/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)blockmean.c	2.23  4/28/95
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/

/*  
   blockmean.c
   reads x, y, data, [weight] on stdin or file and writes out one value
   per cell, where cellular region is bounded by West East South North
   and cell dimensions are delta_x, delta_y.
   
   Latest method uses a hash table and linked lists if the region will
   not simply fit in core memory.
      
   Author:      Walter H. F. Smith
   Version:     3.0, testing hash tables
   Date:        4 October, 1988
   Modified:	26 April 1991-1995 by WHFS for gmt v2.0
   Modified:	3 Jan 1995 by PW for gmt 3.0
*/

#include "gmt.h"

struct	WXYZ {
	double w;
	double	x;
	double	y;
	double	z;
}	*wxyz;	

struct LIST {
	int		key;
	struct WXYZ	sum;
	struct LIST	*p_right;
	struct LIST	*p_left;
}	**hash, *this_box;

int	set_up_arrays();
int	write_output_and_free_space();
struct	LIST *find_box();

main (argc, argv)
int argc;
char **argv;
{

	BOOLEAN	error, weighted, offset, report_weight, nofile = TRUE, done = FALSE, first = TRUE;
	BOOLEAN b_in = FALSE, b_out = FALSE, single_precision = FALSE, more, skip;
	
	FILE *fp = NULL;
	
	double	west, east, south, north, delta_x, delta_y, del_off, *in;
	double	idx, idy;
	
	int	i, j, method, n_x, n_y, ij, x, y, n_expected_fields, n_fields;
	int	n_lost, n_read, n_pitched, n_cells_filled, n_files = 0, fno, n_args;
	
	char	modifier, line[512];

	argc = gmt_begin (argc, argv);

	west = east = south = north = delta_x = delta_y = 0.0;
	del_off = 0.5;
	error = weighted = offset = report_weight = FALSE;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
              
				/* Common parameters */
                      
				case 'H':
				case 'R':
				case 'V':
				case ':':
				case '\0':
                                      error += get_common_args (argv[i], &west, &east, &south, &north);
                                      break;
                              
				/* Supplemental parameters */
                              
				case 'b':
					single_precision = !(argv[i][strlen(argv[i])-1] == 'd');
					if (argv[i][2] == 'i')
						b_in = TRUE;
					else if (argv[i][2] == 'o')
						b_out = TRUE;
					else
						b_in = b_out = TRUE;
					break;
					
				case 'I':
					gmt_getinc (&argv[i][2], &delta_x, &delta_y);
					break;
				case 'N':
					offset = TRUE;
					break;
				case 'W':
					if ( (modifier = argv[i][2]) == 'i' || modifier == 'I') {
						weighted = TRUE;
						report_weight = FALSE;
					}
					else if (modifier == 'O' || modifier == 'o') {
						report_weight = TRUE;
						weighted = FALSE;
					}
					else
						weighted = report_weight = TRUE;
					break;
					
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else
			n_files++;
	}

	if (argc == 1 || gmt_quick) {
		fprintf (stderr, "blockmean %s - Block averaging by L2 norm\n\n", GMT_VERSION);
		fprintf (stderr, "usage: blockmean [infile(s)] -I<xinc[m]>[/<yinc>[m]] -R<west/east/south/north>\n");
		fprintf (stderr, "\t[-H] [-N] [-V] [-W[i][o] ] [-:] [-b[i|o][d]]\n\n");
		
		if (gmt_quick) exit (-1);
		
		fprintf (stderr, "\t-I sets Increment of the grid; enter xinc, optionally xinc/yinc.\n");
		fprintf (stderr, "\t   Default is yinc = xinc.  Append an m [or c] to xinc or yinc to indicate minutes [or seconds],\n");
		fprintf (stderr, "\t   e.g., -I10m/5m grids longitude every 10 minutes, latitude every 5 minutes.\n");
		explain_option ('R');
		fprintf (stderr, "\n\tOPTIONS:\n");
		explain_option ('H');
		fprintf (stderr, "\t-N Offsets registration so block edges are on gridlines.\n");
		explain_option ('V');
		fprintf (stderr, "\t-W sets Weight options.  -WI reads Weighted Input (4 cols: x,y,z,w) but writes only (x,y,z) Output.\n");
		fprintf (stderr, "\t   -WO reads unWeighted Input (3 cols: x,y,z) but reports sum (x,y,z,w) Output.\n");
		fprintf (stderr, "\t   -W with no modifier has both weighted Input and Output; Default is no weights used.\n");
		explain_option (':');
		fprintf (stderr, "\t-b means binary i/o.  Append i or o if only input OR output is binary.\n");
		fprintf (stderr, "\t   Append d for double precision [Default is single precision].\n");
		explain_option ('.');
		exit (-1);
	}
	
	if (!project_info.region_supplied) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify -R option\n", gmt_program);
		error++;
	}
	if (delta_x <= 0.0 || delta_y <= 0.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -I option.  Must specify positive increment(s)\n", gmt_program);
		error++;
	}
	
	if (b_in && gmtdefs.io_header) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data cannot have header -H\n", gmt_program);
		error++;
	}
	if (error) exit (-1);
	
	if (b_in) gmt_input = (single_precision) ? bin_float_input : bin_double_input;
	if (b_out) gmt_output = (single_precision) ? bin_float_output : bin_double_output;
	
	idx = 1.0 / delta_x;
	idy = 1.0 / delta_y;
	n_x = rint((east - west) * idx) + 1;
	n_y = rint((north - south) * idy) + 1;
	
	if (offset) {
		n_x--;
		n_y--;
		del_off = 0.0;
	}
	
	if ( (set_up_arrays(n_x, n_y, &method) ) ) exit (-1);
	
	if (gmtdefs.verbose) {
		fprintf (stderr, "W: %.5lf E: %.5lf S: %.5lf N: %.5lf nx: %d ny: %d\n",
			west, east, south, north, n_x, n_y);
		if (method) {
			fprintf (stderr, "Direct method requires %ld bytes for array.\n", (n_x*n_y*sizeof(struct WXYZ)));
			fprintf (stderr, "Constructing hash table and linked lists.\n");
		}
		else {
			fprintf (stderr, "Region fits inside core memory.\n");
			fprintf (stderr, "Using %ld bytes for array.\n", (n_x*n_y*sizeof(struct WXYZ)));
		}
	}

	gmt_data[3] = 1.0;	/* Default weight if weights not supplied  */

	x = (gmtdefs.xy_toggle) ? 1 : 0;        y = 1 - x;              /* Set up which columns have x and y */
	
	n_read = 0;
	n_pitched = 0;
	n_expected_fields = (weighted) ? 4 : 3;
	
	if (n_files > 0)
		nofile = FALSE;
	else
		n_files = 1;

	n_args = (argc > 1) ? argc : 2;
	
	for (fno = 1; !done && fno < n_args; fno++) {	/* Loop over input files, if any */
		if (!nofile && argv[fno][0] == '-') continue;
		
		if (nofile) {	/* Just read standard input */
			fp = stdin;
			done = TRUE;
		}
		else if ((fp = fopen (argv[fno], "r")) == NULL) {
			fprintf (stderr, "blockmean: Cannot open file %s\n", argv[fno]);
			continue;
		}

		if (!nofile && gmtdefs.verbose) fprintf (stderr, "blockmean: Working on file %s\n", argv[fno]);
		
		if (gmtdefs.io_header) {
			for (i = 0; i < gmtdefs.n_header_recs; i++) {
				fgets (line, 512, fp);
				line[strlen(line)-1] = 0;
				if (first && !b_out) (report_weight && !(weighted)) ? printf ("%s weights\n", line) : printf ("%s\n", line);
			}
			first = FALSE;
		}
	
		more = ((n_fields = gmt_input (fp,  n_expected_fields, &in)) == n_expected_fields);
				
		while (more) {
			skip = FALSE;
			
			if (bad_float (in[2])) skip = TRUE;	/* Skip when z = NaN */
			
			if (!skip) {	/* Check if point is inside area */
				n_read++;
			
				i = floor(((in[x] - west) * idx) + del_off);
				if ( i < 0 || i >= n_x ) skip = TRUE;
				j = floor(((in[y] - south) * idy) + del_off);
				if ( j < 0 || j >= n_y ) skip = TRUE;
			}
			
			if (!skip) {
				if (method) {
					this_box = find_box(i, j, method);
					this_box->key = (n_x > n_y) ? j : i;
					this_box->sum.w += in[3];
					this_box->sum.x += (in[x]*in[3]);
					this_box->sum.y += (in[y]*in[3]);
					this_box->sum.z += (in[2]*in[3]);
				}
				else {
					ij = i * n_y + j;
					wxyz[ij].w += in[3];
					wxyz[ij].x += (in[x]*in[3]);
					wxyz[ij].y += (in[y]*in[3]);
					wxyz[ij].z += (in[2]*in[3]);
				}
				n_pitched++;
			}
			
			more = ((n_fields = gmt_input (fp,  n_expected_fields, &in)) == n_expected_fields);
		}
		if (fp != stdin) fclose(fp);
		
		if (n_fields == -1) n_fields = 0;	/* Ok to get EOF */
		if (n_fields%n_expected_fields) {	/* Got garbage or multiple segment subheader */
			fprintf (stderr, "%s:  Cannot read X Y Z [W] at line %d, aborts\n", gmt_program, n_read);
			exit (-1);
		}
	}

	write_output_and_free_space (n_x, n_y, method, report_weight, &n_cells_filled);

	n_lost = n_read - n_pitched;
	if (gmtdefs.verbose) fprintf (stderr, "N read: %d\tN used: %d\tN outside: %d\tN cells filled: %d\n",
		n_read, n_pitched, n_lost, n_cells_filled);

        gmt_end (argc, argv);
}

int	set_up_arrays(n_x, n_y, method)

int	n_x, n_y, *method;

{

	if ( (wxyz = (struct WXYZ *) calloc ((unsigned int)(n_x * n_y), sizeof (struct WXYZ) ) ) == NULL) {

		/* Full matrix did not fit in core, or failed for other reason.  Try hash table:  */

		if (n_x > n_y) {	/* Hash on x (i); linked list on y (j)  */
			*method = 1;
			if ( (hash = (struct LIST **) calloc ((unsigned int)n_x, sizeof (struct LIST *) ) ) == NULL) {
				fprintf (stderr, "blockmean: Memory allocation error for hash table on x.\n");
				return(-1);
			}
			else {
				return(0);
			}
		}
		else {	/* Hash on y (j); linked list on x (i)  */
			*method = -1;
			if ( (hash = (struct LIST **) calloc ((unsigned int)n_y, sizeof (struct LIST *) ) ) == NULL) {
				fprintf (stderr, "blockmean: Memory allocation error for hash table on y.\n");
				return(-1);
			}
			else {
				return(0);
			}
		}
	}
	else {
		/* Easy method fits in memory.  */
		*method = 0;
		return(0);
	}
}

int	write_output_and_free_space(n_x, n_y, method, report_weight, n_cells_filled)

int	n_x, n_y, method, report_weight, *n_cells_filled;
{
	int	loop, limit, x, y, n_out;
	double	out[4], iw;
	
	x = (gmtdefs.xy_toggle) ? 1 : 0;        y = 1 - x;              /* Set up which columns have x and y */

	n_out = (report_weight) ? 4 : 3;

	*n_cells_filled = 0;
	
	if (method) {	/* We have to loop over the linked lists  */
	
		if (method == 1)
			limit = n_x;
		else
			limit = n_y;
		
		for (loop = 0; loop < limit; loop++) {

			/* If this list is empty, continue:  */
			
			if (hash[loop] == NULL) continue;
			
			/* Go to the leftmost box in the list  */
			
			this_box = hash[loop];
			while (this_box->p_left) this_box = this_box->p_left;
			
			/* While we are at a box, write it, and move right  */
			
			while (this_box) {
			
				(*n_cells_filled)++;
				out[3] = this_box->sum.w;
				iw = 1.0 / out[3];
				out[x] = this_box->sum.x * iw;
				out[y] = this_box->sum.y * iw;
				out[2] = this_box->sum.z * iw;

				gmt_output (stdout, n_out, out);
				
				if (this_box->p_right) {
					this_box = this_box->p_right;
					free ( (char *) this_box->p_left);
				}
				else {
					free ( (char *) this_box);
					this_box = NULL;
				}
			}
		}
		free ((char *) hash);
	}
	
	else {	/* We have a simple array in memory  */
	
		limit = n_x * n_y;
		
		for (loop = 0; loop < limit; loop++) {
		
			if (wxyz[loop].w == 0.0) continue;
			
			(*n_cells_filled)++;
			out[3] = wxyz[loop].w;
			iw = 1.0 / out[3];
			out[x] = wxyz[loop].x * iw;
			out[y] = wxyz[loop].y * iw;
			out[2] = wxyz[loop].z * iw;

			gmt_output (stdout, n_out, out);
		}
		
		free ((char *) wxyz);
	}
}
		
struct LIST *find_box(i, j, method)

int	i, j, method;

{
	int	hash_key, list_key;
	struct LIST *current_box, *temp_box;

	static int last_i = -1;
	static int last_j = -1;
	static struct LIST *last_box = NULL;
	
	if ( (i == last_i) && (j == last_j) ) return (last_box);
	
	/* Get here when we have to search  */
	
	if (method > 0) {
		hash_key = i;
		list_key = j;
	}
	else {
		hash_key = j;
		list_key = i;
	}
	
	current_box = hash[hash_key];
	
	if (current_box) {
	
		/* Hash table has an entry into a linked list;
			if it doesn't match the list_key, search the list */
		
		if (current_box->key < list_key) {
		
			/* Current's key too low; move right while necessary  */
		
			while ( (current_box->p_right) && (current_box->p_right->key <= list_key) )
				current_box = current_box->p_right;
			
			if (current_box->key < list_key) {
			
				/* Current's key still too low  */
			
				if (current_box->p_right) {
				
					/* Next's key too high; insert a box in between  */
					
					if ( (temp_box = (struct LIST *) calloc (1, sizeof (struct LIST))) == NULL) {
						fprintf (stderr, "blockmean: error adding box to list.\n");
						exit(-1);
					}
					temp_box->p_right = current_box->p_right;
					current_box->p_right->p_left = temp_box;
					current_box->p_right = temp_box;
					temp_box->p_left = current_box;
					current_box = temp_box;
				}
				else {
					/* There is no next; make new box at end of list  */
					
					if ( (temp_box = (struct LIST *) calloc (1, sizeof (struct LIST))) == NULL) {
						fprintf (stderr, "blockmean: error adding box to list.\n");
						exit(-1);
					}
					temp_box->p_left = current_box;
					current_box->p_right = temp_box;
					current_box = temp_box;
				}
			}
		}
		
		if (current_box->key > list_key) {
		
			/* Current's key too high; move left while necessary  */
		
			while ( (current_box->p_left) && (current_box->p_left->key >= list_key) )
				current_box = current_box->p_left;
				
			if (current_box->key > list_key) {
			
				/* Current's key still too high  */
			
				if (current_box->p_left) {
				
					/* Next's key too low; insert a box in between  */
					
					if ( (temp_box = (struct LIST *) calloc (1, sizeof (struct LIST))) == NULL) {
						fprintf (stderr, "blockmean: error adding box to list.\n");
						exit(-1);
					}
					temp_box->p_left = current_box->p_left;
					current_box->p_left->p_right = temp_box;
					current_box->p_left = temp_box;
					temp_box->p_right = current_box;
					current_box = temp_box;
				}
				else {
					/* There is no next; make new box at end of list  */
					
					if ( (temp_box = (struct LIST *) calloc (1, sizeof (struct LIST))) == NULL) {
						fprintf (stderr, "blockmean: error adding box to list.\n");
						exit(-1);
					}
					temp_box->p_right = current_box;
					current_box->p_left = temp_box;
					current_box = temp_box;
				}
			}
		}
			

			
		
		
	}
	else  {	
		/* Hash table is NULL; create first box in linked list  */
	
		if ( (current_box = (struct LIST *) calloc (1, sizeof (struct LIST))) == NULL) {
			fprintf (stderr, "blockmean: error allocating first box in list.\n");
			exit(-1);
		}
	}
		
	/* Now we set the state of the static variables and return  */
	
	last_i = i;
	last_j = j;
	last_box = current_box;
	hash[hash_key] = current_box;	/* So we enter lists where we were last time  */
	return (current_box);
}


