/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)gmt_grdio.c	2.8  05 Jul 1995
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 *
 *	G M T _ G R D I O . C   R O U T I N E S
 *
 * Generic routines that take care of all gridfile input/output.
 *
 * Author:	Paul Wessel
 * Date:	9-SEP-1992
 * Modified:	27-JUN-1995
 * Version:	3.0
 *
 * Functions include:
 *
 *	grd_get_i_format :	Get format id for input grdfile
 *	grd_get_o_format :	Get format id for output grdfile
 *	grdio_init :		Set pointers to all i/o routines
 *
 *	read_grd_info :		Read header from file
 *	read_grd :		Read header and data set from file
 *	write_grd_info :	Update header in existing file
 *	write_grd :		Write header and data set to new file
 *
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

#include "gmt.h"

/* GENERIC I/O FUNCTIONS FOR GRIDDED DATA FILES */

int read_grd_info (file, header)
char *file;
struct GRD_HEADER *header; {
	int status;
	char fname[BUFSIZ];
	double scale = 1.0, offset = 0.0;
	
	grd_i_format = grd_get_i_format (file, fname, &scale, &offset);
	status = (*gmtio_readinfo[grd_i_format]) (fname, header);
	header->z_min = header->z_min * scale + offset;
	header->z_max = header->z_max * scale + offset;
	return (status);
}

int write_grd_info (file, header)
char *file;
struct GRD_HEADER *header; {
	int status;
	char fname[BUFSIZ];
	double scale = 1.0, offset = 0.0;
	
	grd_o_format = grd_get_o_format (file, fname, &scale, &offset);
	header->z_min = header->z_min * scale + offset;
	header->z_max = header->z_max * scale + offset;
	status = (*gmtio_writeinfo[grd_o_format]) (fname, header);
	return (status);
}

int read_grd (file, header, grid, w, e, s, n, pad, complex)
char *file;
struct GRD_HEADER *header;
float *grid;
double w, e, s, n;	/* Sub-region to extract  [Use entire file if 0,0,0,0] */
int pad[];		/* padding (# of empty rows/columns) on w, e, s, n of grid, respectively */
BOOLEAN complex;	/* TRUE if array is to hold real and imaginary parts */
{
	int status;
	char fname[BUFSIZ];
	double scale = 1.0, offset = 0.0;
	
	grd_i_format = grd_get_i_format (file, fname, &scale, &offset);
	status = (*gmtio_readgrd[grd_i_format]) (fname, header, grid, w, e, s, n, pad, complex);
	grd_do_scaling (grid, (header->nx * header->ny), scale, offset);
	return (status);
}

int write_grd (file, header, grid, w, e, s, n, pad, complex)
char *file;
struct GRD_HEADER *header;
float *grid;
double w, e, s, n;	/* Sub-region to write out */
int pad[];		/* padding on w, e, s, n of grid, respectively */
BOOLEAN complex;	/* TRUE if array is to hold real and imaginary parts */
{
	int status;
	char fname[BUFSIZ];
	double scale = 1.0, offset = 0.0;

	grd_o_format = grd_get_o_format (file, fname, &scale, &offset);
	grd_do_scaling (grid, (header->nx * header->ny), scale, offset);
	status = (*gmtio_writegrd[grd_o_format]) (fname, header, grid, w, e, s, n, pad, complex);
	return (status);
}

/* Routines to see if a particular grd file format is specified as part of filename. */

int grd_get_i_format (file, fname, scale, offset)
char *file, *fname;
double *scale, *offset; {
	int i = 0, j, n, id = 0;

	strcpy (fname, file);
	
	while (file[i] && file[i] != '=') i++;
	
	if (file[i]) {	/* Check format id */
		i++;
		n = sscanf (&file[i], "%d/%lf/%lf/%lf", &id, scale, offset, &grd_in_nan_value);
		if (n == 1) *scale = 1.0, *offset = 0.0;
		if (id < 0 || id >= N_GRD_FORMATS) {
			fprintf (stderr, "GMT Warning: grdfile format option (%d) unknown, reset to 0\n", id);
			id = 0;
		}
		j = (i == 1) ? i : i - 1;
		fname[j] = 0;
	}
	return (id);
}

int grd_get_o_format (file, fname, scale, offset)
char *file, *fname;
double *scale, *offset; {
	int i = 0, j, n, id = 0;

	strcpy (fname, file);
	
	while (file[i] && file[i] != '=') i++;
	
	if (file[i]) {	/* Check format id */
		i++;
		n = sscanf (&file[i], "%d/%lf/%lf/%lf", &id, scale, offset, &grd_out_nan_value);
		if (n == 1) *scale = 1.0, *offset = 0.0;
		if (id < 0 || id >= N_GRD_FORMATS) {
			fprintf (stderr, "GMT Warning: grdfile format option (%d) unknown, reset to 0\n", id);
			id = 0;
		}
                j = (i == 1) ? i : i - 1;
                fname[j] = 0;
	}
	return (id);
}

/* Routine that scales and offsets the data if specified */

int grd_do_scaling (grid, nm, scale, offset)
float grid[];
int nm;
double scale, offset; {
	int i;
	
	if (scale == 1.0 && offset == 0.0) return;
	
	if (scale == 1.0)
		for (i = 0; i < nm; i++) grid[i] += offset;
	else if (offset == 0.0)
		for (i = 0; i < nm; i++) grid[i] *= scale;
	else
		for (i = 0; i < nm; i++) grid[i] = grid[i] * scale + offset;

	return;
}
