/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)gmt_init.c	2.56  24 Jul 1995
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*									*
 *			G M T _ I N I T . C   V.  2 . 0
 *									*
 *- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * gmt_init.c contains code which is used by all GMT programs
 *
 * Author:	Paul Wessel
 * Date:	27-MAY-1991-1995
 * Version:	2.0, based in part on v1.0
 *
 *
 * The functions are:
 *
 *	explain_option ()  	:	Prints explanations for the options below
 *	get_common_args ()	:	Interprets -B -H -J -K -O -P -R -U -V -X -Y -: -c
 *	get_gmtdefaults ()	:	Initializes the GMT global parameters
 *	gmt_loaddefaults ()	:	Reads the GMT global parameters from .gmtdefaults
 *	gmt_savedefaults ()	:	Writes the GMT global parameters to .gmtdefaults
 *	gmt_setparameter ()	:	Sets a default value given keyord,value-pair
 *	gmt_hash (v)		   : 	Hash functions
 *	init_gmt_hash (v)  	: 	     -"-
 *	gmt_hash_entry (key)	:	     -"-
 *	get_ellipse()	   	:	     -"-
 *	gmt_begin ()   		:	Gets history and init parameters
 *	gmt_end ()	       	:	Updates history and exits
 *	map_getframe ()		:	Scans the -Bstring to set tickinfo
 *	map_getproject ()	   :	Scans the -Jstring to set projection
 *	prepare_three_D ()	:	Initialize 3-D parameters
 *
 */
 
#include "gmt.h"
#include "gmt_init.h"

struct HASH hashnode[HASH_SIZE];

int use_points[3];
double pen_width[3];

int explain_option (option)
char option; {
	/* The function print to stderr a short explanation for the option indicated by
	 * the variable <option>.  Only the common parameter options are covered
	 */
	 
	switch (option) {
	
		case 'B':	/* Tickmark option */
		
			fprintf (stderr, "	-B specifies Boundary info.  <tickinfo> is a textstring made up of one or\n");
			fprintf (stderr, "	   more substrings of the form [t]<tick>[m|c].  The optional [t] can be\n");
			fprintf (stderr, "	   either: a for anotation interval, f for frame interval, or g for\n");
			fprintf (stderr, "	   gridline interval.  If the [t] is not given, a<tick> AND f<tick>\n");
			fprintf (stderr, "	   (but no g<tick>) are set.  The frame interval = anotation interval\n");
			fprintf (stderr, "	   if no separate f<tick> is given.  <tick> is the desired tick-interval.\n");
			fprintf (stderr, "	   The optional [m|s] indicates minutes or seconds.  To specify separate x and y tick-.\n");
			fprintf (stderr, "	   info, separate the strings with a slash [/].  E.g., 5 degree ticks for\n");
			fprintf (stderr, "	   frame AND anotation, 30 minutes grid lines, use\n");
			fprintf (stderr, "	        -B5g30m.   For different y ticks try -B5g30m/2g15m\n");
			fprintf (stderr, "	   [If -Jz is selected, use slashes to separate x, y, and z-tickinfo.]\n");
			fprintf (stderr, "	   Add labels by surrounding them with colons.  If first character is a\n");
			fprintf (stderr, "	   period, then the text is used as the plot title (e.g. :.Plot_Title:).\n");
			fprintf (stderr, "	   Append any combination of w, e, s, n to plot those axes\n");
			fprintf (stderr, "	   only [Default is all].  Append l to anotate log10 (value), p for\n");
			fprintf (stderr, "	   10^(log10(value)).  (See -J for log10 scaling).  For -Jx with power\n");
			fprintf (stderr, "	   scaling, append p to annotate value at equidistant pow increments\n");
			fprintf (stderr, "	   See psbasemap man pages for more details and examples.\n");
			break;
			
		case 'b':	/* Condensed tickmark option */
		
			fprintf (stderr, "	-B Boundary anotation, give -B<xinfo>[/<yinfo>[/<zinfo>]][.:\"title\":][wesnzWESNZ+]\n");
			fprintf (stderr, "	   <?info> is 1-3 substring(s) of form [a|f|g]<tick>[m][l|p] and optionally :\"label\":\n");
			fprintf (stderr, "	   (See psbasemap man pages for more details and examples.)\n");
			break;
			
		case 'H':	/* Header */
		
			fprintf (stderr, "	-H means input/output file has %d Header record(s) [%s]\n",
				gmtdefs.n_header_recs, gmt_choice[gmtdefs.io_header]);
			fprintf (stderr, "	   Optionally, append number of header records\n");
			break;
			
		case 'J':	/* Map projection option */
		
			fprintf (stderr, "	-J Selects the map proJection system. (<mapwidth> is in %s)\n", gmt_unit_names[gmtdefs.measure_unit]);
			
			fprintf (stderr, "	   -Ja<lon0><lat0><scale> OR -JA<lon0><lat0><mapwidth> (Lambert Azimuthal Equal Area)\n");
			fprintf (stderr, "	     lon0/lat0 is the center or the projection.\n");
			fprintf (stderr, "	     Scale is either <1:xxxx> or <radius>/<lat>, where <radius> distance\n");
			fprintf (stderr, "	     is in %s to the oblique parallel <lat>.\n", gmt_unit_names[gmtdefs.measure_unit]);
			
			fprintf (stderr, "	   -Jb<lon0>/<lat0>/<lat1>/<lat2>/<scale> OR -JB<lon0>/<lat0>/<lat1>/<lat2>/<mapwidth> (Albers Equal-Area Conic)\n");
			fprintf (stderr, "	     Give origin, 2 standard parallels, and true scale in %s/degree\n",
				gmt_unit_names[gmtdefs.measure_unit]);
				
			fprintf (stderr, "	   -Jc<lon0>/<lat0><scale> OR -JC<lon0>/<lat0><mapwidth> (Cassini)\n");
			fprintf (stderr, "	     Give central point and scale as 1:xxxx or %s/degree\n", gmt_unit_names[gmtdefs.measure_unit]);
			
			fprintf (stderr, "	   -Je<lon0><lat0><scale> OR -JE<lon0><lat0><mapwidth> (Azimuthal Equidistant)\n");
			fprintf (stderr, "	     lon0/lat0 is the center or the projection.\n");
			fprintf (stderr, "	     Scale is either <1:xxxx> or <radius>/<lat>, where <radius> is distance\n");
			fprintf (stderr, "	     in %s to the oblique parallel <lat>. \n", gmt_unit_names[gmtdefs.measure_unit]);
			
			fprintf (stderr, "	   -Jg<lon0><lat0><scale> OR -JG<lon0><lat0><mapwidth> (Orthographic)\n");
			fprintf (stderr, "	     lon0/lat0 is the center or the projection.\n");
			fprintf (stderr, "	     Scale is either <1:xxxx> or <radius>/<lat>, where <radius> is distance\n");
			fprintf (stderr, "	     in %s to the oblique parallel <lat>. \n", gmt_unit_names[gmtdefs.measure_unit]);
			
			fprintf (stderr, "	   -Jh<lon0>/<scale> OR -JH<lon0>/<mapwidth> (Hammer-Aitoff)\n");
			fprintf (stderr, "	     Give central meridian and scale as 1:xxxx or %s/degree\n", gmt_unit_names[gmtdefs.measure_unit]);
			
			fprintf (stderr, "	   -Ji<lon0>/<scale> OR -JI<lon0>/<mapwidth> (Sinusoidal)\n");
			fprintf (stderr, "	     Give central meridian and scale as 1:xxxx or %s/degree\n", gmt_unit_names[gmtdefs.measure_unit]);

			fprintf (stderr, "	   -Jk<lon0>/<scale> OR -JK<lon0>/<mapwidth> (Eckert IV)\n");
			fprintf (stderr, "	     Give central meridian and scale as 1:xxxx or %s/degree\n", gmt_unit_names[gmtdefs.measure_unit]);

			fprintf (stderr, "	   -Jl<lon0>/<lat0>/<lat1>/<lat2>/<scale> OR -JL<lon0>/<lat0>/<lat1>/<lat2>/<mapwidth> (Lambert Conformal Conic)\n");
			fprintf (stderr, "	     Give origin, 2 standard parallels,  and true scale in %s/degree\n",
				gmt_unit_names[gmtdefs.measure_unit]);
				
			fprintf (stderr, "	   -Jm<scale> OR -JM<mapwidth> (Mercator projection)\n");
			fprintf (stderr, "	     Give true scale at Equator in %s/degree\n",
				gmt_unit_names[gmtdefs.measure_unit]);
				
			fprintf (stderr, "	   -Jn<lon0>/<scale> OR -JN<lon0>/<mapwidth> (Robinson projection)\n");
			fprintf (stderr, "	     Give central meridian and scale as 1:xxxx or %s/degree\n", gmt_unit_names[gmtdefs.measure_unit]);

			fprintf (stderr, "	   -Jo | -JO (Oblique Mercator).  Specify one of three definitions:\n");
			fprintf (stderr, "	      -Joa<orig_lon>/<orig_lat>/<azimuth>/<scale> OR -JOa<orig_lon>/<orig_lat>/<azimuth>/<mapwidth>\n");
			fprintf (stderr, "	      		Give origin and azimuth of oblique equator\n");
			fprintf (stderr, "	      -Job<orig_lon>/<orig_lat>/<b_lon>/<b_lat>/<scale> OR -JOb<orig_lon>/<orig_lat>/<b_lon>/<b_lat>/<mapwidth>\n");
			fprintf (stderr, "	      		Give origin and second point on oblique equator\n");
			fprintf (stderr, "	      -Joc<orig_lon>/<orig_lat>/<pole_lon>/<pole_lat>/<scale> OR -JOc<orig_lon>/<orig_lat>/<pole_lon>/<pole_lat>/<mapwidth>\n");
			fprintf (stderr, "	      		Give origin and pole of projection\n");
			fprintf (stderr, "	        Scale is true scale at oblique equator in %s/degree\n",
				gmt_unit_names[gmtdefs.measure_unit]);
			fprintf (stderr, "	        Specify region in oblique degrees OR use -R<>r\n");
			
			fprintf (stderr, "	   -Jq<lon0>/<scale> OR -JQ<lon0>/<mapwidth> (Equidistant Cylindrical)\n");
			fprintf (stderr, "	     Give central meridian and scale as 1:xxxx or %s/degree\n", gmt_unit_names[gmtdefs.measure_unit]);
			
			fprintf (stderr, "	   -Jr<lon0>/<scale> OR -JR<lon0>/<mapwidth> (Winkel Tripel)\n");
			fprintf (stderr, "	     Give central meridian and scale as 1:xxxx or %s/degree\n", gmt_unit_names[gmtdefs.measure_unit]);

			fprintf (stderr, "	   -Js<lon0><lat0><scale> OR -JS<lon0><lat0><mapwidth> (Stereographic)\n");
			fprintf (stderr, "	     lon0/lat0 is the center or the projection.\n");
			fprintf (stderr, "	     Scale is either <1:xxxx> or <radius>/<lat>, where <radius> distance\n");
			fprintf (stderr, "	     is in %s to the oblique parallel <lat>.\n", gmt_unit_names[gmtdefs.measure_unit]);
			
			fprintf (stderr, "	   -Jt<lon0>/<scale> OR -JT<lon0>/<mapwidth> (Transverse Mercator)\n");
			fprintf (stderr, "	      Give central meridian and scale as 1:xxxx or %s/degree\n",
				gmt_unit_names[gmtdefs.measure_unit]);
				
			fprintf (stderr, "	   -Ju<zone>/<scale> OR -JU<zone>/<mapwidth> (UTM)\n");
			fprintf (stderr, "	     Give zone (negative for S hemisphere) and scale as 1:xxxx or %s/degree\n",
				gmt_unit_names[gmtdefs.measure_unit]);
				
			fprintf (stderr, "	   -Jw<lon0>/<scale> OR -JW<lon0>/<mapwidth> (Mollweide)\n");
			fprintf (stderr, "	     Give central meridian and scale as 1:xxxx or %s/degree\n", gmt_unit_names[gmtdefs.measure_unit]);
			
			fprintf (stderr, "	   -Jy<lon0>/<lats>/<scale> OR -JY<lon0>/<lats>/<mapwidth> (Cylindrical Equal-area)\n");
			fprintf (stderr, "	     Give central meridian, standard parallel and scale as 1:xxxx or %s/degree\n", gmt_unit_names[gmtdefs.measure_unit]);
			fprintf (stderr, "	     <slat> = 45 (Peters), 37.4 (Trystan Edwards), 30 (Behrmann), 0 (Lambert)\n");

			fprintf (stderr, "	   -Jp<scale> OR -JP<mapwidth> (Polar (theta,radius))\n");
			fprintf (stderr, "	     Linear scaling for polar coordinates.  Give scale in %s/units\n",
				gmt_unit_names[gmtdefs.measure_unit]);
				
			fprintf (stderr, "	   -Jx OR -JX for non-map projections.  Scale in %s/units.  Specify one:\n",
				gmt_unit_names[gmtdefs.measure_unit]);
			fprintf (stderr, "	      -Jx<x-scale>		Linear projection\n");
			fprintf (stderr, "	      -Jx<x-scale>l		Log10 projection\n");
			fprintf (stderr, "	      -Jx<x-scale>p<power>	x^power projection\n");
			fprintf (stderr, "	      Use / to specify separate x/y scaling (e.g., -Jx0.5/0.3.)\n");
			fprintf (stderr, "	      If -JX is used then give axes lengths rather than scales\n");
			break;
			
		case 'j':	/* Condensed version of J */
		
			fprintf (stderr, "	-J Selects map proJection. (<scale> in %s/degree, <mapwidth> in %s)\n", gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
			
			fprintf (stderr, "	   -Ja|A<lon0><lat0><scale (or radius/lat)|mapwidth> (Lambert Azimuthal Equal Area)\n");
			
			fprintf (stderr, "	   -Jb|B<lon0>/<lat0>/<lat1>/<lat2>/<scale|mapwidth> (Albers Equal-Area Conic)\n");
			fprintf (stderr, "	   -Jc|C<lon0>/<lat0><scale|mapwidth> (Cassini)\n");
			
			fprintf (stderr, "	   -Je|E<lon0><lat0><scale (or radius/lat)|mapwidth>  (Azimuthal Equidistant)\n");
			
			fprintf (stderr, "	   -Jg|G<lon0><lat0><scale (or radius/lat)|mapwidth>  (Orthographic)\n");
			
			fprintf (stderr, "	   -Jh|H<lon0>/<scale|mapwidth> (Hammer-Aitoff)\n");
			
			fprintf (stderr, "	   -Ji|I<lon0>/<scale|mapwidth> (Sinusoidal)\n");

			fprintf (stderr, "	   -Jk|K<lon0>/<scale|mapwidth> (Eckert IV)\n");

			fprintf (stderr, "	   -Jl|L<lon0>/<lat0>/<lat1>/<lat2>/<scale|mapwidth> (Lambert Conformal Conic)\n");
				
			fprintf (stderr, "	   -Jm|M<scale|mapwidth> (Mercator projection)\n");
				
			fprintf (stderr, "	   -Jn|N<lon0>/<scale|mapwidth> (Robinson projection)\n");

			fprintf (stderr, "	   -Jo|O (Oblique Mercator).  Specify one of three definitions:\n");
			fprintf (stderr, "	      -Jo|Oa<orig_lon>/<orig_lat>/<azimuth>/<scale|mapwidth>\n");
			fprintf (stderr, "	      -Jo|Ob<orig_lon>/<orig_lat>/<b_lon>/<b_lat>/<scale|mapwidth>\n");
			fprintf (stderr, "	      -Jo|Oc<orig_lon>/<orig_lat>/<pole_lon>/<pole_lat>/<scale|mapwidth>\n");
			
			fprintf (stderr, "	   -Jq|Q<lon0>/<scale|mapwidth> (Equidistant Cylindrical)\n");
			
			fprintf (stderr, "	   -Jr|R<lon0>/<scale|mapwidth> (Winkel Tripel)\n");

			fprintf (stderr, "	   -Js|S<lon0><lat0><scale (or radius/lat)|mapwidth> (Stereographic)\n");
			
			fprintf (stderr, "	   -Jt|T<lon0>/<scale|mapwidth> (Transverse Mercator)\n");
				
			fprintf (stderr, "	   -Ju|U<zone>/<scale|mapwidth> (UTM)\n");
				
			fprintf (stderr, "	   -Jw|W<lon0>/<scale|mapwidth> (Mollweide)\n");
			
			fprintf (stderr, "	   -Jy|Y<lon0>/<lats>/<scale|mapwidth> (Cylindrical Equal-area)\n");

			fprintf (stderr, "	   -Jp|P<scale|mapwidth> (Polar (theta,radius))\n");
				
			fprintf (stderr, "	   -Jx|X<x-scale|mapwidth>[l|p<power>][/<y-scale|mapheight>[l|p<power>]] (Linear projections)\n");
			fprintf (stderr, "	   (See psbasemap for more details on projection syntax)\n");
			break;
			
		case 'K':	/* Append-more-PostScript-later */

			fprintf (stderr, "	-K means allow for more plot code to be appended later [%s].\n",
				gmt_choice[!gmtdefs.last_page]);
			break;
			
		case 'O':	/* Overlay plot */

			fprintf (stderr, "	-O means Overlay plot mode [%s].\n",
				gmt_choice[gmtdefs.overlay]);
			break;
			
		case 'P':	/* Portrait or landscape */
		
			fprintf (stderr, "	-P means Portrait page orientation [%s].\n",
				gmt_choice[(gmtdefs.page_orientation & 1)]);
			break;
			
		case 'R':	/* Region option */
		
			fprintf (stderr, "	-R specifies the min/max coordinates of data region in user units.\n");
			fprintf (stderr, "	   Use dd:mm[:ss] format for regions given in degrees and minutes [and seconds].\n");
			fprintf (stderr, "	   Append r if -R specifies the longitudes/latitudes of the lower left\n");
			fprintf (stderr, "	   and upper right corners of a rectangular area\n");
			break;
			
		case 'r':	/* Region option for 3-D */
		
			fprintf (stderr, "	-R specifies the xyz min/max coordinates of the plot window in user units.\n");
			fprintf (stderr, "	   Use dd:mm[:ss] format for regions given in degrees and minutes [and seconds].\n");
			fprintf (stderr, "	   Append r if first 4 arguments to -R specifies the longitudes/latitudes\n");
			fprintf (stderr, "	   of the lower left and upper right corners of a rectangular area\n");
			break;
			
		case 'U':	/* Plot time mark and [optionally] command line */
		
			fprintf (stderr, "	-U to plot Unix System Time stamp [and optionally appended text].\n");
			fprintf (stderr, "	   You may also set the lower left corner position of stamp [%lg/%lg].\n",
				gmtdefs.unix_time_pos[0], gmtdefs.unix_time_pos[1]);
			fprintf (stderr, "	   Give -Uc to have the command line plotted [%s].\n",
				gmt_choice[gmtdefs.unix_time]);
			break;
			
		case 'V':	/* Verbose */
		
			fprintf (stderr, "	-V Run in verbose mode [%s].\n", gmt_choice[gmtdefs.verbose]);
			break;
			
		case 'X':
		case 'Y':	/* Reset plot origin option */
		
			fprintf (stderr, "	-X -Y to shift origin of plot to (<xshift>, <yshift>) [%lg,%lg].\n",
				gmtdefs.x_origin, gmtdefs.y_origin);
			fprintf (stderr, "	   (Note that for overlays (-O), the default is [0,0].)\n");
			break;
			
		case 'Z':	/* Vertical scaling for 3-D plots */
		
			fprintf (stderr, "	   -Jz for z component of 3-D projections.  Same syntax as -Jx.\n");
			break;
			
		case 'c':	/* Set number of plot copies option */
		
			fprintf (stderr, "	-c specifies the number of copies [%d].\n", gmtdefs.n_copies);
			break;

		case ':':	/* lon/lat or lat/lon */

			fprintf (stderr, "	-: Expect lat/lon input rather than lon/lat [%s].\n",
				gmt_choice[gmtdefs.xy_toggle]);
			break;
			
		case '.':	/* Trailer message */
		
			fprintf (stderr, "	(See man page for gmtdefaults to set these and other defaults to ON or OFF)\n");
			break;

		default:
			break;
	}
}

int gmt_fill_syntax (option)
char option; {
	fprintf (stderr, "%s: GMT SYNTAX ERROR -%c option.  Correct syntax:\n", gmt_program, option);
	fprintf (stderr, "\t-%cP|p<size>/<pattern>, size of tile (in inch), pattern from 1-32 or a filename\n", option);
	fprintf (stderr, "\t-%c<red>/<green>/<blue> or -%c<gray>, all in the 0-255 range\n", option, option);
}

int gmt_pen_syntax (option)
char option; {
	fprintf (stderr, "%s: GMT SYNTAX ERROR -%c option.  Correct syntax:\n", gmt_program, option);
	fprintf (stderr, "\t-%c[<width>][/<red>/<green>/<blue> | <gray>][to | ta | t<texture>:<offset>][p]\n", option);
	fprintf (stderr, "\t  <width> >= 0, <red>/<green>/<blue> or <gray> all in the 0-255 range\n");
}

int gmt_rgb_syntax (option)
char option; {
	fprintf (stderr, "%s: GMT SYNTAX ERROR -%c option.  Correct syntax:\n", gmt_program, option);
	fprintf (stderr, "\t-%c<red>/<green>/<blue> or -%c<gray>, all in the 0-255 range\n", option, option);
}

int gmt_syntax (option)
char option; {
	/* The function print to stderr the syntax for the option indicated by
	 * the variable <option>.  Only the common parameter options are covered
	 */
	 
	fprintf (stderr, "%s: GMT SYNTAX ERROR -%c option.  Correct syntax:\n", gmt_program, option);

	switch (option) {
	
		case 'B':	/* Tickmark option */
		
			fprintf (stderr, "\t-B[a|f|g]<tick>[m][l|p]:\"label\":[/.../...]:.\"Title\":[W|w|E|e|S|s|N|n][Z|z]\n");
			break;
			
		case 'H':	/* Header */
		
			fprintf (stderr, "\t-H[n-header-records]\n");
			break;
			
		case 'J':	/* Map projection option */
		
			switch (project_info.projection) {
				case LAMB_AZ_EQ:
					fprintf (stderr, "\t-Ja<lon0><lat0><scale> OR -Ja<lon0><lat0><mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or <radius> (in %s)/<lat>, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case ALBERS:
					fprintf (stderr, "\t-Jb<lon0>/<lat0>/<lat1>/<lat2>/<scale> OR -Jb<lon0>/<lat0>/<lat1>/<lat2>/<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case CASSINI:
					fprintf (stderr, "\t-Jc<lon0>/<lat0><scale> OR -JC<lon0>/<lat0><mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree ,or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case ORTHO:
					fprintf (stderr, "\t-Jg<lon0><lat0><scale> OR -JG<lon0><lat0><mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or <radius> (in %s)/<lat>, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case HAMMER:
					fprintf (stderr, "\t-Jh<lon0>/<scale> OR -JH<lon0>/<mapwidth\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case SINUSOIDAL:
					fprintf (stderr, "\t-Ji<lon0>/<scale> OR -JI<lon0>/<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case LAMBERT:
					fprintf (stderr, "\t-Jl<lon0>/<lat0>/<lat1>/<lat2>/<scale> OR -JL<lon0>/<lat0>/<lat1>/<lat2>/<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case MERCATOR:
					fprintf (stderr, "\t-Jm<scale> OR -JM<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case ROBINSON:
					fprintf (stderr, "\t-Jn<lon0>/<scale> OR -JN<lon0>/<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case OBLIQUE_MERC:
					fprintf (stderr, "\t-JOa<lon0>/<lat0>/<azimuth>/<scale> OR -JOa<lon0>/<lat0>/<azimuth>/<mapwidth>\n");
					fprintf (stderr, "\t-Job<lon0>/<lat0>/<b_lon>/<b_lat>/<scale> OR -JOb<lon0>/<lat0>/<b_lon>/<b_lat>/<mapwidth>\n");
					fprintf (stderr, "\t-Joc<lon0>/<lat0>/<lonp>/<latp>/<scale> OR -JOc<lon0>/<lat0>/<lonp>/<latp>/<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/oblique degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case WINKEL:
					fprintf (stderr, "\t-Jr<lon0>/<scale> OR -JR<lon0><mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case CYL_EQDIST:
					fprintf (stderr, "\t-Jq<lon0>/<scale> OR -JQ<lon0><mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case STEREO:
					fprintf (stderr, "\t-Js<lon0><lat0><scale> OR -JS<lon0><lat0><mapwidth>\n"); 
					fprintf (stderr, "\t  <scale is <1:xxxx> or <radius> (in %s)/<lat>, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case TM:
					fprintf (stderr, "\t-Jt<lon0>/<scale> OR -JT<lon0>/<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case UTM:
					fprintf (stderr, "\t-Ju<zone>/<scale> OR -JU<zone>/<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case MOLLWEIDE:
					fprintf (stderr, "\t-Jw<lon0>/<scale> OR -JW<lon0>/<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case ECKERT:
					fprintf (stderr, "\t-Jk<lon0>/<scale> OR -JK<lon0>/<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case CYL_EQ:
					fprintf (stderr, "\t-Jy<lon0>/<lats>/<scale> OR -JY<lon0>/<lats>/<mapwidth>\n");
					fprintf (stderr, "\t  <scale is <1:xxxx> or %s/degree, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
					break;
				case POLAR:
					fprintf (stderr, "\t-Jp<scale> OR -JP<mapwidth>\n");
					fprintf (stderr, "\t  <scale is %s/units, or use <mapwidth> in %s\n",
						gmt_unit_names[gmtdefs.measure_unit], gmt_unit_names[gmtdefs.measure_unit]);
				case LINEAR:
					fprintf (stderr, "\t-Jx<x-scale>[l|p<power>][/<y-scale>[l|p<power>]], scale in %s/units\n",
						gmt_unit_names[gmtdefs.measure_unit]);
					fprintf (stderr, "\t-Jz<z-scale>[l|p<power>], scale in %s/units\n",
						gmt_unit_names[gmtdefs.measure_unit]);
					fprintf (stderr, "\tUse -JX (and/or -JZ) to give axes lengths rather than scales\n");
					break;
				default:
					fprintf (stderr, "\tProjection not recognized!\n");
					break;
			}
			break;
	
		case 'R':	/* Region option */
		
			fprintf (stderr, "\t-R<xmin>/<xmax>/<ymin>/<ymax>[/<zmin>/<zmax>], dd:mm format ok\n");
			fprintf (stderr, "\tAppend r if giving lower left and upper right coordinates\n");
			break;
			
		case 'U':	/* Set time stamp option */
		
			fprintf (stderr, "\t-U[/<dx>/<dy>/][<string> | c], c will plot command line.\n");
			break;
			
		case 'c':	/* Set number of plot copies option */
		
			fprintf (stderr, "\t-c<copies>, copies is number of copies\n");
			break;

		default:
			break;
	}
}

int gmt_default_error (option)
char option; {
	fprintf (stderr, "%s: GMT SYNTAX ERROR:  Unrecognized option -%c\n", gmt_program, option);
}

int get_common_args (item, w, e, s, n)
char *item;
double *w, *e, *s, *n; {
	char *text, string[100];
	
	/* get_common_args interprets the command line for the common, unique options
	 * -B, -H, -J, -K, -O, -P, -R, -U, -V, -X, -Y, -c, -:, -
	 */
	 
	int i, j, nn, n_slashes, error = 0;
	double *p[6];
	
	switch (item[1]) {
		case '\0':
			gmt_quick = TRUE;
			break;
		case 'B':
			error += (i = map_getframe (&item[2]));
			if (i) gmt_syntax ('B');
			break;
		case 'H':
			if (item[2]) {
				i = atoi (&item[2]);
				if (i < 0) {
					gmt_syntax ('H');
					error++;
				}
				else
					gmtdefs.n_header_recs = i;
			}
			gmtdefs.io_header = (gmtdefs.n_header_recs > 0);
			break;
		case 'J':
			error += (i = map_getproject (&item[2]));
			if (i) gmt_syntax ('J');
			break;
		case 'K':
			gmtdefs.last_page = FALSE;
			break;
		case 'O':
			gmtdefs.overlay = TRUE;
			break;
		case 'P':
			gmtdefs.page_orientation |= 1;	/* Bit arith because eurofont bit may be set */
			break;
		case 'R':
			p[0] = w;	p[1] = e;	p[2] = s;	p[3] = n;
			p[4] = &project_info.z_bottom;	p[5] = &project_info.z_top;
	 		if (item[strlen(item)-1] == 'r') project_info.region = FALSE; /* Rectangular box given */
	 		project_info.region_supplied = TRUE;
			
			i = 0;
			strcpy (string, &item[2]);
			text = strtok (string, "/");
			while (text) {
				*p[i] = ddmmss_to_degree (text);
				i++;
				text = strtok (CNULL, "/");
			}
			if ((i < 4 || i > 6) || (check_region (*p[0], *p[1], *p[2], *p[3]) || (i == 6 && *p[4] >= *p[5]))) {
				error++;
				gmt_syntax ('R');
			}
			break;
		case 'U':
			gmtdefs.unix_time = TRUE;
			for (i = n_slashes = 0; item[i]; i++) if (item[i] == '/') {
				n_slashes++;
				if (n_slashes < 4) j = i;
			}
			if (item[2] == '/' && n_slashes == 2)	/* Gave -U/<dx>/<dy> */
				nn = sscanf (&item[3], "%lf/%lf", &gmtdefs.unix_time_pos[0], &gmtdefs.unix_time_pos[1]);
			else if (item[2] == '/' && n_slashes > 2) {	/* Gave -U<dx>/<dy>/<string> */
				nn = sscanf (&item[3], "%lf/%lf", &gmtdefs.unix_time_pos[0], &gmtdefs.unix_time_pos[1]);
				strcpy (gmtdefs.unix_time_label, &item[j+1]);
			}
			else if (item[2] && item[2] != '/')	/* Gave -U<string> */
				strcpy (gmtdefs.unix_time_label, &item[2]);
			if ((item[2] == '/' && n_slashes == 1) || (item[2] == '/' && n_slashes >= 2 && nn != 2)) {
				error++;
				gmt_syntax ('U');
			}
			break;
		case 'V':
			gmtdefs.verbose = TRUE;
			break;
		case 'X':
		case 'x':
			gmtdefs.x_origin = atof (&item[2]);
			project_info.x_off_supplied = TRUE;
			break;
		case 'Y':
		case 'y':
			gmtdefs.y_origin = atof (&item[2]);
			project_info.y_off_supplied = TRUE;
			break;
		case 'c':
			i = atoi (&item[2]);
			if (i < 1) {
				error++;
				gmt_syntax ('c');
			}
			else
				gmtdefs.n_copies = i;
			break;
		case ':':	/* Toggle lon/lat - lat/lon */
			gmtdefs.xy_toggle = TRUE;
			break;
		default:	/* Should never get here, but... */
			error++;
			fprintf (stderr, "GMT: Warning: bad case in get_common_args\n");
			break;
	}
	
	return (error);
}

double ddmmss_to_degree (text)
char *text; {
	int i, colons = 0;
	double degree, minute, degfrac, second;

	for (i = 0; text[i]; i++) if (text[i] == ':') colons++;
	if (colons == 2) {	/* dd:mm:ss format */
		sscanf (text, "%lf:%lf:%lf", &degree, &minute, &second);
		degfrac = degree + copysign (minute / 60.0, degree) + copysign (second / 3600.0, degree);
	}
	else if (colons == 1) {	/* dd:mm format */
		sscanf (text, "%lf:%lf", &degree, &minute);
		degfrac = degree + copysign (minute / 60.0, degree);
	}
	else
		degfrac = atof (text);
	return (degfrac);
}

int gmt_loaddefaults (file)
char *file; {
	int i, error = 0, entry;
	char line[512], keyword[40], value[80];
	FILE *fp = NULL;
	struct HASH *this;
	
	if ((fp = fopen (file, "r")) == NULL) return (-1);
			
	/* Set up hash table */
	
	init_gmt_hash (hashnode, gmt_keywords, HASH_SIZE, N_KEYS);

	while (fgets (line, 512, fp)) {
		if (line[0] == '#') continue;	/* Skip comments */
		sscanf (line, "%s = %s", keyword, value);
		
		error += gmt_setparameter (keyword, value);
	}
	
	fclose (fp);
	if (gmtdefs.want_euro_font) gmtdefs.page_orientation += 2;
	if (gmtdefs.ps_heximage) gmtdefs.page_orientation += 4;
	if (use_points[0]) gmtdefs.frame_pen = rint (pen_width[0] * gmtdefs.dpi / 72.0), use_points[0] = 2;
	if (use_points[1]) gmtdefs.grid_pen = rint (pen_width[1] * gmtdefs.dpi / 72.0), use_points[1] = 2;
	if (use_points[2]) gmtdefs.tick_pen = rint (pen_width[2] * gmtdefs.dpi / 72.0), use_points[2] = 2;

	if (error) fprintf (stderr, "GMT:  %d conversion errors in file %s!\n", error, file);
	
	return (0);
}

int gmt_setdefaults (argc, argv)
int argc;
char **argv; {
	int i, j, error = 0, entry;
	struct HASH *this;
	
	/* Set up hash table */
	
	init_gmt_hash (hashnode, gmt_keywords, HASH_SIZE, N_KEYS);

	for (j = 1; j < argc; j += 2) error += gmt_setparameter (argv[j], argv[j+1]);

	if (gmtdefs.want_euro_font) gmtdefs.page_orientation += 2;
	if (gmtdefs.ps_heximage) gmtdefs.page_orientation += 4;
	if (use_points[0] == 1) gmtdefs.frame_pen = rint (pen_width[0] * gmtdefs.dpi / 72.0), use_points[0] = 2;
	if (use_points[1] == 1) gmtdefs.grid_pen = rint (pen_width[1] * gmtdefs.dpi / 72.0), use_points[1] = 2;
	if (use_points[2] == 1) gmtdefs.tick_pen = rint (pen_width[2] * gmtdefs.dpi / 72.0), use_points[2] = 2;
	
	if (error) fprintf (stderr, "gmtset:  %d conversion errors\n", error);
}

int gmt_setparameter (keyword, value)
char *keyword, *value; {
	int i, error = FALSE;
	
	switch (gmt_hash_entry (keyword, hashnode, HASH_SIZE)) {
		case 0:
			gmtdefs.anot_min_angle = atof (value);
			break;
		case 1:
			gmtdefs.anot_font = get_entry (value, font_name, N_FONTS);
			break;
		case 2:
			gmtdefs.anot_font_size = atoi (value);
			break;
		case 3:
			gmtdefs.anot_offset = atof (value);
			break;
		case 4:
			strcpy (gmtdefs.basemap_axes, value);
			for (i = 0; value[i]; i++) {
				switch (value[i]) {
					case 'W':	/* Upper case: Draw axis/ticks AND anotate */
						frame_info.side[3] = 2;
						break;
					case 'w':	/* Lower case: Draw axis/ticks only */
						frame_info.side[3] = 1;
						break;
					case 'E':
						frame_info.side[1] = 2;
						break;
					case 'e':
						frame_info.side[1] = 1;
						break;
					case 'S':
						frame_info.side[0] = 2;
						break;
					case 's':
						frame_info.side[0] = 1;
						break;
					case 'N':
						frame_info.side[2] = 2;
						break;
					case 'n':
						frame_info.side[2] = 1;
						break;
					default:
						error = TRUE;
						fprintf (stderr, "%s: GMT SYNTAX ERROR in gmt_setparameter:  Unrecognized modifier for %s: %c\n", gmt_program, keyword, value[i]);
						break;
				}
			}
			break;
		case 5:
			sscanf (value, "%d/%d/%d", &gmtdefs.basemap_frame_rgb[0],
				&gmtdefs.basemap_frame_rgb[1],  &gmtdefs.basemap_frame_rgb[2]);
			break;
		case 6:
			gmtdefs.basemap_type = (strcmp (value, "PLAIN")) ? 0 : 1;
			break;
		case 7:
			sscanf (value, "%d/%d/%d", &gmtdefs.background_rgb[0],
				&gmtdefs.background_rgb[1],  &gmtdefs.background_rgb[2]);
			break;
		case 8:
			sscanf (value, "%d/%d/%d", &gmtdefs.foreground_rgb[0],
				&gmtdefs.foreground_rgb[1],  &gmtdefs.foreground_rgb[2]);
			break;
		case 9:
			sscanf (value, "%d/%d/%d", &gmtdefs.nan_rgb[0],
				&gmtdefs.nan_rgb[1],  &gmtdefs.nan_rgb[2]);
			break;
		case 10:
			if (!strcmp (value, "ADOBE"))
				gmtdefs.color_image = 0;
			else if (!strcmp (value, "TILES"))
				gmtdefs.color_image = 1;
			else if (!strcmp (value, "SEPARATION"))
				gmtdefs.color_image = 2;
			else
				error = TRUE;
			break;
		case 11:
			gmtdefs.color_model = (!strcmp (value, "HSV")) ? 1 : 0;
			break;
		case 12:
			strcpy (gmtdefs.d_format, value);
			break;
		case 13:
			gmtdefs.degree_format = atoi (value);
			break;
		case 14:
			gmtdefs.dpi = atoi (value);
			break;
		case 15:
			gmtdefs.ellipsoid = get_ellipse (value);
			break;
		case 16:
			if (value[strlen(value)-1] == 'p') {
				use_points[0] = 1;
				pen_width[0] = atof (value);
			}
			else {
				use_points[0] = 0;
				gmtdefs.frame_pen = atoi (value);
			}
			break;
		case 17:
			gmtdefs.frame_width = atof (value);
			break;
		case 18:
			gmtdefs.global_x_scale = atof (value);
			break;
		case 19:
			gmtdefs.global_y_scale = atof (value);
			break;
		case 20:
			gmtdefs.grid_cross_size = atof (value);
			break;
		case 21:
			if (value[strlen(value)-1] == 'p') {
				use_points[1] = 1;
				pen_width[1] = atof (value);
			}
			else {
				use_points[1] = 0;
				gmtdefs.grid_pen = atoi (value);
			}
			break;
		case 22:
			gmtdefs.header_font = get_entry (value, font_name, N_FONTS);
			break;
		case 23:
			gmtdefs.header_font_size = atoi (value);
			break;
		case 24:
			gmtdefs.hsv_min_saturation = atof (value);
			break;
		case 25:
			gmtdefs.hsv_max_saturation = atof (value);
			break;
		case 26:
			gmtdefs.hsv_min_value = atof (value);
			break;
		case 27:
			gmtdefs.hsv_max_value = atof (value);
			break;
		case 28:
			if (!strcmp (value, "LINEAR"))
				gmtdefs.interpolant = 0;
			else if (!strcmp (value, "AKIMA"))
				gmtdefs.interpolant = 1;
			else if (!strcmp (value, "CUBIC"))
				gmtdefs.interpolant = 2;
			else
				error = TRUE;
			break;
		case 29:
			gmtdefs.io_header = (strcmp (value, "TRUE")) ? 0 : 1;
			break;
		case 30:
			gmtdefs.n_header_recs = atoi (value);
			break;
		case 31:
			gmtdefs.label_font = get_entry (value, font_name, N_FONTS);
			break;
		case 32:
			gmtdefs.label_font_size = atoi (value);
			break;
		case 33:
			if ((gmtdefs.line_step = atof (value)) <= 0.0) {
#ifdef SI
				gmtdefs.line_step = 0.25;
#else
				gmtdefs.line_step = 0.1;
#endif
				fprintf (stderr, "%s: GMT WARNING in gmt_setparameter: %s given as 0, reset to %lg\n", 
				gmt_program, keyword, gmtdefs.line_step);
			}
			break;
		case 34:
			gmtdefs.map_scale_factor = atof (value);
			break;
		case 35:
			if (!(strcmp (value, "CM") && strcmp (value, "cm"))) 
				gmtdefs.measure_unit = 0;
			else if (!(strcmp (value, "INCH") && strcmp (value, "inch"))) 
				gmtdefs.measure_unit = 1;
			else if (!(strcmp (value, "M") && strcmp (value, "m"))) 
				gmtdefs.measure_unit = 2;
			else
#ifdef SI
				gmtdefs.measure_unit = 0;	/* Let cm be default */
#else
				gmtdefs.measure_unit = 1;	/* Let inch be default */
#endif
			break;
		case 36:
			gmtdefs.n_copies = atoi (value);
			break;
		case 37:
			gmtdefs.oblique_anotation = atoi (value);
			break;
		case 38:
			sscanf (value, "%d/%d/%d", &gmtdefs.page_rgb[0],
				&gmtdefs.page_rgb[1],  &gmtdefs.page_rgb[2]);
			break;
		case 39:
			gmtdefs.page_orientation = (value[0] == 'L' || value[0] == 'l') ? 0 : 1;
			break;
		case 40:
			gmtdefs.paper_width = atof (value);
			break;
		case 41:
			gmtdefs.ps_heximage = (strcmp (value, "HEX")) ? 0 : 1;
			break;
		case 42:
			gmtdefs.tick_length = atof (value);
			break;
		case 43:
			if (value[strlen(value)-1] == 'p') {
				use_points[2] = 1;
				pen_width[2] = atof (value);
			}
			else {
				use_points[2] = 0;
				gmtdefs.tick_pen = atoi (value);
			}
			break;
		case 44:
			gmtdefs.unix_time = (strcmp (value, "TRUE")) ? 0 : 1;
			break;
		case 45:
			sscanf (value, "%lf/%lf", &gmtdefs.unix_time_pos[0], &gmtdefs.unix_time_pos[1]);
			break;
		case 46:
			gmtdefs.vector_shape = atof (value);
			break;
		case 47:
			gmtdefs.verbose = (strcmp (value, "TRUE")) ? 0 : 1;
			break;
		case 48:
			gmtdefs.want_euro_font = (strcmp (value, "TRUE")) ? 0 : 1;
			break;
		case 49:
			gmtdefs.x_axis_length = atof (value);
			break;
		case 50:
			gmtdefs.y_axis_length = atof (value);
			break;
		case 51:
			gmtdefs.x_origin = atof (value);
			break;
		case 52:
			gmtdefs.y_origin = atof (value);
			break;
		case 53:
			gmtdefs.xy_toggle = (strcmp (value, "TRUE")) ? 0 : 1;
			break;
		case 54:
			gmtdefs.y_axis_type = (strcmp (value, "VER_TEXT")) ? 0 : 1;
			break;
		default:
			error = TRUE;
			fprintf (stderr, "%s: GMT SYNTAX ERROR in gmt_setparameter:  Unrecognized keyword %s\n", 
				gmt_program, keyword);
			break;
	}
	
	return (error);
}

int gmt_savedefaults (file)
char *file; {
	FILE *fp;
	double pen;
	
	if (!file)
		fp = stdout;
	else if ((fp = fopen (file, "w")) == NULL) {
		fprintf (stderr, "GMT: Could not create file %s\n", file);
		return (-1);
	}
	
	fprintf (fp, "#\n#	GMT-SYSTEM %s Default file\n#\n", GMT_VERSION);
	fprintf (fp, "ANOT_MIN_ANGLE		= %lg\n", gmtdefs.anot_min_angle);
	fprintf (fp, "ANOT_FONT		= %s\n", font_name[gmtdefs.anot_font]);
	fprintf (fp, "ANOT_FONT_SIZE		= %d\n", gmtdefs.anot_font_size);
	fprintf (fp, "ANOT_OFFSET		= %lg\n", gmtdefs.anot_offset);
	fprintf (fp, "BASEMAP_AXES		= %s\n", gmtdefs.basemap_axes);
	fprintf (fp, "BASEMAP_FRAME_RGB	= %d/%d/%d\n", gmtdefs.basemap_frame_rgb[0],
		gmtdefs.basemap_frame_rgb[1], gmtdefs.basemap_frame_rgb[2]);
	(gmtdefs.basemap_type) ? fprintf (fp, "BASEMAP_TYPE		= PLAIN\n") : fprintf (fp, "BASEMAP_TYPE		= FANCY\n");
	fprintf (fp, "COLOR_BACKGROUND	= %d/%d/%d\n", gmtdefs.background_rgb[0], gmtdefs.background_rgb[1], gmtdefs.background_rgb[2]);
	fprintf (fp, "COLOR_FOREGROUND	= %d/%d/%d\n", gmtdefs.foreground_rgb[0], gmtdefs.foreground_rgb[1], gmtdefs.foreground_rgb[2]);
	fprintf (fp, "COLOR_NAN		= %d/%d/%d\n", gmtdefs.nan_rgb[0], gmtdefs.nan_rgb[1], gmtdefs.nan_rgb[2]);
	fprintf (fp, "COLOR_IMAGE		= ");
	if (gmtdefs.color_image == 0)
		fprintf (fp, "ADOBE\n");
	else if (gmtdefs.color_image == 1)
		fprintf (fp, "TILES\n");
	else if (gmtdefs.color_image == 2)
		fprintf (fp, "SEPARATION\n");
	(gmtdefs.color_model) ? fprintf (fp, "COLOR_MODEL		= HSV\n") : fprintf (fp, "COLOR_MODEL		= RGB\n");
	fprintf (fp, "D_FORMAT		= %s\n", gmtdefs.d_format);
	fprintf (fp, "DEGREE_FORMAT		= %d\n", gmtdefs.degree_format);
	fprintf (fp, "DOTS_PR_INCH		= %d\n", gmtdefs.dpi);
	fprintf (fp, "ELLIPSOID		= %s\n", gmtdefs.ellipse[gmtdefs.ellipsoid].name);
	if (use_points[0] == 2)
		fprintf (fp, "FRAME_PEN		= %.2lfp\n", pen_width[0]);
	else
		fprintf (fp, "FRAME_PEN		= %d\n", gmtdefs.frame_pen);
	fprintf (fp, "FRAME_WIDTH		= %lg\n", gmtdefs.frame_width);
	fprintf (fp, "GLOBAL_X_SCALE		= %lg\n", gmtdefs.global_x_scale);
	fprintf (fp, "GLOBAL_Y_SCALE		= %lg\n", gmtdefs.global_y_scale);
	fprintf (fp, "GRID_CROSS_SIZE		= %lg\n", gmtdefs.grid_cross_size);
	if (use_points[1] == 2)
		fprintf (fp, "GRID_PEN		= %.2lfp\n", pen_width[1]);
	else
		fprintf (fp, "GRID_PEN		= %d\n", gmtdefs.grid_pen);
	fprintf (fp, "HEADER_FONT		= %s\n", font_name[gmtdefs.header_font]);
	fprintf (fp, "HEADER_FONT_SIZE	= %d\n", gmtdefs.header_font_size);
	fprintf (fp, "HSV_MIN_SATURATION	= %lg\n", gmtdefs.hsv_min_saturation);
	fprintf (fp, "HSV_MAX_SATURATION	= %lg\n", gmtdefs.hsv_max_saturation);
	fprintf (fp, "HSV_MIN_VALUE		= %lg\n", gmtdefs.hsv_min_value);
	fprintf (fp, "HSV_MAX_VALUE		= %lg\n", gmtdefs.hsv_max_value);
	fprintf (fp, "INTERPOLANT		= ");
	if (gmtdefs.interpolant == 0)
		fprintf (fp, "LINEAR\n");
	else if (gmtdefs.interpolant == 1)
		fprintf (fp, "AKIMA\n");
	else if (gmtdefs.interpolant == 2)
		fprintf (fp, "CUBIC\n");
	(gmtdefs.io_header) ? fprintf (fp, "IO_HEADER		= TRUE\n") : fprintf (fp, "IO_HEADER		= FALSE\n");
	fprintf (fp, "N_HEADER_RECS		= %d\n", gmtdefs.n_header_recs);
	fprintf (fp, "LABEL_FONT		= %s\n", font_name[gmtdefs.label_font]);
	fprintf (fp, "LABEL_FONT_SIZE		= %d\n", gmtdefs.label_font_size);
	fprintf (fp, "LINE_STEP		= %lg\n", gmtdefs.line_step);
	fprintf (fp, "MAP_SCALE_FACTOR	= %lg\n", gmtdefs.map_scale_factor);
	fprintf (fp, "MEASURE_UNIT		= %s\n", gmt_unit_names[gmtdefs.measure_unit]);
	fprintf (fp, "N_COPIES		= %d\n", gmtdefs.n_copies);
	fprintf (fp, "OBLIQUE_ANOTATION	= %d\n", gmtdefs.oblique_anotation);
	fprintf (fp, "PAGE_COLOR		= %d/%d/%d\n", gmtdefs.page_rgb[0], gmtdefs.page_rgb[1], gmtdefs.page_rgb[2]);
	(gmtdefs.page_orientation & 1) ? fprintf (fp, "PAGE_ORIENTATION	= PORTRAIT\n") : fprintf (fp, "PAGE_ORIENTATION	= LANDSCAPE\n");
	fprintf (fp, "PAPER_WIDTH		= %lg\n", gmtdefs.paper_width);
	(gmtdefs.ps_heximage) ? fprintf (fp, "PSIMAGE_FORMAT		= HEX\n") : fprintf (fp, "PSIMAGE_FORMAT		= BIN\n");
	fprintf (fp, "TICK_LENGTH		= %lg\n", gmtdefs.tick_length);
	if (use_points[2] == 2)
		fprintf (fp, "TICK_PEN		= %.2lfp\n", pen_width[2]);
	else
		fprintf (fp, "TICK_PEN		= %d\n", gmtdefs.tick_pen);
	(gmtdefs.unix_time) ? fprintf (fp, "UNIX_TIME		= TRUE\n") : fprintf (fp, "UNIX_TIME		= FALSE\n");
	fprintf (fp, "UNIX_TIME_POS		= %lg/%lg\n", gmtdefs.unix_time_pos[0], gmtdefs.unix_time_pos[1]);
	fprintf (fp, "VECTOR_SHAPE		= %lg\n", gmtdefs.vector_shape);
	(gmtdefs.verbose) ? fprintf (fp, "VERBOSE			= TRUE\n") : fprintf (fp, "VERBOSE			= FALSE\n");
	(gmtdefs.want_euro_font) ? fprintf (fp, "WANT_EURO_FONT		= TRUE\n") : fprintf (fp, "WANT_EURO_FONT		= FALSE\n");
	fprintf (fp, "X_AXIS_LENGTH		= %lg\n", gmtdefs.x_axis_length);
	fprintf (fp, "Y_AXIS_LENGTH		= %lg\n", gmtdefs.y_axis_length);
	fprintf (fp, "X_ORIGIN		= %lg\n", gmtdefs.x_origin);
	fprintf (fp, "Y_ORIGIN		= %lg\n", gmtdefs.y_origin);
	(gmtdefs.xy_toggle) ? fprintf (fp, "XY_TOGGLE	= TRUE\n") : fprintf (fp, "XY_TOGGLE		= FALSE\n");
	(gmtdefs.y_axis_type == 1) ? fprintf (fp, "Y_AXIS_TYPE		= VER_TEXT\n") : fprintf (fp, "Y_AXIS_TYPE		= HOR_TEXT\n");

	if (fp != stdout) fclose (fp);
	
	return (0);
}

int get_gmtdefaults (this_file)	/* Read user's .gmtdefaults file and initialize parameters */
char *this_file; {
	int i;
	char file[512], *homedir;
/*         static char home [] = "HOME";    */
/* Environmental variable "GMTDIR" must be set to point to the
 * directory where the file ".gmtdefaults" is stored (including drive letter)   */
   static char home [] = "GMTDIR";
	
	 /* Default is to draw AND annotate all sides */
	for (i = 0; i < 5; i++) frame_info.side[i] = 2;
	
	if (!this_file) {	/* Must figure out which file to use */
	
		/* First see if a .gmtdefaults file is present in the current directory */
	
		if (!access (".gmtdefaults", R_OK)) /* Use cwd */
			strcpy (file, ".gmtdefaults");
		else { /* Then look in home directory */
	
			if ((homedir = getenv (home)) == NULL) {
				fprintf (stderr, "GMT Warning: Could not determine home directory!\n");
				return;
			}
			sprintf (file, "%s/.gmtdefaults\0", homedir);
			if (access (file, R_OK)) {	/* No defaults file present, use GMT defaults*/
				if (gmtdefs.want_euro_font) gmtdefs.page_orientation += 2;
				if (gmtdefs.ps_heximage) gmtdefs.page_orientation += 4;
				return;
			}
		}
	}
	else
		strcpy (file, this_file);
	
	gmt_loaddefaults (file);
	
	/* dpi is ALWAYS in dots per inch, so adjust if CM or M is used as unit */
	
	if (gmtdefs.measure_unit == 0) gmtdefs.dpi /= 2.54;
	if (gmtdefs.measure_unit == 2) gmtdefs.dpi /= 0.0254;
}

int gmt_hash (v)
char *v; {
	int h;
	for (h = 0; *v != '\0'; v++) h = (64 * h + (*v)) % HASH_SIZE;
	return (h);
}

int gmt_hash_entry (key, hashnode, n)
char *key;
struct HASH hashnode[];
int n; {
	int i;
	struct HASH *this;
	
	i = gmt_hash (key);
	
	if (i >= n || i < 0 || !hashnode[i].next) return (-1);	/* Bad key */
	this = hashnode[i].next;
	while (this && strcmp (this->key, key)) this = this->next;
	
	return ((this) ? this->id : -1);
}

int init_gmt_hash (hashnode, keys, n_hash, n_keys)
struct HASH hashnode[];
char *keys[];
int n_hash, n_keys; {
	int i, entry;
	struct HASH *this;

	/* Set up hash table */
	
	for (i = 0; i < n_hash; i++) hashnode[i].next = (struct HASH *)0;
	for (i = 0; i < n_keys; i++) {
		entry = gmt_hash (keys[i]);
		this = &hashnode[entry];
		while (this->next) this = this->next;
		this->next = (struct HASH *)memory (CNULL, 1, sizeof (struct HASH), gmt_program);
		this->next->key = keys[i];
		this->next->id = i;
	}
}

int get_ellipse (name)
char *name; {
	int i;
	
	for (i = 0; i < N_ELLIPSOIDS && strcmp (name, gmtdefs.ellipse[i].name); i++);
	if (i == N_ELLIPSOIDS) {	/* Try to open as file */
		FILE *fp;
		char line[512];
			
		if ((fp = fopen (name, "r")) == NULL)
			i = 0;	/* Default is GRS-80 */
		else {	/* Found file, now get parameters */
			i = N_ELLIPSOIDS - 1;
			fgets (line, 512, fp);
			fclose (fp);
			sscanf (line, "%s %d %lf %lf %lf", gmtdefs.ellipse[i].name, 
				&gmtdefs.ellipse[i].date, &gmtdefs.ellipse[i].eq_radius,
				&gmtdefs.ellipse[i].pol_radius, &gmtdefs.ellipse[i].flattening);
		}
	}
			
	return (i);
}

int get_entry (name, list, n)
char *name, *list[];
int n; {
	int i;
	
	for (i = 0; i < n && strcmp (name, list[i]); i++);
	return (i);
}

int gmt_begin (argc, argv)
int argc;
char **argv; {
	/* gmt_begin will merge the command line arguments with the arguments
	 * that were used the last time this programs was called (if any).  This
	 * way one only have to give -R -J to repeat previous map projection instead
	 * of spelling out the wesn and projection parameters every time.
	 * The information is stored in the first line of the file .gmtcommands
	 * in the current directory [or optionally a provided filename] and will
	 * contain the last arguments to the common parameters like
	 * -B, -H, -J, -K, -O, -P, -R, -U, -V, -X, -Y, -c
	 * Since the meaning of -X/-Y depends on weather we have an overlay or not
	 * we maintain -X -Y for absolute shifts and -x -y for relative shifts.
	 * If the argument +file is encountered then file is used in lieu of the
	 * usual .gmtdefaults file and this argument is chopped from argv
	 */
       
	int found = FALSE, need_xy = FALSE, overlay = FALSE;
	int i, j;
	char line[512], *this = CNULL;
	FILE *fp = NULL;

	/* Initialize parameters */
	
	mknan (gmt_NaN);
	oldargc = 0;
	frame_info.plot = FALSE;
	project_info.projection = -1;
	project_info.gave_map_width = FALSE;
	project_info.region = TRUE;
	project_info.compute_scale[0] =  project_info.compute_scale[1] = project_info.compute_scale[2] = FALSE;
	project_info.x_off_supplied = project_info.y_off_supplied = FALSE;
	project_info.region_supplied = FALSE;
	for (j = 0; j < 10; j++) project_info.pars[j] = 0.0;
	project_info.xmin = project_info.ymin = 0.0;
	project_info.z_level = MAXDOUBLE;	/* Will be set in map_setup */
	project_info.xyz_pos[0] = project_info.xyz_pos[1] = TRUE;
	prepare_three_D ();
	gmtdefs.dlon = (project_info.e - project_info.w) / gmtdefs.n_lon_nodes;
	gmtdefs.dlat = (project_info.n - project_info.s) / gmtdefs.n_lat_nodes;
	for (i = 0; i < 4; i++) project_info.edge[i] = TRUE;
	grdio_init ();
	for (i = 0; i < N_UNIQUE; i++) oldargv[i] = CNULL;
	gmt_program = argv[0];
	gmt_input = ascii_input;
	gmt_output = ascii_output;
	grd_in_nan_value = grd_out_nan_value = gmt_NaN;
	use_points[0] = use_points[1] = use_points[2] = 0;
	
	/* Set the gmtdefault parameters from the $HOME/.gmtdefaults (if any) */
	
        /* See if user specified +optional_defaults_file */
        
	for (i = j = 1; i < argc; i++) {
		argv[j] = argv[i];
		if (argv[j][0] == '+' && argv[i][1])
			this = &argv[i][1];
		else
			j++;
	}
	argc = j;

	get_gmtdefaults(this);
	
	/* Open .gmtcommands file and retrive old argv (if any) */

	if ((fp = fopen (".gmtcommands", "r")) == NULL) return (argc);     /* No .gmtcommands file in current directory */
      
	/* Get the common arguments and copy them to array oldargv */
      
	while (fgets (line, 512, fp)) {
		if (line[0] == '#') continue;	/* Skip comments */
		if (line[0] != '-') continue;	/* Possibly reading old .gmtcommands format */
		line[strlen(line)-1] = 0;
		oldargv[oldargc] = (char *) memory (CNULL, strlen (line) + 1, 1, "GMT");
		strcpy (oldargv[oldargc], line);
		oldargc++;
		if (oldargc > N_UNIQUE) {
			fprintf (stderr, "GMT Fatal Error: Failed while decoding common arguments\n");
			exit (-1);
		}
	}
	fclose (fp);
                    
	/* See if (1) We need abs/rel shift and (2) if we have an overlay */
      
	for (i = 1; i < argc; i++) {
		if (argv[i][0] != '-') continue;
		if (argv[i][1] == 'X' || argv[i][1] == 'Y' || argv[i][1] == 'x' || argv[i][1] == 'y') need_xy = TRUE;
		if (argv[i][1] == 'O' || argv[i][1] == 'o') overlay = TRUE;
	}
      
	overlay = (need_xy && overlay); /* -O means overlay only if -X or -Y has been specified */
      
	/* Change x/y to upper case if no overlay and change X/Y to lower case if overlay */
 
	for (i = 1; i < argc; i++) {
		if (argv[i][0] != '-') continue;
		if (overlay) {
			if (argv[i][1] == 'X') argv[i][1] = 'x';
			if (argv[i][1] == 'Y') argv[i][1] = 'y';
		}
		else {
			if (argv[i][1] == 'x') argv[i][1] = 'X';
			if (argv[i][1] == 'y') argv[i][1] = 'Y';
		}
	}

	/* Loop over argv and merge argv and oldargv */
      
	for (i = 1; i < argc; i++) {

		/* Skip if filename or option has modifiers */
              
		if (argv[i][0] != '-') continue;
		if (argv[i][1] != 'J' && argv[i][2] != 0) continue;
		if (argv[i][1] == 'J' && argv[i][3] != 0) continue;

		for (j = 0, found = FALSE; !found && j < oldargc; j++) {
			if (argv[i][1] == 'J')
				found = (oldargv[j][1] == argv[i][1] && oldargv[j][2] == argv[i][2]);
			else
				found = (oldargv[j][1] == argv[i][1]);
		}
		j--;
                              
		/* Skip if not found or oldargv has no modifiers */
              
		if (!found) continue;
		if (argv[i][1] != 'J' && oldargv[j][2] == 0) continue;
		if (argv[i][1] == 'J' && oldargv[j][3] == 0) continue;
      
		/* Here, oldargv has modifiers and argv dont, set pointer */
              
		argv[i] = oldargv[j];
	}
	return (argc);
}
      
int gmt_end (argc, argv)
int argc;
char **argv; {
	/* gmt_end will update (or create) the specified projectfile OR .gmtcommands
	 * file and write out the common parameters.
	 */
       
	int i, j, k, found_new, found_old;
	char hfile[512];
	FILE *fp_new = NULL;

	hfile[0] = 0;
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '+' && argv[i][1]) strcpy (hfile, &argv[i][1]);	/* Alternate commands file */
	}
		
	if (hfile[0] == 0) strcpy (hfile, ".gmtcommands");

	if ((fp_new = fopen (hfile, "w")) == NULL) {
		fprintf (stderr, "GMT Fatal Error: Could not update .gmtcommands file\n");
		exit (-1);
	}
	
	fprintf (fp_new, "# GMT common arguments shelf\n");

	for (i = 0; i < N_UNIQUE; i++) {        /* Loop over unique_option parameters */

		/* First see if an updated value exist for this common parameter */

		for (j = 1, found_new = FALSE; !found_new && j < argc; j++) {
			if (unique_option[i][0] == 'J') /* Range of -J? options */
				found_new = !strncmp (&argv[j][1], unique_option[i], 2);
			else
				found_new = (argv[j][1] == unique_option[i][0]);
		}

		if (found_new) { /* Need to store this updated value */
			fprintf (fp_new, "%s\n", argv[j-1]);
		}
		else  {	 	/* Need to find and store the old value if any */
			for (k = 0, found_old = FALSE; !found_old && k < oldargc; k++) {
				if (unique_option[i][0] == 'J') /* Range of -J? options */
					found_old = !strncmp (&oldargv[k][1], unique_option[i], 2);
				else
					found_old = (oldargv[k][1] == unique_option[i][0]);
			}

			if (found_old)  /* Keep old value */
				fprintf (fp_new, "%s\n", oldargv[k-1]);
		}
	}
	fclose (fp_new);

	for (i = 0; i < N_UNIQUE; i++) if (oldargv[i]) free (oldargv[i]);
	if (gmt_lut) free ((char *)gmt_lut);
	free_plot_array ();

	exit (0);
}

int map_getframe (args)
char *args; {
	/* map_getframe scans an argument string and extract parameters that
	 * set the interval for  tickmars and anotations on the boundary.
	 * The string must be continous, i.e. no whitespace must be present
	 * The string may have 1, 2,  or 3 parts, separated by a slash '/'. All
	 * info after the first slash are assigned to the y-axis.  Info after
	 * the second slash are assigned to the z-axis.  If there is no
	 * slash, x-values are copied to y-values.
	 * A substring looks like [t][value][m|c]. The [t] and [m|c] are optional
	 * ([ and ] are NOT part of the string and are just used to clarify)
	 * [t] can be any of [a](anotation int), [f](frame int), or [g](gridline int).
	 * Default is a AND f. The [m], if present indicates value is in minutes.
	 * The [c], if present indicates value is in seConds (s already used for south...).
	 * Text between : and : are labels for the respective axes. If the first
	 * character of the text is a period, then the rest of the text is used
	 * as the plot title.
	 * For LINEAR axes: If the first characters in args are one or more of w,e,s,n
	 * only those axes will be drawn. Upper case letters means the chosen axes
	 * also will be annotated. Default is all 4 axes drawn/annotated.
	 * For logscale plots:  l will cause log10(x) to be plotted
	 *			p will cause 10 ^ log10(x) to be plotted
	 *	anot/tick/grid interval can here be either:
	 *		1.0	-> Only powers of 10 are anotated
	 *		2.0	-> powers of 10 times (1, 2, 5) are anotated
	 *		3.0	-> powers of 10 times (1,2,3,..9) are anotated
	 */
	 
	int i, i1, i2, xyz_side = 0, n_colons = 0, side[5];
	BOOLEAN xyz_set[3], error = FALSE, set_sides = FALSE;
	double value;
	char flag;
	
	for (i = 0; i < 3; i++) {
		frame_info.anot_int[i] = 0.0;
		frame_info.frame_int[i] = 0.0;
		frame_info.grid_int[i] = 0.0;
		frame_info.label[i][0] = 0;
		frame_info.anot_type[i] = 0;
		xyz_set[i] = FALSE;
	}
	frame_info.plot = TRUE;
	frame_info.draw_box = FALSE;
	frame_info.header[0] = 0;
	for (i = 0; i < 5; i++) side[i] = 0;
	
	i1 = 0;
	while (args[i1]) {
		if (args[i1] == '/') {
			xyz_side++;
			i1++;
		}
		flag = '*';	/* If a,f, or g is not appended, we set all to the same value */
		if (isalpha(args[i1])) {
			flag = args[i1];
			i1++;
		}
		if (flag == 'w' || flag == 'W') {
			side[3] = (flag == 'W') ? 2 : 1;
			set_sides = TRUE;
			continue;
		}
		if (flag == 'e' || flag == 'E') {
			side[1] = (flag == 'E') ? 2 : 1;
			set_sides = TRUE;
			continue;
		}
		if (flag == 's' || flag == 'S') {
			side[0] = (flag == 'S') ? 2 : 1;
			set_sides = TRUE;
			continue;
		}
		if (flag == 'n' || flag == 'N') {
			side[2] = (flag == 'N') ? 2 : 1;
			set_sides = TRUE;
			continue;
		}
		if (flag == 'z' || flag == 'Z') {
			side[4] = (flag == 'Z') ? 2 : 1;
			if (args[i1] == '+') {
				frame_info.draw_box = TRUE;
				i1++;
			}
			set_sides = TRUE;
			continue;
		}
		if (isupper((int)flag)) flag = tolower ((int)flag);
		if (flag == 'l') {	/* log10(val) anotations */
			frame_info.anot_type[xyz_side] = 1;
			continue;
		}
		if (flag == 'p') {	/* Powers of 10 anotations */
			frame_info.anot_type[xyz_side] = 2;
			continue;
		}
		if (args[i1] == ':') {	/* Label string */
			n_colons++;
			flag = ' ';
			i2 = ++i1;
			while (args[i2] && args[i2] != ':') i2++;
			if (args[i2] == ':') n_colons++;
			if (args[i1] == '.') {
				i1++;
				strncpy (frame_info.header, &args[i1], i2-i1);
				frame_info.header[i2-i1] = 0;
			}
			else {
				strncpy (frame_info.label[xyz_side], &args[i1], i2-i1);
				frame_info.label[xyz_side][i2-i1] = 0;
			}
			i1 = i2 + 1;
		}
		i2 = i1;
		while (args[i2] && strchr ("WESNacefglmnpswz/:", (int)args[i2]) == NULL) i2++;
		if (flag != ' ') {	/* Decode the numerical value */
			value = atof (&args[i1]);
			if (args[i2] == 'm') {
				value /= 60.0;
				i2++;
			}
			else if (args[i2] == 'c') {
				value /= 3600.0;
				i2++;
			}
			xyz_set[xyz_side] = TRUE;
		}
		if (flag == '*')
			frame_info.anot_int[xyz_side] = frame_info.frame_int[xyz_side] = value;
		else if (flag == 'a')
			frame_info.anot_int[xyz_side] = value;
		else if (flag == 'f')
			frame_info.frame_int[xyz_side] = value;
		else if (flag == 'g')
			frame_info.grid_int[xyz_side] = value;
		i1 = i2;
	}
	if (!xyz_set[1]) frame_info.anot_type[1] = frame_info.anot_type[0];
	for (i = 0; i < 3; i++) {	/* Set both a and f if only one of them is set */
		if (frame_info.anot_int[i] != 0.0 && frame_info.frame_int[i] == 0.0)
			frame_info.frame_int[i] = frame_info.anot_int[i];
		else if (frame_info.frame_int[i] != 0.0 && frame_info.anot_int[i] == 0.0)
			frame_info.anot_int[i] = frame_info.frame_int[i];
	}
	if (!xyz_set[1]) {	/* No separate y-axis info given, copy x values */
		frame_info.anot_int[1] = frame_info.anot_int[0];
		frame_info.frame_int[1] = frame_info.frame_int[0];
		frame_info.grid_int[1] = frame_info.grid_int[0];
	}
	if (set_sides) for (i = 0; i < 5; i++) frame_info.side[i] = side[i];	/* Override default */

	if (n_colons%2) error++;
	
	return (error);
}

int map_getproject (args)
char *args; {
	/* map_getproject scans the arguments given and extracts the parameters needed
	 * for the specified map projection. These parameters are passed through the
	 * project_info structure.  The function returns TRUE if an error is encountered.
	 */
	 
	int j, k, n, slash, l_pos[2], p_pos[2], id, project = -1, n_slashes = 0;
	BOOLEAN error = FALSE, skip = FALSE;
	double o_x, o_y, b_x, b_y, c, az;
	char type, txt_a[32], txt_b[32], txt_c[32], txt_d[32];
	
	l_pos[0] = l_pos[1] = p_pos[0] = p_pos[1] = 0;
	type = args[0];

	if (strchr ("AaBbCcEeGgHhIiKkLlMmNnOoPpQqRrSsTtUuWwXxYyZz", type) == NULL) return (TRUE);	/* NO valid projection specified */
	args++;

	for (j = 0; args[j]; j++) if (args[j] == '/') n_slashes++;
	 
	if (!(type == 'z' || type == 'Z')) {
		/* Check to see if scale is specified in 1:xxxx */
		for (j = strlen (args), k = -1; j > 0 && k < 0 && args[j] != '/'; j--) if (args[j] == ':') k = j + 1; 
		project_info.units_pr_degree = (k == -1) ? TRUE : FALSE;
		project_info.unit = gmt_units[gmtdefs.measure_unit];
	}
	 
	 switch (type) {
	 	case 'X':
	 		project_info.compute_scale[0] = project_info.compute_scale[1] = TRUE;
	 		if (args[0] == 'v') {
	 			project_info.pars[0] = gmtdefs.y_axis_length;
	 			project_info.pars[1] = gmtdefs.x_axis_length;
	 			skip = TRUE;
	 		}
	 		else if (args[0] == 'h') {
	 			project_info.pars[0] = gmtdefs.x_axis_length;
	 			project_info.pars[1] = gmtdefs.y_axis_length;
	 			skip = TRUE;
	 		}
	 		
	 	case 'x':		/* Linear x/y scaling */
	 		project_info.pars[4] = (strchr (args, 'd')) ? 1.0 : 0.0;	/* TRUE if input is degrees and d is appended */
			error = (n_slashes > 1);
	 		
	 		/* Find occurrences of /, l, or p */
	 		for (j = 0, slash = 0; args[j] && slash == 0; j++) if (args[j] == '/') slash = j;
	 		for (j = id = 0; args[j]; j++) {
	 			if (args[j] == '/') id = 1;
	 			if (args[j] == 'L' || args[j] == 'l') l_pos[id] = j;
	 			if (args[j] == 'P' || args[j] == 'p') p_pos[id] = j;
	 		}
	 		
	 		/* Get x-arguments */
	 		
	 		if (!skip) project_info.pars[0] = atof (args);	/* x-scale */
	 		
	 		if (l_pos[0] > 0)
	 			project_info.xyz_projection[0] = LOG10;
	 		else if (p_pos[0] > 0) {
	 			project_info.xyz_projection[0] = POW;
	 			project_info.pars[2] = atof (&args[p_pos[0]+1]);	/* pow to raise x */
	 		}
	 		
	 		if (slash) {	/* Separate y-scaling desired */
	 			if (!skip) project_info.pars[1] = atof (&args[slash+1]);	/* y-scale */
	 			
	 			if (l_pos[1] > 0)
	 				project_info.xyz_projection[1] = LOG10;
	 			else if (p_pos[1] > 0) {
	 				project_info.xyz_projection[1] = POW;
	 				project_info.pars[3] = atof (&args[p_pos[1]+1]);	/* pow to raise y */
	 			}
	 		}
	 		else {	/* Just copy x parameters */
	 			project_info.xyz_projection[1] = project_info.xyz_projection[0];
	 			if (!skip) project_info.pars[1] = project_info.pars[0];
	 			project_info.pars[3] = project_info.pars[2];
	 		}
	 		project = LINEAR;
	 		if (project_info.pars[0] == 0.0 || project_info.pars[1] == 0.0) error = TRUE;
	 		break;
	 	case 'Z':
			project_info.compute_scale[2] = TRUE;
	 	case 'z':
	 		
			error = (n_slashes > 0);

	 		/* Find occurrences of l or p */
	 		for (j = 0; args[j]; j++) if (args[j] == 'L' || args[j] == 'l') l_pos[0] = j;
	 		for (j = 0; args[j]; j++) if (args[j] == 'P' || args[j] == 'p') p_pos[0] = j;
	 		
	 		/* Get arguments */
	 		
	 		project_info.z_pars[0] = atof (args);	/* z-scale */
	 		
	 		if (l_pos[0] > 0)
	 			project_info.xyz_projection[2] = LOG10;
	 		else if (p_pos[0] > 0) {
	 			project_info.xyz_projection[2] = POW;
	 			project_info.z_pars[1] = atof (&args[p_pos[0]+1]);	/* pow to raise z */
	 		}
	 		if (project_info.z_pars[0] == 0.0) error = TRUE;
	 		break;
	 	case 'P':		/* Polar (theta,r) */
	 		project_info.gave_map_width = TRUE;
	 	case 'p':
	 		n = sscanf (args, "%lf", &project_info.pars[0]);
			error = (n_slashes > 0 || n != 1 || project_info.pars[0] <= 0.0);
	 		project = POLAR;
	 		break;
	 		
	 	/* Map projections */

	 	case 'A':	/* Lambert Azimuthal Equal-Area */
	 		project_info.gave_map_width = TRUE;
	 	case 'a':
	 		if (k >= 0) {	/* Scale entered as 1:mmmmm */
	 			n = sscanf (args, "%[^/]/%[^/]/1:%lf", txt_a, txt_b, &project_info.pars[2]);
	 			if (project_info.pars[2] != 0.0) project_info.pars[2] = 1.0 / (project_info.pars[2] * project_info.unit);
				error = (!(n_slashes == 2 && n == 3));
	 		}
	 		else if (project_info.gave_map_width) {
	 			n = sscanf (args, "%[^/]/%[^/]/%lf", txt_a, txt_b, &project_info.pars[2]);
				error = (!(n_slashes == 2 && n == 3));
	 		}
	 		else {
	 			n = sscanf (args, "%[^/]/%[^/]/%lf/%s", txt_a, txt_b, &project_info.pars[2], txt_c);
				project_info.pars[3] = ddmmss_to_degree (txt_c);
				error = (!(n_slashes == 3 && n == 4));
	 		}
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			project_info.pars[1] = ddmmss_to_degree (txt_b);
			error += (project_info.pars[2] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = LAMB_AZ_EQ;
	 		break;
	 	case 'B':		/* Albers Equal-area Conic */
	 		project_info.gave_map_width = TRUE;
	 	case 'b':
	 		if (k >= 0) {	/* Scale entered as 1:mmmmm */
	 			n = sscanf (args, "%[^/]/%[^/]/%[^/]/%[^/]/1:%lf", txt_a, txt_b, txt_c, txt_d, &project_info.pars[4]);
	 			if (project_info.pars[4] != 0.0) project_info.pars[4] = 1.0 / (project_info.pars[4] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%[^/]/%[^/]/%[^/]/%lf", txt_a, txt_b, txt_c, txt_d, &project_info.pars[4]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			project_info.pars[1] = ddmmss_to_degree (txt_b);
			project_info.pars[2] = ddmmss_to_degree (txt_c);
			project_info.pars[3] = ddmmss_to_degree (txt_d);
			error = !(n_slashes == 4 && n == 5);
			error += (project_info.pars[4] <= 0.0 || project_info.pars[2] == project_info.pars[3]);
			error += (k >= 0 && project_info.gave_map_width);
	 		project = ALBERS;
	 		break;

	 	case 'C':	/* Cassini */
	 		project_info.gave_map_width = TRUE;
	 	case 'c':
	 		if (k >= 0) {	/* Scale entered as 1:mmmmm */
	 			n = sscanf (args, "%[^/]/%[^/]/1:%lf", txt_a, txt_b, &project_info.pars[2]);
	 			if (project_info.pars[2] != 0.0) project_info.pars[2] = 1.0 / (project_info.pars[2] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%[^/]/%lf", txt_a, txt_b, &project_info.pars[2]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			project_info.pars[1] = ddmmss_to_degree (txt_b);
			error = !(n_slashes == 2 && n == 3);
			error += (project_info.pars[2] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = CASSINI;
	 		break;

	 	case 'E':		/* Azimuthal equal-distant */
	 		project_info.gave_map_width = TRUE;
	 	case 'e':
	 		if (k >= 0) {	/* Scale entered as 1:mmmmm */
	 			n = sscanf (args, "%[^/]/%[^/]/1:%lf", txt_a, txt_b, &project_info.pars[2]);
	 			if (project_info.pars[2] != 0.0) project_info.pars[2] = 1.0 / (project_info.pars[2] * project_info.unit);
				error = (!(n_slashes == 2 && n == 3));
	 		}
	 		else if (project_info.gave_map_width) {
		 		n = sscanf (args, "%[^/]/%[^/]/%lf", txt_a, txt_b, &project_info.pars[2]);
				error = (!(n_slashes == 2 && n == 3));
	 		}
	 		else {
	 			n = sscanf (args, "%[^/]/%[^/]/%lf/%s", txt_a, txt_b, &project_info.pars[2], txt_c);
				project_info.pars[3] = ddmmss_to_degree (txt_c);
				error = (!(n_slashes == 3 && n == 4));
	 		}
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			project_info.pars[1] = ddmmss_to_degree (txt_b);
			error += (project_info.pars[2] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = AZ_EQDIST;
	 		break;

	 	case 'G':		/* Orthographic */
	 		project_info.gave_map_width = TRUE;
	 	case 'g':		/* Orthographic */
	 		if (k >= 0) {	/* Scale entered as 1:mmmmm */
	 			n = sscanf (args, "%[^/]/%[^/]/1:%lf", txt_a, txt_b, &project_info.pars[2]);
	 			if (project_info.pars[2] != 0.0) project_info.pars[2] = 1.0 / (project_info.pars[2] * project_info.unit);
				error = (!(n_slashes == 2 && n == 3));
	 		}
	 		else if (project_info.gave_map_width) {
		 		n = sscanf (args, "%[^/]/%[^/]/%lf", txt_a, txt_b, &project_info.pars[2]);
				error = (!(n_slashes == 2 && n == 3));
			}
	 		else {
	 			n = sscanf (args, "%[^/]/%[^/]/%lf/%s", txt_a, txt_b, &project_info.pars[2], txt_c);
				project_info.pars[3] = ddmmss_to_degree (txt_c);
				error = (!(n_slashes == 3 && n == 4));
			}
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			project_info.pars[1] = ddmmss_to_degree (txt_b);
			error += (project_info.pars[2] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = ORTHO;
	 		break;

	 	case 'H':	/* Hammer-Aitoff Equal-Area */
	 		project_info.gave_map_width = TRUE;
	 	case 'h':
	 		if (k >= 0) {
	 			n = sscanf (args, "%[^/]/1:%lf", txt_a, &project_info.pars[1]);
	 			if (project_info.pars[1] != 0.0) project_info.pars[1] = gmtdefs.ellipse[N_ELLIPSOIDS-1].eq_radius / (project_info.pars[1] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%lf", txt_a, &project_info.pars[1]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			error = !(n_slashes == 1 && n == 2);
			error += (project_info.pars[1] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = HAMMER;
	 		break;

	 	case 'I':	/* Sinusoidal Equal-Area */
	 		project_info.gave_map_width = TRUE;
	 	case 'i':
	 		if (k >= 0) {
	 			n = sscanf (args, "%[^/]/1:%lf", txt_a, &project_info.pars[1]);
	 			if (project_info.pars[1] != 0.0) project_info.pars[1] = 2.0 * M_PI * gmtdefs.ellipse[N_ELLIPSOIDS-1].eq_radius / (project_info.pars[1] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%lf", txt_a, &project_info.pars[1]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			error = !(n_slashes == 1 && n == 2);
			error += (project_info.pars[1] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = SINUSOIDAL;
	 		break;

	 	case 'K':	/* Eckert VI projection */
	 		project_info.gave_map_width = TRUE;
	 	case 'k':
	 		if (k >= 0) {
	 			n = sscanf (args, "%[^/]/1:%lf", txt_a, &project_info.pars[1]);
	 			if (project_info.pars[1] != 0.0) project_info.pars[1] = gmtdefs.ellipse[N_ELLIPSOIDS-1].eq_radius / (project_info.pars[1] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%lf", txt_a, &project_info.pars[1]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			error = !(n_slashes == 1 && n == 2);
			error += (project_info.pars[1] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = ECKERT;
	 		break;

	 	case 'L':		/* Lambert Conformal Conic */
	 		project_info.gave_map_width = TRUE;
	 	case 'l':
	 		if (k >= 0) {	/* Scale entered as 1:mmmmm */
	 			n = sscanf (args, "%[^/]/%[^/]/%[^/]/%[^/]/1:%lf", txt_a, txt_b, txt_c, txt_d, &project_info.pars[4]);
	 			if (project_info.pars[4] != 0.0) project_info.pars[4] = 1.0 / (project_info.pars[4] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%[^/]/%[^/]/%[^/]/%lf", txt_a, txt_b, txt_c, txt_d, &project_info.pars[4]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			project_info.pars[1] = ddmmss_to_degree (txt_b);
			project_info.pars[2] = ddmmss_to_degree (txt_c);
			project_info.pars[3] = ddmmss_to_degree (txt_d);
			error = !(n_slashes == 4 && n == 5);
			error += (project_info.pars[4] <= 0.0 || project_info.pars[2] == project_info.pars[3]);
			error += (k >= 0 && project_info.gave_map_width);
	 		project = LAMBERT;
	 		break;
	 	case 'M':		/* Mercator */
	 		project_info.gave_map_width = TRUE;
	 	case 'm':
	 		if (k >= 0) {	/* Scale entered as 1:mmmmm */
	 			n = sscanf (args, "1:%lf", &project_info.pars[0]);
	 			if (project_info.pars[0] != 0.0) project_info.pars[0] = 1.0 / (project_info.pars[0] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%lf", &project_info.pars[0]);
			error = (n_slashes > 0 || n != 1 || project_info.pars[0] <= 0.0);
	 		project = MERCATOR;
	 		break;

	 	case 'N':	/* Robinson Projection */
	 		project_info.gave_map_width = TRUE;
	 	case 'n':
	 		if (k >= 0) {
	 			n = sscanf (args, "%[^/]/1:%lf", txt_a, &project_info.pars[1]);
	 			if (project_info.pars[1] != 0.0) project_info.pars[1] = gmtdefs.ellipse[N_ELLIPSOIDS-1].eq_radius / (project_info.pars[1] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%lf", txt_a, &project_info.pars[1]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			error = !(n_slashes == 1 && n == 2);
			error += (project_info.pars[1] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = ROBINSON;
	 		break;

	 	case 'O':		/* Oblique Mercator */
	 		project_info.gave_map_width = TRUE;
	 	case 'o':
	 		if (args[0] == 'a') {	/* Origin and azimuth specified */
	 			if (k >= 0) {
					n = sscanf (&args[1], "%[^/]/%[^/]/%lf/1:%lf", txt_a, txt_b, &az, &project_info.pars[4]);
	 				if (project_info.pars[4] != 0.0) project_info.pars[4] = 1.0 / (project_info.pars[4] * project_info.unit);
	 			}
	 			else
					n = sscanf (&args[1], "%[^/]/%[^/]/%lf/%lf", txt_a, txt_b, &az, &project_info.pars[4]);

				o_x = ddmmss_to_degree (txt_a);
				o_y = ddmmss_to_degree (txt_b);
	 			c = 10.0;	/* compute point 10 degrees from origin along azimuth */
	 			b_x = o_x + R2D * atan (sind (c) * sind (az) / (cosd (o_y) * cosd (c) - sind (o_y) * sind (c) * cosd (az)));
	 			b_y = R2D * d_asin (sind (o_y) * cosd (c) + cosd (o_y) * sind (c) * cosd (az));
	 			project_info.pars[6] = 0.0;
				error = !(n_slashes == 3 && n == 4);
	 		}
	 		else if (args[0] == 'b') {	/* Origin and second point specified */
	 			if (k >= 0) {
	 				n = sscanf (&args[1], "%[^/]/%[^/]/%[^/]/%[^/]/1:%lf", txt_a, txt_b, txt_c, txt_d, &project_info.pars[4]);
	 				if (project_info.pars[4] != 0.0) project_info.pars[4] = 1.0 / (project_info.pars[4] * project_info.unit);
	 			}
	 			else
	 				n = sscanf (&args[1], "%[^/]/%[^/]/%[^/]/%[^/]/%lf", txt_a, txt_b, txt_c, txt_d, &project_info.pars[4]);
				o_x = ddmmss_to_degree (txt_a);
				o_y = ddmmss_to_degree (txt_b);
				b_x = ddmmss_to_degree (txt_c);
				b_y = ddmmss_to_degree (txt_d);
	 			project_info.pars[6] = 0.0;
				error = !(n_slashes == 4 && n == 5);
	 		}
	 		else if (args[0] == 'c') {	/* Origin and Pole specified */
	 			if (k >= 0) {
	 				n = sscanf (&args[1], "%[^/]/%[^/]/%[^/]/%[^/]/1:%lf", txt_a, txt_b, txt_c, txt_d, &project_info.pars[4]);
	 				if (project_info.pars[4] != 0.0) project_info.pars[4] = 1.0 / (project_info.pars[4] * project_info.unit);
	 			}
	 			else
	 				n = sscanf (&args[1], "%[^/]/%[^/]/%[^/]/%[^/]/%lf", txt_a, txt_b, txt_c, txt_d, &project_info.pars[4]);
				o_x = ddmmss_to_degree (txt_a);
				o_y = ddmmss_to_degree (txt_b);
				b_x = ddmmss_to_degree (txt_c);
				b_y = ddmmss_to_degree (txt_d);
	 			if (b_y < 0.0) {	/* Flip from S hemisphere to N */
	 				b_y = -b_y;
	 				b_x += 180.0;
	 				if (b_x >= 360.0) b_x -= 360.0;
	 			}
	 			project_info.pars[6] = 1.0;
				error = !(n_slashes == 4 && n == 5);
	 		}
	 		else
	 			project = -1;
			error += (project_info.pars[4] <= 0.0);
			error += (k >= 0 && project_info.gave_map_width);
	 		project_info.pars[0] = o_x;	project_info.pars[1] = o_y;
	 		project_info.pars[2] = b_x;	project_info.pars[3] = b_y;
	 		
	 		/* see if wesn is in oblique degrees or just diagonal corners */
	 		
	 		project = OBLIQUE_MERC;
	 		break;

	 	case 'Q':	/* Equidistant Cylindrical (Plate Carree) */
	 		project_info.gave_map_width = TRUE;
	 	case 'q':
	 		if (k >= 0) {
	 			n = sscanf (args, "%[^/]/1:%lf", txt_a, &project_info.pars[1]);
	 			if (project_info.pars[1] != 0.0) project_info.pars[1] = gmtdefs.ellipse[N_ELLIPSOIDS-1].eq_radius / (project_info.pars[1] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%lf", txt_a, &project_info.pars[1]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			error = !(n_slashes == 1 && n == 2);
			error += (project_info.pars[1] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = CYL_EQDIST;
	 		break;

	 	case 'R':	/* Winkel Tripel Modified azimuthal */
	 		project_info.gave_map_width = TRUE;
	 	case 'r':
	 		if (k >= 0) {
	 			n = sscanf (args, "%[^/]/1:%lf", txt_a, &project_info.pars[1]);
	 			if (project_info.pars[1] != 0.0) project_info.pars[1] = gmtdefs.ellipse[N_ELLIPSOIDS-1].eq_radius / (project_info.pars[1] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%lf", txt_a, &project_info.pars[1]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			error = !(n_slashes == 1 && n == 2);
			error += (project_info.pars[1] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = WINKEL;
	 		break;

	 	case 'S':		/* Stereographic */
	 		project_info.gave_map_width = TRUE;
	 	case 's':
	 		if (k >= 0) {	/* Scale entered as 1:mmmmm */
	 			n = sscanf (args, "%[^/]/%[^/]/1:%lf", txt_a, txt_b, &project_info.pars[2]);
	 			if (project_info.pars[2] != 0.0) project_info.pars[2] = 1.0 / (project_info.pars[2] * project_info.unit);
				error = (!(n_slashes == 2 && n == 3));
	 		}
	 		else if (project_info.gave_map_width) {
		 		n = sscanf (args, "%[^/]/%[^/]/%lf", txt_a, txt_b, &project_info.pars[2]);
				error = (!(n_slashes == 2 && n == 3));
	 		}
	 		else {
	 			n = sscanf (args, "%[^/]/%[^/]/%lf/%s", txt_a, txt_b, &project_info.pars[2], txt_c);
				project_info.pars[3] = ddmmss_to_degree (txt_c);
				error = (!(n_slashes == 3 && n == 4));
			}
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			project_info.pars[1] = ddmmss_to_degree (txt_b);
			error += (project_info.pars[2] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = STEREO;
	 		break;

	 	case 'T':	/* Transverse Mercator */
	 		project_info.gave_map_width = TRUE;
	 	case 't':
	 		if (k >= 0) {
	 			n = sscanf (args, "%[^/]/1:%lf", txt_a, &project_info.pars[1]);
	 			if (project_info.pars[1] != 0.0) project_info.pars[1] = 1.0 / (project_info.pars[1] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%lf", txt_a, &project_info.pars[1]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			error = !(n_slashes == 1 && n == 2);
			error += (project_info.pars[1] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = TM;
	 		break;

	 	case 'U':	/* Universal Transverse Mercator */
	 		project_info.gave_map_width = TRUE;
	 	case 'u':
	 		if (k >= 0) {
	 			n = sscanf (args, "%lf/1:%lf", &project_info.pars[0], &project_info.pars[1]);
	 			if (project_info.pars[1] != 0.0) project_info.pars[1] = 1.0 / (project_info.pars[1] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%lf/%lf", &project_info.pars[0], &project_info.pars[1]);
	 		project_info.north_pole = (project_info.pars[0] > 0.0);
	 		project_info.pars[0] = fabs (project_info.pars[0]);
			error = !(n_slashes == 1 && n == 2);
			error += (project_info.pars[1] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = UTM;
	 		break;
	 	case 'W':	/* Mollweide Equal-Area */
	 		project_info.gave_map_width = TRUE;
	 	case 'w':
	 		if (k >= 0) {
	 			n = sscanf (args, "%[^/]/1:%lf", txt_a, &project_info.pars[1]);
	 			if (project_info.pars[1] != 0.0) project_info.pars[1] = gmtdefs.ellipse[N_ELLIPSOIDS-1].eq_radius / (project_info.pars[1] * project_info.unit);
	 		}
	 		else
	 			n = sscanf (args, "%[^/]/%lf", txt_a, &project_info.pars[1]);
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			error = !(n_slashes == 1 && n == 2);
			error += (project_info.pars[1] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = MOLLWEIDE;
	 		break;
	 		
	 	case 'Y':		/* Cylindrical Equal Area */
	 		project_info.gave_map_width = TRUE;
	 	case 'y':
	 		if (k >= 0) {	/* Scale entered as 1:mmmmm */
	 			n = sscanf (args, "%[^/]/%[^/]/1:%lf", txt_a, txt_b, &project_info.pars[2]);
	 			if (project_info.pars[2] != 0.0) project_info.pars[2] = 1.0 / (project_info.pars[2] * project_info.unit);
				error = (!(n_slashes == 2 && n == 3));
	 		}
	 		else {
	 			n = sscanf (args, "%[^/]/%[^/]/%lf", txt_a, txt_b, &project_info.pars[2]);
				error = (!(n_slashes == 2 && n == 3));
			}
			project_info.pars[0] = ddmmss_to_degree (txt_a);
			project_info.pars[1] = ddmmss_to_degree (txt_b);
			error += (fabs(project_info.pars[1]) >= 90.0);
			error += (project_info.pars[2] <= 0.0 || (k >= 0 && project_info.gave_map_width));
	 		project = CYL_EQ;
	 		break;

	 	default:
	 		error = TRUE;
	 		project = -1;
	 		break;
	 }
	 
	 if (!project_info.units_pr_degree && project_info.gave_map_width) {
	 	fprintf (stderr, "%s: GMT SYNTAX ERROR -J%c option: Cannot specify map width with 1:xxxx format\n", gmt_program, type);
	 	error++;
	 }
	 
	 if (!(type == 'z' || type == 'Z')) project_info.projection = project;
	 return (error);
}

int prepare_three_D () {	/* Initialize 3-D parameters */
	project_info.z_pars[0] = project_info.z_pars[1] = 0.0;
	project_info.xyz_pos[2] = TRUE;
	project_info.zmin = project_info.zmax = 0.0;
	z_forward = (PFI) NULL;
	z_inverse = (PFI) NULL;
	z_project.view_azimuth = 180.0;
	z_project.view_elevation = 90.0;
	project_info.z_bottom = project_info.z_top = 0.0;
}

