
/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)gmt_plot.c	2.58  09 Aug 1995
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 *
 *			G M T _ P L O T . C
 *
 *- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * gmt_plot.c contains code related to plotting maps
 *
 * Author:	Paul Wessel
 * Date:	27-JUL-1992
 * Version:	2.1
 *
 *
 * Functions include:
 *
 *	basemap_3D :		Plots 3-D basemap
 *	color_image :		Calls the desired colorimage operator
 *	echo_command :		Puts the command line into the PostScript file as comments
 *	geoplot :		As ps_plot, but using lon/lat directly
 *	gmt_plot_line :		Same for polygons
 *	get_angle :		Sub function to get annotation angles
 *	get_anot_label :	Construct degree/minute label
 *	lambert_map_boundary :	Plot basemap for Lambert conformal conic projection
 *	linear_map_boundary :	Plot basemap for Linear projections
 *	logx_grid :		Draw log10 x grid lines
 *	logy_grid :		Draw log10 y grid lines
 *	prepare_label :		Gets angle and justification for frame anotations
 *	map_anotate :		Annotate basemaps
 *	map_basemap :		Generic basemap function
 *	map_boundary :		Draw the maps boundary
 *	map_clip_off :		Deactivate map region clip path
 *	map_clip_on :		Activate map region clip path
 *	map_gridcross :		Draw grid crosses on maps
 *	map_gridlines :		Draw gridlines on maps
 *	map_latline :		Draw a latitude line
 *	map_lattick :		Draw a latitude tick marck
 *	map_lonline :		Draw a longitude line
 *	map_lontick :		Draw a longitude tick mark
 *	map_symbol :		Plot map annotation
 *	map_symbol_ew :		  for east/west sides
 *	map_symbol_ns :		  for south/north sides
 *	map_tick :		Draw the ticks
 *	map_tickmarks :		Plot tickmarks on maps
 *	merc_map_boundary :	Plot basemap for Mercator projection
 *	ellipse_map_boundary :	Plot basemap for Mollweide and Hammer-Aitoff projections
 *	oblmrc_map_boundary :	Plot basemap for Oblique Mercator projection
 *	polar_map_boundary :	Plot basemap for Polar stereographic projection
 *	rect_map_boundary :	Plot plain basemap for projections with rectangular boundaries
 *	basic_map_boundary :	Plot plain basemap for most projections
 *	timestamp :		Plot UNIZ time stamp with optional string
 *	wesn_map_boundary :	Plot plain basemap for projections with geographical boundaries
 *	x_axis :		Draw x axis
 *	xyz_axis3D :		Draw 3-D axis
 *	y_axis :		Draw y axis
 *	polar_adjust :		Adjust label justification for polar projection
 *	

 */
 
#include "gmt.h"

struct XINGS {
	double xx[2], yy[2];	/* Cartesian coordinates of intersection with map boundary */
	double angle[2];	/* Angles of intersection */
	int sides[2];		/* Side id of intersection */
	int nx;			/* Number of intersections (1 or 2) */
};


/*	LINEAR PROJECTION MAP FUNCTIONS	*/

int linear_map_boundary (w, e, s, n)
double w, e, s, n; {
	double x1, x2, y1, y2, x_length, y_length;
	
	geo_to_xy (w, s, &x1, &y1);
	geo_to_xy (e, n, &x2, &y2);
	if (!project_info.xyz_pos[0]) d_swap (x1, x2);
	if (!project_info.xyz_pos[1]) d_swap (y1, y2);
	x_length = fabs (x2 - x1);
	y_length = fabs (y2 - y1);
	
	if (frame_info.side[3])	/* West or left y-axis */
		y_axis (x1, y1, y_length, s, n, frame_info.anot_int[1], frame_info.frame_int[1],
				frame_info.label[1], project_info.xyz_projection[1], frame_info.anot_type[1], TRUE, frame_info.side[3]-1);
	
	if (frame_info.side[1])	/* East or right y-axis */
		y_axis (x2, y1, y_length, s, n, frame_info.anot_int[1], frame_info.frame_int[1],
			frame_info.label[1], project_info.xyz_projection[1], frame_info.anot_type[1], FALSE, frame_info.side[1]-1);
	
	if (frame_info.side[0])	/* South or lower x-axis */
		x_axis (x1, y1, x_length, w, e, frame_info.anot_int[0], frame_info.frame_int[0],
			frame_info.label[0], project_info.xyz_projection[0], frame_info.anot_type[0], TRUE, frame_info.side[0]-1);
	
	if (frame_info.side[2])	/* North or upper x-axis */
		x_axis (x1, y2, x_length, w, e, frame_info.anot_int[0], frame_info.frame_int[0],
			frame_info.label[0], project_info.xyz_projection[0], frame_info.anot_type[0], FALSE, frame_info.side[2]-1);
}

int x_axis (x0, y0, length, val0, val1, anotation_int, tickmark_int, label, axistype, anottype, below, anotate)
double x0, y0, length, val0, val1, anotation_int, tickmark_int;
char *label;
int axistype, anottype, below, anotate; {
	int i, i_a, i_f, justify, n_anotations = 0, n_tickmarks = 0, test;
	
	BOOLEAN left = FALSE, do_anot, do_tick;
	
	double axis_scale, val, v0, v1, anot_off, label_off, start_val_a, start_val_f, end_val;
	double tvals_a[9], tvals_f[9], sign, dy, tmp, xx, len, start_log_a, start_log_f;
	
	char annotation[80], format[20];
	
	ps_setfont (gmtdefs.anot_font);
	justify = (below) ? 10 : 2;
	sign = (below) ? -1.0 : 1.0;
	dy = sign * gmtdefs.tick_length;
	len = (gmtdefs.tick_length > 0.0) ? gmtdefs.tick_length : 0.0;
	left = (anotation_int < 0.0);
	anotation_int = fabs (anotation_int);
	tickmark_int = fabs (tickmark_int);
	
	do_anot = (anotation_int > 0.0);
	do_tick = (tickmark_int > 0.0);
		
	/* Find number of decimals needed, if any */
	
	get_format (anotation_int, format);

	anot_off = sign * (len + gmtdefs.anot_offset);
	label_off = sign * (len + 2.5 * gmtdefs.anot_offset + (gmtdefs.anot_font_size / gmt_ppu[gmtdefs.measure_unit]) * font_height[gmtdefs.anot_font]);
	
	/* Ready to draw axis */
	
	if (below) ps_comment ("Start of lower x-axis"); else ps_comment ("Start of upper x-axis");
	ps_transrotate (x0, y0, 0.0);
	ps_setline (gmtdefs.frame_pen);
	ps_plot (0.0, 0.0, 3);
	ps_plot (length, 0.0, -2);
	ps_setline (gmtdefs.tick_pen);
	
	i_a = i_f = 0;
	switch (axistype) {
		case POW:	/* Anotate in pow(x) */
			(*x_forward) (val0, &v0);
			(*x_forward) (val1, &v1);
			if (anottype == 2) {
				val = (anotation_int == 0.0) ? 0.0 : floor (v0 / anotation_int) * anotation_int;
				if (fabs (val - v0) > SMALL) val += anotation_int;
				start_val_a = val;	end_val = v1;
				val = (tickmark_int == 0.0) ? 0.0 : floor (v0 / tickmark_int) * tickmark_int;
				if (fabs (val - v0) > SMALL) val += tickmark_int;
				start_val_f = val;
			}
			else {
				val = (anotation_int == 0.0) ? 0.0 : floor (val0 / anotation_int) * anotation_int;
				if (fabs (val - val0) > SMALL) val += anotation_int;
				start_val_a = val;	end_val = val1;
				val = (tickmark_int == 0.0) ? 0.0 : floor (val0 / tickmark_int) * tickmark_int;
				if (fabs (val - val0) > SMALL) val += tickmark_int;
				start_val_f = val;
			}
			break;
		case LOG10:	/* Anotate in d_log10 (x) */
			v0 = d_log10 (val0);
			v1 = d_log10 (val1);
			val = pow (10.0, floor (v0));
			test = rint (anotation_int) - 1;
			if (test < 0 || test > 2) test = 0;
			if (test == 0) {
				n_anotations = 1;
				tvals_a[0] = 10.0;
			}
			else if (test == 1) {
				tvals_a[0] = 1.0;
				tvals_a[1] = 2.0;
				tvals_a[2] = 5.0;
				n_anotations = 3;
			}
			else if (test == 2) {
				n_anotations = 9;
				for (i = 0; i < n_anotations; i++) tvals_a[i] = i + 1;
			}
			test = rint (tickmark_int) - 1;
			if (test < 0 || test > 2) test = 0;
			if (test == 0) {
				n_tickmarks = 1;
				tvals_f[0] = 10.0;
			}
			if (test == 1) {
				tvals_f[0] = 1.0;
				tvals_f[1] = 2.0;
				tvals_f[2] = 5.0;
				n_tickmarks = 3;
			}
			else if (test == 2) {
				n_tickmarks = 9;
				for (i = 0; i < n_tickmarks; i++) tvals_f[i] = i + 1;
			}
			i_a = 0;
			start_log_a = val = pow (10.0, floor (v0));
			while ((val0 - val) > SMALL) {
				if (i_a < n_anotations)
					val = start_log_a * tvals_a[i_a];
				else {
					val = (start_log_a *= 10.0);
					i_a = 0;
				}
				i_a++;
			}
			i_a--;
			start_val_a = val;
			i_f = 0;
			start_log_f = val = pow (10.0, floor (v0));
			while ((val0 - val) > SMALL) {
				if (i_f < n_tickmarks)
					val = start_log_f * tvals_f[i_f];
				else {
					val = (start_log_f *= 10.0);
					i_f = 0;
				}
				i_f++;
			}
			i_f--;
			start_val_f = val;
			end_val = val1;
			break;
		case LINEAR:
			val = (anotation_int == 0.0) ? 0.0 : floor (val0 / anotation_int) * anotation_int;
			if (fabs (val - val0) > SMALL) val += anotation_int;
			start_val_a = val;	end_val = val1;
			val = (tickmark_int == 0.0) ? 0.0 : floor (val0 / tickmark_int) * tickmark_int;
			if (fabs (val - val0) > SMALL) val += tickmark_int;
			start_val_f = val;
			(*x_forward) (val0, &v0);
			(*x_forward) (val1, &v1);
			break;
	}
	
	axis_scale = length / (v1 - v0);
	
	/* Do anotations with tickmarks */
	
	val = (anotation_int == 0.0) ? end_val + 1.0 : start_val_a;
	
	while (do_anot && val <= end_val) {
	
		i_a++;

		switch (anottype) {
			case 0:
				(*x_forward) (val, &tmp);
				xx = (tmp - v0) * axis_scale;
				sprintf (annotation, format, val);
				break;
			case 1:
				sprintf (annotation, "%d\0", (int) d_log10 (val));
				xx = (d_log10 (val) - v0) * axis_scale;
				break;
			case 2:
				if (axistype == POW) {
					xx = (val - v0) * axis_scale;
					(*x_inverse) (&tmp, val);
					sprintf (annotation, format, tmp);
				}
				else {
					xx = (d_log10 (val) - v0) * axis_scale;
					sprintf (annotation, "10@+%d@+\0", (int) d_log10 (val));
				}
				break;
		}
				
		if (left) xx = length - xx;
		
		ps_plot (xx, 0.0, 3);
		ps_plot (xx, dy, -2);
		if (anotate) ps_text (xx, anot_off, gmtdefs.anot_font_size, annotation, 0.0, justify, 0);
		
		if (axistype == LOG10) {
			if (i_a < n_anotations)
				val = start_log_a * tvals_a[i_a];
			else {
				val = (start_log_a *= 10.0);
				i_a = 0;
			}
		}
		else
			val = start_val_a + i_a * anotation_int;
			
	}

	/* Now do frame tickmarks */
	
	dy *= 0.5;
	
	val = (tickmark_int == 0.0) ? end_val + 1.0 : start_val_f;

	while (do_tick && val <= end_val) {
	
		i_f++;
		
		switch (anottype) {
			case 0:
				(*x_forward) (val, &tmp);
				xx = (tmp - v0) * axis_scale;
				break;
			case 1:
				xx = (d_log10 (val) - v0) * axis_scale;
				break;
			case 2:
				xx =  ((axistype == POW) ? (val - v0) : (d_log10 (val) - v0)) * axis_scale;
				break;
		}
				
		if (left) xx = length - xx;
		
		ps_plot (xx, 0.0, 3);
		ps_plot (xx, dy, -2);
		
		if (axistype == LOG10) {
			if (i_f < n_tickmarks)
				val = start_log_f * tvals_f[i_f];
			else {
				val = start_log_f *= 10.0;
				i_f = 0;
			}
		}
		else
			val = start_val_f + i_f * tickmark_int;
	}

	/* Finally do label */
	
	ps_setfont (gmtdefs.label_font);
	if (label && anotate) ps_text (0.5 * length, label_off, gmtdefs.label_font_size, label, 0.0, justify, 0);
	ps_rotatetrans  (-x0, -y0, 0.0);
	ps_comment ("End of x-axis");
}

int y_axis (x0, y0, length, val0, val1, anotation_int, tickmark_int, label, axistype, anottype, left_side, anotate)
double x0, y0, length, val0, val1, anotation_int, tickmark_int;
char *label;
int axistype, anottype, left_side, anotate; {
	int i, i_a, i_f, anot_justify, label_justify, n_anotations = 0, n_tickmarks = 0, ndec, test;
	
	BOOLEAN left = FALSE, do_anot, do_tick;
	
	double axis_scale, val, v0, v1, anot_off, label_off, start_val_a, start_val_f, end_val;
	double tvals_a[9], tvals_f[9], sign, len, dy, tmp, xx, off, angle, start_log_a, start_log_f;
	
	char annotation[80], text_l[80], text_u[80], format[20];
	
	ps_setfont (gmtdefs.anot_font);
	sign = (left_side) ? 1.0 : -1.0;
	len = (gmtdefs.tick_length > 0.0) ? gmtdefs.tick_length : 0.0;
	dy = sign * gmtdefs.tick_length;
	left = (anotation_int < 0.0);
	anotation_int = fabs (anotation_int);
	tickmark_int = fabs (tickmark_int);
	
	do_anot = (anotation_int > 0.0);
	do_tick = (tickmark_int > 0.0);
	
	ndec = get_format (anotation_int, format);
	
	/* Ready to draw axis */
	
	if (left_side) ps_comment ("Start of left y-axis"); else ps_comment ("Start of right y-axis");
	ps_transrotate (x0, y0, 90.0);
	ps_setline (gmtdefs.frame_pen);
	ps_plot (0.0, 0.0, 3);
	ps_plot (length, 0.0, -2);
	ps_setline (gmtdefs.tick_pen);
	
	i_a = i_f = 0;
	switch (axistype) {
		case POW:
			(*y_forward) (val0, &v0);
			(*y_forward) (val1, &v1);
			if (anottype == 2) {
				val = (anotation_int == 0.0) ? 0.0 : floor (v0 / anotation_int) * anotation_int;
				if (fabs (val - v0) > SMALL) val += anotation_int;
				start_val_a = val;	end_val = v1;
				val = (tickmark_int == 0.0) ? 0.0 : floor (v0 / tickmark_int) * tickmark_int;
				if (fabs (val - v0) > SMALL) val += tickmark_int;
				start_val_f = val;
				sprintf (text_l, "%d\0", (int)floor (v0));
				sprintf (text_u, "%d\0", (int)ceil (v1));
			}
			else {	/* Anotate in x */
				val = (anotation_int == 0.0) ? 0.0 : floor (val0 / anotation_int) * anotation_int;
				if (fabs (val - val0) > SMALL) val += anotation_int;
				start_val_a = val;	end_val = val1;
				val = (tickmark_int == 0.0) ? 0.0 : floor (val0 / tickmark_int) * tickmark_int;
				if (fabs (val - val0) > SMALL) val += tickmark_int;
				start_val_f = val;
				sprintf (text_l, "%d\0", (int)floor (val0));
				sprintf (text_u, "%d\0", (int)ceil (val1));
			}
			break;
		case LOG10:
			v0 = d_log10 (val0);
			v1 = d_log10 (val1);
			val = pow (10.0, floor (v0));
			test = rint (anotation_int) - 1;
			if (test < 0 || test > 2) test = 0;
			if (test == 0) {
				n_anotations = 1;
				tvals_a[0] = 10.0;
				if (fabs (val - val0) > SMALL) val *= 10.0;
			}
			else if (test == 1) {
				tvals_a[0] = 1.0;
				tvals_a[1] = 2.0;
				tvals_a[2] = 5.0;
				n_anotations = 3;
			}
			else if (test == 2) {
				n_anotations = 9;
				for (i = 0; i < n_anotations; i++) tvals_a[i] = i + 1;
			}
			test = rint (tickmark_int) - 1;
			if (test < 0 || test > 2) test = 0;
			if (test == 0) {
				n_tickmarks = 1;
				tvals_f[0] = 10.0;
			}
			else if (test == 1) {
				tvals_f[0] = 1.0;
				tvals_f[1] = 2.0;
				tvals_f[2] = 5.0;
				n_tickmarks = 3;
			}
			else if (test == 2) {
				n_tickmarks = 9;
				for (i = 0; i < n_tickmarks; i++) tvals_f[i] = i + 1;
			}
			i_a = 0;
			start_log_a = val = pow (10.0, floor (v0));
			while ((val0 - val) > SMALL) {
				if (i_a < n_anotations)
					val = start_log_a * tvals_a[i_a];
				else {
					val = (start_log_a *= 10.0);
					i_a = 0;
				}
				i_a++;
			}
			i_a--;
			start_val_a = val;
			i_f = 0;
			start_log_f = val = pow (10.0, floor (v0));
			while ((val0 - val) > SMALL) {
				if (i_f < n_tickmarks)
					val = start_log_f * tvals_f[i_f];
				else {
					val = (start_log_f *= 10.0);
					i_f = 0;
				}
				i_f++;
			}
			i_f--;
			start_val_f = val;
			end_val = val1;
			if (anottype == 2) {	/* 10 ^ pow annotations */
				sprintf (text_l, "10%d\0", (int)floor (v0));
				sprintf (text_u, "10%d\0", (int)ceil (v1));
			}
			else {
				sprintf (text_l, "%d\0", (int)floor ((anottype == 0) ? val0 : v0));
				sprintf (text_u, "%d\0", (int)ceil ((anottype == 0) ? val1 : v1));
			}
			break;
		case LINEAR:
			v0 = val0;
			v1 = val1;
			val = (anotation_int == 0.0) ? 0.0 : floor (val0 / anotation_int) * anotation_int;
			if (fabs (val - val0) > SMALL) val += anotation_int;
			start_val_a = val;	end_val = val1;
			val = (tickmark_int == 0.0) ? 0.0 : floor (val0 / tickmark_int) * tickmark_int;
			if (fabs (val - val0) > SMALL) val += tickmark_int;
			start_val_f = val;
			sprintf (text_l, "%d\0", (int)floor (fabs (val0)));
			sprintf (text_u, "%d\0", (int)ceil (fabs (val1)));
			break;
	}
	
	/* Find offset based on no of digits before and after a period, if any */
	
	off = ((MAX (strlen (text_l), strlen (text_u)) + ndec) * 0.55 + ((ndec > 0) ? 0.3 : 0.0) + ((val0 < 0.0) ? 0.3 : 0.0))
		* gmtdefs.anot_font_size / gmt_ppu[gmtdefs.measure_unit];

	axis_scale = length / (v1 - v0);
	
	if (gmtdefs.y_axis_type == 0) {	/* Horizontal anotations */
		if (left_side) {
			anot_off = len + gmtdefs.anot_offset;
			label_off = anot_off + 1.5 * gmtdefs.anot_offset + off;
			anot_justify = 7;
			label_justify = 2;
		}
		else {
			anot_off = -(len + gmtdefs.anot_offset + off);
			label_off = anot_off - 1.5 * gmtdefs.anot_offset;
			anot_justify = 7;
			label_justify = 10;
		}
		angle = -90.0;
	}
	else {
		anot_off = sign * (len + gmtdefs.anot_offset);
		label_off = sign * (len + 2.5 * gmtdefs.anot_offset + (gmtdefs.anot_font_size / gmt_ppu[gmtdefs.measure_unit]) * font_height[gmtdefs.anot_font]);
		anot_justify = label_justify = (left_side) ? 2 : 10;
		angle = 0.0;
	}
	
	/* Do anotations */
	
	val = (anotation_int == 0.0) ? end_val + 1.0 : start_val_a;
	
	while (do_anot && val <= end_val) {
	
		i_a++;
		
		switch (anottype) {
			case 0:
				sprintf (annotation, format, val);
				(*y_forward) (val, &tmp);
				xx = (tmp - v0) * axis_scale;
				break;
			case 1:
				sprintf (annotation, "%d\0", (int) d_log10 (val));
				xx = (d_log10 (val) - v0) * axis_scale;
				break;
			case 2:
				if (axistype == POW) {
					xx = (val - v0) * axis_scale;
					(*y_inverse) (&tmp, val);
					sprintf (annotation, format, tmp);
				}
				else {
					xx = (d_log10 (val) - v0) * axis_scale;
					sprintf (annotation, "10@+%d@+\0", (int) d_log10 (val));
				}
				break;
		}
				
		if (left) xx = length - xx;
		
		ps_plot (xx, 0.0, 3);
		ps_plot (xx, dy, -2);
		if (anotate) ps_text (xx, anot_off, gmtdefs.anot_font_size, annotation, angle, anot_justify, 0);
		
		if (axistype == LOG10) {
			if (i_a < n_anotations)
				val = start_log_a * tvals_a[i_a];
			else {
				val = (start_log_a *= 10.0);
				i_a = 0;
			}
		}
		else
			val = start_val_a + i_a * anotation_int;
			
	}


	/* Now do frame tickmarks */
	
	dy *= 0.5;
	
	val = (tickmark_int == 0.0) ? end_val + 1.0 : start_val_f;

	while (do_tick && val <= end_val) {
	
		i_f++;
		
		switch (anottype) {
			case 0:
				(*y_forward) (val, &tmp);
				xx = (tmp - v0) * axis_scale;
				break;
			case 1:
				xx = (d_log10 (val) - v0) * axis_scale;
				break;
			case 2:
				xx =  ((axistype == POW) ? (val - v0) : (d_log10 (val) - v0)) * axis_scale;
				break;
		}
				
		if (left) xx = length - xx;
		
		ps_plot (xx, 0.0, 3);
		ps_plot (xx, dy, -2);
		
		if (axistype == LOG10) {
			if (i_f < n_tickmarks)
				val = start_log_f * tvals_f[i_f];
			else {
				val = start_log_f *= 10.0;
				i_f = 0;
			}
		}
		else
			val = start_val_f + i_f * tickmark_int;
	}

	/* Finally do label */
	
	ps_setfont (gmtdefs.label_font);
	if (label && anotate) ps_text (0.5 * length, label_off, gmtdefs.label_font_size, label, 0.0, label_justify, 0);
	ps_rotatetrans  (-x0, -y0, -90.0);
	ps_comment ("End of y-axis");
}

int logx_grid (w, e, s, n, dval)
double w, e, s, n, dval; {
	int i, nticks, test;
	double val, v0, end_val, start_log, tvals[9];
	
	test = rint (fabs (dval)) - 1;
	if (test < 0 || test > 2) test = 0;
	if (test == 0) {
		tvals[0] = 1.0;
		nticks = 1;
	}
	if (test == 1) {
		tvals[0] = 1.0;
		tvals[1] = 2.0;
		tvals[2] = 5.0;
		nticks = 3;
	}
	else if (test == 2) {
		nticks = 9;
		for (i = 0; i < nticks; i++) tvals[i] = i + 1;
	}
	
	v0 = d_log10 (w);
	start_log = val = pow (10.0, floor (v0));
	i = 0;
	while ((w - val) > SMALL) {
		if (i < nticks)
			val = start_log * tvals[i];
		else {
			val = (start_log *= 10.0);
			i = 0;
		}
		i++;
	}
	i--;
	end_val = e;
	
	while (val <= end_val) {
		i++;
		geoplot (val, s, 3);
		geoplot (val, n, 2);
		if (i < nticks) 
			val = start_log * tvals[i];
		else {
			start_log *= 10;
			val = start_log;
			i = 0;
		}
	}
}

int logy_grid (w, e, s, n, dval)
double w, e, s, n, dval; {
	int i, nticks, test;
	double val, v0, end_val, start_log, tvals[9];
	
	test = rint (fabs (dval)) - 1;
	if (test < 0 || test > 2) test = 0;
	if (test == 0) {
		tvals[0] = 1.0;
		nticks = 1;
	}
	if (test == 1) {
		tvals[0] = 1.0;
		tvals[1] = 2.0;
		tvals[2] = 5.0;
		nticks = 3;
	}
	else if (test == 2) {
		nticks = 9;
		for (i = 0; i < nticks; i++) tvals[i] = i + 1;
	}
	v0 = d_log10 (s);
	start_log = val = pow (10.0, floor (v0));
	i = 0;
	while ((s - val) > SMALL) {
		if (i < nticks)
			val = start_log * tvals[i];
		else {
			val = (start_log *= 10.0);
			i = 0;
		}
		i++;
	}
	i--;
	end_val = n;

	while (val <= end_val) {
		i++;
		geoplot (w, val, 3);
		geoplot (e, val, 2);
		if (i < nticks)
			val = start_log * tvals[i];
		else {
			start_log *= 10;
			val = start_log;
			i = 0;
		}
	}
}

/*	MERCATOR PROJECTION MAP FUNCTIONS	*/

int merc_map_boundary (w, e, s, n)
double w, e, s, n; {
	double x1, x2, x3, y1, y2, y3, s1, w1, val, v2;
	int shade, i, nx, ny, fat_pen, thin_pen;
	
	if (gmtdefs.basemap_type == 1) {	/* Draw plain boundary and return */
		wesn_map_boundary (w, e, s, n);
		return;
	}
	
	fat_pen = rint (gmtdefs.frame_width * gmtdefs.dpi);
	thin_pen = rint (0.1 * gmtdefs.frame_width * gmtdefs.dpi);
	
	ps_setline (thin_pen);
	if (frame_info.side[3]) {	/* Draw western boundary */
		geo_to_xy (w, s, &x1, &y1);
		y1 -= gmtdefs.frame_width;
		geo_to_xy (w, n, &x2, &y2);
		y2 += gmtdefs.frame_width;
		ps_plot (x1, y1, 3);
		ps_plot (x1, y2, -2);
		x1 -= gmtdefs.frame_width;
		ps_plot (x1, y1, 3);
		ps_plot (x1, y2, -2);
	}
	if (frame_info.side[1]) {	/* Draw eastern boundary */
		geo_to_xy (e, s, &x2, &y1);
		y1 -= gmtdefs.frame_width;
		geo_to_xy (e, n, &x1, &y2);
		y2 += gmtdefs.frame_width;
		ps_plot (x2, y1, 3);
		ps_plot (x2, y2, -2);
		x2 += gmtdefs.frame_width;
		ps_plot (x2, y1, 3);
		ps_plot (x2, y2, -2);
	}
	if (frame_info.side[0]) {	/* Draw southern boundary */
		geo_to_xy (w, s, &x1, &y1);
		x1 -= gmtdefs.frame_width;
		geo_to_xy (e, s, &x2, &y2);
		x2 += gmtdefs.frame_width;
		ps_plot (x1, y1, 3);
		ps_plot (x2, y1, -2);
		y1 -= gmtdefs.frame_width;
		ps_plot (x1, y1, 3);
		ps_plot (x2, y1, -2);
	}
	if (frame_info.side[2]) {	/* Draw northern boundary */
		geo_to_xy (w, n, &x1, &y1);
		x1 -= gmtdefs.frame_width;
		geo_to_xy (e, n, &x2, &y2);
		x2 += gmtdefs.frame_width;
		ps_plot (x1, y2, 3);
		ps_plot (x2, y2, -2);
		y2 += gmtdefs.frame_width;
		ps_plot (x1, y2, 3);
		ps_plot (x2, y2, -2);
	}
	
	/* Draw frame grid for W/E boundaries */
	
	if (frame_info.frame_int[1] != 0.0) {
		shade = TRUE;
		ps_setline(fat_pen);
		s1 = floor (s / frame_info.frame_int[1]) * frame_info.frame_int[1];
		if (fabs (s1 - s) > SMALL) s1 += frame_info.frame_int[1];
		ny = (s1 > n) ? -1 : (n - s1) / frame_info.frame_int[1];
		for (i = 0; i <= ny; i++) {
			val = s1 + i * frame_info.frame_int[1];
			geo_to_xy (w, val, &x1, &y1);
			geo_to_xy (e, val, &x2, &y2);
			if (shade) {
				v2 = val + frame_info.frame_int[1];
				if (v2 > n) v2 = n;
				if (frame_info.side[3]) {
					geo_to_xy (w, v2, &x3, &y3);
					ps_plot (x1-0.5*gmtdefs.frame_width, y1, 3);
					ps_plot (x3-0.5*gmtdefs.frame_width, y3, -2);
				}
				if (frame_info.side[1]) {
					geo_to_xy (e, v2, &x3, &y3);
					ps_plot (x2+0.5*gmtdefs.frame_width, y2, 3);
					ps_plot (x3+0.5*gmtdefs.frame_width, y3, -2);
				}
				shade = FALSE;
			}
			else
				shade = TRUE;
		}
	}
	
	/* Draw Frame grid for N and S boundaries */
	
	if (frame_info.frame_int[0] != 0.0) {
		shade = TRUE;
		w1 = floor (w / frame_info.frame_int[0]) * frame_info.frame_int[0];
		if (fabs (w1 - w) > SMALL) w1 += frame_info.frame_int[0];
		nx = (w1 > e) ? -1 : (e - w1) / frame_info.frame_int[0];
		for (i = 0; i <= nx; i++) {
			val = w1 + i * frame_info.frame_int[0];
			geo_to_xy (val, s, &x1, &y1);
			geo_to_xy (val, n, &x2, &y2);
			if (shade) {
				v2 = val + frame_info.frame_int[0];
				if (v2 > e) v2 = e;
				if (frame_info.side[0]) {
					geo_to_xy (v2, s, &x3, &y3);
					ps_plot (x1, y1-0.5*gmtdefs.frame_width, 3);
					ps_plot (x3, y3-0.5*gmtdefs.frame_width, -2);
				}
				if (frame_info.side[2]) {
					geo_to_xy (v2, n, &x3, &y3);
					ps_plot (x2, y2+0.5*gmtdefs.frame_width, 3);
					ps_plot (x3, y3+0.5*gmtdefs.frame_width, -2);
				}
				shade = FALSE;
			}
			else
				shade = TRUE;
		}
	}
	ps_setline (thin_pen);
}

/*	POLAR STEREOGRAPHIC PROJECTION MAP FUNCTIONS	*/

int polar_map_boundary (w, e, s, n)
double w, e, s, n; {
	int i, nx, ny, shade, thin_pen, fat_pen;
	double anglew, dx2w, dy2w, anglee, dx2e, dy2e;
	double y0, x0, radiuss, radiusn, da, da0, az1, az2, psize;
	double x1, x2, x3, y1, y2, y3, s1, w1, val, v2, dummy, r2, dr;
	
	if (!project_info.region) { /* Draw rectangular boundary and return */
		rect_map_boundary (0.0, 0.0, project_info.xmax, project_info.ymax);
		return;
	}
	
	if (!project_info.north_pole && s == -90.0) /* Cannot have southern boundary */
		frame_info.side[0] = FALSE;
	if (project_info.north_pole && n == 90.0) /* Cannot have northern boundary */
		frame_info.side[2] = FALSE;
	if ((fabs(w-e) == 360.0 || w == e)) {
		frame_info.side[1] = FALSE;
		frame_info.side[3] = FALSE;
	}
	
	if (gmtdefs.basemap_type == 1) { /* Draw plain boundary and return */
		wesn_map_boundary (w, e, s, n);
		return;
	}
	
	/* Here draw fancy map boundary */
	
	fat_pen = rint (gmtdefs.frame_width * gmtdefs.dpi);
	thin_pen = rint (0.1 * gmtdefs.frame_width * gmtdefs.dpi);
	ps_setline (thin_pen);
	
	psize = gmtdefs.frame_width;
	
	/* Angle of western boundary:  */
	
	geo_to_xy (w, n, &x1, &y1);
	geo_to_xy (w, s, &x2, &y2);
	anglew = d_atan2 (y1 - y2, x1 - x2);
	dx2w = -psize * sin (anglew);
	dy2w = psize * cos (anglew);
	
	/* Angle of eastern boundary:  */
	
	geo_to_xy (e, n, &x1, &y1);
	geo_to_xy (e, s, &x2, &y2);
	anglee = d_atan2 (y1 - y2, x1 - x2);
	dx2e = -psize * cos (anglee);
	dy2e = psize * sin (anglee);
	
	geo_to_xy (project_info.central_meridian, project_info.pole, &x0, &y0);
	geo_to_xy (project_info.central_meridian, s, &dummy, &y1);
	geo_to_xy (project_info.central_meridian, n, &dummy, &y2);
	radiuss = fabs(y1 - y0);
	radiusn = fabs(y2 - y0);
	dr = 0.5 * psize;
	
	if (frame_info.side[3]) {	/* Draw western boundary */
		geo_to_xy (w, n, &x1, &y1);
		geo_to_xy (w, s, &x2, &y2);
		ps_plot (x1+dy2w, y1-dx2w, 3);
		ps_plot (x2-dy2w, y2+dx2w, -2);
		x1 += dx2w;
		y1 += dy2w;
		x2 += dx2w;
		y2 += dy2w;
		ps_plot (x1+dy2w, y1-dx2w, 3);
		ps_plot (x2-dy2w, y2+dx2w, -2);
	}
	if (frame_info.side[1]) {	/* Draw eastern boundary */
		geo_to_xy (e, n, &x1, &y1);
		geo_to_xy (e, s, &x2, &y2);
		ps_plot (x1-dx2e, y1+dy2e, 3);
		ps_plot (x2+dx2e, y2-dy2e, -2);
		x1 += dy2e;
		y1 += dx2e;
		x2 += dy2e;
		y2 += dx2e;
		ps_plot (x1-dx2e, y1+dy2e, 3);
		ps_plot (x2+dx2e, y2-dy2e, -2);
	}
	if (frame_info.side[0]) {	/* Draw southern boundary */
		da0 = R2D * psize /radiuss;
		geo_to_xy (e, s, &x1, &y1);
		geo_to_xy (w, s, &x2, &y2);
		az1 = d_atan2 (y1 - y0, x1 - x0) * R2D;
		az2 = d_atan2 (y2 - y0, x2 - x0) * R2D;
		if (project_info.north_pole) {
			r2 = radiuss + psize;
			da = R2D * psize / r2;
			if (az1 <= az2) az1 += 360.0;
			ps_arc (x0, y0, radiuss, az2-da0, az1+da0, 3);
			ps_arc (x0, y0, r2, az2-da, az1+da, 3);
		}
		else {
			r2 = radiuss - psize;
			da = R2D * psize / r2;
			if (az2 <= az1) az2 += 360.0;
			ps_arc (x0, y0, radiuss, az1-da0, az2+da0, 3);
			ps_arc (x0, y0, r2, az1-da, az2+da, 3);
		}
	}
	if (frame_info.side[2]) {	/* Draw northern boundary */
		da0 = R2D * psize / radiusn;
		geo_to_xy (e, n, &x1, &y1);
		geo_to_xy (w, n, &x2, &y2);
		az1 = d_atan2 (y1 - y0, x1 - x0) * R2D;
		az2 = d_atan2 (y2 - y0, x2 - x0) * R2D;
		if (project_info.north_pole) {
			r2 = radiusn - psize;
			da = R2D * psize / r2;
			if (az1 <= az2) az1 += 360.0;
			ps_arc (x0, y0, radiusn, az2-da0, az1+da0, 3);
			ps_arc (x0, y0, r2, az2-da, az1+da, 3);
		}
		else {
			r2 = radiusn + psize;
			da = R2D * psize / r2;
			if (az2 <= az1) az2 += 360.0;
			ps_arc (x0, y0, radiusn, az1-da0, az2+da0, 3);
			ps_arc (x0, y0, r2, az1-da, az2+da, 3);
		}
	}
	
	/* Anotate S-N axes */
	
	ps_setline (fat_pen);
	if (frame_info.frame_int[1] != 0.0) {
		shade = TRUE;
		s1 = floor(s/frame_info.frame_int[1])*frame_info.frame_int[1];
		if (fabs(s1-s) > SMALL) s1 += frame_info.frame_int[1];
		ny = (s1 > n) ? -1 : (n-s1) / frame_info.frame_int[1];
		for (i = 0; i <= ny; i++) {
			val = s1 + i*frame_info.frame_int[1];
			geo_to_xy (w, val, &x1, &y1);
			geo_to_xy (e, val, &x2, &y2);
			if (shade) {
				v2 = val + frame_info.frame_int[1];
				if (v2 > n) v2 = n;
				if (frame_info.side[3]) {
					geo_to_xy (w, v2, &x3, &y3);
					ps_plot (x1+0.5*dx2w, y1+0.5*dy2w, 3);
					ps_plot (x3+0.5*dx2w, y3+0.5*dy2w, -2);
				}
				if (frame_info.side[1]) {
					geo_to_xy (e, v2, &x3, &y3);
					ps_plot (x2+0.5*dy2e, y2+0.5*dx2e, 3);
					ps_plot (x3+0.5*dy2e, y3+0.5*dx2e, -2);
				}
				shade = FALSE;
			}
			else
				shade = TRUE;
		}
	}

	/* Anotate W-E axes */
	
	if (frame_info.frame_int[0] != 0.0) {
		shade = TRUE;
		w1 = floor(w/frame_info.frame_int[0])*frame_info.frame_int[0];
		if (fabs(w1-w) > SMALL) w1 += frame_info.frame_int[0];
		nx = (w1 > e) ? -1 : (e-w1) / frame_info.frame_int[0];
		for (i = 0; i <= nx; i++) {
			val = w1 + i*frame_info.frame_int[0];
			if (shade) {
				v2 = val + frame_info.frame_int[0];
				if (v2 > e) v2 = e;
				if (frame_info.side[0]) {
					geo_to_xy (v2, s, &x1, &y1);
					geo_to_xy (val, s, &x2, &y2);
					az1 = d_atan2 (y1 - y0, x1 - x0) * R2D;
					az2 = d_atan2 (y2 - y0, x2 - x0) * R2D;
					if (project_info.north_pole) {
						if (az1 < az2) az1 += 360.0;
						ps_arc (x0, y0, radiuss+dr, az2, az1, 3);
					}
					else {
						if (az2 < az1) az2 += 360.0;
						ps_arc (x0, y0, radiuss-dr, az1, az2, 3);
					}
				}
				if (frame_info.side[2]) {
					geo_to_xy (v2, n, &x1, &y1);
					geo_to_xy (val, n, &x2, &y2);
					az1 = d_atan2 (y1 - y0, x1 - x0) * R2D;
					az2 = d_atan2 (y2 - y0, x2 - x0) * R2D;
					if (project_info.north_pole) {
						if (az1 < az2) az1 += 360.0;
						ps_arc (x0, y0, radiusn-dr, az2, az1, 3);
					}
					else {
						if (az2 < az1) az2 += 360.0;
						ps_arc (x0, y0, radiusn+dr, az1, az2, 3);
					}
				}
				shade = FALSE;
			}
			else
				shade = TRUE;
		}
	}
	ps_setline (thin_pen);
}

/*	LAMBERT CONFORMAL CONIC PROJECTION MAP FUNCTIONS	*/

int lambert_map_boundary (w, e, s, n)
double w, e, s, n; {
	int i, nx, ny, shade, fat_pen, thin_pen;
	double dx, dy, angle, dx2, dy2, y0, x0, radiuss, radiusn, da, da0, az1, az2, psize;
	double x1, x2, x3, y1, y2, y3, s1, w1, val, v2, rsize;
	
	if (!project_info.region) { /* Draw rectangular boundary and return */
		rect_map_boundary (0.0, 0.0, project_info.xmax, project_info.ymax);
		return;
	}
	if (gmtdefs.basemap_type == 1) { /* Draw plain boundary and return */
		wesn_map_boundary (w, e, s, n);
		return;
	}
	
	fat_pen = rint (gmtdefs.frame_width * gmtdefs.dpi);
	thin_pen = rint (0.1 * gmtdefs.frame_width * gmtdefs.dpi);
	ps_setline (thin_pen);
	
	psize = (project_info.north_pole) ? gmtdefs.frame_width : -gmtdefs.frame_width;
	rsize = fabs (psize);
	geo_to_xy (w, n, &x1, &y1);
	geo_to_xy (w, s, &x2, &y2);
	dx = x1 - x2;
	dy = y1 - y2;
	angle = R2D*d_atan2 (dy, dx) - 90.0;
	if (fabs(angle-180.0) < SMALL) angle = 0.0;
	dx2 = rsize * cos (angle*D2R);
	dy2 = rsize * sin (angle*D2R);
	geo_to_xy (project_info.central_meridian, project_info.pole, &x0, &y0);
	geo_to_xy (e, s, &x1, &y1);
	geo_to_xy (w, n, &x2, &y2);
	radiuss = hypot (x1 - x0, y1 - y0);
	radiusn = hypot (x2 - x0, y2 - y0);
	
	if (frame_info.side[3]) {	/* Draw western boundary */
		geo_to_xy (w, s, &x1, &y1);
		geo_to_xy (w, n, &x2, &y2);
		ps_plot (x1+dy2, y1-dx2, 3);
		ps_plot (x2-dy2, y2+dx2, -2);
		x1 -= dx2;
		y1 -= dy2;
		x2 -= dx2;
		y2 -= dy2;
		ps_plot (x1+dy2, y1-dx2, 3);
		ps_plot (x2-dy2, y2+dx2, -2);
	}
	if (frame_info.side[1]) {	/* Draw eastern boundary */
		geo_to_xy (e, s, &x1, &y1);
		geo_to_xy (e, n, &x2, &y2);
		ps_plot (x1-dy2, y1-dx2, 3);
		ps_plot (x2+dy2, y2+dx2, -2);
		x1 += dx2;
		y1 -= dy2;
		x2 += dx2;
		y2 -= dy2;
		ps_plot (x1-dy2, y1-dx2, 3);
		ps_plot (x2+dy2, y2+dx2, -2);
	}
	if (frame_info.side[0]) {	/* Draw southern boundary */
		da0 = R2D*gmtdefs.frame_width/radiuss;
		da = R2D*gmtdefs.frame_width/(radiuss+psize);
		geo_to_xy (e, s, &x1, &y1);
		geo_to_xy (w, s, &x2, &y2);
		az1 = d_atan2 (y1 - y0, x1 - x0) * R2D;
		az2 = d_atan2 (y2 - y0, x2 - x0) * R2D;
		if (project_info.north_pole) {
			if (az1 <= az2) az1 += 360.0;
			ps_arc (x0, y0, radiuss, az2-da0, az1+da0, 3);
			ps_arc (x0, y0, radiuss + psize, az2-da, az1+da, 3);
		}
		else {
			if (az2 <= az1) az2 += 360.0;
			ps_arc (x0, y0, radiuss, az1-da0, az2+da0, 3);
			ps_arc (x0, y0, radiuss + psize, az1-da, az2+da, 3);
		}
	}
	if (frame_info.side[2]) {	/* Draw northern boundary */
		da0 = R2D*gmtdefs.frame_width/radiusn;
		da = R2D*gmtdefs.frame_width/(radiusn-psize);
		geo_to_xy (e, s, &x1, &y1);
		geo_to_xy (w, s, &x2, &y2);
		az1 = d_atan2 (y1 - y0, x1 - x0) * R2D;
		az2 = d_atan2 (y2 - y0, x2 - x0) * R2D;
		if (project_info.north_pole) {
			if (az1 <= az2) az1 += 360.0;
			ps_arc (x0, y0, radiusn, az2-da0, az1+da0, 3);
			ps_arc (x0, y0, radiusn - psize, az2-da, az1+da, 3);
		}
		else {
			if (az2 <= az1) az2 += 360.0;
			ps_arc (x0, y0, radiusn, az1-da0, az2+da0, 3);
			ps_arc (x0, y0, radiusn - psize, az1-da, az2+da, 3);
		}
	}
	
	/* Anotate S-N axes */
	
	ps_setline (fat_pen);
	if (frame_info.frame_int[1] != 0.0) {
		shade = TRUE;
		s1 = floor(s/frame_info.frame_int[1])*frame_info.frame_int[1];
		if (fabs(s1-s) > SMALL) s1 += frame_info.frame_int[1];
		ny = (s1 > n) ? -1 : (n-s1) / frame_info.frame_int[1];
		for (i = 0; i <= ny; i++) {
			val = s1 + i*frame_info.frame_int[1];
			geo_to_xy (w, val, &x1, &y1);
			geo_to_xy (e, val, &x2, &y2);
			if (shade) {
				v2 = val + frame_info.frame_int[1];
				if (v2 > n) v2 = n;
				if (frame_info.side[3]) {
					geo_to_xy (w, v2, &x3, &y3);
					ps_plot (x1-0.5*dx2, y1-0.5*dy2, 3);
					ps_plot (x3-0.5*dx2, y3-0.5*dy2, -2);
				}
				if (frame_info.side[1]) {
					geo_to_xy (e, v2, &x3, &y3);
					ps_plot (x2+0.5*dx2, y2-0.5*dy2, 3);
					ps_plot (x3+0.5*dx2, y3-0.5*dy2, -2);
				}
				shade = FALSE;
			}
			else
				shade = TRUE;
		}
	}

	/* Anotate W-E axes */
	
	if (frame_info.frame_int[0] != 0.0) {
		shade = TRUE;
		w1 = floor(w/frame_info.frame_int[0])*frame_info.frame_int[0];
		if (fabs(w1-w) > SMALL) w1 += frame_info.frame_int[0];
		nx = (w1 > e) ? -1 : (e-w1) / frame_info.frame_int[0];
		da = dx;
		dx = dy;
		dy = da;
		for (i = 0; i <= nx; i++) {
			val = w1 + i*frame_info.frame_int[0];
			if (shade) {
				v2 = val + frame_info.frame_int[0];
				if (v2 > e) v2 = e;
				if (frame_info.side[0]) {
					geo_to_xy (v2, s, &x1, &y1);
					geo_to_xy (val, s, &x2, &y2);
					az1 = d_atan2 (y1 - y0, x1 - x0) * R2D;
					az2 = d_atan2 (y2 - y0, x2 - x0) * R2D;
					if (project_info.north_pole) {
						if (az1 < az2) az1 += 360.0;
						ps_arc (x0, y0, radiuss+0.5*psize, az2, az1, 3);
					}
					else {
						if (az2 < az1) az2 += 360.0;
						ps_arc (x0, y0, radiuss+0.5*psize, az1, az2, 3);
					}
				}
				if (frame_info.side[2]) {
					geo_to_xy (v2, n, &x1, &y1);
					geo_to_xy (val, n, &x2, &y2);
					az1 = d_atan2 (y1 - y0, x1 - x0) * R2D;
					az2 = d_atan2 (y2 - y0, x2 - x0) * R2D;
					if (project_info.north_pole) {
						if (az1 < az2) az1 += 360.0;
						ps_arc (x0, y0, radiusn-0.5*psize, az2, az1, 3);
					}
					else {
						if (az2 < az1) az2 += 360.0;
						ps_arc (x0, y0, radiusn-0.5*psize, az1, az2, 3);
					}
				}
				shade = FALSE;
			}
			else
				shade = TRUE;
		}
	}
	ps_setline (thin_pen);
}

/*	OBLIQUE MERCATOR PROJECTION MAP FUNCTIONS	*/

int oblmrc_map_boundary (w, e, s, n)
double w, e, s, n; {
	
	rect_map_boundary (0.0, 0.0, project_info.xmax, project_info.ymax);
}
	
/*	MOLLWEIDE and HAMMER-AITOFF EQUAL AREA PROJECTION MAP FUNCTIONS	*/

int ellipse_map_boundary (w, e, s, n)
double w, e, s, n; {
	
	if (!project_info.region) { /* Draw rectangular boundary and return */
		rect_map_boundary (0.0, 0.0, project_info.xmax, project_info.ymax);
		return;
	}
	if (project_info.s == -90.0) /* Cannot have southern boundary */
		frame_info.side[0] = FALSE;
	if (project_info.n == 90.0) /* Cannot have northern boundary */
		frame_info.side[2] = FALSE;

	wesn_map_boundary (w, e, s, n);
	
}
	
int basic_map_boundary (w, e, s, n)
double w, e, s, n; {
	
	if (!project_info.region) { /* Draw rectangular boundary and return */
		rect_map_boundary (0.0, 0.0, project_info.xmax, project_info.ymax);
		return;
	}
	wesn_map_boundary (w, e, s, n);
}
	
/*
 *	GENERIC MAP PLOTTING FUNCTIONS
 */

int wesn_map_boundary (w, e, s, n)
double w, e, s, n; {
	int i, np = 0;
	double *xx, *yy;
	
	ps_setline (gmtdefs.frame_pen);
	
	if (frame_info.side[3]) {	/* West */
		np = map_path (w, s, w, n, &xx, &yy);
		for (i = 0; i < np; i++) geoz_to_xy (xx[i], yy[i], project_info.z_level, &xx[i], &yy[i]);
		ps_line (xx, yy, np, 3, FALSE, TRUE);
		free ((char *)xx);	free ((char *)yy);
	}
	if (frame_info.side[1]) {	/* East */
		np = map_path (e, s, e, n, &xx, &yy);
		for (i = 0; i < np; i++) geoz_to_xy (xx[i], yy[i], project_info.z_level, &xx[i], &yy[i]);
		ps_line (xx, yy, np, 3, FALSE, TRUE);
		free ((char *)xx);	free ((char *)yy);
	}
	if (frame_info.side[0]) {	/* South */
		np = map_path (w, s, e, s, &xx, &yy);
		for (i = 0; i < np; i++) geoz_to_xy (xx[i], yy[i], project_info.z_level, &xx[i], &yy[i]);
		ps_line (xx, yy, np, 3, FALSE, TRUE);
		free ((char *)xx);	free ((char *)yy);
	}
	if (frame_info.side[2]) {	/* North */
		np = map_path (w, n, e, n, &xx, &yy);
		for (i = 0; i < np; i++) geoz_to_xy (xx[i], yy[i], project_info.z_level, &xx[i], &yy[i]);
		ps_line (xx, yy, np, 3, FALSE, TRUE);
		free ((char *)xx);	free ((char *)yy);
	}
}
	
int rect_map_boundary (x0, y0, x1, y1)
double x0, y0, x1, y1; {
	double xt[4], yt[4];
	
	xy_do_z_to_xy (x0, y0, project_info.z_level, &xt[0], &yt[0]);
	xy_do_z_to_xy (x1, y0, project_info.z_level, &xt[1], &yt[1]);
	xy_do_z_to_xy (x1, y1, project_info.z_level, &xt[2], &yt[2]);
	xy_do_z_to_xy (x0, y1, project_info.z_level, &xt[3], &yt[3]);

	ps_setline (gmtdefs.frame_pen);
	
	if (frame_info.side[3]) {	/* West */
		ps_plot (xt[0], yt[0], 3);
		ps_plot (xt[3], yt[3], -2);
	}
	if (frame_info.side[1]) {	/* East */
		ps_plot (xt[1], yt[1], 3);
		ps_plot (xt[2], yt[2], -2);
	}
	if (frame_info.side[0]) {	/* South */
		ps_plot (xt[0], yt[0], 3);
		ps_plot (xt[1], yt[1], -2);
	}
	if (frame_info.side[2]) {	/* North */
		ps_plot (xt[3], yt[3], 3);
		ps_plot (xt[2], yt[2], -2);
	}
}
	
int circle_map_boundary (w, e, s, n)
double w, e, s, n; {
	int i, nr;
	double x0, y0, a, da;
	
	if (!project_info.region) { /* Draw rectangular boundary and return */
		rect_map_boundary (0.0, 0.0, project_info.xmax, project_info.ymax);
		return;
	}
	
	ps_setline (gmtdefs.frame_pen);
	
	nr = gmtdefs.n_lon_nodes + gmtdefs.n_lat_nodes;
	if (nr >= gmt_n_alloc) get_plot_array ();
	da = 2.0 * M_PI / (nr - 1);
	for (i = 0; i < nr; i++) {
		a = i * da;
		x0 = project_info.r * cos (a);
		y0 = project_info.r * sin (a);
		xy_do_z_to_xy (x0, y0, project_info.z_level, &gmt_x_plot[i], &gmt_y_plot[i]);
	}
	geoz_to_xy (project_info.central_meridian, project_info.pole, project_info.z_level, &x0, &y0);
	ps_transrotate (x0, y0, 0.0);
	ps_line (gmt_x_plot, gmt_y_plot, nr, 3, FALSE, TRUE);
	ps_rotatetrans (-x0, -y0, 0.0);
}
	
int theta_r_map_boundary (w, e, s, n)
double w, e, s, n; {
	int i, nr;
	double x0, y0, x, y, a, da;
	
	ps_setline (gmtdefs.frame_pen);
	
	if (s == 0.0) frame_info.side[0] = 0;	/* No donuts, please */
	nr = gmtdefs.n_lon_nodes;
	if (nr >= gmt_n_alloc) get_plot_array ();
	da = fabs (project_info.e - project_info.w) / (nr - 1);
	for (i = 0; i < nr; i++) {
		a = project_info.w + i * da;
		x = project_info.r * cosd (a);
		y = project_info.r * sind (a);
		xy_do_z_to_xy (x, y, project_info.z_level, &gmt_x_plot[i], &gmt_y_plot[i]);
	}
	geoz_to_xy (project_info.central_meridian, project_info.pole, project_info.z_level, &x0, &y0);
	ps_transrotate (x0, y0, 0.0);
	ps_line (gmt_x_plot, gmt_y_plot, nr, 3, FALSE, TRUE);
	if (frame_info.side[0]) {
		double r;
		geo_to_xy (0.0, project_info.s, &r, &y);
		r -= x0;
		for (i = 0; i < nr; i++) {
			a = project_info.w + i * da;
			x = r * cosd (a);
			y = r * sind (a);
			xy_do_z_to_xy (x, y, project_info.z_level, &gmt_x_plot[i], &gmt_y_plot[i]);
		}
		ps_line (gmt_x_plot, gmt_y_plot, nr, 3, FALSE, TRUE);
	}
	ps_rotatetrans (-x0, -y0, 0.0);
	
	if (project_info.w != project_info.e) {	/* Must draw radial borders */
		double xx[2], yy[2];
		if (frame_info.side[1]) {
			geoz_to_xy (project_info.e, project_info.s, project_info.z_level, &xx[0], &yy[0]);
			geoz_to_xy (project_info.e, project_info.n, project_info.z_level, &xx[1], &yy[1]);
			ps_line (xx, yy, 2, 3, FALSE, TRUE);
		}
		if (frame_info.side[3]) {
			geoz_to_xy (project_info.w, project_info.s, project_info.z_level, &xx[0], &yy[0]);
			geoz_to_xy (project_info.w, project_info.n, project_info.z_level, &xx[1], &yy[1]);
			ps_line (xx, yy, 2, 3, FALSE, TRUE);
		}
	}
}
	
int map_latline (lat, west, east)		/* Draws a line of constant latitude */
double lat, west, east; {
	int nn;
	double *llon, *llat;
	char text[20];
	
	nn = latpath (lat, west, east, &llon, &llat);
	
	gmt_n_plot = geo_to_xy_line (llon, llat, nn);
	sprintf (text, "Lat = %lg\0", lat);
	ps_comment (text);
	gmt_plot_line (gmt_x_plot, gmt_y_plot, gmt_pen, gmt_n_plot);
	
	free ((char *)llon);
	free ((char *)llat);
}
	
int map_lonline (lon, south, north)	/* Draws a line of constant longitude */
double lon, south, north; {
	int nn;
	double *llon, *llat;
	char text[20];
	
	nn = lonpath (lon, south, north, &llon, &llat);

	gmt_n_plot = geo_to_xy_line (llon, llat, nn);
	sprintf (text, "Lon = %lg\0", lon);
	ps_comment (text);
	gmt_plot_line (gmt_x_plot, gmt_y_plot, gmt_pen, gmt_n_plot);
	
	free ((char *)llon);
	free ((char *)llat);
}

int map_lontick (lon, south, north)
double lon, south, north; {
	int i, nc;
	struct XINGS *xings;
	
	for (i = 0; i < (nc = map_loncross (lon, south, north, &xings)); i++) {
		map_tick (xings[i].xx, xings[i].yy, xings[i].sides, xings[i].angle, xings[i].nx);
		free ((char *)xings);
	}
}

int map_lattick (lat, west, east)
double lat, west, east; {
	int i, nc;
	
	struct XINGS *xings;
	
	for (i = 0; i < (nc = map_latcross (lat, west, east, &xings)); i++) {
		map_tick (xings[i].xx, xings[i].yy, xings[i].sides, xings[i].angle, xings[i].nx);
		free ((char *)xings);
	}
}

int map_tick (xx, yy, sides, angles, nx)
double *xx, *yy, *angles;
int *sides, nx; {
	double angle, xl, yl, xt, yt;
	int i;
	
	for (i = 0; i < nx; i++) {
		if (!project_info.edge[sides[i]]) continue;
		if (!frame_info.side[sides[i]]) continue;
		angle = angles[i];
		xl = 0.5 * gmtdefs.tick_length * cos (angle * D2R);
		yl = 0.5 * gmtdefs.tick_length * sin (angle * D2R);
		xy_do_z_to_xy (xx[i], yy[i], project_info.z_level, &xt, &yt);
		ps_plot (xt, yt, 3);
		xy_do_z_to_xy (xx[i]+xl, yy[i]+yl, project_info.z_level, &xt, &yt);
		ps_plot (xt, yt, -2);
	}
}

int map_symbol_ew (lat, label, west, east, anot)
double lat, west, east;
char *label;
BOOLEAN anot; {
	int i, nc;
	struct XINGS *xings;
	
	for (i = 0; i < (nc = map_latcross (lat, west, east, &xings)); i++) {
		map_symbol (xings[i].xx, xings[i].yy, xings[i].sides, xings[i].angle, label, xings[i].nx, 1, anot);
		free ((char *)xings);
	}
}

int map_symbol_ns (lon, label, south, north, anot)
double lon, south, north;
char *label;
BOOLEAN anot; {
	int i, nc;
	struct XINGS *xings;
	
	for (i = 0; i < (nc = map_loncross (lon, south, north, &xings)); i++) {
		map_symbol (xings[i].xx, xings[i].yy, xings[i].sides, xings[i].angle, label, xings[i].nx, 0, anot);
		free ((char *)xings);
	}
}

int map_symbol (xx, yy, sides, line_angles, label, nx, type, anot)
double *xx, *yy, *line_angles;
int *sides, nx, type;	/* type = 0 for lon and 1 for lat */
char *label;
BOOLEAN anot; {
	double line_angle, text_angle, len, dx, dy, angle, ca, sa, xt1, yt1, zz;
	int i, justify;
	
	len = ((gmtdefs.tick_length > 0.0) ? gmtdefs.tick_length : 0.0) + gmtdefs.anot_offset;
	
	for (i = 0; i < nx; i++) {
	
		if (prepare_label (line_angles[i], sides[i], xx[i], yy[i], type, &line_angle, &text_angle, &justify)) continue;
		
		angle = line_angle * D2R;
		ca = cos (angle);	sa = sin (angle);
		dx = gmtdefs.tick_length * ca;
		dy = gmtdefs.tick_length * sa;
		z_to_zz (project_info.z_level, &zz);
		xyz_to_xy (xx[i], yy[i], zz, &xt1, &yt1);
		ps_plot (xt1, yt1, 3);
		xyz_to_xy (xx[i]+dx, yy[i]+dy, zz, &xt1, &yt1);
		ps_plot (xt1, yt1, -2);
		xx[i] += len * ca;
		yy[i] += len * sa;
		xyz_to_xy (xx[i], yy[i], zz, &xt1, &yt1);
			
		if (project_info.three_D) {
			int upside = FALSE, k;
			double xp[2], yp[2], xt2, xt3, yt2, yt3, del_y;
			double size, xsize, ysize, xshrink, yshrink, tilt, baseline_shift, cb, sb, a;
				
			upside = (z_project.quadrant == 1 || z_project.quadrant == 4);
			cb = cosd (text_angle);	sb = sind (text_angle);
			if (sides[i]%2 == 0 && (justify%2 == 0)) {
				if (upside) {
					k = (sides[i] == 0) ? 2 : 10;
					text_angle += 180.0;
				}
				else
					k = justify;
				del_y = 0.5 * gmtdefs.anot_font_size * 0.732 * (k/4) / gmt_ppu[gmtdefs.measure_unit];
				justify = 2;
				xx[i] += del_y * ca;	yy[i] += del_y * sa;
				xyz_to_xy (xx[i], yy[i], zz, &xt1, &yt1);
			}
			else {
				del_y = -0.5 * gmtdefs.anot_font_size * 0.732 * (justify/4) / gmt_ppu[gmtdefs.measure_unit];
				if (upside) {
					if (sides[i]%2) del_y = -del_y;
					text_angle += 180.0;
					justify = (justify == 5) ? 7 : 5;
				}
				justify -= 4;
				switch (sides[i]) {
					case 0:
						a = (justify == 1) ? line_angle + 90.0 : line_angle - 90.0;
						break;
					case 1:
						a = line_angle + 90.0;
						break;
					case 2:
						a = (justify == 1) ? line_angle + 90.0 : line_angle - 90.0;
						break;
					case 3:
						a = line_angle - 90.0;
						break;
				}
				ca = cosd (a);	sa = sind (a);
				xx[i] += del_y * ca;	yy[i] += del_y * sa;
				xyz_to_xy (xx[i], yy[i], zz, &xt1, &yt1);
			}
			xp[0] = xx[i] + len * cb;	yp[0] = yy[i] + len * sb;
			xp[1] = xx[i] - len * sb;	yp[1] = yy[i] + len * cb;
			xyz_to_xy (xp[0], yp[0], zz, &xt2, &yt2);
			xyz_to_xy (xp[1], yp[1], zz, &xt3, &yt3);
			xshrink = hypot (xt2-xt1, yt2-yt1) / hypot (xp[0]-xx[i], yp[0]-yy[i]);
			yshrink = hypot (xt3-xt1, yt3-yt1) / hypot (xp[1]-xx[i], yp[1]-yy[i]);
			baseline_shift = d_atan2 (yt2 - yt1, xt2 - xt1) - d_atan2 (yp[0] - yy[i], xp[0] - xx[i]);
			tilt = 90.0 - R2D * (d_atan2 (yt3 - yt1, xt3 - xt1) - d_atan2 (yt2 - yt1, xt2 - xt1));
			tilt = tand (tilt);
			size = gmtdefs.anot_font_size * gmtdefs.dpi / gmt_ppu[gmtdefs.measure_unit];
			xsize = size * xshrink;
			ysize = size * yshrink;
			/* Temporarely modify meaning of F0 */
			printf ("/F0 {pop /%s findfont [%lg 0 %lg %lg 0 0] makefont setfont} bind def\n",
			font_name[gmtdefs.anot_font], xsize, ysize * tilt, ysize);
			ps_setfont (0);
			text_angle += (R2D * baseline_shift);
		}
		if (anot) ps_text (xt1, yt1, gmtdefs.anot_font_size, label, text_angle, justify, 0);
	}
}

int map_latcross (lat, west, east, xings)
double lat, east, west;
struct XINGS *xings[]; {
	int i, go = FALSE, nx, nc = 0, n_alloc = 50;
	double lon, lon_old, this_x, this_y, last_x, last_y, xlon[2], xlat[2];
	double get_angle ();
	struct XINGS *X;
	
	X = (struct XINGS *) memory (CNULL, n_alloc, sizeof (struct XINGS), "map_loncross");
		
	lon_old = west - SMALL;
	map_outside (lon_old, lat);
	geo_to_xy (lon_old, lat, &last_x, &last_y);
	for (i = 1; i <= gmtdefs.n_lon_nodes; i++) {
		lon = (i == gmtdefs.n_lon_nodes) ? east + SMALL : west + i * gmtdefs.dlon;
		map_outside (lon, lat);
		geo_to_xy (lon, lat, &this_x, &this_y);
		nx = 0;
		if ( break_through (lon_old, lat, lon, lat) ) {	/* Crossed map boundary */
			nx = map_crossing (lon_old, lat, lon, lat, xlon, xlat, X[nc].xx, X[nc].yy, X[nc].sides);
			if (nx == 1) X[nc].angle[0] = get_angle (lon_old, lat, lon, lat);
			if (nx == 2) X[nc].angle[1] = X[nc].angle[0] + 180.0;
			if (gmt_corner > 0) {
				X[nc].sides[0] = (gmt_corner%4 > 1) ? 1 : 3;
				gmt_corner = 0;
			}
		}
		if (gmt_world_map) wrap_around_check (X[nc].angle, last_x, last_y, this_x, this_y, X[nc].xx, X[nc].yy, X[nc].sides, &nx);
		if (nx == 2 && (fabs (X[nc].xx[1] - X[nc].xx[0]) - gmt_map_width) < SMALL && !gmt_world_map)
			go = FALSE;
		else if (nx > 0)
			go = TRUE;
		if (go) {
			X[nc].nx = nx;
			nc++;
			if (nc == n_alloc) {
				n_alloc += 50;
				X = (struct XINGS *) memory ((char *)X, n_alloc, sizeof (struct XINGS), "map_loncross");
			}
			go = FALSE;
		}
		lon_old = lon;
		last_x = this_x;	last_y = this_y;
	}
	
	if (nc > 0) {
		X = (struct XINGS *) memory ((char *)X, nc, sizeof (struct XINGS), "map_loncross");
		*xings = X;
	}
	else
		free ((char *)X);
	
	return (nc);
}

int map_loncross (lon, south, north, xings)
double lon, south, north;
struct XINGS *xings[]; {
	int go = FALSE, j, nx, nc = 0, n_alloc = 50;
	double lat, lat_old, this_x, this_y, last_x, last_y, xlon[2], xlat[2];
	double get_angle ();
	struct XINGS *X;
	
	X = (struct XINGS *) memory (CNULL, n_alloc, sizeof (struct XINGS), "map_loncross");
	
	lat_old = ((south - SMALL) >= -90.0) ? south - SMALL : south;	/* Outside */
	if ((north + SMALL) <= 90.0) north += SMALL;
	map_outside (lon, lat_old);
	geo_to_xy (lon, lat_old, &last_x, &last_y);
	for (j = 1; j <= gmtdefs.n_lat_nodes; j++) {
		lat = (j == gmtdefs.n_lat_nodes) ? north: south + j * gmtdefs.dlat;
		map_outside (lon, lat);
		geo_to_xy (lon, lat, &this_x, &this_y);
		nx = 0;
		if ( break_through (lon, lat_old, lon, lat) ) {	/* Crossed map boundary */
			nx = map_crossing (lon, lat_old, lon, lat, xlon, xlat, X[nc].xx, X[nc].yy, X[nc].sides);
			if (nx == 1) X[nc].angle[0] = get_angle (lon, lat_old, lon, lat);
			if (nx == 2) X[nc].angle[1] = X[nc].angle[0] + 180.0;
			if (gmt_corner > 0) {
				X[nc].sides[0] = (gmt_corner < 3) ? 0 : 2;
				gmt_corner = 0;
			}
		}
		if (gmt_world_map) wrap_around_check (X[nc].angle, last_x, last_y, this_x, this_y, X[nc].xx, X[nc].yy, X[nc].sides, &nx);
		if (nx == 2 && (fabs (X[nc].xx[1] - X[nc].xx[0]) - gmt_map_width) < SMALL && !gmt_world_map)
			go = FALSE;
		else if (nx > 0)
			go = TRUE;
		if (go) {
			X[nc].nx = nx;
			nc++;
			if (nc == n_alloc) {
				n_alloc += 50;
				X = (struct XINGS *) memory ((char *)X, n_alloc, sizeof (struct XINGS), "map_loncross");
			}
			go = FALSE;
		}
		lat_old = lat;
		last_x = this_x;	last_y = this_y;
	}
	
	if (nc > 0) {
		X = (struct XINGS *) memory ((char *)X, nc, sizeof (struct XINGS), "map_loncross");
		*xings = X;
	}
	else
		free ((char *)X);
	
	return (nc);
}

int map_gridlines (w, e, s, n)
double w, e, s, n; {
	int i, nx, ny;
	BOOLEAN dash;
	double dx, dy, w1, s1;
	
	if (gmtdefs.grid_cross_size > 0.0) return;
	
	dx = fabs (frame_info.grid_int[0]);
	dy = fabs (frame_info.grid_int[1]);
	
	dash = (!MAPPING && (dx > 0.0 || dy > 0.0));
	
	ps_setline (gmtdefs.grid_pen);
	if (dash) ps_setdash ("1 4", 0);
	if (dx > 0.0 && project_info.xyz_projection[0] == LOG10)
		logx_grid (w, e, s, n, dx);
	else if (dx > 0.0) {	/* Draw grid lines that go E to W */
		w1 = floor (w / dx) * dx;
		if ((w - w1) > SMALL) w1 += dx;
		nx = (w1 > e) ? -1 : (e - w1) / dx;
		for (i = 0; i <= nx; i++) map_lonline (w1 + i * dx, s, n);
	}
	
	if (dy > 0.0 && project_info.xyz_projection[1] == LOG10)
		logy_grid (w, e, s, n, dy);
	else if (dy > 0.0) {	/* Draw grid lines that go S to N */
		s1 = floor (s / dy) * dy;
		if ((s - s1) > SMALL) s1 += dy;
		ny = (s1 > n) ? -1 : (n - s1) / dy;
		for (i = 0; i <= ny; i++) map_latline (s1 + i * dy, w, e);
	}
	if (dash) ps_setdash ((char *)NULL, 0);
}

int map_gridcross (w, e, s, n)
double w, e, s, n; {
	int i, j, nx, ny;
	double dx, dy, w1, s1, lon, lat, x0, y0, x1, y1, xa, xb, ya, yb;
	double x_angle, y_angle, e1, n1, xt1, xt2, yt1, yt2, C, S, L;
	
	if (!MAPPING) return;
	if (gmtdefs.grid_cross_size <= 0.0) return;
	
	dx = fabs (frame_info.grid_int[0]);
	dy = fabs (frame_info.grid_int[1]);
	
	if (dx <= 0.0 || dy <= 0.0) return;
	
	ps_setline (gmtdefs.grid_pen);
	
	w1 = floor (w / dx) * dx;
	e1 = ceil (e / dx) * dx;
	nx = (e1 - w1) / dx;
	s1 = floor (s / dy) * dy;
	n1 = ceil (n / dy) * dy;
	ny = (n1 - s1) / dy;
	
	L = 0.5 * gmtdefs.grid_cross_size;
	
	for (i = 0; i <= nx; i++) {
		lon = w1 + i * dx;
		for (j = 0; j <= ny; j++) {
			lat = s1 + j * dy;
			
			if (!map_outside (lon, lat)) {	/* Inside map */
			
				geo_to_xy (lon, lat, &x0, &y0);
				geo_to_xy (lon + gmtdefs.dlon, lat, &x1, &y1);
				x_angle = d_atan2 (y1-y0, x1-x0);
				y_angle = x_angle + M_PI_2;
				C = cos (x_angle);
				S = sin (x_angle);
				xa = x0 - L * C;
				xb = x0 + L * C;
				ya = y0 - L * S;
				yb = y0 + L * S;
				
				/* Clip to map */
				
				if (xa < 0.0) xa = 0.0;
				if (xb < 0.0) xb = 0.0;
				if (ya < 0.0) ya = 0.0;
				if (yb < 0.0) yb = 0.0;
				if (xa > gmt_map_width) xa = gmt_map_width;
				if (xb > gmt_map_width) xb = gmt_map_width;
				if (ya > gmt_map_height) ya = gmt_map_height;
				if (yb > gmt_map_height) yb = gmt_map_height;
				
				/* 3-D projection */
				
				xy_do_z_to_xy (xa, ya, project_info.z_level, &xt1, &yt1);
				xy_do_z_to_xy (xb, yb, project_info.z_level, &xt2, &yt2);
				ps_plot (xt1, yt1, 3);
				ps_plot (xt2, yt2, -2);
				
				C = cos (y_angle);
				S = sin (y_angle);
				xa = x0 - L * C;
				xb = x0 + L * C;
				ya = y0 - L * S;
				yb = y0 + L * S;
				
				/* Clip to map */
				
				if (xa < 0.0) xa = 0.0;
				if (xb < 0.0) xb = 0.0;
				if (ya < 0.0) ya = 0.0;
				if (yb < 0.0) yb = 0.0;
				if (xa > gmt_map_width) xa = gmt_map_width;
				if (xb > gmt_map_width) xb = gmt_map_width;
				if (ya > gmt_map_height) ya = gmt_map_height;
				if (yb > gmt_map_height) yb = gmt_map_height;
				
				/* 3-D projection */
				
				xy_do_z_to_xy (xa, ya, project_info.z_level, &xt1, &yt1);
				xy_do_z_to_xy (xb, yb, project_info.z_level, &xt2, &yt2);
				ps_plot (xt1, yt1, 3);
				ps_plot (xt2, yt2, -2);
			}
		}
	}
}

int map_tickmarks (w, e, s, n)
double w, e, s, n; {
	int i, nx, ny;
	double dx, dy, w1, s1;
	
	if (!MAPPING || gmtdefs.basemap_type == 0) return;
	
	ps_setline (gmtdefs.tick_pen);
	dx = fabs (frame_info.frame_int[0]);
	dy = fabs (frame_info.frame_int[1]);
	
	on_border_is_outside = TRUE;	/* Temporarily, points on the border are outside */
	
	if (dx > 0.0 && dx != fabs (frame_info.anot_int[0])) {	/* Draw grid lines that go E to W */
		w1 = floor (w / dx) * dx;
		if (fabs (w1 - w) > SMALL) w1 += dx;
		nx = (w1 > e) ? -1 : (e - w1) / dx;
		for (i = 0; i <= nx; i++) map_lontick (w1 + i * dx, s, n);
	}
	
	if (dy > 0.0 && dy != fabs (frame_info.anot_int[1])) {	/* Draw grid lines that go S to N */
		s1 = floor (s / dy) * dy;
		if (fabs (s1 - s) > SMALL) s1 += dy;
		ny = (s1 > n) ? -1 : (n - s1) / dy;
		for (i = 0; i <= ny; i++) map_lattick (s1 + i * dy, w, e);
	}
	
	on_border_is_outside = FALSE;	/* Reset back to default */
}

int map_anotate (w, e, s, n)
double w, e, s, n; {
	double s1, w1, val, dx, dy, x, y;
	int do_minutes, do_seconds, move_up, i, nx, ny, done_zero = FALSE, anot, gmt_world_map_save;
	char label[80];
	
	if (frame_info.header[0]) {	/* Make plot header */
		move_up = (MAPPING || frame_info.side[2] == 2);
		ps_setfont (gmtdefs.header_font);
		x = project_info.xmax * 0.5;
		y = project_info.ymax + ((gmtdefs.tick_length > 0.0) ? gmtdefs.tick_length : 0.0) + 2.5 * gmtdefs.anot_offset;
		y += ((move_up) ? (gmtdefs.anot_font_size + gmtdefs.label_font_size) / gmt_ppu[gmtdefs.measure_unit] : 0.0) + 2.5 * gmtdefs.anot_offset;
		if (project_info.three_D && project_info.z_scale == 0.0) {	/* Only do this if flat 2-D plot */
			double size, xsize, ysize;
			
			ps_setfont (0);
			xy_do_z_to_xy (x, y, project_info.z_level, &x, &y);
			size = gmtdefs.header_font_size * gmtdefs.dpi / gmt_ppu[gmtdefs.measure_unit];
			xsize = size * z_project.xshrink[0];
			ysize = size * z_project.yshrink[0];
			printf ("/F0 {pop /%s findfont [%lg 0 %lg %lg 0 0] makefont setfont} bind def\n",
				font_name[gmtdefs.header_font], xsize, ysize * z_project.tilt[0], ysize);
			
			ps_text (x, y, gmtdefs.header_font_size, frame_info.header, z_project.phi[0], -2, 0);
			printf ("/F0 {/Helvetica Y} bind def\n");	/* Reset definition of F0 */
			ps_setfont (gmtdefs.header_font);
		}
		else if (!project_info.three_D)
			ps_text (x, y, gmtdefs.header_font_size, frame_info.header, 0.0, -2, 0);
	}
	
	if (!MAPPING) return;	/* Annotation already done by linear_axis */
	
	ps_setfont (gmtdefs.anot_font);
	ps_setline (gmtdefs.tick_pen);
	dx = (project_info.edge[0] || project_info.edge[2]) ? fabs (frame_info.anot_int[0]) : 0.0;
	dy = (project_info.edge[1] || project_info.edge[3]) ? fabs (frame_info.anot_int[1]) : 0.0;
	
	on_border_is_outside = TRUE;	/* Temporarily, points on the border are outside */
	gmt_world_map_save = gmt_world_map;
	if (project_info.region) gmt_world_map = FALSE;
	
	if (dx > 0.0) {	/* Anotate the S and N boundaries */
		BOOLEAN full_lat_range, proj_A, proj_B, anot_0_and_360;
		
		/* Determine if we should annotate both 0 and 360 degrees */
		
		full_lat_range = (fabs (180.0 - fabs (project_info.n - project_info.s)) < SMALL);
		proj_A = (project_info.projection == MERCATOR || project_info.projection == OBLIQUE_MERC ||
			project_info.projection == WINKEL || project_info.projection == ECKERT ||
			project_info.projection == ROBINSON || project_info.projection == CYL_EQ ||
			project_info.projection == CYL_EQDIST || project_info.projection == LINEAR);
		proj_B = (project_info.projection == HAMMER || project_info.projection == MOLLWEIDE ||
			project_info.projection == SINUSOIDAL);
/*		anot_0_and_360 = (gmt_world_map_save && ((full_lat_range && proj_A) || (!full_lat_range && proj_B))); */
		anot_0_and_360 = (gmt_world_map_save && (proj_A || (!full_lat_range && proj_B)));
		
		do_minutes = (fabs (fmod (dx, 1.0)) > SMALL);
		do_seconds = (fabs (60.0 * fmod (fmod (dx, 1.0) * 60.0, 1.0)) >= 1.0);
		w1 = floor (w / dx) * dx;
		if (fabs (w1 - w) > SMALL) w1 += dx;
		nx = (w1 > e) ? -1 : (e - w1) / dx;
		for (i = 0; i <= nx; i++) {
			val = w1 + i * dx;
			if (val == 0.0) done_zero = TRUE;
			get_anot_label (val, label, do_minutes, do_seconds, 0);
			anot = anot_0_and_360 || !(done_zero && val == 360.0);
			map_symbol_ns (val, label, s, n, anot);
		}
	}
	
	if (dy > 0.0) {	/* Anotate W and E boundaries */
		do_minutes = (fabs (fmod (dy, 1.0)) > SMALL);
		do_seconds = (fabs (60.0 * fmod (fmod (dy, 1.0) * 60.0, 1.0)) >= 1.0);
		s1 = floor (s / dy) * dy;
		if (fabs (s1 - s) > SMALL) s1 += frame_info.anot_int[1];
		ny = (s1 > n) ? -1: (n - s1) / dy;
		for (i = 0; i <= ny; i++) {
			val = s1 + i * frame_info.anot_int[1];
			if (project_info.polar && fabs (val) == 90.0) continue;
			get_anot_label (val, label, do_minutes, do_seconds, 1);
			map_symbol_ew (val, label, w, e, TRUE);
		}
	}
	
	on_border_is_outside = FALSE;	/* Reset back to default */
	if (project_info.region) gmt_world_map = gmt_world_map_save;
}

int map_boundary (w, e, s, n)
double w, e, s, n; {
	switch (project_info.projection) {
		case LINEAR:
			if (MAPPING)	/* xy is lonlat */
				merc_map_boundary (w, e, s, n);
			else if (project_info.three_D)
				basemap_3D (3);
			else
				linear_map_boundary (w, e, s, n);
			break;
		case POLAR:
			theta_r_map_boundary (w, e, s, n);
			break;
		case MERCATOR:
		case CYL_EQ:
		case CYL_EQDIST:
			merc_map_boundary (w, e, s, n);
			break;
		case LAMBERT:
			lambert_map_boundary (w, e, s, n);
			break;
		case OBLIQUE_MERC:
			oblmrc_map_boundary (w, e, s, n);
			break;
		case STEREO:
		case ORTHO:
		case LAMB_AZ_EQ:
		case AZ_EQDIST:
			if (project_info.polar)
				polar_map_boundary (w, e, s, n);
			else
				circle_map_boundary (w, e, s, n);
			break;
		case HAMMER:
		case MOLLWEIDE:
		case SINUSOIDAL:
			ellipse_map_boundary (w, e, s, n);
			break;
		case TM:
		case UTM:
		case ALBERS:
		case CASSINI:
		case WINKEL:
		case ECKERT:
		case ROBINSON:
			basic_map_boundary (w, e, s, n);
			break;
	}
	
	if (project_info.three_D) vertical_axis (3);
}

/* map_basemap will create a basemap for the given area. 
 * Scaling and wesn are assumed to be passed throught the project_info-structure (see gmt_project.h)
 * Tickmark info are passed through the frame_info-structure
 *
 */
 
int map_basemap () {
	double w, e, s, n;

	w = project_info.w;	e = project_info.e;	s = project_info.s;	n = project_info.n;
	
	ps_comment ("Start of basemap");
	
	ps_comment ("Map gridlines");
	map_gridlines (w, e, s, n);
	map_gridcross (w, e, s, n);
	
	ps_comment ("Map tickmarks");
	map_tickmarks (w, e, s, n);
	
	ps_comment ("Map anotations");
	map_anotate (w, e, s, n);
	
	ps_comment ("Map boundaries");
	map_boundary (w, e, s, n);
	
	ps_comment ("End of basemap");
	
	return (0);
}

int basemap_3D (mode)
int mode; {
	/* Mode means: 1 = background axis, 2 = foreground axis, 3 = all */
	BOOLEAN go[4], back;
	int i;
	
	back = (mode % 2);
	for (i = 0; i < 4; i++) go[i] = (mode == 3) ? TRUE : ((back) ? z_project.draw[i] : !z_project.draw[i]);
	
	if (go[0] && frame_info.side[0])	/* South or lower x-axis */
		xyz_axis3D (0, 'x', frame_info.anot_int[0], frame_info.frame_int[0],
		frame_info.label[0], project_info.xyz_projection[0], frame_info.anot_type[0], frame_info.side[0]-1);
	
	if (go[2] && frame_info.side[2])	/* North or upper x-axis */
		xyz_axis3D (2, 'x', frame_info.anot_int[0], frame_info.frame_int[0],
		frame_info.label[0], project_info.xyz_projection[0], frame_info.anot_type[0], frame_info.side[2]-1);
	
	if (go[3] && frame_info.side[3])	/* West or left y-axis */
		xyz_axis3D (3, 'y', frame_info.anot_int[1], frame_info.frame_int[1],
		frame_info.label[1], project_info.xyz_projection[1], frame_info.anot_type[1], frame_info.side[3]-1);
			
	if (go[1] && frame_info.side[1])	/* East or right y-axis */
		xyz_axis3D (1, 'y', frame_info.anot_int[1], frame_info.frame_int[1],
		frame_info.label[1], project_info.xyz_projection[1], frame_info.anot_type[1], frame_info.side[1]-1);
		
}

int vertical_axis (mode)
int mode; {
	/* Mode means: 1 = background axis, 2 = foreground axis, 3 = all */
	BOOLEAN go[4], fore, back;
	int i, j;
	double xp[2], yp[2];
	
	if (frame_info.anot_int[2] == 0.0) return;

	fore = (mode > 1);	back = (mode % 2);
	for (i = 0; i < 4; i++) go[i] = (mode == 3) ? TRUE : ((back) ? z_project.draw[i] : !z_project.draw[i]);
	
	/* Vertical */
		
	if (fore && frame_info.side[4]) xyz_axis3D (z_project.z_axis, 'z', frame_info.anot_int[2], frame_info.frame_int[2], frame_info.label[2],
		project_info.xyz_projection[2], frame_info.anot_type[2], frame_info.side[4]-1);
			
	if (frame_info.draw_box) {
		go[0] = ( (back && z_project.quadrant == 1) || (fore && z_project.quadrant != 1) );
		go[1] = ( (back && z_project.quadrant == 4) || (fore && z_project.quadrant != 4) );
		go[2] = ( (back && z_project.quadrant == 3) || (fore && z_project.quadrant != 3) );
		go[3] = ( (back && z_project.quadrant == 2) || (fore && z_project.quadrant != 2) );
		for (i = 0; i < 4; i++) {
			if (!go[i]) continue;
			geoz_to_xy (z_project.corner_x[i], z_project.corner_y[i], project_info.z_bottom, &xp[0], &yp[0]);
			geoz_to_xy (z_project.corner_x[i], z_project.corner_y[i], project_info.z_top, &xp[1], &yp[1]);
			ps_line (xp, yp, 2, 3, FALSE, TRUE);
		}
		go[0] = ( (back && (z_project.quadrant == 1 || z_project.quadrant == 4)) || (fore && (z_project.quadrant == 2 || z_project.quadrant == 3)) );
		go[1] = ( (back && (z_project.quadrant == 3 || z_project.quadrant == 4)) || (fore && (z_project.quadrant == 1 || z_project.quadrant == 2)) );
		go[2] = ( (back && (z_project.quadrant == 2 || z_project.quadrant == 3)) || (fore && (z_project.quadrant == 1 || z_project.quadrant == 4)) );
		go[3] = ( (back && (z_project.quadrant == 1 || z_project.quadrant == 2)) || (fore && (z_project.quadrant == 3 || z_project.quadrant == 4)) );
		for (i = 0; i < 4; i++) {
			if (!go[i]) continue;
			j = (i + 1) % 4;
			geoz_to_xy (z_project.corner_x[i], z_project.corner_y[i], project_info.z_top, &xp[0], &yp[0]);
			geoz_to_xy (z_project.corner_x[j], z_project.corner_y[j], project_info.z_top, &xp[1], &yp[1]);
			ps_line (xp, yp, 2, 3, FALSE, TRUE);
		}
	}
	if (back && frame_info.header[0]) {
		ps_setfont (gmtdefs.header_font);
		xp[0] = 0.5 * (z_project.xmin + z_project.xmax);
		yp[0] = z_project.ymax + 0.5;
		ps_text (xp[0], yp[0], gmtdefs.header_font_size, frame_info.header, 0.0, -2, 0);
	}
}

int xyz_axis3D (axis_no, axis, anotation_int, tickmark_int, label, axistype, anottype, anotate)
double anotation_int, tickmark_int;
char *label, axis;
int axis_no, axistype, anottype, anotate; {
	int i, j, i_a, i_f, k, id, justify, n_anotations = 0, n_tickmarks = 0, test;
	
	BOOLEAN do_anot, do_tick;
	
	double val, v0, v1, anot_off, label_off, start_val_a, start_val_f, end_val;
	double tvals_a[9], tvals_f[9], sign, dy, tmp, xyz[3][2], len, x0, x1, y0, y1;
	double pp[3], w[3], xp, yp, del_y, val_xyz[3], phi, size, xsize, ysize;
	double start_log_a, start_log_f, val0, val1;
	
	PFI xyz_forward, xyz_inverse;
	
	char annotation[80], format[20];
	
	id = (axis == 'x') ? 0 : ((axis == 'y') ? 1 : 2);
	j = (id == 0) ? 1 : ((id == 1) ? 0 : z_project.k);
	xyz_forward = (PFI) ((id == 0) ? x_to_xx : ((id == 1) ? y_to_yy : z_to_zz));
	xyz_inverse = (PFI) ((id == 0) ? xx_to_x : ((id == 1) ? yy_to_y : zz_to_z));
	phi = (id < 2 && axis_no > 1) ? z_project.phi[id] + 180.0 : z_project.phi[id];
	
	/* Get projected anchor point */
	
	if (id == 2) {
		geoz_to_xy (z_project.corner_x[axis_no], z_project.corner_y[axis_no], project_info.z_bottom, &x0, &y0);
		k = axis_no;
		geoz_to_xy (z_project.corner_x[axis_no], z_project.corner_y[axis_no], project_info.z_top, &x1, &y1);
		if (j == 0)
			sign = z_project.sign[z_project.z_axis];
		else
			sign = (z_project.z_axis%2) ? -z_project.sign[z_project.z_axis] : z_project.sign[z_project.z_axis];
	}
	else {
		geoz_to_xy (z_project.corner_x[axis_no], z_project.corner_y[axis_no], project_info.z_level, &x0, &y0);
		k = (axis_no + 1) % 4;
		geoz_to_xy (z_project.corner_x[k], z_project.corner_y[k], project_info.z_level, &x1, &y1);
		sign = z_project.sign[axis_no];
	}
	xyz[0][0] = project_info.w;		xyz[0][1] = project_info.e;
	xyz[1][0] = project_info.s;		xyz[1][1] = project_info.n;
	xyz[2][0] = project_info.z_bottom;	xyz[2][1] = project_info.z_top;
	
	size = gmtdefs.anot_font_size * gmtdefs.dpi / gmt_ppu[gmtdefs.measure_unit];
	xsize = size * z_project.xshrink[id];
	ysize = size * z_project.yshrink[id];
	ps_comment ("Start of xyz-axis3D");
	/* Temporarely modify meaning of F0 */
	printf ("/F0 {pop /%s findfont [%lg 0 %lg %lg 0 0] makefont setfont} bind def\n",
		font_name[gmtdefs.anot_font], xsize, ysize * z_project.tilt[id], ysize);
	ps_setfont (0);
	justify = (id == 2) ? 2 : 10;
	dy = sign * gmtdefs.tick_length;
	len = (gmtdefs.tick_length > 0.0) ? gmtdefs.tick_length : 0.0;
	anotation_int = fabs (anotation_int);
	tickmark_int = fabs (tickmark_int);
	
	do_anot = (anotation_int > 0.0);
	do_tick = (tickmark_int > 0.0);
	val0 = xyz[id][0];
	val1 = xyz[id][1];
	
	
	/* Find number of decimals needed, if any */
	
	get_format (anotation_int, format);

	anot_off = sign * (len + gmtdefs.anot_offset);
	label_off = sign * (len + 2.5 * gmtdefs.anot_offset + (gmtdefs.anot_font_size / gmt_ppu[gmtdefs.measure_unit]) * font_height[gmtdefs.anot_font]);
	
	/* Ready to draw axis */
	
	ps_setline (gmtdefs.frame_pen);
	ps_plot (x0, y0, 3);
	ps_plot (x1, y1, -2);
	ps_setline (gmtdefs.tick_pen);
	
	i_a = i_f = 0;
	
	switch (axistype) {
		case POW:	/* Anotate in pow(x) */
			(*xyz_forward) (val0, &v0);
			(*xyz_forward) (val1, &v1);
			if (anottype == 2) {
				val = (anotation_int == 0.0) ? 0.0 : floor (v0 / anotation_int) * anotation_int;
				if (fabs (val - v0) > SMALL) val += anotation_int;
				start_val_a = val;	end_val = v1;
				val = (tickmark_int == 0.0) ? 0.0 : floor (v0 / tickmark_int) * tickmark_int;
				if (fabs (val - v0) > SMALL) val += tickmark_int;
				start_val_f = val;
			}
			else {
				val = (anotation_int == 0.0) ? 0.0 : floor (val0 / anotation_int) * anotation_int;
				if (fabs (val - val0) > SMALL) val += anotation_int;
				start_val_a = val;	end_val = val1;
				val = (tickmark_int == 0.0) ? 0.0 : floor (val0 / tickmark_int) * tickmark_int;
				if (fabs (val - val0) > SMALL) val += tickmark_int;
				start_val_f = val;
			}
			break;
		case LOG10:	/* Anotate in d_log10 (x) */
			v0 = d_log10 (val0);
			v1 = d_log10 (val1);
			val = pow (10.0, floor (v0));
			test = rint (anotation_int) - 1;
			if (test < 0 || test > 2) test = 0;
			if (test == 0) {
				n_anotations = 1;
				tvals_a[0] = 10.0;
				if (fabs (val - val0) > SMALL) val *= 10.0;
			}
			else if (test == 1) {
				tvals_a[0] = 1.0;
				tvals_a[1] = 2.0;
				tvals_a[2] = 5.0;
				n_anotations = 3;
			}
			else if (test == 2) {
				n_anotations = 9;
				for (i = 0; i < n_anotations; i++) tvals_a[i] = i + 1;
			}
			test = rint (tickmark_int) - 1;
			if (test < 0 || test > 2) test = 0;
			if (test == 0) {
				n_tickmarks = 1;
				tvals_f[0] = 10.0;
			}
			else if (test == 1) {
				tvals_f[0] = 1.0;
				tvals_f[1] = 2.0;
				tvals_f[2] = 5.0;
				n_tickmarks = 3;
			}
			else if (test == 2) {
				n_tickmarks = 9;
				for (i = 0; i < n_tickmarks; i++) tvals_f[i] = i + 1;
			}
			i_a = 0;
			start_log_a = val = pow (10.0, floor (v0));
			while ((val0 - val) > SMALL) {
				if (i_a < n_anotations)
					val = start_log_a * tvals_a[i_a];
				else {
					val = (start_log_a *= 10.0);
					i_a = 0;
				}
				i_a++;
			}
			i_a--;
			start_val_a = val;
			i_f = 0;
			start_log_f = val = pow (10.0, floor (v0));
			while ((val0 - val) > SMALL) {
				if (i_f < n_tickmarks)
					val = start_log_f * tvals_f[i_f];
				else {
					val = (start_log_f *= 10.0);
					i_f = 0;
				}
				i_f++;
			}
			i_f--;
			start_val_f = val;
			end_val = val1;
			break;
		case LINEAR:
			v0 = val0;
			v1 = val1;
			val = (anotation_int == 0.0) ? 0.0 : floor (val0 / anotation_int) * anotation_int;
			if (fabs (val - val0) > SMALL) val += anotation_int;
			start_val_a = val;	end_val = val1;
			val = (tickmark_int == 0.0) ? 0.0 : floor (val0 / tickmark_int) * tickmark_int;
			if (fabs (val - val0) > SMALL) val += tickmark_int;
			start_val_f = val;
			break;
	}
	
	del_y = 0.5 * sign * gmtdefs.anot_font_size * 0.732 * (justify/4) / gmt_ppu[gmtdefs.measure_unit];
	
	/* Do anotations with tickmarks */
	
	val_xyz[0] = z_project.corner_x[axis_no];
	val_xyz[1] = z_project.corner_y[axis_no];
	val_xyz[2] = project_info.z_level;
	val = (anotation_int == 0.0) ? end_val + 1.0 : start_val_a;
	while (do_anot && val <= end_val) {
	
		i_a++;
		
		val_xyz[id] = val;
		
		switch (anottype) {
			case 0:
				sprintf (annotation, format, val_xyz[id]);
				break;
			case 1:
				sprintf (annotation, "%d\0", (int) d_log10 (val_xyz[id]));
				break;
			case 2:
				if (axistype == POW) {
					(*xyz_inverse) (&tmp, val_xyz[id]);
					val_xyz[id] = tmp;
					sprintf (annotation, format, val_xyz[id]);
				}
				else
					sprintf (annotation, "10@+%d@+\0", (int) d_log10 (val_xyz[id]));
				break;
		}
		
		project3D (val_xyz[0], val_xyz[1], val_xyz[2], &w[0], &w[1], &w[2]);
		pp[0] = w[0];
		pp[1] = w[1];
		pp[2] = w[2];
		xyz_to_xy (pp[0], pp[1], pp[2], &xp, &yp);
		ps_plot (xp, yp, 3);
		pp[j] += dy;
		xyz_to_xy (pp[0], pp[1], pp[2], &xp, &yp);
		ps_plot (xp, yp, -2);
		pp[j] += anot_off -dy + del_y;
		xyz_to_xy (pp[0], pp[1], pp[2], &xp, &yp);
		if (anotate) {
			if (id < 2)
				ps_text (xp, yp, gmtdefs.anot_font_size, annotation, phi, 2, 0);
			else if (val != project_info.z_level)
				ps_text (xp, yp, gmtdefs.anot_font_size, annotation, phi, 2, 0);
		}
		
		if (axistype == LOG10) {
			if (i_a < n_anotations)
				val = start_log_a * tvals_a[i_a];
			else {
				val = (start_log_a *= 10.0);
				i_a = 0;
			}
		}
		else
			val = start_val_a + i_a * anotation_int;
			
	}

	/* Now do frame tickmarks */
	
	dy *= 0.5;
	
	val_xyz[0] = z_project.corner_x[axis_no];
	val_xyz[1] = z_project.corner_y[axis_no];
	val_xyz[2] = project_info.z_level;
	val = (tickmark_int == 0.0) ? end_val + 1.0 : start_val_f;
	while (do_tick && val <= end_val) {
	
		i_f++;
		
		val_xyz[id] = val;
		if (anottype == 2 && axistype == POW) {
			(*xyz_inverse) (&tmp, val_xyz[id]);
			val_xyz[id] = tmp;
		}
		project3D (val_xyz[0], val_xyz[1], val_xyz[2], &w[0], &w[1], &w[2]);
				
		pp[0] = w[0];
		pp[1] = w[1];
		pp[2] = w[2];
		xyz_to_xy (pp[0], pp[1], pp[2], &xp, &yp);
		ps_plot (xp, yp, 3);
		pp[j] += dy;
		xyz_to_xy (pp[0], pp[1], pp[2], &xp, &yp);
		ps_plot (xp, yp, -2);
		
		if (axistype == LOG10) {
			if (i_f < n_tickmarks)
				val = start_log_f * tvals_f[i_f];
			else {
				val = start_log_f *= 10.0;
				i_f = 0;
			}
		}
		else
			val = start_val_f + i_f * tickmark_int;
	}

	/* Finally do label */
	
	if (label && anotate) {
		val_xyz[0] = z_project.corner_x[axis_no];
		val_xyz[1] = z_project.corner_y[axis_no];
		val_xyz[2] = project_info.z_level;
		size = gmtdefs.label_font_size * gmtdefs.dpi / gmt_ppu[gmtdefs.measure_unit];
		xsize = size * z_project.xshrink[id];
		ysize = size * z_project.yshrink[id];
		printf ("/F0 {pop /%s findfont [%lg 0 %lg %lg 0 0] makefont setfont} bind def\n",
			font_name[gmtdefs.label_font], xsize, ysize * z_project.tilt[id], ysize);
		project3D (val_xyz[0], val_xyz[1], val_xyz[2], &w[0], &w[1], &w[2]);
		x0 = w[id];
		val_xyz[id] = (val_xyz[id] == xyz[id][0]) ? xyz[id][1] : xyz[id][0];
		project3D (val_xyz[0], val_xyz[1], val_xyz[2], &w[0], &w[1], &w[2]);
		x1 = w[id];
		pp[0] = w[0];
		pp[1] = w[1];
		pp[2] = w[2];
		pp[id] = 0.5 * (x1 + x0);
		pp[j] += label_off + del_y;
		xyz_to_xy (pp[0], pp[1], pp[2], &xp, &yp);
	
		ps_text (xp, yp, gmtdefs.label_font_size, label, phi, 2, 0);
		printf ("/F0 {/Helvetica Y} bind def\n");	/* Reset definition of F0 */
	}
	ps_comment ("End of xyz-axis3D");
}

int map_clip_on (r, g, b, flag)
int r, g, b, flag; {
	/* This function sets up a clip path so that only plotting
	 * inside the map area will be drawn on paper. map_setup
	 * must have been called first.  If r >= 0, the map area will
	 * first be painted in the r,g,b colors specified.  flag can
	 * be 0-3, as described in ps_clipon().
	 */
	 
	double *work_x, *work_y;
	int np;
	BOOLEAN donut;
	
	np = map_clip_path (&work_x, &work_y, &donut);
		
	if (donut) {
		ps_clipon (work_x, work_y, np, r, g, b, 1);
		ps_clipon (&work_x[np], &work_y[np], np, r, g, b, 2);
	}
	else
		ps_clipon (work_x, work_y, np, r, g, b, flag);
	
	free ((char *)work_x);
	free ((char *)work_y);
}

int map_clip_off () {
	/* Restores the original clipping path (or pours white paint) */
	
	ps_clipoff ();
}

int geoplot (lon, lat, pen)
double lon, lat;
int pen; {
	/* Computes x/y from lon/lat, then calls plot */
	double x, y;
	
	geo_to_xy (lon, lat, &x, &y);
	ps_plot (x, y, pen);
}

int timestamp (argc, argv)
int argc;
char **argv; {
	time_t right_now, time();
	int i, echo_command = FALSE;
	char *ctime(), label[512];
	double x, y, dim[5];
	
	dim[0] = 0.365; dim[1] = 0.7215; dim[2] = 0.15; dim[3] = 0.075; dim[4] = 0.1;
	if (gmtdefs.measure_unit == 0) for (i = 0; i < 5; i++) dim[i] *= 2.54;
	x = gmtdefs.unix_time_pos[0];
	y = gmtdefs.unix_time_pos[1];
	right_now = time ((time_t *)0);
	strcpy (label, ctime (&right_now));
	label[16] = 0;
	for (i = 1; i < argc && argv[i][1] != 'J'; i++);
	ps_comment (" ");
	ps_comment ("Begin time-stamp");
	ps_transrotate (x, y, 0.0);
	ps_setline (1);
	ps_rect (0.0, 0.0, dim[0]+dim[1], dim[2], gmtdefs.foreground_rgb[0], gmtdefs.foreground_rgb[1], gmtdefs.foreground_rgb[2], TRUE);
	ps_rect (0.0, 0.0, dim[0], dim[2], gmtdefs.background_rgb[0], gmtdefs.background_rgb[1], gmtdefs.background_rgb[2], TRUE);
	ps_setfont (1);
	ps_setpaint (gmtdefs.foreground_rgb[0], gmtdefs.foreground_rgb[1], gmtdefs.foreground_rgb[2]);
	ps_text (0.5*dim[0], dim[3], 10, "GMT", 0.0, 6, 0);
	ps_setfont (0);
	ps_setpaint (gmtdefs.background_rgb[0], gmtdefs.background_rgb[1], gmtdefs.background_rgb[2]);
	ps_text (dim[0]+0.5*dim[1], dim[3], 8, &label[4], 0.0, 6, 0);
	ps_setfont (1);
	label[0] = 0;
	if (gmtdefs.unix_time_label[0] == 'c' && gmtdefs.unix_time_label[1] == 0) {
		echo_command = TRUE;
		gmtdefs.unix_time_label[0] = 0;
	}
	if (echo_command) {
		strcpy (label, argv[0]);
		for (i = 1; i < argc; i++) {
			if (argv[i][0] != '-') continue;
			strcat (label, " ");
			strcat (label, argv[i]);
		}
	}
	else if (gmtdefs.unix_time_label[0])
		strcpy (label, gmtdefs.unix_time_label);
		
	if (label[0]) ps_text (dim[0]+dim[1]+dim[4], dim[3], 7, label, 0.0, 5, 0);
	ps_rotatetrans  (-x, -y, 0.0);
	ps_comment ("End time-stamp\n");
}

int echo_command (argc, argv)
int argc;
char **argv; {
	/* This routine will echo the command and its arguments to the
	 * PostScript output file so that the user can see what scales
	 * etc was used to produce this plot
	 */
	int i, length = 0;
	char outstring[512];
	
	ps_comment ("PostScript produced by:\n");
	outstring[0] = 0;
	for (i = 0; i < argc; i++) {
		strcat (outstring, argv[i]);
		strcat (outstring, " ");
		length += (strlen (argv[i]) + 1);
		if (length >= 80) {
			ps_comment (outstring);
			length = 0;
			outstring[0] = 0;
		}
	}
	if (length > 0) ps_comment (outstring);
	ps_comment ("\n");
}

int gmt_plot_line (x, y, pen, n)
double x[], y[];
int pen[], n; {
	int i, j, i1, way, stop, close;
	double x_cross[2], y_cross[2], *xx, *yy, xt1, yt1, xt2, yt2;
	
	if (n < 2) return;
	
	i = 0;
	while (i < (n-1) && pen[i+1] == 3) i++;	/* Skip repeating pen == 3 in beginning */
	if ((n-i) < 2) return;
	while (n > 1 && pen[n-1] == 3) n--;	/* Cut off repeating pen == 3 at end */
	if ((n-i) < 2) return;
	
	for (j = i + 1; j < n && pen[j] == 2; j++);	/* j == n means no moveto's present */
	close = (j == n) ? (hypot (x[n-1] - x[i], y[n-1] - y[i]) < SMALL) : FALSE;
	
	/* First see if we can use the ps_line call directly to save points */
	
	for (j = i + 1, stop = FALSE; !stop && j < n; j++) stop = (pen[j] == 3 || map_jump (x[j-1], y[j-1], x[j], y[j]));
	if (!stop) {
		if (project_info.three_D) {	/* Must project first */
			xx = (double *) memory (CNULL, n-i, sizeof (double), "gmt_plot_line");
			yy = (double *) memory (CNULL, n-i, sizeof (double), "gmt_plot_line");
			for (j = i; j < n; j++) xy_do_z_to_xy (x[j], y[j], project_info.z_level, &xx[j], &yy[j]);
			ps_line (&xx[i], &yy[i], n - i, 3, close, TRUE);
			free ((char *)xx);
			free ((char *)yy);
		}
		else
			ps_line (&x[i], &y[i], n - i, 3, close, TRUE);
		return;
	}
	
	/* Here we must check for jumps, pen changes etc */
	
	if (project_info.three_D) {
		xy_do_z_to_xy (x[i], y[i], project_info.z_level, &xt1, &yt1);
		ps_plot (xt1, yt1, pen[i]);
	}
	else
		ps_plot (x[i], y[i], pen[i]);

	i++;
	while (i < n) {
		i1 = i - 1;
		if (pen[i] == pen[i1] && (way = map_jump (x[i1], y[i1], x[i], y[i]))) {	/* Jumped across the map */
			get_crossings (x_cross, y_cross, x[i1], y[i1], x[i], y[i]);
			xy_do_z_to_xy (x_cross[0], y_cross[0], project_info.z_level, &xt1, &yt1);
			xy_do_z_to_xy (x_cross[1], y_cross[1], project_info.z_level, &xt2, &yt2);
			if (project_info.three_D) {
				xy_do_z_to_xy (xt1, yt1, project_info.z_level, &xt1, &yt1);
				xy_do_z_to_xy (xt2, yt2, project_info.z_level, &xt2, &yt2);
			}
			if (way == -1) {	/* Add left border point */
				ps_plot (xt1, yt1, 2);
				ps_plot (xt2, yt2, 3);
			}
			else {
				ps_plot (xt2, yt2, 2);
				ps_plot (xt1, yt1, 3);
			}
			close = FALSE;
		}
		if (project_info.three_D) {
			xy_do_z_to_xy (x[i], y[i], project_info.z_level, &xt1, &yt1);
			ps_plot (xt1, yt1, pen[i]);
		}
		else
			ps_plot (x[i], y[i], pen[i]);
		i++;
	}
	if (close) ps_command ("P S") ; else ps_command ("S");
}

int color_image (x0, y0, x_side, y_side, image, nx, ny)
double x0, y0;		/* Lower left corner in inches */
double x_side, y_side;	/* Size of cell in inches */
unsigned char *image;	/* color image  */
int nx, ny; {		/* image size */
	
	/* Call the appropriate image filler (see pslib) */
	
	switch (gmtdefs.color_image) {
		case 0:
			ps_colorimage (x0, y0, x_side, y_side, image, nx, ny);
			break;
		case 1:
			ps_colortiles (x0, y0, x_side, y_side, image, nx, ny);
			break;
		case 2:
			ps_imagergb (x0, y0, x_side, y_side, image, nx, ny);
			break;
	}
}

int gmt_text3d (x, y, z, fsize, fontno, text, angle, justify, form)
double x, y, z, angle;
int fsize, fontno, justify, form;
char *text; {
	double xb, yb, xt, yt, xt1, xt2, xt3, yt1, yt2, yt3, del_y;
	double ca, sa, xshrink, yshrink, tilt, baseline_shift, size, xsize, ysize;
        if (project_info.three_D) {
                ps_setfont (0);
                justify = abs (justify);
                del_y = 0.5 * fsize * 0.732 * (justify / 4) / gmt_ppu[gmtdefs.measure_unit];
                justify %= 4;
                ca = cosd (angle);
                sa = sind (angle);
                x += del_y * sa;	/* Move anchor point down on baseline */
                y -= del_y * ca;
                xb = x + ca;		/* Point a distance of 1.0 along baseline */
                yb = y + sa;
                xt = x - sa;		/* Point a distance of 1.0 normal to baseline */
                yt = y + ca;
                xyz_to_xy (x, y, z, &xt1, &yt1);
                xyz_to_xy (xb, yb, z, &xt2, &yt2);
                xyz_to_xy (xt, yt, z, &xt3, &yt3);
		xshrink = hypot (xt2-xt1, yt2-yt1) / hypot (xb-x, yb-y);	/* How lines in baseline-direction shrink */
		yshrink = hypot (xt3-xt1, yt3-yt1) / hypot (xt-x, yt-y);	/* How lines _|_ to baseline-direction shrink */
		baseline_shift = R2D * (d_atan2 (yt2 - yt1, xt2 - xt1) - d_atan2 (yb - y, xb - x));	/* Rotation of baseline */
		tilt = 90.0 - R2D * (d_atan2 (yt3 - yt1, xt3 - xt1) - d_atan2 (yt2 - yt1, xt2 - xt1));
		tilt = tand (tilt);
		size = fsize * gmtdefs.dpi / gmt_ppu[gmtdefs.measure_unit];
		xsize = size * xshrink;
		ysize = size * yshrink;
		/* Temporarely modify meaning of F0 */
		printf ("/F0 {pop /%s findfont [%lg 0 %lg %lg 0 0] makefont setfont} bind def\n",
			font_name[fontno], xsize, ysize * tilt, ysize);

                ps_text (xt1, yt1, fsize, text, angle + baseline_shift, justify, form);
                printf ("/F0 {/Helvetica Y} bind def\n");       /* Reset definition of F0 */
                ps_setfont (fontno);
        }
        else {
		ps_setfont (fontno);
		ps_text (x, y, fsize, text, angle, justify, form);
	}
}

int gmt_textbox3d (x, y, z, size, font, label, angle, just, outline, dx, dy, r, g, b)
double x, y, z, angle, dx, dy;
int size, font, just, r, g, b;
char *label;
BOOLEAN outline; {
        if (project_info.three_D) {
        	int i, len, ndig = 0, ndash = 0, nperiod = 0;
        	double xx[4], yy[4], h, w, xa, ya, cosa, sina;
		len = strlen (label);
		for (i = 0; label[i]; i++) {
			if (isdigit ((int)label[i])) ndig++;
			if (strchr (label, '.')) nperiod++;
			if (strchr (label, '-')) ndash++;
		}
		len -= (ndig + nperiod + ndash);
		w = ndig * 0.78 + nperiod * 0.38 + ndash * 0.52 + len;
		
		h = 0.58 * font_height[font] * size / gmt_ppu[gmtdefs.measure_unit];
		w *= (0.81 * h);
		just = abs (just);
		y -= (((just/4) - 1) * h);
		x -= (((just-1)%4 - 1) * w);
		xx[0] = xx[3] = -w - dx;
		xx[1] = xx[2] = w + dx;
		yy[0] = yy[1] = -h - dy;
		yy[2] = yy[3] = h + dy;
		cosa = cosd (angle);	sina = sind (angle);
		for (i = 0; i < 4; i++) {
			xa = xx[i] * cosa - yy[i] * sina;
			ya = xx[i] * sina + yy[i] * cosa;
			xx[i] = x + xa;	yy[i] = y + ya;
		}
		d_swap (z, project_info.z_level); 
		two_d_to_three_d (xx, yy, 4);
		d_swap (z, project_info.z_level);
		if (r < 0)
			ps_clipon (xx, yy, 4, r, g, b, 0);
		else
			ps_patch (xx, yy, 4, r, g, b, outline);
	}
	else
		ps_textbox (x, y, size, label, angle, just, outline, dx, dy, r, g, b);
}

int gmt_vector3d (x0, y0, x1, y1, z0, tailwidth, headlength, headwidth, shape, r, g, b, outline)
double x0, y0, x1, y1, z0, tailwidth, headlength, headwidth, shape;
int r, g, b;
BOOLEAN outline; {
	int i;
	double xx[7], yy[7], dx, dy, angle, length;
	
	if (project_info.three_D) {
		angle = atan2 (y1 - y0, x1 - x0);
		length = hypot (y1 - y0, x1 - x0);
		xx[3] = x0 + length * cos (angle);
		yy[3] = y0 + length * sin (angle);
		dx = 0.5 * tailwidth * sin (angle);
		dy = 0.5 * tailwidth * cos (angle);
		xx[0] = x0 + dx;	xx[6] = x0 - dx;
		yy[0] = y0 - dy;	yy[6] = y0 + dy;
		dx = (length - (1.0 - 0.5 * shape) * headlength) * cos (angle);
		dy = (length - (1.0 - 0.5 * shape) * headlength) * sin (angle);
		xx[1] = xx[0] + dx;	xx[5] = xx[6] + dx;
		yy[1] = yy[0] + dy;	yy[5] = yy[6] + dy;
		x0 += (length - headlength) * cos (angle);
		y0 += (length - headlength) * sin (angle);
		dx = headwidth * sin (angle);
		dy = headwidth * cos (angle);
		xx[2] = x0 + dx;	xx[4] = x0 - dx;
		yy[2] = y0 - dy;	yy[4] = y0 + dy;
		for (i = 0; i < 7; i++) {
			xyz_to_xy (xx[i], yy[i], z0, &x0, &y0);
			xx[i] = x0;
			yy[i] = y0;
		}
		ps_polygon (xx, yy, 7, r, g, b, outline);
	}
	else
		ps_vector (x0, y0, x1, y1, tailwidth, headlength, headwidth, gmtdefs.vector_shape, r, g, b, outline);
}

int prepare_label (angle, side, x, y, type, line_angle, text_angle, justify)
double angle, x, y, *line_angle, *text_angle;
int side, type, *justify; {
	BOOLEAN set_angle;
		
	if (!project_info.edge[side]) return -1;		/* Side doesnt exist */
	if (frame_info.side[side] < 2) return -1;	/* Dont want labels here */
		
	if (frame_info.check_side == TRUE) {
		if (type == 0 && side%2) return -1;
		if (type == 1 && !(side%2)) return -1;
	}
	
	if (frame_info.horizontal == 2 && !(side%2)) angle = -90.0;	/* get_label_parameters will make this 0 */
	
	if (angle < 0.0) angle += 360.0;
	
	set_angle = ((project_info.region && !(AZIMUTHAL || CONICAL)) || !project_info.region);
	if (set_angle) {
		if (side == 0 && angle < 180.0) angle -= 180.0;
		if (side == 1 && (angle > 90.0 && angle < 270.0)) angle -= 180.0;
		if (side == 2 && angle > 180.0) angle -= 180.0;
		if (side == 3 && (angle < 90.0 || angle > 270.0)) angle -= 180.0;
	}
	
	if (!get_label_parameters (side, angle, text_angle, justify)) return -1;
	*line_angle = angle;
	
	if (!set_angle) *justify = polar_adjust (side, angle, x, y);
	
	return 0;
}

int get_anot_label (val, label, do_minutes, do_seconds, lonlat)
double val;	/* Degree value of anotation */
char *label;	/* String to hold the final anotation */
int do_minutes;	/* TRUE if degree and minutes are desired, FALSE for just integer degrees */
int do_seconds;	/* TRUE if degree, minutes, and seconds are desired */
int lonlat; {	/* 0 = longitudes, 1 = latitudes */
	int ival, minutes, seconds;
	BOOLEAN zero_fix = FALSE;
	char letter = 0;
	
	if (lonlat == 0) {	/* Fix longitudes to 0.0 <= lon <= 360 first */
		while (val > 360.0) val -= 360.0;
		while (val < 0.0) val += 360.0;
	}

	if (val == 360.0 && !gmt_world_map) val = 0.0;
	if (val == 360.0 && gmt_world_map && project_info.projection == OBLIQUE_MERC) val = 0.0;

	switch (gmtdefs.degree_format) {
		case 0:	/* Use 0 to 360 for longitudes and -90 to +90 for latitudes */
			break;
		case 1:	/* Use -180 to +180 for longitudes and -90 to +90 for latitudes */
			if (lonlat == 0 && val > 180.0) val -= 360.0;
			break;
		case 2:	/* Use unsigned 0 to 180 for longitudes and 0 to 90 for latitudes */
			if (lonlat == 0 && val > 180.0) val -= 360.0;
			val = fabs (val);
			break;
		case 3:	/* Use 0 to 180E and 0 to 180W for longitudes and 90S to 90N for latitudes */
			if (lonlat == 0) {
				if (val > 180.0) val -= 360.0;
				letter = (val == 0.0 || val == 180.0) ? 0 : ((val < 0.0) ? 'W' : 'E');
			}
			else
				letter = (val == 0.0) ? 0 : ((val < 0.0) ? 'S' : 'N');
			val = fabs (val);
			break;
		case 4:	/* Use decimal 0 to 360 for longitudes and -90 to +90 for latitudes */
			break;
		case 5:	/* Use decimal -180 to +180 for longitudes and -90 to +90 for latitudes */
			if (lonlat == 0 && val > 180.0) val -= 360.0;
			break;
	}
	
	if (gmtdefs.degree_format == 6) {	/* theta-r */
		(lonlat) ? sprintf (label, "%lg\0", val) : sprintf (label, "%lg\\312\0", val);
		return;
	}
	
	if (gmtdefs.degree_format > 3) {
		sprintf (label, "%lg\\312\0", val);
		return;
	}
		
	ival = val;	/* Truncate to integer in the direction toward 0 */
	minutes = seconds = 0;
	if (fabs (val - (double) ival) > SMALL) {
		minutes = floor (fabs ((val - ival) * 60.0) + SMALL);
		if (minutes == 60) {
			minutes = 0;
			ival = rint (val);
		}
		seconds = rint (fabs ((val - ival - minutes / 60.0) * 3600.0));
		if (seconds == 60) {
			seconds = 0;
			minutes++;
			if (minutes == 60) {
				minutes = 0;
				ival = rint (val);
			}
		}
	}
	
	if (do_minutes) {
		if (ival == 0 && val < 0.0) {	/* Must write out -0 degrees, do so by writing -1 and change 1 to 0 */
			ival = -1;
			zero_fix = TRUE;
		}
		if (do_seconds) {
			if (letter)
				sprintf (label, "%d\\312 %.2d\\251 %.2d\\042%c\0", ival, minutes, seconds, letter);
			else
				sprintf (label, "%d\\312 %.2d\\251 %.2d\\042\0", ival, minutes, seconds);
		}
		else {
			if (letter)
				sprintf (label, "%d\\312 %.2d\\251%c\0", ival, minutes, letter);
			else
				sprintf (label, "%d\\312 %.2d\\251\0", ival, minutes);
		}
		if (zero_fix) label[1] = '0';	/* Undo the fix above */
	}
	else {
		if (letter)
			sprintf (label, "%d\\312%c\0", ival, letter);
		else
			sprintf (label, "%d\\312\0", ival);
	}
	return;
}

int polar_adjust (side, angle, x, y)
int side;
double angle, x, y; {
	int justify, left, right, top, bottom, low;
	double x0, y0;
	
	geo_to_xy (project_info.central_meridian, project_info.pole, &x0, &y0);
	if (project_info.north_pole) {
		low = 0;
		left = 7;
		right = 5;
	}
	else {
		low = 2;
		left = 5;
		right = 7;
	}
	if (y >= y0) {
		top = 2;
		bottom = 10;
	}
	else {
		top = 10;
		bottom = 2;
	}
	
	if (side%2) {	/* W and E border */
		if (y >= y0)
			justify = (side == 1) ? left : right;
		else
			justify = (side == 1) ? right : left;
	}
	else {
		if (frame_info.horizontal) {
			if (side == low)
				justify = (angle == 180.0) ? bottom : top;
			else
				justify = (angle == 0.0) ? top : bottom;
		}	
		else {
			if (x >= x0)
				justify = (side == 2) ? left : right;
			else
				justify = (side == 2) ? right : left;
		}
	}
	return (justify);
}

double get_angle (lon1, lat1, lon2, lat2)
double lon1, lat1, lon2, lat2; {
	double x1, y1, x2, y2, angle, direction;
	
	geo_to_xy (lon1, lat1, &x1, &y1);
	geo_to_xy (lon2, lat2, &x2, &y2);
	angle = d_atan2 (y2-y1, x2-x1) * R2D;
	
	if (abs (gmt_x_status_old) == 2 && abs (gmt_y_status_old) == 2)	/* Last point outside */
		direction = angle + 180.0;
	else if (gmt_x_status_old == 0 && gmt_y_status_old == 0)		/* Last point inside */
		direction = angle;
	else {
		if (abs (gmt_x_status_new) == 2 && abs (gmt_y_status_new) == 2)	/* This point outside */
			direction = angle;
		else if (gmt_x_status_new == 0 && gmt_y_status_new == 0)		/* This point inside */
			direction = angle + 180.0;
		else {	/* Special case of corners and sides only */
			if (gmt_x_status_old == gmt_x_status_new)
				direction = (gmt_y_status_old == 0) ? angle : angle + 180.0;
			else if (gmt_y_status_old == gmt_y_status_new)
				direction = (gmt_x_status_old == 0) ? angle : angle + 180.0;
			else
				direction = angle;
			
		}
	}
	
	if (direction < 0.0) direction += 360.0;
	if (direction >= 360.0) direction -= 360.0;
	return (direction);
}


int draw_map_scale (x0, y0, lat, length, miles, gave_xy, fancy)
double x0, y0, lat, length;
BOOLEAN miles, gave_xy, fancy; {
	int i, j, k, *rgb, n_a_ticks[9], n_f_ticks[9];
	double dlon, x1, x2, dummy, a, b, tx, ty, off, f_len, a_len, x_left, bar_length;
	double xx[4], yy[4], bx[4], by[4], base, d_base, width, half, bar_width, dx_f, dx_a;
	char txt[80];
	
	if (!MAPPING) return;	/* Only for geographic projections */
	
	if (!gave_xy) {
		geo_to_xy (x0, y0, &a, &b);
		x0 = a;
		y0 = b;
	}
	
	bar_length = (miles) ? 1.609344 * length : length;
	dlon = 0.5 * bar_length * 1000.0 / (M_PR_DEG * cosd (lat));
	
	geoz_to_xy (project_info.central_meridian - dlon, lat, project_info.z_level, &x1, &dummy);
	geoz_to_xy (project_info.central_meridian + dlon, lat, project_info.z_level, &x2, &dummy);
	width = x2 - x1;
	half = 0.5 * width;
	a_len = fabs (gmtdefs.tick_length);
	off = a_len + 0.75 * gmtdefs.anot_offset;
	
        ps_setline (gmtdefs.tick_pen);
	if (fancy) {	/* Fancy scale */
		n_f_ticks[8] = 3;
		n_f_ticks[1] = n_f_ticks[3] = n_f_ticks[7] = 4;
		n_f_ticks[0] = n_f_ticks[4] = 5;
		n_f_ticks[2] = n_f_ticks[5] = 6;
		n_f_ticks[6] = 7;
		n_a_ticks[4] = n_a_ticks[6] = n_a_ticks[8] = 1;
		n_a_ticks[0] = n_a_ticks[1] = n_a_ticks[3] = n_a_ticks[7] = 2;
		n_a_ticks[2] = n_a_ticks[5] = 3;
		base = pow (10.0, floor (d_log10 (length)));
		i = rint (length / base) - 1;
		d_base = length / n_a_ticks[i];
		dx_f = width / n_f_ticks[i];
		dx_a = width / n_a_ticks[i];
		bar_width = 0.5 * fabs (gmtdefs.tick_length);
		f_len = 0.75 * fabs (gmtdefs.tick_length);
		yy[2] = yy[3] = y0;
		yy[0] = yy[1] = y0 - bar_width;
		x_left = x0 - half;
		xyz_to_xy (x_left, y0 - f_len, project_info.z_level, &a, &b);
		ps_plot (a, b, 3);
		xyz_to_xy (x_left, y0, project_info.z_level, &a, &b);
		ps_plot (a, b, 2);
		for (j = 0; j < n_f_ticks[i]; j++) {
			xx[0] = xx[3] = x_left + j * dx_f;
			xx[1] = xx[2] = xx[0] + dx_f;
			for (k = 0; k < 4; k++) xyz_to_xy (xx[k], yy[k], project_info.z_level, &bx[k], &by[k]);
			rgb = (j%2) ? gmtdefs.foreground_rgb : gmtdefs.background_rgb;
			ps_polygon (bx, by, 4, rgb[0], rgb[1], rgb[2], TRUE);
			xyz_to_xy (xx[1], y0 - f_len, project_info.z_level, &a, &b);
			ps_plot (a, b, 3);
			xyz_to_xy (xx[1], y0, project_info.z_level, &a, &b);
			ps_plot (a, b, 2);
		}
		ty = y0 - off;
		for (j = 0; j <= n_a_ticks[i]; j++) {
			tx = x_left + j * dx_a;
			xyz_to_xy (tx, y0 - a_len, project_info.z_level, &a, &b);
			ps_plot (a, b, 3);
			xyz_to_xy (tx, y0, project_info.z_level, &a, &b);
			ps_plot (a, b, 2);
			sprintf (txt, "%lg\0", j * d_base);
			gmt_text3d (tx, ty, project_info.z_level, gmtdefs.anot_font_size, gmtdefs.anot_font, txt, 0.0, 10, 0);
		}
		if (miles)
			strcpy (txt, "miles");
		else
			strcpy (txt, "km");
		xyz_to_xy (x0, y0 + f_len, project_info.z_level, &tx, &ty);
		gmt_text3d (tx, ty, project_info.z_level, gmtdefs.label_font_size, gmtdefs.label_font, txt, 0.0, 2, 0);
	}
	else {	/* Simple scale */
	
		if (miles)
			sprintf (txt, "%lg miles\0", length);
		else
			sprintf (txt, "%lg km\0", length);
		xyz_to_xy (x0 - half, y0 - gmtdefs.tick_length, project_info.z_level, &a, &b);
		ps_plot (a, b, 3);
		xyz_to_xy (x0 - half, y0, project_info.z_level, &a, &b);
		ps_plot (a, b, 2);
		xyz_to_xy (x0 + half, y0, project_info.z_level, &a, &b);
		ps_plot (a, b, 2);
		xyz_to_xy (x0 + half, y0 - gmtdefs.tick_length, project_info.z_level, &a, &b);
		ps_plot (a, b, 2);
		gmt_text3d (x0, y0 - off, project_info.z_level, gmtdefs.anot_font_size, gmtdefs.anot_font, txt, 0.0, 10, 0);
	}
}
