/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grd2xyz.c	2.16  3/13/95
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * grd2xyz.c reads a grd file and prints out the x,y,z values to
 * standard output.
 *
 * Author:	Paul Wessel
 * Date:	3-JAN-1991-1995
 * Version:	2.0	Based on old v1.x
 */

#include "gmt.h"

float *z;

main (argc, argv)
int argc;
char **argv; {
	int i, j, ij, nm, nx, ny, dummy[4], one_or_zero, ix, iy;
	int error = FALSE, global = FALSE, z_only = FALSE, binary = FALSE, single_precision = FALSE;
	double w, e, s, n, d, *x, *y, out[3];
	char *grdfile = NULL;
	struct GRD_HEADER grd;
	
	argc = gmt_begin (argc, argv);

	w = e = s = n = 0.0;
	dummy[0] = dummy[1] = dummy[2] = dummy[3] = 0;
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */
			
				case 'H':
				case 'R':
				case 'V':
				case ':':
				case '\0':
					error += get_common_args (argv[i], &w, &e, &s, &n);
					break;
				case 'b':
					single_precision = !(argv[i][strlen(argv[i])-1] == 'd');
					binary = TRUE;
					break;
				case 'Z':
					z_only = TRUE;
					break;
					
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else
			grdfile = argv[i];
	}
	
	if (argc == 1 || gmt_quick) {
		fprintf (stderr, "grd2xyz %s - Converting a netCDF grdfile to an ASCII xyz-file\n\n", GMT_VERSION);
		fprintf( stderr, "usage: grd2xyz grdfile [-H] [-Rw/e/s/n] [-V] [-Z] [-:] [-b[d]]> xyzfile\n");

		if (gmt_quick) exit (-1);
		
		fprintf (stderr, "	grdfile is the grd file to convert\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		explain_option ('H');
		explain_option ('R');
		explain_option ('V');
		fprintf (stderr, "	-Z Only output the z-array in scanline orientation\n");
		explain_option (':');
		fprintf (stderr, "	-b means binary output.  Append d for double precision [Default is single].\n");
		exit (-1);
	}
	
	if (!grdfile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify input file\n", gmt_program);
		error++;
	}
	if (binary && gmtdefs.io_header) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data cannot have header -H\n", gmt_program);
		error++;
	}
	if (error) exit (-1);

	if (read_grd_info (grdfile, &grd)) {
		fprintf (stderr, "grd2xyz: Error opening file %s\n", grdfile);
		exit (-1);
	}
	
	if (binary) gmt_output = (single_precision) ? bin_float_output : bin_double_output;

	ix = (gmtdefs.xy_toggle) ? 1 : 0;        iy = 1 - ix;              /* Set up which columns have x and y */

	nm = grd.nx * grd.ny;
	
	if (e > w && n > s) {
		global = (fabs (grd.x_max - grd.x_min) == 360.0);
		if (!global && (w < grd.x_min || e > grd.x_max)) error = TRUE;
		if (s < grd.y_min || n > grd.y_max) error = TRUE;
		if (error) {
			fprintf (stderr, "%s: GMT ERROR: Subset exceeds data domain!\n", gmt_program);
			exit (-1);
		}
		one_or_zero = (grd.node_offset) ? 0 : 1;
		nx = rint ((e - w) / grd.x_inc) + one_or_zero;
		ny = rint ((n - s) / grd.y_inc) + one_or_zero;
		
		z = (float *) memory (CNULL, nx * ny, sizeof (float), "grd2xyz");
		
		if (read_grd (grdfile, &grd, z, w, e, s, n, dummy, FALSE)) {
			fprintf (stderr, "grd2xyz: Error reading file %s\n", grdfile);
			exit (-1);
		}
	}
	else {
		z = (float *) memory (CNULL, nm, sizeof (float), "grd2xyz");

		if (read_grd (grdfile, &grd, z, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
			fprintf (stderr, "grd2xyz: Error reading file %s\n", grdfile);
			exit (-1);
		}
	}

	if (!z_only) {
	
		x = (double *) memory (CNULL, grd.nx, sizeof (double), "grd2xyz");
		y = (double *) memory (CNULL, grd.ny, sizeof (double), "grd2xyz");

		/* Compute grid node positions once only */

		d = (grd.node_offset) ? 0.5 : 0.0;

		for (j = 0; j < grd.ny; j++) y[j] = (j == (grd.ny-1)) ? grd.y_min + d * grd.y_inc : grd.y_max - (j + d) * grd.y_inc;

		for (i = 0; i < grd.nx; i++) x[i] = (i == (grd.nx-1)) ? grd.x_max - d * grd.x_inc: grd.x_min + (i + d) * grd.x_inc;
	}

	if (gmtdefs.io_header) {
		if (!grd.x_units[0]) strcpy (grd.x_units, "x");
		if (!grd.y_units[0]) strcpy (grd.y_units, "y");
		if (!grd.z_units[0]) strcpy (grd.z_units, "z");
		if (gmtdefs.xy_toggle)
			printf ("%s\t%s\t%s\n", grd.y_units, grd.x_units, grd.z_units);
		else
			printf ("%s\t%s\t%s\n", grd.x_units, grd.y_units, grd.z_units);
	}
	
	if (z_only) {
		for (ij = 0; ij < nm; ij++) {
			out[0] = z[ij];
			gmt_output (stdout, 1, out);
		}
	}
	else  {
		for (j = ij = 0; j < grd.ny; j++) for (i = 0; i < grd.nx; i++, ij++) {
			out[ix] = x[i];	out[iy] = y[j];	out[2] = z[ij];
			gmt_output (stdout, 3, out);
		}
	}
	
	free ((char *)z);
	
	if (!z_only) {
		free ((char *)x);
		free ((char *)y);
	}
	
	gmt_end (argc, argv);
}
