/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdedit.c	2.14  3/13/95
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * grdedit reads an existing grdfile and takes command line
 * arguments to redefine some of the grdheader parameters:
 *
 *	x_min/x_max OR x_inc (the other is recomputed)
 *	y_min/y_max OR y_inc (the other is recomputed)
 *	z_scale_factor/z_add_offset
 *	x_units/y_units/z_units
 *	title/command/remark
 *
 * Author:	Paul Wessel
 * Date:	9-JAN-1991-1995
 * Version:	2.0
 */
 
#include "gmt.h"

float *a;

main (argc, argv)
int argc;
char **argv; {
	int i, entry, one_or_zero, dummy[4];
	int error = FALSE, got_input = FALSE, got_region = FALSE, shift = FALSE;
	
	double w, e, s, n, shift_amount = 0.0;
	
	char *grdfile, input[512], *ptr;
	
	struct GRD_HEADER grd;
	
	input[0] = 0;
	grdfile = CNULL;
	w = e = s = n = 0.0;
	dummy[3] = dummy[2] = dummy[1] = dummy[0] = 0;
	
	argc = gmt_begin (argc, argv);

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */
			
				case 'R':
				case '\0':
					error += get_common_args (argv[i], &w, &e, &s, &n);
					if (argv[i][1] == 'R') got_region = TRUE;
					break;

				/* Supplemental parameters */
			
				case 'D':
					strcpy (input, &argv[i][2]);
					got_input = TRUE;
					break;
				case 'S':
					shift = TRUE;
					break;
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else
			grdfile = argv[i];
	}
		
	if (argc == 1 || gmt_quick) {
		fprintf (stderr, "grdedit %s - Modifying the header in a netCDF grdfile\n\n", GMT_VERSION);
		fprintf (stderr, "usage: grdedit grdfile [-R<west>/<east>/<south>/<north>\n");
		fprintf (stderr, "	[-S] [-Dxunit/yunit/zunit/scale/offset/title/remark]\n");
		
		if (gmt_quick) exit (-1);
		
		fprintf (stderr, "	grdfile is file to be modified\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf (stderr, "	-D to enter information.  Specify '=' to get default value\n");
		explain_option ('R');
		fprintf (stderr, "	-S For global grids of 360 degree longitude range.\n");
		fprintf (stderr, "	   Will rotate entire grid to coincide with new borders in -R\n");
		exit (-1);
	}
	
	if (shift && !got_region) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S option:  Must also specify -R\n", gmt_program);
		error++;
	}
	if (shift && fabs (w - e) != 360.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S option:  -R longitudes must span exactly 360 degrees\n", gmt_program);
		error++;
	}
	if (!grdfile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify input file\n", gmt_program);
		error++;
	}
	if (error) exit (-1);
			
	if (!strcmp (grdfile,  "=")) {
		fprintf (stderr, "grdedit: Piping of grdfile not supported!\n");
		exit (-1);
	}

	if (read_grd_info (grdfile, &grd)) {
		fprintf (stderr, "grdedit: Error opening file %s\n", grdfile);
		exit (-1);
	}
	
	if (shift && fabs (grd.x_max - grd.x_min) != 360.0) {
		fprintf (stderr, "grdedit: Shift only allowed for global grids\n");
		exit (-1);
	}
	
	/* Decode grd information given, if any */
	
	if (got_input) {
		ptr = strtok (input, "/");
		entry = 0;
		while (ptr) {
			switch (entry) {
				case 0:
					if (ptr[0] != '=') strcpy (grd.x_units, ptr);
					break;
				case 1:
					if (ptr[0] != '=') strcpy (grd.y_units, ptr);
					break;
				case 2:
					if (ptr[0] != '=') strcpy (grd.z_units, ptr);
					break;
				case 3:
					if (ptr[0] != '=') grd.z_scale_factor = atof (ptr);
					break;
				case 4:
					if (ptr[0] != '=') grd.z_add_offset = atof (ptr);
					break;
				case 5:
					if (ptr[0] != '=') strcpy (grd.title, ptr);
					break;
				case 6:
					if (ptr[0] != '=') strcpy (grd.remark, ptr);
					break;
				default:
					break;
			}
			ptr = strtok (CNULL, "/");
			entry++;
		}
	}
	
	if (got_region && !shift) {
		one_or_zero = (grd.node_offset) ? 0 : 1;
		grd.x_min = w;	grd.x_max = e;
		grd.x_inc = (grd.x_max - grd.x_min) / (grd.nx - one_or_zero);
		grd.y_min = s;	grd.y_max = n;
		grd.y_inc = (grd.y_max - grd.y_min) / (grd.ny - one_or_zero);
	}
	
	if (shift) {
		shift_amount = w - grd.x_min;
		a = (float *) memory (CNULL, (int)(grd.nx * grd.ny), sizeof (float), "grdedit");
		read_grd (grdfile, &grd, a, 0.0, 0.0, 0.0, 0.0, dummy, FALSE);
		grd_shift (&grd, a, shift_amount);
		write_grd (grdfile, &grd, a, 0.0, 0.0, 0.0, 0.0, dummy, FALSE);
		free ((char *)a);
	}
	else if (write_grd_info (grdfile, &grd)) {
		fprintf (stderr, "grdedit: Error updating file %s\n", grdfile);
		exit (-1);
	}
	
	gmt_end (argc, argv);
}
