/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdfilter.c	2.22  4/28/95
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * grdfilter.c  reads a grdfile and creates filtered grd file
 *
 * user selects boxcar, gaussian, cosine arch, median, or mode filter
 * user selects distance scaling as appropriate for map work, etc.
 *
 * Author:  W.H.F. Smith
 * Date: 	16 Aug 88
 *
 * Modified:	27 Sep 88 by W.H.F. Smith, to use new median routine.
 *
 * Updated:	PW: 13-Jun-1991 to v2.0
 *		PW: 13-Jul-1992 to actually do v2 i/o.
 *		PW: 15-Jul-1992 to handle arbitrary new -R -I
 *		PW: 3-Jan-1995 to offer mode-filter (from filter1d)
 *
 */
 
#include "gmt.h"

double	get_wt();
int	median();
int	mode();

int	*i_origin;
float	*input, *output;
double	*weight, *work_array, *x_shift;
double	deg2km;

main (argc, argv)
int argc;
char **argv; {

	int	nx_out, ny_out, nx_fil, ny_fil, n_in_median, n_nan = 0;
	int	x_half_width, y_half_width, j_origin, i_out, j_out;
	int	i_in, j_in, ii, jj, i, ij_in, ij_out, ij_wt, effort_level;
	int	distance_flag, filter_type, one_or_zero = 1, dummy[4];
	
	BOOLEAN	error, new_range, new_increment, fast_way, shift, slow;
	
	double	west_new, east_new, south_new, north_new, dx_new, dy_new, offset;
	double	filter_width, x_scale, y_scale, x_width, y_width;
	double	x_out, y_out, wt_sum, value, last_median, this_median, xincnew2, yincnew2;
	double	xincold2, yincold2, y_shift, x_fix = 0.0, y_fix = 0.0;
	
	char	*fin = NULL, *fout = NULL;
	
	struct	GRD_HEADER h;

	argc = gmt_begin (argc, argv);
	
	deg2km = 0.001 * 2.0 * M_PI * gmtdefs.ellipse[gmtdefs.ellipsoid].eq_radius / 360.0;
	error = new_range = new_increment = FALSE;
	fin = fout = NULL;
	filter_width = dx_new = dy_new = west_new = east_new = 0.0;
	filter_type = distance_flag = -1;
	dummy[3] = dummy[2] = dummy[1] = dummy[0] = 0;
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */
			
				case 'R':
				case 'V':
				case '\0':
					error += get_common_args (argv[i], &west_new, &east_new, &south_new, &north_new);
					break;
				
				case 'F':
					switch (argv[i][2]) {
						case 'b':
							filter_type = 0;
							break;
						case 'c':
							filter_type = 1;
							break;
						case 'g':
							filter_type = 2;
							break;
						case 'm':
							filter_type = 3;
							break;
						case 'p':
							filter_type = 4;
							break;
						default:
							error = TRUE;
							break;
					}
					filter_width = atof(&argv[i][3]);
					break;
				case 'G':
					fout = &argv[i][2];
					break;
				case 'D':
					distance_flag = atoi(&argv[i][2]);
					break;
				case 'I':
					gmt_getinc (&argv[i][2], &dx_new, &dy_new);
					new_increment = TRUE;
					break;
				case 'N':	/* Pixel registration */
					one_or_zero = 0;
					break;
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else
			fin = argv[i];
	}
	
	if (argc == 1 || gmt_quick) {
		fprintf (stderr, "grdfilter %s - Filter a 2-D netCDF grdfile in the Time domain\n\n", GMT_VERSION);
		fprintf(stderr,"usage: grdfilter input_file -D<distance_flag> -F<type><filter_width>\n");
		fprintf(stderr,"\t-G<output_file> [-I<xinc>[m|c][/<yinc>[m|c]] ] [-R<west/east/south/north>] [-V]\n");
		
		if (gmt_quick) exit (-1);
		
		fprintf(stderr,"\tDistance flag determines how grid (x,y) maps into distance units of filter width as follows:\n");
		fprintf(stderr,"\t   -D0 grid x,y same units as <filter_width>, cartesian Distances.\n");
		fprintf(stderr,"\t   -D1 grid x,y in degrees, <filter_width> in km, cartesian Distances.\n");
		fprintf(stderr,"\t   -D2 grid x,y in degrees, <filter_width> in km, x_scaled by cos(middle y), cartesian Distances.\n");
		fprintf(stderr,"\t   These first three options are faster; they allow weight matrix to be computed only once.\n");
		fprintf(stderr,"\t   Next two options are slower; weights must be recomputed for each scan line.\n");
		fprintf(stderr,"\t   -D3 grid x,y in degrees, <filter_width> in km, x_scale varies as cos(y), cartesian Distances.\n");
		fprintf(stderr,"\t   -D4 grid x,y in degrees, <filter_width> in km, spherical Distances.\n");
		fprintf(stderr,"\t-F sets the filter type and full (6 sigma) filter-width.  Choose between\n");
		fprintf(stderr,"\t   (b)oxcar, (c)osine arch, (g)aussian, (m)edian filters\n");
		fprintf(stderr,"\t   or p(maximum liklihood Probability estimator -- a mode estimator)\n");
		fprintf(stderr,"\t-G sets output name for filtered grdfile\n");
		fprintf(stderr, "\n\tOPTIONS:\n");
		fprintf(stderr,"\t-I for new Increment of output grid; enter xinc, optionally xinc/yinc.\n");
		fprintf(stderr,"\t   Default is yinc = xinc.  Append an m [or c] to xinc or yinc to indicate minutes [or seconds];\n");
		fprintf(stderr,"\t   The new xinc and yinc should be divisible by the old ones (new lattice is subset of old).\n");
		fprintf(stderr, "\t-N Force pixel registration for output grid [Default is gridline registration]\n");
		fprintf(stderr, "\t-R for new Range of output grid; enter <WESN> (xmin, xmax, ymin, ymax) separated by slashes.\n");
		explain_option ('V');
		exit(-1);
	}

	if (!fout) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G option:  Must specify output file\n", gmt_program);
		error++;
	}
	if (!fin) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify input file\n", gmt_program);
		error++;
	}
	if (distance_flag < 0 || distance_flag > 4) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -D option:  Choose among 1, 2, 3, or 4\n", gmt_program);
		error++;
	}
	if (filter_type < 0 || filter_width <= 0.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -F option:  Correct syntax:\n", gmt_program);
		fprintf (stderr, "\t-FX<width>, with X one of bcgmp, width is filter fullwidth\n");
		error++;
	}
	if (error) exit (-1);
	
	if (west_new != east_new) new_range = TRUE;

	if (read_grd_info (fin, &h)) {
		fprintf (stderr, "grdfilter: Error opening file %s\n", fin);
		exit (-1);
	}
	grd_init (&h, argc, argv, TRUE);	/* Update command history only */

	/* Read the input grd file and close it  */
	
	input = (float *) memory (CNULL, (int)(h.nx * h.ny), sizeof(float), "grdfilter");

	if (read_grd (fin, &h, input, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
		fprintf (stderr, "grdfilter: Error reading file %s\n", fin);
		exit (-1);
	}
	
	last_median = 0.5 * (h.z_min + h.z_max);

	/* Check range of output area and set i,j offsets, etc.  */

	if (!new_range) {
		west_new = h.x_min;
		east_new = h.x_max;
		south_new = h.y_min;
		north_new = h.y_max;
	}
	if (!new_increment) {
		dx_new = h.x_inc;
		dy_new = h.y_inc;
	}

	if (west_new < h.x_min) error = TRUE;
	if (east_new > h.x_max) error = TRUE;
	if (south_new < h.y_min) error = TRUE;
	if (north_new > h.y_max) error = TRUE;
	if (dx_new <= 0.0) error = TRUE;
	if (dy_new <= 0.0) error = TRUE;
	
	if (error) {
		fprintf(stderr,"grdfilter: New WESN incompatible with old.\n");
		exit(-1);
	}

	/* We can save time by computing a weight matrix once [or once pr scanline] only
	   if new grid spacing is multiple of old spacing */
	   
	fast_way = ((rint (dx_new / h.x_inc) == (dx_new / h.x_inc)) && (rint (dy_new / h.y_inc) == (dy_new / h.y_inc)));
	
	if (!fast_way) {
		fprintf (stderr, "grdfilter: Warning - Your output grid spacing is such that filter-weights must\n");
		fprintf (stderr, "be recomputed for every output node, so expect this run to be slow.  Calculations\n");
		fprintf (stderr, "can be speeded up significantly if output grid spacing is chosen to be a multiple\n");
		fprintf (stderr, "of the input grid spacing.  If the odd output grid is necessary, consider using\n");
		fprintf (stderr, "a \'fast\' grid for filtering and then resample onto your desired grid with grdsample.\n");
	}

	nx_out = one_or_zero + rint ( (east_new - west_new) / dx_new);
	ny_out = one_or_zero + rint ( (north_new - south_new) / dy_new);
	
	output = (float *) memory (CNULL, nx_out*ny_out, sizeof(float), "grdfilter");
	i_origin = (int *) memory (CNULL, nx_out, sizeof(int), "grdfilter");
	if (!fast_way) x_shift = (double *) memory (CNULL, nx_out, sizeof(double), "grdfilter");

	xincnew2 = (one_or_zero) ? 0.0 : 0.5 * dx_new;
	yincnew2 = (one_or_zero) ? 0.0 : 0.5 * dy_new;
	xincold2 = (h.node_offset) ? 0.0 : 0.5 * h.x_inc;
	yincold2 = (h.node_offset) ? 0.0 : 0.5 * h.y_inc;
	offset = (h.node_offset) ? 0.0 : 0.5;
	
	if (fast_way && h.node_offset == one_or_zero) {	/* multiple grid but one is pix, other is grid */
		x_fix = 0.5 * h.x_inc;
		y_fix = 0.5 * h.y_inc;
		shift = (x_fix != 0.0 || y_fix != 0.0);
	}
	
	/* Set up weight matrix and i,j range to test  */

	x_scale = y_scale = 1.0;
	if (distance_flag > 0) {
		x_scale *= deg2km;
		y_scale *= deg2km;
	}
	if (distance_flag == 2)
		x_scale *= cos(D2R * 0.5 * ( north_new + south_new) );
	else if (distance_flag > 2) {
		if ( fabs(south_new) > north_new )
			x_scale *= cos(D2R * south_new);
		else
			x_scale *= cos(D2R * north_new);
	}
	x_width = filter_width / (h.x_inc * x_scale);
	y_width = filter_width / (h.y_inc * y_scale);
	y_half_width = ceil(y_width) / 2;
	x_half_width = ceil(x_width) / 2;

	nx_fil = 2 * x_half_width + 1;
	ny_fil = 2 * y_half_width + 1;
	weight = (double *) memory (CNULL, nx_fil*ny_fil, sizeof(double), "grdfilter");

	slow = (filter_type == 3 || filter_type == 4);	/* Will require sorting etc */
	
	if (slow) work_array = (double *) memory (CNULL, nx_fil*ny_fil, sizeof(double), "grdfilter");

	if (gmtdefs.verbose) {
		fprintf(stderr,"grdfilter: Input nx,ny = (%d %d), output nx,ny = (%d %d), filter nx,ny = (%d %d)\n",
			h.nx, h.ny, nx_out, ny_out, nx_fil, ny_fil);
		if (filter_type == 0) fprintf(stderr,"grdfilter: Filter type is Boxcar.\n");
		if (filter_type == 1) fprintf(stderr,"grdfilter: Filter type is Cosine Arch.\n");
		if (filter_type == 2) fprintf(stderr,"grdfilter: Filter type is Gaussian.\n");
		if (filter_type == 3) fprintf(stderr,"grdfilter: Filter type is Median.\n");
		if (filter_type == 4) fprintf(stderr,"grdfilter: Filter type is Mode.\n");
	}
	
	/* Compute nearest xoutput i-indices and shifts once */
	
	for (i_out = 0; i_out < nx_out; i_out++) {
		x_out = west_new + i_out * dx_new + xincnew2;
		i_origin[i_out] = floor(((x_out - h.x_min) / h.x_inc) + offset);
		if (!fast_way) x_shift[i_out] = x_out - (h.x_min + i_origin[i_out] * h.x_inc + xincold2);
	}
	
	/* Now we can do the filtering  */

	/* Determine how much effort to compute weights:
		1 = Compute weights once for entire grid
		2 = Compute weights once per scanline
		3 = Compute weights for every output point [slow]
	*/
	
	if (fast_way && distance_flag <= 2)
		effort_level = 1;
	else if (fast_way && distance_flag > 2)
		effort_level = 2;
	else
		effort_level = 3;
		
	if (effort_level == 1) {	/* Only need this once */
		y_out = north_new - yincnew2;
		set_weight_matrix (nx_fil, ny_fil, y_out, north_new, south_new, h.x_inc, h.y_inc, filter_width, filter_type, distance_flag, x_fix, y_fix, shift);
	}
	
	for (j_out = 0; j_out < ny_out; j_out++) {
	
		if (gmtdefs.verbose) fprintf (stderr, "grdfilter: Processing output line %d\r", j_out);
		y_out = north_new - j_out * dy_new - yincnew2;
		j_origin = floor(((h.y_max - y_out) / h.y_inc) + offset);
		if (effort_level == 2)
			set_weight_matrix (nx_fil, ny_fil, y_out, north_new, south_new, h.x_inc, h.y_inc, filter_width, filter_type, distance_flag, x_fix, y_fix, shift);
		if (!fast_way) y_shift = y_out - (h.y_max - j_origin * h.y_inc - yincold2);
		
		for (i_out = 0; i_out < nx_out; i_out++) {
		
			if (effort_level == 3)
				set_weight_matrix (nx_fil, ny_fil, y_out, north_new, south_new, h.x_inc, h.y_inc, filter_width, filter_type, distance_flag, x_shift[i_out], y_shift, fast_way);
			wt_sum = value = 0.0;
			n_in_median = 0;
			ij_out = j_out * nx_out + i_out;
			
			for (ii = -x_half_width; ii <= x_half_width; ii++) {
				i_in = i_origin[i_out] + ii;
				if ( (i_in < 0) || (i_in >= h.nx) ) continue;

				for (jj = -y_half_width; jj <= y_half_width; jj++) {
					j_in = j_origin + jj;
					if ( (j_in < 0) || (j_in >= h.ny) ) continue;
										
					ij_wt = (jj + y_half_width) * nx_fil + ii + x_half_width;
					if (weight[ij_wt] < 0.0) continue;
					
					ij_in = j_in*h.nx + i_in;
					if (bad_float ((double)input[ij_in])) continue;

					/* Get here when point is usable  */
					if (slow) {
						work_array[n_in_median] = input[ij_in];
						n_in_median++;
					}
					else {
						value += input[ij_in] * weight[ij_wt];
						wt_sum += weight[ij_wt];
					}
				}
			}
			
			/* Now we have done the convolution and we can get the value  */
			
			if (slow) {
				if (n_in_median) {
					if (filter_type == 3)
						median (work_array, n_in_median, h.z_min, h.z_max, last_median, &this_median);
					else
						mode (work_array, n_in_median, n_in_median/2, TRUE, &this_median);
					output[ij_out] = this_median;
					last_median = this_median;
				}
				else {
					output[ij_out] = gmt_NaN;
					n_nan++;
				}
			}
			else {
				if (wt_sum == 0.0) {	/* Assign value = gmt_NaN */
					n_nan++;
					output[ij_out] = gmt_NaN;
				}
				else
					output[ij_out] = value / wt_sum;
			}
		}
	}
	if (gmtdefs.verbose) fprintf (stderr, "\n");
	
	/* At last, that's it!  Output: */

	if (n_nan) fprintf (stderr, "grdfilter: Unable to estimate value at %d nodes, set to NaN\n", n_nan);
	
	h.nx = nx_out;
	h.ny = ny_out;
	h.x_min = west_new;
	h.x_max = east_new;
	h.y_min = south_new;
	h.y_max = north_new;
	h.x_inc = dx_new;
	h.y_inc = dy_new;
	h.node_offset = !one_or_zero;
	
	if (write_grd (fout, &h, output, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
		fprintf (stderr, "grdfilter: Error writing file %s\n", fout);
		exit (-1);
	}
	
	free ((char *) input);
	free ((char *) output);
	free ((char *) weight);
	free ((char *) i_origin);
	if (slow) free ((char *) work_array);
	if (!fast_way) free ((char *) x_shift);
	
	gmt_end (argc, argv);
}

int	set_weight_matrix (nx_f, ny_f, y_0, north, south, dx, dy, f_wid, f_flag, d_flag, x_off, y_off, fast)

double	y_0, north, south, dx, dy, f_wid, x_off, y_off;	/* Last two gives offset between output node and 'origin' input node for this window (0,0 for integral grids) */
int	nx_f, ny_f;
BOOLEAN fast;	/* TRUE when input/output grids are offset by integer values in dx/dy */
int	f_flag, d_flag;
{

	int	i, j, ij, i_half, j_half;
	double	x_scl, y_scl, f_half, r_f_half, sigma, sig_2;
	double	y1, y2, theta, x, y, r;

	/* Set Scales  */

	y_scl = (d_flag) ? deg2km : 1.0;
	if (d_flag < 2)
		x_scl = y_scl;
	else if (d_flag == 2)
		x_scl = deg2km * cos (0.5 * D2R * (north + south));
	else
		x_scl = deg2km * cos (D2R * y_0);

	/* Get radius, weight, etc.  */

	i_half = nx_f / 2;
	j_half = ny_f / 2;
	f_half = 0.5 * f_wid;
	r_f_half = 1.0 / f_half;
	sigma = f_wid / 6.0;
	sig_2 = -0.5 / (sigma * sigma);
	for (i = -i_half; i <= i_half; i++) {
		for (j = -j_half; j <= j_half; j++) {
			ij = (j + j_half) * nx_f + i + i_half;
			if (fast && i == 0)
				r = (j == 0) ? 0.0 : j * y_scl * dy;
			else if (fast && j == 0)
				r = i * x_scl * dx;
			else if (d_flag < 4) {
				x = x_scl * (i * dx - x_off);
				y = y_scl * (j * dy - y_off);
				r = hypot (x, y);
			}
			else {
				theta = D2R * (i * dx - x_off);
				y1 = D2R * (90.0 - y_0);
				y2 = D2R * (90.0 - (y_0 + (j * dy - y_off)) );
				r = d_acos(cos(y1)*cos(y2) + sin(y1)*sin(y2)*cos(theta)) * deg2km * R2D;
			}
			/* Now we know r in f_wid units  */
			
			if (r > f_half) {
				weight[ij] = -1.0;
				continue;
			}
			else if (f_flag == 3) {
				weight[ij] = 1.0;
				continue;
			}
			else {
				if (f_flag == 0)
					weight[ij] = 1.0;
				else if (f_flag == 1)
					weight[ij] = 1.0 + cos(M_PI * r * r_f_half);
				else
					weight[ij] = exp(r * r * sig_2);
			}
		}
	}
}
