/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdgradient.c	2.19  3/13/95
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 *  grdgradient.c
 * read a grdfile and compute gradient in azim direction:
 *
 * azim = azimuth clockwise from north in degrees.
 *
 * gradient = -[(dz/dx)sin(azim) + (dz/dy)cos(azim)].
 *
 * the expression in [] is the correct gradient.  We take
 * -[]  in order that data which goes DOWNHILL in the
 * azim direction will give a positive value; this is
 * for image shading purposes.
 *
 *
 * Author:	W.H.F. Smith
 * Date: 	13 Feb 1991-1995
 * Upgraded to v2.0 15-May-1991-1995 Paul Wessel
 *
 * Modified:	1 Mar 94 by WHFS to make -M scale change with j latitude
 */
 
#include "gmt.h"

float *data;

main (argc, argv)
int argc;
char **argv; {

	int	i, j, ij, k, n, nm, nm2, mx, my, pad[4];
	int p[4], n_used = 0;
	
	BOOLEAN	error = FALSE, map_units = FALSE, normalize = FALSE, atan_trans = FALSE, bad;
	
	double	dx_grid, dy_grid, x_factor, y_factor, dzdx, dzdy, ave_gradient, norm_val = 1.0;
	double	azim, denom, max_gradient = 0.0, min_gradient = 0.0, rpi, m_pr_degree, lat;
	
	char *infile = CNULL, *outfile = CNULL;
	
	struct GRD_HEADER header;

	argc = gmt_begin (argc, argv);
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
			
				/* Common parameters */
				
				case 'V':
				case '\0':
					error += get_common_args (argv[i], 0, 0, 0, 0);
					break;
					
				/* Supplemental parameters */
			
				case 'A':
					azim = atof(&argv[i][2]);
					break;
				case 'G':
					outfile = &argv[i][2];
					break;
				case 'M':
					map_units = TRUE;
					break;
				case 'N':
					normalize = TRUE;
					if (argv[i][2]) {
						if (argv[i][2] == 't' || argv[i][2] == 'T') {
							atan_trans = TRUE;
							if (argv[i][3]) norm_val = atof(&argv[i][3]);
						}
						else
							norm_val = atof (&argv[i][2]);
					}
					break;
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
					
			}
		}
		else
			infile = argv[i];
	}

	if (argc == 1 || gmt_quick) {
		fprintf (stderr,"grdgradient %s - Compute directional gradients from grdfiles\n\n", GMT_VERSION);
		fprintf (stderr, "usage: grdgradient <infile> -A<azim> -G<outfile> [-M] [-N[t][<val>]] [-V]\n");
		if (gmt_quick) exit (-1);
		fprintf (stderr, "	-A sets azimuth (0-360) for directional derivatives\n");
		fprintf (stderr, "	-G output file for intensities\n");
		fprintf (stderr, "	-M to use map units.  In this case, dx,dy of grdfile\n");
		fprintf (stderr, "	   will be converted from degrees lon,lat into meters.\n");
		fprintf (stderr, "	   Default computes gradient in units of data/grid_distance.\n");
		fprintf (stderr, "	-N will normalize gradients so that max |grad| = <val> [1.0]\n");
		fprintf (stderr, "	-Nt will make atan() transform, then normalize max |grad| to <val>\n");
		fprintf (stderr, "	      -Nt useful for gradients to -I option of grdimage.\n");
		explain_option ('V');
		exit (-1);
	}

	if (!outfile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G option:  Must specify output file\n", gmt_program);
		error++;
	}
	if (!infile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify input file\n", gmt_program);
		error++;
	}
	if (azim < 0.0 || azim >= 360.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -A option:  Use 0-360 degree range\n", gmt_program);
		error++;
	}
	if (error) exit (-1);

	read_grd_info (infile, &header);
	grd_init (&header, argc, argv, TRUE);
	nm = header.nx * header.ny;
	mx = header.nx + 2;
	my = header.ny + 2;
	nm2 = mx * my;
	data = (float *) memory (CNULL, nm2, sizeof (float), "grdgradient");
	pad[0] = pad[1] = pad[2] = pad[3] = 1;
	
	read_grd (infile, &header, data, header.x_min, header.x_max, header.y_min, header.y_max, pad, FALSE);
	
	/* set boundary conditions:  */

	for (i = 1; i < mx-1; i++) {	/* Along S and N */
		ij = mx + i;
		data[ij-mx] = 2 * data[ij] - data[ij+mx];
		ij = header.ny * mx + i;
		data[ij+mx] = 2 * data[ij] - data[ij-mx];
	}
	for (j = 1; j < my-1; j++) {	/* Along W and E */
		ij = j * mx + 1;
		data[ij-1] = 2 * data[ij] - data[ij+1];
		ij += (header.nx - 1);
		data[ij+1] = 2 * data[ij] - data[ij-1];
	}

	if (map_units) {
		m_pr_degree = 2.0 * M_PI * gmtdefs.ellipse[gmtdefs.ellipsoid].eq_radius / 360.0;
		dx_grid = m_pr_degree * header.x_inc * cos (M_PI * (header.y_max + header.y_min) / 360.0);
		dy_grid = m_pr_degree * header.y_inc;
	}
	else {
		dx_grid = header.x_inc;
		dy_grid = header.y_inc;
	}
	x_factor = -1.0 / (2.0 * dx_grid);
	y_factor = -1.0 / (2.0 * dy_grid);
	azim *= (M_PI / 180.0);
	x_factor *= sin(azim);
	y_factor *= cos(azim);

	p[0] = 1;	p[1] = -1;	p[2] = mx;	p[3] = -mx;
	
	min_gradient = MAXDOUBLE;	max_gradient = -MAXDOUBLE;
	ave_gradient = 0.0;
	for (j = k = 0; j < header.ny; j++) {
		if (map_units) {
			lat = (header.node_offset) ? -header.y_inc * (j + 0.5) : -header.y_inc * j;
			lat += header.y_max;
			dx_grid = m_pr_degree * header.x_inc * cos (D2R * lat);
			x_factor = -1.0 / (2.0 * dx_grid);
			x_factor *= sin(azim);
		}
		for (i = 0; i < header.nx; i++, k++) {
			ij = (j + 1) * mx + i + 1;
			for (n = 0, bad = FALSE; !bad && n < 4; n++) if (bad_float ((double)data[ij+p[n]])) bad = TRUE;
			if (bad) {	/* One of corners = Nan, skip */
				data[k] = gmt_NaN;
				continue;
			}
			
			dzdx = (data[ij+1] - data[ij-1]) * x_factor;
			dzdy = (data[ij-mx] - data[ij+mx]) * y_factor;
			data[k] = dzdx + dzdy;	/* Write to unused NW corner  */
			ave_gradient += data[k];
			min_gradient = MIN (min_gradient, data[k]);
			max_gradient = MAX (max_gradient, data[k]);
			n_used++;
		}
	}
	
	ave_gradient /= n_used;
	
	if (normalize) {
                if (atan_trans) {
                        denom = 0.0;
                        for (k = 0; k < nm; k++) if (!bad_float ((double)data[k])) denom += pow(data[k] - ave_gradient, 2.0);
                        denom = sqrt( (n_used - 1) / denom);
                        rpi = 2.0 * norm_val / M_PI;
                        for (k = 0; k < nm; k++) if (!bad_float ((double)data[k])) data[k] = (float)(rpi * atan((data[k] - ave_gradient)*denom));
                }
                else {
			if ( (max_gradient - ave_gradient) > (ave_gradient - min_gradient) ) {
				denom = norm_val / (max_gradient - ave_gradient);
			}
			else {
				denom = norm_val / (ave_gradient - min_gradient);
			}
			for (k = 0; k < nm; k++) if (!bad_float ((double)data[k])) data[k] = (data[k] - ave_gradient) * denom;
		}
	}
			
	/* Now we write out: */
	
	strcpy (header.title, "gradients");
	pad[0] = pad[1] = pad[2] = pad[3] = 0;	/* Because of the shift */
	write_grd (outfile, &header, data, 0.0, 0.0, 0.0, 0.0, pad, FALSE);

	free ((char *) data);
	
	if (gmtdefs.verbose) fprintf (stderr ,"grdgradient:  Min Mean Max intensities:\t%.8lg\t%.8lg\t%.8lg\n", min_gradient, ave_gradient, max_gradient);
	
	gmt_end (argc, argv);
}
