/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdinfo.c	2.22  5/25/95
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * grdinfo reads one or more grd file and [optionally] prints out various
 * statistics like mean/standard deviation and median/scale
 *
 * Author:	Paul Wessel
 * Date:	4-JAN-1991-1995
 * Version:	3.0	Based on 2.x
 */

#include "gmt.h"

float *a;
char *type[2] = { "Normal", "Pixel"};

main (argc, argv)
int argc;
char **argv; {
	int nfiles = 0, k, i, j, i_min, i_max, nm, n_nan = 0, n, dummy[4];
	
	BOOLEAN error = FALSE, l1 = FALSE, l2 = FALSE, quick = TRUE, find_max = FALSE;
	
	double x_min, y_min, z_min, x_max, y_max, z_max;
	double mean, median, sum2, stdev, scale, rms, x;
	
	char file[BUFSIZ];
	
	struct GRD_HEADER grd;
	
	argc = gmt_begin (argc, argv);

	dummy[0] = dummy[1] = dummy[2] = dummy[3] = 0;

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */
			
				case '\0':
					error += get_common_args (argv[i], 0, 0, 0, 0);
					break;
				
				/* Supplemental parameters */
			
				case 'M':
					quick = FALSE;
					find_max = TRUE;
					break;
				case 'L':
					quick = FALSE;
					if (argv[i][2] == 0 || argv[i][2] == '2')
						l2 = TRUE;
					else if (argv[i][2] == '1')
						l1 = TRUE;
					else {
						error = TRUE;
						fprintf (stderr, "%s: GMT SYNTAX ERROR -L option:  Choose between -L1 or -L2\n", gmt_program);
					}
					break;
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else
			nfiles ++;
	}
	
	if (argc == 1 || gmt_quick) {
		fprintf (stderr, "grdinfo %s - Extract information from netCDF grdfiles\n\n", GMT_VERSION);
		fprintf (stderr, "usage: grdinfo <grdfiles> [-L1] [-L[2]] [-M]\n");
		
		if (gmt_quick) exit (-1);
		
		fprintf (stderr, "	<grdfiles> may be one or more netCDF grdfiles\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		fprintf (stderr, "	-L1 will report median/L1-scale of data set\n");
		fprintf (stderr, "	-L[2] will report mean/standard deviation/rms of data set\n");
		fprintf (stderr, "	-M will search for location of min/max z-values\n");
		exit (-1);
	}
	
	if (nfiles == 0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR: Must specify one or more input files\n", gmt_program);
		error++;
	}
	if (error) exit (-1);

	a = (float *) memory (CNULL, 1, sizeof (float), "grdinfo");

	for (k = 1; k < argc; k++) {	/* Loop over arguments, skip options */
	
		if (argv[k][0] == '-') continue;
		
		strcpy (file, argv[k]);
		for (j = 0; file[j]; j++) if (file[j] == '=') file[j] = 0;
		if (strcmp (file, "=") && access (file, R_OK)) {
			fprintf (stderr, "grdinfo:  File %s not found\n", file);
			continue;
		}

		grd_init (&grd, argc, argv, FALSE);

		if (read_grd_info (argv[k], &grd)) {
			fprintf (stderr, "grdinfo: Error opening file %s\n", file);
			continue;
		}
		if (grd.z_min == grd.z_max && grd_i_format) quick = FALSE, find_max = TRUE;
		
		printf ("%s: Title: %s\n", file, grd.title);
		printf ("%s: Command: %s\n", file, grd.command);
		printf ("%s: Remark: %s\n", file, grd.remark);
		printf ("%s: %s node registation used\n", file, type[grd.node_offset]);
		printf ("%s: grdfile format # %d\n", file, grd_i_format);
		printf ("%s: x_min: %lg x_max: %lg x_inc: %lg units: %s nx: %d\n", file, grd.x_min, grd.x_max, grd.x_inc, grd.x_units, grd.nx);
		printf ("%s: y_min: %lg y_max: %lg y_inc: %lg units: %s ny: %d\n", file, grd.y_min, grd.y_max, grd.y_inc, grd.y_units, grd.ny);
		if (!quick) {
			nm = grd.nx * grd.ny;
			a = (float *) memory ((char *)a, nm, sizeof (float), "grdinfo");
			if (read_grd (argv[k], &grd, a, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) continue;
		
			z_min = MAXDOUBLE;	z_max = -MAXDOUBLE;
			mean = median = sum2 = 0.0;
			i_min = i_max = 0;
			n_nan = 0;
			for (i = 0; i < nm; i++) {
				if (bad_float ((double)a[i])) {
					n_nan++;
					continue;
				}
				if (find_max) {
					if (a[i] < z_min) {
						z_min = a[i];
						i_min = i;
					}
					if (a[i] > z_max) {
						z_max = a[i];
						i_max = i;
					}
				}
				if (l2) {
					mean += a[i];
					sum2 += a[i]*a[i];
				}
			}
		
			x_min = grd.x_min + (i_min % grd.nx) * grd.x_inc;
			y_min = grd.y_max - (i_min / grd.nx) * grd.y_inc;
			x_max = grd.x_min + (i_max % grd.nx) * grd.x_inc;
			y_max = grd.y_max - (i_max / grd.nx) * grd.y_inc;
		}
		
		if (find_max) 
			printf ("%s: zmin: %lg (at %lg/%lg) zmax: %lg (at %lg/%lg) units: %s\n", file, z_min, x_min, y_min, z_max, x_max, y_max, grd.z_units);
		else
			printf ("%s: zmin: %lg zmax: %lg units: %s\n", file, grd.z_min, grd.z_max, grd.z_units);

		printf ("%s: scale_factor: %lg add_offset: %lg\n", file, grd.z_scale_factor, grd.z_add_offset);
		if (n_nan) printf ("%s: %d nodes set to NaN\n", file, n_nan);
		if (l1) {
			qsort ((char *)a, nm, sizeof (float), comp_float_asc);
			n = nm - n_nan;
			median = (n%2) ? a[n/2] : 0.5*(a[n/2-1] + a[n/2]);
			for (i = 0; i < n; i++) a[i] = fabs (a[i] - median);
			qsort ((char *)a, n, sizeof (float), comp_float_asc);
			scale = (n%2) ? 1.4826 * a[n/2] : 0.7413 * (a[n/2-1] + a[n/2]);
			printf ("%s: median: %lg scale: %lg\n", file, median, scale);
		}
		if (l2) {
			x = nm - n_nan;
			stdev = sqrt((x*sum2 - mean*mean)/(x*(x-1)));
			rms = sqrt (sum2 / x);
			mean /= x;
			printf ("%s: mean: %lg stdev: %lg rms: %lg\n", file, mean, stdev, rms);
		}
	}
	
	free ((char *)a);
	
	gmt_end (argc, argv);
}
