/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdmath_main.c	3.4  06 Jul 1995
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * grdmath.c is a reverse polish calculator that operates on grd files 
 * (and constants) and perform basic matheatical operations
 * on them like add, multiply, etc for a total of 29 operators.
 * Some operators only work on one operand (e.g., log, exp)
 *
 * Author:	Paul Wessel
 * Date:	22-AUG-1988
 * Revised:	27-JUN-1995
 * Version:	3.0 based on old 2.1
 *
 */
 
#include "gmt.h"

#define N_OPERATORS	41
#define STACK_SIZE	50

#define ARG_IS_FILE	0
#define ARG_IS_NUMBER	1
#define ARG_IS_PI	2
#define ARG_IS_E	3
#define ARG_IS_X_MATRIX	4
#define ARG_IS_Y_MATRIX	5
#define ARG_IS_OPERATOR	6

#include "grdmath_def.inc"

PFI call_operator[N_OPERATORS];

int nm = 0, consumed_operands[N_OPERATORS], produced_operands[N_OPERATORS];

struct HASH hashnode[HASH_SIZE];

main (argc, argv)
int argc;
char **argv; {
	int i, j, k, arg, op = 0, nstack = 0, new_stack = -1, last_arg, ok = 1, type, dummy[4];
	
	BOOLEAN constant[STACK_SIZE], error = FALSE, set_r = FALSE, set_inc = FALSE;
	
	float *stack[STACK_SIZE], *x, *y;

	double factor[STACK_SIZE], d;
	
	char *outfile = 0, file[BUFSIZ];
	
	struct GRD_HEADER grd[STACK_SIZE], header;
	
	argc = gmt_begin (argc, argv);
	
	if (argc == 2 && !strcmp (argv[1], "-")) error = gmt_quick = TRUE;
	
	if (argc == 1 || gmt_quick) {
		fprintf (stderr, "grdmath %s - Reverse Polish calculator for grdfiles (element by element)\n\n", GMT_VERSION);
		fprintf (stderr, "usage: grdmath [-R<west/east/south/north> -I<xinc[m|c]>[/<yinc>[m|c] -V] A B op C op D op ... = outfile\n\n");
		
		if (gmt_quick) exit (-1);
		
		fprintf (stderr, "	A, B, etc are grdfiles, constants, or symbols (see below)\n");
		fprintf (stderr, "	The stack can hold up to %d entries (given enough memory)\n", STACK_SIZE);
		fprintf (stderr, "	Trigonometric operators expect radians.  The operators are:\n\n");
		fprintf (stderr, "	Name	#args	Returns:\n");
		fprintf (stderr, "	-----------------------\n");
#include "grdmath_explain.inc"
		fprintf (stderr, "\n	The special symbols are:\n\n");
		fprintf (stderr, "	  PI	= 3.1415926...\n");
		fprintf (stderr, "	  E	= 2.7182818...\n");
		fprintf (stderr, "	  X	= grid with x-coordinates\n");
		fprintf (stderr, "	  Y	= grid with y-coordinates\n");
		fprintf (stderr, "\n\tOPTIONS: (only used if no grdfiles are passed as arguments)\n\n");
		fprintf (stderr, "\t-F Set pixel grid registration [Default is gridline orientation]\n");
		fprintf (stderr, "\t-I sets Increment of the grid; enter xinc, optionally xinc/yinc.\n");
		fprintf (stderr, "\t   Default is yinc = xinc.  Append an m [or c] to indicate minutes [or seconds].\n");
		explain_option ('R');
		explain_option ('V');
		exit (-1);
	}
	
	if (argv[argc-2][0] != '=') {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Usage is <operations> = outfile\n", gmt_program);
		exit (-1);
	}
	
	/* Determine if the user have files with stupid names that will cause conflict */

	for (op = 0; op < N_OPERATORS; op++) {
		if (!access (operator[op], R_OK))
			fprintf (stderr, "%s: Your file %s may be confused with a grdmath operator!\n", gmt_program, operator[op]);
	}
	if (!access ("PI", R_OK)) fprintf (stderr, "%s: Your file PI may be confused with a grdmath symbol!\n", gmt_program);
	if (!access ("pi", R_OK)) fprintf (stderr, "%s: Your file pi may be confused with a grdmath symbol!\n", gmt_program);
	if (!access ("E", R_OK)) fprintf (stderr, "%s: Your file E may be confused with a grdmath symbol!\n", gmt_program);
	if (!access ("e", R_OK)) fprintf (stderr, "%s: Your file e may be confused with a grdmath symbol!\n", gmt_program);
	if (!access ("X", R_OK)) fprintf (stderr, "%s: Your file X may be confused with a grdmath symbol!\n", gmt_program);
	if (!access ("x", R_OK)) fprintf (stderr, "%s: Your file x may be confused with a grdmath operator!\n", gmt_program);
	if (!access ("Y", R_OK)) fprintf (stderr, "%s: Your file Y may be confused with a grdmath symbol!\n", gmt_program);
	if (!access ("+", R_OK)) fprintf (stderr, "%s: Your file + may be confused with a grdmath operator!\n", gmt_program);
	if (!access ("-", R_OK)) fprintf (stderr, "%s: Your file - may be confused with a grdmath operator!\n", gmt_program);
	if (!access ("^", R_OK)) fprintf (stderr, "%s: Your file ^ may be confused with a grdmath operator!\n", gmt_program);

	init_gmt_hash (hashnode, operator, HASH_SIZE, N_OPERATORS);

	dummy[3] = dummy[2] = dummy[1] = dummy[0] = 0;
	for (i = 0; i < STACK_SIZE; i++) {
		constant[i] = FALSE;
		factor[i] = 0.0;
		grd[i].nx = grd[i].ny = 0;
		stack[i] = (float *)NULL;
	}
		
	outfile = argv[argc-1];
	
	last_arg = argc - 2;
	
	/* Get header from one file so we can allocate space */
	
	grd_init (&header, argc, argv, TRUE);

	for (arg = 1; nm == 0 && arg < last_arg; arg++) {
	
		if (decode_argument (argv[arg]) != ARG_IS_FILE) continue;
		
		strcpy (file, argv[arg]);
		for (j = 0; file[j]; j++) if (file[j] == '=') file[j] = 0;
		if (read_grd_info (argv[arg], &header)) {
			fprintf (stderr, "grdmath: Error opening file %s\n", file);
			exit (-1);
		}
		
		nm = header.nx * header.ny;
	}
	
	/* Scan command line for -R, -I */

	for (arg = 1; arg < last_arg; arg++) {
		if (argv[arg][0] == '-') {

			switch (argv[arg][1]) {

				case 'R':
					set_r = TRUE;
				case 'V':
					error += get_common_args (argv[arg], &header.x_min, &header.x_max, &header.y_min, &header.y_max);
					break;

				case 'F':
					header.node_offset = 1;
					break;

				case 'I':
					gmt_getinc (&argv[arg][2], &header.x_inc, &header.y_inc);
					set_inc = (header.x_inc > 0 && header.y_inc > 0);
					break;
			}
		}
	}
	if (nm && set_r && set_inc) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Cannot use -R, -I when grdfiles are specified\n", gmt_program);
		exit (-1);
	}
	if (set_r && set_inc) {
		header.nx = rint ((header.x_max - header.x_min) / header.x_inc) + !header.node_offset;
		header.ny = rint ((header.y_max - header.y_min) / header.y_inc) + !header.node_offset;
		nm = header.nx * header.ny;
	}
	
	if (nm == 0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Expression must contain at least one grdfile or -R, -I\n", gmt_program);
		exit (-1);
	}
	
	stack[0] = (float *) memory (CNULL, nm, sizeof (float), "grdmath");

	/* Get x and y vectors */

	x = (float *) memory (CNULL, header.nx, sizeof (float), gmt_program);
	y = (float *) memory (CNULL, header.ny, sizeof (float), gmt_program);
	d = (header.node_offset) ? 0.5 : 0.0;
	for (j = 0; j < header.ny; j++) y[j] = (j == (header.ny-1)) ? header.y_min + d * header.y_inc : header.y_max - (j + d) * header.y_inc;
	for (i = 0; i < header.nx; i++) x[i] = (i == (header.nx-1)) ? header.x_max - d * header.x_inc: header.x_min + (i + d) * header.x_inc;

	grdmath_init (call_operator, consumed_operands, produced_operands);

	nstack = 0;
	
	for (arg = 1; !error && arg < last_arg; arg++) {
	
		/* First check if we should skip optional arguments */

		if (!(strncmp (argv[arg], "-R", 2) && strncmp (argv[arg], "-I", 2) && strcmp (argv[arg], "-V"))) continue;

		if ((type = decode_argument (argv[arg])) != ARG_IS_OPERATOR) {	/* File name or factor */
		
			if (nstack == STACK_SIZE) {	/* Stack overflow */
				error = TRUE;
				continue;
			}

			if (type == ARG_IS_NUMBER) {
				constant[nstack] = TRUE;
				ok = sscanf (argv[arg], "%lf", &factor[nstack]);
				error = !ok;
				if (gmtdefs.verbose) fprintf (stderr, "%lg ", factor[nstack]);
				nstack++;
				continue;
			}
			else if (type == ARG_IS_PI) {
				constant[nstack] = TRUE;
				factor[nstack] = M_PI;
				if (gmtdefs.verbose) fprintf (stderr, "%lg ", factor[nstack]);
				nstack++;
				continue;
			}
			else if (type == ARG_IS_E) {
				constant[nstack] = TRUE;
				factor[nstack] = M_E;
				if (gmtdefs.verbose) fprintf (stderr, "%lg ", factor[nstack]);
				nstack++;
				continue;
			}

			/* Here we need a matrix */

			grd_init (&grd[nstack], argc, argv, TRUE);
			if (!stack[nstack]) stack[nstack] = (float *) memory (CNULL, nm, sizeof (float), "grdmath");
			constant[nstack] = FALSE;

			if (type == ARG_IS_X_MATRIX) {		/* Need to set up matrix of x-values */
				if (gmtdefs.verbose) fprintf (stderr, "X ");
				for (j = k = 0; j < header.ny; j++, k += header.nx)
					memcpy ((char *)&stack[nstack][k], (char *)x, header.nx * sizeof (float));
			}
			else if (type == ARG_IS_Y_MATRIX) {	/* Need to set up matrix of y-values */
				if (gmtdefs.verbose) fprintf (stderr, "Y ");
				for (j = k = 0; j < header.ny; j++) for (i = 0; i < header.nx; i++, k++)
					stack[nstack][k] = y[j];
			}
			else if (type == ARG_IS_FILE) {		/* Filename given */
				if (gmtdefs.verbose) fprintf (stderr, "%s ", argv[arg]);
				if (read_grd_info (argv[arg], &grd[nstack])) {
					fprintf (stderr, "grdmath: Error opening file %s\n", argv[arg]);
					exit (-1);
				}
				if (grd[nstack].nx != header.nx || grd[nstack].ny != header.ny) {
					fprintf (stderr, "grdmath: grd files not of same size!\n");
					exit (-1);
				}
				else if (grd[nstack].x_min != header.x_min || grd[nstack].x_max != header.x_max || 
					grd[nstack].y_min != header.y_min || grd[nstack].y_max != header.y_max) {
					fprintf (stderr, "grdmath: grd files do not cover the same area!\n");
					exit (-1);
				}
				if (read_grd (argv[arg], &grd[nstack], stack[nstack], 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
					fprintf (stderr, "grdmath: Error reading file %s\n", argv[arg]);
					exit (-1);
				}

			}
			nstack++;
			continue;
		}
		
		/* Here we have an operator */
		
		if ((op = get_operator (argv[arg])) < 0) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  Unrecognized operator %s\n", gmt_program, argv[arg]);
			exit (-1);
		}

		if ((new_stack = nstack - consumed_operands[op] + produced_operands[op]) >= STACK_SIZE) {
			error = TRUE;
			continue;
		}

		if (nstack < consumed_operands[op]) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  Operation \"%s\" requires %d operands\n", gmt_program, operator[op], consumed_operands[op]);
			exit (-1);
		}
		
		if (gmtdefs.verbose) fprintf (stderr, "%s ", operator[op]);

		if (produced_operands[op] > consumed_operands[op] && !stack[nstack+1])	/* Must make space for more */
			stack[nstack+1] = (float *) memory (CNULL, nm, sizeof (float), "grdmath");

		(*call_operator[op]) (stack, constant, factor, nstack - 1);	/* Do it */

		nstack = new_stack;
		
		constant[nstack-1] = FALSE;	/* Now filled with grid */
	}
	
	if (error && !ok) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Unable to decode constant %s (File not found?)\n", gmt_program, argv[i-1]);
		exit (-1);
	}
	
	if (error) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Stack overflow (%s)\n", gmt_program, argv[i-1]);
		exit (-1);
	}
	
	if (gmtdefs.verbose) fprintf (stderr, "= %s", outfile);

	grd_init (&header, argc, argv, TRUE);
	
	if (new_stack < 0 && constant[0]) {	/* Only a constant provided, set grid accordingly */
		for (k = 0; k < nm; k++) stack[0][k] = factor[0];
	}

	if (write_grd (outfile, &header, stack[0], 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
		fprintf (stderr, "grdmath: Error writing file %s\n", outfile);
		exit (-1);
	}
	
	for (i = 0; i < STACK_SIZE; i++) if (stack[i]) free ((char *)stack[i]);
	free ((char *)x);
	free ((char *)y);
	
	if (gmtdefs.verbose) fprintf (stderr, "\n");

	gmt_end (argc, argv);
}

int get_operator (choice)
char *choice; {
	int op;
	
	/* Returns -1 if not a registered operator */

	op = gmt_hash_entry (choice, hashnode, HASH_SIZE);

	if (op < 0 && strlen (choice) == 1) {	/* Check for old-style operators */
	
		switch (choice[0]) {
			case '+':
				op = ADD;
				break;
			case '-':
				op = SUB;
				break;
			case 'x':
				op = MUL;
				break;
			case '/':
				op = DIV;
				break;
			case '^':
				op = RAISE;
				break;
		}
	}

	return (op);
}

/* -----------------------------------------------------------------
 *              Definitions of all operator functions
 * -----------------------------------------------------------------*/

int grd_ABS (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = fabs (a);
	}
}

int grd_ACOS (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && fabs (factor[last]) > 1.0) fprintf (stderr, "grdmath: Warning, |operand| > 1 for ACOS!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = d_acos (a);
	}
}

int grd_ACOSH (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && fabs (factor[last]) > 1.0) fprintf (stderr, "grdmath: Warning, operand < 1 for ACOSH!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = acosh (a);
	}
}

int grd_ADD (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	if (constant[prev] && factor[prev] == 0.0) fprintf (stderr, "grdmath: Warning, operand one == 0!\n");
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand two == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = a + b;
	}
}

int grd_AND (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = (bad_float (a)) ? b : a;
	}
}

int grd_ASIN (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && fabs (factor[last]) > 1.0) fprintf (stderr, "grdmath: Warning, |operand| > 1 for ASIN!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = d_asin (a);
	}
}

int grd_ASINH (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = asinh (a);
	}
}

int grd_ATAN (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = atan (a);
	}
}

int grd_ATAN2 (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	if (constant[prev] && factor[prev] == 0.0) fprintf (stderr, "grdmath: Warning, operand one == 0 for ATAN2!\n");
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand two == 0 for ATAN2!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = d_atan2 (a, b);
	}
}

int grd_ATANH (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && fabs (factor[last]) >= 1.0) fprintf (stderr, "grdmath: Warning, |operand| >= 1 for ATANH!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = atanh (a);
	}
}

int grd_CEIL (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = ceil (a);
	}
}

int grd_COS (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = cos (a);
	}
}

int grd_COSH (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = cosh (a);
	}
}

int grd_DIV (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	
	if (constant[last] && factor[last] == 0.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Cannot divide by zero\n", gmt_program);
		exit (-1);
	}
	if (constant[last]) {	/* Turn divide into multiply */
		factor[last] = 1.0 / factor[last];
		grd_MUL (stack, constant, factor, last);
		return;
	}
	
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = a / b;
	}
}

int grd_DUP (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int next;
	
	next = last + 1;
	memcpy ((char *)stack[next], (char *)stack[last], nm * sizeof (float));
}

int grd_EXCH (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	
	prev = last - 1;
	for (i = 0; i < nm; i++) f_swap (stack[last][i], stack[prev][i]);
}

int grd_EXP (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = exp (a);
	}
}

int grd_FLOOR (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = floor (a);
	}
}

int grd_FMOD (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	if (constant[prev] && factor[prev] == 0.0) fprintf (stderr, "grdmath: Warning, operand one == 0!\n");
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand two == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = fmod (a, b);
	}
}

int grd_HYPOT (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	if (constant[prev] && factor[prev] == 0.0) fprintf (stderr, "grdmath: Warning, operand one == 0!\n");
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand two == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = hypot (a, b);
	}
}

int grd_INV (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && factor[last] == 0.0) {
		fprintf (stderr, "grdmath: Error, Cannot take inverse of zero!\n");
		exit (-1);
	}
	if (constant[last]) factor[last] = 1.0 / factor[last];
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : 1.0 / stack[last][i];
		stack[last][i] = a;
	}
}

int grd_LOG (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && factor[last] == 0.0) {
		fprintf (stderr, "grdmath: Error, Cannot take log10 of zero!\n");
		exit (-1);
	}
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = d_log (a);
	}
}

int grd_LOG10 (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && factor[last] == 0.0) {
		fprintf (stderr, "grdmath: Error, Cannot take log10 of zero!\n");
		exit (-1);
	}
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = d_log10 (a);
	}
}

int grd_MAX (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = MAX (a, b);
	}
}

int grd_MEAN (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, n_a = 0;
	double sum_a = 0.0;

	if (constant[last]) {	/* Trivial case */
		for (i = 0; i < nm; i++) stack[last][i] = factor[last];
		return;
	}
	
	for (i = 0; i < nm; i++) {
		if (bad_float ((double)stack[last][i])) continue;
		sum_a += stack[last][i];
		n_a++;
	}
	sum_a = (n_a) ? sum_a / n_a : 0.0;
	for (i = 0; i < nm; i++) stack[last][i] = sum_a;
}

int grd_MED (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double med;
	
	if (constant[last]) {	/* Trivial case */
		for (i = 0; i < nm; i++) stack[last][i] = factor[last];
		return;
	}

	qsort ((char *)stack[last], nm, sizeof (float), comp_float_asc);
	for (i = nm-1; bad_float ((double)stack[last][i]) && i > 0; i--);
	if (i)
		med = (i%2) ? stack[last][i/2] : 0.5 * (stack[last][(i-1)/2] + stack[last][i/2]);
	else
		med = gmt_NaN;

	for (i = 0; i < nm; i++) stack[last][i] = med;
}

int grd_MIN (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = MIN (a, b);
	}
}

int grd_MUL (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	if (constant[prev] && factor[prev] == 0.0) fprintf (stderr, "grdmath: Warning, operand one == 0!\n");
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand two == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = a * b;
	}
}

int grd_NEG (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = -a;
	}
}

int grd_OR (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = (bad_float (a) || bad_float (b)) ? gmt_NaN : a;
	}
}

int grd_POW (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;

	if (constant[prev] && factor[prev] == 0.0) fprintf (stderr, "grdmath: Warning, operand one == 0!\n");
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand two == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = pow (a, b);
	}
}

int grd_R2 (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	if (constant[prev] && factor[prev] == 0.0) fprintf (stderr, "grdmath: Warning, operand one == 0!\n");
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand two == 0!\n");
	if (constant[prev]) factor[prev] *= factor[prev];
	if (constant[last]) factor[last] *= factor[last];
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i] * stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i] * stack[last][i];
		stack[prev][i] = a + b;
	}
}

int grd_RINT (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = rint (a);
	}
}

int grd_SIGN (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = copysign (1.0, a);
	}
}

int grd_SIN (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = sin (a);
	}
}

int grd_SINH (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = sinh (a);
	}
}

int grd_SQRT (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand one == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = sqrt (a);
	}
}

int grd_STD (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, n_a = 0;
	double sum_a = 0.0, sum_a2 = 0.0, std;
	
	if (constant[last]) {	/* Trivial case */
		memset ((char *)stack[last], 0, nm * sizeof (float));
		return;
	}

	for (i = 0; i < nm; i++) {
		if (bad_float ((double)stack[last][i])) continue;
		sum_a += stack[last][i];
		sum_a2 += (stack[last][i] * stack[last][i]);
		n_a++;
	}
	if (n_a > 1)
		std = sqrt ((n_a * sum_a2 - sum_a * sum_a) / (n_a * (n_a - 1.0)));
	else
		std = 0.0;
	for (i = 0; i < nm; i++) stack[last][i] = std;
}

int grd_SUB (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i, prev;
	double a, b;
	
	prev = last - 1;
	if (constant[prev] && factor[prev] == 0.0) fprintf (stderr, "grdmath: Warning, operand one == 0!\n");
	if (constant[last] && factor[last] == 0.0) fprintf (stderr, "grdmath: Warning, operand two == 0!\n");
	for (i = 0; i < nm; i++) {
		a = (constant[prev]) ? factor[prev] : stack[prev][i];
		b = (constant[last]) ? factor[last] : stack[last][i];
		stack[prev][i] = a - b;
	}
}

int grd_TAN (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = tan (a);
	}
}

int grd_TANH (stack, constant, factor, last)
float *stack[];
BOOLEAN constant[];
double factor[];
int last;
{
	int i;
	double a;
	
	for (i = 0; i < nm; i++) {
		a = (constant[last]) ? factor[last] : stack[last][i];
		stack[last][i] = tanh (a);
	}
}

/* ---------------------- end operator functions --------------------- */

int decode_argument (txt)
char *txt; {
	int j, number, minus, plus, exp, dec, n_digits = 0;
	char *s, file[BUFSIZ];
	
	if (!strcmp (txt, "/")) return ARG_IS_OPERATOR;	/* Special test since / is also a directory */

	/* First see if argument it can be opened as a file */
	
	strcpy (file, txt);
	for (j = 0; file[j]; j++) if (file[j] == '=') file[j] = 0;
	if (!access (file, R_OK)) return ARG_IS_FILE; 	/* returns FALSE if file exists and can be read */

	/* Next look for symbols with special meaning */

	if (!(strcmp (txt, "PI") && strcmp (txt, "pi"))) return ARG_IS_PI;
	if (!(strcmp (txt, "E") && strcmp (txt, "e"))) return ARG_IS_E;
	if (!strcmp (txt, "X")) return ARG_IS_X_MATRIX;
	if (!(strcmp (txt, "Y") && strcmp (txt, "y"))) return ARG_IS_Y_MATRIX;

	/* Here we must check if argument is a numerical value */

	s = txt;
	if (*s == '-' || *s == '+') s++;	/* Skip leading sign */
	
	minus = plus = exp = dec = 0;
	number = TRUE;

	while (number && *s) {
		if (isdigit (*s))
			n_digits++;
		else {
			switch (*s) {
				case '-':
					minus++;
					break;
				case '+':
					plus++;
					break;
				case 'E':
				case 'e':
					exp++;
					break;
				case '.':
					dec++;
					break;
				default:
					number = FALSE;
					break;
			}
		}
		if (minus > 1 || exp > 1 || dec > 1) number = FALSE;
		s++;
	}
	
	return ((number && n_digits > 0) ? ARG_IS_NUMBER : ARG_IS_OPERATOR);
	
}

int grdmath_init (ops, n_args, n_out)
PFI ops[];
int n_args[], n_out[]; {

	/* Operator function		# of operands  		# of outputs */

	ops[0] = grd_ABS;		n_args[0] = 1;		n_out[0] = 1;
	ops[1] = grd_ACOS;		n_args[1] = 1;		n_out[1] = 1;
	ops[2] = grd_ACOSH;		n_args[2] = 1;		n_out[2] = 1;
	ops[3] = grd_ADD;		n_args[3] = 2;		n_out[3] = 1;
	ops[4] = grd_AND;		n_args[4] = 2;		n_out[4] = 1;
	ops[5] = grd_ASIN;		n_args[5] = 1;		n_out[5] = 1;
	ops[6] = grd_ASINH;		n_args[6] = 1;		n_out[6] = 1;
	ops[7] = grd_ATAN;		n_args[7] = 1;		n_out[7] = 1;
	ops[8] = grd_ATAN2;		n_args[8] = 2;		n_out[8] = 1;
	ops[9] = grd_ATANH;		n_args[9] = 1;		n_out[9] = 1;
	ops[10] = grd_CEIL;		n_args[10] = 1;		n_out[10] = 1;
	ops[11] = grd_COS;		n_args[11] = 1;		n_out[11] = 1;
	ops[12] = grd_COSH;		n_args[12] = 1;		n_out[12] = 1;
	ops[13] = grd_DIV;		n_args[13] = 2;		n_out[13] = 1;
	ops[14] = grd_DUP;		n_args[14] = 1;		n_out[14] = 2;
	ops[15] = grd_EXCH;		n_args[15] = 2;		n_out[15] = 2;
	ops[16] = grd_EXP;		n_args[16] = 1;		n_out[16] = 1;
	ops[17] = grd_FLOOR;		n_args[17] = 1;		n_out[17] = 1;
	ops[18] = grd_FMOD;		n_args[18] = 2;		n_out[18] = 1;
	ops[19] = grd_HYPOT;		n_args[19] = 2;		n_out[19] = 1;
	ops[20] = grd_INV;		n_args[20] = 1;		n_out[20] = 1;
	ops[21] = grd_LOG;		n_args[21] = 1;		n_out[21] = 1;
	ops[22] = grd_LOG10;		n_args[22] = 1;		n_out[22] = 1;
	ops[23] = grd_MAX;		n_args[23] = 2;		n_out[23] = 1;
	ops[24] = grd_MEAN;		n_args[24] = 1;		n_out[24] = 1;
	ops[25] = grd_MED;		n_args[25] = 1;		n_out[25] = 1;
	ops[26] = grd_MIN;		n_args[26] = 2;		n_out[26] = 1;
	ops[27] = grd_MUL;		n_args[27] = 2;		n_out[27] = 1;
	ops[28] = grd_NEG;		n_args[28] = 1;		n_out[28] = 1;
	ops[29] = grd_OR;		n_args[29] = 2;		n_out[29] = 1;
	ops[30] = grd_POW;		n_args[30] = 2;		n_out[30] = 1;
	ops[31] = grd_R2;		n_args[31] = 2;		n_out[31] = 1;
	ops[32] = grd_RINT;		n_args[32] = 1;		n_out[32] = 1;
	ops[33] = grd_SIGN;		n_args[33] = 1;		n_out[33] = 1;
	ops[34] = grd_SIN;		n_args[34] = 1;		n_out[34] = 1;
	ops[35] = grd_SINH;		n_args[35] = 1;		n_out[35] = 1;
	ops[36] = grd_SQRT;		n_args[36] = 1;		n_out[36] = 1;
	ops[37] = grd_STD;		n_args[37] = 1;		n_out[37] = 1;
	ops[38] = grd_SUB;		n_args[38] = 2;		n_out[38] = 1;
	ops[39] = grd_TAN;		n_args[39] = 1;		n_out[39] = 1;
	ops[40] = grd_TANH;		n_args[40] = 1;		n_out[40] = 1;
}
