/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdpaste.c	2.13  3/13/95
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * grdpaste.c reads two grdfiles and writes a new file with
 * the first two pasted together along their common edge.
 *
 * Author:	Walter Smith
 * Date:	5 august, 1988
 * Updated to v2.0 20-May-1991-1995 Paul Wessel
 *
 */
 
#include "gmt.h"

float	*c;

main (argc, argv)
int argc;
char **argv;
{
	BOOLEAN	error = FALSE;
	int	i, way, one_or_zero, pad[4];
	char *grd_a, *grd_b, *grd_c;
	struct GRD_HEADER ha, hb, hc;
	
	argc = gmt_begin (argc, argv);

	grd_a = grd_b = grd_c = CNULL;
	
	/* Check and interpret the command line arguments */
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch(argv[i][1]) {
				/* Common parameters */
			
				case 'V':
				case '\0':
					error += get_common_args (argv[i], 0, 0, 0, 0);
					break;
					
				/* Supplemental parameters */
			
	 			case 'A':
	 				grd_a = &argv[i][2];
					break;
	 			case 'B':
	 				grd_b = &argv[i][2];
					break;
	 			case 'G':
	 				grd_c = &argv[i][2];
					break;
				default:		/* Options not recognized */
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else {
			error = TRUE;
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  Options must start with a hyphen\n", gmt_program);
		}
	}
	
	if (gmt_quick || argc == 1) {
		fprintf (stderr,"grdpaste %s - Join two grdfiles along common edge\n\n", GMT_VERSION);
		fprintf (stderr, "usage: grdpaste -A<file_a> -B<file_b> -G<outfile> [-V]\n\n");
		
		if (gmt_quick) exit (-1);
		
		fprintf (stderr, "	 where file_a and file_b are to be combined into outfile.\n");
		fprintf (stderr, "	 file_a and file_b must have same dx,dy and one edge in common.\n");
		fprintf (stderr, "	 If in doubt, run grdinfo first and check your files.\n");
		fprintf (stderr, "	 Use grdcut and/or grdsample to adjust files as necessary.\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		explain_option ('V');
		exit (-1);
	}

	if (!grd_a) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -A:  Must specify input file\n", gmt_program);
		error++;
	}
	if (!grd_b) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -B:  Must specify input file\n", gmt_program);
		error++;
	}
	if (!grd_c) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G:  Must specify output file\n", gmt_program);
		error++;
	}
	if (error) exit (-1);

	/* Check that the options selected make sense */
	

	/* Try to find a common side to join on  */

	grd_init (&hc, argc, argv, FALSE);
	if (read_grd_info (grd_a, &ha)) {
		fprintf (stderr, "grdpaste: Error opening file %s\n", grd_a);
		exit (-1);
	}
	if (read_grd_info (grd_b, &hb)) {
		fprintf (stderr, "grdpaste: Error opening file %s\n", grd_b);
		exit (-1);
	}

	if (ha.node_offset != hb.node_offset) error = TRUE;
	if ((ha.z_scale_factor != hb.z_scale_factor) || (ha.z_add_offset != hb.z_add_offset)) {
		fprintf (stderr, "grdpaste: Scale/offset not compatible!\n");
		exit (-1);
	}
		
	one_or_zero = 1 - ha.node_offset;
		
	if ( (ha.x_min == hb.x_min) && (ha.x_max == hb.x_max) ) {

		if (ha.y_max == hb.y_min) {
			way = 1;
			hc.nx = ha.nx;
			hc.ny = ha.ny + hb.ny - one_or_zero;
			hc.x_min = ha.x_min;
			hc.x_max = ha.x_max;
			hc.y_min = ha.y_min;
			hc.y_max = hb.y_max;
		}
		else if (ha.y_min == hb.y_max) {
			way = 2;
			hc.nx = ha.nx;
			hc.ny = ha.ny + hb.ny - one_or_zero;
			hc.x_min = ha.x_min;
			hc.x_max = ha.x_max;
			hc.y_min = hb.y_min;
			hc.y_max = ha.y_max;
		}
		else {
			fprintf (stderr, "grdpaste:  Grids do not share a common edge!\n");
			exit (-1);
		}
	}
	else if ( (ha.y_min == hb.y_min) && (ha.y_max == hb.y_max) ) {

		if (ha.x_min == hb.x_max) {
			way = 3;
			hc.nx = ha.nx + hb.nx - one_or_zero;
			hc.ny = ha.ny;
			hc.x_min = hb.x_min;
			hc.x_max = ha.x_max;
			hc.y_min = ha.y_min;
			hc.y_max = ha.y_max;
		}
		else if (ha.x_max == hb.x_min) {
			way = 4;
			hc.nx = ha.nx + hb.nx - one_or_zero;
			hc.ny = ha.ny;
			hc.x_min = ha.x_min;
			hc.x_max = hb.x_max;
			hc.y_min = ha.y_min;
			hc.y_max = ha.y_max;
		}
		else {
			fprintf (stderr, "grdpaste:  Grids do not share a common edge!\n");
			exit (-1);
		}
	}
	else {
		fprintf (stderr, "grdpaste:  Grids do not share a common edge!\n");
		exit (-1);
	}

	if ( (ha.x_inc == hb.x_inc) && (ha.y_inc == hb.y_inc) ) {
		hc.x_inc = ha.x_inc;
		hc.y_inc = ha.y_inc;
	}
	else {
		fprintf (stderr, "grdpaste:  Grid intervals do not match!\n");
		exit (-1);
	}

	/* Now we can do it  */


	if (gmtdefs.verbose) {
		fprintf (stderr, "File spec:\tW E S N dx dy nx ny:\n");
		fprintf (stderr, "File_a:\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%d\t%d\n",
			ha.x_min, ha.x_max, ha.y_min, ha.y_max, ha.x_inc, ha.y_inc, ha.nx, ha.ny);
		fprintf (stderr, "File_b:\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%d\t%d\n",
			hb.x_min, hb.x_max, hb.y_min, hb.y_max, hb.x_inc, hb.y_inc, hb.nx, hb.ny);
		fprintf (stderr, "File_c:\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%d\t%d\n",
			hc.x_min, hc.x_max, hc.y_min, hc.y_max, hc.x_inc, hc.y_inc, hc.nx, hc.ny);
	}
	
	c = (float *) memory (CNULL, (int)(hc.nx*hc.ny), sizeof (float), "grdpaste");
	hc.node_offset = ha.node_offset;
	hc.z_scale_factor = ha.z_scale_factor;
	hc.z_add_offset = ha.z_add_offset;
	pad[0] = pad[1] = pad[2] = pad[3] = 0;
	
	switch (way) {
		case 1:
			pad[3] = hb.ny - one_or_zero;
			if (read_grd (grd_a, &ha, c, ha.x_min, ha.x_max, ha.y_min, ha.y_max, pad, FALSE)) {
				fprintf (stderr, "grdpaste: Error reading file %s\n", grd_a);
				exit (-1);
			}
			pad[3] = 0;	pad[2] = ha.ny - one_or_zero;
			if (read_grd (grd_b, &hb, c, hb.x_min, hb.x_max, hb.y_min, hb.y_max, pad, FALSE)) {
				fprintf (stderr, "grdpaste: Error reading file %s\n", grd_b);
				exit (-1);
			}
			break;
		case 2:
			pad[2] = hb.ny - one_or_zero;
			if (read_grd (grd_a, &ha, c, ha.x_min, ha.x_max, ha.y_min, ha.y_max, pad, FALSE)) {
				fprintf (stderr, "grdpaste: Error reading file %s\n", grd_a);
				exit (-1);
			}
			pad[2] = 0;	pad[3] = ha.ny - one_or_zero;
			if (read_grd (grd_b, &hb, c, hb.x_min, hb.x_max, hb.y_min, hb.y_max, pad, FALSE)) {
				fprintf (stderr, "grdpaste: Error reading file %s\n", grd_b);
				exit (-1);
			}
			break;
		case 3:
			pad[0] = hb.nx - one_or_zero;
			if (read_grd (grd_a, &ha, c, ha.x_min, ha.x_max, ha.y_min, ha.y_max, pad, FALSE)) {
				fprintf (stderr, "grdpaste: Error reading file %s\n", grd_a);
				exit (-1);
			}
			pad[0] = 0;	pad[1] = ha.nx - one_or_zero;
			if (read_grd (grd_b, &hb, c, hb.x_min, hb.x_max, hb.y_min, hb.y_max, pad, FALSE)) {
				fprintf (stderr, "grdpaste: Error reading file %s\n", grd_b);
				exit (-1);
			}
			break;
		case 4:
			pad[1] = hb.nx - one_or_zero;
			if (read_grd (grd_a, &ha, c, ha.x_min, ha.x_max, ha.y_min, ha.y_max, pad, FALSE)) {
				fprintf (stderr, "grdpaste: Error reading file %s\n", grd_a);
				exit (-1);
			}
			pad[1] = 0;	pad[0] = ha.nx - one_or_zero;
			if (read_grd (grd_b, &hb, c, hb.x_min, hb.x_max, hb.y_min, hb.y_max, pad, FALSE)) {
				fprintf (stderr, "grdpaste: Error reading file %s\n", grd_b);
				exit (-1);
			}
			break;
	}
						
	if (write_grd (grd_c, &hc, c, 0.0, 0.0, 0.0, 0.0, pad, FALSE)) {
		fprintf (stderr, "grdpaste: Error writing file %s\n", grd_c);
		exit (-1);
	}
	
	free ( (char *) c);
	
	gmt_end (argc, argv);
}
