/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdproject.c	2.19  30 Jun 1995
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * grdproject reads a geographical grdfile and evaluates the grid at new grid positions
 * specified by the map projection and new dx/dy values using a weighted average of all
 * points within the search radius. Optionally, grdproject may perform the inverse
 * transformation, going from rectangular coordinates to geographical.
 *
 * Author:	Paul Wessel
 * Date:	6-JAN-1991-1995
 * Ver:		2.0
 *
 */

#include "gmt.h"

float *geo, *rect, *weight;

main (argc, argv)
int argc;
char **argv; {
	int i, dpi, nx, ny, dummy[4];
	
	BOOLEAN error = FALSE, inverse = FALSE, n_set = FALSE, set_n = FALSE;
	BOOLEAN d_set = FALSE, e_set = FALSE, map_center = FALSE, offset = FALSE;
	
	double w, e, s, n, x_inc = 0.0, y_inc = 0.0, max_radius = 0.0;
	double xmin, xmax, ymin, ymax;
	
	char *infile, *outfile;
	
	struct GRD_HEADER g_head, r_head;
	
	argc = gmt_begin (argc, argv);
	
	infile = outfile = CNULL;
	w = e = s = n = 0.0;
	dummy[3] = dummy[2] = dummy[1] = dummy[0] = 0;
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */
			
				case 'J':
				case 'R':
				case 'V':
				case '\0':
					error += get_common_args (argv[i], &w, &e, &s, &n);
					break;
				
				/* Supplemental parameters */
				
				case 'D':
					gmt_getinc (&argv[i][2], &x_inc, &y_inc);
					d_set = TRUE;
					break;
				case 'E':
					dpi = atoi (&argv[i][2]);
					e_set = TRUE;
					break;
				case 'F':
					offset = TRUE;
					break;
				case 'G':
					outfile = &argv[i][2];
					break;
				case 'I':
					inverse = TRUE;
					break;
				case 'M':
					map_center = TRUE;
					if (argv[i][2] == 'm') gmtdefs.measure_unit = 2;	/* Select meters */
					break;
				case 'N':
					sscanf (&argv[i][2], "%d/%d", &nx, &ny);
					if (ny == 0) ny = nx;
					n_set = TRUE;
					break;
				case 'S':
					max_radius = atof (&argv[i][2]);
					break;
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else 
			infile = argv[i];
	}
	
	if ((d_set + e_set + n_set) == 0) n_set = set_n = TRUE;

	if (argc == 1 || gmt_quick) {
		fprintf (stderr, "grdproject %s - Project geographical grid to/from rectangular grid\n\n", GMT_VERSION);
		fprintf(stderr, "usage: grdproject <in_grdfile> -J<parameters> -R<west/east/south/north>\n");
		fprintf(stderr, "	[-D<dx[m|c]>[/dy[m|c]]] [-E<dpi>] [-F] [-G<out_grdfile>] [-I] [-M]\n");
		fprintf(stderr, "	[-N<nx/ny>] [-S<radius>] [-V]\n\n");
		
		if (gmt_quick) exit (-1);
		
		fprintf(stderr, "	<in_grdfile> is data set to be transformed\n");
		explain_option ('J');
		explain_option ('R');
		fprintf(stderr, "\n\tOPTIONS:\n");
		fprintf(stderr, "	-D sets the grid spacing for the new grid\n");
		fprintf(stderr, "	-E sets dpi for output grid\n");
		fprintf(stderr, "	-F force pixel node registration  [Default is centered]\n");
		fprintf(stderr, "	-G name of output grid\n");
		fprintf(stderr, "	-I Inverse transformation from rectangular to geographical\n");
		fprintf (stderr, "	-M coordinates relative to projection center [Default is relative to lower left corner]\n");
		fprintf(stderr, "	-N sets the number of nodes for the new grid\n");
		fprintf(stderr, "	   Only one of -D, -E, and -N can be specified!\n");
		fprintf(stderr, "	   If none are specified, nx,ny of the input file is used\n");
		fprintf(stderr, "	-S sets the search radius in projected units [Default avoids aliasing]\n");
		explain_option ('V');
		
		exit (-1);
	}
	
	if (!infile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify input file\n", gmt_program);
		error++;
	}
	if (!outfile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G option:  Must specify output file\n", gmt_program);
		error++;
	}
	if (!project_info.region_supplied) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify -R option\n", gmt_program);
		error++;
	}
	if (max_radius < 0.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S:  Max search radius, if specified, must be positive\n", gmt_program);
		error++;
	}
	if ((d_set + e_set + n_set) != 1) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify only one of -D, -E, or -N\n", gmt_program);
		error++;
	}
	if (d_set && (x_inc <= 0.0 || y_inc <= 0.0)) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -D option.  Must specify positive increment(s)\n", gmt_program);
		error++;
	}
	if (n_set && !set_n && (nx <= 0 || ny <= 0)) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -N option.  Must specify positive integers\n", gmt_program);
		error++;
	}
	if (e_set && dpi <= 0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -E option.  Must specify positive dpi\n", gmt_program);
		error++;
	}
	
	if (error) exit (-1);

	map_setup (w, e, s, n);
	
	xmin = (map_center) ? project_info.xmin - project_info.x0 : project_info.xmin;
	xmax = (map_center) ? project_info.xmax - project_info.x0 : project_info.xmax;
	ymin = (map_center) ? project_info.ymin - project_info.y0 : project_info.ymin;
	ymax = (map_center) ? project_info.ymax - project_info.y0 : project_info.ymax;
	
	grd_init (&r_head, argc, argv, FALSE);
	grd_init (&g_head, argc, argv, FALSE);

	if (inverse) {	/* Transforming from rectangular projection to geographical */
	
		if (!project_info.region) d_swap (s, e);  /* Got w/s/e/n, make into w/e/s/n */

		g_head.x_min = w;	g_head.x_max = e;	g_head.y_min = s;	g_head.y_max = n;
	
		if (read_grd_info (infile, &r_head)) {
			fprintf (stderr, "grdproject: Error opening file %s\n", infile);
			exit (-1);
		}
		rect = (float *) memory (CNULL, (int)(r_head.nx * r_head.ny), sizeof (float), "grdproject");
		if (read_grd (infile, &r_head, rect, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
			fprintf (stderr, "grdproject: Error reading file %s\n", infile);
			exit (-1);
		}
	
		if (set_n) {
			nx = r_head.nx;
			ny = r_head.ny;
		}
		grdproject_init (&g_head, x_inc, y_inc, nx, ny, dpi, offset);
		geo = (float *) memory (CNULL, (int)(g_head.nx * g_head.ny), sizeof (float), "grdproject");
		if (gmtdefs.verbose)
			fprintf (stderr, "grdproject:  Transform (%lg/%lg/%lg/%lg) <-- (%lg/%lg/%lg/%lg)\n",
				g_head.x_min, g_head.x_max, g_head.y_min, g_head.y_max, xmin, xmax, ymin, ymax);

		grd_inverse (geo, &g_head, rect, &r_head, max_radius, map_center);
			
		if (write_grd (outfile, &g_head, geo, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
			fprintf (stderr, "grdproject: Error writing file %s\n", outfile);
			exit (-1);
		}
		
	}
	else {	/* Forward projection from geographical to rectangular grid */
	
		if (read_grd_info (infile, &g_head)) {
			fprintf (stderr, "grdproject: Error opening file %s\n", infile);
			exit (-1);
		}

		if (project_info.projection == MOLLWEIDE && (fabs (g_head.x_min - g_head.x_max) < 360.0 || (g_head.y_max - g_head.y_min) < 180.0)) {
			fprintf (stderr, "grdproject: Selected projection works on global datasets only\n");
			exit (-1);
		}
	
		geo = (float *) memory (CNULL, (int)(g_head.nx * g_head.ny), sizeof (float), "grdproject");
		if (read_grd (infile, &g_head, geo, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
			fprintf (stderr, "grdproject: Error reading file %s\n", infile);
			exit (-1);
		}
	
		r_head.x_min = xmin;	r_head.x_max = xmax;
		r_head.y_min = ymin;	r_head.y_max = ymax;
		if (set_n) {
			nx = g_head.nx;
			ny = g_head.ny;
		}
		
		if (gmtdefs.verbose)
			fprintf (stderr, "grdproject:  Transform (%lg/%lg/%lg/%lg) --> (%lg/%lg/%lg/%lg)\n",
				g_head.x_min, g_head.x_max, g_head.y_min, g_head.y_max, xmin, xmax, ymin, ymax);

		grdproject_init (&r_head, x_inc, y_inc, nx, ny, dpi, offset);
		rect = (float *) memory (CNULL, (int)(r_head.nx * r_head.ny), sizeof (float), "grdproject");
		grd_forward (geo, &g_head, rect, &r_head, max_radius, map_center);
		
		if (write_grd (outfile, &r_head, rect, 0.0, 0.0, 0.0, 0.0, dummy, FALSE)) {
			fprintf (stderr, "grdproject: Error writing file %s\n", outfile);
			exit (-1);
		}
		
	}
	
	free ((char *)geo);
	free ((char *)rect);
	
	gmt_end (argc, argv);
}
