/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)grdtrack.c	2.19  3/13/95
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * grdtrack reads a xyfile, opens the 2d binary gridded grdfile, 
 * and samples the dataset at the xy positions with a bilinear or bicubic
 * interpolant.  This new data is added to the input as an extra column
 * and printed to standard output.  In order to evaluate derivatives along
 * the edges of the grdfile region, we assume natural bicubic spline
 * boundary conditions (d2z/dn2 = 0, n being the normal to the edge;
 * d2z/dxdy = 0 in the corners).  Rectangles of size x_inc by y_inc are 
 * mapped to [0,1] x [0,1] by affine transformation, and the interpolation
 * done on the normalized rectangle.
 *
 * Author:	Walter H F Smith
 * Date:	23-SEP-1993
 * 
 * Based on the original grdtrack, which had this authorship/date/history:
 *
 * Author:	Paul Wessel
 * Date:	29-JUN-1988
 * Revised:	5-JAN-1990	PW: Updated to v.2.0
 *		4-AUG-1993	PW: Added -Q
 */

#include "gmt.h"
#include "gmt_bcr.h"

float *f;

main (argc, argv)
int argc;
char **argv; {
	int i, ix, iy, mx, my, n_read, n_points = 0, pad[4];
	
	BOOLEAN error = FALSE, multi_segments = FALSE, bilinear = FALSE, periodic = FALSE;
	
	double value, xy[2];
	
	char *grdfile, stuff[80], line[512], format1[512], format2[512], EOL_flag = '>';
	
	FILE *fp = NULL;
	
	struct GRD_HEADER grd;
	
	grdfile = CNULL;
	
	argc = gmt_begin (argc, argv);

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
			
				/* Common parameters */
			
				case 'H':
				case 'V':
				case ':':
				case '\0':
					error += get_common_args (argv[i], 0, 0, 0, 0);
					break;

				/* Supplemental parameters */
				
				case 'G':
					grdfile = &argv[i][2];
					break;
				case 'L':
					periodic = TRUE;
					break;
				case 'M':
					multi_segments = TRUE;
					if (argv[i][2]) EOL_flag = argv[i][2];
					break;
				case 'Q':
					bilinear = TRUE;
					break;

				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else if ((fp = fopen (argv[i], "r")) == NULL) {
			fprintf (stderr, "grdtrack: Cannot open file %s\n", argv[i]);
			exit (-1);
		}
	}
	
	if (argc == 1 || gmt_quick) {
		fprintf (stderr,"grdtrack %s - Sampling of a 2-D gridded netCDF grdfile along 1-D trackline\n\n", GMT_VERSION);
		fprintf (stderr, "usage: grdtrack <xyfile> -G<grdfile> [-H] [-L] [-M[<flag>]] [-Q] [-V] [-:]\n");
		
		if (gmt_quick) exit(-1);
		
		fprintf (stderr, "	<xyfile> is an multicolumn ASCII file with (lon,lat) in the first two columns\n");
		fprintf (stderr, "	-G <grdfile> is the name of the 2-D binary data set\n");
		fprintf (stderr, "\n\tOPTIONS:\n");
		explain_option ('H');
		fprintf (stderr, "	-L means that x is longitude\n");
		fprintf (stderr, "	-M Input file consist of multiple segments separated by one record\n");
		fprintf (stderr, "	   whose first character is <flag> [>].\n");
		fprintf (stderr, "	-Q Quick mode, use bilinear rather than bicubic interpolation\n");
		explain_option ('V');
		explain_option (':');
		explain_option ('.');
		exit(-1);
	}
	
	if (!grdfile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -G:  Must specify output file\n", gmt_program);
		error++;
	}
	
	if (error) exit (-1);

	if (fp == NULL) {
		fp = stdin;
		fprintf (stderr, "grdtrack: Reads from standard input\n");
	}

	if (read_grd_info (grdfile, &grd)) {
		fprintf (stderr, "grdtrack: Error opening file %s\n", grdfile);
		exit (-1);
	}
	
	mx = grd.nx + 2;
	my = grd.ny + 2;
	
	f = (float *) memory (CNULL, mx * my, sizeof (float), "grdtrack");

	pad[0] = pad[1] = pad[2] = pad[3] = 1;
	if (read_grd (grdfile, &grd, f, grd.x_min, grd.x_max, grd.y_min, grd.y_max, pad, FALSE)) {
		fprintf (stderr, "grdtrack: Error reading file %s\n", grdfile);
		exit (-1);
	}

	/* Initialize bcr structure:  */

	bcr_init (&grd, pad, bilinear);

	/* Set boundary conditions  */
	
	set_bcr_boundaries (&grd, f);
	
	sprintf (format1, "%s\t%s\t%%s\t%s\n\0", gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format);
	sprintf (format2, "%s\t%s\t%s\n\0", gmtdefs.d_format, gmtdefs.d_format, gmtdefs.d_format);
	
	if (gmtdefs.io_header) {	/* First echo headers, if any */
		for (i = 0; i < gmtdefs.n_header_recs - 1; i++) {
			fgets (line, 512, fp);
			printf ("%s", line);
		}
		fgets (line, 512, fp);
		line[strlen(line)-1] = 0;
		printf ("%s\tsample\n", line);
	}

	ix = (gmtdefs.xy_toggle) ? 1 : 0;	iy = 1 - ix;		/* Set up which columns have x and y */
	while (fgets (line, 512, fp)) {

		if (multi_segments && line[0] == EOL_flag) {
			printf ("%s", line);
			continue;
		}

		n_read = sscanf (line, "%lf %lf %[^\n]", &xy[ix], &xy[iy], stuff);

		/* Check that we are inside the grd area */
		
		if (xy[1] < grd.y_min || xy[1] > grd.y_max) continue;
		if (periodic) {
			xy[0] -= 360.0;
			while (xy[0] < grd.x_min) xy[0] += 360.0;
			if (xy[0] > grd.x_max) continue;
		}
		else if (xy[0] < grd.x_min || xy[0] > grd.x_max)
			continue;
		
		value = get_bcr_z(&grd, xy[0], xy[1], f);

		if (n_read == 2)
			printf (format2, xy[0], xy[1], value);
		else
			printf (format1, xy[0], xy[1], stuff, value);
		n_points++;
	}
	fclose(fp);
	
	if (gmtdefs.verbose) fprintf (stderr, "grdtrack: Sampled %d points from grid %s (%d x %d)\n",
		n_points, grdfile, grd.nx, grd.ny);
	
	free ((char *)f);
	
	gmt_end (argc, argv);
}
