/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)makecpt.c	2.12  08 Jun 1995
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * makecpt -C<cont_int_or_file> -S<v1_color_table> [-M<mid_val>]
 *
 * Read a v1.0 color table format, and a contour interval or file
 * of contours, [and a mid-value for contour interval], and produce
 * a v2.0 cpt file.
 * Designed to help people convert from GMTv1 to GMTv2.0 syntax.
 *
 * Author:	Walter H. F. Smith
 * Date:	20 June 1991-1995
 */

#include "gmt.h"
#include <time.h>

struct	CTABLE {
	int	r;
	int	g;
	int	b;
};


main(argc,argv)
int	argc;
char	**argv;
{

	int	i, j, ic, is, ntable, nconts, nshades = -1, nfact, bg, fg;
	BOOLEAN	error = FALSE, makehues = FALSE, continuous = FALSE;
	double	cint = -1.0, midval = 0.0, start_hue, delta_hue, hue, sat, val, *contour, start_cont;
	double	h1, s1, v1, h2, s2, v2;
	FILE	*fpc = NULL, *fps = NULL;
	struct	CTABLE	*ctable;
	time_t	clock;
	char	buffer[120];

	argc = gmt_begin(argc, argv);
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch(argv[i][1]) {
				case 'V':
				case '\0':
					error += get_common_args(argv[i], 0, 0, 0, 0);
					break;
				case 'C':
					ic = i;
					fpc = fopen(&argv[i][2], "r");
					if (fpc == NULL) cint = atof(&argv[i][2]);
					break;
				case 'M':
					midval = atof(&argv[i][2]);
					break;
				case 'S':
					is = i;
					fps = fopen(&argv[i][2], "r");
					if (fps == NULL) {
						j = 2;
						while(argv[i][j]) j++;
						if (argv[i][j-1] == 'c' || argv[i][j-1] == 'C') makehues = TRUE;
						nshades = atoi(&argv[i][2]);
					}
					break;
				case 'Z':
					continuous = TRUE;
					break;
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else {
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  Options must start with a hyphen\n", gmt_program);
			error = TRUE;
		}
	}

	if (argc == 1 || gmt_quick) {
		fprintf(stderr,"makecpt %s - Make GMT color palette tables\n\n", GMT_VERSION);
		fprintf(stderr,"usage:  makecpt -C<cint_or_file> -S<nshades_or_file>[c] [-M<midval>] [-Z]\n\n");
		if (gmt_quick) exit(-1);
		fprintf(stderr,"\t-C If <cint_or_file> can be opened as a file, it is read to get contour intervals;\n");
		fprintf(stderr,"\t   else, <cint_or_file> should be a positive number, the contour interval desired.\n");
		fprintf(stderr,"\t-S If <nshades_or_file> can be opened as a file, it is read as a V1-style color table;\n");
		fprintf(stderr,"\t   else, <nshades_or_file> should be the # of colors/grayshades desired.  In this case,.\n");
		fprintf(stderr,"\t   you may append c after the number to make a color table or default to a gray table.\n");
		fprintf(stderr,"\t-M If you gave a number for -C, you may set the contour value at the middle of the range\n");
		fprintf(stderr,"\t   to <midval>.  [Default is zero; and ignored if -C gives a file.]\n");
		fprintf(stderr,"\t-Z will create a continuous color palette.  Ignored if c_int and n_shades are files\n");
		fprintf(stderr,"\t   [Default is discontinuous, i.e., color is constant for each interval]\n");
		exit(-1);
	}

	if (fpc == NULL && cint <= 0.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -C option:  Must specify contour interval or file\n", gmt_program);
		error++;
	}

	if (fps == NULL && nshades <= 0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S option:  Must specify number of shades or file\n", gmt_program);
		error++;
	}

	if (error) exit (-1);

	/* First get color table.  Read it or make it.  */

	if (fps || fpc) continuous = FALSE;

	if (fps) {	/* Read user's old table  */
		if ( (fgets (buffer, 120, fps) ) == NULL) {
			fprintf(stderr, "makecpt:  Cannot read line 1 of v1 color table file.\n");
			exit(-1);
		}
		nshades = atoi(buffer);
		if (nshades <= 0) {
			fprintf(stderr, "makecpt:  Cannot read 1st line of v1 color table file.\n");
			exit(-1);
		}
		ntable = nshades + 3;
		ctable = (struct CTABLE *) memory (CNULL, ntable, sizeof(struct CTABLE), "makecpt");

		for (i = 0; i < nshades; i++) {
			if ( (fgets (buffer, 120, fps) ) == NULL) {
				fprintf(stderr, "makecpt:  Cannot read line %d of v1 color table file.\n", i+2);
				exit(-1);
			}
			if ((sscanf(buffer, "%d %d %d", &ctable[i].r, &ctable[i].g, &ctable[i].b)) != 3) {
				 fprintf(stderr, "makecpt:  Cannot read line %d of v1 color table file.\n", i+2);
				exit(-1);
			}
		}
		/* Try to read FG and BG colors, if any  */

		bg = i;
		if ( (fgets (buffer, 120, fps) ) != NULL) {
			if ((sscanf(buffer, "%d %d %d", &ctable[i].r, &ctable[i].g, &ctable[i].b)) != 3) {
				 fprintf(stderr, "makecpt:  Cannot read line %d of v1 color table file.\n", i+2);
				exit(-1);
			}
			i++;
		}
		else {
			ctable[i].r = ctable[i].g = ctable[i].b = 0;	/* Default to black  */
			i++;
		}


		fg = i;
		if ( (fgets (buffer, 120, fps) ) != NULL) {
			if ((sscanf(buffer, "%d %d %d", &ctable[i].r, &ctable[i].g, &ctable[i].b)) != 3) {
				 fprintf(stderr, "makecpt:  Cannot read line %d of v1 color table file.\n", i+2);
				exit(-1);
			}
			i++;
		}
		else {
			ctable[i].r = ctable[i].g = ctable[i].b = 255;	/* Default to white  */
		}
		/* Get here when finished reading existing v1 table  */
		fclose(fps);
	}
	else {
		/* Get here if table has to be created.  */
		is = 0;
		ntable = nshades + 3;
		ctable = (struct CTABLE *) memory (CNULL, ntable, sizeof(struct CTABLE), "makecpt");

		if (makehues) {
			/* Decide arbitrarily that first entry will be purple 255 0 255 or H = 0.83333
				and decrease H to last entry where H = 0 (= 1) is red 255 0 0:  */
			nfact = (continuous) ? nshades : nshades - 1;
			start_hue = 5.0/6.0;
			delta_hue = start_hue / nfact;	/* 0.83/nfact  */
			if (gmtdefs.color_model) {
				sat = gmtdefs.hsv_min_saturation;
				val = gmtdefs.hsv_max_value;
			}
			else {
				sat = val = 1.0;
			}
			for (i = 0; i < nshades + continuous; i++) {
				hue = start_hue - i * delta_hue;
				gmt_hsv_to_rgb (&ctable[i].r, &ctable[i].g, &ctable[i].b, 360.0 * hue, sat, val);
			}
		}
		else {
			/* Divide gray scale so first and last entry are not quite black/white;
				thus use nshades, start_hue, delta_hue, hue, sat, val, rr, gg, bb+1 intervals of gray, with 1/2 interval start.  */
			nfact = (continuous) ? nshades + 2 : nshades + 1;
			delta_hue = 1.0 / nfact;
			start_hue = 0.5 * delta_hue;
			for (i = 0; i < nshades + continuous; i++) {
				hue = start_hue + i * delta_hue;
				ctable[i].r = ctable[i].g = ctable[i].b = floor (255.999*hue);
			}
		}
		bg = (continuous) ? nshades+1 : nshades;
		fg = bg + 1;
		ctable[bg].r = ctable[bg].g = ctable[bg].b = 0;	/* Set BG to black  */
		ctable[fg].r = ctable[fg].g = ctable[fg].b = 255;	/* Set FG to white  */
	}

	/* Now we have a color table set up with a FG and BG and also nshades entries.
		Need to read nshades+1 contour boundaries, or create them with cint and midval:  */

	nconts = nshades+1;
	contour = (double *) memory (CNULL, nconts, sizeof(double), "makecpt");

	if (fpc) {
		for (i = 0; i < nconts; i++) {
			if ( (fgets (buffer, 120, fpc) ) == NULL) {
				fprintf(stderr, "makecpt:  Cannot read line %d of v1 contour file.\n", i+2);
				exit(-1);
			}
			if ((sscanf(buffer, "%lf", &contour[i])) != 3) {
				 fprintf(stderr, "makecpt:  Cannot read line %d of v1 contour file.\n", i+2);
				exit(-1);
			}
		}
		fclose(fpc);
	}
	else {
		ic = 0;
		start_cont = midval - cint * ( (int)(nshades/2) );
		for (i = 0; i < nconts; i++) {
			contour[i] = start_cont + i * cint;
		}
	}


	/* Write to stdout.  */

	clock = time ((time_t *)NULL);
	
	printf("#\tcpt file created by makecpt on %s", ctime(&clock));
	if (is) {
		printf("#\tInput V1.0 shade table was %s\n", &argv[is][2]);
	}
	else if (makehues) {
		printf("#\tNo input V1.0 shade table was given; an iso-hue rainbow was made.\n");
	}
	else {
		printf("#\tNo input V1.0 shade table was given; a gray scale was made.\n");
	}
	if (ic) {
		printf("#\tContours were read from %s\n", &argv[ic][2]);
	}
	else {
		printf("#\tContours were made using a mid-value of %.8lg and a contour interval of %.8lg\n", midval, cint);
	}
	if (continuous) printf("#\tColor palette is continous\n");

	printf("#\n");
	if (continuous) {
		if (gmtdefs.color_model) {
			/* Conversion.  This is a new feature:  */
			for (i = 0; i < nshades; i++) {
				gmt_rgb_to_hsv(ctable[i].r, ctable[i].g, ctable[i].b, &h1, &s1, &v1);
				gmt_rgb_to_hsv(ctable[i+1].r, ctable[i+1].g, ctable[i+1].b, &h2, &s2, &v2);
				printf("%.8lg\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%.8lg\t%.8lg\n", contour[i], h1, s1, v1,
					contour[i+1], h2, s2, v2);
			}
		}
		else {
			for (i = 0; i < nshades; i++) {
				printf("%.8lg\t%d\t%d\t%d\t%.8lg\t%d\t%d\t%d\n", contour[i], ctable[i].r, ctable[i].g, ctable[i].b,
					contour[i+1], ctable[i+1].r, ctable[i+1].g, ctable[i+1].b);
			}
		}
	}
	else {
		for (i = 0; i < nshades; i++) {
			printf("%.8lg\t%d\t%d\t%d\t%.8lg\t%d\t%d\t%d\n", contour[i], ctable[i].r, ctable[i].g, ctable[i].b,
				contour[i+1], ctable[i].r, ctable[i].g, ctable[i].b);
		}
	}
	if (gmtdefs.color_model) {
		gmt_rgb_to_hsv(ctable[bg].r, ctable[bg].g, ctable[bg].b, &h1, &s1, &v1);
		gmt_rgb_to_hsv(ctable[fg].r, ctable[fg].g, ctable[fg].b, &h2, &s2, &v2);
		printf("B\t%.8lg\t%.8lg\t%.8lg\n", h1, s1, v1);
		printf("F\t%.8lg\t%.8lg\t%.8lg\n", h2, s2, v2);
	}
	else {
		printf("B\t%d\t%d\t%d\n", ctable[bg].r, ctable[bg].g, ctable[bg].b);
		printf("F\t%d\t%d\t%d\n", ctable[fg].r, ctable[fg].g, ctable[fg].b);
	}

	free((char *)ctable);
	free((char *)contour);
	exit(0);
}

