/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)nearneighbor.c	2.28  18 Jun 1995
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * Based on a specified grid size, nearneighbor reads an xyz file and
 * determines the nearest points to each node in sectors.  The default
 * looks for the nearest point for each quadrant.  The points must also
 * be within a maximum search-radius from the node.  For the nodes that
 * have a full set of nearest neighbors, a weighted average value is
 * computed.
 *
 * Author:	Paul Wessel
 * Date:	7-MAY-1991-1995
 * Version:	2.0
 */
 
#include "gmt.h"

float *grd;

struct NODE {
	float *distance;
	int *datum;
} **grid_node, *add_new_node();

struct POINT {
	float x, y, z, w;
} *point;

main (argc, argv)
int argc;
char **argv; {
	int i, j, k, ij, i0, j0, di, dj, n_sectors = 4, sector, n, n_alloc = 5 * GMT_CHUNK, n_fields;
	int ix, iy, n_set, n_almost, n_none, n_files = 0, n_args, fno, one_or_zero, n_in, pad[4];

	BOOLEAN go, error = FALSE, map_units = FALSE, done = FALSE, first = TRUE, nofile = TRUE;
	BOOLEAN set_empty = FALSE, binary = FALSE, weights = FALSE, more, skip, single_precision = FALSE;

	double radius = 0.0, weight, weight_sum, x0, y0, dx, dy, delta, distance, factor, empty = 0.0;
	double *in, shrink, km_pr_deg, x_left, x_right, y_top, y_bottom, offset, xinc2, yinc2, idx, idy;

	char *outfile = 0, line[512];

	FILE *fp = NULL;

	struct GRD_HEADER header;
	
	argc = gmt_begin (argc, argv);
	
	grd_init (&header, argc, argv, FALSE);

	pad[0] = pad[1] = pad[2] = pad[3] = 0;
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
				/* Common parameters */
			
				case 'R':
				case 'V':
				case ':':
				case '\0':
					error += get_common_args (argv[i], &header.x_min, &header.x_max, &header.y_min, &header.y_max);
					break;
				
				/* Supplemental parameters */
				
				case 'b':	/* Input triplets [quadruplets] are binary, not ascii */
					single_precision = !(argv[i][strlen(argv[i])-1] == 'd');
					binary = TRUE;
					break;
				case 'I':
					gmt_getinc (&argv[i][2], &header.x_inc, &header.y_inc);
					break;
				case 'E':
					if (!argv[i][2]) {
						fprintf (stderr, "%s: GMT SYNTAX ERROR -E option:  Must specify value or NaN\n", gmt_program);
						error++;
					}
					else
						empty = (!strcmp (&argv[i][2], "NaN")) ? gmt_NaN : atof (&argv[i][2]);	
					set_empty = TRUE;
					break;
				case 'F':
					header.node_offset = TRUE;
					break;
				case 'G':
					outfile = &argv[i][2];
					break;
				case 'M':
					map_units = TRUE;
					break;
				case 'N':
					n_sectors = atoi (&argv[i][2]);
					break;
				case 'S':
					gmt_getinc (&argv[i][2], &radius, &radius);
					break;
				case 'W':
					weights = TRUE;
					break;
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else 
			n_files++;
	}
	
	if (argc == 1 || gmt_quick) {
		fprintf (stderr, "nearneighbor %s - A \"Nearest neighbor\" gridding algorithm\n\n", GMT_VERSION);
		fprintf(stderr, "usage: nearneighbor [xyzfile(s)] -G<out_grdfile> -I<dx>[m|c][/<dy>[m|c]]\n");
		fprintf(stderr, "	-N<sectors> -R<west/east/south/north> -S<radius>[m|c] [-E<empty>] [-F] [-H ] [-M ] [-V ] [-W] [-:] [-b[d]]\n\n");
		if (gmt_quick) exit (-1);
		fprintf(stderr, "	-G name of output grid\n");
		fprintf(stderr, "	-I sets the grid spacing for the grid.  Append m for minutes, c for seconds\n");
		fprintf(stderr, "	-N sets number of sectors. Default is quadrant search [4]\n");
		explain_option ('R');
		fprintf(stderr, "	-S sets the search radius.  Append m for minutes, c for seconds\n");
		fprintf(stderr, "\n\tOPTIONS:\n");
		fprintf(stderr, "	-E value to use for empty nodes [Default is NaN]\n");
		fprintf(stderr, "	-F Force pixel registration [Default is gridline registration]\n");
		explain_option ('H');
		fprintf(stderr, "	-M map units.  Input data is in degrees and radius is in km\n");
		explain_option ('V');
		fprintf(stderr, "	-W input file has observation weights in 4th column\n");
		explain_option (':');
		fprintf (stderr, "\t-b means input triplets are binary (double precision).  Default is ascii.\n");
		fprintf (stderr, "\t   Append d for double precision [Default is single precision].\n");		explain_option ('.');
		explain_option ('.');
		
		exit (-1);
	}

	if (!project_info.region_supplied) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify -R option\n", gmt_program);
		error++;
	}
	if (!outfile) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR option -G:  Must specify output file\n", gmt_program);
		error++;
	}
	if (n_sectors <= 0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -N option:  Must specify a positive number of sectors\n", gmt_program);
		error++;
	}
	if (radius <= 0.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -S option:  Must specify a positive search radius\n", gmt_program);
		error++;
	}
	if (header.x_inc <= 0.0 || header.y_inc <= 0.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -I option.  Must specify positive increment(s)\n", gmt_program);
		error++;
	}	
	if (binary && gmtdefs.io_header) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR.  Binary input data cannot have header -H\n", gmt_program);
		error++;
	}
	if (error) exit (-1);

	if (binary) gmt_input = (single_precision) ? bin_float_input : bin_double_input;
	
	n_in = (weights) ? 4 : 3;
	
        ix = (gmtdefs.xy_toggle);       iy = 1 - ix;
        
        if (n_files > 0)
        	nofile = FALSE;
        else
        	n_files = 1;
        n_args = (argc > 1) ? argc : 2;
        
	if (header.node_offset) {
		one_or_zero = 0;
		offset = 0.0;
		xinc2 = 0.5 * header.x_inc;
		yinc2 = 0.5 * header.y_inc;
	}
	else {
		one_or_zero = 1;
		offset = 0.5;
		xinc2 = yinc2 = 0.0;
	}
	
	idx = 1.0 / header.x_inc;
	idy = 1.0 / header.y_inc;
	header.nx = rint ( (header.x_max - header.x_min) * idx) + one_or_zero;
	header.ny = rint ( (header.y_max - header.y_min) * idy) + one_or_zero;
	
	if (gmtdefs.verbose) fprintf (stderr, "nearneighbor:  Grid dimensions are nx = %d, ny = %d\n",
		header.nx, header.ny);

	grid_node = (struct NODE **) memory (CNULL, (int)(header.nx * header.ny), sizeof (struct NODE *), "nearneighbor");
	point = (struct POINT *) memory (CNULL, n_alloc, sizeof (struct POINT), "nearneighbor");
	
	if (map_units) {
		km_pr_deg = 0.001 * 2.0 * M_PI * gmtdefs.ellipse[gmtdefs.ellipsoid].eq_radius / 360.0;
		shrink = cos (0.5 * (header.y_min + header.y_max) * D2R);
		di = ceil (radius / (km_pr_deg * header.x_inc * shrink));
		dj = ceil (radius / (km_pr_deg * header.y_inc));
	}
	else {
		di = ceil (radius * idx);
		dj = ceil (radius * idy);
	}
	factor = n_sectors / (2.0 * M_PI);

	x_left = header.x_min - radius;	x_right = header.x_max + radius;
	y_top = header.y_max + radius;	y_bottom = header.y_min - radius;
	
	n = 0;
	
	for (fno = 1; !done && fno < n_args; fno++) {	/* Loop over input files, if any */
		if (!nofile && argv[fno][0] == '-') continue;
		
		if (nofile) {	/* Just read standard input */
			fp = stdin;
			done = TRUE;
		}
		else if ((fp = fopen (argv[fno], "r")) == NULL) {
			fprintf (stderr, "nearneighbor: Cannot open file %s\n", argv[fno]);
			continue;
		}
		
		if (!nofile && gmtdefs.verbose) fprintf (stderr, "nearneighbor: Working on file %s\n", argv[fno]);
		
		if (gmtdefs.io_header) {
			for (i = 0; i < gmtdefs.n_header_recs; i++) {
				fgets (line, 512, fp);
				if (first) printf ("%s", line);
			}
			first = FALSE;
		}

		more = ((n_fields = gmt_input (fp,  n_in, &in)) == n_in);
			
		while (more) {
	
			skip = FALSE;

			if (in[ix] < x_left || in[ix] > x_right) skip = TRUE;
			if (in[iy] < y_bottom || in[iy] > y_top) skip = TRUE;
		
			if (!skip) {
				point[n].x = in[ix];
				point[n].y = in[iy];
				point[n].z = in[2];
				if (weights) point[n].w = in[3];
		
				i0 = floor (((in[ix] - header.x_min) * idx) + offset);
				j0 = floor (((header.y_max - in[iy]) * idy) + offset);
		
				for (j = j0 - dj; j <= (j0 + dj); j++) {
					if (j < 0 || j >= header.ny) continue;
					for (i = i0 - di; i <= (i0 + di); i++) {
						if (i < 0 || i >= header.nx) continue;
						k = j * header.nx + i;
						x0 = header.x_min + i * header.x_inc + xinc2;
						y0 = header.y_max - j * header.y_inc - yinc2;
						dx = in[ix] - x0;	dy = in[iy] - y0;
						distance = (map_units) ? km_pr_deg * hypot (dx * shrink, dy) : hypot (dx, dy);
						if (distance > radius) continue;
						sector = (d_atan2 (dy, dx) + M_PI) * factor;
						if (sector == n_sectors) sector = 0;
						if (!grid_node[k]) grid_node[k] = add_new_node (n_sectors);
						if (grid_node[k]->datum[sector] == -1 || grid_node[k]->distance[sector] > distance) {
							grid_node[k]->distance[sector] = distance;
							grid_node[k]->datum[sector] = n;
						}
					}
				}
				n++;
				if (n == n_alloc) {
					n_alloc += GMT_CHUNK;
					point = (struct POINT *) memory ((char *)point, n_alloc, sizeof (struct POINT), "nearneighbor");
				}
			}
			more = ((n_fields = gmt_input (fp,  n_in, &in)) == n_in);
		}
		if (fp != stdin) fclose (fp);
		if (n_fields == -1) n_fields = 0;	/* Ok to get EOF */
		if (n_fields%n_in) {	/* Got garbage or multiple segment subheader */
			fprintf (stderr, "%s:  Cannot read X Y Z [W] at line %d, aborts\n", gmt_program, n);
			exit (-1);
		}
	}
	
	point = (struct POINT *) memory ((char *)point, n, sizeof (struct POINT), "nearneighbor");
	grd = (float *) memory (CNULL, (int)(header.nx * header.ny), sizeof (float), "nearneighbor");

	/* Compute weighted averages based on the nearest neighbors */
	
	n_set = n_almost = n_none = 0;

	if (!set_empty) empty = gmt_NaN;
	
	for (j = ij = 0; j < header.ny; j++) {
		for (i = 0; i < header.nx; i++, ij++) {
			grd[ij] = empty;

			if (!grid_node[ij]) {	/* No nearest neighbors */
				n_none++;
				continue;
			}

			for (k = 0, go = TRUE; go && k < n_sectors; k++) if (grid_node[ij]->datum[k] < 0) go = FALSE;
			if (!go) { 	/* Not full set of neighbors */
				n_almost++;
				continue;
			}

			n_set++;
			weight_sum = grd[ij] = 0.0;	/* Replace the empty so that we may compute a sum */
			for (k = 0; k < n_sectors; k++) {
				delta = 3.0 * grid_node[ij]->distance[k] / radius;
				weight = 1.0 / (1.0 + delta * delta);	/* This is distance weight */
				if (weights) weight *= point[grid_node[ij]->datum[k]].w;	/* This is observation weight */
				grd[ij] += weight * point[grid_node[ij]->datum[k]].z;
				weight_sum += weight;
			}
			grd[ij] /= weight_sum;
		}
	}
	
	if (write_grd (outfile, &header, grd, 0.0, 0.0, 0.0, 0.0, pad, FALSE)) {
		fprintf (stderr, "nearneighbor: Error writing file %s\n", outfile);
		exit (-1);
	}
	
	if (gmtdefs.verbose) {
		fprintf (stderr, "nearneighbor: %d nodes assigned, %d nodes partially filled, %d nodes set to ",
			n_set, n_almost, n_none);
		(bad_float (empty)) ? fprintf (stderr, "NaN\n") : fprintf (stderr, "%lg\n", empty);
	}

	free ((char *)grd);
	free ((char *)point);
	free ((char *)grid_node);
	
	gmt_end (argc, argv);
}

struct NODE *add_new_node(n)
int n; {
	struct NODE *new;
	
	new = (struct NODE *) memory (CNULL, 1, sizeof (struct NODE), "nearneighbor");
	new->distance = (float *) memory (CNULL, n, sizeof (float), "nearneighbor");
	new->datum = (int *) memory (CNULL, n, sizeof (int), "nearneighbor");
	while (n > 0) new->datum[--n] = -1;
	
	return (new);
}
