/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)psclip.c	2.20  3/13/95
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * psclip reads one or many xy-files and draws polygons just like psxy
 * with the exception that these polygons are set up to act as clip
 * paths for subsequent plotting.  psclip uses the even-odd winding
 * rule to decide what's inside and outside the path.
 *
 * Author:	Paul Wessel
 * Date:	6-FEB-1991-1995
 * Version:	2.0
 * Limitation:	Total clip path cannot exceed PostScript max path length
 * (~1350 on Laserwriters, memory dependent on SUNs).
 *
 */

#include "gmt.h"

double *xx, *yy;	/* Arrays holding the contour xy values */

main (argc, argv)
int argc;
char **argv; {
	int i, n, fno, ix, iy, n_alloc, n_args, n_files = 0;
	
	BOOLEAN error = FALSE, nofile = TRUE, end_of_clip = FALSE, multi_segments = FALSE;
	BOOLEAN invert = FALSE, done, first;
	
	char line[512], EOL_flag = '>', *more;
	
	double west, east, north, south, x0, y0, xy[2], new_z_level = 0.0;
	
	FILE *fp = NULL;
	
	west = east = north = south = 0.0;

	argc = gmt_begin (argc, argv);

	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch (argv[i][1]) {
		
				/* Common parameters */
			
				case 'B':
				case 'H':
				case 'J':
				case 'K':
				case 'O':
				case 'P':
				case 'R':
				case 'U':
				case 'V':
				case 'X':
				case 'x':
				case 'Y':
				case 'y':
				case 'c':
				case ':':
				case '\0':
					error += get_common_args (argv[i], &west, &east, &south, &north);
					break;
				
				/* Supplemental parameters */
			
				case 'E':
					sscanf (&argv[i][2], "%lf/%lf",&z_project.view_azimuth, &z_project.view_elevation);
					break;
				case 'F':
					if (gmt_getrgb (&argv[i][2], &gmtdefs.basemap_frame_rgb[0], &gmtdefs.basemap_frame_rgb[1], &gmtdefs.basemap_frame_rgb[2])) {
						gmt_pen_syntax ('F');
						error++;
					}
					break;
				case 'N':
					invert = TRUE;
					break;
				case 'M':               /* Multiple line segments */
					multi_segments = TRUE;
					if (argv[i][2]) EOL_flag = argv[i][2];
					break;
				case 'S':
					end_of_clip = TRUE;
					break;
				case 'Z':
					new_z_level = atof (&argv[i][2]);
					break;
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
		else
			n_files++;
	}
	
	if (argc == 1 || gmt_quick) {
		fprintf (stderr,"psclip %s - To set up polygonal clip paths\n\n", GMT_VERSION);
		fprintf (stderr, "usage: psclip -S  OR\n");
		fprintf (stderr, "       psclip <xy-files> -J<params> -R<west/east/south/north>\n");
		fprintf (stderr, "	[-B<tickinfo>] [-Eaz/el] [-F<r/g/b] [-H] [-K] [-M<flag>] [-N] [-O] [-P]\n");
		fprintf (stderr, "	[-U[label] [-V] [-X<x_shift>] [-Y<y_shift>] [-c<ncopies>] [-:]\n\n");
		
		if (gmt_quick) exit(-1);
		
		fprintf (stderr, "	<xy-files> is one or more polygon files.  If none, standard input is read\n");
		explain_option ('j');
		explain_option ('R');
		fprintf (stderr, "\n\tOPTIONS:\n");
		explain_option ('b');
		fprintf (stderr, "      -E set azimuth and elevation of viewpoint for 3-D perspecive [180/90]\n");
		fprintf (stderr, "      -F Set color used for Frame and anotation [%d/%d/%d]\n",
			gmtdefs.basemap_frame_rgb[0], gmtdefs.basemap_frame_rgb[1], gmtdefs.basemap_frame_rgb[2]);
		explain_option ('H');
		explain_option ('K');
		fprintf (stderr, "      -M Input files each consist of multiple segments separated by one record\n");
		fprintf (stderr, "         whose first character is <flag> [>]\n");
		fprintf (stderr, "	-N uses the outside of the polygons as clip paths\n");
		explain_option ('O');
		explain_option ('P');
		fprintf (stderr, "	-S means stop existing clip-path.  No other options required\n");
		explain_option ('U');
		explain_option ('V');
		explain_option ('X');
		explain_option ('c');
		explain_option (':');
		explain_option ('.');
		exit(-1);
	}
	
	if (!end_of_clip) {
		if (!project_info.region_supplied) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR:  Must specify -R option\n", gmt_program);
			error++;
		}
		if (z_project.view_azimuth > 360.0 || z_project.view_elevation <= 0.0 || z_project.view_elevation > 90.0) {
			fprintf (stderr, "%s: GMT SYNTAX ERROR -E option:  Enter azimuth in 0-360 range, elevation in 0-90 range\n", gmt_program);
			error++;
		}
	}
	
	if (error) exit (-1);

	if (end_of_clip) gmtdefs.overlay = TRUE;	/* Force overlay mode */
	
	if (!project_info.x_off_supplied && gmtdefs.overlay) gmtdefs.x_origin = 0.0;
	if (!project_info.y_off_supplied && gmtdefs.overlay) gmtdefs.y_origin = 0.0;

	ps_plotinit (CNULL, gmtdefs.overlay, gmtdefs.page_orientation, gmtdefs.x_origin, gmtdefs.y_origin,
		gmtdefs.global_x_scale, gmtdefs.global_y_scale, gmtdefs.n_copies,
		gmtdefs.dpi, gmtdefs.measure_unit, gmtdefs.paper_width, gmtdefs.page_rgb, gmt_epsinfo (argv[0]));
	echo_command (argc, argv);
	if (gmtdefs.unix_time) timestamp (argc, argv);

	if (end_of_clip && !frame_info.plot) {	/* Just undo previous clip-path, no basemap needed */
		ps_clipoff ();
		ps_plotend (gmtdefs.last_page);
	
		if (gmtdefs.verbose) fprintf (stderr, "psclip: Done!\n");
	
		gmt_end (argc, argv);
	}

	map_setup (west, east, south, north);
		
	if (project_info.three_D) ps_transrotate (-z_project.xmin, -z_project.ymin, 0.0);

	if (new_z_level != 0.0) project_info.z_level = new_z_level;

	ix = (gmtdefs.xy_toggle);	iy = 1 - ix;

	if (!end_of_clip) {	/* Start new clip_path */
	
		if (frame_info.plot) {
			ps_setpaint (gmtdefs.basemap_frame_rgb[0], gmtdefs.basemap_frame_rgb[1], gmtdefs.basemap_frame_rgb[2]);
			map_basemap ();
		}
		
		xx = (double *) memory (CNULL, GMT_CHUNK, sizeof (double), "psclip");
		yy = (double *) memory (CNULL, GMT_CHUNK, sizeof (double), "psclip");
		n_alloc = GMT_CHUNK;
		
		if (invert) map_clip_on (-1, -1, -1, 1);	/* Must clip map */
		first = (invert) ? FALSE : TRUE;
		done = FALSE;
		n_args = (argc > 1) ? argc : 2;
		if (n_files > 0) nofile = FALSE;
		
		for (fno = 1; !done && fno < n_args; fno++) {	/* Loop over all input files */
			if (!nofile && argv[fno][0] == '-') continue;
			if (nofile) {
				fp = stdin;
				done = TRUE;
			}
			else if ((fp = fopen (argv[fno], "r")) == NULL) {
				fprintf (stderr, "psclip: Cannot open file %s\n", argv[fno]);
				continue;
			}

			if (!nofile && gmtdefs.verbose) {
				fprintf (stderr, "psclip: Working on file %s\n", argv[fno]);
				sprintf (line, "File: %s\0", argv[fno]);
				ps_comment (line);
			}
			if (gmtdefs.io_header) for (i = 0; i < gmtdefs.n_header_recs; i++) fgets (line, 512, fp);
			if (multi_segments) {
				fgets (line, 512, fp);
				if (gmtdefs.verbose) ps_comment (line);
			}
			more = fgets (line, 512, fp);
			n = 0;
			while (more) {
				while ((more && !multi_segments) || (more && multi_segments && line[0] != EOL_flag)) {
					sscanf (line, "%lf %lf", &xy[ix], &xy[iy]);
					xx[n] = xy[0];	yy[n] = xy[1];
					n++;
					if (n == n_alloc) {
						n_alloc += GMT_CHUNK;
						xx = (double *) memory ((char *)xx, n_alloc, sizeof (double), "psclip");
						yy = (double *) memory ((char *)yy, n_alloc, sizeof (double), "psclip");
					}
					more = fgets (line, 512, fp);
				}
		
				xx = (double *) memory ((char *)xx, n, sizeof (double), "psclip");
				yy = (double *) memory ((char *)yy, n, sizeof (double), "psclip");
				n_alloc = n;
				
				for (i = 0; i < n; i++) {
					geo_to_xy (xx[i], yy[i], &x0, &y0);
					xx[i] = x0; yy[i] = y0;
				}
				
				if (project_info.three_D) two_d_to_three_d (xx, yy, n);
				
				ps_clipon (xx, yy, n, -1, -1, -1, first);
				first = FALSE;
				
				if (more) more = fgets (line, 512, fp);
			}
		}
		ps_clipon (xx, yy, 0, -1, -1, -1, 2);
		if (fp != stdin) fclose (fp);
		free ((char *)xx);
		free ((char *)yy);
	}

	else {	/* Undo previous clip-path and draw basemap */
		ps_clipoff ();
		ps_setpaint (gmtdefs.basemap_frame_rgb[0], gmtdefs.basemap_frame_rgb[1], gmtdefs.basemap_frame_rgb[2]);
		map_basemap ();
	}
	
	if (project_info.three_D) ps_rotatetrans (z_project.xmin, z_project.ymin, 0.0);

	ps_plotend (gmtdefs.last_page);
	
	if (gmtdefs.verbose) fprintf (stderr, "psclip: Done!\n");
	
	gmt_end (argc, argv);
}
