/*--------------------------------------------------------------------
 *    The GMT-system:	@(#)psscale.c	2.27  27 Jul 1995
 *
 *    Copyright (c) 1991-1995 by P. Wessel and W. H. F. Smith
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 * psscale draws a grayscale or colorscale either vertically or
 * horizontally.  psscale will interpolate colors if the lower
 * and upper rgb values for an interval are different.  The
 * resolution for this interpolation can be set on the command line.
 * If the scale is to be used with illumninated 2-D and 3-D plots
 * then options for intensities are available.
 *
 * Author:	Paul Wessel
 * Created:	10-MAY-1991
 * Modified:	1-FEB-1995
 *
 */

#include "gmt.h"

int bit_dpi = 50;

main(argc, argv)
int argc;
char *argv[]; {
	int i;
	
	BOOLEAN horizontal = FALSE, equi = FALSE, intens = FALSE, error = FALSE;
	
	double xpos, ypos, length, width, max_intens = 1.0;
	double *z_width;
	
	char flag, *cpt_file = NULL, *z_file = NULL, line[512];
	
	FILE *fp = NULL;
	
	argc = gmt_begin (argc, argv);

	length = width = xpos = ypos = 0.0;
	
	/* Check and interpret the command line arguments */
	
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch(argv[i][1]) {
		
				/* Common parameters */

				case 'B':
				case 'K':
				case 'O':
				case 'P':
				case 'U':
				case 'V':
				case 'X':
				case 'x':
				case 'Y':
				case 'y':
				case 'c':
				case '\0':
					error += get_common_args (argv[i], 0, 0, 0, 0);
					break;
				
				/* Supplemental parameters */
			
				case 'C':
					cpt_file = &argv[i][2];
					break;
				case 'D':
					sscanf (&argv[i][2], "%lf/%lf/%lf/%lf", &xpos, &ypos, &length, &width);
					flag = argv[i][strlen(argv[i])-1];
					horizontal = (flag == 'h' || flag == 'H');
					break;
				case 'F':
					if (gmt_getrgb (&argv[i][2], &gmtdefs.basemap_frame_rgb[0], &gmtdefs.basemap_frame_rgb[1], &gmtdefs.basemap_frame_rgb[2])) {
						gmt_pen_syntax ('F');
						error++;
					}
					break;
				case 'I':
					intens = TRUE;
					if (argv[i][2]) max_intens = atof (&argv[i][2]);
					break;
				case 'N':
					if (argv[i][2]) bit_dpi = atoi (&argv[i][2]);
					break;
				case 'L':
					equi = TRUE;
					break;
				case 'Z':
					z_file = &argv[i][2];
					break;
					
				case '0':
					gmtdefs.color_image = 0;
					break;
				case '1':
					gmtdefs.color_image = 1;
					break;
				case '2':
					gmtdefs.color_image = 2;
					break;

				/* Options not recognized */
						
				default:
					error = TRUE;
					gmt_default_error (argv[i][1]);
					break;
			}
		}
	}
	
	if (argc == 1 || gmt_quick) {
		fprintf (stderr,"psscale %s - To create grayscale or colorscale for maps\n\n", GMT_VERSION);
		fprintf(stderr,"usage: psscale -C<cpt_file>  -D<xpos/ypos/length/width[h]> [-B<tickinfo>] [-F<r>/<g>/<b>] [-I[<intens>]\n");
		fprintf(stderr,"	[-K] [-L] [-N<dpi>] [-O] [-P] [-U] [-V] [-X<x_shift>] [-Y<y_shift>] [-Zzfile] [-0 -1 -2] [-c<ncopies>]\n\n");
		if (gmt_quick) exit(-1);
		
		fprintf (stderr,"	-B Set header label as :.\"label\":\n");
		fprintf (stderr,"	-C Color palette file\n");
		fprintf (stderr,"	   By default all color changes are anotated.  To use a subset,\n");
		fprintf (stderr,"	   add an extra column to the cpt-file with a L, U, or B\n");
		fprintf (stderr,"	   to anotate Lower, Upper, or Both color segment boundaries\n");
		fprintf (stderr,"	-D set mid-point position and length/width for scale.\n");
		fprintf (stderr,"	   Append h for horizontal scale\n");
		fprintf (stderr, "	-F Set color used for Frame and anotation [%d/%d/%d]\n",
			gmtdefs.basemap_frame_rgb[0], gmtdefs.basemap_frame_rgb[1], gmtdefs.basemap_frame_rgb[2]);
		fprintf (stderr,"	-I add illumination for +- <max_intens> [1.0]\n");
		explain_option ('K');
		fprintf (stderr,"	-L For equal-sized color rectangles\n");
		fprintf (stderr,"	-N effective dots-per-inch for color scale [50]\n");
		explain_option ('O');
		explain_option ('P');
		explain_option ('U');
		explain_option ('V');
		explain_option ('X');
		fprintf (stderr,"	-Z give colorbar-width (in %s) per color entry\n", gmt_unit_names[gmtdefs.measure_unit]);
		fprintf (stderr,"	   By default, width of entry is scaled to color range\n");
		fprintf (stderr,"	   i.e., z = 0-100 gives twice the width as z = 100-150.\n");
		fprintf (stderr,"	-0 plot color scale using colorimage.\n");
		fprintf (stderr,"	-1 plot color scale using tiles.\n");
		fprintf (stderr,"	-2 plot color scale using psto24.\n");
		fprintf (stderr,"	   [Default = %d]\n", gmtdefs.color_image);
		explain_option ('c');
		explain_option ('.');
		exit(-1);
	}

	/* Check that the options selected are mutually consistant */
	
	if (!cpt_file) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -C option:  Must specify color palette table\n", gmt_program);
		error++;
	}
	if (z_project.view_azimuth > 360.0 || z_project.view_elevation <= 0.0 || z_project.view_elevation > 90.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -E option:  Enter azimuth in 0-360 range, elevation in 0-90 range\n", gmt_program);
		error++;
	}
	if (length <= 0.0 ) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -D option: scale length must be nonzero\n", gmt_program);
		error++;
	}
	if (width <= 0.0) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -D option: scale width must be nonzero\n", gmt_program);
		error++;
	}
	if (bit_dpi < 1) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR -N option: dpi must be nonzero\n", gmt_program);
		error++;
	}
	      
	if (error) exit (-1);

	read_cpt (cpt_file);
	z_width = (double *) memory (CNULL, gmt_n_colors, sizeof (double), "psscale");
	
	if (z_file && (fp = fopen (z_file, "r")) == NULL) {
		fprintf (stderr, "psscale: Unable to open file %s\n", z_file);
		exit (-1);
	}
	else if (z_file) {
		for (i = 0; i < gmt_n_colors; i++) {
			fgets (line, 512, fp);
			if (line[0] == '#') continue;
			sscanf (line, "%lf", &z_width[i]);
		}
		fclose (fp);
	}
	else if (equi) {
		for (i = 0; i < gmt_n_colors; i++) z_width[i] = length / gmt_n_colors;
	}
	else {
		for (i = 0; i < gmt_n_colors; i++) z_width[i] = length * (gmt_lut[i].z_high - gmt_lut[i].z_low) / (gmt_lut[gmt_n_colors-1].z_high - gmt_lut[0].z_low);
	}
		
	if (!project_info.x_off_supplied && gmtdefs.overlay) gmtdefs.x_origin = 0.0;	/* Since map_setup is not called here */
	if (!project_info.y_off_supplied && gmtdefs.overlay) gmtdefs.y_origin = 0.0;

	ps_plotinit (CNULL, gmtdefs.overlay, gmtdefs.page_orientation, gmtdefs.x_origin, gmtdefs.y_origin,
		gmtdefs.global_x_scale, gmtdefs.global_y_scale, gmtdefs.n_copies,
		gmtdefs.dpi, gmtdefs.measure_unit, gmtdefs.paper_width, gmtdefs.page_rgb, gmt_epsinfo (argv[0]));
		
	echo_command (argc, argv);
	if (gmtdefs.unix_time) timestamp (argc, argv);
	
	ps_setpaint (gmtdefs.basemap_frame_rgb[0], gmtdefs.basemap_frame_rgb[1], gmtdefs.basemap_frame_rgb[2]);
	
	draw_colorbar (xpos, ypos, length, width, z_width, horizontal, intens, max_intens);

	ps_plotend (gmtdefs.last_page);
	
	gmt_end (argc, argv);
}

int draw_colorbar (xpos, ypos, length, width, z_width, horizontal, intens, max_intens)
double xpos, ypos;
double length, width, z_width[], max_intens;
BOOLEAN horizontal, intens; {
	int i, j, ndec = -1, dec, r, g, b, rr, gg, bb;
	int nx, ny, nm, nm2, barmem, k, all = TRUE;
	char format[50], text[50], test[50];
	unsigned char *bar;
	double off, anot_off, label_off, len, len2, size, x0, x1;
	double z, xleft, inc_i, inc_j, get_z();
	
	ps_setfont (gmtdefs.anot_font);

	/* Find max decimals needed */
	
	for (i = 0; i < gmt_n_colors; i++) {
		dec = get_format (gmt_lut[i].z_low, text);
		if (dec > ndec) {
			strcpy (format, text);
			ndec = dec;
		}
		if (gmt_lut[i].anot) all = FALSE;
	}
	dec = get_format (gmt_lut[gmt_n_colors-1].z_high, text);
	if (dec > ndec) {
		strcpy (format, text);
		ndec = dec;
	}
	
	len = (gmtdefs.tick_length > 0.0) ? gmtdefs.tick_length : 0.0;
	len2 = 0.5 * len;
	xleft = x0 = -0.5 * length;
	
	/* Make bitimage for colorbar using bit_dpi */
	
	nx = length * bit_dpi;
	ny = width * bit_dpi;
	nm = nx * ny;
	nm2 = 2 * nm;
	inc_i = 1.0 / bit_dpi;
	inc_j = (intens) ? 2.0 * max_intens / (ny - 1) : 0.0;
	barmem = (gmt_gray) ? nm : 3 * nm;
	bar = (unsigned char *) memory (CNULL, barmem, sizeof (char), "psscale");
	
	/* Load bar image */
	
	for (i = 0; i < nx; i++) {
		z = get_z (i * inc_i, z_width, gmt_n_colors);
		get_rgb24 (z, &rr, &gg, &bb);
		for (j = 0; j < ny; j++) {
			r = rr;	g = gg;	b = bb;
			k = j * nx + i;
			if (intens) illuminate (max_intens - j * inc_j, &r, &g, &b);
			if (gmt_gray)	/* Convert to gray using the YIQ transformation */
				bar[k] = (unsigned char) rint (0.299 * r + 0.587 * g + 0.114 * b);
			else if (gmtdefs.color_image == 2) {	/* RGB separation */
				bar[k] = r;
				bar[k+nm] = g;
				bar[k+nm2] = b;
			}
			else {
				k *= 3;
				bar[k++] = r;
				bar[k++] = g;
				bar[k++] = b;
			}
		}
	}
	
	ps_setline (gmtdefs.frame_pen);
	if (horizontal) {
		ps_transrotate (xpos, ypos, 0.0);
		if (gmt_gray)
			ps_image (x0, -width, length, width, bar, nx, ny, 8);
		else 
			color_image (x0, -width, length, width, bar, nx, ny);
			
		anot_off = -(width + len + gmtdefs.anot_offset);
		label_off = -(width + len + 2.5 * gmtdefs.anot_offset + (gmtdefs.anot_font_size / gmt_ppu[gmtdefs.measure_unit]) * font_height[gmtdefs.anot_font]);
		
		ps_plot (xleft, 0.0, 3);	ps_plotr (length, 0.0, 2);
		ps_plot (xleft, -width, 3);	ps_plotr (length, 0.0, 2);
		x1 = xleft;
		for (i = 0; i < gmt_n_colors; i++) {
			ps_plot (x1, 0.0, 3);
			if (all || (gmt_lut[i].anot & 1)) {
				ps_plotr (0.0, -width - len, 2);
				sprintf (text, format, gmt_lut[i].z_low);
				ps_text (x1, anot_off, gmtdefs.anot_font_size, text, 0.0, 10, 0);
			}
			else
				ps_plotr (0.0, -width - len2, 2);
			x1 += z_width[i];
		}
		ps_plot (-xleft, 0.0, 3);
		if (all || (gmt_lut[gmt_n_colors-1].anot & 2)) {
			ps_plotr (0.0, -width - len, 2);
			sprintf (text, format, gmt_lut[gmt_n_colors-1].z_high);
			ps_text (-xleft, anot_off, gmtdefs.anot_font_size, text, 0.0, 10, 0);
		}
		else
			ps_plotr (0.0, -width - len2, 2);

		ps_setfont (gmtdefs.header_font);

		if (frame_info.label[3][0]) ps_text (0.0, label_off, gmtdefs.header_font_size, frame_info.label[3], 0.0, 10, 0);
		if (frame_info.label[0][0]) ps_text (xleft+length+gmtdefs.anot_offset, -0.5*width, gmtdefs.label_font_size, frame_info.label[0], 0.0, 5, 0);
		ps_rotatetrans (-xpos, -ypos, 0.0);
	}
	else {	/* Vertical scale */
		ps_transrotate (xpos, ypos, 90.0);
		if (gmt_gray)
			ps_image (x0, -width, length, width, bar, nx, ny, 8);
		else 
			color_image (x0, -width, length, width, bar, nx, ny);
		sprintf (text, "%d\0", (int)fabs (floor (gmt_lut[0].z_low)));
		sprintf (test, "%d\0", (int)fabs (ceil (gmt_lut[gmt_n_colors-1].z_high)));
		off = ((MAX (strlen (text), strlen (test)) + ndec) * 0.55 + ((ndec > 0) ? 0.3 : 0.0) + ((gmt_lut[0].z_low < 0.0) ? 0.3 : 0.0))
			* gmtdefs.anot_font_size / gmt_ppu[gmtdefs.measure_unit];
		anot_off = -(width + len + gmtdefs.anot_offset + off);
		label_off = anot_off - 0.7 * gmtdefs.anot_font_size / gmt_ppu[gmtdefs.measure_unit] - 1.5 * gmtdefs.anot_offset;
		ps_plot (xleft, 0.0, 3);	ps_plotr (length, 0.0, 2);
		ps_plot (xleft, -width, 3);	ps_plotr (length, 0.0, 2);
		x1 = xleft;
		for (i = 0; i < gmt_n_colors; i++) {
			ps_plot (x1, 0.0, 3);
			if (all || (gmt_lut[i].anot & 1)) {
				ps_plotr (0.0, -width - len, 2);
				sprintf (text, format, gmt_lut[i].z_low);
				ps_text (x1, anot_off, gmtdefs.anot_font_size, text, -90.0, 7, 0);
			}
			else
				ps_plotr (0.0, -width - len2, 2);
			x1 += z_width[i];
		}
		ps_plot (-xleft, 0.0, 3);
		if (all || (gmt_lut[gmt_n_colors-1].anot & 2)) {
			ps_plotr (0.0, -width - len, 2);
			sprintf (text, format, gmt_lut[gmt_n_colors-1].z_high);
			ps_text (-xleft, anot_off, gmtdefs.anot_font_size, text, -90.0, 7, 0);
		}
		else
			ps_plotr (0.0, -width - len2, 2);
			
		ps_setfont (gmtdefs.header_font);
		size = 0.9 * gmtdefs.header_font_size / gmt_ppu[gmtdefs.measure_unit];
		x0 = 0.5 * (strlen (frame_info.label[3]) -1) * size ;
		text[1] = 0;
		for (i = 0; i < strlen (frame_info.label[3]); i++) {
			x1 = x0 - i * size;
			text[0] = frame_info.label[3][i];
			ps_text (x1, label_off, gmtdefs.header_font_size, text, -90.0, 6, 0);
		}
		if (frame_info.label[0][0]) ps_text (-xleft + 2.0 * len, -0.5 * width, gmtdefs.label_font_size, frame_info.label[0], -90.0, 2, 0);
		ps_rotatetrans (-xpos, -ypos, -90.0);

	}
}

double get_z (x, width, n)
double x, width[];
int n; {
	int i = 0;
	double tmp;
	
	tmp = width[0];
	while (i < n && x > tmp) tmp += width[++i];
	if (i == n) return (gmt_lut[gmt_n_colors-1].z_high);
	return (gmt_lut[i].z_low + (x - tmp + width[i]) * (gmt_lut[i].z_high - gmt_lut[i].z_low) / width[i]);
}
