/**/

   /*=================================================================================================*/
   /*  Anton Monroe <amonroe5301@gmail.com>                                                           */
   /*  August 2025                                                                                    */
   /*                                                                                                 */
   /*  RexxMerge.cmd copies procedures from one Rexx file to another, replacing the                   */
   /*  original versions. See RexxMerge.cmd -help                                                     */
   /*                                                                                                 */
   /*  I wrote it because I have a collection of commonly-used routines. For example,                 */
   /*  the ones in this file that start with "0". When I improve or fix a bug in one                  */
   /*  of them, I wanted an easy way to import the new version into any program that                  */
   /*  uses that routine. RexxMerge.cmd replaces the body of the routine with the                     */
   /*  newer version. The "procedure expose" statement from the old and new versions                  */
   /*  are merged, because I sometimes customize it for a particular script.                          */
   /*                                                                                                 */
   /*  This is for OS/2 Rexx; it should work under Regina/OS2 and Regina/Linux but                    */
   /*  I have not tested it much.                                                                     */
   /*                                                                                                 */
   /*=================================================================================================*/
   /*  fix: when updating procedures, if the old label is on two lines, like                          */
   /*              WordsQ:                                                                            */
   /*              procedure expose (Shared_vars)                                                     */
   /*        the replacement combines the lines into                                                  */
   /*              WordsQ: procedure expose (Shared_vars)                                             */
   /*        Worse, it combines the lines even if no update is needed                                 */
   /* ----------------------------------------------------------------------------------------------- */

   Main: /*fold00*/
   /*=================================================================================================*/
   /*=================================================================================================*/
   parse arg MainArgs
   signal on Syntax
   signal on NoValue

   call Init
   Old. = ''
   call FileToArray 'Old.', G.0InfileN
   select
      when translate(G.0Command) == 'REPLACE' then do
            call Task_Replace
         end
      when translate(G.0Command) == 'IMPORT' then do
            call Task_Import
         end
      otherwise
         call Msg G.0meID 'I do not understand the command' G.0Command
         call Msg G.0meID_ind 'use Import or Replace'
         call Egress 'usage'
   end   /*  select  */
   call Egress 'okay'
   /* === End of Main =============================================================================== */

   Init: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   G. = ''
   parse source G.0OpSystem . .
   parse upper version G.0Interpreter .
   if abbrev(G.0Interpreter, 'REXX-REGINA') then
      G.0Interpreter = 'REGINA'
   select
      when G.0Interpreter == 'REXXSAA' then
         if RxFuncQuery('SysLoadFuncs') then do
               call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
               call SysLoadFuncs
            end
      when G.0Interpreter == 'REGINA' then do
            call RxFuncAdd 'SysLoadFuncs', 'regutil', 'SysLoadFuncs'
            call SysLoadFuncs
         end
      otherwise
         call Msg G.0meID 'unknown Rexx interpreter "'G.0Interpreter'"'
         call Msg G.0meID_ind 'I expected REXXSAA or REGINA'
         call Egress 'Usage'
   end

   call Init_Global

   Opt. = ''
   Opt.0progress = 1
   Opt.0debug = 0
   Opt.0strict = 0
   ValidSwitch = 'path from keep where debug progress strict'
   interpret GetOpts(MainArgs, ValidSwitch)

   if Opt.0errors > 0 then
      /*  the message was already displayed when GetOpts() was interpreted                            */
      call Egress 'Usage'
   if Opt.0help \= '' then do
         call Help
         call Egress 'Usage'
      end
   if translate(ArgV.1) == 'HELP' then do
         call Help
         call Egress 'Usage'
      end

   G.0Command = ArgV.1
   G.0InfileN = ArgV.2
   G.0OutfileN = ArgV.3
   if G.0OutfileN = '' then G.0OutfileN = 'STDOUT'
   if G.0InfileN == '' then do
         call Msg G.0meID 'I need a file name'
         call Msg G.0meID_ind 'see' G.0me '-help'
         call Egress 'Usage'
      end   /*  if InfileN == ''  */

   if G.0InfileN \= '' ,
    & \FileExist(G.0InfileN) then do
         call Msg G.0meID 'Cannot find source file "'||G.0InfileN||'"'
         call Egress 'USAGE'
      end

   /*  if outfile exists, do not overwrite it, add a suffix                                           */
   if FileExist(G.0OutfileN) then do
         suff = 2
         do while FileExist(G.0OutfileN||'_'||suff)
            suff = suff +1
         end
         NewN = G.0OutfileN||'_'||suff
         call Msg G.0meID 'file exists:' G.0OutfileN
         call Msg G.0meID_ind 'changed to: ' NewN
         G.0OutfileN = NewN
      end

   return 0
   /* === End of Init =============================================================================== */

   Init_Global: /*fold00*/
   /* =============================================================================================== */
   /*  define constants and shared variables                                                          */
   /* =============================================================================================== */
   Shared_vars = 'sigl Opt. G.'
   line_vars = 'Old. New. Replace. tmp.'

   /* ----------------------------------------------------------------------------------------------- */
   if G.0OpSystem == 'UNIX' then do
         G.0CRLF = '0A'x
         G.0DirSep = '/'
         G.0PathSep = ':'
      end
   else do
         G.0CRLF = '0D0A'x
         G.0DirSep = '\'
         G.0PathSep = ';'
      end
   /*  never use the literal comment strings in a Rexx file                                           */
   /*  also, the '||' is apparently not legal in a parse template                                     */
   G.0CommentBegin = '/'||'*'
   G.0CommentEnd = '*'||'/'
   G.0footerID = G.0CommentBegin||' === End of '
   parse value SysTextScreenSize() with G.0ScreenRows G.0ScreenCols
   /* ----------------------------------------------------------------------------------------------- */
   /*  These can be used for showing messages. 'G.0me' is the name of this file. (When                */
   /*  a Rexx script is invoked from the command line, the name is in ALL CAPS, which                 */
   /*  is hard to read.  To get the correct capitalization of it, the only way I have                 */
   /*  found is a cumbersome call to SysFileTree(), and even that only corrects the                   */
   /*  filename, not the path.)                                                                       */
   /* ----------------------------------------------------------------------------------------------- */
   parse source . . G.0me
   if SysFileTree(G.0me, 'tmp.', 'FO') = 0 then
      G.0me_long = tmp.1
   else
      call ShowError 'SysFileTree() returned 'ret
   drop tmp.

   G.0me = filespec('name', G.0me_long)
   G.0meID = '['G.0me']'
   G.0me_ind = copies(' ', length(G.0me))
   G.0meID_ind = copies(' ', length(G.0meID))
   /* ----------------------------------------------------------------------------------------------- */
   return 0
   /* === End of Init_Global ======================================================================== */

   Task_Import: procedure expose (Shared_vars) (line_vars) /*fold00*/
   /* =============================================================================================== */
   /*  imports the external procedures/files named in Opt.0where                                      */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*     RexxMerge.cmd is called with "-where=Fee"                                                   */
   /*     Fee.cmd contains two procedures                                                             */
   /*           Fee:                                                                                  */
   /*              call Fie                                                                           */
   /*              return                                                                             */
   /*           Fie: procedure                                                                        */
   /*              return                                                                             */
   /*     then both procedures will be imported, and the label of Fee will be changed to              */
   /*           Fee: procedure                                                                        */
   /* =============================================================================================== */

   if Opt.0where == '' then do
         call Msg 'There is nothing to do'
         call Msg 'use -where=<procedure list> to specify which external procedures/files to import'
         call Egress 'usage'
      end
   /*  I sometimes forget the path is supposed to use ; or : as a separator, so fix it here           */
   Opt.0where = translate(Opt.0where, ' ', ',')
   Opt.0path = translate(Opt.0path, ' ', ',')
   Opt.0path = translate(Opt.0path, G.0PathSep, ' ')
   tmp. = ''
   tmp.0 = 0

   if G.0Interpreter == 'REGINA' then do
         /*  see regina.pdf about how Regina looks for external commands                              */
         /*  note that value() requires 'SYSTEM' to be uppercase                                      */
         /*  suffix 'rxc' is for a pre-compiled Rexx macro. RexxMerge.cmd will report that            */
         /*  the file was found, but is not Rexx                                                      */
         suffixlist = value('REGINA_SUFFIXES', , 'SYSTEM')
         if suffixlist \= '' then
            suffixlist = '<none>' suffixlist 'rxc rexx rex cmd rx'
         else
            suffixlist = '<none> rxc rexx rex cmd rx'
         /*  Regina suffix list may have spaces or commas, with or without dots                       */
         suffixlist = translate(suffixlist, ' ', ',')
         suffixlist = ChangeString('.', suffixlist, '')
      end
   else
      suffixlist = '<none> cmd'

   LabelList = LabelList()
   Imported = ''
   do w = 1 to words(Opt.0where)
      /*  get the name of a procedure, say "Foo", then find a file called "Foo.cmd"                   */
      procN = word(Opt.0where, w)
      /*  procN may have a path, extract basename                                                     */
      BaseN = FileBaseName(procN)
      if wordpos(BaseN, LabelList) > 0 then do
            call Msg G.0meID 'procedure' BaseN 'already exists'
            iterate w
         end
      if wordpos(BaseN, Imported) > 0 then do
            /*  a duplicate could happen in the case of                                               */
            /*        call Path\to\Foo                                                                */
            /*        call Path\to\Foo.cmd                                                            */
            /*  we don't want to import procedure Foo again                                           */
            iterate w
         end

      do e = 1 to words(suffixlist)
         ext = word(suffixlist, e)
         if ext == '<none>' then
            ext = ''
         else
            ext = '.'||ext
         procCmd = procN||ext

         /*  if user specified a path, use it. Otherwise search in the OS2 %PATH                      */
         /*  (or Unix $PATH)                                                                          */
         foundCmd = ''
         if Opt.0path \= '' then do
               foundCmd = SearchPath(Opt.0path, procCmd)
               if foundCmd \= '' then leave e
            end
         else do
               /*  SysSearchPath OS2 doesn't handle full path and in Regina seems to be broken        */
               if G.0Interpreter == 'REGINA' then
                  foundCmd = SearchPath(OS2Var('REGINA_MACROS'), procCmd)
               if foundCmd == '' then
                  /*  include the current directory before PATH and after REGINA_MACROS               */
                  foundCmd = SearchPath('.'||G.0PathSep||OS2Var('PATH'), procCmd)
               if foundCmd \= '' then leave e
            end
      end e

      /*  foundCmd == 'Path\to\Foo.cmd'                                                               */
      if foundCmd == '' then do
            call Msg G.0meID 'cannot find' procN'{'delword(ChangeString(' ', suffixlist, ', .'), 1, 1)'}'
            iterate w
         end
      /*  verify that foundCmd is really a Rexx file                                                  */
      RexxID = charin(foundCmd, 1, 2)
      call stream foundCmd, 'C', 'CLOSE'
      if RexxID \== G.0CommentBegin then do
            call Msg G.0meID 'found' foundCmd 'but it is not a Rexx file'
            iterate w
         end

      /*  copy foundCmd into a temporary array, then find the start of procedure Foo                  */
      /*  so anything before the procedure will be ignored                                            */
      call FileToArray 'tmp.', foundCmd
      start = 0
      incomment = 0
      do l = 1 to tmp.0
         /*  Incomment() tracks the start and end comments to determine if we are outside             */
         /*  any comments. It's slow, but the only way to be completely sure that a line              */
         /*  is really a label                                                                        */
         if pos('/'||'*', tmp.l) > 0 ,
          | pos('*'||'/', tmp.l) > 0 then
            incomment = Incomment(tmp.l, incomment)
         if incomment <= 0 then do
               if HasLabel(tmp.l) then do
                     rest = strip(tmp.l)
                     parse var rest label ':' rest
                     if translate(BaseN) == translate(label) then do
                           start = l
                           if rest == '' then
                              /*  change "Foo:" to "Foo: procedure"                                   */
                              tmp.l = tmp.l 'procedure'
                           leave l
                        end
                  end
            end   /*  if incomment <= 0  */
      end l

      if start == 0 then do
            call Msg G.0meID 'could not find a label "'BaseN'" in' foundCmd
            iterate w
         end
      /*  copy lines onto the end of the Old. array                                                   */
      ln = Old.0
      if Old.ln \= '' then do
            ln = ln +1
            Old.ln = ''
            Old.0 = ln
         end
      do l = start to tmp.0
         ln = ln +1
         Old.ln = tmp.l
         Old.0 = ln
      end l
      Imported = Imported BaseN
   end w   /*  for word in Opt.0where  */
   call ArrayToFile 'Old.', G.0OutfileN
   /*  quick fix to report what procedures were imported, for the benefit of standalone.btm           */
   if G.0OutfileN \= 'STDOUT' then
      if Imported \= '' then
         say G.0meID Imported
   return 0
   /* === End of Task_Import ======================================================================== */

   Task_Replace: procedure expose (Shared_vars) (line_vars) Repl. /*fold00*/
   /*=================================================================================================*/
   /*  Build a new array by copying from the Old and Replace arrays-- mostly from Old,                */
   /*  but if a procedure in Old also occurs in Replace, use the version in Replace                   */
   /*  instead.                                                                                       */
   /*  It does not check to see if each replacement version is actually different from                */
   /*  the original; it compares the Old and New arrays at the end to see if anything changed.        */
   /*=================================================================================================*/
   Opt.0where = translate(Opt.0where, ' ', ',')
   Opt.0from = translate(Opt.0from, ' ', ',')
   tmp. = ''
   G.0ReplaceFileN.0 = 0
   do i = 1 to words(Opt.0from)
      Fspec = word(Opt.0from, i)
      call SysFileTree Fspec, 'tmp.', 'FO'
      do j = 1 to tmp.0
         m = G.0ReplaceFileN.0 +1
         G.0ReplaceFileN.m = tmp.j
         G.0ReplaceFileN.0 = G.0ReplaceFileN.0 +1
      end j
   end i
   drop tmp
   if G.0ReplaceFileN.0 == 0 then do
         call Msg 'no files match "'Opt.0from'"'
         call Egress 'usage'
      end

   /*  put all the replacement files into one array                                                   */
   Replace. = ''
   do n = 1 to G.0ReplaceFileN.0
      call FileToArray 'Replace.', G.0ReplaceFileN.n
   end n

   New. = ''
   New.0 = 0
   /*  FindLabelsReplace exposes a list of labels in Repl.labels and their location in Repl.lines     */
   call FindLabelsReplace
   if Opt.0debug then
      call Mark 'Repl.labels Repl.lines'
   call FixExpose
   lnO = 0
   lnN = 0
   incomment = 0
   do while lnO < Old.0
      lnN = lnN +1
      lnO = lnO +1
      New.lnN = Old.lnO
      New.0 = lnN
      /*  Incomment() tracks the start and end comments to determine if we are outside                */
      /*  any comments. It's slow, but the only way to be completely sure that tmp.l                  */
      /*  really is a label                                                                           */
      if pos('/'||'*', Old.lnO) > 0 ,
       | pos('*'||'/', Old.lnO) > 0 then
         incomment = Incomment(Old.lnO, incomment)
      if incomment <= 0 then do
            if HasLabel(Old.lnO) then do
                  parse var Old.lnO label ':' .
                  label = strip(label)
                  /*  Old label == 'Foo'. Check list of Replacement labels for a match                */
                  labelID = wordpos(translate(label), translate(Repl.labels))
                  if labelID > 0 then do
                        if Opt.0where == '' then
                           Match = 1
                        else do
                              Match = 0
                              WhereList = Opt.0where
                              do while WhereList \= ''
                                 parse var WhereList pattern WhereList
                                 if StringMatch(label, pattern, 'I') then do
                                       Match = 1
                                       leave
                                    end
                              end
                           end
                        if Match then do
                              if Opt.0keep == 1 then do
                                    /*  also keep the old version of Foo                              */
                                    do l = lnO +1 to Old_EndOfProc(lnO)
                                       lnN = lnN +1
                                       New.lnN = Old.l
                                    end
                                 end
                              /*  skip to the end of Foo in the original                              */
                              lnO = Old_EndOfProc(lnO)
                              /*  start copying the replacement Foo from the label                    */
                              lnR = word(Repl.lines, labelID)
                              if Opt.0keep == 1 then do
                                    lnN = lnN +1
                                    New.lnN = '/'||'*' ,
                                              copies('%', 20) 'start new' label 'procedure' copies('%', 20) ,
                                              '*'||'/'
                                    New.0 = lnN
                                 end
                              /*  quick fix because we already stored label from Old                  */
                              lnN = lnN -1
                              do l = lnR to Rep_EndOfProc(lnR)
                                 lnN = lnN +1
                                 New.lnN = Replace.l
                                 New.0 = lnN
                              end
                              if Opt.0keep == 1 then do
                                    lnN = lnN +1
                                    New.lnN = '/'||'*' ,
                                              copies('%', 20) 'end new' label 'procedure' copies('%', 22) ,
                                              '*'||'/'
                                    New.0 = lnN
                                 end
                           end   /*  if Match  */
                     end   /*  if LabelID  */
               end   /*  if HasLabel  */
         end   /*  if Incomment <= 0  */
   end   /*  do while  */
   if ArraysMatch() then do
         call Msg G.0meID 'No changes needed for' G.0InfileN
      end
   else do
         call ArrayToFile 'New.', G.0OutfileN
      end
   return 0
   /* === End of Task_Replace ======================================================================= */

   FixExpose: procedure expose (Shared_vars) Old. Replace. Repl. /*fold00*/
   /* =============================================================================================== */
   /*  sometimes I customize the EXPOSE statement after importing a routine. This preserves           */
   /*  the customizing by combining the lists of exposed variables in the old and new lines           */
   /*  But that required other ad-hockery, like ignoring the 0-functions I use and preserving         */
   /*  the text editor's "FOLD00" comments. Not thoroughly tested yet.                                */
   /*  This is also where "procedure" gets added where needed.                                        */
   /* =============================================================================================== */
   PrevLabelN = '[none]'
   lnO = 0
   do while lnO < Old.0
      lnO = lnO +1
      if HasLabel(Old.lnO) then do
            lnO2 = lnO +1
            parse var Old.lnO LabelN ':' .
            LabelN = strip(LabelN)
            commentStr = ''
            /*  workaround to skip 0-functions that I use                                             */
            lnOp = lnO -1
            if HasLabel(Old.lnOp) then do
                  PrevLabelN = LabelN
                  iterate
               end
            labelID = wordpos(translate(LabelN), translate(Repl.labels))
            if labelID > 0 then do
                  lnR = word(Repl.lines, labelID)
                  lnR2 = lnR +1
                  lnRProc = ''
                  lnOProc = ''
                  /*  parse the old line to get the exposed variables and any comment                 */
                  /*  the comment will be added to the replacement line                               */
                  if word(translate(Old.lnO2), 1) == 'PROCEDURE' then do
                        parse var Old.lnO2 . . ExplistO (G.0CommentBegin) commentStr
                        lnOProc = lnO2
                     end
                  else do
                        parse var Old.lnO . ':' . . ExplistO (G.0CommentBegin) commentStr
                        lnOProc = lnO
                     end
                  /*  from the replacement line extract "procedure expose varN"                       */
                  /*  if it is not a procedure, Wprocedure will be ''                                 */
                  if word(translate(Replace.lnR2), 1) == 'PROCEDURE' then do
                        parse var Replace.lnR2 Wprocedure . ExplistR
                        lnRProc = lnR2
                     end
                  else do
                        parse var Replace.lnR . ':' Wprocedure . ExplistR
                        lnRProc = lnR
                     end
                  do while ExplistO \= ''
                     /*  check old expose list for variables that should be added to replacement list */
                     parse var ExplistO varN ExplistO
                     if wordpos(translate(varN), translate(ExplistR)) == 0 then
                        ExplistR = ExplistR varN
                  end
                  /*  keep the original indenting                                                     */
                  Replace.lnRProc = copies(' ', length(Replace.lnRProc) - length(strip(Replace.lnRProc, 'L')))
                  if lnRProc == lnR then
                     /*  a routine from a macro library is a procedure, but does not use              */
                     /*  the word "procedure", so we need to add it                                   */
                     if \abbrev(translate(LabelN), translate(PrevLabelN)'.') then
                        Replace.lnRProc = Replace.lnRProc||LabelN':' 'procedure'
                     else
                        /*  a sublabel, use "procedure" only if it already had one                    */
                        Replace.lnRProc = Replace.lnRProc||LabelN':' Wprocedure
                  else
                     Replace.lnRProc = Replace.lnRProc||Wprocedure
                  if ExplistR \= '' then
                     Replace.lnRProc = Replace.lnRProc 'expose' strip(ExplistR)
                  Replace.lnRProc = strip(Replace.lnRProc, 'T')
                  if commentStr \= '' then
                     Replace.lnRProc = Replace.lnRProc '/'||'*'||commentStr
               end   /*  if labelID  */
            PrevLabelN = LabelN
         end   /*  if HasLabel()  */
   end   /*  do while lnO  */
   return
   /* === End of FixExpose ========================================================================== */

   ArraysMatch: procedure expose (Shared_vars) (line_vars) /*fold00*/
   /*=================================================================================================*/
   /*  returns True if the Old and New arrays are exactly the same                                    */
   /*  By default the comparison is NOT case-sensitive and ignores leading/trailing                   */
   /*  spaces, because I don't want to replace a procedure every time the formatter                   */
   /*  made a slight change in the capitalization or indenting                                        */
   /*=================================================================================================*/
   if Old.0 \= New.0 then return 0
   if Opt.0strict then
      do l = 1 to Old.0
         if Old.l \== New.l then return 0
      end l
   else
      do l = 1 to Old.0
         if strip(translate(Old.l)) \== strip(translate(New.l)) then return 0
      end l
   return 1
   /* === End of ArraysMatch ======================================================================== */

   FindLabelsReplace: procedure expose (Shared_vars) (line_vars) Repl. /*fold00*/
   /* =============================================================================================== */
   /*  locates labels in the replacement text and the lines they are on                               */
   /*  exposes Repl.labels as a space-separated list of label names and Repl.lines as                 */
   /*  the lines they are on                                                                          */
   /* =============================================================================================== */

   Repl.labels = ''
   Repl.lines = ''
   incomment = 0
   do l = 1 to Replace.0
      if pos('/'||'*', Replace.l) > 0 ,
       | pos('*'||'/', Replace.l) > 0 then
         incomment = Incomment(Replace.l, incomment)
      if incomment <= 0 then do
            if HasLabel(Replace.l) then do
                  parse var Replace.l label ':' .
                  Repl.labels = Repl.labels strip(label)
                  Repl.lines = Repl.lines l
               end
         end
   end l
   return 0
   /* === End of FindLabelsReplace ================================================================== */

   Incomment: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     incomment = Incomment(string, incomment)                                                    */
   /*                                                                                                 */
   /*  tracks whether we are inside a comment                                                         */
   /*  returns incomment level, which may be > 1 if inside a nested comment                           */
   /* =============================================================================================== */
   parse arg str , incomment
   Esigl = sigl
   p = 1
   do while p <= length(str)
      c = substr(str, p, 1)
      if incomment <= 0 then
         if c == '"' ,
          | c == "'" then
            /*  skip over anything in quotes                                                          */
            do while p < length(str)
               p = p +1
               if substr(str, p, 1) == c then leave
            end
      cc = substr(str, p, 2)
      if cc == '/'||'*' then do
            incomment = incomment +1
            p = p +1
         end
      if cc == '*'||'/' then do
            incomment = incomment -1
            p = p +1
         end
      p = p +1
   end
   if incomment < 0 then do
         call Msg G.0meID 'Warning: negative incomment value "'incomment'"'
         call Msg 'the line is "'str'"'
         incomment = 0
      end
   return incomment
   /* === End of Incomment ========================================================================== */

   LabelList: procedure expose (Shared_vars) (line_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        LabelList = LabelList()                                                                  */
   /*                                                                                                 */
   /*  returns a list of all labels in the Old. array                                                 */
   /* =============================================================================================== */
   LabelList = ''
   incomment = 0
   do l = 1 to Old.0
      if pos('/'||'*', Old.l) > 0 ,
       | pos('*'||'/', Old.l) > 0 then
         incomment = Incomment(Old.l, incomment)
      if incomment <= 0 then do
            if HasLabel(Old.l) then do
                  parse var Old.l label ':' rest
                  LabelList = LabelList strip(label)
               end
         end
   end l
   return LabelList
   /* === End of LabelList ========================================================================== */

   Old_EndOfProc: procedure expose (Shared_vars) (line_vars) /*fold00*/
   /*=================================================================================================*/
   /*  Usage:                                                                                         */
   /*        EndOfProc = Old_EndOfProc(l)                                                             */
   /*        where l is a line number                                                                 */
   /*                                                                                                 */
   /*  returns the last line of the procedure in the Old array that line s is in                      */
   /*=================================================================================================*/
   arg s
   incomment = 0
   CurLabel = ''
   if HasLabel(Old.s) then do
         aline = strip(Old.s)
         parse var aline CurLabel ':' .
         s = s +1
      end
   do l = s to Old.0
      aline = strip(Old.l)
      if abbrev(translate(aline), translate(G.0footerID)) then
         return l
      if pos('/'||'*', aline) > 0 ,
       | pos('*'||'/', aline) > 0 then
         incomment = Incomment(aline, incomment)
      if incomment <= 0 then do
            if HasLabel(Old.l) then do
                  parse var aline LabelN ':' .
                  /*  ignore sublabel                                                                 */
                  if abbrev(translate(LabelN), translate(CurLabel)||'.') then iterate l
                  CurLabel = LabelN
                  eop = l -1
                  /*  back up to the previous non-blank line                                          */
                  if strip(Old.eop) == '' then
                     do until strip(Old.eop) \== ''
                        eop = eop -1
                     end
                  return eop
               end
         end
   end l
   return Old.0
   /* === End of Old_EndOfProc ====================================================================== */

   Rep_EndOfProc: procedure expose (Shared_vars) (line_vars) /*fold00*/
   /*=================================================================================================*/
   /*  Usage:                                                                                         */
   /*        EndOfProc = Rep_EndOfProc(l)                                                             */
   /*        where l is a line number                                                                 */
   /*                                                                                                 */
   /*  returns the last line of the procedure in the Replace array that line s is in                  */
   /*=================================================================================================*/
   arg s
   incomment = 0
   CurLabel = ''
   if HasLabel(Replace.s) then do
         aline = strip(Replace.s)
         parse var aline CurLabel ':' .
         s = s +1
      end
   do l = s to Replace.0
      aline = strip(Replace.l)
      if abbrev(translate(aline), translate(G.0footerID)) then do
            return l
         end
      if pos('/'||'*', aline) > 0 ,
       | pos('*'||'/', aline) > 0 then
         incomment = Incomment(aline, incomment)
      if incomment <= 0 then do
            if HasLabel(Replace.l) then do
                  parse var aline LabelN ':' .
                  /*  ignore sublabel                                                                 */
                  if abbrev(translate(LabelN), translate(CurLabel)||'.') then iterate l
                  CurLabel = LabelN
                  eop = l -1
                  /*  back up to the previous non-blank line                                          */
                  if strip(Replace.eop) == '' then
                     do until strip(Replace.eop) \== ''
                        eop = eop -1
                     end
                  return eop
               end
         end
   end l
   return Replace.0
   /* === End of Rep_EndOfProc ====================================================================== */

   SearchPath: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        FQFN = SearchPath(<list>, <filename>)                                                    */
   /*  Example:                                                                                       */
   /*        filename = SearchPath(OS2Var('PATH'), filename)                                          */
   /*                                                                                                 */
   /*  Returns the full path+filename if found, or '' if not found                                    */
   /*                                                                                                 */
   /*  Similar to SysSearchPath except that the first argument is not an environment                  */
   /*  variable, it is a semicolon-separated list of directories. Or on Unix, a                       */
   /*  colon-separated list.                                                                          */
   /*  Unlike SysSearchPath, if the filename already has a path+filename, it                          */
   /*  ignores the %path and returns the full filename.                                               */
   /*  Unlike Regina's SysSearchPath, it works                                                        */
   /* =============================================================================================== */
   parse arg PathList , fileN
   foundCmd = ''
   foundCmd = stream(fileN, 'c', 'query exists')
   if foundCmd \= '' then
      return foundCmd
   do while PathList \= ''
      parse var PathList dirN (G.0PathSep) PathList
      foundCmd = stream(dirN||G.0DirSep||fileN, 'c', 'query exists')
      if foundCmd \= '' then leave
   end
   return foundCmd
   /* === End of SearchPath ========================================================================= */

   Help: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   call Msg G.0me 'copies procedures from one Rexx file to another, replacing the original versions'
   call Msg 'Usage:'
   call Msg '   ' G.0me 'REPLACE <old file> [<new file>] -from=<filespec> [-where=<labels>] [-keep] [-strict]'
   call Msg '   ' G.0me 'IMPORT <old file> [<new file>] [-path=<path>] -where=<labels>'
   call Msg ''
   call Msg 'Examples for updating internal procedures:'
   call Msg '     update routines in Foo.cmd from New.cmd:'
   call Msg '     ' G.0me 'replace Foo.cmd Foo.cmd.new -from=New.cmd'
   call Msg ''
   call Msg '     update routines in Foo.cmd from several separate files:'
   call Msg '     ' G.0me 'replace Foo.cmd Foo.cmd.new -from="Fee.cmd Fie.cmd Foe.cmd"'
   call Msg '     ' G.0me 'replace Foo.cmd Foo.cmd.new -from=Fee.cmd,Fie.cmd,Foe.cmd'
   call Msg '     ' G.0me 'replace Foo.cmd Foo.cmd.new -from=F:\RexxFunctions\*.cmd'
   call Msg ''
   call Msg '     update only certain routines in Foo.cmd:'
   call Msg '     ' G.0me 'replace Foo.cmd Foo.cmd.new -from=F:\RexxFunctions\*.cmd -where="Fee Fie"'
   call Msg ''
   call Msg '     the same, but keep the old versions also:'
   call Msg '     ' G.0me 'replace Foo.cmd Foo.cmd.new -from=F:\RexxFunctions\*.cmd -where="Fee Fie" -keep'
   call Msg ''
   call Msg '     procedure names can use leading or trailing * as wildcard'
   call Msg '     ' G.0me 'replace Foo.cmd Foo.cmd.new -from=F:\RexxFunctions\*.cmd -where=F*'
   call Msg ''
   call Msg '     the body of the procedure is copied but the original label line is kept'
   call Msg ''
   call Msg '     by default, the comparison is NOT case-sensitive and ignores leading spaces.'
   call Msg '     That means a procedure will not be replaced if the only difference is in the'
   call Msg '     indenting or capitalization of words.'
   call Msg '     You can override that with the -strict switch.'
   call Msg ''
   call Msg 'Examples for importing external procedures:'
   call Msg '     to import Fee.cmd and Fie.cmd from anywhere in the %PATH:'
   call Msg '     ' G.0me 'import Foo.cmd Foo.cmd.new -where="Fee Fie"'
   call Msg '     if run under Regina, it will also search directories in %REGINA_MACROS'
   call Msg '     and it understands %REGINA_SUFFIXES'
   call Msg ''
   call Msg '     to import F:\Rexx_macros\Fee.cmd and D:\Rexx\Fie.cmd'
   call Msg '     ' G.0me 'import Foo.cmd Foo.cmd.new -path=F:\Rexx_macros;D:\Rexx -where=Fee,Fie'
   call Msg ''
   call Msg '     note that the Import command does not allow wildcards in the value of -where'
   call Msg ''
   call Msg '     the -path= switch uses the normal separator of the operating system, a ":" on Unix'
   call Msg '     and ";" on OS/2. (later-- now you can also use commas)'
   call Msg ''
   call Msg 'remember that Unix/Linux has case-sensitive file names, so the values of -path and -where'
   call Msg 'must match the directory and filenames exactly'
   call Msg ''
   call Msg 'if no <new file> is given, output will be to STDOUT'
   call Msg ''
   return 1
   /* === End of Help =============================================================================== */

   Syntax: /*fold00*/
   0Syntax:
   /* =============================================================================================== */
   /*  depends on variables: G.0me sigl                                                               */
   /*  depends on functions: LastLabel(), ShowError(), Egress()                                       */
   /* ----------------------------------------------------------------------------------------------- */
   /*  traps syntax errors and shows a better error message than Rexx does. In particular,            */
   /*  an incomplete DO/SELECT/IF is fairly common and the error message from Rexx is no              */
   /*  help at all.                                                                                   */
   /*                                                                                                 */
   /*  Some errors cannot be handled by an error handler, like REX06 "unmatched comment or quote"     */
   /*  (Which makes sense. If a defective script cannot be tokenised, then it cannot be run, so       */
   /*  the handler never exists. But the help message for REX06 implies that error 6 may not be       */
   /*  detected until later, so there is something I don't understand.)                               */
   /*                                                                                                 */
   /*  This will always be a work in progress                                                         */
   /* =============================================================================================== */
   SyntaxSigl = sigl
   call lineout 'stderr', ''
   /*  sometimes a "control stack full" will be trapped here and sometimes not. I don't know why.     */
   /*  If the control stack is full, any DO or SELECT after this will probably trigger                */
   /*  another (untrapped) error 11. Get the important part of the message onto the                   */
   /*  screen before that happens, without using DO.                                                  */
   /*  A trivial bug: The default formatting style puts a space before a continuation                 */
   /*  comma. In this case the space becomes part of the string to be displayed, at                   */
   /*  the start of the next line. Not worth fixing.                                                  */
   if rc = 11 then
      call lineout 'stderr' ,
           '[SYNTAX Handler] Rexx error 11: Control Stack full at line' SyntaxSigl||G.0CRLF ,
           '                      ==>' strip(sourceline(SyntaxSigl))||G.0CRLF ,
           '                There may be another error 11 after this'||G.0CRLF
   else do
         select
            when rc == 14 then do
                  /*  incomplete DO/SELECT/IF                                                         */
                  /* Rexx reports an error at the end of file, which is useless                       */
                  call ShowError '[SYNTAX Handler]'
                  call ShowError '    Syntax error 'rc' which is "'errortext(rc)'"', ''
                  if SyntaxSigl >= sourceline() then do
                        call ShowError '    Rexx says it is at line 'SyntaxSigl', which is the end of file.', ''
                     end
                  else do
                        call ShowError '    at line 'SyntaxSigl, ''
                        call ShowError '       ==> 'strip(sourceline(SyntaxSigl)), ''
                     end
               end
            otherwise
               Err_func = LastLabel(SyntaxSigl)
               call ShowError '[SYNTAX Handler] Syntax error at line 'SyntaxSigl 'in label' Err_func
               call ShowError '==>' sourceline(SyntaxSigl), ''
               call ShowError 'Error # 'rc' is "'errortext(rc)'"', ''
               if condition('d') \= '' then call ShowError 'condition(D) is "'condition('d')'"', ''
         end   /*  select  */
      end
   call ShowError 'for more help, type "helpmsg REX'rc'"', ''
   call Egress 'BUG_Syntax'
   /* === End of 0Syntax ============================================================================ */

   NoValue: /*fold00*/
   0NoValue:
   /* =============================================================================================== */
   /*  A simple NoValue handler that shows the name of the uninitialized variable and                 */
   /*  the line where it occurs.                                                                      */
   /*  The only variables it uses are _ and __, which it assumes are not used elsewhere               */
   /* ----------------------------------------------------------------------------------------------- */
   signal off NoValue
   _ = sigl
   parse source . . __
   __ = filespec('name', __)

   call lineout 'stderr', '['__'][NOVALUE Handler] at line '_' for variable "'condition('d')'"'
   call lineout 'stderr', copies(' ', length(__) +2)||'==> 'sourceline(_)
   call lineout 'stderr', copies(' ', length(__) +2)||'NOVALUE can also be triggered by a missing CALL'
   exit 9
   /* === End of 0NoValue =========================================================================== */

   Msg: /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call Msg <text>                                                                          */
   /* ----------------------------------------------------------------------------------------------- */
   /*   sends <text> to stderr.                                                                       */
   /*  this exists because I got tired of typing "call lineout 'stderr',"                             */
   /* =============================================================================================== */
   call lineout 'stderr', arg(1)
   return 0
   /* === End of Msg ================================================================================ */

   Mark: /*fold00*/
   0Mark:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL Mark [<list of variable names>] [, <string>]                                         */
   /* ----------------------------------------------------------------------------------------------- */
   /*  Shows the filename and line number it was called from, the current values of the               */
   /*  variables named, and an informational string                                                   */
   /*  Arg(1) is a list of variable names                                                             */
   /*  Arg(2) can be any descriptive text                                                             */
   /*  Examples:                                                                                      */
   /*      CALL mark 'varA varB', 'At top of the "DO x = 1 to 5" loop'                                */
   /*  If no variable names are given, it will just show a message:                                   */
   /*      CALL mark '', 'At top of the "DO x = 1 to 5" loop'                                         */
   /*                                                                                                 */
   /*  This must share all the variables of whatever routine it was called from.                      */
   /*  I cannot think of a way to do it without setting at least one variable here,                   */
   /*  so reserve the variable '_' for the use of Mark and similar routines.                          */
   /* =============================================================================================== */
   _ = sigl
   call lineout 'stderr', ''
   if arg(2, 'e') ,
    & arg(2) = '' then nop
   else call lineout 'stderr', '[Mark]['value('G.0me')':'_']' arg(2)
   do _ = 1 to 99
      if word(arg(1), _) = '' then leave _
      if symbol(word(arg(1), _)) = 'VAR' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||'= "'||value(word(arg(1), _))||'"'
      if symbol(word(arg(1), _)) = 'LIT' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||' is not defined'
      if symbol(word(arg(1), _)) = 'BAD' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||' is not a variable'
   end _
   drop _
   return 0
   /* === End of 0Mark ============================================================================== */

   LineNo: /*fold00*/
   0LineNo:
   /* =============================================================================================== */
   /*  returns the number of the current line--    say 'now at line 'LineNo()                         */
   /* =============================================================================================== */
   _ = sigl
   return _
   /* === End of 0LineNo ============================================================================ */

   LineID: /*fold00*/
   0LineID:
   /* =============================================================================================== */
   /*    LineID()                                                                                     */
   /* ----------------------------------------------------------------------------------------------- */
   /*  returns a string identifying this file and the line number it was called from                  */
   _ = sigl
   return '['G.0me':'right(_, 5, '0')']'
   /* === End of 0LineID ============================================================================ */

   SubID: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     SubID = SubID()                                                                             */
   /*     call lineout 'stderr', 'Error in ' SubID()                                                  */
   /*                                                                                                 */
   /*  returns a string identifying the label name of the current routine                             */
   /* =============================================================================================== */
   Esigl = sigl
   return '['LastLabel(Esigl)']'
   /* === End of SubID ============================================================================== */

   ShowError: procedure expose (Shared_vars) /*fold00*/
   0ShowError:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call ShowError <Arg1> [, <Arg2>]                                                         */
   /* ----------------------------------------------------------------------------------------------- */
   /*   sends a message to stderr, adding a header with the program name, function name,              */
   /*   and line number to show where the message comes from.                                         */
   /*   Use '' as the second arg to improve the display of a multi-line message:                      */
   /*                                                                                                 */
   /*   Example: in function Add, to show a multi-line message:                                       */
   /*       call ShowError 'Warning: unexpected value for variable varA: "'varA'"'                    */
   /*       call ShowError 'It should be a number', ''                                                */
   /*   displays:                                                                                     */
   /*       [Foo.cmd:Add:33] Warning: unexpected value for variable varA: "foo"                       */
   /*                        It should be a number                                                    */
   /*                                                                                                 */
   /* =============================================================================================== */
   seSigl = sigl

   if Opt.0verbose \= 0 then
      seID = '['G.0me':'LastLabel(seSigl)':'seSigl']'
   else
      seID = '['G.0me':'seSigl']'

   if arg(2, 'e') ,
    & arg(2) = '' then
      seID = copies(' ', length(seID))
   else
      call lineout 'stderr', ''

   call lineout 'stderr', seID arg(1)
   return 0
   /* === End of 0ShowError ========================================================================= */

   Egress: /*fold00*/
   0Egress:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     call Egress <exit type>                                                                     */
   /* ----------------------------------------------------------------------------------------------- */
   /*  All exits come through here.                                                                   */
   /*  If you need to, say, clean up temporary files, do it from here                                 */
   /*                                                                                                 */
   /*  OKAY    == normal exit                                                                         */
   /*  USAGE   == user error                                                                          */
   /*  ERROR   == something else wrong                                                                */
   /*  BUG     == bug in the program                                                                  */
   /*                                                                                                 */
   /*  if you customize this, remember to delete the "0Egress" label                                  */
   /* =============================================================================================== */
   signal off NoValue
   signal off Syntax
   parse upper arg ExitCode

   /*    handle my typing mistakes                                                                    */
   if ExitCode = 'OKAY' then exit 0
   if ExitCode = '' then exit 0
   if ExitCode = 0 then exit 0
   if abbrev(ExitCode, 'HALT') then exit 1
   if abbrev(ExitCode, 'USAGE') then exit 1
   if abbrev(ExitCode, 'ERROR') then exit 3
   if abbrev(ExitCode, 'BUG') then exit 9
   /*    in case I used some other exit string and forgot to add it here:                             */
   exit 99
   /* === End of 0Egress ============================================================================ */

   ArrayToFile: procedure expose (Shared_vars) (line_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL ArrayToFile 'Stemvar.' , 'FileName'                                                  */
   /* ----------------------------------------------------------------------------------------------- */
   /*  writes a stem array to a file                                                                  */
   /*  the stem =name= is passed, but the stem =variable= itself must be exposed                      */
   /*  if filename is not given, default is stdout                                                    */
   /* =============================================================================================== */
   parse arg stemN , fileN
   if fileN == '' then fileN = 'stdout'
   if stemN == '' then do
         call ShowError 'In ArrayToFile, invalid arguments'
         call Egress 'BUG_Function'
      end
   !listN = value('stemN')
   if !listN == '' then do
         call ShowError 'In ArrayToFile, invalid arguments'
         call Egress 'BUG_Function'
      end

   /*  cosmetic improvement, keep the progress and output from stepping on each other                 */
   if fileN == 'stdout' then
      call Msg ''
   else
      if Opt.0progress then call Progress SubID()

   if right(!listN, 1) \= '.' then !listN = !listN||'.'
   ListHigh = value(!listN||0)
   do i = 1 to ListHigh
      thisline = value(!listN||i)
      call lineout fileN, thisline
   end
   /*  make sure there is an ending CRLF                                                              */
   if value(!listN||ListHigh) \== '' then
      call lineout fileN, ''
   call lineout fileN
   if Opt.0progress then call Progress
   return 0
   /* === End of ArrayToFile ======================================================================== */

   FileToArray: procedure expose (Shared_vars) (line_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL FileToArray 'Stemvar.' , 'FileName'                                                  */
   /* ----------------------------------------------------------------------------------------------- */
   /*  reads input file 'FileName' into an array named 'Stemvar.', return 0 if successful             */
   /*                                                                                                 */
   /*  The name of the variable is passed, but the variable itself must be exposed                    */
   /*  The stem variable should already be initialized                                                */
   /* =============================================================================================== */
   Esigl = sigl
   if Opt.0progress then call Progress SubID()
   parse arg stemN , fileN
   !listN = value('stemN')
   if !listN = '' ,
    | fileN = '' ,
    | right(!listN, 1) \= '.' then do
         call ShowError 'In FileToArray, invalid argument(s)'
         call Mark 'Esigl stemN fileN !listN'
         call Egress 'BUG_Function'
      end

   i = value(!listN||0)
   if i == '' then i = 0
   /*  interpreting a line is slightly faster than a loop using 'value' for each assignment           */
   FileArray_script = ,
     'do while chars(fileN) > 0 ; i = i +1 ; '!listN||'i = linein(fileN) ; end ; '!listN||'0 = i'
   interpret FileArray_script
   call stream fileN, 'c', 'CLOSE'
   if Opt.0progress then call Progress
   return 0
   /* === End of FileToArray ======================================================================== */

   LastLabel: procedure expose (Shared_vars) sigl /*fold00*/
   0LastLabel:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       ret = LastLabel()                                                                         */
   /*       ret = LastLabel(<line number>)                                                            */
   /* ----------------------------------------------------------------------------------------------- */
   /*  returns the name of the nearest label =in this program= previous to <line number>              */
   /*  if <line> is not given then it is the current line                                             */
   /*  returns '' if no label was found or if <line> is not a valid line number                       */
   /*                                                                                                 */
   /*  it is normally used to determine the name of a routine or function:                            */
   /*      Foo: procedure                                                                             */
   /*      say 'Now in procedure 'Lastlabel()       --> "Now in procedure Foo"                        */
   /*                                                                                                 */
   /*  akm: This partly depends on the style of writing I use, where                                  */
   /*  multiline comments begin at the beginning of a line and end at the end of a                    */
   /*  line. It could be fooled by something that looks like a label inside other                     */
   /*  kinds of multiline comments. But it is useful, and if it is wrong it will only                 */
   /*  affect the progress and debugging messages. This is about as reliable as I can                 */
   /*  make it without parsing every comment letter by letter.                                        */
   /* =============================================================================================== */
   if arg(1, 'o') then sline = sigl ; else sline = arg(1)
   if datatype(sline, 'W') \= 1 then return ''
   if sline > sourceline() then return ''
   incomment = 0
   do l = sline to 1 by -1
      aline = strip(sourceline(l))
      if left(aline, 2) == '/'||'*' ,
       | right(aline, 2) == '*'||'/' then do
            /*  counter-intuitively, incomment is incremented by the end of comment,                  */
            /*  because we are working backwards                                                      */
            incomment = incomment - CountStr('/'||'*', aline) + CountStr('*'||'/', aline)
         end

      if incomment <= 0 then do
            if HasLabel(aline) then do
                  parse var aline label ':' .
                  label = strip(label)
                  return label
               end
         end
   end l
   return ''
   /* === End of 0LastLabel ========================================================================= */

   HasLabel: procedure expose (Shared_vars) /*fold00*/
   0HasLabel:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        if HasLabel(<string>) then ...                                                           */
   /*  returns True if <string> is a line that looks like a label                                     */
   /*  the calling routine is responsible for identifying multi-line comments                         */
   /* =============================================================================================== */
   /*  it is a label if str is one word followed by ':', followed by '', 'procedure', or a comment    */
   parse arg str
   str = strip(str)
   if pos(':', str) == 0 then return 0
   if left(str, 2) == '/'||'*' then return 0
   parse var str label ':' rest
   rest = strip(rest)
   if words(label) \= 1 then return 0
   if symbol(label) == 'BAD' then return 0
   if rest \= '' then
      select
         when translate(word(rest, 1)) == 'PROCEDURE' then
            return 1
         when left(rest, 2) == '/'||'*' then
            return 1
         otherwise
            return 0
      end   /*  select  */
   return 1
   /* === End of 0HasLabel ========================================================================== */

   Progress: procedure expose (Shared_vars) /*fold00*/
   0ShowProgressUnix:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL ShowProgressUnix <string> [, <row>, <column>]                                        */
   /*                                                                                                 */
   /* ----------------------------------------------------------------------------------------------- */
   /*  a version of ShowProgress for either OS/2 or Linux                                             */
   /*  On OS/2 writes <string> to the screen at <row>, <col> and returns cursor to where it was.      */
   /*  On Linux, writes <string> at the bottom of the screen.                                         */
   /*                                                                                                 */
   /*  <string> is padded or trimmed to screen width                                                  */
   /*  If <row> or <col> are not given, defaults to current cursor position                           */
   /*  Call it from within a loop to show progress. When done, call it with a null string             */
   /*  to erase the progress display                                                                  */
   /*  returns 1 on success, 0 if bad parameter                                                       */
   /* =============================================================================================== */
   /*  On Linux:                                                                                      */
   /*  Without a fully-working SysCurPos(), the best I can do is to put the progress                  */
   /*  display at the bottom of the screen.                                                           */
   /*  '/dev/tty' seems to be the Unix equivalent of the OS/2 SCREEN$ device (?)                      */
   /*  BUT, SysCurPos() on Linux seems to send its output to stdout, so redirecting stdout            */
   /*  will result in the screen display being garbled and the redirected output will                 */
   /*  have ANSI terminal sequences mixed with it.                                                    */
   /*  If you really want to redirect stdout, you will want to use "-progress=0" on the               */
   /*  command line. There is no problem with error messages because they go to stderr.               */
   /*                                                                                                 */
   /*  Someone who knows more about Regina and Linux could probably improve this                      */
   /* ----------------------------------------------------------------------------------------------- */
   parse arg msg , row , col
   select
      when G.0OpSystem == 'UNIX' then do
            parse value SysTextScreenSize() with maxRow maxCol
            row = maxRow
            col = 0
            call SysCurPos row, col
            msgLen = (maxCol - col -1)
            msg = left(msg, msgLen)
            call charout '/dev/tty', msg
            call SysCurPos row, col
         end
      otherwise
         /*  presumably OS/2. This might also work on Windows?                                        */
         parse value SysCurPos() with origRow origCol
         if row = '' then row = origRow
         if col = '' then col = origCol
         if \datatype(row, 'w') ,
          | \datatype(col, 'w') then return 0
         parse value SysTextScreenSize() with maxRow maxCol
         maxRow = maxRow -1 ; maxCol = maxCol -1
         if row = '' then row = origRow
         if col = '' then col = origCol
         if row > maxRow then row = maxRow
         if col > maxCol then col = maxCol
         if row < 0 then row = 0
         if col < 0 then col = 0
         msgLen = (maxCol - col)
         if row < maxRow then msgLen = msgLen +1
         msg = left(msg, msgLen)
         call SysCurPos row, col
         call charout 'SCREEN$', msg
         call SysCurPos origRow, origCol
   end   /*  select  */
   return 1
   /* === End of 0ShowProgressUnix ================================================================== */

   ChangeString: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        NewString = ChangeString(<Old>, <String>, <New>)                                         */
   /*                                                                                                 */
   /*  replaces every occurrence of <Old> with <New> in <String>                                      */
   /*  <Old> and <New> do not have to be the same length                                              */
   /*  if <New> is '', <Old> will be removed from <String>                                            */
   /*                                                                                                 */
   /*  this is similar to the ChangeStr() function from Object Rexx                                   */
   /* =============================================================================================== */
   parse arg Old , string , New
   charpos = 1
   do forever
      charpos = pos(Old, string, charpos)
      if charpos = 0 then leave
      string = left(string, charpos -1)||New||substr(string, charpos + length(Old))
      charpos = charpos + length(New)
   end
   return string
   /* === End of ChangeString ======================================================================= */

   CountStr: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        Count = CountStr(<needle>, <haystack>)                                                   */
   /*                                                                                                 */
   /*  counts how many times the string <needle> occurs in the string <haystack>, without overlap     */
   /*  if either <needle> or <haystack> is null, returns 0                                            */
   /*                                                                                                 */
   /*  this is an imitation of the CountStr() function from Object Rexx                               */
   /* =============================================================================================== */
   parse arg needle , haystack

   found = 0
   p = pos(needle, haystack)
   do while p > 0
      found = found +1
      p = pos(needle, haystack, p + length(needle))
   end
   return found
   /* === End of CountStr =========================================================================== */

   FileBaseName: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        name = FileBaseName(<filename>)                                                          */
   /*  returns basename of a filename, like 4os2 "%@name[]" or almost like                            */
   /*  Unix "basename -s"                                                                             */
   /* =============================================================================================== */
   parse arg fileN
   fileN = filespec('N', fileN)
   if fileN == '' then return ''
   if pos('.', fileN) == 0 then return fileN
   return substr(fileN, 1, lastpos('.', fileN) -1)
   /* === End of FileBaseName ======================================================================= */

   FileExist: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        if FileExist(<filename>) then                                                            */
   /*                                                                                                 */
   /*  returns 1 (True) if <filename> exists and 0 (False) if it does not.                            */
   /*                                                                                                 */
   /*  <filename> must be a real file. This returns 0 for devices or for streams like STDOUT.         */
   /*  Granted, STDOUT does exist, but it is more useful this way. I mostly use this when I want      */
   /*  to avoid clobbering an existing file,                                                          */
   /*                                                                                                 */
   /*  The main reason for this is that                                                               */
   /*        if FileExist(filename) then                                                              */
   /*  is easier to remember than                                                                     */
   /*         if stream(filename, 'c', 'query exists') then                                           */
   /* =============================================================================================== */
   /*  Most of the complications are because OS/2 Rexx and Regina return different values             */
   /*  from stream() for stdout and for devices. Regina usually returns 0 for stdout, but             */
   /*  not always. I don't know why.                                                                  */
   parse arg fileN
   if fileN = '' then return 0
   if wordpos(translate(fileN), 'STDOUT STDERR') > 0 then return 0
   ret = stream(fileN, 'c', 'query exists')
   if ret == '' then return 0
   if left(translate(ret), 5) == '\DEV\' then return 0
   if left(translate(ret), 5) == '/DEV/' then return 0
   return 1
   /* === End of FileExist ========================================================================== */

   GetOpts: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     INTERPRET GetOpts(args [,valid])                                                            */
   /*                                                                                                 */
   /*  where 'args' is the command line to be parsed, and                                             */
   /*  'valid' is an optional space-separated list of valid switches                                  */
   /*  'help' and '?' are always valid                                                                */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*     parse arg args                                                                              */
   /*     interpret GetOpts(args, 'switchA switchB')                                                  */
   /*                                                                                                 */
   /* ----------------------------------------------------------------------------------------------- */
   /*  Switches must start with '-'. For example:                                                     */
   /*                                                                                                 */
   /*         -Foo                (Opt.0Foo == 1)                                                     */
   /*         -Foo=1              (Opt.0Foo == 1)                                                     */
   /*         -!Foo or -Foo=0     (Opt.0Foo == 0)                                                     */
   /*         -Foo=string         (Opt.0Foo == 'string')                                              */
   /*         -Foo="string"       (Opt.0Foo == 'string')                                              */
   /*         -Foo=a,b,c          (Opt.0Foo == 'a,b,c')                                               */
   /*         -Foo="a, b, c"      (Opt.0Foo == 'a, b, c')                                             */
   /*         -Foo="a b c"        (Opt.0Foo == 'a b c')                                               */
   /*         -Foo=123            (Opt.0Foo == '123')                                                 */
   /*         -Foo= or -Foo=''    (Opt.0Foo == '')                                                    */
   /*                                                                                                 */
   /*  switches must not have spaces before or after the '='                                          */
   /*  values with spaces must be in quotes. the quotes will be stripped.                             */
   /*                                                                                                 */
   /*  a switch without a value is set to '1'                                                         */
   /*        -switchA                                                                                 */
   /*  is the same as                                                                                 */
   /*        -switchA=1                                                                               */
   /*                                                                                                 */
   /*  a switch can be negated with "!", so                                                           */
   /*        -!switchA                                                                                */
   /*  is the same as                                                                                 */
   /*        -switchA=0                                                                               */
   /*                                                                                                 */
   /*     -? is the equivalent of -help                                                               */
   /*     they will both result in:                                                                   */
   /*        Opt.0help    == 1                                                                        */
   /*                                                                                                 */
   /*  Interpreting what GetOpts returns will create several variables that the                       */
   /*  caller can use--                                                                               */
   /*     Opt.              is a stem variable that holds whatever was specified by switches          */
   /*     Opt.0optlist      a space-separated list of options that were read                          */
   /*     Opt.0errors       number of errors; that is, when GetOpts() did not understand something    */
   /*                                                                                                 */
   /*                       elements of the Opt. stem variable all start with '0', to avoid           */
   /*                       name conflicts with simple variables                                      */
   /*                                                                                                 */
   /*     ArgV.             is a stem that holds the non-switch arguments                             */
   /*                       (ArgV.0 is the name of the .cmd script, NOT the number of items           */
   /*                       in the stem)                                                              */
   /*     ArgC              is the number of non-switch arguments                                     */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*     Foo.cmd -dir=up -msg="that way" -v "From here"                                              */
   /*     will result in these variables:                                                             */
   /*        Opt.0optlist == opt.0dir opt.0msg                                                        */
   /*        Opt.0dir     == up                                                                       */
   /*        Opt.0msg     == that way                                                                 */
   /*        Opt.0v       == 1                                                                        */
   /*        ArgV.0       == Foo.cmd                                                                  */
   /*        ArgV.1       == From here                                                                */
   /*        ArgC         == 1                                                                        */
   /*                                                                                                 */
   /*  in the case of errors, GetOpts() will set Opt.0errors to the number of errors                  */
   /*  found, and cause the caller to show a message about it on stderr. The caller                   */
   /*  should probably abort if Opt.0errors is greater than 0.                                        */
   /* =============================================================================================== */
   /* Some important variables (inside GetOpts) are:                                                  */
   /*    Str.SetOpts       == commands the the caller will execute to set options                     */
   /*                          in the Opt. stem variable                                              */
   /*    Str.SetArgs       == commands the the caller will execute to put the arguments               */
   /*                          in the ArgV. and ArgC variables                                        */
   /*    Str.Errors        == commands the caller will execute to report errors                       */
   /*                          reading the command line                                               */
   /*    ReturnStr         == all of the above, combined and returned to caller                       */
   /*    flag.             == variables for GetOpt's internal use                                     */
   /*                          flag.error == number of errors trying to read command line             */
   /*    Str.SwitchList    == list of options (switches) used on command line                         */
   /*    ArgC              == number of non-switch arguments                                          */
   /*    ArgV.             == array of non-switch arguments                                           */
   /*    valid             == list of valid switches                                                  */
   /*                                                                                                 */
   /* =============================================================================================== */
   flag. = ''
   flag.errors = 0
   str. = ''
   ReturnStr = ''
   ArgC = 0
   ArgV. = ''
   parse arg args , valid
   do i = 1 to WordsQ(args)
      thisArg = WordQ(args, i)
      if left(thisArg, 1) == '-' then
         call GetOpts.ReadSwitch thisArg
      else do
            /*    not a switch, add to Arg list                                                       */
            ArgC = ArgC +1
            ArgV.ArgC = thisArg
         end
   end
   /* ----------------------------------------------------------------------------------------------- */
   /*    now put it all together into a series of commands for the caller to interpret                */
   if flag.errors > 0 then do
         str.SwitchList = strip(str.SwitchList' Opt.0Errors')
      end
   str.SetOpts = str.SetOpts' Opt.0Errors = 'flag.errors ';'
   str.SetOpts = str.SetOpts 'Opt.0optlist = "'str.SwitchList'" ;'

   str.SetArgs = 'ArgV. = "" ; parse source . . ArgV.0 ;  ArgC = "'ArgC'" ;'
   do i = 1 to ArgC
      varVal = Unquoted(ArgV.i)
      str.SetArgs = str.SetArgs' ArgV.'i' = 'Quoted(varVal)' ;'
   end

   ReturnStr = str.errors str.SetOpts str.SetArgs
   return ReturnStr

   GetOpts.ReadSwitch: procedure expose flag. str. valid (Shared_vars)
   /* =============================================================================================== */
   /*  this is part of GetOpts()                                                                      */
   /* =============================================================================================== */
   parse arg thisArg
   Negate = 0
   switch = strip(thisArg, 'L', '-')
   if left(switch, 1) == '!' then do
         switch = strip(switch, 'L', '!')
         Negate = 1
      end
   parse var switch optName '=' optVal
   select
      /* -? and -help have fixed meanings and are allowed even if not included in 'valid'             */
      when translate(optName) == '?' then
         optName  = 'Opt.0Help'
      when translate(optName) == 'HELP' then
         optName  = 'Opt.0Help'
      when optName == '' then do
            str.errors = ,
              str.errors||'call lineout "stderr", "[GetOpts] `-` without a switch name" ;'
            flag.errors = flag.errors +1
            return
         end
      when valid \= '' ,
         & wordpos(translate(optName), translate(valid)) == 0 then do
            str.errors = str.errors|| ,
                         'call lineout "stderr", "[GetOpts] I do not understand switch `'thisArg'`" ;'
            flag.errors = flag.errors +1
            return
         end
      otherwise
         optName = 'Opt.0'optName
   end   /*  select  */

   select
      when Negate then
         if optVal == '' then
            optVal = 0
         else do
               str.errors = str.errors|| ,
                            'call lineout "stderr", "[GetOpts] I do not understand switch `'thisArg'`" ;'
               flag.errors = flag.errors +1
               return
            end
      when optVal == '' ,
         & right(switch, 1) \= '=' then
         /*  "-switch=" means Opt.0switch == '', not 1                                                */
         optVal = '1'
      otherwise
         /*    try to be tolerant of unusual arguments; only strip quotes if they are balanced        */
         /*    and only one level of nested quotes                                                    */
         optVal = Unquoted(optVal)
   end   /*  select  */

   if wordpos(optName, str.SetOpts) > 0 then do
         str.errors = ,
           str.errors||'call lineout "stderr", "[GetOpts] duplicate switch `'thisArg'`" ;'
         flag.errors = flag.errors +1
      end
   else do
         str.SetOpts = str.SetOpts' 'optName' = 'Quoted(optVal)' ;'
         str.SwitchList = strip(str.SwitchList' 'optName)
      end
   return
   /* === End of GetOpts ============================================================================ */

   OS2Var: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        VarValue = OS2var(<variable name>)                                                       */
   /*        OS2Path = OS2var('PATH')                                                                 */
   /*                                                                                                 */
   /*     returns value of a variable in the OS/2 environment,                                        */
   /*     or '' if not defined                                                                        */
   /* =============================================================================================== */
   parse arg Pname
   return value(Pname, , 'OS2ENVIRONMENT')
   /* === End of OS2Var ============================================================================= */

   Quoted: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        string = Quoted(string)                                                                  */
   /*                                                                                                 */
   /*    if possible, quotes a string with the kind of quotes it does not contain                     */
   /*        {He said "no"}   --> {'He said "no"'}                                                    */
   /*        {He said 'no'}   --> {"He said 'no'"}                                                    */
   /*    otherwise uses double quotes and lets the caller deal with the errors                        */
   /* =============================================================================================== */
   parse arg string
   select
      when pos('"', string) == 0 then
         newstring = '"'string'"'
      when pos("'", string) == 0 then
         newstring = "'"string"'"
      otherwise
         newstring = '"'string'"'
   end
   return newstring
   /* === End of Quoted ============================================================================= */

   StringMatch: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        ret = StringMatch(<string>, <template> [,'I']                                            */
   /*                                                                                                 */
   /*  returns 1 if <string> matches <template>                                                       */
   /*  template can contain * and ? as wild characters                                                */
   /*  by default the comparison is case-sensitive.                                                   */
   /*  A third argument of 'I' cause it to be case-insensitive                                        */
   /*                                                                                                 */
   /*  this is adapted from Marcel Mller's StringMatchQ procedure in RXMMUTIL.CMD                    */
   /*  at <http://www.maazl.de>                                                                       */
   /*  case-insensitive option added by Anton Monroe                                                  */
   /* =============================================================================================== */
   str = arg(1)
   template = arg(2)
   if translate(left(arg(3), 1)) == 'I' then do
         str = translate(str)
         template = translate(template)
      end

   StringMatch.redo:
   if length(str) = 0 then
      return verify(template, '*') = 0            /* '' matches only if template consists only of '*' */
   p = verify(template, '?*', 'M')
   if p = 0 then
      return compare(str, template, '*') = 0                       /* no wildcards => compare strings */
   if compare(left(str, p -1), left(template, p -1, '*')) \= 0 then
      return 0                                              /* compare of non-wildcard section failed */
   j = p
   do i = p to verify(template' ', '*?', , p +1) -1   /*  count # of ?  */
      if substr(template, i, 1) = '?' then
         j = j +1
   end
   if j > length(str) +1 then
      return 0                                                           /* more ? than length of str */
   template = substr(template, i)
   str = substr(str, j)
   if i = j then
      signal StringMatch.redo
   /* '*'                                                                                             */
   if length(template) = 0 then
      return 1                                                                   /* nothing after '*' */
   do p = 1
      p = pos(left(template, 1), str, p)
      if p = 0 then
         return 0                                                    /* character after '*' not found */
      if StringMatch(substr(str, p +1), substr(template, 2)) then
         return 1                                                                          /* got it! */
   end
   /* variant 2                                                                                       */
   /*     if i \= j then do                      /* '*' */                                            */
   /*           if length(template) = 0 then                                                          */
   /*              return 1                      /* nothing after '*' */                              */
   /*           p = lastpos(left(template, 1), str)                                                   */
   /*           say 'p*: 'p                                                                           */
   /*           if p = 0 then                                                                         */
   /*              return 0                      /* character after '*' not found */                  */
   /*           str = substr(str, p +1)                                                               */
   /*           template = substr(template, 2)                                                        */
   /*        end                                                                                      */
   /*     signal StringMatch.redo                                                                     */
   /* === End of StringMatch ======================================================================== */

   Unquoted: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        string = Unquoted(string)                                                                */
   /*                                                                                                 */
   /*    strips quotes from a string if and only if it is enclosed in a balanced pair                 */
   /*    of single or double quotes                                                                   */
   /* =============================================================================================== */
   parse arg string
   if length(string) < 2 then return string
   select
      when left(string, 1) == '"' ,
         & right(string, 1) == '"' then
         string = substr(string, 2, length(string) -2)
      when left(string, 1) == "'" ,
         & right(string, 1) == "'" then
         string = substr(string, 2, length(string) -2)
      otherwise
         nop
   end
   return string
   /* === End of Unquoted =========================================================================== */

   WordQ: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     foo = WordQ(<string>, <word number>, <separator characters>, <quote characters>)            */
   /*                                                                                                 */
   /*  The last two parameters are optional, so it will work                                          */
   /*  as a replacement for the Rexx Word() function.                                                 */
   /* ----------------------------------------------------------------------------------------------- */
   /*  A better Word() function--                                                                     */
   /*  It recognizes a quoted string as one word                                                      */
   /*  Optional parameters let you decide which characters                                            */
   /*  count as word separators and as quotes.                                                        */
   /*                                                                                                 */
   /*  Default word separators are <space> and <tab>                                                  */
   /*  Default quote characters are single, double, and                                               */
   /*  back quotes-- ' " `                                                                            */
   /* =============================================================================================== */

   parse arg str , thisnum , chrsW , chrsQ
   if chrsW == '' then chrsW = ' '||d2c(9)
   if chrsQ == '' then chrsQ = '"'||"'"||'`'
   i         = 1
   quote     = ''
   numwords  = 0
   Thisword  = ''
   w1        = 0
   w2        = 0

   /* find start of first word by trimming blanks and word characters                                 */
   do while i <= length(str) ,
          & pos(substr(str, i, 1), chrsW) > 0
      i = i +1
   end
   if i <= length(str) then numwords = numwords +1
   if numwords == thisnum then w1 = i

   do while (i <= length(str))
      if pos(substr(str, i, 1), chrsQ) \= 0 then do
            /* start of quoted string;
             nothing counts until we get to the matching quote (or end of string) */
            quote = substr(str, i, 1)
            tsti = i +1
            /*  bugfix for "-opt=''"                                                                  */
            do while tsti < length(str) ,
                   & substr(str, tsti, 1) \= quote
               tsti = tsti +1
            end
            if substr(str, tsti, 1) == quote then
               /* found the matching quote                                                            */
               i = tsti
            else
               /* ran out of string with no closing quote, so ignore this quote                       */
               nop
            i = i +1
         end   /*  quote  */

      if pos(substr(str, i, 1), chrsW) \= 0 ,
       & i <= length(str) then do
            /* a word separator, skip all blanks, find start of next word                             */
            /* unless the end of the word we want, then quit                                          */
            if w1 > 0 then do
                  w2 = i -1
                  leave
               end
            do while (pos(substr(str, i, 1), chrsW) \= 0) ,
                   & (i <= length(str))
               i = i +1
            end
            if i <= length(str) then numwords = numwords +1
            /* if this is the start of the word we want, note position in string                      */
            if numwords == thisnum then w1 = i
         end   /*  wordsep  */
      else
         /* a normal character                                                                        */
         i = i +1
   end   /*  do i  */
   if i > length(str) then i = length(str)
   if w1 > 0 then do
         /* we know w1, if we ran out of string, w2 is the end of string                              */
         if w2 == 0 then w2 = i
         /* found start and end of word, copy it                                                      */
         wlen = w2 - w1 +1
         Thisword = substr(str, w1, wlen)
      end
   return Thisword
   /* === End of WordQ ============================================================================== */

   WordsQ: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     foo = WordsQ(string, <separator characters>, <quote characters>)                            */
   /*                                                                                                 */
   /*  The last two parameters are optional, so it will work                                          */
   /*  as a drop-in replacement for the Rexx Words() function.                                        */
   /* ----------------------------------------------------------------------------------------------- */
   /*  A better Words() function--                                                                    */
   /*  It recognizes a quoted string as one word                                                      */
   /*  Optional parameters let you decide which characters                                            */
   /*  count as word separators and as quotes.                                                        */
   /*                                                                                                 */
   /*  Default word separators are <space> and <tab>                                                  */
   /*  Default quote characters are single, double, and                                               */
   /*  back quotes-- ' " `                                                                            */
   /* =============================================================================================== */

   parse arg str , chrsW , chrsQ
   if chrsW == '' then chrsW = ' '||d2c(9)
   if chrsQ == '' then chrsQ = '"'||"'"||'`'
   i         = 1
   quote     = ''
   numwords  = 0

   /* find start of first word by trimming blanks and word characters                                 */
   do while i <= length(str) ,
          & pos(substr(str, i, 1), chrsW) > 0
      i = i +1
   end
   if i <= length(str) then numwords = numwords +1

   do while (i <= length(str))
      if pos(substr(str, i, 1), chrsQ) \= 0 then do
            /* start of quoted string;
             nothing counts until we get to the matching quote (or end of string) */
            quote = substr(str, i, 1)
            tsti = i +1
            /*  bugfix for "-opt=''"                                                                  */
            do while tsti < length(str) ,
                   & substr(str, tsti, 1) \= quote
               tsti = tsti +1
            end
            if substr(str, tsti, 1) == quote then
               /* found the matching quote                                                            */
               i = tsti
            else
               /* ran out of string with no closing quote, so ignore this quote                       */
               nop
            i = i +1
         end   /*  quote  */

      if pos(substr(str, i, 1), chrsW) \= 0 ,
       & i <= length(str) then do
            /* a word separator, skip all blanks, find start of next word                             */
            do while (pos(substr(str, i, 1), chrsW) \= 0) ,
                   & (i <= length(str))
               i = i +1
            end
            if i <= length(str) then numwords = numwords +1
         end
      else
         /* a normal character                                                                        */
         i = i +1
   end   /*  do i  */
   return numwords
   /* === End of WordsQ ============================================================================= */


