/**/

   /***************************************************************************************************/
   /*     RexxTool.cmd   an indenter / formatter / analyzer / etc for Rexx                            */
   /*     Anton Monroe   <amonroe5301@gmail.com>  August 2025                                         */
   /*                                                                                                 */
   /*     see "RexxTool.cmd help" for how to use it                                                   */
   /*                                                                                                 */
   /*  If you see anything here that you can use in your own work, feel free to copy,                 */
   /*  adapt, use, or abuse it. The last section of this file has some procedures that                */
   /*  might be worth looking at. In fact, I copied a couple of them from other                       */
   /*  people. Of course, it is polite to give credit.                                                */
   /***************************************************************************************************/

   /***************************************************************************************************/
   /*  the initialization is divided into two parts-- Init, at the start of the                       */
   /*  program, and Setup, at the start of a script. The result is that practically                   */
   /*  all of the variables can be customized.                                                        */
   /*                                                                                                 */
   /*  The work flow looks something like this:                                                       */
   /*                                                                                                 */
   /*      Main                                                                                       */
   /*        |__Init                  Detect OS, interpreter, set global variables,                   */
   /*        |                        set default values for switches                                 */
   /*        |                        parse command line to validate syntax                           */
   /*        |                                                                                        */
   /*        |__Style_<name>          (if specified on command line) Customize various things         */
   /*        |                        by making changes to variables set in Init                      */
   /*        |                                                                                        */
   /*        |__Script_<name>         Can change some default values if it needs to.                  */
   /*             |                   (which will override Init and Style)                            */
   /*             |                                                                                   */
   /*             |__Setup            Define more variables, based on Init, Style, and Script.        */
   /*             |                   Then re-read command-line switches, which will                  */
   /*             |                   override any values set before this point.                      */
   /*             |                   Setup also reads the file into the SrcIn array.                 */
   /*             |                                                                                   */
   /*             |__Task_A           Tasks do the work and save their results to SrcOut array        */
   /*             |    |                                                                              */
   /*             |    |__task subroutines, sometimes                                                 */
   /*             |    |__low-level tool procedures                                                   */
   /*             |                                                                                   */
   /*             |__Task_B                                                                           */
   /*             |    |__ ...                                                                        */
   /*             |                                                                                   */
   /*             .                                                                                   */
   /*             .                                                                                   */
   /*             .                                                                                   */
   /*             |__ArrayToFile      Send contents of SrcOut array to screen or file.                */
   /*             |                                                                                   */
   /*             |__Egress           All exits go through Egress.                                    */
   /*                                                                                                 */
   /* =============================================================================================== */

   /*
   */

   /* == Main level ================================================================================= */ /*fold00*/
   /* =============================================================================================== */
   /*  decide which script to call, call it, and quit                                                 */
   /* =============================================================================================== */
   parse arg MainArgs
   signal on Syntax
   signal on NoValue
   signal on Error
   signal on Failure name Error

   call Init

   Command = translate(G.0Command)
   if Opt.0Style \== '' then
      if wordpos(translate(Opt.0Style), translate(G.0Styles)) > 0 then
         interpret 'call' 'Style_'||Opt.0Style
      else do
            call Msg 'unknown style "'Opt.0Style'"'
            call Egress 'USAGE'
         end

   /*  the scripts do not return, they exit by calling Egress                                         */
   if Command == 'FORMAT' then call Script_Format
   if Command == 'CHANGE'  then call Script_Change
   if Command == 'DECONTROL'  then call Script_Decontrol
   if Command == 'EXTRACT'  then call Script_Extract
   if Command == 'ADDCOUNTER' then call Script_AddCounter

   if Command == 'VARIABLES'  then call Script_Variables
   if Command == 'DEPENDS'  then call Script_Depends
   if Command == 'CALLS'  then call Script_Calls
   if Command == 'UNUSED'  then call Script_Unused
   if Command == 'TREE'  then call Script_Tree
   if Command == 'CHECKONLY' then call Script_CheckOnly

   if Command == 'TEST' then call Script_Test
   if Command == 'TOKERONLY' then call Script_TokerOnly
   if Command == 'USAGE' then call Script_Usage
   if Command == 'DEVHELP' then call Script_DevHelp
   if Command == 'HELP' then call Script_Help

   /*  this lets the user type "RexxTool.cmd -config" without a command                               */
   if Opt.0config then call Script_Format

   call Msg 'unrecognized command "'G.0Command'"'
   call Egress 'USAGE'

   /* =============================================================================================== */
   Init: /*fold00*/
   /* =============================================================================================== */
   /*  initializes a lot of variables                                                                 */
   /* =============================================================================================== */
   /*  Initialization has been split off into subroutines to make things manageable                   */
   call Init_Global
   call Init_DefineBIFs
   /*  Init_Config has the descriptions and default values of all the command line options            */
   call Init_Config
   call Init_CommandLine
   return 0
   /* === End of Init =============================================================================== */

   Init_Global: /*FOLD00*/
   /* =============================================================================================== */
   /*  define constants and shared variables                                                          */
   /* =============================================================================================== */
   /*  the G. stem is a set of global variables that are available wherever needed, so I do           */
   /*  not need to add to Shared_vars every time I invent a new global variable.                      */
   /*  Granted, having to type 'G.0' every time I use a variable is annoying. The                     */
   /*  advantage of it is that EXPOSEing a long list of variables means it takes more                 */
   /*  time to call a procedure. The program runs faster if most of the global                        */
   /*  variables are combined into one stem.                                                          */
   G. = ''
   parse source G.0OpSystem . G.0me
   parse upper version G.0Interpreter .
   if abbrev(G.0Interpreter, 'REXX-REGINA') then
      G.0Interpreter = 'REGINA'
   select
      when G.0Interpreter == 'REXXSAA' then do
            if RxFuncQuery('SysLoadFuncs') then do
                  call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
                  call SysLoadFuncs
               end
            /*  the DumpVars routine will use RXU.DLL if it is available.                             */
            /*  If you do not have RXU.DLL installed, you could just comment out these lines.         */
            if RxFuncQuery('RxVList') then do
                  call RxFuncAdd 'RxuInit', 'RXU', 'RxuInit'
                  call RxuInit
               end
         end
      when G.0Interpreter == 'REGINA' then do
            call RxFuncAdd 'SysLoadFuncs', 'regutil', 'SysLoadFuncs'
            call SysLoadFuncs
         end
      otherwise
         call Msg 'unknown Rexx interpreter "'G.0Interpreter'"'
         call Msg 'I expected REXXSAA or REGINA'
         call Egress 'USAGE'
   end
   /* ----------------------------------------------------------------------------------------------- */
   /*  These can be used for showing messages. 'G.0me' is the name of this file. When                 */
   /*  a Rexx program is invoked from the command line, the name is in ALL CAPS, which                */
   /*  is hard to read.  To get the correct capitalization of it, the only way I have                 */
   /*  found is a cumbersome call to SysFileTree(). And even that only corrects the                   */
   /*  filename, not the path.                                                                        */
   parse source . . G.0me
   call SysFileTree G.0me, 'tmp.', 'FO'
   G.0me_long = tmp.1
   drop tmp.
   G.0me = filespec('name', G.0me_long)
   G.0meID = '['G.0me']'
   G.0me_ind = copies(' ', length(G.0me))
   G.0meID_ind = copies(' ', length(G.0meID))
   G.0Tab = '    '

   G.0version = '1.4.3'
   if G.0OpSystem == 'UNIX' then do
         G.0CRLF = '0A'x
         G.0DirSep = '/'
         G.0PathSep = ':'
      end
   else do
         G.0CRLF = '0D0A'x
         G.0DirSep = '\'
         G.0PathSep = ';'
      end
   /*  a subroutine "End of" comment starts with this                                                 */
   G.0footerID = ' === End of '
   /*  never use the literal comment strings in a Rexx file                                           */
   /*  also, the '||' is apparently not legal in a parse template                                     */
   G.0CommentBegin = '/'||'*'
   G.0CommentEnd = '*'||'/'
   parse value SysTextScreenSize() with G.0ScreenRows G.0ScreenCols
   /*  G.0sep is sometimes used as a field separator in strings                                       */
   G.0sep = '1F'x
   G.0ControlChars = xrange('00'x, '08'x)||xrange('10'x, '1F'x)||'7F'x
   G.0OP_math = '+ - * / // % **'
   /*  Some functions will set G.0fatalerror if they have a serious problem                           */
   G.0fatalerror = 0
   /*  see if this is useful; a flag to show that a task made some change to the file                 */
   G.0FileChanged = 0
   /*  Toker will set this to 1 if the file has any procedures                                        */
   G.0HasProcedures = 0

   /* ----------------------------------------------------------------------------------------------- */
   /*  set some other variables                                                                       */
   /* ----------------------------------------------------------------------------------------------- */
   /*  most procedures will need to use "expose (Shared_vars)"                                        */
   Shared_vars = 'sigl Opt. G. debug. SrcIn.'
   /*  all_tokens is a list of all the t_xxx. arrays that describe the tokens.                        */
   /*  anything that uses tokens must expose them.                                                    */
   all_tokens = 't_class. t_type. t_prefix. t_val. t_line. t_col. t_indent. t_readonly.'

   /*  G.0token_vars tells the insert/delete functions which token arrays to act on.                  */
   /*  Since inserting/deleting tokens is so slow, only maintain the arrays that are                  */
   /*  needed by the tasks. The t_col. array is not included, because it is only                      */
   /*  used at the very beginning. The t_line array is included because the line number               */
   /*  of a token is sometimes used in error messages.                                                */
   /*  If you create another token array, remember to add it to the lists                             */
   G.0token_vars = 't_class. t_type. t_prefix. t_val. t_line.'
   /*    G.0track_vars is a temporary variable I can use for debugging purposes                       */
   G.0track_vars = ''
   return 0
   /* === End of Init_Global ======================================================================== */

   Init_DefineBIFs: /*fold00*/
   /* =============================================================================================== */
   /*  define lists of Built-In-Functions and keywords                                                */
   /*  Setup_Dialect will decide what combination of these to use when creating G.0AllBIFs.           */
   /*                                                                                                 */
   /*  If you want to add a set of BIFs from another library, this is the place to define it.         */
   /*  The name must be in the form "BIFs.0<library name>". For instance, BIFs.0vrexx from VREXX.DLL  */
   /*  Be sure to also include the name of the library in BIFs.0OS2Libraries                          */
   /*                                                                                                 */
   /*  I experimented with pruning the BIF list to only include libraries that were                   */
   /*  actually loaded with RxFuncAdd, but it did not save any significant amount of time.            */
   /*  And this way the Depends command gives a better idea of what libraries are used                */
   /*  when the program does not explicitly load a library.                                           */
   /*                                                                                                 */
   /*  You could also customize these in a script. See Style_akm for an example.                      */
   /* =============================================================================================== */

   BIFs.0OS2Libraries = 'Rexx RexxUtil RXU RxSock VRexx RxUtilex RxUls Rmxlib RexxINI PR1util' ,
                        'RxUnlock RxLVM WPTools RwINI'

   /*  BIFs included in OS/2 Rexx                                                                     */
   BIFs.0rexx = 'Abbrev Abs Address Arg B2X Beep BitAnd BitOr BitXor C2D C2X Center' ,
                'Centre CharIn CharOut Chars Compare Condition Copies D2C D2X DataType' ,
                'Date dbAdjust dbBracket dbCenter dbCJustify dbLeft dbRight dbRLeft' ,
                'dbRRight dbToDBCS dbToSBCS dbUnBracket dbValidate dbWidth DelStr DelWord' ,
                'Digits Directory EndLocal ErrorText Filespec Form Format Fuzz Insert' ,
                'LastPos Left Length LineIn LineOut Lines Max Min Overlay Pos Queued' ,
                'Random Reverse Right RxFuncAdd RxFuncDrop RxFuncQuery RxQueue SetLocal' ,
                'Sign SourceLine Space Stream Strip SubStr SubWord Symbol Time Trace' ,
                'Translate Trunc Value Verify Word WordIndex WordLength WordPos Words X2B' ,
                'X2C X2D XRange'

   /*  BIFs from OS/2 REXXUTIL.DLL                                                                    */
   BIFs.0rexxutil = 'RxMessageBox SysAddFileHandle SysAddRexxMacro SysBootDrive' ,
                    'SysClearRexxMacroSpace SysCloseEventSem SysCloseMutexSem SysCLS' ,
                    'SysCopyObject SysCreateEventSem SysCreateMutexSem SysCreateObject' ,
                    'SysCreateShadow SysCurPos SysCurState SysDeregisterObjectClass' ,
                    'SysDestroyObject SysDriveInfo SysDriveMap SysDropFuncs SysDropFuncs' ,
                    'SysDropRexxMacro SysDumpVariables SysElapsedTime SysFileDelete' ,
                    'SysFileSearch SysFileSystemType SysFileTree SysGetCollate SysGetEA' ,
                    'SysGetFileDateTime SysGetKey SysGetMessage SysIni SysLoadFuncs' ,
                    'SysLoadRexxMacroSpace SysMapCase SysMkDir SysMoveObject' ,
                    'SysNationalLanguageCompare SysOpenEventSem SysOpenMutexSem SysOpenObject' ,
                    'SysOS2Ver SysPostEventSem SysProcessType SysPutEA SysQueryClassList' ,
                    'SysQueryEAList SysQueryExtLIBPATH SysQueryProcessCodePage' ,
                    'SysQueryRexxMacro SysQuerySwitchList SysRegisterObjectClass' ,
                    'SysReleaseMutexSem SysReorderRexxMacro SysRequestMutexSem' ,
                    'SysResetEventSem SysRmDir SysSaveObject SysSaveRexxMacroSpace' ,
                    'SysSearchPath SysSetExtLIBPATH SysSetFileDateTime SysSetFileHandle' ,
                    'SysSetIcon SysSetObjectData SysSetPriority SysSetProcessCodePage' ,
                    'SysShutDownSystem SysSleep SysStemCopy SysStemDelete SysStemInsert' ,
                    'SysStemSort SysSwitchSession SysTempFilename SysTextScreenRead' ,
                    'SysTextScreenSize SysUtilVersion SysVersion SysWaitEventSem' ,
                    'SysWaitForShell SysWaitNamedPipe SysWildCard'

   /*  BIFs from RXU.DLL for OS/2                                                                     */
   /*  list was adapted from Doug Rickman's CodeAnalyzer.cmd                                          */
   BIFs.0rxu = 'RxAdd2Ptr RxAddMacro RxAddMuxWaitSem RxAddQueue RxAllocMem' ,
               'RxAllocSharedMem RxC2F RxCallEntryPoint RxCallFuncAddress RxCallInStore' ,
               'RxCallProcAddr RxClearMacroSpace RxCloseEventSem RxCloseH' ,
               'RxCloseMutexSem RxCloseMuxWaitSem RxCloseQueue RxConnectNPipe' ,
               'RxCreateEventSem RxCreateMutexSem RxCreateMuxWaitSem RxCreateNPipe' ,
               'RxCreatePipe RxCreateQueue RxCreateRexxThread RxCreateThread' ,
               'RxDeleteMuxWaitSem RxDeregisterExit RxDestroyPipe RxDetachRexxPgm' ,
               'RxDevConfig RxDevIOCtl RxDisConnectNPipe RxDosRead RxDosWrite' ,
               'RxDropMacro RxDupHandle RxExecI RxExecO RxExecPgm RxExitList RxF2C' ,
               'RxFree RxFreeMem RxFreeModule RxGetInfoBlocks RxGetNamedSharedMem' ,
               'RxGetSharedMem RxGiveSharedMem RxGlobalVar RxKbdCharIn RxKillProcess' ,
               'RxKillThread RxLineInH RxLoadMacroSpace RxLoadModule RxMalloc' ,
               'RxNbSessionStatus RxNet RxOpen RxOpenEventSem RxOpenMutexSem' ,
               'RxOpenMuxWaitSem RxOpenQueue RxPBNBufSize RxPassByName RxPeekQueue' ,
               'RxPhysicalDisk RxPmPrintf RxPostEventSem RxProcId RxPullQueue' ,
               'RxPurgeQueue RxQExists RxQProcStatus RxQueryAppType RxQueryEventSem' ,
               'RxQueryExit RxQueryExtLibPath RxQueryFHState RxQueryMacro RxQueryMem' ,
               'RxQueryModuleHandle RxQueryModuleName RxQueryMutexSem RxQueryMuxWaitSem' ,
               'RxQueryProcAddr RxQueryProcType RxQueryQueue RxQuerySysInfo RxQueued' ,
               'RxRSi2F RxRead RxReadQueue RxReadQueueStr RxRegisterExitDll' ,
               'RxRegisterExitExe RxRegisterFuncAddress RxRegisterFunctionExe' ,
               'RxReleaseMutexSem RxReorderMacro RxReplaceModule RxRequestMutexSem' ,
               'RxResetEventSem RxResumeThread RxReturnByName RxRsoe2f RxRsoe2q' ,
               'RxSaveMacroSpace RxScount RxSearchPath RxSetError RxSetExceptionExit' ,
               'RxSetExtLibPath RxSetFHState RxSetMaxFH RxSetMem RxSetNPHState' ,
               'RxSetPriority RxSetRelMaxFH RxSi2H RxSoSe2H RxStartRexxSession' ,
               'RxStartSession RxStem2Struct RxStorage RxStruct2Stem RxStructMap' ,
               'RxSubAllocMem RxSubFreeMem RxSubSetMem RxSubUnsetMem RxSuspendThread' ,
               'RxThunkAddr RxTmrQueryFreq RxTmrQueryTime RxTokenize RxUpm RxVioEndPopUp' ,
               'RxVioPopUp RxVioWrtCharStrAtt RxVlist RxWaitChild RxWaitEventSem' ,
               'RxWaitMuxWaitSem RxWinDestroyObject RxWinQueryObject RxWinSetPresParam' ,
               'RxWinSetSelf RxWrite RxWriteQueue RxuMthacos RxuMthasin RxuMthatan' ,
               'RxuMthatan2 RxuMthceil RxuMthcos RxuMthcosh RxuMtherf RxuMtherfc' ,
               'RxuMthexp RxuMthfabs RxuMthfloor RxuMthfmod RxuMthfrexp RxuMthgamma' ,
               'RxuMthhypot RxuMthldexp RxuMthlog RxuMthlog10 RxuMthmodf RxuMthpow' ,
               'RxuMthsin RxuMthsinh RxuMthsqrt RxuMthtan RxuMthtanh RxuQuery RxuTerm' ,
               'RxuInit'

   /*  BIFs from RxSock.dll                                                                           */
   BIFs.0rxsock = 'SockAccept SockBind SockClose SockConnect SockDropFuncs' ,
                  'SockGetHostByAddr SockGetHostByName SockGetHostID SockGetPeerName' ,
                  'SockGetSockName SockGetSockOpt SockInit SockIOctl SockListen' ,
                  'SockLoadFuncs SockPsock_ErrNo SockRecv SockRecvFrom SockSelect SockSend' ,
                  'SockSendTo SockSetSockOpt SockShutdown SockSocket SockSock_ErrNo' ,
                  'SockSoClose SockVersion'

   /*  BIFs from VREXX.DLL                                                                            */
   BIFs.0vrexx = 'VArc VBackcolor VCheckBox VClearWindow VCloseWindow VColorBox' ,
                 'VDialogPos VDraw VDrawParms VExit VFileBox VFontBox VForeColor' ,
                 'VGetVersion VInit VInputBox VListBox VMsgBox VMultBox VOpenWindow' ,
                 'VRadioBox VResize VSay VSetFont VSetTitle VTableBox'

   /*  BIFs from RXUTILEX.DLL                                                                         */
   BIFs.0rxutilex = 'Sys2BytesRemaining Sys2CheckNamedPipe Sys2Close Sys2ConnectNamedPipe' ,
                    'Sys2CreateNamedPipe Sys2DisconnectNamedPipe Sys2DropFuncs' ,
                    'Sys2FormatNumber Sys2FormatTime Sys2GetClipboardText Sys2GetEpochTime' ,
                    'Sys2KillProcess Sys2LoadFuncs Sys2LocateDLL Sys2Open Sys2PutClipboardText' ,
                    'Sys2QueryDriveInfo Sys2QueryForegroundProcess Sys2QueryPhysicalMemory' ,
                    'Sys2QueryProcess Sys2QueryProcessList Sys2QuerySysValue Sys2Read' ,
                    'Sys2ReadLine Sys2ReplaceModule Sys2Seek Sys2SyncBuffer Sys2Version' ,
                    'Sys2Write'

   /*  BIFs from RXULS.DLL (Unicode functions)                                                        */
   BIFs.0rxuls = 'ULSLoadFuncs ULSDropFuncs ULSConvertCodePage ULSCountryLocale ULSFindAttr' ,
                 'ULSFormatTime ULSGetLocales ULSGetUnicodeClipboard ULSPutUnicodeClipboard' ,
                 'ULSQueryAttr ULSQueryLocaleItem ULSTransform ULSVersion'

   /*  BIFs from RMXLIB.DLL and the older names from previous version called REXXLIB.DLL              */
   BIFs.0rmxlib = 'RMxLibQuerySwitchList RMxLibCopy RMxLibStartProgramme RMxLibDelete' ,
                  'RMxLibCloseObject RMxLibDelTree RMxLibMakePath RMxLibSearchPath' ,
                  'RexxLibQuerySwitchList RexxLibCopy RexxLibStartProgramme RexxLibDelete' ,
                  'RexxLibCloseObject RexxLibDelTree RexxLibMakePath RexxLibSearchPath'

   /*  BIFs from REXXINI.DLL                                                                          */
   BIFs.0rexxini = 'IniLoadFuncs IniDropFuncs IniLoad IniEnum IniGet IniDel' ,
                   'IniSet IniEnumSections IniOpen IniClose IniSave IniCopy'

   /*  BIFs from PR1UTIL.DLL                                                                          */
   BIFs.0pr1util = 'PRLoadFuncs PRDropFuncs PRCloseWindow PRGetFileAttr PRGetGMTOffset' ,
                   'PRGetMXAddress PRGetPid PRKillProcess PRProcessList PRQueryIFConfig' ,
                   'PRQuerySwitchList PRQuitWindow PRReplaceModule PRReplaceObjectClass' ,
                   'PRSetFileAttr PRStatSockets PRSwitchToProgram PRVersion'

   /*  BIFs from RXUNLOCK.DLL                                                                         */
   BIFs.0rxunlock = 'RxUnlockLoadFuncs RxUnlockDropFuncs RxUnlockDebug RxUnlockFiles' ,
                    'RxReplaceModule RxBldLevel RxMD5 RxCRC32 RxFileCompare RxCpuId' ,
                    'RxStemSearch3 RxRandomize RxRandom RxExecutableType'

   /*  BIFs from RXLVM.DLL                                                                            */
   BIFs.0rxlvm = 'RxLVMBootMgrInfo RxLVMBootMgrMenu RxLVMDropFuncs RxLVMEngineClose' ,
                 'RxLVMEngineOpen RxLVMGetDisks RxLVMGetPartitions RxLVMGetVolumes' ,
                 'RxLVMLoadFuncs RxLVMRediscoverPRM RxLVMVersion'

   /*     BIFs from WPTOOLS.DLL                                                                       */
   BIFs.0wptools = 'WPToolsLoadFuncs WPToolsQueryObject WPToolsFolderContent' ,
                   'WPToolsSetObjectData WPToolsVersion'

   /*     BIFs from RWINI.DLL                                                                         */
   BIFs.0rwini = 'rwIniLoadFuncs rwIniDropFuncs rwIniVersion rwIniOpen rwIniClose rwIni'

   /* ----------------------------------------------------------------------------------------------- */
   /*  BIFs included in Regina                                                                        */
   BIFs.0regina = 'Abbrev Abs Address Arg B2X Beep BitAnd BitChg BitClr BitComp BitOr' ,
                  'BitSet BitTst BitXor BufType C2B C2D C2X CD Center Centre ChangeStr' ,
                  'CharIn CharOut Chars ChDir Close Compare Compress Condition Copies' ,
                  'CountStr Crypt D2C D2X DataType Date DelStr DelWord DesBuf Digits' ,
                  'Directory DropBuf EOF ErrorText Exists Export Filespec Find Fork Form' ,
                  'Format FreeSpace Fuzz GetCaller GetCallStack GetEnv GetPID GetSpace' ,
                  'GetTID Hash Import Index Insert Justify LastPos Left Length LineIn' ,
                  'LineOut Lines Lower MakeBuf Max Min Open Overlay PoolID Popen Pos PutEnv' ,
                  'Qualify Queued Random Randu ReadCh ReadLn Reverse Right RxFuncAdd' ,
                  'RxFuncDrop RxFuncErrMsg RxFuncQuery RxQueue Seek Show Sign Sleep' ,
                  'SourceLine Space State Storage Stream Strip SubStr SubWord Symbol Time' ,
                  'Trace Translate Trim Trunc Uname UnixError Upper UserID Value Verify' ,
                  'Word WordIndex WordLength WordPos Words WriteCh WriteLn X2B X2C X2D' ,
                  'XRange'

   /*     RegUtil BIFs (from Regina)                                                                  */
   /*  some are not available on all platforms                                                        */
   BIFs.0regutil = 'RegMultiStemSort RegStemDoOver RegStemRead RegStemSearch RegStemWrite' ,
                   'RxMessageBox RxWinExec SysAddRexxMacro SysArcCos SysArcSin SysArcTan' ,
                   'SysBootDrive SysClearRexxMacroSpace SysCloseEventSem SysCloseMutexSem' ,
                   'SysCls SysCopyObject SysCos SysCosH SysCotan SysCreateEventSem' ,
                   'SysCreateMutexSem SysCreateShadow SysCurPos SysCurState SysDriveInfo' ,
                   'SysDriveMap SysDropFuncs SysDropRexxMacro SysDumpVariables SysExp' ,
                   'SysFileDelete SysFileSearch SysFileSystemType SysFileTree SysFromUnicode' ,
                   'SysGetErrorText SysGetFileDateTime SysGetKey SysGetLine' ,
                   'SysGetLineHistory SysHomeDirectory SysIni SysLinVer SysLoadFuncs' ,
                   'SysLoadRexxMacroSpace SysLog SysLog10 SysMkDir SysMoveObject' ,
                   'SysOpenEventSem SysOpenMutexSem SysOS2Ver SysPi SysPostEventSem SysPower' ,
                   'SysPulseEventSem SysQueryProcess SysQueryRexxMacro SysReleaseMutexSem' ,
                   'SysReorderRexxMacro SysRequestMutexSem SysResetEventSem SysRmDir' ,
                   'SysSaveRexxMacroSpace SysSearchPath SysSetFileDateTime SysSetPriority' ,
                   'SysSin SysSinH SysSleep SysSqrt SysStemCopy SysStemDelete SysStemInsert' ,
                   'SysStemSort SysSwitchSession SysSystemDirectory SysTan SysTanH' ,
                   'SysTempFileName SysTextScreenRead SysTextScreenSize SysToUnicode' ,
                   'SysUtilVersion SysVersion SysVolumeLabel SysWaitEventSem' ,
                   'SysWaitNamedPipe SysWinDecryptFile SysWinEncryptFile SysWinVer'

   /* ----------------------------------------------------------------------------------------------- */
   /*  keywords start a clause:                                                                       */
   /*  This is where you would add keywords for some other dialect of Rexx                            */
   keywords_rexx = 'Address Arg Call Do Drop Exit If Interpret Iterate' ,
                   'Leave Nop Numeric Options Parse Procedure Pull Push Queue Return Say Select' ,
                   'Signal Then Trace Else Otherwise End When'
   /*  sub-keywords:                                                                                  */
   keywords2_rexx = 'Expose Name Off On Upper With To by For Forever Until While Over' ,
                    'Digits Form Scientific Engineering Fuzz Source Var Version' ,
                    'Error Failure Halt NoValue Syntax NotReady'

   keywords_regina = keywords_rexx 'Upper'
   keywords2_regina = keywords2_rexx ,
                      'With Input Normal NoEOL Output Append Replace Error Stream Stem LIFO FIFO'
   return 0
   /* === End of Init_DefineBIFs ==================================================================== */

   Init_Config: /*fold00*/
   /* =============================================================================================== */
   /*  a list of configuration variables that can be changed on the command line                      */
   /* ----------------------------------------------------------------------------------------------- */
   /*  To create a new command-line switch "-xyz" you need to                                         */
   /*     1) add a line to the list in the form                                                       */
   /*             Opt.0xyz    = <default value>    /@  a short description $ <possible values>  @/    */
   /*        note the $ that separates the two parts of the comment                                   */
   /*     2) add 'xyz' to the variable ValidSwitch                                                    */
   /* =============================================================================================== */
   /*  This routine is read by ShowConfig to get a list of all the variable names. It looks           */
   /*  for lines that match                                                                           */
   /*       Opt.0<name> = xxx                                                                         */
   /*  the values listed below are defaults. Showconfig will show the actual values after             */
   /*  the command line switches are read.                                                            */
   /*                                                                                                 */
   /*  Doing it this way means I do not have to maintain a separate list for documentation,           */
   /*  and it lets me show the names with the proper capitalization. The alternative would be         */
   /*  having Rexx dump the variable list as OPT.0WIDTH, OPT.0FORMATCOMMENTS, etc.,                   */
   /*  which is hard to read.                                                                         */
   /*                                                                                                 */
   /*  IMPORTANT: do not change the layout of this unless you understand how                          */
   /*  ShowConfig parses it                                                                           */
   /* =============================================================================================== */
   if translate(arg(1)) = 'SHOW' then do
         call ShowConfig
         call Egress 'OKAY'
      end
   /* ----------------------------------------------------------------------------------------------- */
   /*  $format comment off                                                                            */
   Opt. = ''
   /*  General options:                                                                               */
   Opt.0config          = 0        /*  show this list                                     $ 0 or 1    */
   Opt.0debug           = 'eoc,arg,fatal' /*  debug categories; See "help debug"          $ <list>    */
   Opt.0dialect         = ''       /*  see "help dialect"                     $ '', REXXSAA or REGINA */
   Opt.0help            = ''       /*  -help or help <switch> or help <subject>           $ <string>  */
   Opt.0idle            = 1        /*  use lower (idle) priority                          $ 0 or 1    */
   Opt.0progress        = 1        /*  show progress                                      $ 0 or 1    */
   Opt.0verbose         = 0        /*  verbose error messages                             $ 0 or 1    */
   Opt.0warn            = 1        /*  Tokeniser warns about faulty code                  $ 0 or 1    */
   /*  Formatting options:                                                                            */
   Opt.0AddFooter       = 0        /*  insert footer comment at end of routine            $ 0 or 1    */
   Opt.0AddHeader       = 0        /*  insert empty header at start of routine      $ 0, 1, or ABOVE  */
   Opt.0Bare            = 0        /*  do a bare format only                              $ 0 or 1    */
   Opt.0BIFs            = ''       /*  how to recap Built-In Functions     $ UPPER,LOWER,MIXED, or '' */
   Opt.0EndComment      = 'LEFT'   /*  how to shift comment after END                    $ LEFT or '' */
   Opt.0FoldLine        = 0        /*  break IF statement at logical OPs                  $ 0 or 1    */
   Opt.0Footer          = 1        /*  maintain subroutine footers, if any                $ 0 or 1    */
   Opt.0FormatComments  = 1        /*  format comments                                    $ 0 or 1    */
   Opt.0FTE             = 1        /*  recognize FTE/eFTE comments                        $ 0 or 1    */
   Opt.0Indent          = 1        /*  do indenting task                                  $ 0 or 1    */
   Opt.0IndentContinued = ''       /*  how much to indent continued lines   $ '', SMART, or <number>  */
   Opt.0IndentDivider   = 1        /*  align divider comments with Rexx code              $ 0 or 1    */
   Opt.0IndentEnd       = 0        /*  indent END or align with DO                        $ 0 or 1    */
   Opt.0IndentMulti     = 1        /*  align multi-line comments with Rexx code           $ 0 or 1    */
   Opt.0IndentWhen      = 1        /*  indent WHEN                                        $ 0 or 1    */
   Opt.0Keywords        = ''       /*  how to recap keywords               $ UPPER,LOWER,MIXED, or '' */
   Opt.0Margin          = 3        /*  number of leading spaces (left margin)             $ <number>  */
   Opt.0MoveComment     = ''       /*  how to move right comments       $ RIGHT,LEFT,EXPAND,<col>, '' */
   Opt.0OutdentThenDo   = 0        /*  shift block under THEN DO to left                  $ 0 or 1    */
   Opt.0Procedure       = ''       /*  where to move PROCEDURE                      $ UP, DOWN, or '' */
   Opt.0Quick           = 0        /*  do a quick format only                             $ 0 or 1    */
   Opt.0ReCap           = 1        /*  recapitalize symbols and keywords                  $ 0 or 1    */
   Opt.0ReSpace         = 1        /*  do Respace task                                    $ 0 or 1    */
   Opt.0RexxID          = 1        /*  make sure it starts with a RexxID comment          $ 0 or 1    */
   Opt.0SpaceConcat     = 0        /*  add spaces around "||" operator                 $ 0, 1, or ''  */
   Opt.0SpaceNot        = 0        /*  add space after \                               $ 0, 1, or ''  */
   Opt.0SpaceParenth    = 0        /*  add spaces at ( and )                           $ 0, 1, or ''  */
   Opt.0Spacer          = ''       /*  number of empty lines between routines       $ '' or <number>  */
   Opt.0Style           = ''       /*  built-in formatting style to use             $ '' or <string>  */
   Opt.0Tab             = 3        /*  number of spaces per indent level                  $ <number>  */
   Opt.0Then            = ''       /*  how to move THEN                             $ UP, DOWN, or '' */
   Opt.0ThenDo          = ''       /*  how to move THEN DO                   $ UP, DOWN, SPLIT, or '' */
   Opt.0Unwrap          = 0        /*  unwrap continued lines                             $ 0 or 1    */
   Opt.0Width           = 101      /*  length of lines (plus left margin)                 $ <number>  */
   Opt.0Wrap            = 0        /*  try to wrap long lines to width (incomplete)       $ 0 or 1    */
   /*  Check options:                                                                                 */
   Opt.0Check           = 'ERROR,WARN' /*  what warnings to show                          $ <list>    */
   Opt.0CheckStem       = ''       /*  stem variable to check; See "help stem"        $ '' or <list>  */
   Opt.0Fix             = 0        /*  fix some errors; See "help fix"                    $ 0 or 1    */
   /*  Options for other tasks:                                                                       */
   Opt.0Terse           = 0        /*  give shorter answer                                $ 0 or 1    */
   Opt.0Show            = ''       /*  limits what is shown by some tasks                 $ <list>    */
   Opt.0Flat            = 0        /*  assume no procedures, for some commands            $ 0 or 1    */
   Opt.0Var             = ''       /*  see "help change"                                  $ string    */
   Opt.0Proc            = ''       /*  see "help change"                                  $ string    */
   Opt.0Where           = ''       /*  which procedures to look at, for some commands $ '' or string  */
   Opt.0alt             = 0        /*  use an alternate method for some tasks             $ 0 or 1    */
   Opt.0ascii           = 0        /*  use ASCII characters to draw lines                 $ 0 or 1    */
   /*  $format comment on                                                                             */

   ValidSwitch = 'debug verbose ShowGetOpts' ,
                 'Warn Progress test Idle' ,
                 'Help Version' ,
                 'Dialect' ,
                 'Indent IndentEnd IndentWhen IndentContinued' ,
                 'Width Margin Tab' ,
                 'Check CheckStem' ,
                 'AddFooter AddHeader Footer' ,
                 'Config' ,
                 'FormatComments EndComment MoveComment' ,
                 'IndentDivider IndentMulti' ,
                 'Then ThenDo Procedure' ,
                 'ReCap Keywords BIFs' ,
                 'ReSpace SpaceConcat SpaceNot SpaceParenth Spacer' ,
                 'FoldLine Wrap Fix OutdentThenDo Unwrap' ,
                 'FTE Terse Show Flat Where Var Proc' ,
                 'Quick Bare Style RexxID alt ascii'
   /*  if you add a Style_xxx routine, add "xxx" to this list                                         */
   G.0Styles = 'akm alternate'
   return 0
   /* === End of Init_Config ======================================================================== */

   Init_CommandLine: /*fold00*/
   /* =============================================================================================== */
   /*  interpreting the command line:                                                                 */
   /*  Override default values of Opt.0xxx variables by reading the command line.                     */
   /*  respond to -version switch                                                                     */
   /*  determine name of input file and verify that it exists                                         */
   /*  determine name of output file                                                                  */
   /* =============================================================================================== */
   /*  Usually I just use "interpret GetOpts(MainArgs, ValidSwitch)", but in this case                */
   /*  I want a script to be able to set its own defaults, then re-evaluate the                       */
   /*  command-line switches with                                                                     */
   /*        interpret G.0ConfigString                                                                */
   G.0ConfigString = GetOpts(MainArgs, ValidSwitch)
   interpret G.0ConfigString
   Opt.0Where = translate(Opt.0Where, ' ', ',')
   if Opt.0ShowGetOpts \== '' then do
         call ShowGetOpts
         call Egress 'OKAY'
      end
   if Opt.0errors > 0 then
      /*  the message was already displayed when GetOpts() was interpreted                            */
      call Egress 'USAGE'

   if Opt.0version \= '' then do
         say G.0me G.0version
         call Egress 'OKAY'
      end
   /* ----------------------------------------------------------------------------------------------- */
   do c = 1 to ArgC
      if pos('=', ArgV.c) > 0 then do
            call Msg ' "'ArgV.c'"'
            call Msg '     looks like a mistake. Did you mean'
            call Msg ' "-'ArgV.c'" ?'
            call Egress 'USAGE'
         end
   end c

   if FileExist(ArgV.1) then do
         call Msg '"'MainArgs'"'
         call Msg 'I need a command, like'
         call Msg '   'G.0me 'format' ArgV.1
         call Msg 'or 'G.0me 'tree' ArgV.1
         call Egress 'USAGE'
      end

   G.0Command = ArgV.1
   /*  some newer commands do not use the normal <command> <infile> <outfile> syntax                  */
   /*  so put the clumsy workarounds here                                                             */
   select
      when Opt.0config ,
         | translate(G.0Command) == 'CONFIG' then do
            Opt.0config = 1
            G.0InfileN = G.0me_long
            G.0OutFileN = 'stdout'
         end
      when translate(G.0Command) == 'HELP' then do
            G.0InfileN = G.0me_long
            G.0OutFileN = 'stdout'
            if ArgV.2 \== '' then do
                  Opt.0help = ArgV.2
                  /*  quick hack to allow multi-word help string                                      */
                  /*  (because GetOpts strips quotes)                                                 */
                  if words(Opt.0help) > 1 then
                     Opt.0help = '"'||Opt.0help||'"'
               end
            else
               Opt.0help = 1
         end
      when Opt.0help \== '' ,
         & G.0Command == '' then do
            /*  this allows "-help=format" as an equivalent of the newer "HELP format"                */
            /*  one could argue that "help" should be a command instead of an option                  */
            /*  but many people will expect a program to understand "-help"                           */
            G.0Command = 'HELP'
            G.0InfileN = G.0me_long
            G.0OutFileN = 'stdout'
         end
      when translate(G.0Command) == 'DEVHELP' then do
            /*  this allows "RexxTool.cmd devhelp <where>" as an equivalent of                        */
            /*  "RexxTool.cmd usage RexxTool.cmd -where=<where>"                                      */
            G.0InfileN = G.0me_long
            G.0OutFileN = 'stdout'
            if ArgV.2 \== '' then
               Opt.0Where = ArgV.2
            else
               Opt.0Where = ''
         end
      otherwise
         G.0InfileN = ArgV.2
         G.0OutFileN = ArgV.3
         if G.0OutFileN = '' then G.0OutFileN = 'stdout'
         if ArgV.4 \= '' then do
               list = ''
               do c = 2 to ArgC
                  list = list ArgV.c
               end c
               list = Quoted(strip(list))
               call Msg G.0meID 'too many filenames:' list
               call Egress 'USAGE'
            end
   end   /*  select  */

   if G.0InfileN == '' then do
         call Msg G.0meID 'I need a file name'
         call Egress 'USAGE'
      end   /*  if InfileN == ''  */

   if \FileExist(G.0InfileN) then do
         call Msg G.0meID 'Cannot find source file "'||G.0InfileN||'"'
         call Egress 'USAGE'
      end

   /*  if outfile exists, do not overwrite it, add a suffix                                           */
   if FileExist(G.0OutFileN) then do
         suff = 2
         do while FileExist(G.0OutFileN||'_'||suff)
            suff = suff +1
         end
         NewN = G.0OutFileN||'_'||suff
         call Msg 'file exists:' G.0OutFileN
         call Msg 'changed to: ' NewN
         G.0OutFileN = NewN
      end
   return 0
   /* === End of Init_CommandLine =================================================================== */

   ShowConfig: /*fold00*/
   /* =============================================================================================== */
   /*  Shows the current value of the variables in the Opt. stem                                      */
   /* =============================================================================================== */
   start = sigl
   varN. = ''
   varN.0 = 0
   v = 0
   do ln = start to sourceline()
      text = StripStr(sourceline(ln))
      if left(text, 2) = '/'||'*' then iterate
      if word(translate(text), 1) = 'RETURN' then leave
      /*  apparently a literal '/'||'*' is not allowed in a template                                  */
      parse var text varname '=' . (G.0CommentBegin) descr (G.0CommentEnd) .
      varname = strip(varname)
      if abbrev(translate(varname), 'OPT.0') then do
            v = v +1
            descr = strip(descr)
            varN.v = varname||' '||descr
            varN.0 = v
         end
   end
   call SysStemSort 'varN.', 'a', 'i'
   varNameLen = 20
   varValLen = 25
   possibleLen = 30
   call lineout 'stdout', ,
        G.0Tab||left('switch:', varNameLen -1)||left('value:', varValLen)||left('possible:', possibleLen)||'description:'
   do v = 1 to varN.0
      parse var varN.v varN.v ' ' descr
      varVal = value(varN.v)
      varname = left('  '||'-'||substr(varN.v, 6), varNameLen)||'= '
      varVal =  left("'"||varVal||"'", varValLen)
      parse var descr descr '$' possible
      possible =  left(possible, possibleLen)
      descr = strip(descr, 't')
      call lineout 'stdout', varname||varVal||possible||descr
   end

   call lineout 'stdout', '  Built-in styles are:'
   do i = 1 to words(G.0Styles)
      call lineout 'stdout', G.0Tab||word(G.0Styles, i)
   end i
   call lineout 'stdout', ''
   call lineout 'stdout', '  For more help about each one, type 'G.0me' help <switch>'

   return 0
   /* === End of ShowConfig ========================================================================= */

   Setup: /*fold00*/
   /* =============================================================================================== */
   /*  get ready to start tasks:                                                                      */
   /*     re-interpret command line (to override any default values from the script)                  */
   /*     read the file and verify that it is not compiled or corrupt                                 */
   /*     determine the Rexx dialect, and set some variables based on it                              */
   /*     do a few other things based on command-line switches                                        */
   /* =============================================================================================== */
   Esigl = sigl
   interpret G.0ConfigString
   Opt.0Where = translate(Opt.0Where, ' ', ',')
   if Opt.0Where = 1 then Opt.0Where = ''
   Opt.0debug = translate(Opt.0debug, ' ', ',')
   debug. = 0
   do w = 1 to words(Opt.0debug)
      wordw = translate(word(Opt.0debug, w))
      if wordw == 'ALL' then do
            debug. = 1
            debug.0fatal = 0
         end
      interpret 'debug.0'||wordw '= 1'
   end w
   /*  quick hack to allow multi-word help string                                                     */
   /*  (because GetOpts strips quotes)                                                                */
   if words(Opt.0help) > 1 then
      Opt.0help = '"'||Opt.0help||'"'
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0config then do
         call Init_Config 'SHOW'
         call Egress 'OKAY'
      end
   /* ----------------------------------------------------------------------------------------------- */
   /*  initialize the output array. A task should not initialize it or assume it is empty             */
   SrcOut. = ''
   SrcOut.0 = 0
   /*  read the input file into the SrcIn. array                                                      */
   /*  Check for character 0 that indicates a compiled file. Search the whole file                    */
   /*  because I have seen damaged files with binary junk in the middle.                              */
   SrcIn. = '' ; SrcIn.0 = 0
   if G.0InfileN \= '' then do
         call FileToArray 'SrcIn.', G.0InfileN
         do ln = 1 to SrcIn.0
            if pos('00'x, SrcIn.ln) > 0 then do
                  call Msg 'character '00'x at line 'ln'; file is compiled or corrupt'
                  call Egress 'NotRexx'
               end
            /*  StripStr() strips <Space> and <Tab>                                                   */
            SrcIn.ln = StripStr(SrcIn.ln, 't')
         end
         drop ln
      end   /*  if G.0InfileN \= ''  */
   /* ----------------------------------------------------------------------------------------------- */
   call Setup_Dialect
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0idle then do
         if G.0Interpreter == 'REGINA' then do
               nop
               /*  fix: Regina/RegUtil apparently uses a Unix 'nice' level between 20 and -20,        */
               /*  but nothing I have tried actually changes the priority, either on OS/2 or Linux    */
               /*  It doesn't matter much-- Regina does not hog the CPU like OS/2 Rexx does           */
            end
         else do
               /*  akm: this is the highest priority that doesn't slow down my text editor            */
               /*  in another window. You could also try                                              */
               /*        SysSetPriority(0, -31)                                                       */
               ret = SysSetPriority(1, 31)
               if ret \= 0 then call Mark 'ret', 'from SysSetPriority, OS/2'
            end
      end
   /* ----------------------------------------------------------------------------------------------- */
   return 0
   /* === End of Setup ============================================================================== */

   Setup_Dialect: /*fold00*/
   /* =============================================================================================== */
   /*  if a file has been read into SrcIn, Look at it to guess which dialect of Rexx it is for.       */
   /*  Otherwise assume it is the same as the interpreter being used                                  */
   /*  The '-dialect=' switch forces RexxTool.cmd to use that dialect regardless                      */
   /*  of what the file seems to be.                                                                  */
   /* =============================================================================================== */
   G.0dialect = ''
   select
      when Opt.0dialect \= '' then do
            G.0dialect = translate(Opt.0dialect)
            if G.0dialect = 'REXX' then
               G.0dialect = 'REXXSAA'
         end
      when SrcIn.0 > 0 then do
            /*  bugfix for people who share code snippets that are indented or do not have the        */
            /*  /@ or #! on the first line-- Look at the beginning of the first non-empty line.       */
            /*  We are not verifying that the file is executable, just trying to deduce what          */
            /*  the writer intended.                                                                  */
            do ln = 1 to SrcIn.0
               if StripStr(SrcIn.ln) \== '' then leave ln
            end

            FirstLine = StripStr(SrcIn.ln)
            if G.0dialect == '' then do
                  select
                     when left(FirstLine, 2) == '/'||'*' then
                        G.0dialect = G.0Interpreter
                     when left(FirstLine, 2) == '#!' then do
                           Proc = translate(strip(substr(FirstLine, 3)))
                           parse var Proc Proc rest .
                           Proc = WordQ(Proc, WordsQ(Proc, '/\'), '/\')
                           select
                              when Proc == 'REXX' ,
                                 | Proc == 'REXX.EXE' ,
                                 | Proc == 'REGINA' ,
                                 | Proc == 'REGINA.EXE' then
                                 G.0dialect = 'REGINA'
                              when Proc == 'ENV' ,
                                 & rest == 'REGINA' then
                                 /*  bugfix for "#!/usr/bin/env regina"                               */
                                 G.0dialect = 'REGINA'
                              otherwise
                                 G.0dialect = strip(substr(FirstLine, 3))
                           end

                        end
                     when translate(word(FirstLine, 1)) == 'EXTPROC' then do
                           parse upper var FirstLine . Proc .
                           Proc = WordQ(Proc, WordsQ(Proc, '/\'), '/\')
                           if Proc == 'REXX' ,
                            | Proc == 'REXX.EXE' ,
                            | Proc == 'REGINA' ,
                            | Proc == 'REGINA.EXE' then
                              G.0dialect = 'REGINA'
                           else
                              G.0dialect = word(FirstLine, 2)
                           drop Proc
                        end
                     when translate(FileExt(G.0InfileN)) == 'REXX' ,
                        | translate(FileExt(G.0InfileN)) == 'REX' then
                        /*  assume that a filename that ends with 'rex' or 'rexx'                     */
                        /*  is supposed to be some kind of Rexx                                       */
                        G.0dialect = G.0Interpreter
                     when G.0OpSystem == 'OS/2' then
                        if translate(FileExt(G.0InfileN)) == 'CMD' then do
                              /*  a reasonable assumption is that any *.cmd file without              */
                              /*  a starting Rexx comment is intended for cmd.exe (or 4os2.exe)       */
                              G.0dialect = 'cmd.exe'
                           end
                     otherwise nop
                  end   /*  select  */
               end   /*  if Dialect == ''  */
            if G.0dialect == '' then
               G.0dialect = 'UNKNOWN'
         end   /*  when SrcIn.0 > 0  */
      otherwise
         G.0dialect = G.0Interpreter
   end   /*  select  */
   /* ----------------------------------------------------------------------------------------------- */
   /*  the BIFs.0Used variable will be used by Task_Calls/Task_Depends to identify which library      */
   /*  an external BIF belongs to                                                                     */
   select
      when G.0dialect == 'REXXSAA' then do
            G.0AllBIFs = BIFs.0rexx
            tmp = BIFs.0OS2Libraries
            BIFs.0used = BIFs.0OS2Libraries
            do while tmp \= ''
               parse var tmp libname tmp
               !libname = 'BIFS.0'value(libname)
               if symbol(!libname) == 'VAR' then do
                     list = value(!libname)
                     if list \== '' then
                        G.0AllBIFs = G.0AllBIFs list
                  end
            end
            G.0AllBIFs = strip(G.0AllBIFs)
            BIFs.0used = strip(BIFs.0used)
            G.0Keywords = keywords_rexx
            G.0Keywords2 = keywords2_rexx
         end
      when G.0dialect == 'REGINA' then do
            G.0AllBIFs = BIFs.0regina BIFs.0regutil
            BIFs.0used = 'regutil'
            G.0Keywords = keywords_regina
            G.0Keywords2 = keywords2_regina
         end
      otherwise
         call Msg G.0meID 'I do not know what to do with' G.0InfileN
         call Msg G.0meID_ind 'the dialect seems to be "'G.0dialect'"'
         call Msg G.0meID_ind 'If you are sure this is Rexx, use'
         call Msg G.0meID_ind '-dialect=RexxSAA or -dialect=Regina'
         call Egress 'NotRexx'
   end
   return 0
   /* === End of Setup_Dialect ====================================================================== */

   Prep_MarkRO: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Format $Format FTE eFTE readonly                                                         */
   /*                                                                                                 */
   /*    -FTE=1       do not change comments added by the FTE or eFTE folding text editor             */
   /*                 On by default. If you do not use FTE/eFTE, it should be harmless.               */
   /*                                                                                                 */
   /*  About the $FORMAT directive:                                                                   */
   /*                                                                                                 */
   /*  If none of the command-line switches do what you want, you can prevent part of                 */
   /*  a file from being formatted with a comment in the form                                         */
   /*      /@ $FORMAT <class> <state> @/                                                              */
   /*  where <class> is 'comment' 'rexx' or 'all', and <state> is either 'on' or 'off'                */
   /*  "$FORMAT off" is understood as short for "$FORMAT all off".                                    */
   /*  The first word of the comment must be "$FORMAT". "COMMENTS" is the same as "COMMENT".          */
   /*  Case does not matter, and there can be other text after <state>.                               */
   /*                                                                                                 */
   /*  Examples:                                                                                      */
   /*    |  /@  $FORMAT all off               @/                                                      */
   /*    |  /@  $FORMAT comment off           @/                                                      */
   /*    |  /@  $FORMAT rexx on but do not touch my carefully arranged comments            @/         */
   /*                                                                                                 */
   /*  ==> In this context, "formatting" means "changing the spacing or order of words".              */
   /*      It does not control recapitalization of words. For now, the only way to                    */
   /*      turn off recapitalization is to do it globally with the -!ReCap or -ReCap=0 switch.        */
   /*      It also will not prevent changes to variable names by the Change command.                  */
   /* =============================================================================================== */
   /*  a formatting task should check the t_readonly attribute before changing a token:               */
   /*            if \t_readonly.t then                                                                */
   /*               <do something>                                                                    */
   /*                                                                                                 */
   /*  However, an indenting task should be allowed to set the t_indent.N attribute of                */
   /*  token N. It is information that might (conceivably) be needed for something. If                */
   /*  t_readonly.N is True, then t_indent.N will be ignored when applying the                        */
   /*  indenting.                                                                                     */
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0progress then call Progress SubID()

   G.0token_vars = G.0token_vars 't_readonly.'
   t_readonly. = 0 ; t_readonly.0 = t_type.0
   /*  map "ON","OFF" to the t_readonly.t flag. you could allow synonyms like "true" or "yes"         */
   Formatting.on = 0
   Formatting.off = 1
   RO_Rexx = 0
   RO_Comment = 0

   /*  the RexxID is the starting "/@" at the first column of the first line                          */
   /*  Make sure that it has no prefix and mark it readonly.                                          */
   /*  same applies if it is an empty COMMENT_BEG at the start of a #! or EXTPROC line                */
   t = RexxID()
   if t_type.t == 'COMMENT_BEG' then
      t_readonly.t = 1

   do while t < t_type.0
      t = t +1
      /*  set permission for the token:                                                               */
      if t_class.t = 'COMMENT' then
         t_readonly.t = RO_Comment
      else
         t_readonly.t = RO_Rexx

      /*  check comment for new $FORMAT instruction                                                   */
      if abbrev(t_type.t, 'COMMENT_VAL') then do
            Tval = StripStr(translate(t_val.t))
            select
               /*  if a $format instruction, read it                                                  */
               when abbrev(Tval, '$FORMAT ') then do
                     BadFmt = 0
                     parse upper var t_val.t . class  !state  .
                     if class = 'ON' ,
                      | class = 'OFF' then do ; !state = class ; class = 'ALL' ; end
                     /*  allow "comments" as well as "comment"                                        */
                     if class = 'COMMENTS' then class = 'COMMENT'
                     if !state \= 'ON' ,
                      & !state \= 'OFF' then BadFmt = 1
                     if class \= 'COMMENT' ,
                      & class \= 'REXX' ,
                      & class \= 'ALL' then BadFmt = 1

                     if \BadFmt then do
                           if class = 'ALL' ,
                            | class = 'REXX' then
                              RO_Rexx = Formatting.!state
                           if class = 'ALL' ,
                            | class = 'COMMENT' then do
                                 RO_Comment = Formatting.!state
                                 /*  are $FORMAT comments themselves formattable? Let's say No:       */
                                 do s = t -1 to t +1
                                    t_readonly.s = 1
                                 end
                                 /*  skip the rest of this comment                                    */
                                 t = t +1
                              end   /*  end if class  */
                        end   /*  if BadFmt  */
                     else do
                           call Msg G.0meID 'unrecognized $FORMAT instruction at line' t_line.t':'
                           call Msg '   "'||strip(t_val.t)||'"'
                        end
                  end   /*  when $format  */
               otherwise
                  /*  akm: this is for the FTE folding editor that I use; I do not want               */
                  /*  to tamper with its comments, so mark them as ReadOnly.                          */
                  /*  by default FTE and eFTE use "fold00", "fold01", etc.                            */
                  /*  This is on by default; if you do not use FTE/eFTE, it should be harmless.       */
                  if Opt.0FTE then
                     if length(t_val.t) == 6 then
                        if left(translate(t_val.t), 4) == 'FOLD' then
                           if datatype(substr(t_val.t, 5), 'W') then do
                                 do s = t -1 to t +1
                                    t_readonly.s = 1
                                 end
                                 t = t +1
                              end
            end   /*  select  */
         end   /*  end if a COMMENT_VAL  */
      /* -------------------------------------------------------------------------------------------- */
   end   /*  of t loop  */
   if Opt.0progress then call Progress
   return 0
   /* === End of Prep_MarkRO ======================================================================== */

   Prep_UnMargin: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  After the Tokeniser, adjust a multi-line comment to correct the prefixes                       */
   /* ----------------------------------------------------------------------------------------------- */
   /*  The Toker does not, and should not, try to understand formatting. In a comment, it             */
   /*  considers everything between the "/@" and "@/" as the contents of the comment and              */
   /*  stores it in one or more COMMENT_VAL tokens. So in a multi-line comment like this:             */
   /*          |   /@ start a multi-line comment                                                      */
   /*          |     line two                                                                         */
   /*          |   @/                                                                                 */
   /*  the first token of the comment is "/@" with a prefix of "  ";                                  */
   /*  the first COMMENT_VAL token is " start a multi-line comment",                                  */
   /*  but the next COMMENT_VAL token includes the three-space margin plus the indenting:             */
   /*            "     line two"                                                                      */
   /*  instead of                                                                                     */
   /*            "  line two"                                                                         */
   /*  The result is that the middle of a multi-line comment will move farther right every            */
   /*  time it is formatted. We need to preserve the alignment by removing the extra spaces           */
   /*  from the token's value.                                                                        */
   /* =============================================================================================== */
   /*  akm: I wrote most of this long ago and do not entirely understand it. Surely it                */
   /*       could be done more simply. But if it ain't broke, don't fix it.                           */
   /*                                                                                                 */
   /*  fix: I found one combination of comment and column positions that needed a                     */
   /*        second format to get the indent right.                                                   */
   /*        (note from akm to akm-- so why didn't you tell us what that combination is?)             */
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0progress then call Progress SubID()

   do t = 2 to t_type.0 -1
      if t_class.t \= 'COMMENT' then iterate
      ctype = C_CommentType(t)
      select
         when abbrev(t_type.t, 'COMMENT_BEG') then
            if abbrev(ctype, 'MULTI_') then do
                  /*  the beginning of a multi-line comment                                           */
                  /*  bugfix-- instead of using the prefix of COMMENT_BEG as the margin,              */
                  /*  search the comment for the line that is the farthest left;                      */
                  /*  consider that position as the left side of the comment                          */

                  /*  the indent of the RexxID comment is meaningless here                            */
                  if t = RexxID() then indent = 999
                  else indent = t_col.t -1

                  start = t +1
                  do s = start to L_FirstToken(C_OtherEnd(t))
                     /*  also check the class, to exclude line-endings                                */
                     if abbrev(t_class.s, 'COMMENT') then
                        if abbrev(C_CommentType(s), 'MULTI_M', 7) then
                           /*  bugfix-- also exclude empty lines                                      */
                           if t_val.s \== '' then do
                                 thisindent = length(t_val.s) - length(StripStr(t_val.s, 'l'))
                                 indent = min(indent, length(t_val.s) - length(StripStr(t_val.s, 'l')))
                              end
                  end s
                  to_remove = copies(' ', indent)
               end
         when t_type.t = 'COMMENT_END' ,
            & abbrev(ctype, 'MULTI_') then
            drop indent to_remove
         when abbrev(C_CommentType(t), 'MULTI_M', 7) then do
               if t = L_FirstToken(t) then
                  if indent > 0 then
                     if abbrev(t_val.t, to_remove) then do
                           t_val.t = substr(t_val.t, indent +1)
                           t_prefix.t = to_remove
                        end
            end
         otherwise nop
      end   /*  select  */
   end t
   if Opt.0progress then call Progress
   return 0
   /* === End of Prep_UnMargin ====================================================================== */

   Prepare: /*fold00*/
   /* =============================================================================================== */
   /*  This is the standard set of routines to prepare the token arrays before the                    */
   /*  formatting tasks. Most formatting scripts will want to call this.                              */
   /* =============================================================================================== */
   call Prep_UnMargin
   call Prep_MarkRO
   return 0
   /* === End of Prepare ============================================================================ */

   /***************************************************************************************************/
   /*  start Scripts                                                                                  */
   /***************************************************************************************************/

   Script_AddCounter: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: AddCounter Counter count commands                                                        */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*        RexxTool.cmd ADDCOUNTER <filename> <newfile>                                             */
   /*                                                                                                 */
   /*  Inserts code into <newfile> that will count how many times each routine was called.            */
   /*  See instructions at the top of the new file.                                                   */
   /*                                                                                                 */
   /* =============================================================================================== */
   call Setup
   call Task_Tokenize
   call Task_AddCounter
   /*  Task_AddCounter creates the SrcOut array, so do not call Task_Rebuild                          */
   if debug.0token then call SaveTokensF filespec('N', G.0InfileN)||'_Tokens_End'
   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_AddCounter ================================================================== */

   Script_Bare: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Bare                                                                                     */
   /*                                                                                                 */
   /*      RexxTool.cmd FORMAT <filename> [<newfile>] -bare                                           */
   /*                 makes a new file of code only, without comments, empty lines,                   */
   /*                 or indenting.                                                                   */
   /*                 It is probably not very useful                                                  */
   /*                                                                                                 */
   /*  Theoretically a Rexx program will run faster if the comments are removed, but                  */
   /*  removing them from RexxTool.cmd only saves me about 2% of the running time.                    */
   /* =============================================================================================== */
   /*  Task_RebuildBare depends on having a RexxID                                                    */
   Opt.0RexxID = 1
   call Setup

   call Task_Tokenize
   /*   skip Prep_UnMargin because comments will be removed anyway                                    */
   call Prep_MarkRO
   call Task_FixRexxID
   /*   skip Task_Indent, _Indent2, and _ApplyIndent because indenting will be stripped               */
   call Task_ReSpace
   call Task_Recap
   call Task_Then
   call Task_ThenDo

   call Task_RebuildBare
   if debug.0token then call SaveTokensF filespec('N', G.0InfileN)||'_Tokens_End'
   call ArrayToFile 'SrcOut.', G.0OutFileN
   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
         call Egress 'BUG'
      end
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_Bare ======================================================================== */

   Script_Calls: /*fold00*/
   /* =============================================================================================== */
   /*  Help: Call Calls commands where handlers show                                                  */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*       RexxTool.cmd CALLS <filename> [<new filename>] [options]                                  */
   /*             For each routine, show all of the routines that it calls and is called by           */
   /*                                                                                                 */
   /*  Options:                                                                                       */
   /*       -where=<list>                                                                             */
   /*             Show calls only for labels in <list>                                                */
   /*             list must be in quotes or comma-separated without spaces                            */
   /*             labels may use ? and * as wild characters                                           */
   /*             "+handlers" will be expanded to the list of routines that are the                   */
   /*             object of SIGNAL ON or CALL ON                                                      */
   /*             the start of the file before the first label can be specified with                  */
   /*                   -where=[Unlabeled]                                                            */
   /*       -show=<list>                                                                              */
   /*             only show calls of type <list> where types are Internal, External, and Built-in     */
   /*             default is Int,Ext,BIF                                                              */
   /*                                                                                                 */
   /*  Examples:                                                                                      */
   /*       RexxTool.cmd CALLS                                                                        */
   /*       RexxTool.cmd CALLS -where=Label1,Label2                                                   */
   /*       RexxTool.cmd CALLS -where="Label1 Label2"                                                 */
   /*       RexxTool.cmd CALLS -where=Label* -show=Int                                                */
   /*                                                                                                 */
   /*  Output is to <new filename> or STDOUT                                                          */
   /*                                                                                                 */
   /* =============================================================================================== */
   call Setup
   call Task_Tokenize
   /*  skip the Prep_xxx routines because we are not changing anything                                */
   call Task_Calls
   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Egress 'BUG'
      end
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_Calls ======================================================================= */

   Script_Change: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Change variable variables var commands where                                             */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*     RexxTool.cmd CHANGE <filename> [<new file>] -proc="<old>=<new> [<old>=<new>] ..."           */
   /*                   change label/function name from <old> to <new>                                */
   /*     RexxTool.cmd CHANGE <filename> [<new file>] -var="<old>=<new> [<old>=<new>] ..."            */
   /*                   change variable name from <old> to <new>                                      */
   /*                                                                                                 */
   /*  Examples:                                                                                      */
   /*     RexxTool.cmd CHANGE -proc="In=Out"                                                          */
   /*                   change label "In" and all calls to it to "Out"                                */
   /*                                                                                                 */
   /*     RexxTool.cmd CHANGE -var="In=Out"                                                           */
   /*                   change all variables named "In" to "Out"                                      */
   /*                                                                                                 */
   /*     RexxTool.cmd CHANGE -var="In=Out Up=Down" or                                                */
   /*     RexxTool.cmd CHANGE -var=In=Out,Up=Down                                                     */
   /*                   change all variables named "In" to "Out" and "Up" to "Down"                   */
   /*     So this:                                                                                    */
   /*                In = ''                                                                          */
   /*                Stem.Up = ''                                                                     */
   /*     becomes this:                                                                               */
   /*                Out = ''                                                                         */
   /*                Stem.Down = ''                                                                   */
   /*                                                                                                 */
   /*     You can restrict the replacement to only stem or tail variables by including                */
   /*     a dot in <old> and <new>:                                                                   */
   /*           RexxTool.cmd CHANGE -var="Stem.=StemVar."                                             */
   /*     so                                                                                          */
   /*              Stem = 'abc'                                                                       */
   /*              Stem.Up = 'abc'                                                                    */
   /*     becomes                                                                                     */
   /*              Stem = 'abc'                                                                       */
   /*              StemVar.Up = 'abc'                                                                 */
   /*                                                                                                 */
   /*                                                                                                 */
   /*     The -where switch will limit the changes to certain procedures                              */
   /*                                                                                                 */
   /*          RexxTool.cmd CHANGE -var="In=Out" -where=LabelOne                                      */
   /*                         change "In" to "Out", but only in routine LabelOne                      */
   /*          RexxTool.cmd CHANGE -var="In=Out" -where=Toker_*                                       */
   /*                         change "In" to "Out", but only in routines that start with "Toker_"     */
   /*                                                                                                 */
   /*                                                                                                 */
   /*                                                                                                 */
   /*  There is another use for this command. It is a common practice to prefix tails of              */
   /*  a stem variable with a number or other special character to prevent confusion                  */
   /*  with simple variables that have the same name as the tail. Say you have a stem                 */
   /*  G. with tails like G.up and G.down. You can change them to G.0up and G.0down                   */
   /*  with this                                                                                      */
   /*     RexxTool.cmd CHANGE -var="G.=G.0"                                                           */
   /*                                                                                                 */
   /*  You can also change the prefix from "0" to "!" with                                            */
   /*     RexxTool.cmd CHANGE -var="G.0=G.!"                                                          */
   /*                                                                                                 */
   /*  Note that in the case of stem variables, the "0" and "!" are not complete                      */
   /*  tails, just prefixes. So do not try to change the name of an individual                        */
   /*  stem variable. This will probably not do what you want:                                        */
   /*                                                                                                 */
   /*     RexxTool.cmd CHANGE -var="G.me=G.you"                                                       */
   /*                                                                                                 */
   /*  because it will change anything that begins with "G.me", like "G.meeting"                      */
   /*  to "G.youeting"                                                                                */
   /*                                                                                                 */
   /*  Each token is examined only once, which means it is safe to swap values like this              */
   /*     RexxTool.cmd CHANGE -var="Up=Down Down=Up"                                                  */
   /*                                                                                                 */
   /*  The search for <old> is not case-sensitive                                                     */
   /*                                                                                                 */
   /*  the $FORMAT OFF instruction is ignored. That is, variable names will be changed                */
   /*  regardless of a token's t_readonly flag.                                                       */
   /*                                                                                                 */
   /*  The Change command can sometimes be faster and more accurate than a manual                     */
   /*  search-and-replace in a text editor, but you need to understand what you are doing.            */
   /*                                                                                                 */
   /*  In a quoted string or a comment, it is not possible to be sure that a string is                */
   /*  really a variable/procedure name, so it will ask if you want to change it.                     */
   /*                                                                                                 */
   /*  the Change command will end with exit code 4 if the file was changed, 0 if not changed         */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  All we want to do here is change variable names, so skip the preparation and                   */
   /*  indenting steps. Set margin to 0 to prevent adding a new margin                                */
   Opt.0Margin = 0
   call Setup
   call Task_Tokenize
   call Task_Change

   if debug.0token then call SaveTokensF filespec('N', G.0InfileN)||'_Tokens_End'

   /*  combine tokens into a new array of lines:                                                      */
   call Task_Rebuild
   /*  Save to the new file and quit                                                                  */
   call ArrayToFile 'SrcOut.', G.0OutFileN

   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
         call Egress 'BUG'
      end
   if G.0FileChanged then
      call Egress 'CHANGED'
   else
      if wordpos('FILE', G.0TokerExit) > 0 then
         call Egress 'FILE'
      else
         call Egress 'OKAY'
   /* === End of Script_Change ====================================================================== */

   Script_CheckOnly: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Check CheckOnly commands                                                                 */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*        RexxTool.cmd CHECKONLY <filename> [<newfile>] [-check=<list>]                            */
   /*                  just check for errors, no formatting                                           */
   /*     Options:                                                                                    */
   /*        -check=<list>                                                                            */
   /*                 check <list> error categories                                                   */
   /*                 default is -check=all                                                           */
   /*                 -check=all is the same as -check=error,warn,4OS2,style,expose                   */
   /*                                                                                                 */
   /*    See also "help Check"                                                                        */
   /* =============================================================================================== */
   Opt.0Check = 'ALL'
   Opt.0warn = 1
   /*     Opt.0verbose = 1                                                                            */

   call Setup

   call Task_Tokenize
   /*  skip the Prep_xxx routines because we are not changing anything                                */
   ErrorCount = Task_Check()
   if ErrorCount == 0 then
      ret = 'OKAY'
   else
      ret = 'ERROR'
   if wordpos('FILE', G.0TokerExit) > 0 then do
         call Msg G.0meID 'the Tokeniser found errors in the file'
         ret = 'FILE'
      end
   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
         call Egress 'BUG'
      end
   call Egress ret
   return 0
   /* === End of Script_CheckOnly =================================================================== */

   Script_Decontrol: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Decontrol Control                                                                        */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*                                                                                                 */
   /*     RexxTool.cmd DECONTROL <filename> [<new file>]                                              */
   /*                                                                                                 */
   /*  replaces control characters with hex strings                                                   */
   /*                                                                                                 */
   /*  I have seen a number of Rexx scripts that use control characters to display                    */
   /*  graphics symbols on the screen. It works, but I do not think it is a good                      */
   /*  idea to put the literal control characters in the Rexx file. Because of the                    */
   /*  control characters, a utility like 'file' will mis-identify it as a                            */
   /*  binary file, and a utility that converts line-endings will refuse to change it.                */
   /*  Using 'type' to send the file to the screen will garble the output because                     */
   /*  the terminal tries to interpret some of the control codes. Probably the same                   */
   /*  thing would happen if you tried to print the file.                                             */
   /*                                                                                                 */
   /*  The Decontrol command will change a literal control character to a hex string.                 */
   /*  It does not change what the script does, just how the character is represented.                */
   /*                                                                                                 */
   /*  the $FORMAT OFF instruction is ignored. That is, variable names will be changed                */
   /*  regardless of a token's t_readonly flag.                                                       */
   /*                                                                                                 */
   /*  the program will end with exit code 4 if the file was changed, 0 if not changed                */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  All we want to do here is change something, so skip the preparation and                        */
   /*  indenting steps. Set margin to 0 to prevent adding a new margin                                */
   Opt.0Margin = 0
   call Setup
   call Task_Tokenize
   call Task_DeControl

   if debug.0token then call SaveTokensF filespec('N', G.0InfileN)||'_Tokens_End'

   /*  combine tokens into a new array of lines:                                                      */
   call Task_Rebuild
   /*  Save to the new file and quit                                                                  */
   call ArrayToFile 'SrcOut.', G.0OutFileN

   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
         call Egress 'BUG'
      end
   if G.0FileChanged then
      call Egress 'CHANGED'
   else
      if wordpos('FILE', G.0TokerExit) > 0 then
         call Egress 'FILE'
      else
         call Egress 'OKAY'
   /* === End of Script_Decontrol =================================================================== */

   Script_Depends: /*fold00*/
   /* =============================================================================================== */
   /*  Help: depends depend required require requires show terse commands where handlers              */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*     RexxTool.cmd DEPENDS <filename> [<newfile>] [options]                                       */
   /*        show all internal, external, and built-in functions used,                                */
   /*        and (when possible) the libraries used                                                   */
   /*        and system commands used                                                                 */
   /*                                                                                                 */
   /*  Options:                                                                                       */
   /*     -where=LabelA,LabelB        show only routines used by these labels and their subroutines   */
   /*     -where=Label*               show only routines used by these labels and their subroutines   */
   /*     -where=LabelA,+handlers     "+handlers" will be replaced by a list of routines that are     */
   /*                                 the object of SIGNAL ON or CALL ON                              */
   /*     -where=[Unlabeled]          the pseudo-label [Unlabeled] refers to the start of the         */
   /*                                 file before the first label                                     */
   /*                                                                                                 */
   /*     -show=<list>                only show <list> types of calls                                 */
   /*                                 default is 'Int,Ext,BIF,Lib,Sys'                                */
   /*                                                                                                 */
   /*     -terse                      show each list on one line                                      */
   /*                                 (not for system commands)                                       */
   /*                                 this only exists for the convenience of standalone.btm          */
   /*                                                                                                 */
   /*  Sub-labels and local error-handlers that are recognized as sub-labels will not be listed       */
   /*  (see HELP SubLabel)                                                                            */
   /*                                                                                                 */
   /*  For example, I have a macro library of external procedures.                                    */
   /*  Before I released RexxTool.cmd I used                                                          */
   /*           RexxTool.cmd depends RexxTool.cmd -show=ext                                           */
   /*  to get a list of external procedures that I needed to copy into it.                            */
   /*                                                                                                 */
   /*  Sometimes it is not possible to determine which library was loaded because its                 */
   /*  name is represented by a variable. In that case the library name will be shown as "???".       */
   /*                                                                                                 */
   /*  The Depends command is one case that I know of where RexxTool.cmd should be run by             */
   /*  the Rexx interpreter that the file is intended for. That is, using Regina to analyze           */
   /*  dependencies of an OS/2 Rexx file could give wrong results.  Or vice-versa.                    */
   /*  If anyone really needs to do that, let me know and I'll fix it.                                */
   /* =============================================================================================== */
   call Setup
   call Task_Tokenize
   call Task_Depends
   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
         call Egress 'BUG'
      end
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_Depends ===================================================================== */

   Script_DevHelp: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: DevHelp commands tasks                                                                   */
   /*  Usage:                                                                                         */
   /*     RexxTool.cmd DEVHELP [<procedure> | TASKS]                                                  */
   /*              shows help for RexxTool.cmd procedures                                             */
   /*                                                                                                 */
   /*  <procedure> can have ? and * as wild characters.                                               */
   /*  All procedures with names that match will be shown.                                            */
   /*  The comparison is not case-sensitive.                                                          */
   /*                                                                                                 */
   /*  Examples:                                                                                      */
   /*     to show help for all the tool functions:                                                    */
   /*          RexxTool.cmd DEVHELP *                                                                 */
   /*     to show only the L_xxx functions:                                                           */
   /*          RexxTool.cmd DEVHELP L_*                                                               */
   /*     to show any function whose name contains "AsString":                                        */
   /*          RexxTool.cmd DEVHELP *AsString*                                                        */
   /*                                                                                                 */
   /*     or instead, to show the headers of all the Script and Task procedures:                      */
   /*          RexxTool.cmd DEVHELP tasks                                                             */
   /*                                                                                                 */
   /*  This is part of the help system of RexxTool.cmd, but it is really just a                       */
   /*  front end to the Usage command, searching RexxTool.cmd itself. This:                           */
   /*          RexxTool.cmd DEVHELP L_*                                                               */
   /*  is the same as this:                                                                           */
   /*          RexxTool.cmd USAGE path\to\RexxTool.cmd -where=L_*                                     */
   /* =============================================================================================== */
   call Setup
   if translate(Opt.0Where) == 'TASKS' then
      call Task_ShowTasks
   else
      call Task_UsageHeaders

   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   call Egress 'OKAY'
   /* === End of Script_DevHelp ===================================================================== */

   Script_Extract: /*fold00*/
   /* =============================================================================================== */
   /*  Help: Extract Require Requirements commands where handlers                                     */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*        RexxTool.cmd EXTRACT <filename> [<newfile>] -where=<list>                                */
   /*                          Copies routine(s) in <list> and all subroutines they depend            */
   /*                          on into a new file                                                     */
   /*                                                                                                 */
   /*  Options:                                                                                       */
   /*        -where=<label>[,<label>] [+handlers]                                                     */
   /*                          "+handlers" will be replaced by a list of any routines that            */
   /*                          are the object of SIGNAL ON or CALL ON instructions                    */
   /*                          <label> may use ? and * as wild characters                             */
   /*                          the pseudo-label [Unlabeled] refers to the start of the file           */
   /*                          before the first label                                                 */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*  Say you want to make a separate formatter from RexxTool.cmd. You would want                    */
   /*  to start by copying the formatting tasks listed in Script_Format into a new                    */
   /*  file. You would then have to locate all the routines that those tasks depend                   */
   /*  on, and all the routines that those routines depend on. You would also want to                 */
   /*  copy the Init_xxx routines and probably the error handlers. Instead of doing                   */
   /*  all that manually, it is easier to type                                                        */
   /*                                                                                                 */
   /*       RexxTool.cmd EXTRACT RexxTool.cmd New.cmd -where=Init,Script_Format,+handlers             */
   /*                                                                                                 */
   /*  As with many tasks, this assumes well-structured code-- that each routine is a                 */
   /*  complete unit from one label to the next label.                                                */
   /* =============================================================================================== */
   call Setup
   call Task_Tokenize
   call Prep_UnMargin
   call Task_Extract
   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
      end
   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_Extract ===================================================================== */

   Script_Format: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Format commands                                                                          */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*        RexxTool.cmd FORMAT <filename> [<newfile>] [options]                                     */
   /*                          Reformats <oldfile> and saves it as <newfile>                          */
   /*                          If <newfile> is not given, output is to STDOUT                         */
   /*                                                                                                 */
   /*        there are many options. For short descriptions type                                      */
   /*                  RexxTool.cmd -config                                                           */
   /*        or for long descriptions type                                                            */
   /*                  RexxTool.cmd help recap                                                        */
   /*                  RexxTool.cmd help respace                                                      */
   /*                  etc.                                                                           */
   /* =============================================================================================== */
   /*  The task order will depend on what the tasks do, but the scheme I am evolving is this:         */
   /*                                                                                                 */
   /*     * make an array of lines                                                                    */
   /*     * call any tasks that do not depend on tokenizing                                           */
   /*                                                                                                 */
   /*     the Tokeniser                                                                               */
   /*     * call Toker to make arrays of tokens (Toker_Rexx, Toker_Comment, Toker_Store)              */
   /*     * extra preparation like identifying labels, keywords, variables.                           */
   /*       (Toker_Fixup, Toker_SubKeywords, Toker_IDfunctions)                                       */
   /*     * also split up any stem variables into stem and tail parts (Toker_Fixup)                   */
   /*                                                                                                 */
   /*     * fix the prefix of multi-line comments. (Prep_UnMargin)                                    */
   /*     * read $format instructions to flag some tokens as ReadOnly (Prep_ReadOnly)                 */
   /*                                                                                                 */
   /*     Token Tasks  --tasks that do not care about lines                                           */
   /*     * check for errors                                                                          */
   /*     * re-capitalize keywords, variable names                                                    */
   /*     * inserting subroutine headers, footers                                                     */
   /*     * adjusting spaces between tokens                                                           */
   /*     * any other tasks that involves moving, inserting, deleting tokens                          */
   /*                                                                                                 */
   /*     Indent Tasks                                                                                */
   /*     * calculate indent levels based on keywords (Task_Indent)                                   */
   /*     * adjust indenting for personal preferences (Task_Indent2)                                  */
   /*                                                                                                 */
   /*     Line Tasks   --tasks that need to know what the lines are, and how long they are            */
   /*     * wrap long lines                                                                           */
   /*     * indent continued lines                                                                    */
   /*     * apply the indenting to the first token of each line                                       */
   /*     * some tasks that resize and shift comments                                                 */
   /*                                                                                                 */
   /*     * combine tokens into lines                                                                 */
   /*     * save lines to new file                                                                    */
   /*                                                                                                 */
   /*     * the plan is subject to change                                                             */
   /* =============================================================================================== */
   /*  fix: calling Script_Quick from here is several seconds faster than calling it                  */
   /*  directly from the Main level. I do not understand that.                                        */
   if Opt.0Quick then
      call Script_Quick
   if Opt.0Bare then
      call Script_Bare
   call Setup
   /* ----------------------------------------------------------------------------------------------- */
   /*  This is where you would put any tasks that involve lines instead of tokens.                    */
   /*  Some line-oriented tasks have been moved to RexxMerge.cmd.                                     */
   /*  The only line-oriented tasks left are Task_ShowTasks and Task_UsageHeaders,                    */
   /*  and they have their own scripts.                                                               */
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  this is the Tokeniser stage.                                                                   */
   /*  The Toker is a set of separate routines that all start with "Toker_",  Do not try              */
   /*  to write a script that will leave out one of the tokenising steps. They depend                 */
   /*  on each other in non-obvious ways.                                                             */
   call Task_Tokenize
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  Prepare the tokens for the formatting tasks. A script that does not produce                    */
   /*  formatted output can skip these steps:                                                         */
   /* fix up the margin of multi-line comments                                                        */
   /*      call Prep_UnMargin                                                                         */
   /* flag some tokens as Read Only                                                                   */
   /*      call Prep_MarkRO                                                                           */
   /*  "call Prepare" will do both of the above but it is faster. I do not understand why.            */
   call Prepare
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  First, anything that does not change tokens.                                                   */
   /*  You could use the return value from the Task_Checkxxx routines to decide whether to continue   */
   ret = Task_Check()
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  Then tasks that make changes to tokens:                                                        */
   /*  Inserting, deleting, or moving tokens must be done before indenting is calculated.             */
   /*  Re-spacing labels should come after tasks that move tokens or change spacing.                  */
   /*  Re-capitalizing can be done any time after tasks that insert new tokens                        */
   call Task_FixRexxID
   call Task_Unwrap
   call Task_Procedure
   call Task_HeaderFooter
   call Task_FixFooters
   call Task_Then
   call Task_ThenDo
   call Task_FoldLine
   call Task_SpaceLabels
   call Task_Recap
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  calculate indenting                                                                            */
   /*  This must come after any tasks that insert tokens, and before any tasks that need              */
   /*  to know the line length                                                                        */
   call Task_Indent
   /*  then any adjustments to the indenting to suit user preferences.                                */
   call Task_Indent2
   /* ----------------------------------------------------------------------------------------------- */

   /*  things that change token prefixes go here. Anything (like Task_Indent2) that looks             */
   /*  at original prefixes must come before this point.                                              */
   call Task_ReSpace
   /* ----------------------------------------------------------------------------------------------- */
   /*  things that depend on knowing the length of lines, like wrapping long lines and                */
   /*  formatting comments to the width of the page                                                   */
   call Task_Wrap

   /*  Task_debugEOC is a debugging tool but I usually include it if tokens have been moved           */
   /*  It takes very little time                                                                      */
   call Task_debugEOC

   /*  apply the indenting that was calculated above by changing the prefix of the                    */
   /*  first token in each line                                                                       */
   call Task_ApplyIndent

   /*  move comments, justify right margin of comments, etc.                                          */
   call Task_FormatComments
   /* ----------------------------------------------------------------------------------------------- */
   /*  if wanted, save a list of tokens in their final state                                          */
   if debug.0token then call SaveTokensF filespec('N', G.0InfileN)||'_Tokens_End'

   /*  combine tokens into a new array of lines:                                                      */
   call Task_Rebuild

   /*  Save to the new file and quit                                                                  */
   call ArrayToFile 'SrcOut.', G.0OutFileN

   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
         call Egress 'BUG'
      end
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_Format ====================================================================== */

   Script_Help: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: help                                                                                     */
   /*                                                                                                 */
   /*  RexxTool.cmd HELP <subject>                                                                    */
   /*  RexxTool.cmd HELP all                                                                          */
   /*                                                                                                 */
   /*  RexxTool has a HELP command that searches the comments in RexxTool.cmd                         */
   /*  for a subject. For example, to learn about the format command and the                          */
   /*  options that it uses, type                                                                     */
   /*        RexxTool.cmd help format                                                                 */
   /*  Or if you only want to know about the -MoveComment switch, use                                 */
   /*        RexxTool.cmd help movecomment                                                            */
   /*                                                                                                 */
   /*  The help system may not be as readable as a separate documentation file but it                 */
   /*  is much easier to maintain. If you want everything in one file, type                           */
   /*        RexxTool.cmd help all                                                                    */
   /*  and redirect stdout to a file                                                                  */
   /*                                                                                                 */
   /*  Internally, it works by searching for key words in the header of any routine                   */
   /*  that starts with Script_, Style_, Task_, Prep_, or Help_                                       */
   /*                                                                                                 */
   /*  If you are working on RexxTool and need information about low-level procedures,                */
   /*  use the DEVHELP command. See, of course,                                                       */
   /*        RexxTool.cmd help devhelp                                                                */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  This is the original help routine, before I made things complicated.                           */
   /* ----------------------------------------------------------------------------------------------- */
   select
      when Opt.0help == 1 then do
            say  ''
            say  '    Usage:'
            say  '        'G.0me '[<options>] <command> <filename> [<new file>] [<options>]'
            say  '    where <command> is one of'
            say  '         Format <filename> [<newfile>] [options]'
            say  '               indent and reformat the file'
            say  '         CheckOnly <filename> [<newfile>] [-check=<list>]'
            say  '               only check for errors, no formatting'
            say  '         Calls <filename> [<newfile>] [-where=<list>'
            say  '               show what a routine calls and is called by'
            say  '         Depends <filename> [<newfile>] [-where=<list>] [-show=<list> [-terse]'
            say  '               show what internal, external, and built-in functions'
            say  '               that a file or routine depends on'
            say  '         Unused <filename> [<newfile>] [-flat]'
            say  '               show routines and variables that may be unused'
            say  '         Variables <filename> [<newfile>] [-where=<list>] [-show=compact|lines|unused] [-flat]'
            say  '               show information about variables'
            say  '         Tree <filename> [<newfile>] [-where=<list>] [-show=<list>]'
            say  '               draw a tree diagram of calls'
            say  '         Change <filename> [<newfile>] -var="<old>=<new>" -proc="<old>=<new>"'
            say  '               change variable names and/or routine names'
            say  '         Decontrol <filename> [<newfile>]'
            say  '               replace control characters with hex strings'
            say  '         Extract <filename> [<newfile>] [-where=<list>]'
            say  '               extract a routine with all its dependencies'
            say  '         AddCounter <filename> [<newfile>]'
            say  '               insert code to count calls for each routine'
            say  '         Usage <filename> [<newfile>] [-where=<list>]'
            say  '               show "Usage:" headers for tool procedures'
            say  ''
            say  '    <command> <filename> <newfile> must be in that order'
            say  '    options can be anywhere'
            say  '    if <newfile> is not given, output will go to STDOUT'
            say  ''
            say  '    for help about each command, type'
            say  '        'G.0me 'help format'
            say  '        'G.0me 'help calls'
            say  '         etc.'
            say  ''
            say  '    Some of the options:'
            say  '       -config             or'
            say  '       <command> -config <options>'
            say  '                           shows all the options, their current values, and a short'
            say  '                           description. All options can be changed on the command line.'
            say  '                           An option that is on by default can be turned off with'
            say  '                              -option=0 or -option=""'
            say  '                              -!option is the same as -option=0'
            say  '       help, -help         this is it'
            say  '       help <subject>>     longer description of a switch or subject'
            say  '       help all            all the documentation at once'
            say  ''
            say  '    formatting for parts of a file can be turned on or off. See'
            say  '        'G.0me 'help $format'
            say  ''
            say  '    If you want to redirect output, all warnings and error messages go to STDERR'
            say  ''
            say  ''
            say  '    These commands are only of interest if you working on' G.0me 'itself'
            say  '       DevHelp             shows help for tool functions'
            say  '       TokerOnly           for testing and debugging the tokeniser'
            say  '       Test                runs the Script_Test script'
            say  '       help hack           some suggestions about how you can add features'
            say  '       help types          information about tokens and how they are described'
            say  '       help todo           bugs to fix and ideas about new features'
            say  ''
            say  '    Exit codes:'
            say  '         0     normal exit'
            say  '         1     usage error'
            say  '         2     file is not Rexx'
            say  '         3     defective file'
            say  '         4     file was changed. Used by the Change and Decontrol commands only'
            say  '         5     error found by CheckOnly command'
            say  '         9     error, either a bug in RexxTool.cmd or defective code'
            say  '               that it could not handle.'
            say  ''
            /* ====================================================================================== */
         end
      otherwise
         G.0InfileN = G.0me_long
         call Setup
         call Task_ShowHelp
   end   /*  select  */
   call Egress 'OKAY'
   /* === End of Script_Help ======================================================================== */

   Script_Quick: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Quick                                                                                    */
   /*                                                                                                 */
   /*    RexxTool.cmd FORMAT <filename> [<newfile>] -quick [options]                                  */
   /*                                                                                                 */
   /*    This does a quick format-- code indenting and basic error-checking only.                     */
   /*    It takes about half the time of a full format.                                               */
   /*                                                                                                 */
   /*    by default it assumes "-check=error,4os2 -debug=fatal -idle=0" but those can be              */
   /*    overridden on the command line.                                                              */
   /* =============================================================================================== */
   Opt.0Check = 'ERROR,4OS2'
   Opt.0debug = 'fatal'
   Opt.0idle = 0
   call Setup
   call Task_Tokenize
   call Prepare
   call Task_Check
   call Task_Indent
   call Task_Indent2
   call Task_ApplyIndent
   if debug.0token then call SaveTokensF filespec('N', G.0InfileN)||'_Tokens_End'

   /*  combine tokens into a new array of lines:                                                      */
   call Task_Rebuild

   /*  Save to the new file and quit                                                                  */
   call ArrayToFile 'SrcOut.', G.0OutFileN

   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
         call Egress 'BUG'
      end
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_Quick ======================================================================= */

   Script_Test: /*fold00*/
   /* =============================================================================================== */
   /*  Help: Test debug verbose warn                                                                  */
   /*                                                                                                 */
   /*  Options for testing and debugging:                                                             */
   /*                                                                                                 */
   /*     -verbose               Show more verbose messages. Mostly for debugging.                    */
   /*                            Among other things, it means that error messages will show           */
   /*                            the procedure name as well as the line number that the               */
   /*                            message comes from. If there are a lot of error messages,            */
   /*                            -verbose can make RexxTool.cmd noticeably slower.                    */
   /*                                                                                                 */
   /*    -debug=<list>           Turn on debugging for categories in <list>. For example              */
   /*                               debug=token,loop,eoc                                              */
   /*                               debug="token loop eoc"                                            */
   /*                            You can implement a new category xxx simply by using it,             */
   /*                            with a line in your code like this                                   */
   /*                                if debug.xxx then ...                                            */
   /*                            Please document it in the header of a Script or Task routine         */
   /*                                                                                                 */
   /*    -debug=arg              Do extra debugging to verify arguments to procedures                 */
   /*                                                                                                 */
   /*    -debug=token            Save list of tokens after tokenising and after formatting            */
   /*                                                                                                 */
   /*    -debug=loop             Break out of a loop if it seems to be endless.                       */
   /*                            In one or two places this is always on to protect against            */
   /*                            defective code.                                                      */
   /*                                                                                                 */
   /*    -debug=fatal            Abort if a debugging error was detected.                             */
   /*                            Without this, RexxTool.cmd just shows a warning and continues        */
   /*                            Also sets the G.0fatalerror flag that a script can use               */
   /*                            to decide what to do                                                 */
   /*                                                                                                 */
   /*    -warn                   Turns on some messages from the Tokeniser about errors               */
   /*                            that it finds. This is separate from the -Check switch               */
   /*                            and partly duplicates what it does.                                  */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  this header is as good a place as any to document debugging-related switches                   */
   /* ----------------------------------------------------------------------------------------------- */
   call Setup
   call Task_Tokenize
   do t = 2 to t_type.0
      if t_val.t \= '' then do
            ret = R_Infunction(t)
            call Mark 'ret t_val.t'
         end
   end t


   call Egress 'OKAY'

   /*     call Prep_UnMargin                                                                          */
   /*     call Prep_MarkRO                                                                            */

   /*     call Task_SpaceLabels                                                                       */

   /*     call Task_Check                                                                             */
   /*     call Task_HeaderFooter                                                                      */
   /*     call Task_Then                                                                              */
   /*     call Task_ThenDo                                                                            */
   /*     call Task_FoldLine                                                                          */
   /*     call Task_ReSpace                                                                           */
   /*     call Task_SpaceLabels                                                                       */
   /*     call Task_Recap                                                                             */
   /*     if debug.0eoc then call Task_debugEOC                                                       */
   /*     call Task_Indent                                                                            */
   /*     call Task_UnIndentComments                                                                  */
   /*     call Task_Wrap                                                                              */
   /*     call Task_FormatComments                                                                    */

   /* ----------------------------------------------------------------------------------------------- */
   if debug.0token then call SaveTokensF filespec('N', G.0InfileN)||'_Tokens_End'
   /*     call Task_ApplyIndent                                                                       */

   /*     call Task_Rebuild                                                                           */
   /*     call ArrayToFile 'SrcOut.', G.0OutFileN                                                     */

   call Egress 'OKAY'
   /* === End of Script_Test ======================================================================== */

   Script_TokerOnly: /*fold00*/
   /* =============================================================================================== */
   /*  Help: TokerOnly Toker Tokenizer Tokeniser                                                      */
   /*                                                                                                 */
   /*  RexxTool.cmd TOKERONLY <filename> [<newfile>]                                                  */
   /*                                                                                                 */
   /*  Script_TokerOnly is for testing the Tokeniser/UnMargin stage.                                  */
   /*  The resulting file should be exactly the same as the original, except that--                   */
   /*       an ending linefeed will be added if needed,                                               */
   /*       trailing spaces at the ends of lines will be removed                                      */
   /*       any End-Of-File character (Ctrl-Z) will be removed                                        */
   /* =============================================================================================== */
   Opt.0warn = 1
   Opt.0Margin = 0
   Opt.0debug = 'eoc arg loop'
   Opt.0Check = 'toker'
   call Setup
   call Task_Tokenize
   call Prep_UnMargin
   call Task_Check
   call Task_Rebuild
   if debug.0token then call SaveTokensF filespec('N', G.0InfileN)||'_Tokens_End'
   call ArrayToFile 'SrcOut.', G.0OutFileN
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_TokerOnly =================================================================== */

   Script_Tree: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: tree show commands where handlers ascii                                                  */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*     RexxTool.cmd TREE <filename> [<newfile>] [-where=<list>] [-show=<list>]                     */
   /*                    Shows a tree diagram of an internal routine(s) and all                       */
   /*                    subroutines that it depends on.                                              */
   /*                                                                                                 */
   /*  Options:                                                                                       */
   /*     -where=<label>[,<label>]                                                                    */
   /*                    the root(s) of the tree diagram(s)                                           */
   /*                    <label> may use ? and * as wild characters                                   */
   /*                    "+handlers" will be expanded to a list of routines that are                  */
   /*                    the object of SIGNAL ON or CALL ON                                           */
   /*                    [Unlabeled] is the start of the file before the first label                  */
   /*                                                                                                 */
   /*    -show=<list>                                                                                 */
   /*                    Types of routines to show                                                    */
   /*                    One or more of 'Int', 'Ext' or 'BIF', in a comma-separated list.             */
   /*                    Default is "Int,Ext"                                                         */
   /*                    Internal routines will always be shown                                       */
   /*                                                                                                 */
   /*    -ascii          Use ASCII characters to draw lines                                           */
   /*                    default is to use line-drawing characters if codepage is 850 or 437          */
   /*                                                                                                 */
   /*                                                                                                 */
   /*  Examples:                                                                                      */
   /*    RexxTool.cmd TREE <Infile> <Outfile> -where=Frodo -show=Ext                                  */
   /*                    Shows a tree of internal and external routines                               */
   /*                    used by routine Frodo.                                                       */
   /*                                                                                                 */
   /*    RexxTool.cmd TREE <Infile> <Outfile>                                                         */
   /*                    Shows everything. Each routine that is not called by any other routine       */
   /*                    will be the start of a separate tree.                                        */
   /*                    Probably more information than you really want                               */
   /*                                                                                                 */
   /*  A '...' in the tree diagram indicates a recursive loop                                         */
   /*                                                                                                 */
   /*  Output is to the output file or stdout                                                         */
   /*                                                                                                 */
   /*  Known bug:                                                                                     */
   /*  This will not automatically show a top-level recursive loop of more than two                   */
   /*  calls. As in                                                                                   */
   /*        Main                                                                                     */
   /*          |___SubA                                                                               */
   /*               |___SubB                                                                          */
   /*                    |___Main                                                                     */
   /*                                                                                                 */
   /*  I haven't thought of a fix yet.                                                                */
   /*  Of course, you still see it by using "-where=Main"                                             */
   /*                                                                                                 */
   /*  ==> if routines are very deeply nested, this may abort with an error 11,                       */
   /*      "Control Stack Full". OS/2 Regina might do better, but it has its own                      */
   /*      problems with very large files. Linux Regina should not be a problem.                      */
   /*      The only file I have run across that still has that problem is ppwizard.cmd                */
   /* =============================================================================================== */
   call Setup
   call Task_Tokenize
   /*  skip the Prep_xxx routines because we are not changing anything                                */
   call Task_Tree
   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
         call Egress 'BUG'
      end
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_Tree ======================================================================== */

   Script_Unused: /*fold00*/
   /* =============================================================================================== */
   /*  Help: Unused commands flat alt where                                                           */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*         RexxTool.cmd UNUSED <filename> [<newfile>] [options]                                    */
   /*                      show routines and variables that seem to be unused                         */
   /*                                                                                                 */
   /*  Options:                                                                                       */
   /*         -flat          assume all variables are global                                          */
   /*         -alt           use the alternate method for identifying unused stem                     */
   /*                        variables whose stem is exposed by another routine.                      */
   /*                        This switch will reduce the number of false positives but                */
   /*                        can also miss some unused variables.                                     */
   /*         -where=<list>  limit display to listed routines                                         */
   /*                        do not combine this with -flat                                           */
   /*                                                                                                 */
   /*  Lists labels that seem to be unused. That is, ones that are not the object of                  */
   /*  a CALL, SIGNAL, or function call. It will show a false positive in a case where                */
   /*  execution falls through to the next label.                                                     */
   /*                                                                                                 */
   /*  Lists any variable that seems to be unused in its routine, but assumes that an                 */
   /*  exposed variable is used somewhere else.                                                       */
   /*                                                                                                 */
   /*  If the -flat switch is used, all variables are assumed to be global, even if                   */
   /*  the program has procedures with local variables. In some cases that might be                   */
   /*  more useful. If there are no procedures, -flat is assumed automatically.                       */
   /*                                                                                                 */
   /*  There can be a lot of false hits from this, but if you are looking for unused                  */
   /*  variables it will cut down the list of suspects to something manageable.                       */
   /* =============================================================================================== */
   /*  fix: this involves two calls to GetCalls, one from Task_ShowUnCalled and                       */
   /*        one from Task_UnusedVars.  Causes duplicate error messages.                              */
   /*  fix: make this smarter.                                                                        */
   call Setup
   call Task_Tokenize
   call Task_ShowUnCalled
   call Out ''
   if \G.0HasProcedures then
      if Opt.0Where == '' then
         Opt.0Flat = 1
   call Task_UnusedVars

   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
         call Egress 'BUG'
      end
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_Unused ====================================================================== */

   Script_Usage: /*fold00*/
   /* =============================================================================================== */
   /*  Help: UsageHeaders commands where UsageHeader header headers usage                             */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*        RexxTool.cmd USAGE <filename> [<newfile>] [-where=<label>[,<label>]]                     */
   /*                          shows the "Usage:" headers for routines                                */
   /*                          -where limits the results to the listed labels                         */
   /*                          <label> may use ? and * as wild characters                             */
   /*                                                                                                 */
   /*  Option:                                                                                        */
   /*       -where='CL_*'       show only headers whose name starts with "CL_", like CL_FirstWord     */
   /*       -where='*string*'   show only headers whose name contains "string", like L_AsString       */
   /*                           or ChangeString                                                       */
   /*                                                                                                 */
   /* =============================================================================================== */
   call Setup
   call Task_UsageHeaders
   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   call Egress 'OKAY'
   /* === End of Script_Usage ======================================================================= */

   Script_Variables: /*fold00*/
   /* =============================================================================================== */
   /*  Help: Variable Variables debug commands where show flat debug                                  */
   /*                                                                                                 */
   /*  Usage:                                                                                         */
   /*         RexxTool.cmd VARIABLES <filename> [<newfile>] [options]                                 */
   /*                   show variables exposed, assigned, and used by each routine                    */
   /*                                                                                                 */
   /*  Options:                                                                                       */
   /*         -where=<list>                                                                           */
   /*                    only show routines in <list>                                                 */
   /*                    label names may use ? and * as wild characters                               */
   /*                    use "[Unlabeled]" for the top level outside any label                        */
   /*         -flat      assume all variables are global                                              */
   /*                    If your program has no procedures, use this.                                 */
   /*                    Cannot be combined with -where                                               */
   /*         -show=compact                                                                           */
   /*                    a more compact display of the same information                               */
   /*         -show=lines                                                                             */
   /*                   include line numbers                                                          */
   /*         -debug=var                                                                              */
   /*                    also dump the unprocessed variables returned by MakeVarListsX                */
   /*                                                                                                 */
   /*  These are intended as examples of how you might show information about variables.              */
   /*  See the Task_Variables, MakeVarListsL, MakeVarListsC procedures and adapt them                 */
   /*  to suit yourself.                                                                              */
   /* =============================================================================================== */
   call Setup
   call Task_Tokenize
   call Task_Variables

   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   if G.0fatalerror \== 0 then do
         call Msg G.0meID 'There were serious errors that indicate either'
         call Msg G.0meID_ind 'defective code in' G.0InfileN 'or a bug in' G.0me
         call Msg G.0meID_ind 'Check the results carefully'
         call Egress 'BUG'
      end
   if wordpos('FILE', G.0TokerExit) > 0 then
      call Egress 'FILE'
   else
      call Egress 'OKAY'
   /* === End of Script_Variables =================================================================== */

   Style_akm: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Format akm style                                                                         */
   /*                                                                                                 */
   /*  Options:                                                                                       */
   /*    -style=akm        use Anton Monroe's preferred style. Add headers                            */
   /*                      and footers, fold compound statements, move                                */
   /*                      THEN and THEN DO up, etc. For a list, see                                  */
   /*                            RexxTool.cmd FORMAT -style=akm -config                               */
   /*                      Also customizes capitalization of BIFs, and                                */
   /*                      saves a little time by only recognizing external BIFs                      */
   /*                      from the RexxUtil and RXU libraries, because those are the                 */
   /*                      only libraries I normally use.                                             */
   /*                                                                                                 */
   /*  this is an example of how to customize options                                                 */
   /*  invoke it with "RexxTool.cmd format -style=akm"                                                */
   /* =============================================================================================== */
   /*  $format comment off                                                                            */
   /*  the options for this style:                                                                    */
   Opt.0debug           = 'eoc,arg,fatal'    /*  debug categories   See "help debug"      $ <list>    */
   Opt.0dialect         = ''       /*  see "help dialect"                     $ '', REXXSAA or REGINA */
   Opt.0idle            = 0        /*  use lower (idle) priority                          $ 0 or 1    */
   Opt.0progress        = 1        /*  show progress                                      $ 0 or 1    */
   Opt.0verbose         = 1        /*  verbose error messages                             $ 0 or 1    */
   Opt.0warn            = 1        /*  Tokeniser warns about faulty code                  $ 0 or 1    */
   /*  Formatting options:                                                                            */
   Opt.0AddFooter       = 1        /*  insert footer comment at end of subroutine         $ 0 or 1    */
   Opt.0AddHeader       = 1        /*  insert empty header at start of subroutine         $ 0 or 1    */
   Opt.0Bare            = 0        /*  do a bare format only                              $ 0 or 1    */
   Opt.0BIFs            = 'MIXED'  /*  how to recap Built-In Functions     $ UPPER,LOWER,MIXED, or '' */
   Opt.0EndComment      = 'LEFT'   /*  how to shift comment after END                    $ LEFT or '' */
   Opt.0FoldLine        = 1        /*  break IF statement at logical OPs                  $ 0 or 1    */
   Opt.0Footer          = 1        /*  maintain subroutine footers, if any                $ 0 or 1    */
   Opt.0FormatComments  = 1        /*  format comments                                    $ 0 or 1    */
   Opt.0FTE             = 1        /*  do not change FTE comments                         $ 0 or 1    */
   Opt.0Indent          = 1        /*  do indenting                                       $ 0 or 1    */
   Opt.0IndentContinued = 'SMART'  /*  how much to indent continued lines           $ '' or <number>  */
   Opt.0IndentDivider   = 1        /*  align divider comments with Rexx code              $ 0 or 1    */
   Opt.0IndentEnd       = 0        /*  indent END or align with DO                        $ 0 or 1    */
   Opt.0IndentMulti     = 1        /*  align multi-line comments with Rexx code           $ 0 or 1    */
   Opt.0IndentWhen      = 1        /*  indent WHEN or align with SELECT                   $ 0 or 1    */
   Opt.0Keywords        = 'MIXED'  /*  how to recap keywords               $ UPPER,LOWER,MIXED, or '' */
   Opt.0Margin          = 3        /*  number of leading spaces (left margin)             $ <number>  */
   Opt.0MoveComment     = 'RIGHT'  /*  how to move right comments   $ RIGHT,LEFT,EXPAND,<col>, ''     */
   Opt.0OutdentThenDo   = 0        /*  shift block under THEN DO to left                  $ 0 or 1    */
   Opt.0Procedure       = 'UP'     /*  where to move PROCEDURE                      $ UP, DOWN, or '' */
   Opt.0Quick           = 0        /*  do a quick format only                             $ 0 or 1    */
   Opt.0ReCap           = 1        /*  recapitalize symbols and keywords                  $ 0 or 1    */
   Opt.0ReSpace         = 1        /*  change spacing between words                       $ 0 or 1    */
   Opt.0RexxID          = 1        /*  make sure it starts with a RexxID comment          $ 0 or 1    */
   Opt.0SpaceConcat     = 0        /*  add spaces around "||" operator                 $ 0, 1, or ''  */
   Opt.0SpaceNot        = 0        /*  add space after \                               $ 0, 1, or ''  */
   Opt.0SpaceParenth    = 0        /*  add spaces at ( and )                           $ 0, 1, or ''  */
   Opt.0Spacer          = '1'      /*  number of empty lines between routines       $ '' or <number>  */
   Opt.0Tab             = 3        /*  number of spaces per indent level                  $ <number>  */
   Opt.0Then            = 'UP'     /*  how to move THEN                             $ UP, DOWN, or '' */
   Opt.0ThenDo          = 'UP'     /*  how to move THEN DO                    $ UP DOWN, SPLIT, or '' */
   Opt.0Unwrap          = 0        /*  unwrap continued lines                             $ 0 or 1    */
   Opt.0Width           = 101      /*  length of lines (plus margin)                      $ <number>  */
   Opt.0Wrap            = 0        /*  try to wrap long lines to width (incomplete)       $ 0 or 1    */
   /*  Check options:                                                                                 */
   Opt.0Check           = 'ALL'
   Opt.0CheckStem       = ''       /*  stem variable to check                         $ '' or <list>  */
   Opt.0Fix             = 0        /*  fix some errors            See help fix"           $ 0 or 1    */
   /*  $format comment on                                                                             */

   /*  for the -AddHeader option, you can define what the empty header will be                        */
   Header.1 = ' '||copies('=', 78)||' '
   Header.2 = left('  Usage: ', 80)
   Header.3 = copies(' ', 80)
   Header.4 = ' '||copies('=', 78)||' '
   Header.0 = 4

   /*  for Task_Recap--                                                                               */
   /*  akm: my own writing style seems to use lower case for Rexx built-in functions and              */
   /*  most keywords, but mixed case for the functions with long names from RexxUtil and              */
   /*  other libraries. The formatter does not distinguish between them, but I can get                */
   /*  what I want by customizing the BIF lists that were defined in Init_DefineBIFs                  */

   BIFs.0rexx = 'abbrev abs address arg b2x beep bitand bitor bitxor c2d c2x center' ,
                'centre charin charout chars compare condition copies d2c d2x datatype' ,
                'date dbAdjust dbBracket dbCenter dbCJustify dbLeft dbRight dbRLeft' ,
                'dbRRight dbToDBCS dbToSBCS dbUnBracket dbValidate dbWidth delstr delword' ,
                'digits directory endlocal errortext filespec form format fuzz insert' ,
                'lastpos left length linein lineout lines max min overlay pos queued' ,
                'random reverse right RxFuncAdd RxFuncDrop RxFuncQuery rxqueue setlocal' ,
                'sign sourceline space stream strip substr subword symbol time trace' ,
                'translate trunc value verify word wordindex wordlength wordpos words x2b' ,
                'x2c x2d xrange'

   keywords_rexx = translate(keywords_rexx, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')
   keywords2_rexx = ,
     'expose name off on then upper with to by for forever until while over' ,
     'digits form scientific engineering fuzz source var version' ,
     'Error Failure Halt NoValue Syntax NotReady'
   BIFs.0regina = translate(BIFs.0regina, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')
   keywords_regina = ,
     translate(keywords_regina, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')
   keywords2_regina = ,
     keywords2_rexx 'with input normal noEOL output append replace error stream stem LIFO FIFO'

   /*  Most of the included BIF lists are for libraries I don't normally use.  Cut down the           */
   /*  list of libraries to the one used here. It saves about 2 seconds of execution time.            */
   BIFs.0OS2Libraries = 'rexx rexxutil rxu'
   return 0
   /* === End of Style_akm ========================================================================== */

   Style_Alternate: /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Format Alternate style                                                                   */
   /*                                                                                                 */
   /*  Options:                                                                                       */
   /*    -style=Alternate                                                                             */
   /*                                                                                                 */
   /*  this is another example of a style that you can customize                                      */
   /*  it is a more conservative style with                                                           */
   /*    80 column width                                                                              */
   /*    4 spaces per tab                                                                             */
   /*    no margin                                                                                    */
   /*    does not indent WHEN                                                                         */
   /*    OutdentThenDo                                                                                */
   /*    customized capitalization as in Style_akm                                                    */
   /*    add spaces around concatenation operator                                                     */
   /*    do not move right-hand comments                                                              */
   /*    do not fold compound statements                                                              */
   /*    do not add footers or headers                                                                */
   /*  and generally tampers less with the existing style                                             */
   /*  See RexxTool.cmd format style=alternate -config                                                */
   /* =============================================================================================== */
   /*  $format comment off                                                                            */
   /*  the options for this style:                                                                    */

   Opt.0debug           = 'eoc,arg,fatal'    /*  debug categories   See "help debug"      $ <list>    */
   Opt.0dialect         = ''       /*  see "help dialect"                     $ '', REXXSAA or REGINA */
   Opt.0idle            = 0        /*  use lower (idle) priority                          $ 0 or 1    */
   Opt.0progress        = 1        /*  show progress                                      $ 0 or 1    */
   Opt.0verbose         = 1        /*  verbose error messages                             $ 0 or 1    */
   Opt.0warn            = 1        /*  Tokeniser warns about faulty code                  $ 0 or 1    */
   /*  Formatting options:                                                                            */
   Opt.0AddFooter       = 0        /*  insert footer comment at end of subroutine         $ 0 or 1    */
   Opt.0AddHeader       = 0        /*  insert empty header at start of subroutine         $ 0 or 1    */
   Opt.0BIFs            = 'MIXED'  /*  how to recap Built-In Functions     $ UPPER,LOWER,MIXED, or '' */
   Opt.0EndComment      = ''       /*  how to shift comment after END                    $ LEFT or '' */
   Opt.0FoldLine        = 0        /*  break IF statement at logical OPs                  $ 0 or 1    */
   Opt.0Footer          = 1        /*  maintain subroutine footers, if any                $ 0 or 1    */
   Opt.0FormatComments  = 1        /*  format comments                                    $ 0 or 1    */
   Opt.0FTE             = 1        /*  do not change FTE comments                         $ 0 or 1    */
   Opt.0Indent          = 1        /*  do indenting                                       $ 0 or 1    */
   Opt.0IndentContinued = 'SMART'  /*  how much to indent continued lines           $ '' or <number>  */
   Opt.0IndentDivider   = 1        /*  align divider comments with Rexx code              $ 0 or 1    */
   Opt.0IndentEnd       = 0        /*  indent END or align with DO                        $ 0 or 1    */
   Opt.0IndentMulti     = 1        /*  align multi-line comments with Rexx code           $ 0 or 1    */
   Opt.0IndentWhen      = 0        /*  indent WHEN or align with SELECT                   $ 0 or 1    */
   Opt.0Keywords        = 'MIXED'  /*  how to recap keywords               $ UPPER,LOWER,MIXED, or '' */
   Opt.0Margin          = 0        /*  number of leading spaces (left margin)             $ <number>  */
   Opt.0MoveComment     = ''       /*  how to move right comments   $ RIGHT,LEFT,EXPAND,<col>, ''     */
   Opt.0OutdentThenDo   = 1        /*  shift block under THEN DO to left                  $ 0 or 1    */
   Opt.0Procedure       = ''       /*  where to move PROCEDURE                      $ UP, DOWN, or '' */
   Opt.0ReCap           = 1        /*  recapitalize symbols and keywords                  $ 0 or 1    */
   Opt.0ReSpace         = 1        /*  change spacing between words                       $ 0 or 1    */
   Opt.0SpaceConcat     = 1        /*  add spaces around "||" operator                 $ 0, 1, or ''  */
   Opt.0SpaceNot        = 1        /*  add space after \                               $ 0, 1, or ''  */
   Opt.0SpaceParenth    = 0        /*  add spaces at ( and )                           $ 0, 1, or ''  */
   Opt.0Spacer          = ''       /*  number of empty lines between routines       $ '' or <number>  */
   Opt.0Tab             = 4        /*  number of spaces per indent level                  $ <number>  */
   Opt.0Then            = 'UP'     /*  how to move THEN                             $ UP, DOWN, or '' */
   Opt.0ThenDo          = 'UP'     /*  how to move THEN DO                    $ UP DOWN, SPLIT, or '' */
   Opt.0Unwrap          = 0        /*  unwrap continued lines                             $ 0 or 1    */
   Opt.0Width           = 80       /*  length of lines (plus margin)                      $ <number>  */
   Opt.0Wrap            = 0        /*  try to wrap long lines to width (incomplete)       $ 0 or 1    */
   /*  Check options:                                                                                 */
   Opt.0Check           = 'Error,Warn'
   Opt.0CheckStem       = ''    /*  stem variable to check           $ '' or <list>  */
   Opt.0Fix             = 0        /*  fix some errors            See help fix"           $ 0 or 1    */
   /*  $format comment on                                                                             */

   /*  for Task_Recap--                                                                               */
   /*  akm: my own writing style seems to use lower case for Rexx built-in functions and              */
   /*  most keywords, but mixed case for the functions with long names from RexxUtil and              */
   /*  other libraries. The formatter does not distinguish between them, but I can get                */
   /*  what I want by customizing the BIF lists that were defined in Init_DefineBIFs                  */

   BIFs.0rexx = 'abbrev abs address arg b2x beep bitand bitor bitxor c2d c2x center' ,
                'centre charin charout chars compare condition copies d2c d2x datatype' ,
                'date dbAdjust dbBracket dbCenter dbCJustify dbLeft dbRight dbRLeft' ,
                'dbRRight dbToDBCS dbToSBCS dbUnBracket dbValidate dbWidth delstr delword' ,
                'digits directory endlocal errortext filespec form format fuzz insert' ,
                'lastpos left length linein lineout lines max min overlay pos queued' ,
                'random reverse right RxFuncAdd RxFuncDrop RxFuncQuery rxqueue setlocal' ,
                'sign sourceline space stream strip substr subword symbol time trace' ,
                'translate trunc value verify word wordindex wordlength wordpos words x2b' ,
                'x2c x2d xrange'

   keywords_rexx = translate(keywords_rexx, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')
   keywords2_rexx = ,
     'expose name off on then upper with to by for forever until while over' ,
     'digits form scientific engineering fuzz source var version' ,
     'Error Failure Halt NoValue Syntax NotReady'
   BIFs.0regina = translate(BIFs.0regina, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')
   keywords_regina = ,
     translate(keywords_regina, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')
   keywords2_regina = ,
     keywords2_rexx 'with input normal noEOL output append replace error stream stem LIFO FIFO'
   return 0
   /* === End of Style_Alternate ==================================================================== */

   Task_AddCounter: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  See Script_AddCounter                                                                          */
   /*                                                                                                 */
   /*  Task_AddCounter has examples of several ways you can insert tokens and entire lines            */
   /*  into a file-- In the token stage it adds tokens to expose a new variable in the                */
   /*  'procedure' lines. In the line-oriented stage it adds whole lines to the SrcOut                */
   /*  array, and uses the Text procedure to copy text verbatim from RexxTool.cmd                     */
   /*                                                                                                 */
   /*  This might be hard to understand because some of what looks like Rexx code is                  */
   /*  actually text that will be copied into the new file. That is, anything between                 */
   /*  a "call Text" and an "<ENDTEXT>". It is likely to confuse a syntax highlighter.                */
   /* =============================================================================================== */
   SubID = SubID()
   if Opt.0progress then call Progress SubID

   FoundExit = 0
   FoundProc = 0
   stemN = '!?!ProcCount.'
   ProcList = ''
   LabelN = ''
   t = 1
   do while t < t_type.0
      t = t +1
      select
         when t_type.t == 'REXX_W_KEY' then do
               Tval = translate(t_val.t)
               /*  a RETURN from the top level counts as an EXIT                                      */
               if Tval == 'EXIT' ,
                | (Tval == 'RETURN' ,
                & LabelN == '') then do
                     FoundExit = 1
                     /*  found an exit at token t; insert a call to CountProcedures                   */
                     /*  use O_InsertToken/Line because it is easier than adding each                 */
                     /*  word separately. Getting the tokenizing right does not matter                */
                     /*  for our purposes                                                             */
                     PrevW = translate(T_Val(R_PrevWord(t)))
                     if PrevW == 'THEN' ,
                      | PrevW == 'ELSE' then do
                           /*  change "then exit" to "then do ; call ... ; exit ; end"                */
                           Saved = CL_AsString(t)
                           call CL_Clear t
                           call O_InsertToken t, 'DO ; CALL !?!CountProcedures ; 'Saved' ; END'
                           t_prefix.t = ' '
                        end
                     else
                        call O_InsertLine t, 'CALL !?!CountProcedures'
                     /*  update t to point to 'exit' again                                            */
                     t = t +2
                  end   /*  if EXIT  */
            end   /*  when KEY  */
         when t_type.t == 'REXX_LABEL' then
            if \abbrev(translate(t_val.t), translate(LabelN)||'.') then do
                  LabelN = t_val.t
                  FoundProc = 1
                  if Opt.0progress then call Progress SubID 'Add counter for label' LabelN
                  ProcList = ProcList LabelN
                  IsProc = 0 ; Exposed = 0
                  t2 = R_NextWord(t)
                  if translate(t_val.t2) == 'PROCEDURE' then do
                        IsProc = 1
                        /*  move t to "procedure" because it could be on a separate line from the label */
                        t = t2
                        t2 = R_NextWord(t2)
                        if translate(t_val.t2) == 'EXPOSE' then
                           Exposed = 1
                     end
                  if IsProc then do
                        if \Exposed then do
                              /*  insert EXPOSE after the last word of the line                       */
                              New = L_LastRexxWord(t) +1
                              call O_InsertToken New, 'expose'
                           end
                        /*  insert variable name after the last word of the line                      */
                        New = L_LastRexxWord(t) +1
                        call O_InsertToken New, stemN
                     end   /*  if IsProc  */
                  eol = L_LineEOL(t)
                  /*  insert a line of code to increment counter for this label                       */
                  /*  it is not necessary to insert our new line as a lot of separate tokens because  */
                  /*  no other tasks will need to look at them. Type and class could be anything.     */
                  Command = '!?!LabelN = c2x(translate("'LabelN'")) ;' ,
                            '!?!ProcCount.!?!LabelN = !?!ProcCount.!?!LabelN +1'
                  loc = eol +1
                  call O_InsertLine loc, Command
               end   /*  when Label  */
         otherwise nop
      end   /*  select t_type  */
   end   /*  do while t  */

   if \FoundProc then do
         call Msg SubID 'There is nothing to do for' G.0InfileN 'because there are no labels'
         return 1
      end
   if \FoundExit then
      call Msg SubID 'I could find no exits, you will need to edit 'G.0OutFileN

   ProcList = WordSort(ProcList)
   /*  copy intro text to start of output array                                                       */
   call Out '/'||'*'
   call Text LineNo() +2, '<<ENDTEXT>>'
   /*
   This file has been changed by RexxTool.cmd to count how many times each
   routine is called.
   If RexxTool.cmd is not smart enough to know where your program exits,
   you may need to edit this file. Just before the program exits, there
   should be a line that says
         CALL !?!CountProcedures
   If it is not there, add it.

   When you run the program, it will end by displaying a list (on STDERR)
   of each routine and how many times it was called.  If you want the
   output to go a file, separate from the program's normal STDERR output,
   you can change the line in the !?!CountProcedures procedure at the
   end of this file that says
         FileOut = 'stderr'
   to, for example,
         FileOut = 'ProcedureCount.txt'
   <<ENDTEXT>>
   */
   call Out '*'||'/'
   call Out ''
   call Out '!?!ProcCount. = 0'

   /*  add tokens to the output array                                                                 */
   call Task_Rebuild

   /*  add more text to the end of the array                                                          */
   /*  this gets a little messy because Text() cannot do variables,                                   */
   /*  but using it is easier than trying to escape all the single quotes                             */
   call Out ''
   call Out '/'||'*'||copies('*', 76)||'*'||'/'
   call Out '!?!CountProcedures: procedure expose' stemN
   /*  wrap the Procedure list to avoid a "name or string too long" error                             */
   ProcList = WrapVar(ProcList, 75, 14)
   call Out "   ProcList = '"||ProcList||"'"
   call Text LineNo() +3, '<<ENDTEXT>>'
   /* _ begin Text block ____________________________________________________________________________ */
   /* everything from here to <<ENDTEXT>> is a comment in RexxTool.cmd, but it may confuse a syntax highlighter

   FileOut = 'stderr'
   /* or use a file name:                                                                             */
   /*   FileOut = 'ProcedureCount.txt'                                                                */
   width = 15
   CountList = ''
   sep = '1F'x
   do while ProcList \= ''
      parse var ProcList ProcN ProcList
      !ProcN = c2x(translate(ProcN))

      count = !?!ProcCount.!ProcN
      count = right(count, width, '!')
      CountList = CountList count||sep||ProcN

   end
   /*     sort the list of count:name pairs by count, then reverse the order so                       */
   /*     the display is from highest count to lowest                                                 */
   CountList = wordsort(CountList)
   CountList = !?!ReverseWords(CountList)

   call lineout FileOut, ''
   call lineout FileOut, '==============================================================================='
   call lineout FileOut, 'How many times each routine was called'
   call lineout FileOut, right('count:', width) ' routine name:'
   do while CountList \= ''
      parse var CountList Pair CountList
      parse var Pair count (sep) ProcN
   count = strip(count, 'l', '!')
      call lineout FileOut, right(count, width)||'  '||ProcN
   end
   call lineout FileOut, '==============================================================================='
   call lineout FileOut
   return 0

   !?!ReverseWords:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        list = !?!ReverseWords(list)                                                             */
   /*  where list is a string of space-separated words                                                */
   /*                                                                                                 */
   /*  reverses the order of words in a list                                                          */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*        !?!ReverseWords('Fee Fie Foe Fum')                                                       */
   /*     returns 'Fum Foe Fie Fee'                                                                   */
   /* =============================================================================================== */
   parse arg ListOld

   ListNew = ''
   do while ListOld \= ''
      parse var ListOld aword ListOld
      ListNew = aword ListNew
   end
   return ListNew
   /* === End of !?!ReverseWords ======================================================================= */
   <<ENDTEXT>>
   */
   /* _ end of the Text block _______________________________________________________________________ */
   call Out '/'||'*'||copies('*', 76)||'*'||'/'
   call Out ''

   if Opt.0progress then call Progress
   return 0
   /* === End of Task_AddCounter ==================================================================== */

   Task_ApplyIndent: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Task_Indent and Task_Indent2 calculate the indenting. Task_ApplyIndent applies                 */
   /*  that indenting by changing the prefix of the first token of each line (unless                  */
   /*  the token is marked ReadOnly).                                                                 */
   /* =============================================================================================== */
   if \Opt.0Indent then return 0
   if Opt.0Margin < 0 then do
         call Msg 'Error: bad value for -margin "'Opt.0Margin'"'
         call Egress 'USAGE'
      end

   if Opt.0progress then call Progress SubID()
   do t = 1 to t_type.0
      /*  if token is an EOL, then change prefix of first token in the next line                      */
      if t_class.t == 'FILE_EOL' ,
       |  t_class.t == 'FILE_BOF' then do
            word1 = t +1
            if abbrev(t_class.word1, 'FILE_') then iterate t
            if \t_readonly.word1 then do
                  spaces = t_indent.word1 * Opt.0Tab
                  /*  round to the nearest whole space                                                */
                  spaces = format(spaces, , 0)
                  t_prefix.word1 = copies(' ', spaces)
                  /*  add margin                                                                      */
                  t_prefix.word1 = t_prefix.word1||copies(' ', Opt.0Margin)
               end
         end   /*  if FILE_EOL  */
   end t

   /*  a RexxID comment gets no indent, even if it is read-only                                       */
   t = RexxID()
   if t_type.t == 'COMMENT_BEG' then
      t_prefix.t = ''

   if Opt.0progress then call Progress
   return 0
   /* === End of Task_ApplyIndent =================================================================== */

   Task_Calls: procedure expose (Shared_vars) (all_tokens) SrcOut. BIFs. /*fold00*/
   /* =============================================================================================== */
   /*  used by Script_Calls                                                                           */
   /*                                                                                                 */
   /*  shows a list of routines, what they call and where they are called from                        */
   /* =============================================================================================== */
   /*  LL.0LabelList holds a list of all labels found.                                                */
   /*  for label XXX, calls are recorded in variables named LL.0XXX_Parent and LL.0XXX_Child,         */
   /*  except the XXX is represented as a hex string. See header under GetCalls routine               */
   /*  for an explanation                                                                             */
   /* ----------------------------------------------------------------------------------------------- */
   SubID = SubID()
   if Opt.0progress then call Progress SubID
   if Opt.0Where == '' then Opt.0Where = '*'
   Opt.0Where = ExpandLabels(Opt.0Where)
   if Opt.0Show == 1 then Opt.0Show = ''
   if Opt.0Show = '' then
      Show = 'Int,Ext,BIF'
   else
      Show = Opt.0Show
   Show = translate(Show, ' ', ',')
   Show = translate(Show)

   Tab = copies(' ', 4)
   call GetCalls
   Include = Opt.0Where
   /*  verify that the routines exist, and correct capitalization while we are at it                  */
   incList = Include
   Include = ''
   do while incList \= ''
      parse var incList inc incList
      lpos = wordpos(translate(inc), translate(LL.0LabelList))
      if lpos == 0 then
         call Msg SubID 'Warning: cannot find routine named "'inc'"'
      else
         Include = Include word(LL.0LabelList, lpos)
   end
   Include = strip(Include)

   call Out 'Calls in file: 'G.0InfileN
   if Include \= '' then do
         call Out 'included routines are:'
         call Out WordWrap(Include)
      end
   else do
         call Out 'included routines are: (none)'
         return 0
      end

   call BIFAliases

   Tlist = LL.0LabelList
   do while Tlist \= ''
      parse var Tlist LabelName Tlist
      if Opt.0Where \= '' ,
       & wordpos(translate(LabelName), translate(Include)) == 0 then iterate
      !LabelName = Hexed(LabelName)
      !tFrom = 'LL.0'||!LabelName||'_Parent'
      ParentList = strip(value(!tFrom))
      !tTo = 'LL.0'||!LabelName||'_Child'
      ChildList = strip(value(!tTo))
      call Out ''
      call Out LabelName':'

      if wordpos('INT', Show) > 0 then do
            /*  a routine can only be called from an internal routine, so if INT is                   */
            /*  not to be shown, skip the whole "Called from" display.                                */
            call Out Tab||'Called from:'
            if ParentList == '' then do
                  call Out Tab||Tab||'(never called)'
               end

            do while ParentList \= ''
               parse var ParentList Pname ParentList
               if wordpos(translate(Pname), translate(LL.0LabelList)) > 0 then
                  call Out Tab||Tab||'[Int]  'Pname
               else do
                     call ShowError 'BUG: in' SubID 'Pname should be a label in LL.0LabelList'
                     call Mark 'ParentList Pname LL.0LabelList', ''
                     call Fatality
                  end
            end
         end

      call Out Tab||'Calls to:'
      do while ChildList \= ''
         parse var ChildList Pname ChildList
         select
            when wordpos(translate(Pname), translate(LL.0LabelList)) > 0 then
               if wordpos('INT', Show) > 0 then
                  call Out Tab||Tab||'[Int] 'Pname
            when wordpos(translate(Pname), translate(G.0AllBIFs G.0BifAliases)) > 0 then
               if wordpos('BIF', Show) > 0 then do
                     !FuncN = Hexed(Pname)
                     libname = ''
                     hit = 0
                     alias = ''
                     if BifAlias.!FuncN \== '' then do
                           /*  replace alias with the real BIF name it stands for                     */
                           alias = Pname
                           Pname = BifAlias.!FuncN
                        end
                     do c = 1 to words(BIFs.0used)
                        libname = word(BIFs.0used, c)
                        biflist = value('BIFs.0'||libname)
                        hit = 0
                        if wordpos(translate(Pname), translate(biflist)) > 0 then do
                              hit = 1
                              leave c
                           end
                     end c
                     if hit then
                        if alias \== '' then
                           call Out Tab||Tab||'[BIF]' Pname '(from' libname', registered as' alias')'
                        else
                           if translate(libname) == 'REXX' then
                              call Out Tab||Tab||'[BIF]' Pname
                           else
                              call Out Tab||Tab||'[BIF]' Pname '(from' libname')'
                     else
                        call Out Tab||Tab||'[BIF]' Pname
                  end
            otherwise
               if wordpos('EXT', Show) > 0 then
                  call Out Tab||Tab||'[Ext] 'Pname
         end   /*  select  */
      end   /*  while ChildList  */
   end   /*  while Tlist  */
   if LL.0Handlers \= '' then do
         call Out ''
         call Out 'May also use the error handlers:'
         tempstr = LL.0Handlers
         do while tempstr \= ''
            parse var tempstr word1 tempstr
            call Out Tab||Tab||word1
         end
      end

   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Calls ========================================================================= */

   Task_Change: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Change                                                                                   */
   /*                                                                                                 */
   /*  used by Script_Change                                                                          */
   /*                                                                                                 */
   /*  This task does not respect the token concept. It may replace a token with a                    */
   /*  string that rightfully should be represented by several tokens. There is no                    */
   /*  harm in that if the tokens are just going to be reassembled afterward, as they                 */
   /*  are by the Change command. But don't combine this with tasks that would be                     */
   /*  affected by inaccurate tokenizing.                                                             */
   /* =============================================================================================== */
   if Opt.0Var == '' ,
    & Opt.0Proc == '' then do
         call Msg 'there is nothing to change'
         call Msg 'use -var= or -proc='
         call Egress 'usage'
      end
   SubID = SubID()
   Opt.0Var = translate(Opt.0Var, ' ', ',')
   Opt.0Proc = translate(Opt.0Proc, ' ', ',')
   Opt.0Where = ExpandLabels(Opt.0Where)
   Opt.0Where = translate(Opt.0Where)
   /*  first verify the switches                                                                      */
   do w = 1 to words(Opt.0Var)
      Pair = word(Opt.0Var, w)
      parse var Pair Old '=' New
      select
         when right(Old, 1) == '.' ,
            & right(New, 1) == '.' then nop
         when left(Old, 1) == '.' ,
            & left(New, 1) == '.' then nop
         when pos('.', Old) > 0 ,
            & pos('.', New) > 0 then nop
         when Old == Pair ,
            | Old == '' ,
            | New == '' then do
               call Msg SubID 'I do not understand "'Pair'"'
               call Egress 'USAGE'
            end
         otherwise nop
      end   /*  select  */
   end w

   do w = 1 to words(Opt.0Proc)
      Pair = word(Opt.0Proc, w)
      parse var Pair Old '=' New
      if Old == Pair ,
       | Old == '' ,
       | New == '' then do
            call Msg SubID 'I do not understand "'Pair'"'
            call Egress 'USAGE'
         end
   end w

   call ProcMap
   /* ----------------------------------------------------------------------------------------------- */
   /*  Do the substitution. For each SYMBOL token, parse the var string and see if it                 */
   /*  matches the token value. This makes for a lot of redundant parsing, but doing only             */
   /*  one pass through the tokens means the user can swap variables like                             */
   /*        -vars="Up:Down Down:Up"                                                                  */
   /*  Doing it the other way, "Up" would be changed to "Down" on the first pass and                  */
   /*  changed back to "Up" on the second pass.                                                       */
   if Opt.0progress then call Progress SubID
   Bounders = '"''`%&()*+,-/:;<=>[\]^{}|~ '||'09'x
   if Opt.0Where == '' then
      Include = '[All]'
   else
      Include = Opt.0Where

   do while Include \= ''
      parse var Include ProcN Include
      !ProcN = Hexed(ProcN)
      do t = Map.!ProcN.start to Map.!ProcN.stop
         select
            when wordpos(t_type.t, 'REXX_LABEL REXX_W_PROCEDURE') > 0 then
               do w = 1 to words(Opt.0Proc)
                  Pair = word(Opt.0Proc, w)
                  parse var Pair Old '=' New
                  if Unquoted(translate(t_val.t)) == Unquoted(translate(Old)) then do
                        t_val.t = New
                        G.0FileChanged = 1
                        iterate t
                     end
               end w   /*  end when LABEL or PROCEDURE  */
            when t_type.t == 'REXX_W_SYMBOL' then
               do w = 1 to words(Opt.0Var)
                  Pair = word(Opt.0Var, w)
                  parse var Pair Old '=' New
                  select
                     when right(Old, 1) == '.' ,
                        & right(New, 1) == '.' then do
                           /*  change the stem only                                                   */
                           Old = strip(Old, 't', '.')
                           New = strip(New, 't', '.')
                           n = t +1
                           if t_val.n == '.' ,
                            & t_prefix.n == '' then
                              if translate(t_val.t) == translate(Old) then do
                                    t_val.t = New
                                    G.0FileChanged = 1
                                    iterate t
                                 end
                        end
                     when left(Old, 1) == '.' ,
                        & left(New, 1) == '.' then do
                           /*  change the tail only                                                   */
                           Old = strip(Old, 'l', '.')
                           New = strip(New, 'l', '.')
                           p = t -1
                           if t_val.p == '.' ,
                            & t_prefix.t == '' then
                              if translate(t_val.t) == translate(Old) then do
                                    t_val.t = New
                                    G.0FileChanged = 1
                                    iterate t
                                 end
                        end
                     when pos('.', Old) > 0 ,
                        & pos('.', New) > 0 then do
                           /*  <old> is a stem and the prefix of the tail                             */
                           parse var Old oldstem '.' oldprefix
                           parse var New newstem '.' newprefix
                           if translate(t_val.t) \== translate(oldstem) then iterate t
                           n = t +1
                           nn = t +2
                           if t_val.n \== '.' then iterate t
                           if t_prefix.n \== '' then iterate t
                           if oldprefix == '' ,
                            | left(t_val.nn, length(oldprefix)) == oldprefix then do
                                 /*  it is what we are looking for                                    */
                                 t_val.t = newstem
                                 /*  do not add new prefix if the original has                        */
                                 /*  no tail, as in "expose Opt." or "Opt. = 0"                       */
                                 if right(S_CompoundVar(t), 1) \= '.' then
                                    t_val.nn = newprefix||substr(t_val.nn, length(oldprefix) +1)
                                 G.0FileChanged = 1
                                 iterate t
                              end
                        end   /*  when  */
                     otherwise
                        /*  a simple substitution                                                     */
                        if translate(t_val.t) == translate(Old) then do
                              t_val.t = New
                              G.0FileChanged = 1
                              iterate t
                           end
                  end   /*  select  */
               end w   /*  end when SYMBOL  */
            when t_type.t == 'REXX_W_LITERAL' then do
                  /*  in some cases (like SysFileTree or Value) a variable name can be in quotes.     */
                  /*  also "call mark 'varName1 varName2'"                                            */
                  /*  No need to do a procedure name because it won't be quoted.                      */
                  /*  fix: what about an interpreted string that may contain a procedure name?        */
                  do w = 1 to words(Opt.0Var)
                     Pair = word(Opt.0Var, w)
                     parse var Pair Old '=' New
                     strpos = pos(translate(Old), translate(t_val.t))
                     if strpos > 0 then do
                           /*  Old2 is needed because ChangeString is case-sensitive                  */
                           Old2 = substr(t_val.t, strpos, length(Old))
                           if pos('.', Old2) > 0 then
                              newstr = ChangeString(Old2, t_val.t, New)
                           else
                              newstr = ChangeStringWord(Old2, t_val.t, New, Bounders)
                           if newstr \== t_val.t then do
                                 IsVar = QuotedVar(t)
                                 if IsVar \= '' then do
                                       t_val.t = newstr
                                       G.0FileChanged = 1
                                    end
                                 else do
                                       call Msg 'in literal at line' t_line.t
                                       call Msg Qline(t_line.t)
                                       call Msg 'change'
                                       call Msg G.0Tab||t_val.t
                                       call Msg 'to'
                                       call Msg G.0Tab||newstr
                                       if Confirm('?') then do
                                             t_val.t = newstr
                                             G.0FileChanged = 1
                                          end
                                    end
                              end
                        end
                  end w
               end
            when t_type.t == 'COMMENT_VAL' then do
                  do w = 1 to words(Opt.0Var)
                     Pair = word(Opt.0Var, w)
                     parse var Pair Old '=' New
                     strpos = pos(translate(Old), translate(t_val.t))
                     if strpos > 0 then do
                           /*  Old2 is needed because ChangeString is case-sensitive                  */
                           Old2 = substr(t_val.t, strpos, length(Old))
                           if pos('.', Old2) > 0 then
                              newstr = ChangeString(Old2, t_val.t, New)
                           else
                              newstr = ChangeStringWord(Old2, t_val.t, New, Bounders)
                           if newstr \== t_val.t then do
                                 call Msg 'in comment at line' t_line.t
                                 call Msg Qline(t_line.t)
                                 call Msg 'change'
                                 call Msg G.0Tab||strip(t_val.t)
                                 call Msg 'to'
                                 call Msg G.0Tab||strip(newstr)
                                 if Confirm('?') then do
                                       t_val.t = newstr
                                       G.0FileChanged = 1
                                    end
                              end
                        end
                  end w
                  do w = 1 to words(Opt.0Proc)
                     Pair = word(Opt.0Proc, w)
                     parse var Pair Old '=' New
                     strpos = pos(translate(Old), translate(t_val.t))
                     if strpos > 0 then do
                           /*  Old2 is needed because ChangeString is case-sensitive                  */
                           Old2 = substr(t_val.t, strpos, length(Old))
                           newstr = ChangeStringWord(Old2, t_val.t, New, Bounders)
                           if newstr \== t_val.t then do
                                 call Msg 'in comment at line' t_line.t
                                 call Msg Qline(t_line.t)
                                 call Msg 'change'
                                 call Msg G.0Tab||strip(t_val.t)
                                 call Msg 'to'
                                 call Msg G.0Tab||strip(newstr)
                                 if Confirm('?') then do
                                       t_val.t = newstr
                                       G.0FileChanged = 1
                                    end
                              end
                        end
                  end w
               end   /*  when  */
            otherwise
               iterate t
         end   /*  select  */
      end t
   end   /*  while include  */
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Change ======================================================================== */

   Task_Check: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Check Fix                                                                                */
   /*                                                                                                 */
   /*  This is a container for all the Task_Checkxxx tasks.                                           */
   /*                                                                                                 */
   /*  Options:                                                                                       */
   /*     -check=<list>                    check for errors in <list> categories                      */
   /*     -fix                             fix errors when possible                                   */
   /*                                                                                                 */
   /*     Examples:                                                                                   */
   /*     -check=error,warn,4OS2,style     Check for errors in those categories                       */
   /*     -check=all                       Check for errors in all categories                         */
   /*     -check                           Check for errors in all categories                         */
   /*     -check=style -fix                Check for and fix style errors only                        */
   /*     -check=''                        Do no checking at all                                      */
   /*                                                                                                 */
   /*  See also help for the individual categories, Error, Warn, 4OS2, Style, Expose                  */
   /*                                                                                                 */
   /* =============================================================================================== */
   /* this is a container for all the Task_Checkxxx routines                                          */
   /*  if you write another Task_Checkxxx routine, include a call to it here                          */
   /* =============================================================================================== */
   /*  change "error,warn" to "ERROR WARN"                                                            */
   Opt.0Check = translate(Opt.0Check)
   Opt.0Check = translate(Opt.0Check, ' ', ',')

   ret = Task_CheckError()
   call Task_CheckWarn
   call Task_Check4OS2
   call Task_CheckStyle
   call Task_CheckExpose
   call Task_CheckToker
   return ret
   /* === End of Task_Check ========================================================================= */

   Task_CheckError: procedure expose (Shared_vars) (all_tokens) SrcOut. /*FOLD00*/
   /* =============================================================================================== */
   /*  HELP: Check Error CheckStem unbalanced duplicate "error handler"                               */
   /*                                                                                                 */
   /*    -Check=Error           Check for Rexx errors                                                 */
   /*    -CheckStem=<list>      Check for correct prefix of tails in stem variables                   */
   /*                                                                                                 */
   /*    this task looks for:                                                                         */
   /*                                                                                                 */
   /*       unbalanced quotes                                                                         */
   /*             Including mismatched single/double quotes                                           */
   /*             Bad quoting can confuse the Tokeniser and lead to more error messages               */
   /*                                                                                                 */
   /*       unbalanced DO/END and SELECT/END pairs                                                    */
   /*             It will give a description and approximate location of the error                    */
   /*                                                                                                 */
   /*             It assumes that all of a DO loop is inside one routine, so the                      */
   /*             indenting level will return to 0 by the end of the routine. If it                   */
   /*             does not, you will see a warning. If your writing style involves                    */
   /*             putting labels inside DO loops, this can result in false warnings.                  */
   /*             The symptom is a 'missing END' message followed by an 'unmatched END'               */
   /*             message.                                                                            */
   /*                                                                                                 */
   /*       unbalanced parentheses                                                                    */
   /*                                                                                                 */
   /*       unbalanced comments                                                                       */
   /*             It counts the total number of comment beginning and ending strings                  */
   /*             in the file to make sure they are equal.                                            */
   /*                                                                                                 */
   /*       duplicate labels                                                                          */
   /*             Rexx will not report an error, but it will never see the second label               */
   /*                                                                                                 */
   /*       missing error handler                                                                     */
   /*             like a "SIGNAL ON Syntax" instruction when there is no 'Syntax' label               */
   /*                                                                                                 */
   /*       "==" mistakenly used in an assignment, as in                                              */
   /*             Foo == 1                                                                            */
   /*                                                                                                 */
   /*       comment starting with "--" if the dialect is not Regina                                   */
   /*                                                                                                 */
   /*       missing "BY -1" in a DO loop that counts down, as in                                      */
   /*                   DO a = 5 to 1                                                                 */
   /*             This can cause bugs that are hard to track down-- Rexx does                         */
   /*             not see it as an error, it just ignores the loop.                                   */
   /*             Of course, the start and end values must be numbers, not variables.                 */
   /*                                                                                                 */
   /*       PARSE VAR ... WITH, as in                                                                 */
   /*                   PARSE VAR email WITH user '@' host                                            */
   /*             WITH is only used with PARSE VALUE. That example creates a variable                 */
   /*             named "with", which is probably not what was intended.                              */
   /*                                                                                                 */
   /*       multiple clauses after WHEN without a DO/END around them, as in                           */
   /*                  WHEN a == 1 then                                                               */
   /*                     say 'fee'                                                                   */
   /*                     say 'fie'                                                                   */
   /*                  OTHERWISE                                                                      */
   /*                                                                                                 */
   /*       if the -CheckStem switch is also used, it will verify that certain                        */
   /*       stem variables have tails with the right prefix. For example,                             */
   /*       when formatting RexxTool.cmd I use                                                        */
   /*             -CheckStem=G.0,Opt.0,debug.0,Tkr.0                                                  */
   /*       to verify that all tails of those variables begin with "0"                                */
   /*       and show a warning if one does not                                                        */
   /*                                                                                                 */
   /*  You may want to also use the -warn switch, which causes the tokeniser                          */
   /*  to warn about errors that it finds.                                                            */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  Some of the things this checks for are errors that the Rexx interpreter will detect better.    */
   /*  But this does not require running the script, so I can check for errors as I work.             */
   /*  And it checks the whole file, including parts that may not be executed normally.               */
   /* ----------------------------------------------------------------------------------------------- */
   /* fix: if switch -Warn is used, Toker will duplicate some of this                                 */
   /*      Does Toker warn about anything that is not here?                                           */
   /*         --yes, unexpected character, unexpected End of File, PARSE VALUE without WITH, etc.     */
   /*      I suppose I could capture Toker warnings into an array and display them here               */
   /* =============================================================================================== */
   if Opt.0Check \== 'ALL' ,
    & wordpos('ERROR', Opt.0Check) == 0 then return 0
   SubID = SubID()
   if Opt.0progress then call Progress SubID
   ErrorCount = 0
   WarnCount = 0
   /* ----------------------------------------------------------------------------------------------- */
   /*  check for missing CONTINUATION comma in a compound IF statement                                */
   /*  This only works if logical operator is at start of second line;                                */
   /*  if it is at end of first line, the check below for missing THEN will catch it.                 */
   t = R_EndEOC(1)
   do while t < t_type.0 -2
      w1 = R_NextWord(t)
      if t_type.w1 == 'REXX_W_LOGIC' then
         if \L_IsContinued(t, -1) then do
               call CheckOut SubID 'Error: missing continuation comma at line' t_line.t
               call CheckOut Qline(t_line.t)
               ErrorCount = ErrorCount +1
            end
      t = R_EndEOC(w1)
   end   /*  while t  */
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  check for unbalanced QUOTES                                                                    */
   do t = 2 to t_type.0 -1
      Qerror = 0
      if t_type.t \= 'REXX_W_LITERAL' then iterate
      if left(t_val.t, 1) == "'" ,
       & right(t_val.t, 1) \= "'" then Qerror = 1
      if right(t_val.t, 1) == "'" ,
       & left(t_val.t, 1) \= "'" then Qerror = 1
      if left(t_val.t, 1) == '"' ,
       & right(t_val.t, 1) \= '"' then Qerror = 1
      if right(t_val.t, 1) == '"' ,
       & left(t_val.t, 1) \= '"' then Qerror = 1
      if t_val.t == "'" ,
       | t_val.t == '"' then Qerror = 1
      if Qerror then do
            call CheckOut SubID 'Error: bad quoting in line 't_line.t
            call CheckOut Qline(t_line.t)
            ErrorCount = ErrorCount +1
         end
   end t
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  check for unbalanced PARENTHESES                                                               */
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_EOC' then do
            start = t +1
            if abbrev(t_type.start, 'REXX_W') then do
                  stop = CL_LastWord(start)
                  inpar = 0
                  do v = start to stop
                     select
                        when t_type.v == 'REXX_W_(' then
                           inpar = inpar +1
                        when t_type.v == 'REXX_W_)' then
                           inpar = inpar -1
                        otherwise nop
                     end
                  end v
                  if inpar > 0 then do
                        call CheckOut SubID 'Error: missing ")" at line' t_line.start
                        ErrorCount = ErrorCount +1
                     end
                  if inpar < 0 then do
                        call CheckOut SubID 'Error: missing "(" at line' t_line.start
                        ErrorCount = ErrorCount +1
                     end
               end   /*  if abbrev()  */
         end   /*  if REXX_EOC  */
   end t
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  check for unmatched DO/END or SELECT/END                                                       */
   /*                                                                                                 */
   /*  This assumes that there will never be a label inside a DO loop, so the indenting level         */
   /*  will be 0 at the end of each routine.  That is usually a safe assumption, but I have           */
   /*  seen a very few cases of jumping to a label from another point inside the loop, or             */
   /*  even from outside the loop. That will trigger two false warnings from this section--           */
   /*  one about a missing END and then a warning about an unmatched END.                             */
   /* ----------------------------------------------------------------------------------------------- */
   /*  the misleadingly named inDo variable shows what kind of structure the current                  */
   /*  token is in-- DO, SELECT, or WHEN. The Stack variable is a list of nested                      */
   /*  inDo values. Warning and error count for an unmatched END is under the line that says          */
   /*  "when Tval == 'END'". Warning and error count for missing END is at the end of                 */
   /*  the loop. Other warnings are just trying to be helpful by narrowing down where                 */
   /*  the error is.                                                                                  */
   /* ----------------------------------------------------------------------------------------------- */
   t_from = 2
   t_to = 0
   LabelN = '[Unlabeled]'
   do while t_to < t_type.0
      if debug.0loop then call LoopCounter
      t_to = R_NextLabel(t_from)
      Stack = ''
      inDo = ''
      do t = t_from to t_to
         if t_type.t = 'REXX_W_KEY' then do
               if inDo = '' then Last0 = t_line.t
               Tval = translate(t_val.t)
               select
                  when Tval == 'DO' then do
                        Stack = inDo Stack
                        inDo = Tval
                     end

                  when Tval == 'SELECT' then do
                        Stack = inDo Stack
                        inDo = Tval
                     end

                  when Tval == 'WHEN' ,
                     | Tval == 'OTHERWISE' then do
                        select
                           when inDo == 'WHEN' then
                              /*  close out the previous WHEN                                         */
                              parse var Stack inDo Stack
                           when inDo == 'SELECT' then
                              /*  this is what it should be for the first WHEN                        */
                              nop
                           when inDo == 'DO' then do
                                 /*  Errorcount for missing/unmatched END will be at end of loop      */
                                 if word(Stack, 1) == 'WHEN' then do
                                       /* a missing END in the previous WHEN                          */
                                       call CheckOut SubID ,
                                            'Error: unexpected' Tval 'at line' t_line.t 'under label' LabelN
                                       call CheckOut G.0Tab'probably a missing END at line' t_line.t -1
                                       /*  close out the unclosed DO loop                             */
                                       parse var Stack inDo Stack
                                    end
                                 else do
                                       /* probably an unmatched END in the previous WHEN              */
                                       call CheckOut SubID ,
                                            'Error: unexpected' Tval 'at line' t_line.t 'under label' LabelN
                                       call CheckOut G.0Tab'probably an unmatched END somewhere before this line'
                                    end
                                 /*  inDo should now be 'WHEN'                                        */
                              end
                           when inDo = '' then do
                                 call CheckOut SubID ,
                                      'Error:' Tval 'without a SELECT at line' t_line.t 'under label' LabelN
                                 ErrorCount = ErrorCount +1
                              end
                           otherwise
                              call ShowError 'BUG: invalid value for inDo:' inDo
                              call Mark 'inDo Stack Tval t_line'
                              call Fatality
                        end   /*  select  */
                        Stack = inDo Stack
                        inDo = 'WHEN'
                     end   /*  when Tval == WHEN  */

                  when Tval == 'END' then do
                        select
                           when inDo == '' then do
                                 call CheckOut SubID ,
                                      'Error: unmatched END at line' t_line.t 'under label' LabelN
                                 ErrorCount = ErrorCount +1
                              end
                           when inDo == 'WHEN' then do
                                 /*  close out the WHEN or OTHERWISE                                  */
                                 parse var Stack inDo Stack
                                 /*  close out the SELECT                                             */
                                 parse var Stack inDo Stack
                              end
                           when inDo == 'DO' then do
                                 /* close out the DO                                                  */
                                 parse var Stack inDo Stack
                              end
                           when inDo == 'SELECT' then
                              /* an error should have already been shown                              */
                              parse var Stack inDo Stack
                           otherwise
                              call ShowError 'BUG: invalid value for inDo:' inDo
                              call Mark 'inDo Stack Tval t_line'
                              call Fatality
                        end   /*  select  */
                     end   /*  when END  */

                  otherwise nop
               end   /*  select  */
            end   /*  if a keyword  */
      end t
      if inDo \= '' then do
            call CheckOut SubID 'Error: missing END somewhere after line' Last0 'under label' LabelN
            ErrorCount = ErrorCount +1
         end
      t_from = t_to
      LabelN = t_val.t_from
   end   /*  while t_to < t_type.0  */
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  check for unbalanced COMMENT                                                                   */
   /*  The Toker will insert a dummy COMMENT_BEG or COMMENT_END with a null value for                 */
   /*  the convenience of formatting tasks that expect a complete comment. Which means                */
   /*  comments are never actually unbalanced. But for our purposes we ignore the dummies.            */
   incomment = 0
   Cbeg = ''
   do t = 2 to t_type.0 -1
      if incomment == 0 then RexxLine = t_line.t
      select
         when t_type.t == 'COMMENT_BEG' then do
               if t_val.t \== '' then incomment = incomment +1
               Cbeg = t_val.t
            end
         when t_type.t == 'COMMENT_END' then do
               if t_val.t == '*'||'/' ,
                | Cbeg == '--' then incomment = incomment -1
               if incomment < 0 then do
                     call CheckOut SubID 'Error: unmatched End Of Comment at line 't_line.t
                     call CheckOut Qline(t_line.t)
                     ErrorCount = ErrorCount +1
                     incomment = 0
                  end
            end
         otherwise nop
      end   /*  select  */
   end t
   if incomment > 0 then do
         call CheckOut SubID 'Error: missing End Of Comment at or after line 'RexxLine
         call CheckOut Qline(RexxLine)
         ErrorCount = ErrorCount +1
      end
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  check for wrong kind of COMMENT                                                                */
   if G.0dialect \= 'REGINA' then
      do t = 2 to t_type.0 -1
         if t_val.t == '--' then
            if t_type.t == 'COMMENT_BEG' then do
                  call CheckOut SubID 'Error: Regina or Object Rexx comment at line' t_line.t
                  call CheckOut Qline(t_line.t)
                  ErrorCount = ErrorCount +1
               end
      end t

   /* ----------------------------------------------------------------------------------------------- */
   do t = 2 to t_type.0 -1
      /* -------------------------------------------------------------------------------------------- */
      /*  Object Rexx DIRECTIVE or 4OS2 comment                                                       */
      if t_type.t == 'COLON2' then do
            call CheckOut SubID ,
                 'Error: line' t_line.t 'looks like Object Rexx, or a 4OS2 comment'
            call CheckOut Qline(t_line.t)
            ErrorCount = ErrorCount +1
         end
      /* -------------------------------------------------------------------------------------------- */
      /*  "==" used in an ASSIGNMENT                                                                  */
      if t_val.t == '==' ,
       & t_type.t == 'REXX_W_ASSIGN' then do
            call CheckOut SubID 'Error: "==" used for assignment at line' t_line.t
            call CheckOut Qline(t_line.t)
            call CheckOut G.0Tab'this might also be caused by something that confused the Tokeniser'
            ErrorCount = ErrorCount +1
         end
      /* -------------------------------------------------------------------------------------------- */
   end t

   /* ----------------------------------------------------------------------------------------------- */
   /*  check for duplicate labels                                                                     */
   LabelList = ''
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_LABEL' then do
            if wordpos(translate(t_val.t), LabelList) > 0 then do
                  call CheckOut SubID 'Error: duplicate label "'t_val.t'" at line 't_line.t
                  ErrorCount = ErrorCount +1
               end
            else
               LabelList = LabelList translate(t_val.t)
         end
   end t
   /* ----------------------------------------------------------------------------------------------- */
   /*  check for missing object of "signal on" or "call on" instruction                               */
   /*  Note that this depends on the Label list created in the check for duplicate labels above       */
   do t = 2 to t_type.0 -1
      Tval = R_KeyVal(t)
      if Tval == 'SIGNAL' ,
       | Tval == 'CALL' then do
            w2 = R_NextWord(t)
            if translate(t_val.w2) == 'ON' then do
                  wlast = CL_LastWord(t)
                  if wordpos(translate(t_val.wlast), LabelList) == 0 then do
                        call CheckOut SubID 'Error: line' t_line.t 'refers to a non-existent label "'t_val.wlast'"'
                        call CheckOut Qline(t_line.t)
                        ErrorCount = ErrorCount +1
                     end
               end
         end
   end t
   /* ----------------------------------------------------------------------------------------------- */
   /*  missing BY in a DO loop                                                                        */
   /*  I sometimes type                                                                               */
   /*        DO foo = 5 to 1                                                                          */
   /*  which means                                                                                    */
   /*        DO foo = 5 to 1 by 1                                                                     */
   /*  Rexx will not complain, it will just not execute the loop because the default                  */
   /*  increment value is +1 and 5 is already greater than 1. The line needs to be                    */
   /*        DO foo = 5 to 1 by -1                                                                    */
   /*                                                                                                 */
   /*   Of course this will not detect an error if the start or stop values are variables.            */
   do t = 2 to t_type.0 -1
      if t_type.t \== 'REXX_W_KEY' then iterate t
      if translate(t_val.t) \== 'DO' then iterate t
      wvar = R_NextWord(t)
      if t_type.wvar \== 'REXX_W_SYMBOL' then iterate t
      wvar = R_Tail(wvar)
      wOP = R_NextWord(wvar)
      if t_val.wOP \== '=' then iterate t
      wstart = R_NextWord(wOP)
      wTo = R_NextWord(wstart)
      wstop = R_NextWord(wTo)
      wBy = R_NextWord(wstop)
      if translate(t_val.wTo) \== 'TO' then iterate t
      if translate(t_val.wBy) == 'BY' then iterate t
      if t_type.wstart == 'REXX_W_CONSTANT' ,
       & t_type.wstop == 'REXX_W_CONSTANT' then
         if t_val.wstart > t_val.wstop then do
               call CheckOut SubID 'Error: DO loop will never run because of missing BY <negative number>'
               call CheckOut Qline(t_line.t)
               ErrorCount = ErrorCount +1
            end
   end t
   /* ----------------------------------------------------------------------------------------------- */
   /*  missing THEN seems to be a common mistake; probably a CMD.EXE or 4OS2 habit.                   */
   t = 1
   do while t < t_type.0 -1
      t = t +1
      if t_type.t \== 'REXX_W_KEY' then iterate
      if translate(t_val.t) == 'IF' ,
       | translate(t_val.t) == 'WHEN' then do
            i = t
            do forever
               i = R_NextToken(i)
               select
                  when i >= t_type.0 then do
                        call CheckOut SubID 'Error: missing THEN at line 't_line.t
                        call CheckOut Qline(t_line.t)
                        ErrorCount = ErrorCount +1
                        leave   /*  do forever  */
                     end
                  when t_type.i == 'REXX_EOC' then do
                        /*  note that j is the next word, not the next token--                        */
                        j = R_NextWord(i)
                        select
                           when translate(t_val.j) == 'THEN' then do
                                 /*  as in "IF a THEN DO Something"                                   */
                                 t = j
                                 leave   /*  do forever  */
                              end
                           when t_type.j == 'REXX_W_LOGIC' then do
                                 /*  as in "IF a & b THEN DO Something"                               */
                                 t = j
                                 leave   /*  do forever  */
                              end
                           otherwise
                              /*  as in "IF a DO Something"                                           */
                              call CheckOut SubID 'Error: missing THEN or continuation comma at line 't_line.t
                              call CheckOut Qline(t_line.t)
                              ErrorCount = ErrorCount +1
                              leave   /*  do forever  */
                        end   /*  select  */
                     end
                  otherwise nop
               end
            end   /*  do forever  */
         end   /*  when IF or WHEN  */
   end   /*  while t  */
   /* ----------------------------------------------------------------------------------------------- */
   /*  check for multiple clauses after WHEN ... THEN                                                 */
   do t = 2 to t_type.0 -1
      if t_type.t \== 'REXX_W_KEY' then iterate t
      if translate(t_val.t) == 'WHEN' then do
            w1 = CL_FirstWord(t, +1)
            w2 = CL_FirstWord(t, +2)
            w3 = CL_FirstWord(t, +3)
            if translate(t_val.w1) == 'THEN' then do
                  if wordpos(translate(t_val.w2), 'DO IF') == 0 then do
                        /*  WHEN/THEN without DO or IF, so next clause +1 should be WHEN or OTHERWISE */
                        if wordpos(translate(t_val.w3), 'WHEN OTHERWISE') == 0 then
                           if translate(t_val.w3) \== 'END' then do
                                 call CheckOut SubID 'Error: at line' t_line.w3 'expected WHEN or OTHERWISE'
                                 call CheckOut G.0Tab'probably THEN on line' t_line.w1 'needs a DO/END after it'
                                 call CheckOut Qline(t_line.w1)
                                 if t_line.w2 > t_line.w1 then
                                    call CheckOut Qline(t_line.w2)
                                 if t_line.w3 > t_line.w2 then
                                    call CheckOut Qline(t_line.w3)
                                 ErrorCount = ErrorCount +1
                              end
                     end
               end
         end
   end t
   /* ----------------------------------------------------------------------------------------------- */
   /*  check tails of stem variables                                                                  */
   if Opt.0CheckStem \= '' then do
         Opt.0CheckStem = translate(Opt.0CheckStem, ' ', ',')
         do w = 1 to words(Opt.0CheckStem)
            wordw = word(Opt.0CheckStem, w)
            parse var wordw stem '.' pre .
            if pre == '' then do
                  call CheckOut SubID 'Error: bad value for -checkstem switch "'wordw'"'
                  iterate w
               end
            do t = 2 to t_type.0 -1
               if t_type.t \== 'REXX_W_SYMBOL' then iterate t
               if translate(t_val.t) \== translate(stem) then iterate t
               t1 = t +1
               t2 = t +2
               if t_val.t1 \== '.' then iterate t
               if t_type.t2 \== 'REXX_W_SYMBOL' then iterate t
               if t_prefix.t2 \== '' then iterate t

               if \abbrev(t_val.t2, pre) then do
                     call CheckOut SubID 'Error: at line' t_line.t', missing "'pre'" in "'S_CompoundVar(t)'"'
                     ErrorCount = ErrorCount +1
                  end
            end t
         end w
      end
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  check for PARSE VAR <variable> WITH ...                                                        */
   /*  Do not try to fix it because we cannot be sure if                                              */
   /*        PARSE VAR email WITH '@' host                                                            */
   /*  is intended to be                                                                              */
   /*        PARSE VAR email user '@' host                                                            */
   /*  or                                                                                             */
   /*        PARSE VAR email . '@' host                                                               */
   /*  or                                                                                             */
   /*        there really is a variable called "with"                                                 */
   Warned = 0
   t = 1
   do while t < t_type.0 -1
      t = t +1
      if t_type.t \== 'REXX_W_KEY' then iterate
      if translate(t_val.t) == 'PARSE' then do
            t = R_NextWord(t)
            if translate(t_val.t) \== 'VAR' then iterate
            do while t_type.t == 'REXX_W_KEY2'
               t = R_NextWord(t)
            end
            /*  found the variable name, get first word of template                                   */
            t = R_NextWord(t)
            if translate(t_val.t) == 'WITH' then do
                  WarnCount = WarnCount +1
                  call CheckOut SubID 'Warning: PARSE VAR ... WITH at line' t_line.t
                  call CheckOut Qline(t_line.t)
                  if \Warned then do
                        call CheckOut G.0Tab'WITH is only used with PARSE VALUE.'
                        call CheckOut G.0Tab'PARSE VAR ... WITH creates a variable called "with",'
                        call CheckOut G.0Tab'which is probably not what you intended.'
                        Warned = 1
                     end
               end
         end   /*  if PARSE  */
   end   /*  do while t  */

   if Opt.0progress then call Progress
   if ErrorCount + WarnCount > 0 ,
    | Opt.0verbose then
      call CheckOut left(SubID, 20) 'Errors/Warnings:' ErrorCount'/'WarnCount
   return ErrorCount
   /* === End of Task_CheckError ==================================================================== */

   Task_CheckWarn: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Check Warn Fix                                                                           */
   /*                                                                                                 */
   /*    -Check=Warn           Warn about things that may not be legal or may not                     */
   /*                          be portable,                                                           */
   /*                                                                                                 */
   /*    items labeled (Fixable) can be fixed by the formatter if you use the -fix switch             */
   /*                                                                                                 */
   /*    this task looks for:                                                                         */
   /*                                                                                                 */
   /*       (Fixable) CALL with parentheses                                                           */
   /*             A line like                                                                         */
   /*                   CALL Foo(1)                                                                   */
   /*             Is technically illegal. Some Rexx interpreters will tolerate it,                    */
   /*             but only when there is exactly one argument. This would be better:                  */
   /*                   CALL Foo 1                                                                    */
   /*                                                                                                 */
   /*       (Fixable) Unneeded continuation comma, as in                                              */
   /*                   IF a == 1,                                                                    */
   /*                   THEN                                                                          */
   /*             Rexx will tolerate it, but it could slightly confuse the formatter                  */
   /*                                                                                                 */
   /*       Broken continuation, as in                                                                */
   /*                   SAY Fee ,                                                                     */
   /*                     /@ Fie , @/                                                                 */
   /*                     Foe                                                                         */
   /*             commenting out the second line will not work as intended, because the               */
   /*             End-Of-Line after the comment stops the continuation. If you want to                */
   /*             comment out one continued line, you need to leave the comma outside                 */
   /*             the comment, like this:                                                             */
   /*                   SAY Fee ,                                                                     */
   /*                     /@ Fie @/ ,                                                                 */
   /*                     Foe                                                                         */
   /*                                                                                                 */
   /*       (Fixable) Trailing comma in a function call, like                                         */
   /*                   SAY Foo(a,)                                                                   */
   /*             It is supposed to be illegal in Rexx BIFs, though Rexx will usually                 */
   /*             tolerate it. It's better to be safe and change it to                                */
   /*                   SAY Foo(a)                                                                    */
   /*                                                                                                 */
   /*       Assignment of a special variable, also known as an internal or reserved variable,         */
   /*       like RC, RESULT, SIGL. They are assigned values by Rexx; you should not use               */
   /*       those names for your variables.                                                           */
   /*                                                                                                 */
   /*                                                                                                 */
   /* =============================================================================================== */
   if Opt.0Check \== 'ALL' ,
    & wordpos('WARN', Opt.0Check) == 0 then return 0
   SubID = SubID()
   if Opt.0progress then call Progress SubID
   WarnCount = 0
   /* ----------------------------------------------------------------------------------------------- */
   /*  check for CALL <procedure> with PARENTHESES                                                    */
   /*  OS/2 Rexx does tolerate parentheses, but only if there is one argument.                        */
   /*  I am not sure about other Rexx interpreters, but it is worth a warning.                        */
   t = 1
   do while t < t_type.0 -1
      t = t +1
      if t_type.t == 'REXX_W_PROCEDURE' then do
            if translate(T_Val(R_PrevWord(t))) == 'CALL' then do
                  Lpar = R_NextWord(t)
                  if t_type.Lpar == 'REXX_W_(' then
                     if Opt.0Fix then do
                           Rpar = R_MatchPar(Lpar)
                           Arg1 = R_NextToken(Lpar)
                           if Rpar > Lpar then do
                                 if t_val.Arg1 \== '' then
                                    t_prefix.Arg1 = ' '
                                 call T_DeleteToken Rpar
                                 call T_DeleteToken Lpar
                                 if Opt.0verbose then
                                    call CheckOut SubID 'remove parentheses at line' t_line.t
                              end
                           else do
                                 call ShowError SubID 'Error matching parenthesis at line' t_line.t
                                 call ShowError Qline(t_line.t), ''
                              end
                        end   /*  if Opt.0Fix  */
                     else do
                           call CheckOut SubID 'warning: CALL used with () at line 't_line.t
                           call CheckOut Qline(t_line.t)
                           WarnCount = WarnCount +1
                        end
               end   /*  if call  */
         end   /*  if procedure  */
   end   /*  do while t  */
   /* ----------------------------------------------------------------------------------------------- */
   /*  check for a TRAILING COMMA in a function call without a last argument.                         */
   /*  Apparently it is technically illegal in Rexx BIFs but often tolerated.                         */
   /*  Sounds like a good thing to avoid.                                                             */
   t = 1
   do while t < t_type.0 -1
      t = t +1
      if t_type.t == 'REXX_W_COMMA' then
         if T_Type(R_NextWord(t)) == 'REXX_W_)' then
            if Opt.0Fix then do
                  if Opt.0verbose then
                     call CheckOut SubID 'remove trailing comma in function arguments, line' t_line.t
                  call T_DeleteToken t
                  /*  move t back because the previous word may now be a trailing comma               */
                  t = R_PrevWord(t) -1
               end
            else do
                  call CheckOut SubID 'Warning: trailing comma in function arguments at line' t_line.t
                  call CheckOut Qline(t_line.t)
                  WarnCount = WarnCount +1
               end
   end   /*  do while t  */
   /* ----------------------------------------------------------------------------------------------- */
   /*  Check for UNNEEDED CONTINUATION COMMA:                                                         */
   /*  In this:                                                                                       */
   /*        IF a ,                                                                                   */
   /*           THEN DO                                                                               */
   /*  the comma is legal, but not needed.                                                            */
   /*  It can affect the indenting of THEN DO, because there will be a hidden                         */
   /*  EOC token before THEN.                                                                         */
   /*                                                                                                 */
   /*                                                                                                 */
   /*  Check for BROKEN CONTINUATION                                                                  */
   /*  In this example, trying to comment out one line of a continuation will not work                */
   /*  as intended, because the line-ending after the comment ends the continuation:                  */
   /*           SAY a ,                                                                               */
   /*             /@  b ,  @/                                                                         */
   /*             c                                                                                   */
   Warned = 0
   t = 1
   do while t < t_type.0 -1
      t = t +1
      if t_type.t == 'NULL' then do
            select
               when T_Type(R_NextWord(t)) == 'REXX_W_KEY' ,
                  | R_AtAssignment(R_NextWord(t)) then do
                     if Opt.0Fix then do
                           /*  find the preceding continuation comma.                                 */
                           /*  R_PrevToken() will not work here because it ignores continuation commas. */
                           do comma = t to L_FirstToken(t) by -1
                              if t_type.comma == 'CONTINUE' then leave comma
                           end comma
                           if t_type.comma == 'CONTINUE' then do
                                 t_type.t = 'REXX_EOC'
                                 if Opt.0verbose then
                                    call CheckOut SubID 'remove unneeded continuation comma at line' t_line.comma
                                 call T_DeleteToken comma
                                 /*  move t back because we deleted a token before it                 */
                                 t = t -1
                              end
                           else do
                                 call ShowError SubID 'Error finding continuation comma'
                                 call ShowError Qline(t_line.t)
                                 call Mark 't t_type.t comma t_val.comma t_type.comma', ''
                                 call ShowError 'This could be a bug in the Tokeniser or in' SubID, ''
                                 call Fatality
                              end
                        end   /*  end if Opt.0Fix  */
                     else do
                           call CheckOut SubID 'Warning: unneeded continuation comma at line' t_line.t
                           call CheckOut Qline(t_line.t)
                           WarnCount = WarnCount +1
                           if \Warned  then do
                                 call CheckOut 'It may affect the indenting'
                                 Warned = 1
                              end
                        end
                  end
               when \L_HasRexx(t, +1) then do
                     call CheckOut SubID 'Warning: line' t_line.t +1 'will break the continued line above it'
                     do c = 0 to 2 by 1
                        call CheckOut ' ' right(t_line.t + c, 5)':   ' L_AsString(t, c)
                     end
                     WarnCount = WarnCount +1
                  end
               otherwise nop
            end   /*  select  */
         end
   end
   /* ----------------------------------------------------------------------------------------------- */
   /*  check for assignment of a RESERVED VARIABLE NAME                                               */
   do t = 1 to t_type.0 -1
      if t_type.t == 'REXX_W_SYMBOL' then
         if wordpos(translate(t_val.t), 'SIGL RESULT RC') > 0 then
            if R_AtAnyAssignment(t) then do
                  call CheckOut 'Warning: assignment of internal variable' translate(t_val.t) 'at line' t_line.t
                  WarnCount = WarnCount +1
               end
   end t
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0progress then call Progress
   if WarnCount > 0 ,
    | Opt.0verbose then
      call CheckOut left(SubID, 20) 'Warnings:' WarnCount
   return WarnCount
   /* === End of Task_CheckWarn ===================================================================== */

   Task_Check4OS2: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Check 4OS2                                                                               */
   /*                                                                                                 */
   /*    this task looks for:                                                                         */
   /*                                                                                                 */
   /*       CMD.EXE keywords, 4OS2 keywords, backquotes, and 4OS2 logic I might use by                */
   /*       mistake. Like 'IFF', 'IF NOT', 'SET Foo =', 'ECHO', etc.                                  */
   /*                                                                                                 */
   /* =============================================================================================== */
   if Opt.0Check \== 'ALL' ,
    & wordpos('4OS2', Opt.0Check) == 0 then return 0
   SubID = SubID()
   if Opt.0progress then call Progress SubID
   ErrorCount = 0
   WarnCount = 0

   /* ----------------------------------------------------------------------------------------------- */
   /*  4OS2 KEYWORDS:                                                                                 */
   /*  as in "IFF A == B THEN"                                                                        */
   /*  This assumes that "IFF" is a mistake for "IF".                                                 */
   /*  Task_CheckStyle would also throw a warning, on the assumption that "IFF" is                    */
   /*  an unquoted system command.                                                                    */
   Bad = 'SET UNSET IFF ENDIFF ELSEIFF ECHO ECHOERR CANCEL GOSUB REM ON'
   do t = 2 to t_type.0 -1
      /*  An invalid keyword will not be flagged as a keyword, so identify it                         */
      /*  by looking for the next token after EOC                                                     */
      if T_Type(t -1) == 'REXX_EOC' then do
            if t_type.t \== 'REXX_W_SYMBOL' then iterate t
            if wordpos(translate(t_val.t), Bad) > 0 then do
                  if \R_AtAssignment(t) then do
                        call CheckOut SubID 'Error: 4OS2 / CMD.EXE keyword "'t_val.t'" at line' t_line.t
                        call CheckOut Qline(t_line.t)
                        ErrorCount = ErrorCount +1
                     end
               end
         end
   end t
   /* ----------------------------------------------------------------------------------------------- */
   /*  IF NOT                                                                                         */
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_W_KEY' then do
            keyW = t
            nextW = R_NextWord(t)
            select
               /*  "IF NOT" instead of "IF \"                                                         */
               when translate(t_val.keyW) == 'IF' ,
                  & translate(t_val.nextW) == 'NOT' then do
                     call CheckOut SubID 'Error: 4OS2 or CMD.EXE keyword "'t_val.nextW'" at line' t_line.nextW
                     call CheckOut Qline(t_line.nextW)
                     ErrorCount = ErrorCount +1
                  end
               /*  moved ELSE IF to Task_CheckStyle                                                   */
               otherwise nop
            end   /*  select  */
         end   /*  if REXX_W_KEY  */
   end t
   /* ----------------------------------------------------------------------------------------------- */
   /*  check for 4OS2-style BACKQUOTES                                                                */
   t = 1
   do while t < t_type.0 -1
      t = t +1
      if t_type.t == 'REXX_W_`' then do
            call CheckOut SubID 'Error: backquote at line' t_line.t
            call CheckOut Qline(t_line.t)
            ErrorCount = ErrorCount +1
            /*  only need to show message once per line                                               */
            do while t_type.t \== 'REXX_EOC'
               t = t +1
            end
         end
   end   /*  while t  */
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0progress then call Progress
   if ErrorCount + WarnCount > 0 ,
    | Opt.0verbose then
      call CheckOut left(SubID, 20) 'Errors/Warnings:' ErrorCount'/'WarnCount
   return ErrorCount
   /* === End of Task_Check4OS2 ===================================================================== */

   Task_CheckStyle: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Check Style Fix "else if" otherwise                                                      */
   /*                                                                                                 */
   /*        -Check=Style           show suggestions about writing style                              */
   /*                                                                                                 */
   /*    Items labeled (Fixable) can be fixed by the formatter if you use the -fix switch             */
   /*                                                                                                 */
   /*    This task looks for:                                                                         */
   /*                                                                                                 */
   /*       * Comment begin/end strings inside quotes, represented here as /@ and @/.                 */
   /*             They are legal-- the quotes protect them from being interpreted                     */
   /*             as the beginning or end of a comment. So you can write                              */
   /*                   SAY '/@'                                                                      */
   /*             But if sometime later you comment out that line:                                    */
   /*                   /@ SAY '/@' @/                                                                */
   /*             Rexx will ignore the quotes, so the /@ is interpreted as the beginning              */
   /*             of a nested comment. That can result in strange Rexx errors farther                 */
   /*             along in the program. Since any line could be commented out for some                */
   /*             reason, it is not really safe to rely on the quoting.                               */
   /*             A safer alternative is                                                              */
   /*                   SAY '/'||'@'                                                                  */
   /*                                                                                                 */
   /*                                                                                                 */
   /*       * (Fixable) Unneeded ending ";". As in                                                    */
   /*                   say 'Yes' ;                                                                   */
   /*                                                                                                 */
   /*                                                                                                 */
   /*       * Multiple clauses on one line after THEN. This is misleading:                            */
   /*                   IF a THEN                                                                     */
   /*                       b = 1 ; c = 1 ; d = 1                                                     */
   /*             That does not do what it looks like, because THEN only applies to                   */
   /*             the first clause after it, not the whole line. Either group the                     */
   /*             assignments with a DO/END, or make it clearer by writing                            */
   /*                   IF a THEN                                                                     */
   /*                       b = 1                                                                     */
   /*                   c = 1                                                                         */
   /*                   d = 1                                                                         */
   /*                                                                                                 */
   /*                                                                                                 */
   /*       * (Fixable) ELSE IF:                                                                      */
   /*             The only thing wrong with ELSE IF is that it leads people to misunderstand          */
   /*             how Rexx works, especially if they are used to the ELSEIFF keyword in 4OS2.         */
   /*             In Rexx, ELSE IF is not a keyword, it is an ELSE followed by another IF test.       */
   /*             The Rexx documentation says                                                         */
   /*                "ELSE binds to the nearest IF at the same level"                                 */
   /*             In other words,                                                                     */
   /*                ==> an IF can have only one ELSE.                                                */
   /*                ==> IF is not a multiple-choice test                                             */
   /*                                                                                                 */
   /*             Here is an example of an easy mistake to make. It does not do what the              */
   /*             indenting implies:                                                                  */
   /*                                                                                                 */
   /*                    a = 3                                                                        */
   /*                    b = 2                                                                        */
   /*                    if a == 0 then                                                               */
   /*                       say 0                                                                     */
   /*                    else if a == 1 then                                                          */
   /*                       say 'a is 1'                                                              */
   /*                    else if a == 2 then                                                          */
   /*                       if b == 2 then                                                            */
   /*                          say 'a is 2, b is 2'                                                   */
   /*                    else if a == 3 then                                                          */
   /*                       say 'a is 3'            /@ this line will never execute @/                */
   /*                    else                                                                         */
   /*                       say 'a > 3'                                                               */
   /*                                                                                                 */
   /*            Using the -fix switch to put the IF on a new line and correct the                    */
   /*            indenting makes the logic clearer and shows why it does not work as                  */
   /*            intended:                                                                            */
   /*                                                                                                 */
   /*                   a = 3                                                                         */
   /*                   b = 2                                                                         */
   /*                   if a == 0 then                                                                */
   /*                      say 0                                                                      */
   /*                   else                                                                          */
   /*                      if a == 1 then                                                             */
   /*                         say 'a is 1'                                                            */
   /*                      else                                                                       */
   /*                         if a == 2 then                                                          */
   /*                            if b == 2 then                                                       */
   /*                               say 'a is 2, b is 2'                                              */
   /*                            else                                                                 */
   /*                               if a == 3 then                                                    */
   /*                                  say 'a is 3'  /@ this line will never execute @/               */
   /*                               else                                                              */
   /*                                  say 'a > 3'                                                    */
   /*                                                                                                 */
   /*            You could fix it by inserting an "ELSE nop" after the nested IF, or                  */
   /*            in this case a SELECT structure would be simpler                                     */
   /*                                                                                                 */
   /*                                                                                                 */
   /*       * External (System) commands that are not quoted. As in:                                  */
   /*                   dir '*.cmd'                                                                   */
   /*            If Rexx does not recognize "dir" as a Rexx command it will evaluate                  */
   /*            it as a variable, then pass the resulting string to the system                       */
   /*            command processor (shell). If "dir" is not a variable (and you do not                */
   /*            use 'signal on NoValue') it will evaluate to "DIR". Since the OS/2                   */
   /*            command processor is not case-sensitive, it executes the 'dir' command.              */
   /*            But if there is a variable named "dir", the command processor                        */
   /*            will not see the command that you intended. If the word should not be                */
   /*            evaluated as a variable, it is safer to quote it:                                    */
   /*                 'dir *.cmd'                                                                     */
   /*                                                                                                 */
   /*            There could be some warnings that you do not want. For instance, this                */
   /*            is one way to execute the 'dir' command:                                             */
   /*                ToExec = 'dir'                                                                   */
   /*                ToExec '*.cmd'                                                                   */
   /*            The second line will result in an 'unquoted system command' warning                  */
   /*            from this task, because RexxTool.cmd is not smart enough to know                     */
   /*            that "ToExec" is really a variable to be evaluated.                                  */
   /*            A clearer way to write it in OS/2 Rexx might be:                                     */
   /*                ToExec = 'dir'                                                                   */
   /*                ADDRESS CMD ToExec '*.cmd'                                                       */
   /*            The Regina equivalent would be:                                                      */
   /*                ADDRESS SYSTEM ToExec '*.cmd'                                                    */
   /*                                                                                                 */
   /*                                                                                                 */
   /*       * SELECT/WHEN structure without OTHERWISE                                                 */
   /*            If none of the WHEN conditions match, Rexx will look for an OTHERWISE.               */
   /*            If it does not exist, Rexx gives an Error 7, WHEN or OTHERWISE expected.             */
   /*            If you are sure all possible conditions are accounted for, you can                   */
   /*            do without an OTHERWISE. Sometimes it is useful to put an error handler              */
   /*            under OTHERWISE, as in                                                               */
   /*                OTHERWISE                                                                        */
   /*                   say 'Unexpected value for variable "foo":' foo                                */
   /*                                                                                                 */
   /*                                                                                                 */
   /*       * Control characters used for graphics                                                    */
   /*            The control characters below 32 are sometimes used as graphics                       */
   /*            to be displayed on the screen. But using some control characters                     */
   /*            in a Rexx file can cause problems. See                                               */
   /*                RexxTool.cmd help control                                                        */
   /*            You can convert the literal characters to hex strings with                           */
   /*                RexxTool.cmd Decontrol                                                           */
   /*                                                                                                 */
   /* =============================================================================================== */
   if Opt.0Check \== 'ALL' ,
    & wordpos('STYLE', Opt.0Check) == 0 then return 0
   SubID = SubID()
   if Opt.0progress then call Progress SubID
   WarnCount = 0
   /* ----------------------------------------------------------------------------------------------- */
   /*  MULTIPLE CLAUSES after THEN                                                                    */
   t = 2
   do while t < t_type.0 -1
      t = t +1
      if t_type.t \= 'REXX_W_KEY' then iterate
      if translate(t_val.t) \== 'THEN' then iterate
      firstW = R_NextWord(t)
      if L_ClauseCount(firstW) == 1 then iterate
      lastW = L_LastClause(firstW)
      CLcount = 0
      do w = firstW to lastW
         if t_type.w \== 'REXX_W_KEY' then iterate w
         if translate(t_val.w) == 'IF' ,
          | translate(t_val.w) == 'ELSE' ,
          | translate(t_val.w) == 'END' ,
          | translate(t_val.w) == 'DO' then do
               CLcount = 1
               leave
            end
         CLcount = CLcount +1
      end w
      if CLcount > 1 then do
            call CheckOut 'Warning: there are multiple clauses on line' t_line.firstW
            call CheckOut '         but THEN will only apply to the first one'
            call CheckOut Qline(t_line.firstW)
            WarnCount = WarnCount +1
         end
      t = lastW
   end
   /* ----------------------------------------------------------------------------------------------- */
   /*  LITERAL COMMENT STRINGS                                                                        */
   Warned = 0
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_W_LITERAL' then do
            if pos('/'||'*', t_val.t) > 0 then do
                  call CheckOut SubID 'Warning: "'||'/'||'*'||'" in quotes at line 't_line.t
                  if \Warned then do
                        call CheckOut '         It is safer to use "/"||"*"  See "help style"'
                        Warned = 1
                     end
                  WarnCount = WarnCount +1
               end
            if pos('*'||'/', t_val.t) > 0 then do
                  WarnCount = WarnCount +1
                  call CheckOut SubID 'Warning: "'||'*'||'/'||'" in quotes at line 't_line.t
                  if \Warned then do
                        call CheckOut '         It is safer to use "*"||"/"  See "help style"'
                        Warned = 1
                     end
               end
         end
   end t
   /* ----------------------------------------------------------------------------------------------- */
   /*  UNQUOTED SYSTEM COMMANDS;                                                                      */
   Warned = 0 ; Warned2 = 0
   do t = 2 to t_type.0 -1
      /*  A system command will not be tagged as a keyword, so identify it                            */
      /*  by looking for an EOC followed by a SYMBOL.                                                 */
      if T_Type(R_PrevToken(t)) \== 'REXX_EOC' then iterate t
      select
         when t_type.t == 'REXX_W_SYMBOL' then
            if pos(translate(t_val.t), translate(G.0Keywords)) == 0 then
               if \R_AtAssignment(t) then do
                     Tval = S_CompoundVar(t)
                     WarnCount = WarnCount +1
                     call CheckOut SubID 'Warning: unrecognized word "'Tval'" at line' t_line.t
                     call CheckOut '      'Qline(t_line.t)
                     if \Warned then do
                           if G.0Interpreter == 'REXXSAA' then
                              call CheckOut '   If it is a system command, quote it or use ADDRESS CMD' Tval
                           else
                              call CheckOut '   If it is a system command, quote it or use ADDRESS SYSTEM' Tval
                           Warned = 1
                        end
                  end
         when t_type.t == 'REXX_W_PROCEDURE' then do
               WarnCount = WarnCount +1
               call CheckOut SubID 'Warning: line' t_line.t 'is ambiguous'
               call CheckOut '      'Qline(t_line.t)
               if \Warned2 then do
                     if G.0dialect == 'REXXSAA' then
                        call CheckOut '   it would be clearer to use "CALL' t_val.t'" or "ADDRESS CMD' t_val.t'"'
                     else
                        call CheckOut '   it would be clearer to use "CALL' t_val.t'" or "ADDRESS SYSTEM' t_val.t'"'
                     Warned2 = 1
                  end
            end
         otherwise nop
      end   /*  select  */
   end t
   /* ----------------------------------------------------------------------------------------------- */
   /*  UNNEEDED ending ";"                                                                            */
   /*  a ";" is not needed at end of line, unless it is there to prevent a trailing                   */
   /*  comma from being interpreted as a line continuation                                            */
   t = 1
   do while t < t_type.0
      t = t +1
      if t_val.t == ';' then do
            /*  translation: if the next token (ignoring comments) is an EOL, then                    */
            /*  if the previous Rexx word (ignoring comments) is not a comma, then...                 */
            if abbrev(T_Class(R_NextToken(t)), 'FILE_') then
               if T_Type(R_PrevWord(t)) \== 'REXX_W_COMMA' then
                  if Opt.0Fix then do
                        if Opt.0verbose then
                           call CheckOut SubID 'remove unneeded ";" at line' t_line.t
                        call T_DeleteToken t
                        t = t -1
                     end
                  else do
                        call CheckOut 'Warning: unneeded ";" at end of line' t_line.t
                        call CheckOut Qline(t_line.t)
                        WarnCount = WarnCount +1
                     end
         end   /*  if t_val.t == ';'  */
   end
   /* ----------------------------------------------------------------------------------------------- */
   /*  ELSE IF                                                                                        */
   Warned = 0
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_W_KEY' then do
            nextW = R_NextWord(t)
            if R_KeyVal(t) == 'ELSE' ,
             & R_KeyVal(nextW) == 'IF' ,
             & L_OnSameLine(t, nextW) then
               /*  bugfix-- ignore a line like "if a then ... ; else if b then ... ; else ..."        */
               if t = L_FirstRexxWord(t) then do
                     if Opt.0Fix then do
                           /*  change the hidden EOC to an EOL                                        */
                           eoc = R_NextToken(t)
                           if t_type.eoc == 'REXX_EOC' ,
                            & t_val.eoc == '' then do
                                 t_class.eoc = 'FILE_EOL'
                                 if Opt.0verbose then
                                    call CheckOut SubID 'insert line break into ELSE IF at line' t_line.t
                              end
                           else do
                                 call ShowError 'BUG: token' eoc 'should be a hidden EOC but is not'
                                 call ShowError '     this looks like a bug in' G.0me
                                 call ShowError Qline(t_line.t), ''
                                 call Mark 't_class.eoc t_type.eoc t_val.eoc'
                                 call Fatality
                              end
                        end
                     else do
                           WarnCount = WarnCount +1
                           call CheckOut SubID 'Warning: at line' t_line.t', ELSE IF'
                           if \Warned then do
                                 call CheckOut '   can be misleading and could lead to errors.'
                                 call CheckOut '   It would be clearer to put IF on the next line, or'
                                 call CheckOut '   use SELECT if there are multiple choices.'
                                 call CheckOut '   See 'G.0me 'help "else if"'
                              end
                           call CheckOut Qline(t_line.t)
                           Warned = 1
                        end   /*  else  */
                  end
         end   /*  if REXX_W_KEY  */
   end t
   /* ----------------------------------------------------------------------------------------------- */
   /*  check for SELECT without OTHERWISE                                                             */
   Warned = 0
   t = 1
   do while t < t_type.0
      t = t +1
      if t_type.t == 'REXX_W_KEY' then
         if translate(t_val.t) == 'SELECT' then do
               endselect = R_OtherEnd(t)
               found = 0
               s = t
               do while s < endselect
                  s = s +1
                  if t_type.s == 'REXX_W_KEY' then
                     select
                        when translate(t_val.s) == 'SELECT' then
                           /*  skip over a nested SELECT structure                                    */
                           s = R_OtherEnd(s)
                        when translate(t_val.s) == 'OTHERWISE' then do
                              found = 1
                              leave
                           end
                        otherwise nop
                     end   /*  select  */
               end   /*  do while s  */
               if \found then do
                     call CheckOut SubID 'SELECT structure that ends at line' t_line.endselect 'has no OTHERWISE'
                     if \Warned then do
                           call CheckOut '      If you are sure all possible conditions will be caught, you do not need it'
                           call CheckOut '      see "'G.0me 'help otherwise"'
                        end
                     Warned = 1
                     WarnCount = WarnCount +1
                  end
            end   /*  if SELECT  */
   end   /*  do while t  */
   /* ----------------------------------------------------------------------------------------------- */
   /*  check for CONTROL CHARACTERS used as graphics                                                  */
   Warned = 0
   do l = 1 to SrcIn.0
      if verify(SrcIn.l, G.0ControlChars, 'M') > 0 then do
            call CheckOut SubID 'control character(s) found at line' l
            call CheckOut Qline(l)
            if \Warned then
               call CheckOut 'you can convert them with "Rexxtool Decontrol"'
            Warned = 1
         end
   end l
   if Warned then WarnCount = WarnCount +1
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0progress then call Progress
   if WarnCount > 0 ,
    | Opt.0verbose then
      call CheckOut left(SubID, 20) 'Warnings:' WarnCount
   return WarnCount
   /* === End of Task_CheckStyle ==================================================================== */

   Task_CheckExpose: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Check Expose                                                                             */
   /*                                                                                                 */
   /*        -Check=Expose           check for missing EXPOSE <variable> statement                    */
   /*                                                                                                 */
   /*  In a case like this:                                                                           */
   /*                                                                                                 */
   /*               Fee = 'Fee'                                                                       */
   /*               Fie = 'Fie'                                                                       */
   /*               call Label1                                                                       */
   /*               return                                                                            */
   /*                                                                                                 */
   /*               Label1: procedure expose Fee Fie                                                  */
   /*               say Fie                                                                           */
   /*               call Label2                                                                       */
   /*               return                                                                            */
   /*                                                                                                 */
   /*               Label2: procedure                                                                 */
   /*               call Label3                                                                       */
   /*               return                                                                            */
   /*                                                                                                 */
   /*               Label3: procedure expose Fee                                                      */
   /*               say Fee                                                                           */
   /*               return                                                                            */
   /*                                                                                                 */
   /*                                                                                                 */
   /*  This task will point out that Label2 does not expose variable Fee, which                       */
   /*  is required by Label3. You will need to decide for yourself if the exposed                     */
   /*  variable is supposed to be the same as the upper-level variable.                               */
   /*                                                                                                 */
   /*  If you get a "missing EXPOSE" error that doesn't seem to make sense, check your                */
   /*  error handlers. Say your Syntax handler uses the variable 'Fee'. Since an error                */
   /*  handler can potentially be called from any place, any procedure will need to                   */
   /*  expose Fee. RexxTool.cmd is an example-- the syntax handler calls ShowError,                   */
   /*  which uses the Opt. and G. stem variables. So all procedures must expose                       */
   /*  those variables, even a procedure that does not use them itself.                               */
   /*                                                                                                 */
   /*  This should be accurate most of the time, but there are unlikely cases that                    */
   /*  could confuse it. For instance, if you define a list of global variables inside                */
   /*  a procedure, or define the global list in more than one place.                                 */
   /*                                                                                                 */
   /*  The special variables RC, RESULT, and SIGL are ignored.                                        */
   /*                                                                                                 */
   /*  ==> if routines are very deeply nested, this may abort with an error 11,                       */
   /*      "Control Stack Full". OS/2 Regina might do better, but it has its own                      */
   /*      problems with very large files. Linux Regina should not be a problem.                      */
   /*      I have worked around that problem to the point that the only file I have                   */
   /*      seen that still triggers it is ppwizard.cmd                                                */
   /* =============================================================================================== */
   /*  fix: I don't quite remember how this works. Leave it for now; maybe I will have                */
   /*        a stroke of inspiration and perfect it. See also ExpandVariable()                        */
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0Check \== 'ALL' ,
    & wordpos('EXPOSE', Opt.0Check) == 0 then return 0
   SubID = SubID()
   if Opt.0progress then call Progress SubID

   ProcList = ProcedureList()
   call GetCalls '-sublabel'
   call GetAncestors
   /*  awkward-- we need an ancestor list to figure out the handlers, but then need to                */
   /*  call GetAncestors again to re-figure the ancestors of the handlers                             */
   call AddHandlersToChildren
   call GetAncestors
   if Opt.0progress then call Progress
   call MakeVarListsE
   if Opt.0progress then call Progress SubID

   WarnCount = 0
   !Level0 = Hexed('[Level0]')
   LabelList = LL.0LabelList
   do while LabelList \= ''
      parse var LabelList LabelN LabelList
      if wordpos(LabelN, ProcList) == 0 then iterate
      !LabelN = Hexed(LabelN)
      if Exposed.!LabelN \== '' then do
            VarList = Exposed.!LabelN
            do while VarList \= ''
               parse var VarList varN VarList
               if wordpos(translate(varN), 'RC RESULT SIGL') > 0 then iterate
               AncestorList = value('LL.0'!LabelN'_Ancestors')
               do while AncestorList \= ''
                  parse var AncestorList AncestorN AncestorList
                  !AncestorN = Hexed(AncestorN)
                  /*  if AncestorN is not a procedure, 'expose' does not apply                        */
                  if wordpos(AncestorN, ProcList) == 0 then iterate
                  /*  if varN is used by AncestorN and not exposed by AncestorN, then                 */
                  /*  varN is probably not intended as a global variable                              */
                  if wordpos(translate(varN), translate(Assigned.!AncestorN Used.!AncestorN)) > 0 then iterate
                  /*  if varN is not used above AncestorN, it is not what we care about               */
                  AncestorVars = translate(AncestorVariables(AncestorN))
                  AncestorVars = AncestorVars translate(Assigned.!Level0)
                  if wordpos(translate(varN), AncestorVars) == 0 then iterate
                  if wordpos(translate(varN), translate(Exposed.!AncestorN)) == 0 then do
                        call CheckOut SubID 'Warning: upper-level variable "'varN'" is exposed by procedure "'LabelN'"'
                        call CheckOut G.0Tab'but not by the intermediate procedure "'AncestorN'"'
                        call CheckOut G.0Tab'the upper-level labels where it is used are:'
                        ULProcList = value('LL.0'!AncestorN'_Ancestors') LL.0Level0_Labels
                        ULProcList = UniqueWord(ULProcList)
                        ULProcList = WordSort(ULProcList)
                        do while ULProcList \= ''
                           parse var ULProcList ULprocName ULProcList
                           !ULprocName = Hexed(ULprocName)
                           if wordpos(translate(varN), ,
                              translate(value('Assigned.'!ULprocName) value('Used.'!ULprocName))) > 0 then
                              call CheckOut G.0Tab||G.0Tab ULprocName
                        end
                        WarnCount = WarnCount +1
                     end
               end
            end   /*  while VarList  */
         end   /*  if Exposed  */
   end   /*  while LabelList  */
   if Opt.0progress then call Progress
   if WarnCount > 0 ,
    | Opt.0verbose then
      call CheckOut left(SubID, 20) 'Warnings:' WarnCount
   return WarnCount
   /* === End of Task_CheckExpose =================================================================== */

   Task_CheckToker: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Check Toker Tokeniser Tokenizer                                                          */
   /*                                                                                                 */
   /*  Options:                                                                                       */
   /*     -check=toker                     check for possible Tokeniser errors                        */
   /*                                                                                                 */
   /*  This is only for testing the Tokeniser, and it is not included in "-check=all"                 */
   /*  It checks for variable names that match a keyword, sub-keyword, or BIF, which                  */
   /*  =could= indicate a tokeniser error.  Most warnings will be false positives.                    */
   /*                                                                                                 */
   /* =============================================================================================== */
   if wordpos('TOKER', Opt.0Check) == 0 then return 0
   SubID = SubID()
   if Opt.0progress then call Progress SubID
   WarnCount = 0
   /* ----------------------------------------------------------------------------------------------- */
   Keywords = translate(G.0Keywords)
   Keywords2 = translate(G.0Keywords2)
   BIFs = translate(G.0AllBIFs)

   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_W_SYMBOL' then do
            Tval = translate(t_val.t)
            if wordpos(Tval, Keywords) > 0 then
               /*  exclude some cases where it probably is a real variable                            */
               if \R_AtAnyAssignment(t) then
                  if \(T_Val(t -1) == '.' ,
                   & t_prefix.t == '') then
                     if \(T_Val(t +1) == '.' ,
                      & T_Prefix(t +1) == '') then do
                           call CheckOut SubID 'symbol name matches keyword     "'t_val.t'" at line' t_line.t
                           call CheckOut strip(Qline(t_line.t))
                           WarnCount = WarnCount +1
                        end
            if wordpos(Tval, Keywords2) > 0 then
               if \R_AtAnyAssignment(t) then
                  if \(T_Val(t -1) == '.' ,
                   & t_prefix.t == '') then
                     if \(T_Val(t +1) == '.' ,
                      & T_Prefix(t +1) == '') then do
                           call CheckOut SubID 'symbol name matches sub-keyword "'t_val.t'" at line' t_line.t
                           call CheckOut strip(Qline(t_line.t))
                           WarnCount = WarnCount +1
                        end
            if wordpos(Tval, BIFs) > 0 then
               if \R_AtAnyAssignment(t) then
                  if \(T_Val(t -1) == '.' ,
                   & t_prefix.t == '') then
                     if \(T_Val(t +1) == '.' ,
                      & T_Prefix(t +1) == '') then do
                           call CheckOut SubID 'symbol name matches BIF         "'t_val.t'" at line' t_line.t
                           call CheckOut strip(Qline(t_line.t))
                           WarnCount = WarnCount +1
                        end
         end
   end t
   return WarnCount
   /* === End of Task_CheckToker ==================================================================== */

   Task_debugEOC: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  HELP: debug EOC                                                                                */
   /*                                                                                                 */
   /*    -debug=EOC              make sure all the hidden EOC tokens are in the right place           */
   /*                                                                                                 */
   /*  This is a debugging tool the verifies that End-Of-Clause tokens are in the                     */
   /*  right place. It is useful if you are working on a task that involves moving                    */
   /*  Rexx tokens. You might as well leave it on, because it takes very little time.                 */
   /* =============================================================================================== */
   if \debug.0eoc then return 0
   if Opt.0progress then call Progress SubID()
   do t = 2 to t_type.0 -1
      select
         when t_type.t == 'REXX_W_KEY' then
            if wordpos(translate(t_val.t), 'THEN ELSE OTHERWISE :') > 0 then
               call debugEOC t
         when t_type.t == 'REXX_EOC' then
            if t_val.t == '' then
               if t_prefix.t \== '' then do
                     call ShowError 'Error: hidden EOC token' t 'has prefix "'t_prefix.t'" at line' t_line.t
                     call ShowError Qline(t_line.t), ''
                     call ShowError '"'L_AsString(t)'"', ''
                     call Fatality
                  end
         when t_class.t == 'COMMENT' then nop
         when t_val.t \== '' then nop
         otherwise
            /*  also check for other null tokens that have prefix. Null comments have                 */
            /*  already been excluded                                                                 */
            if t_prefix.t \== '' then do
                  call ShowError 'Error: token' t 'at line' t_line.t 'has no value but it has a prefix "'t_prefix.t'"'
                  call ShowError 'the class is' t_class.t 'and the type is' t_type.t, ''
                  call ShowError '     ==>' L_AsString(t), ''
               end
      end   /*  select  */
   end t

   if Opt.0progress then call Progress
   return 0
   /* === End of Task_debugEOC ====================================================================== */

   Task_DeControl: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  HELP: DeControl Control                                                                        */
   /*                                                                                                 */
   /*  This task simply replaces each control character with its hex representation.                  */
   /*  Control characters are defined here as characters 0 through 31 and character 127,              */
   /*  except for Tab.                                                                                */
   /*                                                                                                 */
   /*  For the sake of speed, this replaces a token with a string that rightfully                     */
   /*  should be split into several tokens. So do not combine this with a task that                   */
   /*  depends on accurate tokenizing. That is why this is not included in the things                 */
   /*  that can be fixed with "-check -fix".                                                          */
   /*                                                                                                 */
   /* =============================================================================================== */
   SubID = SubID()
   if Opt.0progress then call Progress SubID

   do t = 2 to t_type.0 -1
      if verify(t_val.t, G.0ControlChars, 'M') > 0 then do
            select
               when t_type.t == 'REXX_W_LITERAL' then do
                     call DeControl t
                     G.0FileChanged = 1
                  end
               when t_type.t == 'COMMENT_VAL' then do
                     call DeControl t
                     G.0FileChanged = 1
                  end
               otherwise
                  call ShowError 'BUG: control character in a token that is not a LITERAL or COMMENT'
                  call ShowError 'at line' t_line.t, ''
                  call Mark 't t_type.t t_val.t'
                  call Fatality
            end
            /* -------------------------------------------------------------------------------------- */
         end   /*  if verify  */
   end t
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_DeControl ===================================================================== */

   Task_Depends: procedure expose (Shared_vars) (all_tokens) SrcOut. BIFs. /*fold00*/
   /* =============================================================================================== */
   /*  used by Script_Depends                                                                         */
   /*                                                                                                 */
   /*  This task displays the internal, external, and built-in functions that the                     */
   /*  specified routine(s) depends on.                                                               */
   /* =============================================================================================== */
   Tab = copies(' ', 4)
   if wordpos('+HANDLERS', translate(Opt.0Where)) > 0 then
      IncludeHandlers = 1
   else
      IncludeHandlers = 0
   Opt.0Where = ExpandLabels(Opt.0Where)
   Include = Opt.0Where
   if Opt.0Show == 1 then Opt.0Show = ''
   if Opt.0Show = '' then
      Show = 'Int,Ext,BIF,Lib,Sys'
   else
      Show = Opt.0Show
   Show = translate(Show, ' ', ',')
   Show = translate(Show)

   call BIFAliases
   call GetDeps Include
   Include = Required.0Include
   if Opt.0Terse then do
         /*  -terse is a quick hack because sometimes I want a one-line answer to                     */
         /*     "Depends -show=ext"                                                                   */
         if wordpos('INT', Show) > 0 then do
               call Out Required.0Int
            end
         if wordpos('EXT', Show) > 0 then do
               call Out Required.0Ext
            end
         if wordpos('BIF', Show) > 0 then do
               call Out Required.0BIF
            end
         if wordpos('LIB', Show) > 0 then do
               call Out LibList()
            end
      end   /*  if Opt.0Terse  */
   else do
         if Opt.0Where == '' then do
               call Out 'file' G.0InfileN' depends on'
            end
         else do
               call Out 'file' G.0InfileN', dependencies for these routines:'
               call Out ChangeString(' ', Include, ', ')
            end

         if wordpos('INT', Show) > 0 then do
               call Out Tab||'internal routines:'
               do while Required.0Int \= ''
                  parse var Required.0Int FuncN Required.0Int
                  call Out Tab||Tab||FuncN
               end
            end

         if wordpos('EXT', Show) > 0 then do
               if G.0dialect == 'REGINA' then do
                     /*  see regina.pdf about how Regina looks for external commands                  */
                     suffixlist = EnvVar('REGINA_SUFFIXES')
                     if suffixlist = '' then
                        suffixlist = '<none> rxc rexx rex cmd rx'
                     else
                        suffixlist = '<none>' suffixlist 'rxc rexx rex cmd rx'
                     /*  Regina suffix list may have spaces or commas, with or without dots           */
                     suffixlist = translate(suffixlist, ' ', ',')
                     suffixlist = ChangeString('.', suffixlist, '')
                  end   /*  if Regina  */
               else
                  suffixlist = '<none> cmd'
               call Out Tab||'external procedures:'
               do r = 1 to words(Required.0Ext)
                  FuncN = word(Required.0Ext, r)
                  call Out Tab||Tab||FuncN
                  foundN = ''
                  do s = 1 to words(suffixlist)
                     suff = word(suffixlist, s)
                     if suff == '<none>' then
                        suff = ''
                     else
                        suff = '.'||suff
                     srchfor = FuncN||suff
                     /*  use SearchPath() to avoid broken SysSearchPath() in Regina/OS2               */
                     /*  also, SearchPath() handles a filename that already has a path                */
                     if G.0dialect == 'REGINA' then
                        foundN = SearchPath(EnvVar('REGINA_MACROS'), srchfor)
                     if foundN == '' then
                        foundN = SearchPath(EnvVar('PATH'), srchfor)
                     if foundN \= '' then leave s
                  end s
                  if foundN \== '' then
                     call Out Tab||'    (found)  'foundN
                  else
                     call Out Tab||'(not found)  'FuncN
               end r
            end   /*  if EXT  */

         /*  BIFs and their libraries are done together                                               */
         Liblist = ''
         if wordpos('BIF', Show) > 0 then do
               call Out Tab||'Built-In functions:'
            end
         do while Required.0BIF \= ''
            parse var Required.0BIF FuncN Required.0BIF
            !FuncN = Hexed(FuncN)
            libname = ''
            hit = 0
            alias = ''
            if BifAlias.!FuncN \== '' then do
                  /*  replace alias with the real BIF name it stands for                              */
                  alias = FuncN
                  FuncN = BifAlias.!FuncN
               end
            do c = 1 to words(BIFs.0used)
               libname = word(BIFs.0used, c)
               biflist = value('BIFs.0'libname)
               hit = 0
               if wordpos(translate(FuncN), translate(biflist)) > 0 then do
                     hit = 1
                     /*  add to list of required libs, if not already there                           */
                     if wordpos(libname, Liblist) == 0 then do
                           Liblist = Liblist libname
                        end
                     leave c
                  end
            end c
            if wordpos('BIF', Show) > 0 then do
                  if hit then
                     if alias \== '' then
                        call Out Tab||Tab||FuncN '('libname')' '(registered as' alias')'
                     else
                        call Out Tab||Tab||FuncN '('libname')'
                  else
                     call Out Tab||Tab||FuncN
               end
         end   /*  while Required.0BIF  */

         if wordpos('LIB', Show) > 0 then do
               Liblist = strip(Liblist)
               call Out Tab||'Known libraries that are used:'
               do while Liblist \= ''
                  parse var Liblist libname Liblist
                  if translate(libname) \== 'REXX' then
                     call Out Tab||Tab||libname
               end
               Liblist = LibList()
               call Out Tab||'Libraries loaded with RxFuncAdd or SysLoadRexxMacroSpace:'
               do while Liblist \= ''
                  parse var Liblist libname Liblist
                  call Out Tab||Tab||libname
               end
            end

         if \IncludeHandlers then
            if wordpos('INT', Show) > 0 then
               if LL.0Handlers \= '' then do
                     call Out Tab||'May also depend on the error-handlers:'
                     tempstr = LL.0Handlers
                     do while tempstr \= ''
                        parse var tempstr FuncN tempstr
                        call Out Tab||Tab||FuncN
                     end
                  end

         if wordpos('SYS', Show) > 0 then do
               /*  show system commands used                                                          */
               Include = translate(Include)
               cmdlist = ''
               VarList = ''
               CurLabel = translate('[Unlabeled]')
               do t = 1 to t_type.0 -1
                  if t_type.t \= 'REXX_EOC' then iterate t
                  w1 = t +1
                  select
                     when t_type.w1 == 'REXX_LABEL' then
                        CurLabel = translate(t_val.w1)
                     when t_type.w1 == 'REXX_W_LITERAL' then do
                           /*  looks like a quoted system command                                     */
                           if wordpos(CurLabel, Include) > 0 then do
                                 if pos(translate(t_val.w1), translate(cmdlist)) == 0 then
                                    cmdlist = cmdlist||G.0sep||t_val.w1
                              end
                        end
                     when t_type.w1 == 'REXX_W_SYMBOL' then
                        /*  word w1 may be a stem, find tail of stem and check next word for "="      */
                        if T_Type(R_NextWord(R_Tail(w1))) \= 'REXX_W_ASSIGN' then
                           /*  looks like a symbol used as a system command                           */
                           if wordpos(CurLabel, Include) > 0 then do
                                 varN = S_CompoundVar(w1)
                                 if wordpos(translate(varN), translate(VarList)) == 0 then
                                    VarList = VarList varN
                              end
                     when t_type.w1 == 'REXX_W_KEY' then
                        if translate(t_val.w1) == 'ADDRESS' then do
                              /*  an ADDRESS CMD <system command>                                     */
                              if wordpos(CurLabel, Include) > 0 then do
                                    wvar = R_NextWord(R_NextWord(w1))
                                    if t_type.wvar == 'REXX_W_LITERAL' then do
                                          /*  quoted system command                                   */
                                          if pos(translate(t_val.wvar), translate(cmdlist)) == 0 then
                                             cmdlist = cmdlist||G.0sep||t_val.wvar
                                       end
                                    else do
                                          /*  var used as system command                              */
                                          varN = S_CompoundVar(wvar)
                                          if wordpos(translate(varN), translate(VarList)) == 0 then
                                             VarList = VarList varN
                                       end
                                 end
                           end
                     otherwise nop
                  end   /*  select  */
               end t
               VarList = strip(VarList)
               cmdlist = strip(cmdlist, , G.0sep)

               if cmdlist \= '' then do
                     call Out 'System Commands Used:'
                     do while cmdlist \= ''
                        parse var cmdlist cmd (G.0sep) cmdlist
                        call Out G.0Tab||cmd
                     end
                  end

               /*  In the case of                                                                     */
               /*     Todo = 'dir *.*'                                                                */
               /*     address cmd Todo                                                                */
               /*  the only way to resolve Todo is to search the whole file for a simple assignment.  */
               /*  "parse var string Todo" or value('Todo', <newvalue>) probably cannot be resolved,  */
               /*  but surely nobody would use those to assign Todo (?) --Yes, they would; I found    */
               /*  several examples. We cannot do much about that.                                    */
               if VarList \= '' then do
                     call Out 'Variables used as system commands:'
                     do while VarList \= ''
                        parse var VarList varN VarList
                        call Out G.0Tab||varN
                        do t = 1 to t_type.0 -1
                           if t_type.t \= 'REXX_EOC' then iterate t
                           w1 = t +1
                           if t_type.w1 \= 'REXX_W_SYMBOL' then iterate t
                           if translate(S_CompoundVar(w1)) \= translate(varN) then iterate t
                           w2 = R_Tail(w1) +1
                           if t_type.w2 \= 'REXX_W_ASSIGN' then iterate t
                           /*  found an assignment for varN                                           */
                           cmd = ''
                           do w = w2 +1 to R_EndEOC(w2)
                              cmd = cmd||t_prefix.w||t_val.w
                           end w
                           cmd = strip(cmd)
                           call Out G.0Tab||G.0Tab||'which is' cmd
                           cmd = ''
                        end t
                     end
                  end
            end
      end   /*  if \ Opt.0Terse  */
   return 0
   /* === End of Task_Depends ======================================================================= */

   Task_Extract: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  used by Script_Extract                                                                         */
   /*                                                                                                 */
   /*  This task extracts the specified routine(s) and all internal routines it depends on            */
   /* =============================================================================================== */
   if Opt.0Where == '' ,
    | Opt.0Where == 1 then do
         call Msg G.0meID 'I need to know what to extract'
         call Msg 'use "-where=<label list>"'
         call Egress 'USAGE'
      end
   Include = ExpandLabels(Opt.0Where)
   if Include == '' then do
         call Msg G.0meID 'found nothing to extract'
         call Egress 'okay'
      end

   if Opt.0progress then call Progress SubID()
   call ProcMap
   call GetDeps Include
   ToExtract = ''
   /*  if Required.0Include is '' there is nothing to do. That could happen if                        */
   /*  -where=+handlers and there are no handlers                                                     */
   if Required.0Include \== '' then
      ToExtract = strip(Required.0Include Required.0Int)
   ln = SrcOut.0 +1
   SrcOut.ln = '/'||'*' ,
               ||' extracted from '||G.0InfileN||' with "'G.0me 'extract -where='Opt.0Where||'" ' ,
               ||'*'||'/'
   /*  ln +1 adds a blank line after comment                                                          */
   SrcOut.0 = ln +1

   /*  sort the list of procs to extract into the same order as they appear in the original file      */
   tmp = LL.0LabelList
   ToExtractS = ''
   do while tmp \= ''
      parse var tmp LabelN tmp
      if wordpos(LabelN, ToExtract) > 0 then
         ToExtractS = ToExtractS LabelN
   end
   /*  do the extracting                                                                              */
   do while ToExtractS \= ''
      parse var ToExtractS LabelN ToExtractS
      !LabelN = Hexed(LabelN)
      call Task_Rebuild Map.!LabelN.start, Map.!LabelN.stop
   end
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Extract ======================================================================= */

   Task_FixFooters: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: Footer Format                                                                            */
   /*                                                                                                 */
   /*  used by Script_Format                                                                          */
   /*                                                                                                 */
   /*  Makes sure the "End of <routine>" text always matches the name of the routine.                 */
   /*  If a footer does not exist, this will not create one                                           */
   /*                                                                                                 */
   /*    -Footer                    maintain the subroutine footers                                   */
   /*    -!Footer or -Footer=0      do not maintain the subroutine footers                            */
   /*                                                                                                 */
   /* =============================================================================================== */
   if \Opt.0Footer then return 0
   if Opt.0progress then call Progress SubID()
   CurLabel = ''
   do t = 2 to t_type.0 -1
      select
         when t_class.t == 'COMMENT' then nop
         when t_type.t == 'REXX_LABEL' then do
               /*  a sub-label does not count. See "Help sublabel"                                    */
               if \abbrev(translate(t_val.t), translate(CurLabel)||'.') then
                  CurLabel = t_val.t
               iterate t
            end
         otherwise iterate t
      end   /*  select  */
      if C_IsFooter(t) then do
            /*  the comment will be resized later by the comment formatter                            */
            t_val.t = G.0footerID||CurLabel||' '||copies('=', 70)||' '
         end
   end t
   return 0
   /* === End of Task_FixFooters ==================================================================== */

   Task_FixRexxID: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: RexxID                                                                                   */
   /*                                                                                                 */
   /*  used by Script_Format, and possibly other scripts                                              */
   /*                                                                                                 */
   /*  adds a RexxID comment only if there is none. The RexxID is the starting comment                */
   /*  on the first line that identifies the file as a Rexx script.                                   */
   /*  If there is none, add "/@@/" to beginning of file.                                             */
   /*  If there are empty lines before the first comment, delete the empty lines                      */
   /*                                                                                                 */
   /*  controlled by the -RexxID switch                                                               */
   /*                                                                                                 */
   /* =============================================================================================== */
   if \Opt.0RexxID then return 0
   rID = RexxID()
   /*  usually the RexxID is where it should be, at token 2                                           */
   if rID == 2 then return 0
   if rID == 1 then
      /*  there is no RexxID comment; add "/@@/" after Beginning-Of-File token                        */
      call C_InsertCommentLine 2, ''
   else
      /*  there is a RexxID comment, but not at the first line. Remove empty lines above it.          */
      call T_DeleteToken 2, rID -2
   return 0
   /* === End of Task_FixRexxID ===================================================================== */

   Task_FoldLine: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: FoldLine Format                                                                          */
   /*                                                                                                 */
   /*  used by Script_Format                                                                          */
   /*                                                                                                 */
   /*    -FoldLine               break up an IF or WHEN clause at logical operators                   */
   /*                            so                                                                   */
   /*                                  IF a == 1 | b == 1 | c == 1 THEN                               */
   /*                            becomes                                                              */
   /*                                  IF a == 1 ,                                                    */
   /*                                   | b == 1 ,                                                    */
   /*                                   | c == 1 THEN                                                 */
   /*                                                                                                 */
   /*                            it will also force an operator to the start of the next line         */
   /*                                  IF a == 1 | ,                                                  */
   /*                                  b == 1 | ,                                                     */
   /*                                  c == 1 THEN                                                    */
   /*                            becomes the same as above                                            */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  Writing a general-purpose line-wrapping routine would take a lot of work, but this             */
   /*  could be one part of it                                                                        */
   /* =============================================================================================== */
   if \Opt.0FoldLine then return 0
   SubID = SubID()
   if Opt.0progress then call Progress SubID

   t = 1
   do while t < t_type.0 -1
      t = t +1
      if t_readonly.t then iterate
      if t_type.t == 'REXX_W_LOGIC' then do
            select
               when t == L_FirstRexxWord(t) then
                  /*  "| fie THEN"                                                                    */
                  nop
               when t == L_LastRexxWord(t) then do
                     /*  "IF fee | ,"                                                                 */
                     /*  move the "|" to before first word of next line                               */
                     n = R_NextWord(t)
                     t_prefix.n = ' '
                     call T_MoveToken t, n -1
                  end
               otherwise
                  /*  "IF fee | fie THEN"                                                             */
                  call LineWrapAtOp t
            end   /*  select  */
         end
   end   /*  while t  */
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_FoldLine ====================================================================== */

   Task_FormatComments: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: FormatComments EndComment MoveComment Comment Comments Format                            */
   /*                                                                                                 */
   /*  finds comments and changes their width or position                                             */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -!FormatComments        do not reformat any comment                                          */
   /*    -FormatComments=0       do not reformat any comment                                          */
   /*                                                                                                 */
   /*    -MoveComment='RIGHT'    move right-hand comment to the right margin                          */
   /*    -MoveComment='LEFT'     move right-hand comment to the left                                  */
   /*    -MoveComment=<col>      move right-hand comment to column <col>                              */
   /*    -MoveComment='EXPAND'   do not move right-hand comment, but expand to the right margin       */
   /*    -MoveComment=''         do not change right-hand comment                                     */
   /*                                                                                                 */
   /*    -EndComment='LEFT'      move a short comment after END, SELECT, or DO to the left,           */
   /*                            like this:                                                           */
   /*                                 END   /@  of inner loop  @/                                     */
   /*                            instead of this:                                                     */
   /*                                 END                                  /@  of inner loop  @/      */
   /*                                                                                                 */
   /*    -EndComment=''          do not treat END/SELECT/DO comments specially                        */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  All comment-related functions will assume that the token passed to them is the                 */
   /*  COMMENT_VAL part of the comment.                                                               */
   /* ----------------------------------------------------------------------------------------------- */
   if \Opt.0FormatComments then return 0
   if Opt.0progress then call Progress SubID()
   /*  give special handling to the starting comment. Since it cannot have a margin,                  */
   /*  add Opt.0Margin to the width to make the right side line up. If the starting                   */
   /*  comment is an EXTPROC or #! line, C_JustifyRight will pad it with trailing                     */
   /*  spaces. That is harmless because they will be removed later.                                   */
   rID = RexxID() +1
   if t_type.rID == 'COMMENT_VAL' then
      select
         when C_IsDivider(rID) then
            call C_Fmt_Divider rID, Opt.0Width + Opt.0Margin
         when C_CommentType(rID) == 'FULL' then
            if t_val.rID \= '' then
               call C_JustifyRight rID, Opt.0Width + Opt.0Margin
         otherwise nop
      end

   do t = rID +1 to t_type.0
      if t_type.t \= 'COMMENT_VAL' then iterate t
      if t_readonly.t then iterate t

      ctype = C_CommentType(t)
      select
         when C_IsDivider(t) then
            call C_Fmt_Divider t
         when ctype = 'FULL' then
            call C_JustifyRight t
         when ctype = 'RIGHT' then
            call C_Fmt_Right t
         otherwise nop
      end
   end t

   if Opt.0progress then call Progress
   return 0
   /* === End of Task_FormatComments ================================================================ */

   Task_HeaderFooter: procedure expose (Shared_vars) (all_tokens)  SrcOut. Header. /*fold00*/
   /* =============================================================================================== */
   /*  Help: AddHeader AddFooter Header Footer Headers Footers Format                                 */
   /*  Help: sub-label sublabel sub-labels sublabels                                                  */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -AddHeader              inserts an empty subroutine header under the label,                  */
   /*                            unless the label already has a header, or is followed by             */
   /*                            a line of #####, *****, or ===== characters                          */
   /*                                                                                                 */
   /*                            You could customize the header by setting the                        */
   /*                            'Header.' array in a Style_xxx routine.  The lines in the            */
   /*                            Header array are the comment values only; the begin comment          */
   /*                            and end comment tokens will be added. Which means that a             */
   /*                            header cannot be a multi-line comment.                               */
   /*                                                                                                 */
   /*    -AddHeader=above        inserts an empty subroutine header above the label,                  */
   /*                            unless there is already a header, or the label is                    */
   /*                            preceded by a line of #####, *****, or ===== characters              */
   /*                                                                                                 */
   /*    -AddFooter              inserts a footer comment to identify the end of a routine,           */
   /*                            a footer comment is in the form                                      */
   /*                               "/@=== End of <name of routine> ==================@/              */
   /*                                                                                                 */
   /*  -AddHeader and -AddFooter can be very slow on large files, because they                        */
   /*  must insert tokens into the arrays.                                                            */
   /*                                                                                                 */
   /*  This task will not add headers or footers for labels that it recognizes as                     */
   /*  sub-labels.                                                                                    */
   /*                                                                                                 */
   /*  About sub-labels:                                                                              */
   /*    A sub-label is a "local" label that is logically part of a larger routine.                   */
   /*    For instance a local error-handler, or a loop written like this:                             */
   /*                                                                                                 */
   /*          Foo: procedure                                                                         */
   /*          signal on error name Foo.error                                                         */
   /*          Foo.Loop:                                                                              */
   /*          ...                                                                                    */
   /*          signal Foo.Loop                                                                        */
   /*          Foo.error:                                                                             */
   /*          ...                                                                                    */
   /*                                                                                                 */
   /*    I give the loop and error-handler labels names that start with "Foo." to                     */
   /*    indicate that they are intended to be used as part of the Foo procedure.                     */
   /*    It also reduces the chance of duplicate label names. This is just a naming                   */
   /*    convention I invented to make the procedure easier to understand; to Rexx                    */
   /*    there is no such thing as a local label.                                                     */
   /*                                                                                                 */
   /*    RexxTool.cmd will recognize a sub-label if it has the form                                   */
   /*         <label name>.<text>                                                                     */
   /*    So the above block of code would be treated as one procedure, and the                        */
   /*    Calls, Depends, Extract, and Tree commands will be more accurate and/or useful.              */
   /*    In RexxTool.cmd, the GetAncestors and StringMatch procedures have sub-labels.                */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  I have been experimenting with double labels, hence the places in this section                 */
   /*  where it looks to see if the previous or next line is a label. The labels in this              */
   /*  file that start with "0" are examples.                                                         */
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0AddHeader = 0 ,
    & Opt.0AddFooter = 0 then return 0
   if wordpos(translate(Opt.0AddHeader), '0 1 ABOVE') == 0 then do
         call Msg G.0meID 'unrecognized option: -AddHeader='Opt.0AddHeader
         call Egress 'usage'
      end
   if Opt.0progress then call Progress SubID()
   /*  if caller did not set a Header, use the default:                                               */
   if symbol('Header.0') == 'LIT' then do
         Header.1 = ' '||copies('=', 78)||' '
         Header.2 = ' '||copies('=', 78)||' '
         Header.0 = 2
      end

   hasFooter = 0
   CurLabel = ''
   PrevLabel = ''
   t = 1
   do while t < t_type.0
      t = t +1
      select
         when t_type.t == 'COMMENT_VAL' then do
               if C_IsFooter(t) then
                  hasFooter = 1
               iterate
            end
         when t_type.t == 'REXX_LABEL' then do
               /*  for our purposes a sub-label is not a label; keep going                            */
               if abbrev(translate(t_val.t), translate(CurLabel)||'.') then iterate
               PrevLabel = CurLabel
               CurLabel = t_val.t
            end
         when t_class.t == 'FILE_EOF' then
            /*  pretend the EOF is a label so the last routine will get a footer                      */
            PrevLabel = CurLabel
         otherwise iterate
      end   /*  select  */

      /* -------------------------------------------------------------------------------------------- */
      /*  at this point, token t must be a label or the End-Of-File,                                  */
      /*  because everything else iterated                                                            */

      /*  a label in the middle of the line is too weird to get a header                              */
      if t \= L_FirstToken(t) then do
            hasFooter = 0
            iterate
         end
      /* -------------------------------------------------------------------------------------------- */
      if Opt.0AddFooter then
         if \hasFooter then
            if PrevLabel \= '' then
               if T_Type(L_FirstToken(t, -1)) \== 'REXX_LABEL' then do
                     /*  (ie, if previous line is not also a label)                                   */
                     /*  add a footer for the previous block. The insertion point is one              */
                     /*  token past the end of that block. B_EndOfPrev() identifies where             */
                     /*  that is. If there are intervening comments, it does the best it can          */
                     /*  to judge which block they belong to.                                         */
                     point = B_EndOfPrev(t) +1
                     /*  will be resized to the right width later by comment formatter                */
                     Cval = G.0footerID||PrevLabel||' '||copies('=', 70)||' '
                     /*  is token insert -3 already the COMMENT_VAL of a divider?                     */
                     /*  If so, replace it; otherwise insert a new footer comment                     */
                     cval? = point -3
                     if Opt.0progress then
                        call Progress 'adding footer at line' ,
                             t_line.point 'of' SrcIn.0 'for label 'PrevLabel
                     if C_IsDivider(cval?) then
                        call T_Val cval?, Cval
                     else do
                           call C_InsertCommentLine point, Cval
                           /*  update t because we inserted four tokens before it                     */
                           t = t +4
                        end
                     if Opt.0progress then call Progress
                  end
      /* -------------------------------------------------------------------------------------------- */
      /*  a header after the End Of File would be a contradiction in terms.                           */
      if t_class.t == 'FILE_EOF' then
         iterate
      if \abbrev(translate(t_val.t), translate(CurLabel)||'.') then
         hasFooter = 0
      /* -------------------------------------------------------------------------------------------- */
      /*  the HasHeader flag really means "do not add a header"                                       */
      if Opt.0AddHeader \= 0 then do
            HasHeader = 0
            /* -------------------------------------------------------------------------------------- */
            if translate(Opt.0AddHeader) == 'ABOVE' then do                /* header goes ABOVE label */
                  /*  test for things that would exclude adding a header:                             */
                  if T_Type(L_FirstToken(t, -1)) == 'REXX_LABEL' then
                     HasHeader = 1
                  /*  does previous line look like the end of a header?                               */
                  testline = L_AsString(t, -1)
                  if strip(testline) \= '' then do
                        dtype = U_DividerType(testline)
                        if pos(dtype, '#*=') > 0 then
                           HasHeader = 1
                        l = Header.0
                        if strip(S_UnComment(testline)) = strip(Header.l) then
                           HasHeader = 1
                     end
                  if \HasHeader then do
                        p = L_LineEOL(t, -1)
                        if Opt.0progress then
                           call Progress 'adding header at line' ,
                                t_line.t -1 'of' SrcIn.0 'for label' t_val.t
                        call T_InsertToken p, Header.0 * 4
                        do l = 1 to Header.0
                           t_class.p = 'FILE_EOL'
                           t_type.p = 'REXX_EOC'
                           t_val.p = ''
                           t_prefix.p = ''
                           p = p +1
                           t_class.p = 'COMMENT'
                           t_type.p = 'COMMENT_BEG'
                           t_val.p = '/'||'*'
                           t_prefix.p = ''
                           p = p +1
                           t_class.p = 'COMMENT'
                           t_type.p = 'COMMENT_VAL'
                           t_val.p = Header.l
                           t_prefix.p = ''
                           p = p +1
                           t_class.p = 'COMMENT'
                           t_type.p = 'COMMENT_END'
                           t_val.p = '*'||'/'
                           t_prefix.p = ''
                           p = p +1
                        end l
                        /*  update t because we inserted tokens before it                             */
                        t = t + (Header.0 * 4)
                        if Opt.0progress then call Progress
                     end   /*  if \HasHeader  */
               end   /*  if ABOVE  */
            /* -------------------------------------------------------------------------------------- */
            else do                                                        /* header goes BELOW label */
                  /*  test for things that would exclude adding a header:                             */
                  nl = L_FirstToken(t, +1)
                  if t_type.nl == 'REXX_LABEL' then
                     if \abbrev(translate(t_val.nl), translate(CurLabel)||'.') then
                        HasHeader = 1
                  /*  in a case of "Label: a = 2 ; b = 3; ...", skip the header                       */
                  if (L_HasRexxWord(t) ,
                   & translate(T_Val(CL_FirstWord(t, +1))) \== 'PROCEDURE') then
                     HasHeader = 1
                  /*  does next line look like the start of a header?                                 */
                  testline = L_AsString(t, +1)
                  if strip(testline) \= '' then do
                        dtype = U_DividerType(testline)
                        if pos(dtype, '#*=') > 0 then
                           HasHeader = 1
                        l = Header.0
                        if strip(S_UnComment(testline)) = strip(Header.l) then
                           HasHeader = 1
                     end
                  if \HasHeader then do
                        p = L_LineEOL(t)
                        if Opt.0progress then
                           call Progress 'adding header at line' ,
                                t_line.t +1 'of' SrcIn.0 'for label' t_val.t
                        call T_InsertToken p, Header.0 * 4

                        do l = 1 to Header.0
                           t_class.p = 'FILE_EOL'
                           t_type.p = 'REXX_EOC'
                           t_val.p = ''
                           t_prefix.p = ''
                           p = p +1
                           t_class.p = 'COMMENT'
                           t_type.p = 'COMMENT_BEG'
                           t_val.p = '/'||'*'
                           t_prefix.p = ''
                           p = p +1
                           t_class.p = 'COMMENT'
                           t_type.p = 'COMMENT_VAL'
                           t_val.p = Header.l
                           t_prefix.p = ''
                           p = p +1
                           t_class.p = 'COMMENT'
                           t_type.p = 'COMMENT_END'
                           t_val.p = '*'||'/'
                           t_prefix.p = ''
                           p = p +1
                        end l
                        if Opt.0progress then call Progress
                     end   /*  if \HasHeader  */
               end   /*  else  */
            /* -------------------------------------------------------------------------------------- */
         end   /*  if Opt.0AddHeader  */
   end   /*  while t < t_type.0  */
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_HeaderFooter ================================================================== */

   Task_Indent: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: Indent IndentEnd IndentWhen Margin Width Tab Format                                      */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -Indent                 do indenting of lines                                                */
   /*    -!Indent                do not change indenting                                              */
   /*    -Indent=0               same as -!Indent                                                     */
   /*                                                                                                 */
   /*    -IndentEnd              indent END by one tab                                                */
   /*    -!IndentEnd             align END below DO                                                   */
   /*                                                                                                 */
   /*    -IndentWhen             indent WHEN by one tab                                               */
   /*    -!IndentWhen            align WHEN below SELECT                                              */
   /*                                                                                                 */
   /*    -Margin=3               add a three-space left margin in addition to any indenting           */
   /*    -Margin=0               no margin                                                            */
   /*                                                                                                 */
   /*    -Width=101              justify right side of comments at this position, if possible         */
   /*                                                                                                 */
   /*    -Tab=3                  indent each level by 3 spaces                                        */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  This routine only calculates the logical indenting for tokens.  Later tasks may                */
   /*  change the indenting. After that, the Task_ApplyIndent routine will set the indenting          */
   /*  of each line by adjusting the prefix of the first word. Task_Rebuild will then reassemble      */
   /*  the lines and add the margin.                                                                  */
   /* ----------------------------------------------------------------------------------------------- */
   /*  akm: I think of a "logical" indent that reflects what Rexx does, based on keywords             */
   /*  and groups of statements. A "cosmetic" indent arranges lines to get the appearance             */
   /*  someone wants. It is not practical to do them separately, but making the distinction           */
   /*  between logical and cosmetic helps me understand what I am doing. My compromise plan           */
   /*  is to limit this task to logical indenting, with a couple of exceptions. A second              */
   /*  indenting task (Task_Indent2) is where people can make whatever cosmetic changes they want.    */
   /* ----------------------------------------------------------------------------------------------- */
   /*  keywords that increment the next indent until 'END' are 'DO', 'SELECT', 'OTHERWISE'            */
   /*  keywords that increment the next indent for only one statement are THEN and ELSE               */
   /*                                                                                                 */
   /*  I try to not think in terms of lines here. The indent value is how much a word                 */
   /*  =would= be indented, if that word were at the beginning of a line, if it were legal            */
   /*  for that word to start a line. Most tokens do not start a line, but it is easier               */
   /*  to store an indent value for a token than to figure out whether it really needs it.            */
   /* ----------------------------------------------------------------------------------------------- */
   if \Opt.0Indent then return 0
   if Opt.0progress then call Progress SubID()

   if Opt.0Tab < 0 ,
    | \datatype(Opt.0Tab, 'W') then do
         call Msg G.0meID 'Error: cannot use -Tab='Opt.0Tab
         call Egress 'USAGE'
      end
   if Opt.0Width < 20 ,
    | \datatype(Opt.0Width, 'W') then do
         call Msg G.0meID 'Error: cannot use -Width='Opt.0Width
         call Egress 'USAGE'
      end
   if Opt.0Margin < 0 ,
    | \datatype(Opt.0Margin, 'W') then do
         call Msg G.0meID 'Error: cannot use -Margin='Opt.0Margin
         call Egress 'USAGE'
      end

   /*  Indent values will be stored in a t_indent array. Add it to the set of token arrays:           */
   G.0token_vars = G.0token_vars 't_indent.'
   t_indent. = 0 ; t_indent.0 = 0

   Tval = ''
   t = 1
   do while t < t_type.0
      t = t +1
      if t_type.t = 'REXX_W_KEY' then do
            Tval = translate(t_val.t)
            select   /*  based on token Value  */
               when Tval = 'SELECT' then do
                     /* ----------------------------------------------------------------------------- */
                     if Opt.0IndentWhen then do
                           /*  indent all the WHEN blocks between SELECT and END                      */
                           EndDo = R_OtherEnd(t)
                           do s = t +1 to EndDo -1
                              t_indent.s = t_indent.t +1
                           end
                           /*  give END the same indent as SELECT                                     */
                           t_indent.EndDo = t_indent.t
                        end   /*  if Opt.0indentWhen  */
                     /* ----------------------------------------------------------------------------- */
                     else do
                           EndDo = R_OtherEnd(t)
                           do s = t +1 to EndDo -1
                              t_indent.s = t_indent.t
                           end
                           /*  give END the same indent as SELECT                                     */
                           t_indent.EndDo = t_indent.t
                        end
                  end   /*  when SELECT  */

               when Tval = 'THEN' then do
                     /*  indent all of the next clause                                                */
                     start_S = R_NextWord(t)
                     end_S = CL_LastWord(start_S)
                     /*  if THEN is on next line I like to indent it also                             */
                     t_indent.t = T_Indent(R_PrevWord(t)) +1
                     do s = t +1 to end_S
                        t_indent.s = t_indent.t
                     end
                  end   /*  when THEN  */

               /*  ELSE is like IF...THEN but with no THEN to indent                                  */
               when Tval = 'ELSE' then do
                     start_S = R_NextWord(t)
                     end_S = CL_LastWord(start_S)
                     t_indent.t = T_Indent(R_MatchELSE(t))
                     do s = t +1 to end_S
                        t_indent.s = t_indent.t +1
                     end
                  end   /*  when Tval  */

               /*  indent everything from DO to END by one                                            */
               /*  or rather, from the end of the DO statement "DO a = 1 to 3 by 1"                   */
               when Tval == 'DO' then do
                     EndDo = R_OtherEnd(t)
                     start = R_EndEOC(t)
                     do s = start to EndDo -1
                        t_indent.s = t_indent.t +1
                     end
                     t_indent.EndDo = t_indent.t
                     /* ----------------------------------------------------------------------------- */
                     if Opt.0IndentEnd then
                        t_indent.EndDo = t_indent.t +1
                     /* ----------------------------------------------------------------------------- */
                  end   /*  when DO  */

               when Tval = 'OTHERWISE' then do
                     /*  unlike the others, OTHERWISE does not need a DO to group the                 */
                     /*  commands that come after it.  I think of OTHERWISE as having                 */
                     /*  an implied THEN DO ... END, so indent it the same as DO.                     */
                     /*  I suppose that logically it should be double-indented, to align              */
                     /*  with a block inside WHEN...THEN DO. But logic is just a tool.                */
                     EndDo = R_OtherEnd(t)
                     do s = t +1 to EndDo -1
                        t_indent.s = t_indent.t +1
                     end
                  end   /*  when OTHERWISE  */

               otherwise nop
            end   /*  end select Tval  */
         end   /*  end if REXX_W_KEY  */

      if t_indent.t < 0 then do
            if Opt.0verbose then do
                  call Msg SubID 'Warning: negative indent at "'t_val.t'" line 't_line.t
               end
            t_indent.t = 0
         end
   end   /*  do while t  */
   t_indent.0 = t_type.0

   /* ----------------------------------------------------------------------------------------------- */
   /*  for people who put labels inside loops; move label to left side instead of                     */
   /*  indenting it with the code. You could make it configurable if you want labels indented.        */
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_LABEL' then
         t_indent.t = 0
   end t
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Indent ======================================================================== */

   Task_Indent2: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: Indent IndentContinued Continued Outdent ThenDo OutdentThendo Format                     */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -OutdentThenDo          shift indenting one step left after THEN DO or ELSE DO,              */
   /*                            so the block under it is only indented one tab and the               */
   /*                            END is aligned with IF                                               */
   /*                                                                                                 */
   /*    -!OutdentThenDo         use default indenting style                                          */
   /*    -OutdentThenDo=0                                                                             */
   /*                                                                                                 */
   /*    -IndentContinued=''     If indenting is needed, continued lines will be indented             */
   /*                            as a block, to preserve their relative position.                     */
   /*                            In other words, do not change indenting of continued lines.          */
   /*                                                                                                 */
   /*    -IndentContinued=<amount>                                                                    */
   /*                            where <amount> is a multiple of the Tab value.                       */
   /*                            It does not need to be an integer.                                   */
   /*                                                                                                 */
   /*  For example, if the Tab value is 3 spaces:                                                     */
   /*    -IndentContinued=''     do not change indenting of continued lines                           */
   /*    -IndentContinued=0      no extra indent; align continued line with previous line             */
   /*    -IndentContinued=1      indent continued lines 3 spaces                                      */
   /*    -IndentContinued=.3     indent continued lines 1 space                                       */
   /*    -IndentContinued=1.6    indent continued lines 5 spaces                                      */
   /*                                                                                                 */
   /*    -IndentContinued='SMART'                                                                     */
   /*                            is harder to describe. The indent varies depending on the            */
   /*                            line above it, to make it more readable. For example                 */
   /*                                    when a = b ,                                                 */
   /*                                       & a = c then                                              */
   /*                                       if d = e ,                                                */
   /*                                        & d = f then                                             */
   /*                                          ggggg = 'aaaa' ,                                       */
   /*                                                  'bbbb' ,                                       */
   /*                                                  'cccc'                                         */
   /* ----------------------------------------------------------------------------------------------- */
   /*  This task will contain adjustments to the default (logical) indenting to allow                 */
   /*  for individual styles. If you want add new options to the indenting, this is                   */
   /*  probably the place to do it.                                                                   */
   /* =============================================================================================== */
   if Opt.0progress then call Progress SubID()
   if \Opt.0Indent then return 0

   /* ----------------------------------------------------------------------------------------------- */
   /*  adjust indenting one step left after THEN DO                                                   */
   if Opt.0OutdentThenDo then
      do t = 2 to t_type.0 -1
         if R_KeyVal(t) == 'DO' then do
               p = R_PrevWord(t)
               if wordpos(translate(t_val.p), 'THEN ELSE OTHERWISE') > 0 then
                  if L_OnSameLine(p, t) then do
                        EndDo = R_OtherEnd(t)
                        start = R_EndEOC(t)
                        do s = start to EndDo
                           t_indent.s = t_indent.s -1
                        end
                     end
            end
      end t
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  move comments to the left margin instead of indenting them                                     */
   call Task_UnIndentComments
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  indent CONTINUED lines                                                                         */
   do t = 2 to t_type.0 -1
      if t_class.t == 'FILE_EOL' then do
            word1 = t +1
            if L_IsContinued(word1, -1) then
               call DoSmartIndent word1
         end   /*  if FILE_EOL  */
   end t
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Indent2 ======================================================================= */

   Task_Procedure: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: Procedure Format                                                                         */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -Procedure=DOWN              move "procedure" down to start of line after label, as in       */
   /*                                      Label:                                                     */
   /*                                      Procedure expose varA varB                                 */
   /*                                                                                                 */
   /*    -Procedure=UP                move "procedure" up to label line, as in                        */
   /*                                      Label: Procedure expose varA varB                          */
   /*                                                                                                 */
   /*    -Procedure=''                leave "procedure" as it is                                      */
   /*                                                                                                 */
   /* =============================================================================================== */
   if Opt.0Procedure == '' then return 0
   ValidOpts = 'UP DOWN'
   if wordpos(translate(Opt.0Procedure), ValidOpts) == 0 then do
         call Msg 'unrecognized option: "-Procedure='Opt.0Procedure'"'
         return 0
      end

   if Opt.0progress then call Progress SubID()
   do t = 2 to t_type.0 -1
      if t_type.t \= 'REXX_COLON' then iterate t
      if t_readonly.t then iterate t
      eoc = R_NextToken(t)
      Proc = R_NextToken(eoc)
      if translate(t_val.Proc) \== 'PROCEDURE' then iterate t
      if t_type.eoc \== 'REXX_EOC' then do
            call ShowError 'BUG: token' eoc 'should be type REXX_EOC, but is' t_type.eoc
            call ShowError 'This is probably a bug in the Tokeniser or Formatter'
            call Fatality
         end

      if translate(Opt.0Procedure) == 'DOWN' then do
            /*  change the '' EOC to an End-Of-Line EOC                                               */
            t_class.eoc = 'FILE_EOL'
            t_prefix.eoc = ''
            t_val.eoc = ''
         end

      if translate(Opt.0Procedure) == 'UP' then do
            /*  change the '' EOC to an End-Of-Line EOC                                               */
            t_class.eoc = 'REXX'
            t_prefix.eoc = ''
            t_val.eoc = ''
            t_prefix.Proc = ' '
         end
   end t
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Procedure ===================================================================== */

   Task_Rebuild: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call Task_Rebuild [<start>, <stop>]                                                      */
   /*  where <start> and <stop> are token numbers                                                     */
   /*                                                                                                 */
   /*  adds lines to the SrcOut. array by combining the prefix and value of tokens                    */
   /*  if start is not given it defaults to the first token                                           */
   /*  if stop is not given it defaults to the last token                                             */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg start , stop
   if start == '' then start = 1
   if stop == '' then stop = t_type.0
   if debug.0arg then do
         call T_Valid start, Esigl
         call T_Valid stop, Esigl
      end

   /*  a workaround to do what the old Task_Rebuild did when doing all tokens.                        */
   /*  Other tasks that call this to do only certain tokens                                           */
   /*  must handle the progress display themselves                                                    */
   if start == 1 then
      if Opt.0progress then call Progress SubID()

   ln = SrcOut.0 +1
   SrcOut.ln = ''
   do t = start to stop
      SrcOut.ln = SrcOut.ln||t_prefix.t||t_val.t
      if t_class.t == 'FILE_EOL' then do
            SrcOut.ln = strip(SrcOut.ln, 't')
            ln = ln +1
            SrcOut.ln = ''
         end
   end
   SrcOut.0 = ln -1

   if start == 1 then
      if Opt.0progress then call Progress
   return 0
   /* === End of Task_Rebuild ======================================================================= */

   Task_RebuildBare: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  an alternative to Task_Rebuild that removes all comments and empty lines                       */
   /*  Unlike Task_Rebuild, it only works on the whole token array                                    */
   /* =============================================================================================== */
   if Opt.0progress then call Progress SubID()
   /*  a RexxID comment must be included; all other comments are ignored                              */
   rID = RexxID()
   ln = SrcOut.0 +1
   do t = rID to rID +2
      SrcOut.ln = SrcOut.ln||t_val.t
   end t
   ln = ln +1
   do t = 1 to t_type.0
      if t_class.t == 'COMMENT' then iterate
      SrcOut.ln = SrcOut.ln||t_prefix.t||t_val.t
      if t_class.t == 'FILE_EOL' then do
            /*  strip indenting and ignore empty lines                                                */
            SrcOut.ln = strip(SrcOut.ln, 'B')
            if SrcOut.ln \= '' then ln = ln +1
            SrcOut.ln = copies(' ', Opt.0Margin)
         end
   end
   SrcOut.0 = ln -1
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_RebuildBare =================================================================== */

   Task_Recap: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: ReCap BIFs Keywords Capitalization Format                                                */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -ReCap                  recapitalize variables, procedures, Built-In Functions               */
   /*    -!ReCap                 do not recapitalize anything                                         */
   /*    -ReCap=0                                                                                     */
   /*                                                                                                 */
   /*    -BIFs=upper             make all Built-In Functions upper case                               */
   /*    -BIFs=lower             make all Built-In Functions lower case                               */
   /*    -BIFs=mixed             make all Built-In Functions mixed case                               */
   /*    -BIFs=''                capitalize each BIF according to its first occurrence                */
   /*                                                                                                 */
   /*    -Keywords=upper         make all keywords upper case                                         */
   /*    -Keywords=lower         make all keywords lower case                                         */
   /*    -Keywords=mixed         make all keywords mixed case                                         */
   /*    -Keywords=''            capitalize each keyword according to its first occurrence            */
   /*                                                                                                 */
   /*    The name of an internal routine is always capitalized to match its label.                    */
   /*                                                                                                 */
   /*    Hex/binary constants are always capitalized as in: 'FF'x                                     */
   /*                                                                                                 */
   /*    The first place a variable name is used will determine how it is capitalized.                */
   /* =============================================================================================== */
   /*  To re-capitalize variables, keep a list of all variable names. The list                        */
   /*  represents the correct capitalization for each name. For each variable t_val.t,                */
   /*  if it is already in the list, copy name from the list to t_val.t. If it is not                 */
   /*  in the list, then add t_val.t to the list. The result is that each variable will               */
   /*  be capitalized according to the first occurrence of that variable.                             */
   /*  Procedures and keywords are done similarly, but the lists are first                            */
   /*  seeded with the preferred capitalization.                                                      */
   /*                                                                                                 */
   /*  Note that in a line like                                                                       */
   /*        signal on novalue                                                                        */
   /*  "novalue" is a sub-keyword, not a procedure name, so it will not be recapped                   */
   /*  to match the label "NoValue".                                                                  */
   /* ----------------------------------------------------------------------------------------------- */
   if \Opt.0ReCap then return 0
   ValidOpts = 'UPPER LOWER MIXED'
   if Opt.0Keywords \= '' ,
    & wordpos(translate(Opt.0Keywords), ValidOpts) == 0 then do
         call Msg 'unrecognized option: "-Keywords='Opt.0Keywords'"'
      end
   if Opt.0BIFs \= '' ,
    & wordpos(translate(Opt.0BIFs), ValidOpts) == 0 then do
         call Msg 'unrecognized option: "-BIFs='Opt.0BIFs'"'
      end
   SubID = SubID()
   if Opt.0progress then call Progress SubID

   /* ----------------------------------------------------------------------------------------------- */
   /*  Seed the lists:                                                                                */
   /*  for internal routines, the label is the authoritative source for capitalization                */
   LabelList = ''
   VarList = ''
   KeyList = ''
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_LABEL' then do
            /*  note that for this purpose, sub-labels are included                                   */
            if wordpos(translate(t_val.t), translate(LabelList)) == 0 then
               LabelList = LabelList t_val.t
         end   /*  if LABEL  */
   end t

   /*  if Opt.0BIFs == '' then do not seed the list, just re-cap according to the                     */
   /*  first one found. Otherwise seed it with the upper, lower, or mixed-case                        */
   /*  versions of the BIFs.                                                                          */
   if Opt.0BIFs \= '' then do
         AllBIFs = G.0AllBIFs
         select
            when translate(Opt.0BIFs) == 'UPPER' then
               AllBIFs = translate(AllBIFs)
            when translate(Opt.0BIFs) == 'LOWER' then
               AllBIFs = translate(AllBIFs, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')
            otherwise nop
         end
         LabelList = LabelList AllBIFs
      end   /*  if Opt.0BIFs  */

   /*  As above, if Opt.0Keywords == '' then do not seed the list.                                    */
   if Opt.0Keywords \= '' then do
         KeyList = G.0Keywords G.0Keywords2
         select
            when translate(Opt.0Keywords) == 'UPPER' then
               KeyList = translate(KeyList)
            when translate(Opt.0Keywords) == 'LOWER' then
               KeyList = translate(KeyList, 'abcdefghijklmnopqrstuvwxyz', 'ABCDEFGHIJKLMNOPQRSTUVWXYZ')
            otherwise nop
         end
      end   /*  if Opt.0keywords  */

   /* ----------------------------------------------------------------------------------------------- */
   do t = 2 to t_type.0 -1
      select
         /* ----------------------------------------------------------------------------------------- */
         when t_type.t == 'REXX_W_PROCEDURE' then do
               /*  a procedure name in quotes has a special meaning and should not be tampered with   */
               if \S_IsQuoted(t_val.t) then do
                     NamePos = wordpos(translate(t_val.t), translate(LabelList))
                     if NamePos == 0 then
                        LabelList = LabelList t_val.t
                     else
                        t_val.t = word(LabelList, NamePos)
                  end
            end   /*  when REXX_W_PROCEDURE  */
         /* ----------------------------------------------------------------------------------------- */
         when t_type.t == 'REXX_W_SYMBOL' then do
               NamePos = wordpos(translate(t_val.t), translate(VarList))
               if NamePos == 0 then
                  VarList = VarList t_val.t
               else
                  t_val.t = word(VarList, NamePos)
            end   /*  when REXX_W_SYMBOL  */
         /* ----------------------------------------------------------------------------------------- */
         when abbrev(t_type.t, 'REXX_W_KEY') then do
               NamePos = wordpos(translate(t_val.t), translate(KeyList))
               if NamePos == 0 then
                  KeyList = KeyList t_val.t
               else
                  t_val.t = word(KeyList, NamePos)
            end   /*  if REXX_W_KEY  */
         /* ----------------------------------------------------------------------------------------- */
         when t_type.t == 'REXX_W_CONSTANT' then do
               if R_IsHexBin(t) then do
                     base = right(t_val.t, 1)
                     str = left(t_val.t, length(t_val.t) -1)
                     str = translate(str)
                     base = translate(base, 'bx', 'BX')
                     t_val.t = str||base
                  end
            end
         /* ----------------------------------------------------------------------------------------- */
         when t_type.t == 'REXX_W_LITERAL' then do
               IsVar = translate(QuotedVar(t))
               if IsVar \= '' then do
                     /*  a quoted var name. if it is stem, strip '.'                                  */
                     varN = Unquoted(t_val.t)
                     Qchar = left(t_val.t, 1)
                     if right(varN, 1) == '.' then
                        dotChar = '.'
                     else
                        dotChar = ''
                     varN = strip(varN, 'T', '.')
                     NamePos = wordpos(translate(varN), translate(VarList))
                     if NamePos == 0 then
                        VarList = VarList varN
                     else
                        t_val.t = Qchar||word(VarList, NamePos)||dotChar||Qchar
                  end   /*  select  */
            end   /*  when  */
         /* ----------------------------------------------------------------------------------------- */
         otherwise nop
      end   /*  select  */
   end t
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Recap ========================================================================= */

   Task_ReSpace: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: ReSpace SpaceConcat Space SpaceNot SpaceParenth Parentheses Concatenation Format         */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -ReSpace                change spaces around comments, operators, commas, etc.               */
   /*    -!ReSpace               do not change spacing                                                */
   /*    -ReSpace=0                                                                                   */
   /*                                                                                                 */
   /*    -SpaceConcat                                                                                 */
   /*    -SpaceConcat=1          put spaces around the concatenation operator '||', as in             */
   /*                                 SAY abc || def                                                  */
   /*    -SpaceConcat=0          no spaces around '||', as in                                         */
   /*                                 SAY abc||def                                                    */
   /*    -SpaceConcat=''         leave as-is                                                          */
   /*                                                                                                 */
   /*    -SpaceNot                                                                                    */
   /*    -SpaceNot=1             put a space after the NOT operator, as in                            */
   /*                                if \ abc then                                                    */
   /*    -!SpaceNot                                                                                   */
   /*    -SpaceNot=0             no space after the NOT operator, as in                               */
   /*                                if \abc then                                                     */
   /*    -SpaceNot=''            leave spacing as-is                                                  */
   /*                                                                                                 */
   /*    -SpaceParenth                                                                                */
   /*    -SpaceParenth=1         put spaces around words inside parentheses, as in                    */
   /*                                 ret = Foo( a, b, c )                                            */
   /*                                 IF ( a == b )                                                   */
   /*    -!SpaceParenth                                                                               */
   /*    -SpaceParenth=0         do not use spaces, as in                                             */
   /*                                 ret = Foo(a, b, c)                                              */
   /*                                 IF (a == b)                                                     */
   /*    -SpaceParenth=''        leave as-is                                                          */
   /*                                                                                                 */
   /* =============================================================================================== */
   if Opt.0ReSpace == 0 then return 0
   if Opt.0progress then call Progress SubID()
   do t = 2 to t_type.0 -1
      if t_readonly.t then iterate
      if t_class.t == 'FILE_EOL' then iterate
      n = R_NextWord(t)
      p = R_PrevWord(t)

      /*  un-crowd COMMENTS:                                                                          */
      /* it is ugly and hard to read, but some people write things like                               */
      /*          IF xxx THEN/@ a crowded comment @/                                                  */
      if abbrev(t_type.t, 'COMMENT_BEG') then do
            if C_CommentType(t) = 'RIGHT' then do
                  if length(t_prefix.t) < 3 then
                     t_prefix.t = '   '
                  iterate
               end
         end

      /*  Re-space Rexx OPERATORS:                                                                    */
      /*  sometimes people write statements that are aligned, like                                    */
      /*     a        = 1                                                                             */
      /*     bbbb     = 1                                                                             */
      /*  try to preserve the alignment by only forcing a prefix if it is '',                         */
      select
         when t_type.t == 'REXX_W_NOT' then do
               /*  I like "if \foo" better than "if \ foo"                                            */
               if t_prefix.t == '' then t_prefix.t = ' '
               select
                  when Opt.0SpaceNot == 1 then
                     t_prefix.n = ' '
                  when Opt.0SpaceNot == 0 then
                     t_prefix.n = ''
                  otherwise nop
               end
            end

         when t_type.t == 'REXX_W_LOGIC' then do
               /*  BOOLEAN operators get spaces:                                                      */
               /*  if t_val.t == '&'    then iterate                                                  */
               /*  if t_val.t == '|'    then iterate                                                  */
               /*  if t_val.t == '&&'   then iterate                                                  */
               if t_prefix.t == '' then t_prefix.t = ' '
               if t_prefix.n == '' then t_prefix.n = ' '
            end   /*  if LOGIC  */

         /*  COMPARISON operators and ASSIGNMENT operators:                                           */
         when t_type.t == 'REXX_W_COMPARE' ,
            | t_type.t == 'REXX_W_ASSIGN' then
            select
               /*  if you want to leave one of these operators as-is, uncomment its line              */
               /*  when t_val.t == '<'    then iterate                                                */
               /*  when t_val.t == '>'    then iterate                                                */
               /*  when t_val.t == '='    then iterate                                                */
               /*  when t_val.t == '=='   then iterate                                                */
               /*  when t_val.t == '<='   then iterate                                                */
               /*  when t_val.t == '>='   then iterate                                                */
               /*  when t_val.t == '<>'   then iterate                                                */
               /*  when t_val.t == '<<'   then iterate                                                */
               /*  when t_val.t == '>>'   then iterate                                                */
               /*  when t_val.t == '<<='  then iterate                                                */
               /*  when t_val.t == '>>='  then iterate                                                */
               when 1 == 2 then nop
               otherwise
                  if t_type.p == 'REXX_W_NOT' then
                     /*  "if t \==" rather than "if t \ =="                                           */
                     t_prefix.t = ''
                  else
                     if t_prefix.t == '' then t_prefix.t = ' '
                  if t_prefix.n == '' then  t_prefix.n = ' '
            end   /*  when compare or assign  */

         when t_type.t == 'REXX_W_CONCAT' then do
               /*  CONCATENATION operator:                                                            */
               /*  I prefer "xxx||yyy" to "xxx || yyy"                                                */
               select
                  /* -------------------------------------------------------------------------------- */
                  when Opt.0SpaceConcat == 1 then do
                        t_prefix.t = ' '
                        t_prefix.n = ' '
                     end
                  when Opt.0SpaceConcat == 0 then do
                        t_prefix.t = ''
                        t_prefix.n = ''
                     end
                  /* -------------------------------------------------------------------------------- */
                  otherwise nop
               end
            end

         when t_type.t == 'REXX_W_MATH' then
            select
               /*  MATH operators                                                                     */
               /*  The spacing here is inconsistent but it is the way that looks best to me           */
               /*  It will want tweaking as I find things that I don't like.                          */
               /*  Most math operators get both spaces, but a '-' could mean either                   */
               /*  subtraction or a negative number, so I prefer "b = -1", "b = b -1"                 */
               /*                                                                                     */
               /*  I should have left more notes about what I was doing here...                       */
               when t_val.t == '+' ,
                  | t_val.t == '-' then do
                     select
                        when datatype(t_val.n, 'n') ,
                           & t_type.p \= 'REXX_W_COMPARE' ,
                           & t_type.p \= 'REXX_W_ASSIGN' then do
                              /*  a = b -1 ; a = b +1                                                 */
                              /*  say --1                                                             */
                              if t_prefix.t == '' ,
                               & pos(t_val.p, '+-') == 0 then
                                 t_prefix.n = ''
                              if pos(t_val.p, '-+') == 0 then
                                 if t_prefix.t == '' then t_prefix.t = ' '
                           end

                        when datatype(t_val.n, 's') ,
                           & t_type.p \= 'REXX_W_COMPARE' ,
                           & t_type.p \= 'REXX_W_ASSIGN' then do
                              /*  say --b                                                             */
                              if t_prefix.t == '' ,
                               & pos(t_val.p, '+-') == 0 then
                                 t_prefix.n = ''
                              if pos(t_val.p, '-+') == 0 then
                                 if t_prefix.t == '' then t_prefix.t = ' '
                           end

                        when pos(t_val.n, '-+') > 0 ,
                           & t_type.p \= 'REXX_W_COMPARE' ,
                           & t_type.p \= 'REXX_W_ASSIGN' then do
                              /*  say --1 ; say -+1                                                   */
                              if t_prefix.t == '' then t_prefix.t = ' '
                              t_prefix.n = ''
                           end


                        when t_type.p == 'REXX_W_COMPARE' ,
                           | t_type.p == 'REXX_W_ASSIGN' then do
                              /*  a = -c ; if a <= -c                                                 */
                              if t_prefix.t == '' then t_prefix.t = ' '
                              t_prefix.n = ''
                           end

                        when t_type.p == 'REXX_W_(' then do
                              /* a = Foo(-c)                                                          */
                              t_prefix.t = ''
                              t_prefix.n = ''
                           end

                        otherwise
                           /*  a = b - c                                                              */
                           if t_prefix.t == '' then t_prefix.t = ' '
                           if t_prefix.n == '' then t_prefix.n = ' '
                     end   /*  select  */

                  end   /*  when t_val.t == '+' or '-'  */

               /*  when t_val.t == '*'    then nop                                                    */
               /*  when t_val.t == '/'    then nop                                                    */
               /*  when t_val.t == '//'   then nop                                                    */
               /*  when t_val.t == '%'    then nop                                                    */
               /*  when t_val.t == '**'   then nop                                                    */
               otherwise
                  if t_prefix.t == '' then t_prefix.t = ' '
                  if t_prefix.n == '' then t_prefix.n = ' '

            end   /*  when MATH  */
         otherwise nop
      end   /*  select t_type.t  */

      /* -------------------------------------------------------------------------------------------- */
      if Opt.0SpaceParenth == 0 then do
            /*  ReSpace PARENTHESES:                                                                  */
            /*  as in:                                                                                */
            /*        a = Foo(a, b, c)                                                                */
            /*  if previous token is "(", no prefix for this token, regardless of above               */
            /*  if this token is ")", no prefix for it                                                */
            if t_type.p == 'REXX_W_(' then
               t_prefix.t = ''
            if t_type.t == 'REXX_W_)' then
               t_prefix.t = ''
         end
      else do
            /*  ReSpace PARENTHESES, alternate version:                                               */
            /*  as in:                                                                                */
            /*        a = Foo( a, b, c )                                                              */
            if t_type.p == 'REXX_W_(' then
               t_prefix.t = ' '
            if t_type.t == 'REXX_W_)' then
               t_prefix.t = ' '
         end
      /* -------------------------------------------------------------------------------------------- */

      /*  ReSpace SEMICOLON:                                                                          */
      if t_type.t == 'REXX_EOC' ,
       & t_val.t = ';' then do
            if t_prefix.t == '' then t_prefix.t = ' '
            if t_prefix.n == '' then t_prefix.n = ' '
         end

      /*  ReSpace COLON:                                                                              */
      if t_type.t == 'REXX_COLON' then do
            t_prefix.t = ''
            if L_OnSameLine(t, n) then t_prefix.n = ' '
         end

      /*  ReSpace COMMA:                                                                              */
      /*  Spaced commas make CALL statements easier to read and spot mistakes,                        */
      /*  This could probably use some tweaking                                                       */
      /*  If you want to change the comma spacing, probably the best way                              */
      /*  is to insert some "if Opt.0xxx then..." lines into this block.                              */
      select
         when t_type.t = 'REXX_W_COMMA' then do
               t_w1 = CL_FirstWord(t)
               select
                  when translate(t_val.t_w1) == 'CALL' then do
                        if T_Type(t -1) \= 'REXX_W_COMMA' then
                           t_prefix.t = ''
                        if t_prefix.n == '' then t_prefix.n = ' '
                     end
                  when translate(t_val.t_w1) == 'PARSE' then do
                        t_prefix.t = ' '
                        t_prefix.n = ' '
                     end
                  otherwise
                     /*  you could make function calls more compact, like "function(x,y,z)"           */
                     /*                t_prefix.t = ''                                                */
                     /*                t_prefix.n = ''                                                */
                     /*  or the same as CALL--                                                        */
                     if T_Type(t -1) \= 'REXX_W_COMMA' then
                        t_prefix.t = ''
                     t_prefix.n = ' '
               end   /*  end SELECT t_w1  */
            end   /*  end DO if a Comma  */

         when t_type.t = 'CONTINUE' then
            /*  this comma should have at least one space before, but maybe more                      */
            if t_prefix.t == '' then t_prefix.t = ' '
         otherwise nop
      end   /*  select t_type.t  */
   end t
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_ReSpace ======================================================================= */

   Task_SpaceLabels: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Spacer Space Blank Label Format                                                          */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -Spacer=<number>              space routines by putting <number> empty lines                 */
   /*                                   above each label                                              */
   /*  Examples:                                                                                      */
   /*    -Spacer=1                     put one empty line above each label                            */
   /*    -Spacer=0                     no empty lines                                                 */
   /*    -Spacer=''                    do not change spacing                                          */
   /* =============================================================================================== */
   if Opt.0Spacer == '' then return 0
   SubID = SubID()
   if Opt.0progress then call Progress SubID
   CurLabel = ''
   t = 1
   do while t <= t_type.0 -1
      t = t +1
      if t_type.t == 'REXX_LABEL' then
         /*  ignore sub-labels                                                                        */
         if \abbrev(translate(t_val.t), translate(CurLabel)||'.') then do
               CurLabel = t_val.t
               /*  only if the label is at the start of a line                                        */
               if t == L_FirstRexxToken(t) then do
                     EOR = B_EndOfPrev(t)
                     /*  count the existing empty lines                                               */
                     NumBlank = 0
                     do while L_IsEmpty(EOR, NumBlank +1)
                        NumBlank = NumBlank +1
                     end

                     select
                        when NumBlank > Opt.0Spacer then do
                              /*  remove some empty lines                                             */
                              if Opt.0progress then call Progress SubID 'respace at label' t_val.t
                              call T_DeleteToken EOR +1, NumBlank - Opt.0Spacer
                              t = EOR + Opt.0Spacer +2
                           end
                        when NumBlank < Opt.0Spacer then do
                              /*  insert some empty lines                                             */
                              /*  bugfix-- for double labels, no blank line(s) between them           */
                              if L_HasRexx(EOR) then
                                 PrevWordType = T_Type(L_FirstRexxToken(EOR))
                              else
                                 PrevWordType = ''
                              if PrevWordType \== 'REXX_LABEL' then do
                                    if Opt.0progress then call Progress SubID 'respace at label' t_val.t
                                    call T_InsertToken EOR +1, Opt.0Spacer - NumBlank
                                    do lf = EOR +1 to EOR + Opt.0Spacer
                                       t_class.lf = 'FILE_EOL'
                                       t_type.lf = 'REXX_EOC'
                                       t_prefix.lf = ''
                                       t_val.lf = ''
                                    end
                                    t = EOR + Opt.0Spacer +2
                                 end   /*  if no double label  */
                           end
                        otherwise nop
                     end   /*  select  */
                  end   /*  if first token  */
            end
   end   /*  while t  */

   if Opt.0progress then call Progress
   return 0
   /* === End of Task_SpaceLabels =================================================================== */

   Task_ShowHelp: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  show help for a subject from all Task_X, Script_X, Style_X, Prep_X, Help_X routines            */
   /*  the first line(s) of the label header must be  HELP: <subject> <subject> ...                   */
   /* =============================================================================================== */
   Patterns = 'Script_* Style_* Task_* Prep_* Help_*'
   call ExtractHelpHeader Patterns, Opt.0help
   if SrcOut.0 > 0 then do
         call ArrayToFile 'SrcOut.', G.0OutFileN
      end
   return 0
   /* === End of Task_ShowHelp ====================================================================== */

   Task_ShowTasks: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: ShowTasks Tasks                                                                          */
   /*                                                                                                 */
   /*  Extracts headers from all of the Script_xxx and Task_xxx routines                              */
   /*  This task is only used by the help system                                                      */
   /* =============================================================================================== */
   if Opt.0progress then call Progress SubID()
   incomment = 0
   do l = 1 to SrcIn.0
      if pos('/'||'*', SrcIn.l) > 0 ,
       | pos('*'||'/', SrcIn.l) > 0 then
         incomment = Incomment(SrcIn.l, incomment)
      if incomment <= 0 then
         if HasLabel(SrcIn.l) then do
               /*  remember the name of the procedure                                                 */
               parse var SrcIn.l label ':' .
               label = strip(label)
               if abbrev(translate(label), 'TASK_') ,
                | abbrev(translate(label), 'SCRIPT_') then do
                     m = l +1
                     if U_DividerType(SrcIn.m) == '=' then do
                           /*  subroutine header starts with "===" divider after the label            */
                           call Out ''
                           call Out '  '||S_UnComment(SrcIn.m)
                           call Out '  '||' ['label']'
                           m = m +1
                           do while m <= SrcIn.0
                              /*  copy everything until the next divider or non-comment               */
                              if \U_IsComment(SrcIn.m) then leave
                              call Out '  '||S_UnComment(SrcIn.m)
                              if U_DividerType(SrcIn.m) == '=' then leave
                              m = m +1
                           end
                           call Out ''
                        end   /*  if found starting divider  */
                  end   /*  if TASK or SCRIPT  */
            end   /*  if is a label  */
   end l
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_ShowTasks ===================================================================== */

   Task_ShowUnCalled: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  shows only a list of labels that are never used                                                */
   /* =============================================================================================== */
   /*  LL.0LabelList holds a list of all labels found.                                                */
   /*  for label XXX, calls are recorded in variables named LL.0XXX_Parent and LL.0XXX_Child,         */
   /*  except the XXX is represented as a hex string. See comments under GetCalls label.              */
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0progress then call Progress SubID()
   call GetCalls
   call Out 'Labels in file 'G.0InfileN 'that are never called:'
   Tlist = LL.0LabelList
   do while Tlist \= ''
      parse var Tlist LabelName Tlist
      /*  exclude the top level and handlers                                                          */
      if LabelName == '[Unlabeled]' then iterate
      if wordpos(LabelName, LL.0Handlers) > 0 then iterate
      !LabelName = Hexed(LabelName)
      !tFrom = 'LL.0'||!LabelName||'_Parent'
      ParentList = strip(value(!tFrom))
      if ParentList == '' then
         call Out G.0Tab||LabelName
   end   /*  while Tlist  */
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_ShowUnCalled ================================================================== */

   Task_Then: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: Then Format                                                                              */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -Then=down              move THEN down to start of next line                                 */
   /*    -Then=up                move THEN up to end of previous line                                 */
   /*    -Then=''                leave THEN as it is                                                  */
   /*                                                                                                 */
   /*                            "DOWN" changes this:                                                 */
   /*                                    IF a THEN                                                    */
   /*                                       SAY a                                                     */
   /*                                                                                                 */
   /*                             to this:                                                            */
   /*                                    IF a                                                         */
   /*                                       THEN SAY a                                                */
   /*                                                                                                 */
   /*                            "UP" does the reverse                                                */
   /*                                                                                                 */
   /*  See also ThenDo                                                                                */
   /* =============================================================================================== */
   /*  this is a simple example of how tokens can be moved. What makes it a little tricky is that     */
   /*  a THEN is surrounded by real or hidden EOC tokens.  They must be moved along with THEN and     */
   /*  sometimes modified.                                                                            */
   /* =============================================================================================== */
   if Opt.0Then == '' then return 0
   if Opt.0progress then call Progress SubID()
   ValidOpts = 'UP DOWN'
   if wordpos(translate(Opt.0Then), ValidOpts) == 0 then do
         call Msg 'unrecognized option: "-Then='Opt.0Then'"'
         return 0
      end

   do t = 2 to t_type.0 -1
      if t_type.t \= 'REXX_W_KEY' then iterate
      if t_readonly.t then iterate
      if translate(t_val.t) = 'THEN' then do
            select
               when translate(Opt.0Then) == 'DOWN' then do
                     /*  if THEN is the last word, and not on a line by itself,                       */
                     /*  move it down                                                                 */
                     if t == L_LastRexxWord(t) ,
                      & t \= L_FirstRexxWord(t) then
                        call R_MoveClauseDown t
                     else do
                           /*  "IF a THEN b = c"                                                      */
                           /*  in this case we must create a new line starting with THEN.             */
                           /*  Just change the EOC before THEN to an EOL                              */
                           tp = R_PrevToken(t)
                           tn = R_NextToken(t)
                           if t_type.tp \= 'REXX_EOC' ,
                            | t_type.tn \= 'REXX_EOC' then do
                                 call ShowError 'EOC error at line' t_line.t
                                 call ShowError Qline(t_line.t), ''
                                 call Fatality
                              end
                           if t_class.tp == 'REXX' ,
                            & t_class.tn == 'REXX' then do
                                 t_class.tp = 'FILE_EOL'
                              end
                        end
                  end
               when translate(Opt.0Then) == 'UP' then do
                     if t == L_FirstRexxToken(t) then
                        call R_MoveClauseUp t
                  end
               otherwise nop
            end   /*  select Opt.0Then  */
         end
   end t
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Then ========================================================================== */

   Task_ThenDo: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*     Help: ThenDo Format                                                                         */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    if the original is:                                                                          */
   /*            IF a == b THEN                                                                       */
   /*               DO                                                                                */
   /*                                                                                                 */
   /*    -ThenDo=UP gives:                                                                            */
   /*            IF a == b THEN DO                                                                    */
   /*                                                                                                 */
   /*    -ThenDo=DOWN gives:                                                                          */
   /*            IF a == b                                                                            */
   /*            THEN DO                                                                              */
   /*                                                                                                 */
   /*    -ThenDo=SPLIT gives:                                                                         */
   /*            IF a == b THEN                                                                       */
   /*               DO                                                                                */
   /*                                                                                                 */
   /*    -ThenDo=''  does not change anything                                                         */
   /*                                                                                                 */
   /*     but a DO followed by more words will be left on a separate line:                            */
   /*            IF a == b THEN                                                                       */
   /*               DO a = 1 to 5                                                                     */
   /*            IF a == b THEN                                                                       */
   /*               DO ; ... ; END                                                                    */
   /*                                                                                                 */
   /*  see also Then                                                                                  */
   /* =============================================================================================== */
   /*  The comments assume an IF...THEN DO phrase where token t_then is "THEN" and                    */
   /*  t_do is "DO". But the phrase may be ELSE DO or OTHERWISE DO, in which case                     */
   /*  t_then is really "ELSE" or "OTHERWISE".                                                        */
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0ThenDo == '' then return 0
   if Opt.0progress then call Progress SubID()
   ValidOpts = 'UP DOWN SPLIT'
   if wordpos(translate(Opt.0ThenDo), ValidOpts) == 0 then do
         call Msg 'unrecognized option: "-ThenDo='Opt.0ThenDo'"'
         return 0
      end

   do t = 2 to t_type.0 -1
      if t_type.t \= 'REXX_W_KEY' then iterate t
      if wordpos(translate(t_val.t), 'THEN ELSE OTHERWISE') == 0 then iterate t
      if T_Type(R_NextWord(t)) \= 'REXX_W_KEY' then iterate t
      if translate(T_Val(R_NextWord(t))) \= 'DO' then iterate t
      if t_readonly.t then iterate t
      /*  found a keyword THEN + keyword DO                                                           */
      t_then = t
      t_do = R_NextWord(t_then)
      select
         /* ----------------------------------------------------------------------------------------- */
         when translate(Opt.0ThenDo) == 'UP' then do
               /*  IF...THEN...DO already; nothing needed                                             */
               /*  bugfix-- use ">= t_do" to recognize "THEN DO ; nop ; END"                          */
               if L_FirstRexxWord(t_then) \= t_then ,
                & L_LastClause(t_then) >= t_do then iterate t
               if translate(t_val.t_then) == 'THEN' then
                  if t_then == L_FirstRexxWord(t_then) then do
                        /*  move the THEN up to the IF line, only if it is not already there:         */
                        /*  this will handle both   IF a              IF a                            */
                        /*                          THEN      and     THEN DO                         */
                        /*                          DO                                                */
                        call R_MoveClauseUp t_then
                     end   /*  if t_then ==  */
               /*  the lines are now:                                                                 */
               /*                 IF a THEN                                                           */
               /*                 DO                                                                  */
               /*  the next step is to move DO up, except--                                           */
               /*  I prefer to keep DO on a separate line if it has parameters.                       */
               /*  Also if the whole DO ... END are on one line, as in                                */
               /*        IF a THEN                                                                    */
               /*           DO ; NOP ; END                                                            */

               /*  Note that the tokens before DO have been moved around, but we did not need         */
               /*  to update the value of t_do because the total number of tokens is the same.        */
               /*  In some other cases we would have to find the DO token again, using                */
               /*  R_NextWord(t_then)                                                                 */
               if L_FirstRexxWord(t_do) == t_do then
                  if translate(T_Val(L_LastRexxWord(t_do))) \= 'END' ,
                   & R_NextWord(t_do) \= t_do +1 then
                     call R_MoveClauseUp t_do
            end   /*  when UP  */
         /* ----------------------------------------------------------------------------------------- */
         when translate(Opt.0ThenDo) == 'DOWN' then do
               select
                  when L_FirstRexxWord(t_then) == t_then ,
                     & L_LastClause(t_then) == t_do then do
                        if translate(t_val.t_then) == 'THEN' then
                           /*  THEN DO on one line; nothing to do                                     */
                           nop
                        else
                           /*  ELSE DO ; move DO down                                                 */
                           call R_MoveClauseDown t_do, 'newline'
                     end
                  when L_FirstRexxWord(t_then) \= t_then ,
                     & L_LastClause(t_then) == t_do then do
                        /*  "IF a THEN DO"                                                            */
                        call R_MoveClauseDown t_then, 'newline'
                     end
                  when L_FirstRexxWord(t_then) == t_then ,
                     & L_LastClause(t_do) == t_do then do
                        /*  "THEN <EOL> DO" on two lines; move DO up                                  */
                        if translate(t_val.t_then) == 'THEN' then
                           call R_MoveClauseUp t_do
                     end
                  when L_LastClause(t_then) == t_then ,
                     & L_FirstRexxWord(t_do) == t_do then do
                        /*  IF THEN <EOL> DO                                                          */
                        call R_MoveClauseDown t_then
                     end
                  otherwise nop
               end   /*  select  */
            end   /*  when DOWN  */

         /* ----------------------------------------------------------------------------------------- */
         when translate(Opt.0ThenDo) == 'SPLIT' then do
               select
                  when L_FirstRexxWord(t_then) == t_then ,
                     & L_LastClause(t_then) == t_do then do
                        /*  THEN DO on one line; move THEN up                                         */
                        if translate(t_val.t_then) == 'THEN' then
                           call R_MoveClauseUp t_then
                        else
                           call R_MoveClauseDown t_do, 'newline'
                     end   /*  when  */
                  when L_FirstRexxWord(t_then) \= t_then ,
                     & L_LastClause(t_then) == t_do then do
                        /*  "IF a THEN DO"                                                            */
                        /*  move DO to a new line. But we cannot move words to a line                 */
                        /*  that does not exist yet. We can make the new line and move                */
                        /*  the words by simply changing the <EOC> before DO to an <EOL>.             */
                        /*  That will carry any comment(s) to the new line, which is                  */
                        /*  probably not what the user wants, so move the comment back.               */
                        call R_MoveClauseDown t_do, 'newline'
                     end   /*  when  */
                  when L_FirstRexxWord(t_then) == t_then ,
                     & L_LastClause(t_do) == t_do then do
                        /*  "THEN <EOL> DO" on two lines                                              */
                        /*  move THEN up                                                              */
                        if translate(t_val.t_then) == 'THEN' then
                           call R_MoveClauseUp t_then
                     end   /*  when  */
                  otherwise nop
               end   /*  select  */
            end   /*  when SPLIT  */
         otherwise nop
      end   /*  select Opt.0ThenDo  */
   end t

   if Opt.0progress then call Progress
   return 0
   /* === End of Task_ThenDo ======================================================================== */

   Task_Tokenize: procedure expose (Shared_vars) (all_tokens) BIFs. /*fold00*/
   /* =============================================================================================== */
   /*  call the Tokeniser and do some processing of the tokens it returned                            */
   /* ----------------------------------------------------------------------------------------------- */
   /*  the Toker will return one or more words in the variable G.0TokerExit                           */
   /*  ''       --succeeded without problems                                                          */
   /*  FILE     --something wrong with the source file. We already checked it for                     */
   /*             null characters, so it is probably defective Rexx code. Should not                  */
   /*             be considered a fatal error, but the result of trying to format                     */
   /*             defective code should be checked carefully.                                         */
   /*  NOTREXX  --the Toker thinks the file is not Rexx (Object Rexx, for example)                    */
   /*             if the -Dialect switch was used, we continue. Otherwise we quit.                    */
   /*  BUG      --indicates a probable bug in the Toker itself, so we should quit.                    */
   /*                                                                                                 */
   /*  if the -Warn switch was used, the Toker will show messages about problems it found             */
   /*                                                                                                 */
   /* =============================================================================================== */
   G.0TokerExit = Toker()
   if debug.0token then call SaveTokensT filespec('N', G.0InfileN)||'_Tokens_Begin'
   select
      when G.0TokerExit = '' then nop
      when wordpos('BUG', G.0TokerExit) > 0 then do
            call ShowError 'Error: Tokeniser returned "BUG" for' G.0InfileN
            call Fatality
         end
      when wordpos('NOTREXX', G.0TokerExit) > 0 then do
            if Opt.0dialect == '' then do
                  call Msg G.0meID 'Tokeniser returned "NOTREXX" for' G.0InfileN
                  call Msg G.0meID_ind 'if you are sure this is Rexx, use -dialect=RexxSAA or -dialect=Regina'
                  call Egress 'NotRexx'
               end
            else do
                  call Msg G.0meID 'Warning: Tokeniser returned "NOTREXX" for' G.0InfileN
                  call Msg G.0meID_ind 'Continuing, but there may be more errors'
               end
         end
      when wordpos('FILE', G.0TokerExit) > 0 then do
            /*  some other kind of defective file. If the Toker continued, we will also,              */
            /*  but there is no telling what will happen next                                         */
            /*  fix: maybe there should be an option to tell it to abort if there were                */
            /*  file errors                                                                           */
            if translate(G.0Command) \= 'CHECKONLY' then do
                  call Msg G.0meID 'Warning: Tokeniser returned error "FILE" for' G.0InfileN
                  call Msg G.0meID_ind 'Continuing, but there may be more errors'
               end
         end
      otherwise
         /*  if Toker returned something else, there is a bug in the Toker                            */
         call ShowError 'BUG: Toker returned "'G.0TokerExit'" for' G.0InfileN
         call Fatality
   end   /*  select G.0TokerExit  */
   return 0
   /* === End of Task_Tokenize ====================================================================== */

   Task_Tree: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  used by Script_Tree                                                                            */
   /*                                                                                                 */
   /*  draws a tree diagram for the file, or for labels specified with the -where= switch             */
   /* =============================================================================================== */
   SubID = SubID()
   if Opt.0progress then call Progress SubID

   Tree. = ''
   Tree.0Indent = 0
   Tree.0maxIndent = 0
   Tree.0tabval = 10
   Tree.0ancestors = ''
   /*  the 'Conn' string uses ASCII characters unless we are using OS/2 codepage 850 or 437           */
   /*  with line-drawing characters.                                                                  */
   /*  check interpreter because Regina Regutil does not have SysQueryProcessCodePage()               */
   Tree.0conn = ' |___'
   if \Opt.0ascii then
      if G.0OpSystem == 'OS/2' then
         if G.0Interpreter == 'REXXSAA' then do
               cp = SysQueryProcessCodePage()
               if cp == '850' ,
                | cp == '437' then
                  Tree.0conn = ' '
            end
   Lchar = left(strip(Tree.0conn), 1)
   if Lchar == '' then do
         Ichar = ''
         Tchar = ''
      end
   else do
         Ichar = '|'
         Tchar = '|'
      end

   Opt.0Where = ExpandLabels(Opt.0Where)
   Include = Opt.0Where
   if Opt.0Show == 1 then Opt.0Show = ''
   if Opt.0Show == '' then
      Tree.0Show = 'Int,Ext'
   else
      Tree.0Show = Opt.0Show
   Tree.0Show = translate(Tree.0Show, ' ', ',')
   Tree.0Show = translate(Tree.0Show)

   /* ----------------------------------------------------------------------------------------------- */
   call GetCalls
   /*  if -where was not used, map the whole thing. But we do not want to draw a tree for             */
   /*  each routine; that would be redundant and pointless. Only start a tree from                    */
   /*  any routine that has no calls to it.                                                           */
   if Include == '' then do
         Tlist = LL.0LabelList
         do while Tlist \= ''
            parse var Tlist LabelName Tlist
            !LabelName = Hexed(LabelName)
            !tFrom = 'LL.0'||!LabelName||'_Parent'
            ParentList = strip(value(!tFrom))
            !tTo = 'LL.0'||!LabelName||'_Child'
            ChildList = strip(value(!tTo))
            if ParentList == '' then
               Include = Include LabelName
            else do
                  /*  if all parents are also children, it is a loop and we need to draw this label.  */
                  /*  if any parent is not a child, we do not.                                        */
                  /*  fix: this still won't catch a 3-part loop: Main-->Sub1-->Sub2-->Main...         */
                  DrawIt = 1
                  do while ParentList \= ''
                     parse var ParentList Parent ParentList
                     if wordpos(translate(Parent), translate(LabelName ChildList)) == 0 then do
                           DrawIt = 0
                           leave
                        end
                  end
                  if DrawIt then
                     Include = Include LabelName
               end
         end
      end
   Include = strip(Include)
   if Include == '' then
      call Out 'there is nothing to show'
   /* ----------------------------------------------------------------------------------------------- */
   /*  verify that the routines exist, and correct capitalization while we are at it                  */
   incList = Include
   do while incList \= ''
      parse var incList inc incList
      if wordpos(translate(inc), translate(LL.0LabelList)) == 0 then
         call Msg SubID 'Warning: cannot find subroutine named "'inc'"'
   end

   /*  now save the results to SrcOut.                                                                */
   ln = SrcOut.0
   Tlist = LL.0LabelList
   /*  the tree diagram:                                                                              */
   do while Tlist \= ''
      parse var Tlist LabelName Tlist
      if wordpos(translate(LabelName), translate(Include)) > 0 then do
            if Opt.0progress then call Progress SubID LabelName
            ln = ln +1
            SrcOut.ln = copies(' ', Tree.0Indent * Tree.0tabval) '    [Int] 'LabelName
            SrcOut.0 = ln
            call Tree_OneLabel LabelName
            ln = SrcOut.0
         end
   end
   if Opt.0progress then call Progress

   /* ----------------------------------------------------------------------------------------------- */
   /*  make the tree easier to read by connecting the lines                                           */
   do ln = 1 to SrcOut.0
      if Opt.0progress then
         if ln // 1000 == 0 then
            call Progress SubID 'draw connectors for line' ln 'of' SrcOut.0
      lpos = pos(Lchar, SrcOut.ln)
      if lpos > 0 then
         do l = ln -1 to 1 by -1
            if substr(SrcOut.l, lpos, 1) == Lchar then do
                  SrcOut.l = overlay(Tchar, SrcOut.l, lpos)
               end
            if substr(SrcOut.l, lpos, 1) \== ' ' then leave l
            SrcOut.l = overlay(Ichar, SrcOut.l, lpos)
         end l
   end ln

   if LL.0Handlers \= '' then do
         ln = SrcOut.0 +1 ; SrcOut.0 = ln
         SrcOut.ln = 'May also use the error-handlers:'
         tempstr = LL.0Handlers
         do while tempstr \= ''
            parse var tempstr FuncN tempstr
            ln = ln +1
            SrcOut.ln = G.0Tab||FuncN
            SrcOut.0 = ln
         end
      end
   ln = ln +1
   SrcOut.ln = 'Maximum nesting level was' Tree.0maxIndent
   SrcOut.0 = ln

   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Tree ========================================================================== */

   Tree_OneLabel: procedure expose (Shared_vars) SrcOut. LL. Tree. /*fold00*/
   /* =============================================================================================== */
   /*  This is part of Task_Tree                                                                      */
   /*  It shows the subroutines called by one routine.  For each internal subroutine, it calls        */
   /*  itself to show the next level of calls                                                         */
   /* ----------------------------------------------------------------------------------------------- */
   /*  The lists are contained in variables named after the routines. They are all tails              */
   /*  of the stem variable 'LL.' That way this routine can expose those variables without            */
   /*  knowing their names.                                                                           */
   /*  Note that the tails are stored as hex strings. See comments under GetCalls for explanation.    */
   /* ----------------------------------------------------------------------------------------------- */
   parse arg LabelName

   ln = SrcOut.0
   !LabelName = Hexed(LabelName)
   !ChildList = 'LL.0'||!LabelName||'_Child'
   ChildList = strip(value(!ChildList))

   do while ChildList \= ''
      parse var ChildList Pname ChildList
      /*  to reduce the chance of a "Control Stack Full" error I replaced a SELECT block              */
      /*  with a series of  "IF-THEN-ITERATE" blocks and moved "call Tree_OneLabel"                   */
      /*  outside the IF-THEN-ELSE DO block. It means two fewer control structures for each           */
      /*  level of recursion                                                                          */
      if wordpos(translate(Pname), translate(LL.0LabelList)) > 0 then do
            ln = ln +1
            SrcOut.ln = copies(' ', (Tree.0Indent +1) * Tree.0tabval)||Tree.0conn||'[Int] 'Pname
            SrcOut.0 = ln
            /*  recursion check: do not descend if Pname is its own parent, grandparent, etc.         */
            Tree.0ancestors = LabelName Tree.0ancestors
            if wordpos(Pname, Tree.0ancestors) > 0 then do
                  SrcOut.ln = SrcOut.ln||' ...'
                  parse var Tree.0ancestors . Tree.0ancestors
                  iterate
               end

            Tree.0Indent = Tree.0Indent +1
            Tree.0maxIndent = max(Tree.0maxIndent, Tree.0Indent)
            call Tree_OneLabel Pname
            ln = SrcOut.0
            Tree.0Indent = Tree.0Indent -1
            parse var Tree.0ancestors . Tree.0ancestors
            iterate
         end

      if wordpos(translate(Pname), translate(G.0AllBIFs)) > 0 then do
            if wordpos('BIF', Tree.0Show) > 0 then do
                  ln = ln +1
                  SrcOut.ln = copies(' ', (Tree.0Indent +1) * Tree.0tabval)||Tree.0conn||'[BIF] 'Pname
                  SrcOut.0 = ln
               end
            iterate
         end

      if wordpos('EXT', Tree.0Show) > 0 then do
            ln = ln +1
            SrcOut.ln = copies(' ', (Tree.0Indent +1) * Tree.0tabval)||Tree.0conn||'[Ext] 'Pname
            SrcOut.0 = ln
         end
   end   /*  while ChildList \= ''  */
   return 0
   /* === End of Tree_OneLabel ====================================================================== */

   Task_UnIndentComments: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: IndentDivider IndentMulti Indent Comments Divider Format                                 */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -IndentDivider           divider comments are indented along with the Rexx statements        */
   /*    -IndentDivider=0         divider comments are full-width starting at the left margin         */
   /*    -!IndentDivider                                                                              */
   /*                                                                                                 */
   /*    -IndentMulti             multi-line comments are indented with the Rexx statements           */
   /*    -IndentMulti=0           multi-line comments start at the left margin                        */
   /*    -!IndentMulti                                                                                */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  This must be called after Task_Indent, because this overrides some of the                      */
   /*  indent values that Task_Indent assigned to comment tokens.                                     */
   /*  This must be called before Task_FormatComments, which will correct the divider widths          */
   /* ----------------------------------------------------------------------------------------------- */
   Esigl = sigl
   if Opt.0IndentDivider ,
    & Opt.0IndentMulti then return 0
   if symbol('t_indent.0') \== 'VAR' then do
         call ShowError 'BUG: in UnIndentComments, indenting has not been done yet'
         call ShowError 'called from line' Esigl, ''
         call Fatality
         return 0
      end

   if Opt.0progress then call Progress SubID()
   do t = 2 to t_type.0 -1
      if t_readonly.t then iterate
      if t_type.t \= 'COMMENT_BEG' then iterate t
      ctype = C_CommentType(t +1)
      select
         when ctype = 'FULL' then do
               if \Opt.0IndentDivider then do
                     if C_IsDivider(t +1) then
                        do s = t to C_OtherEnd(t)
                           t_indent.s = 0
                        end
                  end
            end
         when abbrev(ctype, 'MULTI') then do
               if \Opt.0IndentMulti then
                  do s = t to C_OtherEnd(t)
                     t_indent.s = 0
                  end
            end
         otherwise nop
      end   /*  select  */
   end t
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_UnIndentComments ============================================================== */

   Task_UnusedVars: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  used by Script_Unused                                                                          */
   /*                                                                                                 */
   /*  shows unused variables, See Script_Unused for the switches it uses                             */
   /* =============================================================================================== */
   SubID = SubID()
   Opt.0Where = ExpandLabels(Opt.0Where)
   Include = Opt.0Where
   if Opt.0Flat then
      LabelList = '[Unlabeled]'
   else do
         LabelList = LabelList()
         LabelList = WordSort(LabelList)
         LabelList = strip(LabelList)
      end

   do i = 1 to words(Include)
      ProcN = word(Include, i)
      if wordpos(translate(ProcN), translate(LabelList)) == 0 then
         call Msg SubID 'Warning: cannot find a label called "'ProcN'"'
   end i
   Include = translate(Include)
   call GetCalls
   if Opt.0Flat then
      call MakeVarListsE '-FLAT'
   else
      call MakeVarListsE
   if Opt.0progress then call Progress SubID
   ProcList = translate(ProcedureList())

   if Opt.0Flat then
      call Out 'variables that may be unused:'
   else
      call Out 'variables that may be unused, by label:'

   do w = 1 to words(LabelList)
      ProcName = word(LabelList, w)
      if Include == '' ,
       | wordpos(translate(ProcName), Include) > 0 then do
            !ProcName = Hexed(ProcName)
            ExposedList = translate(Exposed.!ProcName)
            VarList = WordSort(Assigned.!ProcName)
            labelshown = 0
            if words(VarList) > 0 then do
                  do x = 1 to words(VarList)
                     thisvar = word(VarList, x)
                     /*  first eliminate variables assigned in the procedure                          */
                     if wordpos(translate(thisvar), translate(Used.!ProcName)) > 0 then iterate x
                     /*  eliminate exposed variables; they are probably used elsewhere                */
                     if wordpos(translate(thisvar), ExposedList) > 0 then iterate x
                     /*  also eliminate variables whose stem is exposed                               */
                     dot = pos('.', thisvar)
                     if dot > 0 then do
                           stem = left(thisvar, dot)
                           if wordpos(translate(stem), ExposedList) > 0 then iterate x
                        end
                     /* ----------------------------------------------------------------------------- */
                     /*  if current label is not a procedure, eliminate inherited variables           */
                     /*  fix: maybe. Note that we only check one level up and down, so there          */
                     /*        could still be false positives here                                    */
                     if wordpos(translate(ProcName), ProcList) == 0 then do
                           !ParentList = 'LL.0'||!ProcName||'_Parent'
                           ParentList = value(!ParentList)
                           do while ParentList \= ''
                              parse var ParentList Parent ParentList
                              !Parent = Hexed(Parent)
                              PvarList = Used.!Parent
                              if wordpos(translate(thisvar), translate(PvarList)) > 0 then
                                 /*  thisvar is used in a parent                                      */
                                 iterate x
                           end
                        end
                     /*  if thisvar is either used or exposed by a child, eliminate it.               */
                     /*  if a Level0 label, check the children of all Level0 labels. That is,         */
                     /*  if ProcName is in LL.0Level0_Labels, use LL.0[Level0]_Child for child list   */
                     if wordpos(translate(ProcName), translate(LL.0Level0_Labels)) > 0 then
                        !ChildList = 'LL.0'||Hexed('[Level0]')||'_Child'
                     else
                        !ChildList = 'LL.0'!ProcName||'_Child'
                     ChildList = value(!ChildList)
                     do while ChildList \= ''
                        parse var ChildList Child ChildList
                        !Child = Hexed(Child)
                        CvarList = Used.!Child
                        if wordpos(translate(Child), ProcList) == 0 then do
                              if wordpos(translate(thisvar), translate(CvarList)) > 0 then
                                 /*  thisvar is used in a child routine                               */
                                 iterate x
                           end
                        else do
                              /*  if thisvar is exposed by a child procedure, eliminate it            */
                              ChildExposed = translate(Exposed.!Child)
                              if wordpos(translate(thisvar), ChildExposed) > 0 then
                                 iterate x

                              /*  excluding variables whose stem is exposed will reduce the           */
                              /*  number of false positives, but can also miss some variables         */
                              /*  that really are unused. Let's have yet another switch.              */
                              if Opt.0alt then do
                                    dot = pos('.', thisvar)
                                    if dot > 0 then do
                                          stem = translate(left(thisvar, dot))
                                          if wordpos(stem, ChildExposed) > 0 then
                                             iterate x
                                       end
                                 end
                           end
                     end
                     /* ----------------------------------------------------------------------------- */
                     if \labelshown ,
                      & \Opt.0Flat then do
                           call Out G.0Tab||ProcName':'
                           labelshown = 1
                        end
                     call Out G.0Tab||G.0Tab||thisvar
                  end x
               end   /*  if words(VarList)  */
         end   /*  if in Include  */
   end w
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_UnusedVars ==================================================================== */

   Task_UsageHeaders: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  used by Script_Usage                                                                           */
   /* ----------------------------------------------------------------------------------------------- */
   /*  extracts the usage headers from procedures.                                                    */
   /*  A usage header looks like this:                                                                */
   /*                                                                                                 */
   /*          Label:                                                                                 */
   /*          /@ ===================== @/                                                            */
   /*          Usage:                                                                                 */
   /*                <text>                                                                           */
   /*                                                                                                 */
   /*  This is intended for extracting headers from RexxTool.cmd itself, but it could be              */
   /*  used on any file that uses the same header style.                                              */
   /* =============================================================================================== */
   if Opt.0progress then call Progress SubID()
   Filter = ExpandLabels(Opt.0Where)
   Filter = translate(Filter)
   if Filter = '' then
      if Opt.0Where = '' then
         Filter = translate(U_LabelList())
      else do
            call Out 'nothing matches "'Opt.0Where'"'
            return
         end
   incomment = 0
   do l = 1 to SrcIn.0
      if pos('/'||'*', SrcIn.l) > 0 ,
       | pos('*'||'/', SrcIn.l) > 0 then
         incomment = Incomment(SrcIn.l, incomment)
      if incomment <= 0 then
         if HasLabel(SrcIn.l) then do
               parse var SrcIn.l label ':' .                    /* remember the name of the procedure */
               label = strip(label)
               if wordpos(translate(label), Filter) > 0 then do
                     m = l +1
                     u = m +1
                     Usage = strip((S_UnComment(SrcIn.u)))
                     if U_DividerType(SrcIn.m) == '=' ,
                      & abbrev(translate(Usage), 'USAGE:') then do
                           /*  subroutine header starts with "===" divider after the label            */
                           /*  and the next line is "Usage:"                                          */
                           call Out '  '||S_UnComment(SrcIn.m)
                           call Out '  '||' ['label']'
                           m = m +1
                           do while m <= SrcIn.0
                              /*  copy everything until the next divider or non-comment               */
                              if \U_IsComment(SrcIn.m) then do
                                    call Msg 'Error: defective header for procedure' label
                                    leave
                                 end
                              call Out '  '||S_UnComment(SrcIn.m)
                              if U_DividerType(SrcIn.m) == '=' then leave
                              m = m +1
                           end
                           Last = SrcOut.0
                           if strip(SrcOut.Last) \== '' then do
                                 call Out ''
                              end
                        end   /*  if Usage: header  */
                     /*  else                                                                         */
                     /*     if Opt.0verbose then call Msg 'no Usage: in header at' label              */
                  end   /*  if abbrev()  */
            end   /*  if is a label  */
   end l
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_UsageHeaders ================================================================== */

   Task_Wrap: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: wrap wrapping linewrap format                                                            */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -Wrap                   try to wrap long lines to fit page width                             */
   /*                                                                                                 */
   /*  This is incomplete and not very intelligent about how it wraps lines. It is                    */
   /*  more of a framework that you can add new line-wrapping strategies onto. For                    */
   /*  instance, you could write a special routine for wrapping long assignment                       */
   /*  statements, like the ones in the Init_DefineBIFs section.                                      */
   /*                                                                                                 */
   /*  -Wrap can be very slow on large files, because it must insert tokens into the arrays           */
   /* =============================================================================================== */
   /*  for each long line, try different wrapping strategies, in order of preference                  */
   /*  if a procedure returns success, go to next line                                                */
   /*  if no procedure succeeds in wrapping, leave the long line as-is.                               */
   /*  This way I can improve the task gradually by writing new procedures                            */
   /*  So far, I have                                                                                 */
   /*     LineWrapAtEOC     breaks line at ";" or at a hidden EOC (THEN, etc)                         */
   /*     LineWrapAtOp      breaks line at logical operators, like option -Foldline                   */
   /*     LineWrap          try to guess the best place to break the line                             */
   /*                                                                                                 */
   /*  to add:                                                                                        */
   /*        break and merge long strings at an assignment?                                           */
   /* =============================================================================================== */
   if \Opt.0Wrap then return 0
   SubID = SubID()
   if Opt.0progress then call Progress SubID

   t = 1
   /*  use the loopcounter because I am not sure the wrapping routines are bugless                    */
   loop_max = t_type.0 * 2
   do while t < t_type.0 -1
      t = t +1
      call LoopCounter
      if t_class.t \= 'FILE_EOL' then iterate
      /*  token t is the EOL of its line; the line we want is +1                                      */
      /*  token t2 is the beginning of the next line                                                  */
      if \L_HasRexxWord(t, +1) then iterate
      if length(L_AsString(t, +1)) > Opt.0Width then do
            t2 = L_FirstToken(t, +1)
            if t_readonly.t2 then iterate
            /*  A line-wrapping routine returns 1 if it wrapped the line. That does not               */
            /*  always guarantee the line is short enough, but the routine did the best it could.     */
            /*  And I decided not to use the return value anyway.                                     */
            /*                                                                                        */
            /*  preferred strategies for wrapping get the first chance.                               */
            if Opt.0progress then call Progress SubID 'wrap line' t_line.t2 'of' SrcIn.0
            ret = LineWrapAtEOC(t2)
            if length(L_AsString(t, +1)) > Opt.0Width then do
                  ret = LineWrapAtOp(t2)
               end
            if length(L_AsString(t, +1)) > Opt.0Width then do
                  ret = LineWrap(t2)
               end
            if Opt.0progress then call Progress SubID
         end
   end   /*  while t  */
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Wrap ========================================================================== */

   Task_Unwrap: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Help: unwrap wrapping linewrap format                                                          */
   /*                                                                                                 */
   /*  used by Script_Format. The switches that control it are:                                       */
   /*                                                                                                 */
   /*    -Unwrap                   unwrap long lines                                                  */
   /*                                                                                                 */
   /*  This can be combined with -Wrap to change the line wrapping, as in:                            */
   /*             RexxTool.cmd FORMAT -Unwrap -Wrap                                                   */
   /*  -Unwrap can be very slow on large files, because it must delete tokens from the arrays         */
   /*                                                                                                 */
   /* =============================================================================================== */
   if \Opt.0Unwrap then return 0
   if Opt.0progress then call Progress SubID()

   do t = 2 to t_type.0 -1
      if t_type.t == 'CONTINUE' then do
            if t_readonly.t then iterate t
            /*  bugfix-- don't mess with continued line that has a comment                            */
            if L_HasComment(t) then iterate t
            call UnWrapLine t
         end   /*  if t == CONTINUE  */
   end t
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Unwrap ======================================================================== */

   Task_Variables: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  used by Script_Variables                                                                       */
   /*                                                                                                 */
   /*  shows the variable lists, See Script_Variables for the switches it uses                        */
   /*                                                                                                 */
   /*  fix: the MakeVarLists<X> routines were written at different times for slightly                 */
   /*  different purposes. It might be possible to combine them.                                      */
   /* =============================================================================================== */
   SubID = SubID()
   Opt.0Where = ExpandLabels(Opt.0Where)
   Include = Opt.0Where
   LabelList = LabelList()
   LabelList = WordSort(LabelList)
   LabelList = strip(LabelList)

   do i = 1 to words(Include)
      ProcN = word(Include, i)
      if wordpos(translate(ProcN), translate(LabelList)) == 0 then
         call Msg SubID 'Warning: cannot find a label called "'ProcN'"'
   end i
   Include = translate(Include)
   select
      when Opt.0Show == 'compact' then do
            call MakeVarListsC
            if Opt.0progress then call Progress SubID
            if debug.0Var then do
                  /*  note that DumpVars goes to STDOUT                                               */
                  if G.0Interpreter == 'REXXSAA' ,
                   & RxFuncQuery('RxVlist') == 0 then do
                        say 'Dump the variables returned from MakeVarListsC'
                        say 'Label list:'
                        call DumpVars 'LabelList'
                        say 'Exposed. lists:'
                        call DumpVars 'Exposed.'
                        say 'Vars. lists:'
                        call DumpVars 'Vars.'
                     end
                  else
                     call SysDumpVariables
               end
            if \Opt.0Flat then do
                  call Out copies('=', 80)
                  call Out 'Exposed variables by procedure'
                  call Out '~ indicates variable inferred by expanding a variable in ()'
                  do w = 1 to words(LabelList)
                     ProcName = word(LabelList, w)
                     if Include == '' ,
                      | wordpos(translate(ProcName), Include) > 0 then
                        if ProcName \= '[Unlabeled]' then do
                              !ProcName = Hexed(ProcName)
                              outstring = Exposed.!ProcName
                              call Out G.0Tab||ProcName':'
                              if outstring \= '' then
                                 call Out '      '||outstring
                           end
                  end w
                  call Out copies('=', 80)
                  call Out ''
               end   /*  if \Opt.0flat  */
            call Out copies('=', 80)
            if Opt.0Flat then do
                  call Out 'Global variables:'
                  call Out '   "=" indicates the variable was assigned'
                  call Out '   "+" indicates the variable was used'
                  ProcName = '[Unlabeled]'
                  !ProcName = Hexed(ProcName)
                  VarList = WordSort(Vars.!ProcName, , 3)
                  do x = 1 to words(VarList)
                     /*  the string from MakeVarListsC has no spaces, so it can be sorted with        */
                     /*  WordSort().  Make it easier to read by rewriting the prefix                  */
                     thisvar = word(VarList, x)
                     prefix1 = ChangeString('.', substr(thisvar, 1, 1), ' ')
                     prefix2 = ChangeString('.', substr(thisvar, 2, 1), ' ')
                     thisvar = substr(thisvar, 3)
                     call Out G.0Tab||G.0Tab||prefix1 prefix2 thisvar
                  end x
                  call Out copies('=', 80)
               end   /*  if flat  */
            else do
                  call Out 'Variables by label:'
                  call Out '   "=" indicates the variable was assigned'
                  call Out '   "+" indicates the variable was used'
                  do w = 1 to words(LabelList)
                     ProcName = word(LabelList, w)
                     if Include == '' ,
                      | wordpos(translate(ProcName), Include) > 0 then do
                           !ProcName = Hexed(ProcName)
                           call Out G.0Tab||'variables in '||ProcName':'
                           VarList = WordSort(Vars.!ProcName, , 3)
                           do x = 1 to words(VarList)
                              /*  the string from MakeVarListsC has no spaces, so it can be sorted    */
                              /*  with WordSort().  Make it easier to read by rewriting the prefix    */
                              thisvar = word(VarList, x)
                              prefix1 = ChangeString('.', substr(thisvar, 1, 1), ' ')
                              prefix2 = ChangeString('.', substr(thisvar, 2, 1), ' ')
                              thisvar = substr(thisvar, 3)
                              call Out G.0Tab||G.0Tab||prefix1 prefix2 thisvar
                           end x
                        end
                  end w
                  call Out copies('=', 80)
               end
         end   /*  when compact  */
      /* -------------------------------------------------------------------------------------------- */
      when Opt.0Show == 'lines' ,
         | Opt.0Show == '' then do
            if Opt.0Show == 'lines' then
               call MakeVarListsL 'lines'
            else
               call MakeVarListsL
            if Opt.0progress then call Progress SubID

            if debug.0Var then do
                  /*  note that DumpVars goes to STDOUT                                               */
                  if G.0Interpreter == 'REXXSAA' ,
                   & RxFuncQuery('RxVlist') == 0 then do
                        say 'Dump the variables returned from MakeVarListsL'
                        say 'Label list:'
                        call DumpVars 'LabelList'
                        say 'Exposed. lists:'
                        call DumpVars 'Exposed.'
                        say 'Assigned. lists:'
                        call DumpVars 'Assigned.'
                        say 'Used. lists:'
                        call DumpVars 'Used.'
                     end
                  else
                     call SysDumpVariables
               end
            /*  alphabetize the variable lists for each label                                         */
            do w = 1 to words(LabelList)
               ProcName = word(LabelList, w)
               !ProcName = Hexed(ProcName)
               Used.!ProcName = WordSort(Used.!ProcName)
               Assigned.!ProcName = WordSort(Assigned.!ProcName)
            end w

            call Out ''
            call Out copies('=', 80)
            call Out 'variables exposed by each procedure'
            call Out '~ indicates variable inferred by expanding a variable in ()'
            do w = 1 to words(LabelList)
               ProcName = word(LabelList, w)
               if Include == '' ,
                | wordpos(translate(ProcName), Include) > 0 then
                  if ProcName \= '[Unlabeled]' then do
                        !ProcName = Hexed(ProcName)
                        call Out G.0Tab||ProcName':'
                        do x = 1 to words(Exposed.!ProcName)
                           thisvar = word(Exposed.!ProcName, x)
                           parse var thisvar varN (G.0sep) lineNo
                           if Opt.0Show == 'lines' then
                              call Out G.0Tab||G.0Tab||'line' lineNo  varN
                           else
                              call Out G.0Tab||G.0Tab||varN
                        end x
                     end
            end w
            call Out copies('=', 80)

            call Out ''
            call Out copies('=', 80)
            call Out 'Variables assigned, by label'
            do w = 1 to words(LabelList)
               ProcName = word(LabelList, w)
               if Include == '' ,
                | wordpos(translate(ProcName), Include) > 0 then do
                     !ProcName = Hexed(ProcName)
                     call Out G.0Tab||ProcName':'
                     do x = 1 to words(Assigned.!ProcName)
                        thisvar = word(Assigned.!ProcName, x)
                        parse var thisvar varN (G.0sep) lineNo
                        if Opt.0Show == 'lines' then
                           call Out G.0Tab||G.0Tab||'line' lineNo  varN
                        else
                           call Out G.0Tab||G.0Tab||varN
                     end x
                  end
            end w
            call Out copies('=', 80)

            call Out ''
            call Out copies('=', 80)
            call Out 'Variables used, by label'
            do w = 1 to words(LabelList)
               ProcName = word(LabelList, w)
               if Include == '' ,
                | wordpos(translate(ProcName), Include) > 0 then do
                     !ProcName = Hexed(ProcName)
                     call Out G.0Tab||ProcName':'
                     do x = 1 to words(Used.!ProcName)
                        thisvar = word(Used.!ProcName, x)
                        parse var thisvar varN (G.0sep) lineNo
                        if Opt.0Show == 'lines' then
                           call Out G.0Tab||G.0Tab||'line' lineNo  varN
                        else
                           call Out G.0Tab||G.0Tab||varN
                     end x
                  end
            end w
            call Out copies('=', 80)
            call Out ''
         end   /*  when 'lines' or ''  */
      otherwise
         call Msg G.0meID 'I do not understand "-show='Opt.0Show'"'
         call Egress 'Usage'
   end   /*  select  */
   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Variables ===================================================================== */

   AddHandlersToChildren: procedure expose (Shared_vars) (all_tokens) LL. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call AddHandlersToChildren                                                               */
   /*                                                                                                 */
   /*  This is used by Task_CheckExpose                                                               */
   /*                                                                                                 */
   /*  For checking EXPOSE statements, we need to treat an error-handler as a child of                */
   /*  any routines that it could be invoked from. This tries to identify the routine                 */
   /*  where a SIGNAL ON takes effect and adds the handler to the child list of each of               */
   /*  that routine's descendants.                                                                    */
   /*  This is experimental.                                                                          */
   /* =============================================================================================== */
   CurLabelN = translate('[Unlabeled]')
   do t = 1 to t_type.0
      select
         when t_type.t == 'REXX_LABEL' then
            CurLabelN = translate(t_val.t)
         when wordpos(translate(t_val.t), 'SIGNAL CALL') > 0 then
            if translate(T_Val(t +1)) == 'ON' then do
                  /*  a SIGNAL ON / CALL ON in this routine. extract name of handler routine          */
                  lw = CL_LastWord(t)
                  HandlerN = t_val.lw
                  !HandlerN = Hexed(HandlerN)
                  tmp = LL.0LabelList
                  do while tmp \= ''
                     parse var tmp LabelN tmp
                     !LabelN = Hexed(LabelN)
                     /*  if CurLabelN is one of the ancestors of LabelN, then the SIGNAL ON will      */
                     /*  (presumably) be in effect when LabelN executes. There could be odd cases     */
                     /*  when that is not true, but it's the best we can do. Since the handler could  */
                     /*  be called from LabelN, add the handler to the list of LabelN's children      */
                     /*  and LabeN to the handler's parents.                                          */
                     Alist = value('LL.0'!LabelN'_Ancestors')
                     if wordpos(CurLabelN, translate(Alist)) > 0 then do
                           call value 'LL.0'!LabelN'_Child', value('LL.0'!LabelN'_Child') HandlerN
                           call value 'LL.0'!HandlerN'_Parent', value('LL.0'!HandlerN'_Parent') LabelN
                        end
                  end
               end
         otherwise nop
      end   /*  select  */
   end t
   return
   /* === End of AddHandlersToChildren ============================================================== */

   AncestorVariables: procedure expose (Shared_vars) (all_tokens) LL. Used. Assigned. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        varlist = AncestorVariables(<Label>)                                                     */
   /*                                                                                                 */
   /*  returns a list of all variables used or assigned in any ancestor of <Label>.                   */
   /*  EXPOSEd variables are not included.                                                            */
   /*                                                                                                 */
   /*  this depends on the LL. Used. and Assigned. stem variables, which means                        */
   /*  that GetCalls and MakeVarListsE must have already been called.                                 */
   /*                                                                                                 */
   /*  intended to be called by the Check_Expose task                                                 */
   /* =============================================================================================== */
   parse upper arg LabelN
   !LabelN = Hexed(LabelN)
   VarList = ''
   Alist = value('LL.0'!LabelN'_Ancestors')
   do while Alist \= ''
      parse var Alist ProcN Alist
      !ProcN = Hexed(ProcN)
      VarList = VarList Assigned.!ProcN Used.!ProcN
   end
   return strip(VarList)
   /* === End of AncestorVariables ================================================================== */

   ArrayToFile: procedure expose (Shared_vars) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL ArrayToFile 'Stemvar.' , 'FileName'                                                  */
   /*                                                                                                 */
   /*  writes a stem array to a file                                                                  */
   /*  the stem =name= is passed, but the stem =variable= itself must be exposed                      */
   /*  if filename is not given, default is stdout                                                    */
   /* =============================================================================================== */
   parse arg stemN , FileN
   if FileN == '' then FileN = 'stdout'
   if stemN == '' then do
         call ShowError 'In ArrayToFile, invalid arguments'
         call Egress 'BUG'
      end
   !listN = value('stemN')
   if !listN == '' then do
         call ShowError 'In ArrayToFile, invalid arguments'
         call Egress 'BUG'
      end

   /*  cosmetic fix, keep the progress and output from stepping on each other                         */
   if FileN == 'stdout' then
      call Msg ''
   else
      if Opt.0progress then call Progress SubID()

   if right(!listN, 1) \= '.' then !listN = !listN||'.'
   ListHigh = value(!listN||0)
   do i = 1 to ListHigh
      thisline = value(!listN||i)
      call lineout FileN, thisline
   end
   /*  every file should end with a line-ending                                                       */
   if thisline \== '' then
      call lineout FileN, ''
   call lineout FileN
   if Opt.0progress then call Progress
   return 0
   /* === End of ArrayToFile ======================================================================== */

   FileToArray: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL FileToArray 'Stemvar.' , 'FileName'                                                  */
   /*                                                                                                 */
   /*  reads input file 'FileName' into an array named 'Stemvar.', returns 0 if successful            */
   /*                                                                                                 */
   /*  The name of the variable is passed, but the variable itself must be exposed                    */
   /*  The stem variable should already be initialized                                                */
   /*  This can be used to add lines to an existing array                                             */
   /* =============================================================================================== */
   if Opt.0progress then call Progress SubID()
   parse arg stemN , FileN
   !listN = value('stemN')
   if !listN = '' ,
    | FileN = '' ,
    | right(!listN, 1) \= '.' then do
         call ShowError 'In FileToArray, invalid argument(s)'
         call Egress 'BUG'
      end

   i = value(!listN||0)
   if i == '' then i = 0
   /*  interpreting a line is slightly faster than a loop using 'value' for each assignment           */
   FileArray_script = ,
     'do while chars(fileN) > 0 ; i = i +1 ; '!listN||'i = linein(fileN) ; end ; '!listN||'0 = i'
   interpret FileArray_script
   call stream FileN, 'c', 'CLOSE'
   if Opt.0progress then call Progress
   return 0
   /* === End of FileToArray ======================================================================== */

   BIFAliases: procedure expose (Shared_vars) (all_tokens) BifAlias. BifAliasList /*fold00*/
   /* =============================================================================================== */
   /*  produces a stem variable BifAlias that maps library BIFs to their aliases                      */
   /*  in this case:                                                                                  */
   /*           call RxFuncAdd 'cls', 'RexxUtil', 'SysCLS'                                            */
   /*  produces:                                                                                      */
   /*           BifAlias.cls == 'SysCLS'                                                              */
   /*  except the variable tail is uppercased and converted to hex, so it is really                   */
   /*           BifAlias.434C53 == 'SysCLS'                                                           */
   /* =============================================================================================== */
   BifAlias. = ''
   G.0BifAliases = ''
   do t = 2 to t_type.0 -1
      if t_type.t \= 'REXX_W_PROCEDURE' then iterate t
      if translate(t_val.t) \= 'RXFUNCADD' then iterate t
      n = t +1
      if t_type.n == 'REXX_W_(' then
         n = n +1
      alias = n
      bif = CL_LastWord(n)
      if t_type.bif == 'REXX_W_)' then
         bif = R_PrevWord(bif)
      if wordpos(t_type.bif, 'REXX_W_SYMBOL REXX_W_LITERAL') == 0 then do
            /*  writer did something fancy that we cannot understand, or a bug here                   */
            call ShowError 'error parsing RxFuncAdd statement at line' t_line.t
            call ShowError Qline(t_line.t), ''
            iterate t
         end
      /*  exclude variables. Some people do not quote arguments to RxFuncAdd,                         */
      /*  relying on the fact that a variable defaults to its own uppercased value                    */
      /*  (if there is no NoValue handler)                                                            */
      /*  We can't do anything with it because it might really be a variable. As in                   */
      /*           call RxFuncAdd FuncName, DLLName, FuncName                                         */
      if t_type.alias \== 'REXX_W_LITERAL' ,
       | t_type.bif \== 'REXX_W_LITERAL' then iterate t
      if translate(t_val.alias) \== translate(t_val.bif) then do
            alias = Unquoted(t_val.alias)
            !alias = Hexed(alias)
            BifAlias.!alias = Unquoted(t_val.bif)
            G.0BifAliases = G.0BifAliases alias
         end
   end t
   G.0BifAliases = strip(G.0BifAliases)
   return 0
   /* === End of BIFAliases ========================================================================= */

   CheckOut: procedure expose (Shared_vars) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call CheckOut <string>                                                                   */
   /*                                                                                                 */
   /*  if formatting, sends <string> to stderr                                                        */
   /*  if command is CHECKONLY, adds <string> to the SrcOut array                                     */
   /*                                                                                                 */
   /*  this is only used by the Task_CheckXXX routines                                                */
   /* =============================================================================================== */
   /*  this is a hack-- the Task_CheckXXX routines were originally written to                         */
   /*  send warnings to stderr when formatting a file, while the formatted text goes                  */
   /*  to stdout or a file. But when using the CheckOnly command, I want TaskCheckXXX                 */
   /*  to send output to the SrcOut array like the other informational commands do.                   */
   /* =============================================================================================== */
   parse arg args
   if translate(G.0Command) \== 'CHECKONLY' then
      call lineout 'stderr', args
   else do
         ln = SrcOut.0 +1
         SrcOut.ln = args
         SrcOut.0 = ln
      end
   return 0
   /* === End of CheckOut =========================================================================== */

   debugEOC: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     call debugEOC                                                                               */
   /*                                                                                                 */
   /*  This is a debugging tool to make sure the hidden EOC tokens are in the right place             */
   /*  Use it after a routine that moves tokens                                                       */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   AddBefore = 0 ; AddAfter = 0
   select
      when t_type.t == 'REXX_W_KEY' then do
            Tval = translate(t_val.t)
            select
               when Tval == 'THEN' then do
                     if T_Type(R_PrevToken(t)) \= 'REXX_EOC' then AddBefore = 1
                     if T_Type(R_NextToken(t)) \= 'REXX_EOC' then AddAfter = 1
                  end
               when Tval == 'ELSE' then
                  if T_Type(R_NextToken(t)) \= 'REXX_EOC' then AddAfter = 1
               when Tval == 'OTHERWISE' then
                  if T_Type(R_NextToken(t)) \= 'REXX_EOC' then AddAfter = 1
               otherwise
                  call ShowError 'bad value for token t 't' "'t_val.t'" (called from line 'Esigl')'
            end   /*  select  */
         end
      when t_type.t == 'REXX_COLON' then
         if T_Type(R_NextToken(t)) \= 'REXX_EOC' then AddAfter = 1
      otherwise
         call ShowError 'bad value for token t 't' "'t_val.t'" (called from line 'Esigl')'
   end

   if AddBefore then do
         call ShowError 'no EOC before token 't' "'t_val.t'" (called from line 'Esigl')'
         call ShowError 'possibly at line' t_line.t, ''
      end
   if AddAfter then do
         call ShowError 'no EOC after token 't' "'t_val.t'" (called from line 'Esigl')'
         call ShowError 'possibly at line' t_line.t, ''
      end
   return 0
   /* === End of debugEOC =========================================================================== */

   DeControl: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call DeControl t                                                                         */
   /*        where t is a token number and token t is either type REXX_W_LITERAL                      */
   /*        or COMMENT_VAL                                                                           */
   /*                                                                                                 */
   /*  replaces literal control characters with their hex values                                      */
   /*                                                                                                 */
   /*  where # represents a control character,                                                        */
   /*           say '#'                                                                               */
   /*  becomes                                                                                        */
   /*           say '23'x                                                                             */
   /*  or if commented out it becomes                                                                 */
   /*           /@ say '23'x  @/                                                                      */
   /*  but in a regular comment                                                                       */
   /*           /@ use character #  @/                                                                */
   /*  becomes                                                                                        */
   /*           /@ use character <23x>  @/                                                            */
   /*                                                                                                 */
   /*  This was written for Task_Decontrol. It seems to works but it is not thoroughly tested         */
   /* =============================================================================================== */
   /*  variables:                                                                                     */
   /*     q     the quote character being used, either ' or "                                         */
   /*     p     position in the string t_val.t                                                        */
   /*     c     current character, at position p                                                      */
   /*     inQ   is True if inside quotes                                                              */
   /*     str   the new string                                                                        */
   /*     prev  whether the previous character was stored as a character or hex string                */
   /* ----------------------------------------------------------------------------------------------- */
   /*  This is a new version that should work for literal tokens, commented-out code,                 */
   /*  or normal comments                                                                             */
   /*  If a REXX_W_LITERAL token, q is the first character of the string.                             */
   /*  If a COMMENT_VAL token, it could be a commented-out line of Rexx code-- try to guess           */
   /*  what q should be by searching for single or double quotes around the control character.        */
   /*  If none is found then q == '', we assume the string is simply a comment, and                   */
   /*  leave out the quotes and "||" symbols.                                                         */
   /* ----------------------------------------------------------------------------------------------- */
   parse arg t
   if t_type.t == 'REXX_W_LITERAL' then do
         q = left(t_val.t, 1)
         prev = ''
         str = ''
         do p = 2 to length(t_val.t) -1
            c = substr(t_val.t, p, 1)
            if pos(c, G.0ControlChars) == 0 then do
                  select
                     when prev == '' then
                        str = q||c
                     when prev = 'hex' then
                        str = str||'||'||q||c
                     otherwise
                        str = str||c
                  end
                  prev = 'char'
               end
            else do
                  /*  found a control char                                                            */
                  select
                     when prev == '' then
                        str = q||c2x(c)||q||'x'
                     when prev == 'char' then
                        str = str||q||'||'||q||c2x(c)||q||'x'
                     when prev == q then do
                           /*  bugfix for escaped quote character                                     */
                           if p > 2 then do
                                 if substr(t_val.t, p -2, 1) == q then
                                    str = str||q||'||'||q
                              end
                           str = str||c2x(c)||q||'x'
                        end
                     otherwise
                        /*  prev == hex                                                               */
                        str = str||'||'||q||c2x(c)||q||'x'
                  end
                  prev = 'hex'
               end
         end p
         if prev \== 'hex' then
            str = str||q
         /*  bugfix for concatenation by juxtaposition, like this:                                    */
         /*            say varA'#'varB                                                                */
         if T_Prefix(t +1) == '' ,
          & (T_Type(t +1) == 'REXX_W_SYMBOL' ,
          | T_Type(t +1) == 'REXX_W_LITERAL') then
            str = str||'||'
         t_val.t = str
      end   /*  if LITERAL  */
   if t_type.t == 'COMMENT_VAL' then do
         /*  t_val.t is a COMMENT_VAL string. This will be trickier.                                  */
         /*  It may be commented-out code, in which case the control character(s) is                  */
         /*  in quotes. Try to guess the correct quote character by finding the control               */
         /*  character, then find the next quote character. If q == '' then we assume                 */
         /*  this is just a regular comment. Otherwise assume commented-out code.                     */
         pc = verify(t_val.t, G.0ControlChars, 'M')
         pq = verify(t_val.t, '"''', 'M', pc)
         if pq == 0 then
            q = ''
         else
            q = substr(t_val.t, pq, 1)

         prev = ''
         str = ''
         inQ = 0
         p = 0
         do while p < length(t_val.t)
            p = p +1
            c = substr(t_val.t, p, 1)
            if pos(c, G.0ControlChars) == 0 then do
                  select
                     when c == q then do
                           str = str||c
                           prev = q
                           inQ = \inQ
                        end
                     when prev = 'hex' then do
                           if q == '' then
                              str = str||c
                           else do
                                 if inQ then
                                    str = str||'||'||q||c
                                 else
                                    /*  bugfix for concatenation without "||" in                      */
                                    /*  commented-out code. If comment is not code, it will           */
                                    /*  add an unneeded "||", but it is better to mangle a            */
                                    /*  comment than code.                                            */
                                    if substr(t_val.t, p, 2) == '||' then
                                       str = str||c
                                    else do
                                          if pos(c, ' '||'09'x) > 0 then
                                             str = str||c
                                          else
                                             str = str||'||'||c
                                       end
                              end
                           prev = 'char'
                        end
                     otherwise
                        str = str||c
                        prev = 'char'
                  end
               end
            else do
                  /*  c is a control character                                                        */
                  select
                     when prev == q then do
                           /*  bugfix for escaped quote character                                     */
                           if p > 2 then do
                                 cpp = substr(t_val.t, p -2, 1)
                                 if cpp == q then
                                    str = str||q||'||'||q
                              end
                           str = str||c2x(c)||q||'x'
                        end
                     when prev == 'char' then do
                           if q == '' then
                              /*  a regular comment, change control char to <FFx>                     */
                              str = str||'<'||c2x(c)||'x'||'>'
                           else
                              /*  commented-out code, change control char to 'FF'x                    */
                              str = str||q||'||'||q||c2x(c)||q||'x'
                        end
                     when prev == '' then
                        /*  beginning of the string                                                   */
                        str = q||c2x(c)||q||'x'
                     otherwise
                        /*  prev == hex                                                               */
                        if q == '' then
                           /*  a regular comment, change to <FFx>                                     */
                           str = str||'<'||c2x(c)||'x'||'>'
                        else
                           /*  commented code, change to 'FF'x                                        */
                           str = str||'||'||q||c2x(c)||q||'x'
                  end
                  prev = 'hex'
                  /*  bugfix for escaped quote character                                              */
                  if substr(t_val.t, p +1, 1) == q then
                     if substr(t_val.t, p +2, 1) == q then
                        str = str||q
                     else do
                           /*  discard the closing quote                                              */
                           p = p +1
                           inQ = 0
                        end
               end
         end   /*  do p  */
         t_val.t = str
      end   /*  a COMMENT_VAL token  */
   return 0
   /* === End of DeControl ========================================================================== */

   DoSmartIndent: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call DoSmartIndent t                                                                     */
   /*     where t is the first word of a line continuation                                            */
   /*                                                                                                 */
   /*  indents the first word of a continuation according to the value of Opt.0IndentContinued        */
   /* ----------------------------------------------------------------------------------------------- */
   /*  this was moved from Task_Indent2 to a separate procedure so it could be called                 */
   /*  by the Wrap procedure, or anywhere else it might be needed.                                    */
   /* =============================================================================================== */
   Esigl = sigl
   /*  if not indenting, we cannot smart indent, obviously                                            */
   if symbol('t_indent.0') \== 'VAR' then return 0
   parse arg word1
   if \L_IsContinued(word1, -1) ,
    | L_FirstToken(word1) \== word1 then do
         call ShowError 'BUG: in DoSmartIndent, not the first word of a continuation'
         call ShowError 'called from line' Esigl, ''
         call Fatality
         return 0
      end

   select
      when Opt.0IndentContinued == '' then do
            /*  keep the two parts of a continued line in the same position                           */
            /*  relative to each other by examining the original prefixes:                            */
            ref = L_FirstToken(word1, -1)
            diff = length(t_prefix.word1) - length(t_prefix.ref)
            t_indent.word1 = diff / Opt.0Tab + t_indent.ref
            if t_indent.word1 < 0 then
               t_indent.word1 = 0
         end
      when translate(Opt.0IndentContinued) == 'SMART' then do
            ref = L_FirstToken(word1, -1)
            ref2 = ref +1
            select
               when L_IsContinued(word1, -2) then
                  /*  second continued line, just copy indent from the previous line                  */
                  t_indent.word1 = t_indent.ref

               when R_AtAssignment(ref) then do
                     /*  "Foo = 1 ,"                                                                  */
                     ref2 = R_Tail(ref) +1
                     ref3 = ref2 +1
                     if L_LastRexxWord(ref) <= ref2 then do
                           /*  "Foo = ,"                                                              */
                           /*  in this case, if the next line is indented to align,                   */
                           /*  it will be as long as if it were all one line, which                   */
                           /*  is probably not what the writer wants. Just give it                    */
                           /*  a half tab.                                                            */
                           t_indent.word1 = t_indent.ref + 2 / Opt.0Tab
                        end
                     else do
                           str = strip(T_AsString(ref, ref2))||t_prefix.ref3
                           t_indent.word1 = length(str) / Opt.0Tab + t_indent.ref
                        end
                  end
               when t_type.ref == 'REXX_W_KEY' then do
                     if t_type.word1 == 'REXX_W_LOGIC' then do
                           if wordpos(translate(t_val.ref2), 'WHILE UNTIL') > 0 then
                              /*  "do while a == 1 ,"                                                 */
                              /*  "       & b == 2"                                                   */
                              t_indent.word1 = ((length(t_val.ref||t_prefix.ref2||t_val.ref2) ,
                                               - length(t_val.word1)) / Opt.0Tab) + t_indent.ref

                           else
                              /*  "when Foo == 1 ,"                                                   */
                              /*  "   & Fee == 2"                                                     */
                              t_indent.word1 = ((length(t_val.ref) - length(t_val.word1)) / Opt.0Tab) ,
                                               + t_indent.ref
                        end
                     else
                        t_indent.word1 = (length(t_val.ref||t_prefix.ref2) / Opt.0Tab) ,
                                         + t_indent.ref
                  end
               otherwise
                  str = t_val.ref||t_prefix.ref2
                  t_indent.word1 = (length(str) / Opt.0Tab) + t_indent.ref
            end   /*  select  */
         end
      otherwise
         t_indent.word1 = t_indent.word1 + Opt.0IndentContinued
   end   /*  select  */
   return 0
   /* === End of DoSmartIndent ====================================================================== */

   EnvVar: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        VarValue = EnvVar(<variable name>)                                                       */
   /*        OS2Path = EnvVar('PATH')                                                                 */
   /*                                                                                                 */
   /*     returns value of a variable in the environment,                                             */
   /*     or '' if not defined                                                                        */
   /* =============================================================================================== */
   parse arg Pname
   if G.0Interpreter == 'REGINA' then
      /*  'SYSTEM' must be uppercase                                                                  */
      return value(Pname, , 'SYSTEM')
   else
      return value(Pname, , 'OS2ENVIRONMENT')
   /* === End of EnvVar ============================================================================= */

   ExpandLabels: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     string = ExpandLabels('<label pattern> [<label pattern>...]')                               */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*     labels = ExpandLabels('Task_* *AsString*')                                                  */
   /*     labels = ExpandLabels('Task_*,*AsString*')                                                  */
   /*     Opt.0where = ExpandLabels(Opt.0where)                                                       */
   /*                                                                                                 */
   /*  Expands the wildcards in a list of label patterns.  Words without wildcards                    */
   /*  will be kept as they are.                                                                      */
   /*  The word "+handlers" will be expanded to all routines that are the target of                   */
   /*  a CALL ON or SIGNAL ON statement                                                               */
   /*                                                                                                 */
   /*  Returns a space-separated list of matching labels                                              */
   /*                                                                                                 */
   /*  note that if tokenizing has not been done, "+handlers" expands to nothing                      */
   /* =============================================================================================== */
   parse arg oldstring
   /*  accept comma-separated lists also:                                                             */
   oldstring = translate(oldstring, ' ', ',')

   x = wordpos('+HANDLERS', translate(oldstring))
   if x > 0 then
      oldstring = delword(oldstring, x, 1) HandlerList()

   /*  if no tokenizing has been done, we must extract labels directly from the SrcIn array           */
   if symbol('t_type.0') == 'VAR' then do
         /*  LabelList includes sub-labels                                                            */
         LabelList = LabelList()
      end
   else
      LabelList = U_LabelList()
   newstring = ''
   do j = 1 to words(oldstring)
      wd = word(oldstring, j)
      do i = 1 to words(LabelList)
         LabelName = word(LabelList, i)
         if StringMatch(LabelName, wd, 'I') then
            newstring = newstring LabelName
      end i
   end j
   newstring = strip(UniqueWord(newstring))
   return newstring
   /* === End of ExpandLabels ======================================================================= */

   ExpandVariable: procedure expose (Shared_vars) (all_tokens) LL. ExpVar2. Map. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        ret = ExpandVariable(t, [label list])                                                    */
   /*                                                                                                 */
   /*  where t is a REXX_W_SYMBOL token in parentheses as part of an EXPOSE statement.                */
   /*                                                                                                 */
   /*  This was written to guess what variables are represented by (Shared_vars)                      */
   /*  in a case like                                                                                 */
   /*        procedure expose (Shared_vars)                                                           */
   /*  It is no good for anything else.                                                               */
   /*                                                                                                 */
   /*  It would be easy to contrive an example that would confuse this, but it should work            */
   /*  for most normal cases, and it is better than nothing. It could be rewritten to make            */
   /*  it somewhat more accurate, but it would be slower.                                             */
   /* =============================================================================================== */
   /* ----------------------------------------------------------------------------------------------- */
   /*     the example:                                                                                */
   /*          LabelA: procedure expose (Shared_vars)                                                 */
   /* ----------------------------------------------------------------------------------------------- */
   Esigl = sigl
   parse arg v , list
   if symbol('LL.0Level0_Labels') \= 'VAR' then do
         call ShowError 'BUG: GetCalls must be called before ExpandVariable'
         call ShowError 'called from line' Esigl 'in' LastLabel(Esigl), ''
         call Egress 'bug'
      end
   !tmp = Hexed('[All]')
   if symbol('Map.'!tmp'.start') \= 'VAR' then do
         call ShowError 'BUG: ProcMap must be called before ExpandVariable'
         call ShowError 'called from line' Esigl 'in' LastLabel(Esigl), ''
         call Egress 'bug'
      end

   varN = translate(S_CompoundVar(v))
   !VarN = Hexed(varN)
   CurLabelN = T_Val(R_PrevLabel(v))
   !CurLabelN = Hexed(CurLabelN)
   ret = ''
   /*  Search for statements like "Shared_vars = 'Fee Fie Foe'"                                       */
   if list = '' then do
         /*  examine Level 0 labels, plus parents of current label                                    */
         if symbol('ExpVar2.!VarN') \== 'VAR' then do
               /*  resolve var from Level0 first, then save the results for future use                */
               AncestorList = LL.0Level0_Labels
               do while AncestorList \= ''
                  parse var AncestorList LabelN AncestorList
                  call ExpandVariable.OneLabel
               end   /*  while AncestorList  */
               ExpVar2.!VarN = ret
            end   /*  if ExpVar2 not exist  */
         else
            /*  this VarN has already been resolved (at Level 0)                                      */
            ret = ExpVar2.!VarN
         /*  then check the parent routines of the current label                                      */
         tmp = value('LL.0'!CurLabelN'_Parent')
         AncestorList = ''
         do while tmp \= ''
            parse var tmp LabelN tmp
            if wordpos(LabelN, LL.0Level0_Labels) == 0 then
               AncestorList = AncestorList LabelN
         end
         do while AncestorList \= ''
            parse var AncestorList LabelN AncestorList
            call ExpandVariable.OneLabel
         end   /*  while AncestorList  */
         return ret
      end   /*  if list == ''  */
   else do
         /*  examine labels passed as argument                                                        */
         AncestorList = list
         do while AncestorList \= ''
            parse var AncestorList LabelN AncestorList
            call ExpandVariable.OneLabel
         end   /*  while AncestorList  */
         return ret
      end

   /* ----------------------------------------------------------------------------------------------- */
   ExpandVariable.OneLabel:
   !LabelN = Hexed(LabelN)
   t = Map.!LabelN.start
   do while t < Map.!LabelN.stop
      t = t +1
      if t_type.t \= 'REXX_W_SYMBOL' then iterate
      if translate(S_CompoundVar(t)) \= varN then iterate
      if R_AtAssignment(t) then do
            /*  look at token(s) after the "="                                                        */
            s = t
            do while t_type.s \== 'REXX_W_ASSIGN'
               s = s +1
            end
            s = s +1
            e = R_EndEOC(t)
            do while s < e
               if wordpos(t_type.s, 'REXX_W_SYMBOL REXX_W_LITERAL') == 0 then
                  leave
               if t_type.s == 'REXX_W_SYMBOL' then
                  if translate(S_CompoundVar(s)) \= varN then do
                        /*  in "Shared_vars = Shared_vars Fum"                                        */
                        /*  Fum is itself a list; expand it                                           */
                        ret = strip(ret ExpandVariable(s, LabelN))
                        s = R_Tail(s)
                     end
               if t_type.s == 'REXX_W_LITERAL' then
                  /*  "Shared_vars = 'Foe Fum'"                                                       */
                  /*  a quoted list of variable names, just unquote it                                */
                  ret = ret Unquoted(t_val.s)
               s = s +1
            end   /*  while s  */
            t = R_Tail(t)
         end   /*  if at assignment  */
   end   /*  while t  */
   ret = strip(ret)
   return 0
   /* === End of ExpandVariable ===================================================================== */

   ExtractHelpHeader: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*           call ExtractHelpHeader <pattern>, <keyword>                                           */
   /*                                                                                                 */
   /*  Extracts the header from any label that matches <pattern> and                                  */
   /*  contains a Help keyword that matches <keyword>                                                 */
   /*                                                                                                 */
   /*  the SrcIn. array must already exist                                                            */
   /*  the SrcOut. array must already exist                                                           */
   /*                                                                                                 */
   /*  This is only for reading RexxTool.cmd itself. For the sake of speed, it does not               */
   /*  track the comment level, which works because RexxTool.cmd never uses multi-line                */
   /*  comments or commented-out routines                                                             */
   /* =============================================================================================== */
   parse upper arg Labels , KeyWord
   Labels = translate(ExpandLabels(Labels))
   l = 1
   do while l <= SrcIn.0
      /*  if it looks like a label followed by a "===" divider then                                   */
      if pos(':', SrcIn.l) > 0 then
         if HasLabel(SrcIn.l) then do
               m = l +1
               if U_DividerType(SrcIn.m) == '=' then do
                     parse var SrcIn.l label ':' .
                     label = strip(label)
                     if wordpos(translate(label), Labels) > 0 then do
                           m = m +1
                           HelpLine = translate(strip(S_UnComment(SrcIn.m)))
                           parse var HelpLine word1 topics
                           /*  quick fix to allow more than one HELP: line                            */
                           if word1 == 'HELP:' then do
                                 do n = m +1 to SrcIn.0
                                    HelpLine = translate(strip(S_UnComment(SrcIn.n)))
                                    parse var HelpLine word1 rest
                                    if word1 == 'HELP:' then
                                       topics = topics rest
                                    else do
                                          m = n -1
                                          leave n
                                       end
                                 end n
                                 /*  allows searching for a quoted string                             */
                                 if pos(' '||KeyWord||' ', ' '||topics||' ') > 0 ,
                                  | KeyWord == 'ALL' then do
                                       m = m +1
                                       call Out 'HELP for' label':'
                                       do while m < SrcIn.0
                                          /*  copy everything until the next divider                  */
                                          /*  In order to allow multiline comments in the             */
                                          /*  header, I had to remove "if \U_IsComment()",            */
                                          /*  which means the header =must= end with a divider        */
                                          /*  comment.                                                */
                                          if U_DividerType(SrcIn.m) \= '' then do
                                                call Out strip(S_UnComment(SrcIn.m), 't')
                                                leave
                                             end
                                          call Out strip(S_UnComment(SrcIn.m), 't')
                                          m = m +1
                                       end
                                    end   /*  if wordpos()  */
                              end   /*  if HELP:  */
                        end   /*  if a matching label  */
                     l = m
                  end   /*  if followed by a divider  */
            end   /*  if is a label  */
      l = l +1
   end   /*  do while l  */
   return 0
   /* === End of ExtractHelpHeader ================================================================== */

   Fatality: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call Fatality                                                                            */
   /*                                                                                                 */
   /*  Call this when a bug occurs. It either exits immediately or sets the G.0fatalerror flag,       */
   /*  depending on the value of debug.0fatal.                                                        */
   /*                                                                                                 */
   /*  this exists simply to save me typing these four lines again and again                          */
   /* =============================================================================================== */
   if debug.0fatal then
      call Egress 'BUG'
   else
      G.0fatalerror = 1
   return 0
   /* === End of Fatality =========================================================================== */

   GetAncestors: procedure expose (Shared_vars) (all_tokens) LL. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call GetAncestors                                                                        */
   /*                                                                                                 */
   /*  GetCalls must be called before this is called.                                                 */
   /*  This is an addition to GetCalls. For each label XXX, it adds this to the LL. stem              */
   /*                                                                                                 */
   /*     LL.0XXX_Ancestors      is a list of the ancestors of XXX. That is,                          */
   /*                            routines that call XXX directly or indirectly.                       */
   /*                                                                                                 */
   /*  This is currently only used by the Task_CheckExpose procedure. It was moved out                */
   /*  of the GetCalls routine because some source files can cause it to fail with a                  */
   /*  Rexx error 11 "Control Stack Full".  This way other tasks that do not need the                 */
   /*  ancestors will still work on those files.                                                      */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  for the first call:                                                                            */
   if arg() == 0 then do
         LabelList = LL.0LabelList
         do while LabelList \= ''
            parse var LabelList LabelN LabelList
            LL.0BeenThere = ''
            Ancestors = GetAncestors(LabelN)
            !LabelN = Hexed(LabelN)
            call value 'LL.0'!LabelN'_Ancestors', Ancestors
         end
         return 0
      end
   /* ----------------------------------------------------------------------------------------------- */
   /*  for recursive calls:                                                                           */
   parse arg LabelN
   !LabelN = Hexed(LabelN)
   Parents = value('LL.0'!LabelN'_Parent')
   if Parents = '' then return ''
   Alist = ''
   tmp = Parents

   GetAncestors.loop:
   /*  normally I would use "do while tmp \= ''", but I want to reduce the number of                  */
   /*  control structures to avoid Rexx error 11 from deeply-nested recursion.                        */
   parse var tmp Parent tmp
   !Parent = Hexed(Parent)
   if value('LL.0'!Parent'_Ancestors') \= '' then
      Alist = Alist value('LL.0'!Parent'_Ancestors')
   else
      if wordpos(Parent, LabelN LL.0BeenThere) == 0 then do
            LL.0BeenThere = LL.0BeenThere Parent
            Alist = Alist GetAncestors(Parent)
         end
   if tmp \= '' then
      signal GetAncestors.loop
   Alist = UniqueWord(Parents Alist)
   return Alist
   /* === End of GetAncestors ======================================================================= */

   GetCalls: procedure expose (Shared_vars) (all_tokens) LL. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     CALL GetCalls [-sublabel]                                                                   */
   /*                                                                                                 */
   /*  if -sublabel is used, sublabels will be treated as normal labels,                              */
   /*  and ProcMap should also be called with a -sublabel switch                                      */
   /*  (see HELP SubLabel)                                                                            */
   /*                                                                                                 */
   /*  finds all calls to/from all routines in the input file                                         */
   /*                                                                                                 */
   /*  returns these variables by exposing them:                                                      */
   /*     LL.0LabelList          list of all labels                                                   */
   /*     LL.0Handlers           list of all routines that are the object of a                        */
   /*                               SIGNAL ON or CALL ON instruction                                  */
   /*  for each routine XXX                                                                           */
   /*     LL.0XXX_Child          is a list of routines called from XXX                                */
   /*     LL.0XXX_Parent         is a list of routines that call XXX                                  */
   /*                                                                                                 */
   /*  The beginning of the file before any labels will be given the name "[Unlabeled]"               */
   /*  and included in LL.0LabelList.                                                                 */
   /*                                                                                                 */
   /*     LL.0Level0_Labels      is a list of Level 0 routines                                        */
   /*     LL.0[Level0]_Child     is a list of routines called by Level 0                              */
   /*                                                                                                 */
   /*  Rexx scripts often have more than one routine besides [Unlabeled] at the Main                  */
   /*  level. RexxTool.cmd is an example, where the label 'Main' calls several Init_*                 */
   /*  routines. GetCalls will create LL.0XXX_* variables for each of them, but it                    */
   /*  will also create a fake "label" called [Level0] that is a combination of all                   */
   /*  routines at the Main level. That is, routines called from [Unlabeled] that are                 */
   /*  not procedures. Those routines will be listed in the LL.0[Level0]_Labels                       */
   /*  variable. LL.0[Level0]_Child lists all calls from any of those routines.                       */
   /*  [Level0] is not included in LL.0LabelList, on the grounds that there is no                     */
   /*  such place in the file and that it duplicates the real labels.                                 */
   /*                                                                                                 */
   /*  But note that the XXX in the tail is actually a hex string using                               */
   /*  c2x(translate(XXX)), in case the label has characters that are illegal in                      */
   /*  a variable name.  So if you want to dump variables for debugging, the                          */
   /*  variables for a label named 'Foo' would be named LL.0464F4F_Parent and                         */
   /*  LL.0464F4F_Child                                                                               */
   /* =============================================================================================== */
   SubID = SubID()
   parse upper arg args
   if args == '-SUBLABEL' then
      includeSublabels = 1
   else
      includeSublabels = 0
   LL. = ''
   CurLabel = '[Unlabeled]'
   !CurLabel = Hexed(CurLabel)
   LL.0LabelList = CurLabel

   /* ----------------------------------------------------------------------------------------------- */
   do t = 2 to t_type.0 -1
      select
         when t_type.t == 'REXX_LABEL' then do
               if (L_HasLabel(t, -1)) ,
                | abbrev(translate(t_val.t), translate(CurLabel)||'.') then do
                     /*  consider a double label or a sub-label as a call from the first to           */
                     /*  the second label, just as if t_type.t were a REXX_W_PROCEDURE:               */
                     LabelName = t_val.t
                     !LabelName = Hexed(LabelName)
                     /*  add this label to the list of routines that called this label                */
                     !ParentList = 'LL.0'||!LabelName||'_Parent'
                     ParentList = value(!ParentList)
                     !ChildList = 'LL.0'!CurLabel||'_Child'
                     ChildList = value(!ChildList)
                     if wordpos(translate(CurLabel), translate(ParentList)) == 0 then
                        call value !ParentList, ParentList||' '||CurLabel
                     if wordpos(translate(LabelName), translate(ChildList)) == 0 then
                        call value !ChildList, ChildList||' '||LabelName
                  end
               /*  update CurLabel only if not a sub-label                                            */
               if \abbrev(translate(t_val.t), translate(CurLabel)||'.') ,
                | includeSublabels then do
                     CurLabel = t_val.t
                     !CurLabel = Hexed(CurLabel)
                  end
               LL.0LabelList = LL.0LabelList||' '||t_val.t
            end
         when t_type.t == 'REXX_W_KEY' then do
               /*  "signal on syntax name SyntaxHandler" is not a call to SyntaxHandler               */
               /*  but add it to the list of handlers                                                 */
               if wordpos(translate(t_val.t), 'SIGNAL CALL') > 0 then do
                     w2 = R_NextWord(t)
                     if translate(t_val.w2) == 'ON' then do
                           wlast = CL_LastWord(t)
                           /*  exclude local error handler, recognizable by a sub-label name          */
                           if \abbrev(translate(t_val.wlast), translate(CurLabel)||'.') then
                              if wordpos(translate(t_val.wlast), translate(LL.0Handlers)) == 0 then
                                 LL.0Handlers = LL.0Handlers t_val.wlast
                        end
                  end
            end
         when t_type.t == 'REXX_W_PROCEDURE' then
            /*  "signal on error name ErrorHandler" is not a call to ErrorHandler                     */
            if translate(T_Val(t -1)) \== 'NAME' then do
                  LabelName = Unquoted(t_val.t)
                  !LabelName = Hexed(LabelName)
                  /*  add this label to the list of routines that called this label                   */
                  !ParentList = 'LL.0'||!LabelName||'_Parent'
                  ParentList = value(!ParentList)
                  !ChildList = 'LL.0'!CurLabel||'_Child'
                  ChildList = value(!ChildList)
                  if wordpos(translate(CurLabel), translate(ParentList)) == 0 then
                     call value !ParentList, ParentList||' '||CurLabel
                  if wordpos(translate(LabelName), translate(ChildList)) == 0 then
                     call value !ChildList, ChildList||' '||LabelName
               end
         otherwise nop
      end
   end t

   /* ----------------------------------------------------------------------------------------------- */
   /*  bugfix-- some scripts start with a label. That is, the "unlabeled" section has a label         */
   /*           that is not explicitly called; execution falls into it from the top.                  */
   /*           Try to identify such a label by finding the first label. Then, only if it was         */
   /*           not called from anywhere else, record it as called from [Unlabeled].                  */
   PrevLabel = '[Unlabeled]'
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_LABEL' then do
            if translate(T_Val(CL_FirstWord(t, 1))) == 'PROCEDURE' then
               leave t
            !ParentList = 'LL.0'||Hexed(t_val.t)||'_Parent'
            ParentList = value(!ParentList)
            if ParentList \= '' then
               leave t
            !ChildList = 'LL.0'||Hexed(PrevLabel)||'_Child'
            ChildList = value(!ChildList)
            call value !ParentList, PrevLabel
            call value !ChildList, strip(ChildList||' '||t_val.t)
            ParentList = value(!ParentList)
            ChildList = value(!ChildList)
            PrevLabel = t_val.t
         end
   end t
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  add a fake label called "[Level0]" which will be a combination of all the top-level            */
   /*  labels. That is, [Unlabeled] and any routines it calls that are not procedures.                */
   /*  LL.0Level0_Labels will be a list of all labels included in the top level.                      */
   /*  LL.0[Level0]_Child will contain all calls from all top-level routines.                         */
   ProcList = translate(ProcedureList())
   LL.0Level0_Labels = '[Unlabeled]'
   !Level0_To = 'LL.0'||Hexed('[Level0]')||'_Child'
   tolist = value('LL.0'Hexed('[Unlabeled]')'_Child')
   do while tolist \= ''
      parse var tolist label tolist
      if wordpos(translate(label), translate(LL.0LabelList)) > 0 then
         if wordpos(translate(label), ProcList) == 0 then
            if wordpos(translate(label), translate(LL.0Level0_Labels)) == 0 then do
                  LL.0Level0_Labels = LL.0Level0_Labels label
                  SrcToList = value('LL.0'Hexed(label)||'_Child')
                  call value !Level0_To, value(!Level0_To) SrcToList
                  tolist = tolist SrcToList
               end
   end
   /* ----------------------------------------------------------------------------------------------- */

   /* ----------------------------------------------------------------------------------------------- */
   /*  verify that error-handlers exist and correct capitalization:                                   */
   incList = LL.0Handlers
   LL.0Handlers = ''
   do while incList \= ''
      parse var incList inc incList
      lpos = wordpos(translate(inc), translate(LL.0LabelList))
      if lpos == 0 then do
            if Opt.0verbose then
               call Msg SubID 'Warning: in' G.0InfileN', cannot find error handler named "'inc'"'
            LL.0Handlers = LL.0Handlers inc
         end
      else
         LL.0Handlers = LL.0Handlers word(LL.0LabelList, lpos)
   end
   LL.0Handlers = strip(LL.0Handlers)
   /* ----------------------------------------------------------------------------------------------- */
   return 0
   /* === End of GetCalls =========================================================================== */

   GetDeps: procedure expose (Shared_vars) (all_tokens) LL. Required. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     call GetDeps [label [...]]                                                                  */
   /*                                                                                                 */
   /*  Examples:                                                                                      */
   /*     call GetDeps                                                                                */
   /*     call GetDeps LabelA LabelB                                                                  */
   /*     call GetDeps LabelA +Handlers                                                               */
   /*  where the special string "+Handlers" will be replaced by the list of error handlers            */
   /*  (routines that are the object of a SIGNAL ON or CALL ON instruction)                           */
   /*                                                                                                 */
   /*  See also GetCalls                                                                              */
   /*                                                                                                 */
   /*  GetDeps makes lists of sub-routines that one or more parent routines depend on                 */
   /*                                                                                                 */
   /*  arg(1) contains the names of the parent routines                                               */
   /*                                                                                                 */
   /*  Like GetCalls, it returns (by exposing) the stem variable LL.:                                 */
   /*                                                                                                 */
   /*     LL.0LabelList          list of all labels in the file                                       */
   /*     LL.0Handlers           list of all routines that are the object of a                        */
   /*                               SIGNAL ON or CALL ON instruction                                  */
   /*     LL.0XXX_Parent         is a list of routines called from XXX                                */
   /*     LL.0XXX_Child          is a list of routines that call XXX                                  */
   /*                                                                                                 */
   /*  But note that the XXX in the tail is actually a hex string using                               */
   /*  c2x(translate(XXX)), in case the routine name has characters that are                          */
   /*  illegal in a variable name. That can happen if we are using OS/2 Rexx to                       */
   /*  analyze a Regina file, or in the case of a call to an external routine like                    */
   /*           CALL 'H:\Path\to\Foo.cmd'                                                             */
   /*  So if you want to examine variables for debugging, the variables for a routine                 */
   /*  named 'Foo' would be named LL.0464F4F_Parent and LL.0464F4F_Child                              */
   /*                                                                                                 */
   /*  GetDeps also returns (by exposing) the Required. stem:                                         */
   /*     Required.0Include      the names of the parent routines being queried,                      */
   /*                            after verification that they exist and corrected                     */
   /*                            for capitalization                                                   */
   /*     Required.0Int          is a list of all Internal routines that are required                 */
   /*                            by the parent routines                                               */
   /*     Required.0Ext          list of External routines required by parent(s)                      */
   /*     Required.0BIF          list of Built-In-Functions required by parents)                      */
   /*                                                                                                 */
   /* =============================================================================================== */
   parse arg Include
   /*  first get the call lists for the whole file                                                    */
   call GetCalls

   x = wordpos('+HANDLERS', translate(Include))
   if x > 0 then do
         Include = delword(Include, x, 1) LL.0Handlers
         IncludeHandlers = 1
      end
   else
      IncludeHandlers = 0
   Include = strip(Include)
   if Include == '' then
      /*  bugfix for "+handlers" when there are no handlers:                                          */
      if \IncludeHandlers then
         Include = LL.0LabelList
   /*  Include is the list of routines we are interested in. Verify that they exist,                  */
   /*  and correct the capitalization while we are at it.                                             */
   incList = Include
   Include = ''
   do while incList \= ''
      parse var incList inc incList
      lpos = wordpos(translate(inc), translate(LL.0LabelList))
      if lpos == 0 then
         call Msg SubID 'Error: cannot find subroutine named "'inc'"'
      else
         Include = Include word(LL.0LabelList, lpos)
   end
   Include = strip(Include)
   Required. = ''
   Required.0Include = Include
   if Include == '' then
      return 0

   /* ----------------------------------------------------------------------------------------------- */
   /*  add the first level calls to Called (but not the starting routines in Include)                 */
   Called = ''
   BeenThere = ''
   do while Include \= ''
      parse var Include LabelName Include
      !LabelName = Hexed(LabelName)
      !ChildList = 'LL.0'||!LabelName||'_Child'
      ChildList = strip(value(!ChildList))

      do while ChildList \= ''
         parse var ChildList OneCall ChildList
         /*  skip a sub-label; it is the same as a dependency on LabelName itself                     */
         if \abbrev(translate(OneCall), translate(LabelName)||'.') then
            Called = Called OneCall
      end
      BeenThere = BeenThere LabelName
   end
   /*  for each name Foo in Called, get the list that Foo calls, and add it to Called                 */
   w = 1
   do while w <= words(Called)
      LabelName = word(Called, w)
      if wordpos(LabelName, BeenThere) == 0 then do
            !LabelName = Hexed(LabelName)
            !ChildList = 'LL.0'||!LabelName||'_Child'
            ChildList = strip(value(!ChildList))
            do while ChildList \= ''
               parse var ChildList OneCall ChildList
               /*  skip a sub-label; it is the same as a dependency on LabelName itself               */
               if \abbrev(translate(OneCall), translate(LabelName)||'.') then
                  Called = Called OneCall
            end
            BeenThere = BeenThere LabelName
         end
      w = w +1
   end   /*  while w  */

   Called = UniqueWord(Called)
   /*  now we have a list of all required calls in Called                                             */
   /*  sort them into Required.Int, Required.BIF, or Required.Ext                                     */
   do while Called \= ''
      parse var Called Cname Called
      select
         when wordpos(translate(Cname), translate(LL.0LabelList)) > 0 then do
               Required.0Int = Required.0Int Cname
            end
         when wordpos(translate(Cname), translate(G.0AllBIFs)) > 0 then
            Required.0BIF = Required.0BIF Cname
         when wordpos(translate(Cname), translate(G.0BifAliases)) > 0 then
            Required.0BIF = Required.0BIF Cname
         otherwise
            /*  it must be an external routine                                                        */
            if wordpos(translate(Cname), translate(Required.0Ext)) == 0 then
               Required.0Ext = Required.0Ext Cname
      end   /*  select  */
   end   /*  while Called \= ''  */

   Required.0Int = strip(WordSort(Required.0Int))
   Required.0Ext = strip(WordSort(Required.0Ext))
   Required.0BIF = strip(WordSort(Required.0BIF))
   return 0
   /* === End of GetDeps ============================================================================ */

   HandlerList: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     string = HandlerList()                                                                      */
   /*                                                                                                 */
   /*  returns a list of all routines that are the target of SIGNAL ON or CALL ON.                    */
   /*  local error handlers recognizable by a sub-label name are =not= included.                      */
   /* =============================================================================================== */
   Handlers = ''
   CurLabel = '[Unlabeled]'
   do t = 2 to t_type.0 -1
      select
         when t_type.t == 'REXX_W_KEY' then do
               if wordpos(translate(t_val.t), 'SIGNAL CALL') > 0 then do
                     w2 = R_NextWord(t)
                     if translate(t_val.w2) == 'ON' then do
                           wlast = CL_LastWord(t)
                           /*  exclude local error handler, recognizable by a sub-label name          */
                           if \abbrev(translate(t_val.wlast), translate(CurLabel)||'.') then
                              if wordpos(translate(t_val.wlast), translate(Handlers)) == 0 then
                                 Handlers = Handlers t_val.wlast
                        end
                  end
            end
         when t_type.t == 'REXX_LABEL' then
            /*  a sub-label does not count                                                            */
            if \abbrev(translate(t_val.t), translate(CurLabel)||'.') then
               CurLabel = t_val.t
         otherwise nop
      end
   end t
   return strip(Handlers)
   /* === End of HandlerList ======================================================================== */

   Hexed: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        !ProcN = Hexed(ProcN)                                                                    */
   /*                                                                                                 */
   /*  uppercases a string and converts it to a hex string                                            */
   /*  This is intended for turning a label name into a safe variable name                            */
   /* =============================================================================================== */
   parse arg args
   return c2x(translate(args))
   /* === End of Hexed ============================================================================== */

   Incomment: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     incomment = Incomment(string, incomment)                                                    */
   /*                                                                                                 */
   /*  tracks whether we are inside a comment                                                         */
   /*  returns incomment level, which may be > 1 if inside a nested comment                           */
   /* =============================================================================================== */
   parse arg str , incomment
   Esigl = sigl
   p = 1
   do while p <= length(str)
      c = substr(str, p, 1)
      if incomment <= 0 then
         if c == '"' ,
          | c == "'" then
            /*  skip over anything in quotes                                                          */
            do while p < length(str)
               p = p +1
               if substr(str, p, 1) == c then leave
            end
      cc = substr(str, p, 2)
      if cc == '/'||'*' then do
            incomment = incomment +1
            p = p +1
         end
      else
         if cc == '*'||'/' then do
               incomment = incomment -1
               p = p +1
            end
      p = p +1
   end
   if incomment < 0 then do
         call Msg G.0meID 'Warning: negative incomment value "'incomment'"'
         call Msg 'the line is "'str'"'
         incomment = 0
      end
   return incomment
   /* === End of Incomment ========================================================================== */

   LabelList: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        LabelList = LabelList()                                                                  */
   /*        LabelList = LabelList('-nosub')                                                          */
   /*                                                                                                 */
   /*  returns a list of all labels in the file                                                       */
   /*  It also includes the pseudo-label '[Unlabeled]'                                                */
   /*                                                                                                 */
   /*  If the argument -nosub is used, it will omit sub-labels                                        */
   /* =============================================================================================== */
   Esigl = sigl
   parse upper arg option
   CurLabel = '[Unlabeled]'
   LabelList = CurLabel
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_LABEL' then
         if option == '-NOSUB' then do
               if \abbrev(translate(t_val.t), translate(CurLabel)||'.') then do
                     CurLabel = t_val.t
                     if wordpos(translate(t_val.t), translate(LabelList)) == 0 then
                        LabelList = LabelList t_val.t
                  end
            end
         else
            if wordpos(translate(t_val.t), translate(LabelList)) == 0 then
               LabelList = LabelList t_val.t
   end t
   return strip(LabelList)
   /* === End of LabelList ========================================================================== */

   LibList: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        LibList = LibList()                                                                      */
   /*                                                                                                 */
   /*  returns a space-separated list of all libraries loaded with the 'RxFuncAdd' or                 */
   /*  'SysLoadMacroSpace' functions                                                                  */
   /* =============================================================================================== */
   Liblist = ''
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_W_PROCEDURE' then
         select
            when translate(t_val.t) == 'RXFUNCADD' then do
                  select
                     when T_Val(t +1) == '(' then do
                           /*  Look for the comma instead of just counting tokens. This               */
                           /*  is a bugfix for people who write things like                           */
                           /*        if RxFuncAdd('Sys' || OS, 'RxUtils', 'Rx' || OS ) = 0 then       */
                           lib = t +2
                           do until t_type.lib == 'REXX_W_COMMA'
                              lib = lib +1
                           end
                           lib = lib +1
                        end
                     when translate(T_Val(t -1)) == 'CALL' then do
                           lib = t +1
                           do until t_type.lib == 'REXX_W_COMMA'
                              lib = lib +1
                           end
                           lib = lib +1
                        end
                     otherwise
                        call ShowError 'BUG: error parsing RxFuncAdd statement at line' t_line.t
                        call ShowError Qline(t_line.t), ''
                        call Fatality
                  end
                  libname = Unquoted(t_val.lib)
                  /*  the "library name" could be a variable, in which case we cannot know            */
                  /*  what it really is                                                               */
                  /*  sometimes people write something without quotes like                            */
                  /*           Call RxFuncAdd SysLoadFuncs, RexxUtil, SysLoadFuncs                    */
                  /*  which works (if there is no NoValue handler) because the default value          */
                  /*  of the variable RexxUtil is "REXXUTIL". But we have no way of knowing           */
                  /*  if that variable has been assigned. The only safe thing to do is call           */
                  /*  the library "???".                                                              */
                  if libname == t_val.lib then
                     libname = '???'
                  if wordpos(translate(libname), translate(Liblist)) == 0 then
                     Liblist = Liblist libname
               end
            when translate(t_val.t) == 'SYSLOADREXXMACROSPACE' then do
                  select
                     when T_Val(t +1) == '(' then
                        lib = t +2
                     when translate(T_Val(t -1)) == 'CALL' then
                        lib = t +1
                     otherwise
                        call ShowError 'BUG: error parsing SysLoadRexxMacroSpace statement at line' t_line.t
                        call ShowError Qline(t_line.t), ''
                        call Fatality
                  end
                  libname = Unquoted(t_val.lib)
                  if wordpos(translate(libname), translate(Liblist)) == 0 then
                     Liblist = Liblist libname
               end
            otherwise nop
         end   /*  select  */
   end t
   return strip(Liblist)
   /* === End of LibList ============================================================================ */

   LineWrap: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        CALL LineWrap t                                                                          */
   /*  or                                                                                             */
   /*        ret = LineWrap(t)                                                                        */
   /*  where t is any token in the line                                                               */
   /*                                                                                                 */
   /*  Wraps a line to fit the line width Opt.0Width                                                  */
   /*                                                                                                 */
   /*  This needs much more work                                                                      */
   /*                                                                                                 */
   /*  This breaks the line once. If the remainder of the line is still too long, it will             */
   /*  be wrapped on the next call to this procedure                                                  */
   /*  It does not try to right-justify comments, so Task_FormatComments should probably              */
   /*  be called after this                                                                           */
   /*                                                                                                 */
   /*  Indenting must have already been calculated by Task_Indent                                     */
   /* =============================================================================================== */
   parse arg t
   /*  if no Rexx, there is nothing we can do here                                                    */
   if \L_HasRexx(t) then return 0
   if symbol('t_indent.0') \== 'VAR' then return 0
   ret = 0
   First = L_FirstRexxToken(t)
   Last = L_LastRexxToken(t)
   /*  if all one word, there is nothing we can do                                                    */
   if First >= Last then return 0
   /* ----------------------------------------------------------------------------------------------- */
   /*  if the line is too long only because of a comment, do not wrap Rexx code, just shorten         */
   /*  the comment, move it left and quit. If it is still too long, user must deal with it.           */
   if T_Class(L_LastToken(t)) == 'COMMENT' then do
         LastComment = L_LastToken(t)
         EndOfRexx = T_TokenColumn(L_LastRexxToken(t) +1)
         if EndOfRexx <= Opt.0Width then
            /*  only the comment makes it too long. strip trailing spaces from comment.               */
            /*  (Otherwise all comments justified on right side will be too long)                     */
            do w = Last to L_LastToken(t) by 1
               if t_type.w == 'COMMENT_VAL' then
                  if C_CommentType(w) == 'RIGHT' then
                     if \t_readonly.w then do
                           call C_StripVal w, 't'
                           if length(L_AsString(t)) <= Opt.0Width then
                              return 1
                           else do
                                 /*  if that was not enough, call C_MoveRight() to shorten            */
                                 /*  the comment as much as possible                                  */
                                 call C_MoveRight w
                                 return 1
                              end
                        end
                     else
                        return 0
            end w
      end   /*  if a comment  */
   /* ----------------------------------------------------------------------------------------------- */
   str = copies(' ', format(Opt.0Tab * t_indent.First, , 0))||t_val.First
   /*  find the token that puts the line over the limit (MaxW), then work backwards to                */
   /*  find a good place to break the line (break). Use Width -3 to allow for the inserted comma      */
   MaxW = 0
   do w = First +1 to Last
      if t_class.w == 'COMMENT' then do
            MaxW = w
            leave w
         end
      str = str||t_prefix.w||t_val.w
      if length(str) > Opt.0Width -3 then do
            MaxW = w
            leave w
         end
   end w
   /*  temporary error-checking                                                                       */
   if MaxW == 0 then do
         call ShowError 'BUG: In Wrap, MaxW == 0, which should never happen'
         call ShowError Qline(t_line.w), ''
         call ShowContext w
         call Mark 'w str'
         call Fatality
      end
   /* ----------------------------------------------------------------------------------------------- */
   /*  find a place to break the line:                                                                */
   /*  provide a little bit of intelligence in choosing a break point                                 */
   /*  insert more tests as I think of them                                                           */
   break = ''
   w = MaxW
   do while w >= First +1
      /*  try to keep function arguments or other parentheses together                                */
      if t_type.w == 'REXX_W_)' then do
            w2 = R_MatchPar(w)
            if L_OnSameLine(First, w) then
               w = w2
         end   /*  if REXX_W_)  */

      if T_Type(w -1) == 'REXX_W_ASSIGN' then do
            /*  w is "1" in "Foo = 1" ; break after the "="                                           */
            break = w
            leave
         end
      w = w -1
   end   /*  DO while  */

   /*  if we did not find a good break point, we will have to do it the stupid way                    */
   if break == '' then
      do w = MaxW to First +1 by -1
         if t_prefix.w \== '' then do
               break = w
               leave w
            end
      end w
   /* ----------------------------------------------------------------------------------------------- */
   if break \= '' then do
         /*  break points to the token that will start the continued line                             */
         breakval1 = t_val.break
         call T_InsertContinuation break
         /*  break now points to the comma, adjust it                                                 */
         break = L_FirstToken(break, +1)
         breakval2 = t_val.break
         if breakval1 \== breakval2 then do
               call ShowError 'BUG: when wrapping'
               call Mark 'break breakval1 breakval2', 'Bug: these should match'
               call Fatality
            end
         call DoSmartIndent break
         ret = 1
      end
   return ret
   /* === End of LineWrap =========================================================================== */

   LineWrapAtEOC: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        CALL LineWrapAtEOC t                                                                     */
   /*        Folded = LineWrapAtEOC(t)                                                                */
   /*                                                                                                 */
   /*  where t is any token in the line, but usually the keyword that starts the line                 */
   /*                                                                                                 */
   /*  Wraps a long line by breaking it at ';' End-of-Clause tokens, or hidden EOCs                   */
   /*  Returns 1 if the line was folded, 0 otherwise                                                  */
   /*  Returns 0 if the line is already a continued line                                              */
   /* =============================================================================================== */
   parse arg t
   ret = 0

   w = L_LastRexxWord(t)
   do while w >= t
      w = w -1
      if t_type.w == 'REXX_EOC' then
         if t_class.w \= 'FILE_EOL' then do
               select
                  when t_val.w == ';' then do
                        /*  change the ';' EOC to an End-Of-Line EOC                                  */
                        t_class.w = 'FILE_EOL'
                        t_prefix.w = ''
                        t_val.w = ''
                        ret = 1
                     end
                  when t_val.w == '' then
                     /*  a hidden EOC                                                                 */
                     select
                        when translate(T_Val(R_PrevWord(w))) == 'THEN' then
                           if Opt.0Then == '' ,
                            | translate(Opt.0Then) == 'UP' then do
                                 /*  change the hidden EOC =after= THEN to an EOL                     */
                                 t_class.w = 'FILE_EOL'
                                 t_prefix.w = ''
                                 t_val.w = ''
                                 ret = 1
                              end
                        when translate(T_Val(R_NextWord(w))) == 'THEN' then
                           if Opt.0Then == '' ,
                            | translate(Opt.0Then) == 'DOWN' then do
                                 /*  change the hidden EOC =before= THEN to an EOL                    */
                                 t_class.w = 'FILE_EOL'
                                 t_prefix.w = ''
                                 t_val.w = ''
                                 ret = 1
                              end
                        when translate(T_Val(R_NextWord(w))) == 'PROCEDURE' then
                           if translate(Opt.0Procedure) \== 'UP' then do
                                 /*  change the hidden EOC before PROCEDURE to an EOL                 */
                                 t_class.w = 'FILE_EOL'
                                 t_prefix.w = ''
                                 t_val.w = ''
                                 ret = 1
                              end
                        when wordpos(translate(T_Val(R_PrevWord(w))), 'OTHERWISE ELSE') > 0 then do
                              /*  after OTHERWISE or ELSE                                             */
                              /*  change the hidden EOC to an EOL                                     */
                              t_class.w = 'FILE_EOL'
                              t_prefix.w = ''
                              t_val.w = ''
                              ret = 1
                           end
                        when T_Type(w -1) == 'REXX_COLON' then do
                              t_class.w = 'FILE_EOL'
                              t_prefix.w = ''
                              t_val.w = ''
                              ret = 1
                           end
                        otherwise
                           call ShowError 'BUG: EOC token; expected THEN, ELSE, OTHERWISE, PROCEDURE, or :'
                           call ShowError Qline(t_line.w), ''
                           wp = T_Val(w -1)
                           wn = T_Val(w +1)
                           call Mark 'w t_class.w t_type.w t_val.w wp wn'
                           call Fatality
                     end   /*  when val == ''  */
                  otherwise nop
               end   /*  select  */
            end   /*  if hidden EOC  */
   end   /*  while w  */
   return ret
   /* === End of LineWrapAtEOC ====================================================================== */

   LineWrapAtOp: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        CALL LineWrapAtOp t                                                                      */
   /*        Folded = LineWrapAtOp(t)                                                                 */
   /*                                                                                                 */
   /*  where t is any token of the line                                                               */
   /*                                                                                                 */
   /*  Wraps a long line by breaking it at logical operators                                          */
   /*  Returns 1 if the line was folded, 0 otherwise                                                  */
   /* =============================================================================================== */
   /*  this was part of Task_FoldLine; moved to a procedure so it can be called from a                */
   /*  more general line-wrapping task someday                                                        */
   /* =============================================================================================== */
   parse arg t
   ret = 0
   if \L_HasRexxWord(t) then return ret
   w = L_LastRexxWord(t)
   f = L_FirstRexxWord(t)
   do while w > f
      if t_type.w == 'REXX_W_LOGIC' then do
            /*  word w is a logical operator:                                                         */
            /*  insert two new tokens: a line-continuation comma and an End-Of-Line                   */
            if Opt.0progress then call Progress G.0meID 'fold line' t_line.w 'of' SrcIn.0
            call T_InsertToken w, 2
            /*  the operator at w is now at w +2                                                      */
            t_class.w = 'REXX'
            t_type.w = 'CONTINUE'
            t_val.w = ','
            t_prefix.w = ' '
            call T_Class w +1, 'FILE_EOL'
            call T_Type w +1, 'NULL'
            call T_Val w +1, ''
            call T_Prefix w +1, ''
            ret = 1
            /*  if indenting has already been calculated, re-calculate indent of the new line         */
            call DoSmartIndent w +2
         end   /*  if a logical operator  */
      w = w -1
   end   /*  while w  */
   return ret
   /* === End of LineWrapAtOp ======================================================================= */

   MakeVarListsL: procedure expose (Shared_vars) (all_tokens) Assigned. Used. Exposed. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL MakeVarListsL ['lines']                                                              */
   /*                                                                                                 */
   /*  builds lists of variables assigned and used in each subroutine, with or                        */
   /*  without line numbers.                                                                          */
   /*                                                                                                 */
   /*  for each subroutine XXX, returns these lists of variables by EXPOSEing them:                   */
   /*        Exposed.XXX                                                                              */
   /*        Assigned.XXX                                                                             */
   /*        Used.XXX                                                                                 */
   /*  for example, in procedure Sub1:                                                                */
   /*        Exposed.!Sub1 == 'Fee Fie Foe Fum'                                                       */
   /*        Assigned.!Sub1 == 'Foo'                                                                  */
   /*        Used.!Sub1 == 'Foo'                                                                      */
   /*                                                                                                 */
   /*  where "!Sub1" represents the hex version of "Sub1". That is c2x(translate(Sub1))               */
   /*  so the variable names are really  Exposed.53554231, etc.                                       */
   /*                                                                                                 */
   /*  if the argument 'lines' is used, line numbers will be included, separated                      */
   /*  from the variable name by the separator character G.0sep                                       */
   /*  to split them you can use                                                                      */
   /*        parse var <word> variableName (G.0sep) lineNo                                            */
   /* =============================================================================================== */
   Esigl = sigl
   SubID = SubID()
   if Opt.0progress then call Progress SubID
   parse upper arg args
   showlines = 0
   if args == 'LINES' then showlines = 1
   Used. = ''                                                       /* variables used in each routine */
   Assigned. = ''                                               /* variables assigned in each routine */
   Exposed. = ''                                                 /* variables exposed by each routine */
   ProcN = '[Unlabeled]'                                                        /* current label name */
   !ProcN = Hexed(ProcN)             /* hex representation, because ProcN may not be a legal var name */
   ProcVars = ''                                      /* temporary list of variables in current label */
   call GetCalls
   call ProcMap

   t = 1
   do while t < t_type.0
      t = t +1
      select
         when t_type.t == 'REXX_LABEL' ,
            | t_class.t == 'FILE_EOF' then
            /*  set up for next label, unless this is a sublabel                                      */
            if \abbrev(translate(t_val.t), translate(ProcN)||'.') then do
                  Exposed.!ProcN = strip(Exposed.!ProcN)
                  Assigned.!ProcN = strip(Assigned.!ProcN)
                  Used.!ProcN = strip(Used.!ProcN)
                  ProcN = t_val.t
                  !ProcN = Hexed(ProcN)
                  ProcVars = ''
               end
         when t_type.t == 'REXX_W_SYMBOL' then do
               /*  for example, t_val.t == "Stem", varN == "Stem.Tail"                                */
               varN = S_CompoundVar(t)
               if wordpos(translate(varN), translate(ProcVars)) == 0 then
                  ProcVars = ProcVars varN
               select
                  /*  add the variable name and line number to the appropriate list                   */
                  /*  The variable name and line number are separated by a <G.0sep> character,        */
                  /*  and name/line pairs are separated by spaces. That way the list can be           */
                  /*  alphabetized later using the WordSort() function.                               */
                  when translate(T_Val(CL_FirstWord(t))) == 'PROCEDURE' then do
                        /*  token t is an exposed variable.                                           */
                        /*  Expand "(Shared_vars)" to the list of variables it stands for.            */
                        /*  ProcVars is a list of variables available in this procedure               */
                        if T_Type(t -1) == 'REXX_W_(' then do
                              Expanded = ExpandVariable(t)
                              ProcVars = ProcVars Expanded
                              varN = '('||t_val.t||')'
                           end
                        else
                           Expanded = ''
                        if showlines then do
                              Exposed.!ProcN = Exposed.!ProcN' 'varN||G.0sep||right(t_line.t, 5, '0')
                              do while Expanded \= ''
                                 parse var Expanded avarN Expanded
                                 Exposed.!ProcN = Exposed.!ProcN' ~'avarN||G.0sep||right(t_line.t, 5, '0')
                              end
                           end
                        else do
                              Exposed.!ProcN = Exposed.!ProcN varN
                              do while Expanded \= ''
                                 parse var Expanded avarN Expanded
                                 Exposed.!ProcN = Exposed.!ProcN '~'avarN
                              end
                           end
                     end
                  when R_AtAnyAssignment(t) then do
                        /*  an assignment, like "Stem.Tail = 1"                                       */
                        if showlines then do
                              Assigned.!ProcN = Assigned.!ProcN' 'varN||G.0sep||right(t_line.t, 5, '0')
                              /*  "DO aa = 1 to 3" also counts as usage of variable aa                */
                              if R_KeyVal(R_PrevWord(t)) == 'DO' then
                                 Used.!ProcN = Used.!ProcN' 'varN||G.0sep||right(t_line.t, 5, '0')
                           end
                        else do
                              if wordpos(translate(varN), translate(Assigned.!ProcN)) == 0 then
                                 Assigned.!ProcN = Assigned.!ProcN' 'varN
                              /*  "DO aa = 1 to 3" also counts as usage of variable aa                */
                              if R_KeyVal(R_PrevWord(t)) == 'DO' then do
                                    if wordpos(translate(varN), translate(Used.!ProcN)) == 0 then
                                       Used.!ProcN = Used.!ProcN varN
                                 end
                           end
                        /*  if a stem variable, check the elements for known variable names and       */
                        /*  mark them as Used                                                         */
                        if t_val.t \= varN then
                           call MakeVarListsL.StemUsed
                     end
                  otherwise
                     /*  a variable being used, as in "say Stem.Tail"                                 */
                     if showlines then
                        Used.!ProcN = Used.!ProcN' 'varN||G.0sep||right(t_line.t, 5, '0')
                     else
                        if wordpos(translate(varN), translate(Used.!ProcN)) == 0 then
                           Used.!ProcN = Used.!ProcN' 'varN
                     if t_val.t \= varN then
                        call MakeVarListsL.StemUsed
               end   /*  select  */
               /*  skip over the rest of a stem variable, so it is only listed once                   */
               t = R_Tail(t)
            end   /*  when  */

         when t_type.t == 'REXX_W_LITERAL' then do
               IsVar = QuotedVar(t)
               if IsVar \= '' then do
                     varN = Unquoted(t_val.t)
                     varN = strip(varN, 't', '.')
                     /*  Add it to list of variables in this routine                                  */
                     /*  and initialize it as not set and not used                                    */
                     if wordpos(translate(varN), translate(ProcVars)) == 0 then do
                           ProcVars = ProcVars varN
                        end
                     if IsVar == 'SET' then do
                           if showlines then
                              Assigned.!ProcN = Assigned.!ProcN' 'varN||G.0sep||right(t_line.t, 5, '0')
                           else
                              if wordpos(translate(varN), translate(Assigned.!ProcN)) == 0 then
                                 Assigned.!ProcN = Assigned.!ProcN' 'varN
                           /*  if a stem variable, check the elements for known variable names and    */
                           /*  mark them as Used                                                      */
                           if pos('.', varN) > 0 then
                              call MakeVarListsL.StemUsed
                        end
                     else do
                           if showlines then
                              Used.!ProcN = Used.!ProcN' 'varN||G.0sep||right(t_line.t, 5, '0')
                           else
                              if wordpos(translate(varN), translate(Used.!ProcN)) == 0 then
                                 Used.!ProcN = Used.!ProcN' 'varN
                           if pos('.', varN) > 0 then
                              call MakeVarListsL.StemUsed
                        end
                  end   /*  if IsVar  */
            end   /*  when  */
         otherwise nop
      end   /*  select  */
   end   /*  do while t  */
   if Opt.0progress then call Progress
   return 0

   /* ----------------------------------------------------------------------------------------------- */
   MakeVarListsL.StemUsed:
   /* ----------------------------------------------------------------------------------------------- */
   /*  This is part of MakeVarListsL.                                                                 */
   /*  It adds Stem and Tail elements to the Used list                                                */
   /*  See more comments in MakeVarListsC.StemUsed                                                    */
   /* ----------------------------------------------------------------------------------------------- */
   tmp = varN
   stemstr = ''
   do while tmp \= ''
      parse var tmp stem '.' tmp
      stemstr = stemstr||stem||'.'
      if wordpos('~'||translate(stem), translate(Exposed.!ProcN)) > 0 ,
       | wordpos(translate(stem), translate(Exposed.!ProcN)) > 0 then
         if wordpos(translate(stem), translate(ProcVars)) == 0 then do
               ProcVars = ProcVars stem
            end
      if wordpos(translate(stem), translate(ProcVars)) > 0 then
         /*  stem is a variable, add to Used list                                                     */
         if showlines then
            Used.!ProcN = Used.!ProcN stem||G.0sep||right(t_line.t, 5, '0')
         else
            if wordpos(translate(stem), translate(Used.!ProcN)) == 0 then
               Used.!ProcN = Used.!ProcN stem
            else
               nop
      else do
            /*  stem is not a variable                                                                */
            if tmp \= '' then do
                  stem = stemstr
                  if showlines then
                     Used.!ProcN = Used.!ProcN stem||G.0sep||right(t_line.t, 5, '0')
                  else
                     if wordpos(translate(stem), translate(Used.!ProcN)) == 0 then
                        Used.!ProcN = Used.!ProcN stem
               end
         end
   end
   return
   /* === End of MakeVarListsL ====================================================================== */

   MakeVarListsC: procedure expose (Shared_vars) (all_tokens) Vars. Exposed. LL. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL MakeVarListsC                                                                        */
   /*                                                                                                 */
   /*  builds lists of variables assigned and used in each subroutine                                 */
   /*                                                                                                 */
   /*  Exposes a stem variable called Exposed. listing exposed variables for each label.              */
   /*  Exposes a stem variable called Vars. listing variables found under each label.                 */
   /*                                                                                                 */
   /*  so Exposed.!Foo is a list of variables exposed by procedure Foo, and                           */
   /*  Vars.!Foo is a list of variables in Foo, where !Foo is Hexed('Foo').                           */
   /*  Variable names in a list are separated by a space. Each variable is                            */
   /*  preceded by a "=" if it was set in that routine and a "+" if it was used.                      */
   /*                                                                                                 */
   /*  If the option -flat is used, all variables are considered global and the only                  */
   /*  label name is "[Unlabeled]".                                                                   */
   /* =============================================================================================== */
   /*  Variables are named after labels and variables, but as hex strings because                     */
   /*  the text version may not be a legal variable name.                                             */
   /*  ProcN and varN refer to label name and variable name as plain text.                            */
   /*  !ProcN and !varN are their equivalents represented as hex strings.                             */
   /*                                                                                                 */
   /*  the Set. and Used. stems are flags that show if a variable is set or used in the               */
   /*  routine. So if variable Foo is used in routine LabelA then Used.!LabelA.!Foo is True.          */
   /* ----------------------------------------------------------------------------------------------- */
   SubID = SubID()
   if Opt.0progress then call Progress SubID
   call GetCalls
   call ProcMap
   Vars. = ''
   Exposed. = ''
   Set. = ''
   Used. = ''
   ProcN = '[Unlabeled]'
   !ProcN = Hexed(ProcN)
   !ProcN.Vars = ''
   Set.!ProcN. = 0
   Used.!ProcN. = 0
   t = 1
   do while t <= t_type.0
      t = t +1
      select
         when t_type.t == 'REXX_W_SYMBOL' then do
               if t_val.t == '.' then iterate
               varN = S_CompoundVar(t)
               /*  for example, t_type.t == "List", varN == "List.1"                                  */
               if translate(T_Val(CL_FirstWord(t))) == 'PROCEDURE' then do
                     /*  varN is part of an EXPOSE statement                                          */
                     if T_Val(t -1) == '(' then do
                           /*  add varN to the exposed list, in parentheses                           */
                           Exposed.!ProcN = Exposed.!ProcN' ('varN')'
                           /*  expand varN and add the result to exposed list                         */
                           Expanded = ExpandVariable(t)
                           do while Expanded \= ''
                              parse var Expanded avarN Expanded
                              Exposed.!ProcN = Exposed.!ProcN '~'avarN
                           end
                        end
                     else
                        /*  no parentheses, just add varN to exposed list                             */
                        Exposed.!ProcN = Exposed.!ProcN varN
                  end
               else do
                     /*  variable is not part of an EXPOSE statement.                                 */
                     /*  Add it to list of variables in this routine                                  */
                     /*  and initialize it as not set and not used                                    */
                     !VarN = Hexed(varN)
                     if wordpos(translate(varN), translate(!ProcN.Vars)) == 0 then do
                           !ProcN.Vars = !ProcN.Vars varN
                           Set.!ProcN.!VarN = 0
                           Used.!ProcN.!VarN = 0
                        end
                     if R_AtAnyAssignment(t) then do
                           /*  if var is assigned, mark it as set                                     */
                           Set.!ProcN.!VarN = 1
                           /*  "DO aa = 1 to 3" also counts as usage of variable aa                   */
                           if R_KeyVal(R_PrevWord(t)) == 'DO' then
                              Used.!ProcN.!VarN = 1
                           if t_val.t \= varN then
                              /*  "Stem.Tail = 1"                                                     */
                              call MakeVarListsC.StemUsed
                        end
                     else do
                           /*  not an assignment, mark var as used                                    */
                           Used.!ProcN.!VarN = 1
                           if t_val.t \= varN then
                              /*  "foo = Stem.Tail"                                                   */
                              call MakeVarListsC.StemUsed
                        end   /*  else  */

                  end   /*  else  */
               /*  skip over the rest of a stem variable, so it is only listed once                   */
               t = R_Tail(t)
            end   /*  when  */

         when t_type.t == 'REXX_W_LITERAL' then do
               IsVar = QuotedVar(t)
               /*  If a literal looks like a variable name in quotes, for instance as an argument     */
               /*  to SysFileTree(), IsVar will be 'SET' or 'USED'. Otherwise IsVar will be ''        */
               if IsVar \= '' then do
                     varN = Unquoted(t_val.t)
                     !VarN = Hexed(varN)
                     /*  Add it to list of variables in this routine                                  */
                     /*  and initialize it as not set and not used                                    */
                     if wordpos(translate(varN), translate(!ProcN.Vars)) == 0 then do
                           !ProcN.Vars = !ProcN.Vars varN
                           Set.!ProcN.!VarN = 0
                           Used.!ProcN.!VarN = 0
                        end
                     if IsVar == 'SET' then do
                           Set.!ProcN.!VarN = 1
                           if pos('.', varN) > 0 then
                              call MakeVarListsC.StemUsed
                        end
                     else do
                           Used.!ProcN.!VarN = 1
                           if pos('.', varN) > 0 then
                              call MakeVarListsC.StemUsed
                        end
                  end   /*  if IsVar  */
            end   /*  when  */

         when t_type.t == 'REXX_LABEL' then do
               /*  if a label and not a sublabel, make Vars.!ProcN list for the previous label        */
               if \abbrev(translate(t_val.t), translate(ProcN)||'.') then
                  if \Opt.0Flat then do
                        Vars.!ProcN = ''
                        do w = 1 to words(!ProcN.Vars)
                           varN = word(!ProcN.Vars, w)
                           !VarN = Hexed(varN)
                           if Set.!ProcN.!VarN == 1 then
                              Vars.!ProcN = Vars.!ProcN||' '||'='
                           else
                              Vars.!ProcN = Vars.!ProcN||' '||'.'

                           if Used.!ProcN.!VarN == 1 then
                              Vars.!ProcN = Vars.!ProcN||'+'
                           else
                              Vars.!ProcN = Vars.!ProcN||'.'
                           Vars.!ProcN = Vars.!ProcN||varN
                        end w
                        Vars.!ProcN = strip(Vars.!ProcN)
                        /*  setup for new label                                                       */
                        ProcN = t_val.t
                        !ProcN = Hexed(ProcN)
                        !ProcN.Vars = ''
                        Set.!ProcN. = 0
                        Used.!ProcN. = 0
                     end
            end
         when t_class.t == 'FILE_EOF'then do
               /*  make the vars.!ProcN list for the previous label, or the                           */
               /*  whole file if there were no procedures, or if option -flat was used                */
               Vars.!ProcN = ''
               do w = 1 to words(!ProcN.Vars)
                  varN = word(!ProcN.Vars, w)
                  !VarN = Hexed(varN)
                  if Set.!ProcN.!VarN == 1 then
                     Vars.!ProcN = Vars.!ProcN||' '||'='
                  else
                     Vars.!ProcN = Vars.!ProcN||' '||'.'

                  if Used.!ProcN.!VarN == 1 then
                     Vars.!ProcN = Vars.!ProcN||'+'
                  else
                     Vars.!ProcN = Vars.!ProcN||'.'
                  Vars.!ProcN = Vars.!ProcN||varN
               end w
               Vars.!ProcN = strip(Vars.!ProcN)
            end
         otherwise
            nop
      end   /*  select  */
   end   /*  while t  */
   if Opt.0progress then call Progress
   return 0

   /* ----------------------------------------------------------------------------------------------- */
   MakeVarListsC.StemUsed:
   /* ----------------------------------------------------------------------------------------------- */
   /*  This is part of MakeVarListsC                                                                  */
   /*  It adds Stem and Tail elements to the Used list, if and only if they are variables.            */
   /*  exposed variables will be added to !ProcN.vars only if they are actually used.                 */
   /*  That way, Task_Variables will not clutter the list with a lot of unused exposed variables      */
   /* ----------------------------------------------------------------------------------------------- */
   tmp = varN
   stemstr = ''
   do while tmp \= ''
      parse var tmp stem '.' tmp
      stemstr = stemstr||stem||'.'
      !stem = Hexed(stem)
      /*  bugfix-- the var may or may not have a ~ in front of it                                     */
      if wordpos('~'||translate(stem), translate(Exposed.!ProcN)) > 0 ,
       | wordpos(translate(stem), translate(Exposed.!ProcN)) > 0 then
         if wordpos(translate(stem), translate(!ProcN.Vars)) == 0 then do
               !ProcN.Vars = !ProcN.Vars stem
               Set.!ProcN.!stem = 0
               Used.!ProcN.!stem = 0
            end
      if wordpos(translate(stem), translate(!ProcN.Vars)) > 0 then do
            /*  stem is a variable                                                                    */
            Used.!ProcN.!stem = 1
         end
      else do
            /*  stem is not a variable                                                                */
            /*  in "StemA.1 = 1", mark "StemA." as used,                                              */
            /*  in "StemA.StemB.1 = 1", mark "StemA.StemB." as used,                                  */
            /*  in "StemA. = ''", do not mark "StemA." as used,                                       */
            if tmp \= '' then do
                  stem = stemstr
                  !stem = Hexed(stem)
                  Used.!ProcN.!stem = 1
               end
         end
   end
   return
   /* === End of MakeVarListsC ====================================================================== */

   MakeVarListsE: procedure expose (Shared_vars) (all_tokens) Assigned. Used. Exposed. LL. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL MakeVarListsE                                                                        */
   /*                                                                                                 */
   /*  builds lists of variables assigned and used in each routine                                    */
   /*                                                                                                 */
   /*  for each subroutine XXX, returns these lists of variables by exposing them:                    */
   /*        Exposed.XXX                                                                              */
   /*        Assigned.XXX                                                                             */
   /*        Used.XXX                                                                                 */
   /*  for example, in procedure Sub1:                                                                */
   /*        Exposed.Sub1 == 'Fee Fie Foe Fum'                                                        */
   /*        Assigned.Sub1 == 'Foo'                                                                   */
   /*        Used.Sub1 == 'Foo'                                                                       */
   /*                                                                                                 */
   /*  This is for Task_CheckExpose                                                                   */
   /*  Unlike MakeVarListsL                                                                           */
   /*     it does not show line numbers                                                               */
   /*     it expands the variable in "expose (Shared_vars)"                                           */
   /*     it will show variables for the fictional label called "[Level0]"                            */
   /*     it provides Assigned.[Level0] and Used.[Level0] variable lists                              */
   /*     a sublabel is treated as a regular label                                                    */
   /*  Now also used by Task_UnusedVars, added -flat switch                                           */
   /* =============================================================================================== */
   SubID = SubID()
   if symbol('LL.') \= 'VAR' then do
         call ShowError 'BUG: in' SubID 'the stem LL. has not been initialized'
         call ShowError 'The GetCalls procedure must be called first', ''
         call Egress 'BUG'
      end
   parse upper arg args
   if Opt.0progress then call Progress SubID
   call ProcMap '-sublabel'
   Used. = ''
   Assigned. = ''
   Exposed. = ''
   ProcN = '[Unlabeled]'
   !ProcN = Hexed(ProcN)
   ProcVars = ''                                    /* temporary list of variables in current routine */

   t = 1
   do while t < t_type.0
      t = t +1
      select
         when (t_type.t == 'REXX_LABEL' ,
            & args \= '-FLAT') ,
            | t_class.t == 'FILE_EOF' then do
               Exposed.!ProcN = strip(Exposed.!ProcN)
               Assigned.!ProcN = strip(Assigned.!ProcN)
               Used.!ProcN = strip(Used.!ProcN)
               ProcN = t_val.t
               !ProcN = Hexed(ProcN)
               ProcVars = ''
            end

         when t_type.t == 'REXX_W_SYMBOL' then do
               if t_val.t \= '.' then do
                     varN = S_CompoundVar(t)
                     if wordpos(translate(varN), translate(ProcVars)) == 0 then
                        ProcVars = ProcVars varN
                     select
                        when translate(T_Val(CL_FirstWord(t))) == 'PROCEDURE' then do
                              if T_Val(t -1) == '(' then do
                                    varX = ExpandVariable(t)
                                    if varX == '' then
                                       if Opt.0warn then do
                                             call Msg '[MakeVarListsE]: unable to expand exposed variable list ('S_CompoundVar(t)')'
                                             call Msg '  in procedure' ProcN 'at line' t_line.t
                                          end
                                    varN = varX t_val.t
                                 end

                              do while varN \= ''
                                 parse var varN OneVarN varN
                                 if wordpos(translate(OneVarN), translate(Exposed.!ProcN)) == 0 then
                                    Exposed.!ProcN = Exposed.!ProcN OneVarN
                              end
                           end
                        when R_AtAnyAssignment(t) then do
                              if wordpos(translate(varN), translate(Assigned.!ProcN)) == 0 then
                                 Assigned.!ProcN = Assigned.!ProcN' 'varN
                              /*  "Foo.1 = 1" also counts as an assignment of "Foo."                  */
                              if pos('.', varN) > 0 then
                                 if wordpos(translate(t_val.t)||'.', translate(Assigned.!ProcN)) == 0 then
                                    Assigned.!ProcN = Assigned.!ProcN' 't_val.t||'.'
                              /*  "DO aa = 1 to 3" also counts as usage of variable aa                */
                              if R_KeyVal(R_PrevWord(t)) == 'DO' then do
                                    if wordpos(translate(varN), translate(Used.!ProcN)) == 0 then
                                       Used.!ProcN = Used.!ProcN varN
                                 end

                              if t_val.t \= varN then do
                                    /*  "Stem.Tail = 1"                                               */
                                    /*  mark Stem and Tail as used variables, if they are variables   */
                                    tmp = varN
                                    do while tmp \= ''
                                       parse var tmp stem '.' tmp
                                       if wordpos(translate(stem), translate(ProcVars)) > 0 then
                                          if wordpos(translate(stem), translate(Used.!ProcN)) == 0 then
                                             Used.!ProcN = Used.!ProcN stem
                                    end
                                 end
                           end
                        otherwise
                           if wordpos(translate(varN), translate(Used.!ProcN)) == 0 then do
                                 Used.!ProcN = Used.!ProcN varN
                                 /*  "x = Foo.1" also counts as a usage of "Foo."                     */
                                 if pos('.', varN) > 0 then
                                    if wordpos(translate(t_val.t)||'.', translate(Used.!ProcN)) == 0 then
                                       Used.!ProcN = Used.!ProcN' 't_val.t||'.'
                              end
                           if t_val.t \= varN then do
                                 /*  "say Stem.Tail"                                                  */
                                 /*  mark Stem and Tail as used variables, if they are variables      */
                                 tmp = varN
                                 do while tmp \= ''
                                    parse var tmp stem '.' tmp
                                    if wordpos(translate(stem), translate(ProcVars)) > 0 then
                                       if wordpos(translate(stem), translate(Used.!ProcN)) == 0 then
                                          Used.!ProcN = Used.!ProcN stem
                                 end
                              end
                     end   /*  select  */
                     /*  skip over the rest of a stem variable, so it is only listed once             */
                     t = R_Tail(t)
                  end
            end
         when t_type.t == 'REXX_W_LITERAL' then do
               IsVar = QuotedVar(t)
               if IsVar \= '' then do
                     varN = Unquoted(t_val.t)
                     varN = strip(varN, 't', '.')
                     !VarN = Hexed(varN)
                     /*  Add it to list of variables in this routine                                  */
                     if wordpos(translate(varN), translate(ProcVars)) == 0 then do
                           ProcVars = ProcVars varN
                        end

                     if IsVar == 'SET' then do
                           if wordpos(translate(varN), translate(Assigned.!ProcN)) == 0 then
                              Assigned.!ProcN = Assigned.!ProcN' 'varN
                           /*  if a stem variable, check the elements for known variable names and    */
                           /*  mark them as Used                                                      */
                           if pos('.', varN) > 0 then
                              call MakeVarListsE.StemUsed
                        end
                     else do
                           if wordpos(translate(varN), translate(Used.!ProcN)) == 0 then
                              Used.!ProcN = Used.!ProcN' 'varN
                           if pos('.', varN) > 0 then
                              call MakeVarListsE.StemUsed
                        end
                  end   /*  if IsVar  \= ''  */
            end   /*  when  */
         otherwise nop
      end   /*  select  */
   end   /*  do while t  */

   /* ----------------------------------------------------------------------------------------------- */
   /*  make a combined variable list for all top-level routines:                                      */
   tmp = LL.0Level0_Labels
   !Level0 = Hexed('[Level0]')
   do while tmp \= ''
      parse var tmp label tmp
      !Label = Hexed(label)
      Assigned.!Level0 = Assigned.!Level0 Assigned.!Label
      Used.!Level0 = Used.!Level0 Used.!Label
   end
   Assigned.!Level0 = strip(Assigned.!Level0)
   Used.!Level0 = strip(Used.!Level0)
   /* ----------------------------------------------------------------------------------------------- */
   if Opt.0progress then call Progress
   return 0

   /* ----------------------------------------------------------------------------------------------- */
   MakeVarListsE.StemUsed:
   /* ----------------------------------------------------------------------------------------------- */
   /*  This is part of MakeVarListsE.                                                                 */
   /*  It adds Stem and Tail elements to the Used list                                                */
   /*  See more comments in MakeVarListsC.StemUsed                                                    */
   /* ----------------------------------------------------------------------------------------------- */
   tmp = varN
   stemstr = ''
   do while tmp \= ''
      parse var tmp stem '.' tmp
      stemstr = stemstr||stem||'.'
      if wordpos(translate(stem), translate(ProcVars)) == 0 then do
            ProcVars = ProcVars stem
         end
      if wordpos(translate(stem), translate(ProcVars)) > 0 then
         /*  stem is a variable, add to Used list                                                     */
         if wordpos(translate(stem), translate(Used.!ProcN)) == 0 then
            Used.!ProcN = Used.!ProcN stem
         else
            nop
      else do
            /*  stem is not a variable                                                                */
            if tmp \= '' then do
                  stem = stemstr
                  Used.!ProcN = Used.!ProcN stem
               end
         end
   end
   return
   /* === End of MakeVarListsE ====================================================================== */

   Msg: /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call Msg <text>                                                                          */
   /*                                                                                                 */
   /*   sends <text> to stderr.                                                                       */
   /*  this exists because I got tired of typing "call lineout 'stderr',"                             */
   /* =============================================================================================== */
   call lineout 'stderr', arg(1)
   return 0
   /* === End of Msg ================================================================================ */

   Out: procedure expose (Shared_vars) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call Out <string>                                                                        */
   /*                                                                                                 */
   /*  adds <string> to the SrcOut. array                                                             */
   /*  the array must already have been initialized                                                   */
   /* =============================================================================================== */
   parse arg args
   ln = SrcOut.0 +1
   SrcOut.ln = args
   SrcOut.0 = ln
   return 0
   /* === End of Out ================================================================================ */

   ProcedureList: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        ProcedureList = ProcedureList()                                                          */
   /*                                                                                                 */
   /*  returns a list of all labels that are procedures                                               */
   /*  it =does= include sub-labels                                                                   */
   /* =============================================================================================== */
   list = ''
   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_LABEL' then
         if translate(T_Val(R_NextWord(t))) == 'PROCEDURE' then
            if wordpos(translate(t_val.t), translate(list)) == 0 then
               list = list t_val.t
   end t
   return strip(list)
   /* === End of ProcedureList ====================================================================== */

   ProcMap: procedure expose (Shared_vars) (all_tokens) Map. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*    call ProcMap [-sublabel]                                                                     */
   /*                                                                                                 */
   /*    if -sublabel is used, sublabels will be treated as ordinary labels                           */
   /*    and GetCalls should also be called with the -sublabel switch                                 */
   /*                                                                                                 */
   /*  exposes a stem variable called Map that records the starting and ending tokens                 */
   /*  of each routine. For Label 'Foo',                                                              */
   /*     Map.!Foo.label   is the token where the label 'Foo' is                                      */
   /*     Map.!Foo.start   is the token where the 'Foo' block starts                                  */
   /*     Map.!Foo.stop    is the last token of the 'Foo' block                                       */
   /*  where !Foo is the Hexed() representation of 'Foo'                                              */
   /*  The pseudo-label '[Unlabeled]' starts at token 1                                               */
   /*  The pseudo-label '[All]' is the entire token array                                             */
   /*                                                                                                 */
   /*  The matching list of labels can be obtained with                                               */
   /*     LabeList = LabelList('-nosub') '[All]'                                                      */
   /*  or                                                                                             */
   /*     LabeList = LabelList() '[All]'                                                              */
   /*                                                                                                 */
   /*  The start and end positions are consistent with the B_xxx functions. That is,                  */
   /*  this tries to identify a block of code that may include a header comment above                 */
   /*  the label. So the starting token may not be the label that identifies the                      */
   /*  routine. See comments for the B_xxx functions.                                                 */
   /* =============================================================================================== */
   /*  fix: what should the .label tail be for pseudo-labels? It is not currently used anyway.        */
   parse upper arg args
   if args == '-SUBLABEL' then
      includeSublabels = 1
   else
      includeSublabels = 0
   LabelN = '[Unlabeled]'
   LabelT = 1
   !LabelN = Hexed(LabelN)
   Map.!LabelN.start = 1
   Map.!LabelN.label = 1

   do t = 2 to t_type.0 -1
      if t_type.t == 'REXX_LABEL' then
         if \abbrev(translate(t_val.t), translate(LabelN)||'.') ,
          | includeSublabels then do
               Map.!LabelN.stop = B_EndBlock(LabelT)
               LabelT = t
               LabelN = t_val.t
               !LabelN = Hexed(LabelN)
               Map.!LabelN.label = t
               Map.!LabelN.start = B_StartBlock(t)
            end
   end t
   Map.!LabelN.stop = t_type.0

   LabelN = '[All]'
   !LabelN = Hexed(LabelN)
   Map.!LabelN.start = 1
   Map.!LabelN.label = 1
   Map.!LabelN.stop = t_type.0
   return 0
   /* === End of ProcMap ============================================================================ */

   Qline: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call Msg Qline(<line>)                                                                   */
   /*  where <line> is a line number of the SrcIn. array                                              */
   /*                                                                                                 */
   /*  this provides a consistent way to quote a line from the input file, for use                    */
   /*  in error messages. It is easier than typing                                                    */
   /*        ln = t_line.t +1                                                                         */
   /*        call lineout 'stderr', '   ==>  'strip(ScrIn.ln)                                         */
   /*                                                                                                 */
   /*  Qline() shows the line =as it appears in the original file=                                    */
   /*  L_AsString() shows the current state of the tokens                                             */
   /*                                                                                                 */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg ln
   if ln < 1 ,
    | ln > SrcIn.0 then do
         call ShowError 'Invalid argument "'ln'" to Qline at line' Esigl
         call ShowError '  ==> 'sourceline(Esigl)
         call Fatality
         return ''
      end
   return '     ==>  'strip(SrcIn.ln)
   /* === End of Qline ============================================================================== */

   QuotedVar: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       IsVar = QuotedVar(t)                                                                      */
   /*                                                                                                 */
   /*  if token t looks like a quoted variable name, as in an argument to SysFileTree,                */
   /*  Value, SysStemSort, etc., returns 'SET' if the variable is being assigned                      */
   /*  and 'USED' if it is used.                                                                      */
   /*                                                                                                 */
   /*  returns '' otherwise                                                                           */
   /*                                                                                                 */
   /*  token t must be type W_LITERAL, that is, in quotes                                             */
   /*                                                                                                 */
   /*  this is called by the MakeVarListsX routines; not very useful for anything else                */
   /* =============================================================================================== */
   parse arg t
   qfuncs = 'SYSQUERYCLASSLIST SYSGETEA SYSFILESEARCH SYSFILETREE SYSSTEMSORT SYSSTEMCOPY' ,
            'SYSSTEMINSERT SYSSTEMDELETE VALUE SYMBOL'

   /*  in complicated statements like this, we need to ignore a literal string that                   */
   /*  is not an arg to value()                                                                       */
   /*      call value 'ANSI.__' || word( 'DELEOL ... WORDWRAPON',i ), ansi.__ESC || word( 'K ... 7h', i ) */
   FuncN = R_Infunction(t)
   if FuncN \= '' then
      if wordpos(translate(FuncN), qfuncs) == 0 then
         return ''

   k = CL_FirstWord(t)
   funcT = ''
   FuncN = ''
   /*  find the function name, either after CALL or before (                                          */
   if translate(t_val.k) == 'CALL' then do
         funcT = R_NextWord(k)
         FuncN = translate(t_val.funcT)
         firstarg = R_NextWord(funcT)
      end
   else
      do s = t to k by -1
         if t_type.s = 'REXX_W_(' then do
               funcT = R_PrevWord(s)
               FuncN = translate(t_val.funcT)
               firstarg = R_Word(funcT, +2)
               leave s
            end
      end s

   if FuncN == '' then
      return ''
   if wordpos(FuncN, qfuncs) == 0 then
      return ''

   /*  fix: quick, ugly, unsatisfactory bugfix for something like "value(varN||'.0')":                */
   /*  trying to weed out quoted strings that are not variable names                                  */
   /*  "datatype(unquoted(t_val.t), 'S')" won't work here                                             */
   unQvarN = Unquoted(t_val.t)
   if datatype(unQvarN) == 'NUM' then
      return ''
   /*  if it has non-symbol characters, reject it. Note that this includes Regina characters          */
   if verify(translate(unQvarN), '_ABCDEFGHIJKLMNOPQRSTUVWXYZ?!1234567890.$#@') > 0 then
      return ''
   if left(unQvarN, 1) == '.' then
      return ''

   /*  a == 1 means that token t is part of the first argument, not necessarily all of it             */
   /*  for instance, 'fee.' in symbol('fee.'fie'.foe')                                                */
   argnum = R_ArgNum(t)
   select
      when FuncN == 'SYSFILETREE' then
         /*  it is a variable name only if it is second argument to SySFileTree                       */
         if argnum = 2 then
            return 'SET'
      when wordpos(FuncN, 'SYSFILESEARCH SYSGETEA') > 0 then
         /*  variable only if third argument                                                          */
         if argnum = 3 then
            return 'SET'
      when FuncN = 'SYSSTEMCOPY' then do
            /*  variable only if first or second argument                                             */
            if argnum = 1 then
               return 'USED'
            if argnum = 2 then
               return 'SET'
         end
      when FuncN == 'SYMBOL' then
         if argnum = 1 then
            /*  in                                                                                    */
            /*        symbol('foo')                                                                   */
            /*  'foo' is a variable name                                                              */
            /*  in                                                                                    */
            /*        symbol('fee.'fie'.foe')                                                         */
            /*  'fee.' and '.foe' are literal strings, not variable names                             */
            if T_Type(t -1) == 'REXX_W_(' ,
             & T_Type(t +1) == 'REXX_W_)' then
               return 'USED'
      when FuncN == 'VALUE' then do
            if argnum = 1 then do
                  /*  in                                                                              */
                  /*        value('foo', 1)                                                           */
                  /*  variable 'foo' is set, not used.                                                */
                  /*  but in                                                                          */
                  /*        value('foo.'k, 1)                                                         */
                  /*  stem variable 'foo.' is used, not set                                           */
                  /*  and in                                                                          */
                  /*        value('foo', , 'os2environment')                                          */
                  /*  variable 'foo' is not a Rexx variable at all                                    */
                  hasArg2 = 0
                  commas = ''
                  do s = t to CL_LastWord(t)
                     if t_type.s == 'REXX_W_COMMA' then
                        commas = commas||' '||s
                     if t_type.s == 'REXX_W_)' then
                        leave s
                  end s
                  if words(commas) >= 2 then
                     /*  as in value('etc', , 'os2environment')                                       */
                     return ''
                  if words(commas) > 0 then do
                        c1 = word(commas, 1)
                        if T_Type(c1 +1) \= 'REXX_W_COMMA' then
                           hasArg2 = 1
                     end
                  iscompound = 0
                  if hasArg2 then do
                        if T_Type(t -1) \= 'REXX_W_(' ,
                         & T_Type(t -1) \= 'REXX_W_PROCEDURE' then
                           iscompound = 1
                        if T_Type(t +1) \= 'REXX_W_COMMA' ,
                         & T_Type(t +1) \= 'REXX_W_)' then
                           iscompound = 1
                     end
                  if hasArg2 ,
                   & \iscompound then
                     return 'SET'
                  else
                     return 'USED'
               end   /*  if argnum = 1  */
         end
      when wordpos(FuncN, 'SYSQUERYCLASSLIST SYSSTEMINSERT SYSSTEMDELETE SYSSTEMSORT') > 0 then
         if argnum = 1 then
            return 'SET'
      otherwise
         call ShowError 'unknown function name:' FuncN
         call ShowError 'in QuotedVar()', ''
         call Fatality
   end   /*  select  */
   return ''
   /* === End of QuotedVar ========================================================================== */

   RexxID: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        R = RexxID()                                                                             */
   /*                                                                                                 */
   /*  Returns the token number of the first comment (type COMMENT_BEG)                               */
   /*  that identifies a file as Rexx.                                                                */
   /*                                                                                                 */
   /*  It is usually the starting "/@", but may be a "#!" or "EXTPROC"                                */
   /*  If there is no starting comment, returns 1 (the FILE_BOF token)                                */
   /*                                                                                                 */
   /*  Since the Toker allows for leading empty lines, the RexxID comment                             */
   /*  may not be on the first line.                                                                  */
   /* =============================================================================================== */
   ret = 1
   do t = 2 to t_type.0 -1
      if t_type.t == 'COMMENT_BEG' then do
            ret = t
            leave t
         end
      if t_val.t \== '' then do
            /*  found something but there was no preceding comment, return the BOF token              */
            ret = 1
            leave t
         end
   end t
   return ret
   /* === End of RexxID ============================================================================= */

   SaveTokensT: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     call SaveTokensT <filename>                                                                 */
   /*                                                                                                 */
   /*  This is a debugging tool intended to be called at the end of the Tokeniser stage.              */
   /*  Shows each line of the original file, the tokens in each line, and their                       */
   /*  locations.                                                                                     */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg TokenFileN
   if TokenFileN == '' then do
         call ShowError 'BUG: in SaveTokensT, no file name given.  Called from line' Esigl
         call ShowError '      'sourceline(Esigl)
         call Egress 'BUG'
      end

   if Opt.0progress then call Progress SubID()
   if FileExist(TokenFileN) then do
         suff = 2
         do while FileExist(TokenFileN||'_'||suff)
            suff = suff +1
         end
         NewN = TokenFileN||'_'||suff
         call Msg G.0meID 'token file' TokenFileN 'changed to' NewN
         TokenFileN = NewN
      end

   /*   header lines and tab positions                                                                */
   tab_Source = 22
   tab_sigline = 7
   tab_Token     = tab_sigline +9
   tab_Line      = tab_Token   +10
   tab_column    = tab_Line    +10
   tab_class     = tab_column  +8
   tab_type      = tab_class   +12
   tab_val       = tab_type    +22
   Header_1 = left(' ', tab_Source)'000000000'
   Header_2 = left(' ', tab_Source)'123456789'
   do i = 1 to 11
      c = i // 10
      Header_1 = Header_1||copies(c, 10)
      Header_2 = Header_2||' 123456789'
   end
   Header_1 = overlay(' Column # -->', Header_1, 1)
   Header_3 = ' '
   Header_3 = overlay('Token #', Header_3, tab_Token)
   Header_3 = overlay('Line #', Header_3, tab_Line)
   Header_3 = overlay('Col #', Header_3, tab_column)
   Header_3 = overlay('Class', Header_3, tab_class)
   Header_3 = overlay('Type', Header_3, tab_type)
   Header_3 = overlay('{Prefix}{Value}', Header_3, tab_val)

   s_printed = 0
   do t = 1 to t_type.0
      Header_3 = overlay('Toker line:', Header_3, 1)
      Outline = copies(' ', 140)
      Outline = overlay(format(t, 7), Outline, tab_Token)
      Outline = overlay(format(t_line.t, 6), Outline, tab_Line)
      Outline = overlay(format(t_col.t, 5), Outline, tab_column)
      Outline = overlay(t_class.t, Outline, tab_class)
      Outline = overlay(t_type.t, Outline, tab_type)
      Outline = overlay("{"t_prefix.t"}{"t_val.t"}", Outline, tab_val)
      st_ln = t_line.t
      if st_ln > s_printed then do
            s_printed = st_ln
            srcline = 'source line # 'right(st_ln, 5, '0')
            srcline = overlay('{'SrcIn.st_ln'}', srcline, tab_Source)
            call lineout TokenFileN, ''
            call lineout TokenFileN, Header_1
            call lineout TokenFileN, Header_2
            call lineout TokenFileN, srcline
            call lineout TokenFileN, Header_3
         end
      Outline = strip(Outline, 't')
      call lineout TokenFileN, Outline
   end t
   call lineout TokenFileN
   if Opt.0progress then call Progress
   say G.0meID 'Token list was written to 'TokenFileN
   return 0
   /* === End of SaveTokensT ======================================================================== */

   SaveTokensF: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     call SaveTokensF                                                                            */
   /*                                                                                                 */
   /*  This is a debugging tool intended to be called at any time during the formatting stage.        */
   /*  Shows the current state of the token array at the time it was called.                          */
   /* =============================================================================================== */
   Esigl = sigl
   if Opt.0progress then call Progress SubID()
   parse arg TokenFileN
   if TokenFileN == '' then do
         call ShowError 'BUG: in SaveTokensF, no file name given.  Called from line' Esigl
         call ShowError '      'sourceline(Esigl)
         call Egress 'BUG'
      end

   if FileExist(TokenFileN) then do
         suff = 2
         do while FileExist(TokenFileN||'_'||suff)
            suff = suff +1
         end
         NewN = TokenFileN||'_'||suff
         call Msg G.0meID 'token file' TokenFileN 'changed to' NewN
         TokenFileN = NewN
      end

   /*  bugfix-- this might be called before the t_indent array has been created. If so,               */
   /*  initialize it to avoid errors below                                                            */
   if symbol('t_indent.0') == 'LIT' then
      t_indent. = 0

   /*   header lines and tab positions                                                                */
   /*  to add an attribute, you need to                                                               */
   /*     insert a "tab_xxx = " line here,                                                            */
   /*     insert a "Header_3 =" line in the next block                                                */
   /*     insert a "OutLine = overlay(" line in the next block                                        */

   tab_sigline = 0
   tab_Token     = tab_sigline +3
   tab_RO        = tab_Token   +10
   tab_Line      = tab_RO      +12
   tab_indent    = tab_Line    +12
   tab_class     = tab_indent  +12
   tab_type      = tab_class   +12
   tab_val       = tab_type    +22

   Header_1 = 'Note that a token line is the location of the token in the original file'
   Header_2 = 'Any new tokens will show a line of 0'
   Header_3 = ' '
   Header_3 = overlay('Token #', Header_3, tab_Token)
   Header_3 = overlay('ReadOnly:', Header_3, tab_RO)
   Header_3 = overlay('Line:', Header_3, tab_Line)
   Header_3 = overlay('Indent:', Header_3, tab_indent)
   Header_3 = overlay('Class:', Header_3, tab_class)
   Header_3 = overlay('Type:', Header_3, tab_type)
   Header_3 = overlay('{Prefix}{Value}:', Header_3, tab_val)


   call lineout TokenFileN, Header_1
   call lineout TokenFileN, Header_2
   call lineout TokenFileN, Header_3
   do t = 1 to t_type.0
      /*        Header_3 = overlay('Toker line:', Header_3, 1)                                        */
      Outline = copies(' ', 140)
      Outline = overlay(format(t, 7), Outline, tab_Token)
      Outline = overlay(format(t_readonly.t, 6), Outline, tab_RO)
      Outline = overlay(format(t_line.t, 6), Outline, tab_Line)
      Outline = overlay(format(t_indent.t, 4), Outline, tab_indent)
      Outline = overlay(t_class.t, Outline, tab_class)
      Outline = overlay(t_type.t, Outline, tab_type)
      Outline = overlay("{"t_prefix.t"}{"t_val.t"}", Outline, tab_val)

      if t // 25 == 0 then do
            call lineout TokenFileN, ''
            call lineout TokenFileN, Header_3
         end
      Outline = strip(Outline, 't')
      call lineout TokenFileN, Outline
   end t
   call lineout TokenFileN
   if Opt.0progress then call Progress
   say G.0meID 'Token list was written to 'TokenFileN
   return 0
   /* === End of SaveTokensF ======================================================================== */

   SearchPath: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        FQFN = SearchPath(<list>, <filename>)                                                    */
   /*  Example:                                                                                       */
   /*        filename = SearchPath(EnvVar('PATH'), filename)                                          */
   /*                                                                                                 */
   /*  Returns the full path+filename if found, or '' if not found                                    */
   /*                                                                                                 */
   /*  Similar to SysSearchPath except that the first argument is not an environment                  */
   /*  variable, it is a semicolon-separated list of directories. Or on Unix, a                       */
   /*  colon-separated list.                                                                          */
   /*  Unlike SysSearchPath, if the filename already has a path+filename, it                          */
   /*  ignores the %path and returns the full filename.                                               */
   /*  Unlike Regina's SysSearchPath, it works                                                        */
   /* =============================================================================================== */
   parse arg PathList , FileN
   foundCmd = ''
   foundCmd = stream(FileN, 'c', 'query exists')
   if foundCmd \= '' then
      return foundCmd
   do while PathList \= ''
      parse var PathList dirN (G.0PathSep) PathList
      foundCmd = stream(dirN||G.0DirSep||FileN, 'c', 'query exists')
      if foundCmd \= '' then leave
   end
   return foundCmd
   /* === End of SearchPath ========================================================================= */

   ShowContext: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*           call ShowContext t [, <count>] [, <message>]                                          */
   /*                                                                                                 */
   /*  a debugging tool for use in error messages                                                     */
   /*  Shows the <count> lines before and after the line of token t                                   */
   /*  if <message> is given, it is used as a header                                                  */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count , msg
   if msg == '' then msg = 'called from line' Esigl
   if count == '' then count = 2
   call Msg overlay(' '||msg||' ', '-', 5, 80, '-')
   do a = -count to count
      call Msg '['a']' L_AsString(t, a)
   end
   call Msg copies('-', 84)
   return 0
   /* === End of ShowContext ======================================================================== */

   StemDelete: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        CALL StemDelete <stem> , <where> , <how many>                                            */
   /*        CALL StemDelete 't_type.' , 100 , 3                                                      */
   /*                                                                                                 */
   /*  where <stem> is the name of the stem variable, <where> is which element to                     */
   /*  delete, and <how many> is how many elements to delete.                                         */
   /*                                                                                                 */
   /*  deletes 1 or more elements from an array                                                       */
   /*  the array name must be exposed by including it in (all_tokens)                                 */
   /*  this is faster than SysStemDelete(), especially if <how many> is greater than 1                */
   /* ----------------------------------------------------------------------------------------------- */
   /*  this is intended to be called by the T_DeleteToken function, but I suppose it might            */
   /*  have other uses.                                                                               */
   /* =============================================================================================== */
   Esigl = sigl
   ArrayN = arg(1)
   where = arg(2)
   HowMany = arg(3)
   if HowMany == '' then HowMany = 1
   if debug.0arg then do
         if ArrayN == '' ,
          | \datatype(where, 'w') ,
          | \datatype(HowMany, 'w') then do
               call ShowError 'in StemDelete, invalid argument(s) "'ArrayN where HowMany'"'
               call Fatality
            end
         if value(ArrayN) == translate(ArrayN) then do
               call ShowError 'StemDelete: unknown array "'ArrayN'"'
               call Fatality
            end
      end
   !Array = value('ArrayN')
   old0 = value(!Array||0)
   new0 = max(0, old0 - HowMany)

   /*  shift higher-numbered elements down by <HowMany> positions                                     */
   call value !Array||0, new0
   do n = where to new0 by 1
      o = n + HowMany
      call value !Array||n, value(!Array||o)
   end

   /*  erase vacated positions by overwriting with initial value of this array                        */
   /*  this is not really needed, but it might help detect errors elsewhere                           */
   do i = new0 +1 to old0 by 1
      call value !Array||i, value(!Array)
   end
   return 0
   /* === End of StemDelete ========================================================================= */

   StemInsert: procedure expose  (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        CALL StemInsert <stem> , <where> , <how many>                                            */
   /*        CALL StemInsert 't_type.' , 100 , 3                                                      */
   /*                                                                                                 */
   /*  where <stem> is the name of the stem variable, <where> is which element to                     */
   /*  delete, and <how many> is how many elements to insert.                                         */
   /*                                                                                                 */
   /*  inserts 1 or more new elements into an array                                                   */
   /*  the array name must be included in the (all_tokens) list                                       */
   /*  this is faster than SysStemInsert(), especially if <how many> is greater than 1                */
   /* ----------------------------------------------------------------------------------------------- */
   /*  this is intended to be called by the T_InsertToken function, but I suppose it might            */
   /*  have other uses.                                                                               */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg ArrayN , where , HowMany
   if HowMany == '' then HowMany = 1
   if debug.0arg then do
         if ArrayN == '' ,
          | \datatype(where, 'w') ,
          | \datatype(HowMany, 'w') then do
               call ShowError 'In StemInsert, invalid argument(s)'
               call Fatality
            end
         if value(ArrayN) == translate(ArrayN) then do
               call ShowError 'StemInsert: unknown array "'ArrayN'"'
               call ShowError 'called from line' Esigl, ''
               call Fatality
            end
      end

   !Array = value('ArrayN')
   /*  shift tokens with numbers higher than where, to make room                                      */
   do o = value(!Array||0) to where by -1
      n = o + HowMany
      call value !Array||n, value(!Array||o)
   end
   /*  update value of Array.0                                                                        */
   call value !Array||0, value(!Array||0) + HowMany

   /*  use the initialized value of the array for the new tokens                                      */
   do n = where to where + HowMany -1
      call value !Array||n, value(!Array)
   end
   return 0
   /* === End of StemInsert ========================================================================= */

   Text: procedure expose (Shared_vars) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call Text <line number> , <end string>                                                   */
   /*                                                                                                 */
   /*  copies lines from this file starting at <line number> and ending at a line that                */
   /*  says <end string>                                                                              */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*        call Text LineNo() + 2, '<<END>>'                                                        */
   /*        /@                                                                                       */
   /*        This line of text will be copied to the SrcOut array                                     */
   /*        <<END>>                                                                                  */
   /*        @/                                                                                       */
   /*                                                                                                 */
   /*  This is similar to using a TEXT/ENDTEXT block in 4OS2 or "cat <<EOF" in a Unix                 */
   /*  shell script.                                                                                  */
   /*  Note that the example uses "LineNo() +2" to skip over the beginning of the comment             */
   /* =============================================================================================== */
   parse arg start , endstr

   do ln = start to sourceline()
      aline = sourceline(ln)
      if strip(aline) == endstr then leave ln
      call Out aline
   end ln
   return 0
   /* === End of Text =============================================================================== */

   UnWrapLine: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     call UnWrapLine t                                                                           */
   /*  where t is the continuation comma of a continued line                                          */
   /*                                                                                                 */
   /*  Unwraps the line by deleting the comma and the null End-Of-Line token                          */
   /*  If the resulting line is itself continued, it can be unwrapped in a subsequent call            */
   /*  to UnWrapLine                                                                                  */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   n = L_LineEOL(t)
   if t_type.t \= 'CONTINUE' then do
         call ShowError 'BUG:' SubID() 'called from line' Esigl
         call ShowError 'token t is not a continuation comma', ''
         call Mark 't t_type.t t_val.t'
         call Fatality
      end
   if t_type.n \= 'NULL' then do
         call ShowError 'BUG:' SubID() 'called from line' Esigl
         call ShowError 'to unwrap line that is not unwrappable', ''
         if t_class.n == 'COMMENT' then do
               call ShowError 'because it has a comment. The calling routine should', ''
               call ShowError 'check for that first.', ''
            end
         call Mark 't t_type.t t_val.t n t_type.n t_val.n t_line.t'
         call Fatality
      end

   nn = n +1
   /*  remove the line indent from the next word and change the prefix to ' '.                        */
   /*  bugfix-- for people who put an unneeded comma before THEN, because                             */
   /*  the Toker will add a hidden EOC                                                                */
   do while t_type.nn == 'REXX_EOC'
      nn = nn +1
   end
   t_prefix.nn = ' '

   /*  unwrap line by deleting the continuation comma and the following EOL.                          */
   /*  usually they are adjacent tokens, so save time by deleting both in one call                    */
   /*  to T_DeleteToken. If not adjacent, we must delete them separately, in reverse                  */
   /*  order because T_DeleteToken changes the token numbering.                                       */
   if n == t +1 then
      call T_DeleteToken t, 2
   else do
         call T_DeleteToken n
         call T_DeleteToken t
      end
   return 0
   /* === End of UnWrapLine ========================================================================= */

   WordsOnly: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       ret = WordsOnly(string)                                                                   */
   /*  Returns a string with all non-word characters replaced with spaces.                            */
   /*  For this purpose, word characters are the characters that are legal in variable names.         */
   /*  That is,  [a-z][A-Z][0-9]._?!                                                                  */
   /*                                                                                                 */
   /*  This is for Task_Change                                                                        */
   /* =============================================================================================== */
   parse arg args
   ret = translate(args, ' ', '`~@#$%^&*()+-={}|[]\:'';",>/<')
   return ret
   /* === End of WordsOnly ========================================================================== */

   DumpVars: /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        CALL DumpVars <pattern>                                                                  */
   /*        ret = DumpVars(<pattern>)                                                                */
   /*                                                                                                 */
   /*  Dumps current variables to stdout. This can be called from anywhere.                           */
   /*                                                                                                 */
   /*  Most of the time 'CALL Mark <variable list>' is more useful                                    */
   /*                                                                                                 */
   /*  If using OS/2 Rexx with RXU.DLL:                                                               */
   /*  The optional parameter is a name pattern; only variables that start with that string           */
   /*  will be shown. The list will be alphabetized and variables will be shown in the form           */
   /*  of assignment statements.                                                                      */
   /*                                                                                                 */
   /* =============================================================================================== */
   Esigl = sigl
   select
      when G.0Interpreter == 'REGINA' then
         call SysDumpVariables
      when G.0Interpreter == 'REXXSAA' ,
         & RxFuncQuery('RxVlist') == 0 then do
            do while queued() ; call linein 'queue:' ; end
            if arg(1) == 1 then
               call RxVlist '', 'V'
            else
               call RxVlist arg(1), 'V'

            return Dumpvars2()
         end
      otherwise
         call SysDumpVariables
   end
   return 0
   /* === End of DumpVars =========================================================================== */

   Dumpvars2: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*    ret = Dumpvars2(<[list of variable names>])                                                  */
   /* ----------------------------------------------------------------------------------------------- */
   /*  I want a one-line command to dump all variables in the current environment and filter them.    */
   /*                                                                                                 */
   /*  fix: WrapVar() assumes the value is a string of words and that spaces are not significant,     */
   /*       which is not good enough. It should preserve all spaces, splitting string at spaces       */
   /*       if possible, insert single quotes, "||", and line-continuation comma                      */
   /*       and it should replace non-printable characters with hex strings                           */
   /*       and it should replace Comment_Beg with '/'||'*'                                           */
   /*       and maybe deal with other dangerous strings                                               */
   /*       But this does what I need for now.                                                        */
   /* =============================================================================================== */
   v = 0
   varN. = '' ; varN.0 = 0
   varV. = '' ; varV.0 = 0

   do while queued() > 0
      v = v +1
      varN.v = left(linein('queue:'), 50)||v
      varV.v = linein('queue:')
      varN.0 = v
      varV.0 = v
   end
   if varV.0 == 0 then return 0

   call SysStemSort 'varN.', 'A', 'I'
   varNameLen = 45
   indent = varNameLen +4
   maxValLen = G.0ScreenCols - varNameLen -20
   do v = 1 to varN.0
      parse var varN.v varname varNum .
      if varname = 'SIGL' then iterate
      if varname = 'RESULT' then iterate
      if varname = 'RC' then iterate
      varname = left('  '||varname, varNameLen)||'=   '
      varVal = varV.varNum
      if WordsQ(varVal) > 1 then
         varVal = WrapVar("'"||varVal||"'", maxValLen, indent)
      else
         varVal = "'"||varVal||"'"
      call lineout 'stdout', varname||varVal
   end
   return 0
   /* === End of Dumpvars2 ========================================================================== */

   WrapVar: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  this is used only by Dumpvars2                                                                 */
   /* ----------------------------------------------------------------------------------------------- */
   /*  splits a long variable value into shorter strings, adding single quotes and                    */
   /*  line-continuation to produce a variable assignment statement                                   */
   /*  it assumes the string is made of words and that spaces are not significant.                    */
   /* =============================================================================================== */
   len = 80
   indent = 0
   string = arg(1)
   if arg(2, 'e') then len = arg(2)
   if arg(3, 'e') then indent = arg(3)
   if length(string) < len then return string
   oldstring = ''
   newstring = ''
   do while string \= ''
      parse var string aword string
      if length(newstring||' '||aword) < len then newstring = newstring||' '||aword
      else do
            oldstring = strip(oldstring, 'l')||newstring||"'"||" ,"||G.0CRLF||copies(' ', indent)
            newstring = "'"||aword
         end
   end
   return strip(oldstring||newstring)
   /* === End of WrapVar ============================================================================ */

   WordWrap: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     say WordWrap(<string>,[<width>])                                                            */
   /*  where <string> is composed of words                                                            */
   /*                                                                                                 */
   /*  wraps a string of words to the specified width by inserting line-endings into it               */
   /*  default width is 80                                                                            */
   /*                                                                                                 */
   /*  if a word is longer than <width> it will result in a longer line than was requested            */
   /* =============================================================================================== */
   len = 80

   string = arg(1)
   if arg(2, 'e') then len = arg(2)
   if length(string) <= len then return string
   oldstring = ''
   newstring = ''
   do while string \= ''
      parse var string aword string
      if length(newstring||' '||aword) <= len then
         newstring = strip(newstring||' '||aword, 'l')
      else do
            oldstring = oldstring||newstring||G.0CRLF
            newstring = aword
         end
   end
   return strip(oldstring||newstring)
   /* === End of WordWrap =========================================================================== */

   /*  B_   Block-related procedures                                                           */ /*fold00*/
   B_EndBlock: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        r = B_EndBlock(t)                                                                        */
   /*  where t is a token of type REXX_LABEL                                                          */
   /*                                                                                                 */
   /*  returns the EOL token that ends the block that token t identifies                              */
   /* =============================================================================================== */
   parse arg t
   return B_EndOfPrev(B_NextLabel(t))
   /* === End of B_EndBlock ========================================================================= */

   B_EndOfPrev: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        ret = B_EndOfPrev(t)                                                                     */
   /*  where t is a token of type REXX_LABEL                                                          */
   /*                                                                                                 */
   /*  Returns the EOL token that marks the end of the previous block.                                */
   /*  What we want is the line that the author (presumably) thinks of as the end of                  */
   /*  the routine. That is, after a terminating footer or divider or empty line, but                 */
   /*  before any header or other comment above the current label.                                    */
   /*  If label t is at the start of the file, returns the EOL of the comment on line 1               */
   /*  or the FILE_BOF if there is no comment                                                         */
   /*                                                                                                 */
   /*  This is the basis for the other B_xxx functions. What I call a block is a                      */
   /*  section of  code that (as near as we can tell) is considered a unit, including                 */
   /*  sub-labels and top and bottom comments. Like this:                                             */
   /*                                                                                                 */
   /*     /@ ======================== @/                                                              */
   /*     /@ a header above the label @/                                                              */
   /*     /@ ======================== @/                                                              */
   /*     Foo:                                                                                        */
   /*     nop                                                                                         */
   /*     Foo.loop:                                                                                   */
   /*     nop                                                                                         */
   /*     signal Foo.loop                                                                             */
   /*     return                                                                                      */
   /*     /@ a trailing comment @/                                                                    */
   /*                                                                                                 */
   /*  This could be confused by a comment after the end of a routine that is                         */
   /*  separated from the routine by a blank line-- it will be mistaken for part of                   */
   /*  the header of the following routine. There is probably no fix for that.                        */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   /*  any comments at end of file are part of the last block                                         */
   /*     if t_class.t == 'FILE_EOF' then return t -1                                                 */
   if t_class.t == 'FILE_EOF' then do
         Lcount = -1
         do forever
            if \L_IsEmpty(t, Lcount) then leave
            Lcount = Lcount -1
         end
         EndPrev = L_LineEOL(t, Lcount)
         return EndPrev
      end

   if t_type.t \== 'REXX_LABEL' then do
         call ShowError 'Bug: invalid argument to B_EndOfPrev, from line' Esigl 'in' LastLabel(Esigl)
         call Mark 'Esigl t t_class.t t_type.t t_val.t'
         call Fatality
      end
   rID = RexxID()
   if t_type.rID == 'COMMENT_BEG' then
      BoF = L_LineEOL(C_OtherEnd(rID))
   else
      BoF = 1

   Lcount = -1
   LastRexx = 0
   /*  search backward from t for the last Rexx line:                                                 */
   do forever
      if L_HasRexx(t, Lcount) then do
            LastRexx = L_LineEOL(t, Lcount)
            leave
         end
      if L_IsFooter(t, Lcount) then do
            LastRexx = L_LineEOL(t, Lcount)
            leave
         end
      Lcount = Lcount -1
      if L_LineEOL(t, Lcount) <= BoF then
         leave
   end
   /*  if not found, return Beginning-Of-File token                                                   */
   if LastRexx == 0 then
      return BoF

   /*  now search forward from here to label for a blank line. If none, assume all                    */
   /*  comment lines are a top header for the label                                                   */
   Lcount = 1
   do forever
      if L_IsEmpty(LastRexx, Lcount) then do
            EndPrev = L_LineEOL(LastRexx, Lcount -1)
            leave
         end
      if L_LineEOL(LastRexx, Lcount) >= t then do
            EndPrev = L_LineEOL(LastRexx)
            leave
         end
      Lcount = Lcount +1
   end   /*  forever  */
   return EndPrev
   /* === End of B_EndOfPrev ======================================================================== */

   B_NextLabel: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        r = B_NextLabel(t)                                                                       */
   /*  where t is a token of type REXX_LABEL                                                          */
   /*                                                                                                 */
   /*  returns the token number of the next label.                                                    */
   /*  If there is no next label, returns the FILE_EOF token                                          */
   /*  unlike R_NextLabel, it ignores sub-labels that start with <label>.                             */
   /*                                                                                                 */
   /*  This was written for B_EndBlock() and is probably not useful anywhere else                     */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if t_class.t \== 'FILE_BOF' ,
    & t_type.t \== 'REXX_LABEL' then do
         call ShowError 'Error: invalid argument to B_NextLabel, from line' Esigl 'in' LastLabel(Esigl)
         call Mark 'Esigl t t_type.t t_val.t'
         call Fatality
      end
   do s = t +1 to t_type.0
      if t_type.s == 'REXX_LABEL' then do
            /*  a sub-label does not count. See "Help sublabel"                                       */
            if abbrev(translate(t_val.s), translate(t_val.t)||'.') then iterate s
            return s
         end
   end s
   return t_type.0
   /* === End of B_NextLabel ======================================================================== */

   B_PrevLabel: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = B_PrevLabel(t)                                                                        */
   /*                                                                                                 */
   /*  returns the token number of the previous label,                                                */
   /*                                                                                                 */
   /*  Do not use this. It is left in as a reminder that a B_PrevLabel function is impossible.        */
   /*                                                                                                 */
   /*  fix: explain why. The reason isn't obvious and I can't remember what it was.                   */
   /*        --for one thing, it cannot recognize a sub-label with a name based on the                */
   /*        previous label, but that was not the original reason.                                    */
   /* =============================================================================================== */
   return
   /* === End of B_PrevLabel ======================================================================== */

   B_StartBlock: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        r = B_StartBlock(t)                                                                      */
   /*                                                                                                 */
   /*  returns the token that starts this block                                                       */
   /*  where t is the label token                                                                     */
   /*                                                                                                 */
   /*  If there is a top header, r will be the start of the header. Otherwise r will be               */
   /*  the End-Of-Line token after the end of the previous block. In other words, empty               */
   /*  lines before the label are considered a top header.                                            */
   /* =============================================================================================== */
   parse arg t
   return B_EndOfPrev(t) +1
   /* === End of B_StartBlock ======================================================================= */

   /*  C_   comment-related procedures                                                         */ /*fold00*/
   /***************************************************************************************************/
   C_CommentType: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     if C_CommentType(t) == ...                                                                  */
   /*  where t is any COMMENT token                                                                   */
   /*                                                                                                 */
   /*  returns what kind of comment that token 't' is a part of.                                      */
   /*  'FULL'   is a one-line comment,                                                                */
   /*  'MULTI'  a multi-line comment,                                                                 */
   /*  'RIGHT'  on the right side of a Rexx statement                                                 */
   /*  'LEFT'   on the left side,                                                                     */
   /*  'MIDDLE' in the middle between two Rexx statements, or between Rexx and a comment              */
   /*  'OTHER'  means there is something missing in this subroutine                                   */
   /*  ''       means token t is not a comment                                                        */
   /*  LEFT and MIDDLE comments are legal in Rexx, but anyone who uses that kind of                   */
   /*  commenting style will have to write their own formatter.                                       */
   /* =============================================================================================== */
   /*  look at the tokens before and after the comment. the order is important--                      */
   /*  check FILE_EOL (a real EOL) before REXX_EOC (either a real EOL or ";")                         */
   /* ----------------------------------------------------------------------------------------------- */
   /*  to understand the patterns: (L == line, R == Rexx, C == Comment, M == middle)                  */
   /*                                                                                                 */
   /*     /@ L L @/                                                                                   */
   /*                                                                                                 */
   /*     say 'Hi' /@ R L @/                                                                          */
   /*     say 'Hi' /@ R C @/  /@ C L @/                                                               */
   /*                                                                                                 */
   /*     /@ L M                                                                                      */
   /*        M M                                                                                      */
   /*        M L @/                                                                                   */
   /*                                                                                                 */
   /*     say 'Hi' /@ R M                                                                             */
   /*     M M                                                                                         */
   /*     M R @/ say 'Hi'                                                                             */
   /*                                                                                                 */
   /*     say  /@ R R @/ 'Hi'                                                                         */
   /*                                                                                                 */
   /* ----------------------------------------------------------------------------------------------- */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   if \abbrev(t_type.t, 'COMMENT') then do
         call C_Errorhandler t, Esigl
         return ''
      end
   Pat = ' '
   do s = t to 2 by -1
      if abbrev(T_Type(s), 'COMMENT_BEG') then do
            select
               when abbrev(T_Class(s -1), 'FILE_') then Pat = 'L'Pat
               when T_Class(s -1) == 'REXX' then Pat = 'R'Pat
               when T_Class(s -1) == 'COMMENT' then Pat = 'C'Pat
               otherwise Pat = 'U'Pat
            end
            leave
         end
      if T_Type(s) = 'COMMENT_EOL' then do
            Pat = 'M'Pat
            leave
         end
   end
   do s = t to t_type.0
      if T_Type(s) = 'COMMENT_END' then do
            select
               when abbrev(T_Class(s +1), 'FILE_') then Pat = Pat'L'
               when T_Class(s +1) == 'REXX' then Pat = Pat'R'
               when T_Class(s +1) == 'COMMENT' then Pat = Pat'C'
               otherwise Pat = Pat'U'
            end
            leave
         end
      if T_Type(s) = 'COMMENT_EOL' then do
            Pat = Pat'M'
            leave
         end
   end
   select
      when Pat = 'L L' then return 'FULL'
      when Pat = 'L R' then return 'LEFT'
      when Pat = 'L C' then return 'LEFT'
      when Pat = 'R L' then return 'RIGHT'
      when Pat = 'R R' then return 'MIDDLE'
      when Pat = 'R C' then return 'MIDDLE'
      when Pat = 'C L' then return 'RIGHT'
      when Pat = 'C R' then return 'MIDDLE'
      when Pat = 'C C' then return 'MIDDLE'
      when Pat = 'M M' then return 'MULTI_MM'
      when Pat = 'M R' then return 'MULTI_MR'
      when Pat = 'M L' then return 'MULTI_ML'
      when Pat = 'M C' then return 'MULTI_MC'
      when Pat = 'R M' then return 'MULTI_RM'
      when Pat = 'L M' then return 'MULTI_LM'
      when Pat = 'C M' then return 'MULTI_CM'
      otherwise
         call ShowError 'BUG: unknown comment type: OTHER_'Pat
         call ShowError 'called from Esigl' Esigl, ''
         call ShowError Qline(t_line.t), ''
         call Fatality
   end   /*  select  */
   /*  this will never happen                                                                         */
   call C_Errorhandler t, Esigl
   return ''
   /* === End of C_CommentType ====================================================================== */

   C_Errorhandler: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*         if <something wrong> then call C_Errorhandler <token>, Esigl                            */
   /*                                                                                                 */
   /*  for debugginng; reports some information when a C_xxx function has a problem                   */
   /*  this makes it easier for the C_ functions to report problems, like an invalid argument         */
   /*                                                                                                 */
   /*  Some routines use                                                                              */
   /*        if <something wrong> then return C_Errorhandler(t, Esigl)                                */
   /*  which is short for                                                                             */
   /*        if <something wrong> then do                                                             */
   /*           call C_Errorhandler t, Esigl                                                          */
   /*           return 0                                                                              */
   /*        end                                                                                      */
   /*  But do not use that construction if the routine normally returns a token number                */
   /*  because C_Errorhandler always returns 0                                                        */
   /*                                                                                                 */
   /*  fix: this is seldom used since I finished the C_ functions. Maybe turn it into general         */
   /*        purpose errorhandler and use it for other things.  Maybe using ShowContext()             */
   /* =============================================================================================== */
   Esigl = sigl
   Err_func = LastLabel(Esigl)
   parse arg t , Lastsigl
   call ShowError '', ''
   call ShowError '[C_Errorhandler] Error in "'Err_func'" line' Esigl
   call ShowError '      'sourceline(Esigl), ''
   if G.0track_vars \= '' then
      call Mark G.0track_vars, 'for C_Errorhandler'
   if t \= '' then
      call ShowError 'parameter to C_Errorhandler was "'t', 'Lastsigl'"', ''
   if datatype(t, 'w') ,
    & t > 0 ,
    & t <= t_type.0 then
      call Mark 't t_class.t t_type.t t_prefix.t t_val.t', ''
   call Fatality
   return 0
   /* === End of C_Errorhandler ===================================================================== */

   C_Fmt_Divider: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL C_Fmt_Divider t                                                                      */
   /*  where t is a COMMENT_VAL token                                                                 */
   /*                                                                                                 */
   /*  grow or shrink the width of a divider comment                                                  */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , width
   if width == '' then width = Opt.0Width
   if debug.0arg then call T_Valid t, Esigl
   if \C_IsDivider(t) then return 0
   lastP = T_TokenColumn(t)
   /*  lastP is after "/@", subtract length of "@/"                                                   */
   C_width = width - (lastP -1) -2
   c_val = StripStr(t_val.t)
   chr = right(c_val, 1)
   /*  the only choices are 1 space at each end, or no spaces                                         */
   blanks = 0 ; if length(t_val.t) > length(c_val) then blanks = 2

   /*  bugfix-- dividers can now have text. Regardless of width, we need to make sure                 */
   /*  all the text is included as well as at least 30 copies of chr (so the comment                  */
   /*  can be identified as a divider by S_DividerType)                                               */
   c_val1 = strip(c_val, 't', chr)
   min_width = length(c_val1) + 30 + blanks
   if C_width < min_width then
      C_width = min_width
   c_valNew = left(c_val, C_width - blanks, chr)
   if blanks > 0 then c_valNew = ' '||c_valNew||' '
   t_val.t = c_valNew
   return 0
   /* === End of C_Fmt_Divider ====================================================================== */

   C_Fmt_Right: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL C_Fmt_Right t                                                                        */
   /*  where t is a COMMENT_VAL token of a Right-side comment                                         */
   /*                                                                                                 */
   /*  adjusts size and position of Right-side comments                                               */
   /*  see switches -MoveComment and -EndComment                                                      */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   if t_type.t \= 'COMMENT_VAL' then return C_Errorhandler(t)
   if C_CommentType(t) \= 'RIGHT' then return C_Errorhandler(t)
   ValidOpts = 'LEFT'
   if Opt.0EndComment \= '' ,
    & wordpos(translate(Opt.0EndComment), ValidOpts) == 0 then do
         call Msg 'unrecognized option: "-EndComment='Opt.0EndComment'"'
      end
   ValidOpts = 'RIGHT LEFT EXPAND'
   if Opt.0MoveComment \= '' ,
    & \datatype(Opt.0MoveComment, 'w') ,
    & wordpos(translate(Opt.0MoveComment), ValidOpts) == 0 then do
         call Msg 'unrecognized option: "-MoveComment='Opt.0MoveComment'"'
      end
   /* ----------------------------------------------------------------------------------------------- */
   ctBeg = t -1                                                            /* the "/@" of the comment */
   tFirst = L_FirstToken(t)                                                /* first token of the line */
   tFirstVal = T_Val(tFirst)
   /* ----------------------------------------------------------------------------------------------- */
   /*  special handling for a short note after an END statement, to look like this:                   */
   /*    |  Foo = 1                                /@ a comment about Foo @/                          */
   /*    |  END   /@ of SELECT @/                                                                     */
   /*  Opt.0EndComment is misnamed, because now it also applies to LEAVE, SELECT and DO               */
   if translate(Opt.0EndComment) = 'LEFT' then do
         if abbrev(t_type.tFirst, 'REXX_W') ,
          & wordpos(translate(tFirstVal), 'END SELECT LEAVE DO') > 0 then do
               if length(strip(t_val.t)) < 30 then do
                     /*  a short prefix and short spaces around comment_val; do not right-justify     */
                     ret = C_StripVal(t, , '  ')
                     ret = C_ShiftLeft(t, '   ')
                     return 0
                  end
            end
      end   /*  if End_Comment = LEFT  */

   select
      when translate(Opt.0MoveComment) == 'RIGHT' then do
            /*  move comment text to the right margin                                                 */
            call C_MoveRight t
         end
      when translate(Opt.0MoveComment) == 'LEFT' then do
            /*  move comment text to the left                                                         */
            call C_MoveLeft t
         end
      when datatype(Opt.0MoveComment, 'w') then do
            call T_FloatToken ctBeg, Opt.0MoveComment
         end
      when translate(Opt.0MoveComment) == 'EXPAND' then do
            /*    right-justify end of comment but do not move text                                   */
            call C_JustifyRight t
         end
      otherwise nop
   end   /*  select  */
   return 0
   /* === End of C_Fmt_Right ======================================================================== */

   C_InsertComment: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL C_InsertComment t, <text>                                                            */
   /*                                                                                                 */
   /*  inserts a new comment of <text> before token t                                                 */
   /*  note that it adds a fixed prefix of four spaces before the comment.                            */
   /*  The caller must decide whether to change it.                                                   */
   /*  Remember that <text> must include its leading and trailing spaces.                             */
   /*  This is intended for inserting a right-hand comment, when t is an End Of Line token.           */
   /*  But it will also insert left or middle comments if you want.                                   */
   /*  For a full-width comment, see C_InsertCommentLine                                              */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , Tval
   if debug.0arg then call T_Valid t, Esigl
   call T_InsertToken t, 3
   /*  t now points to the first of the new, empty, tokens                                            */

   t_class.t = 'COMMENT'
   t_type.t = 'COMMENT_BEG'
   t_val.t = '/'||'*'
   t_prefix.t = '    '

   t = t +1
   t_class.t = 'COMMENT'
   t_type.t = 'COMMENT_VAL'
   t_val.t = Tval
   t_prefix.t = ''

   t = t +1
   t_class.t = 'COMMENT'
   t_type.t = 'COMMENT_END'
   t_val.t = '*'||'/'
   t_prefix.t = ''

   return 0
   /* === End of C_InsertComment ==================================================================== */

   C_InsertCommentLine: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL C_InsertCommentLine t, <text>                                                        */
   /*                                                                                                 */
   /*  inserts a new Line Comment of <text> above the line that token t is in                         */
   /*  it does not break the current line                                                             */
   /* =============================================================================================== */
   /*  the tokens will be                                                                             */
   /*        '/@'  '<text>'  '@/'  <EOL>                                                              */
   /*  inserted before the beginning of the current line                                              */
   /* ----------------------------------------------------------------------------------------------- */
   parse arg t , Tval
   p = L_PrevEOL(t) +1
   /*  p is the first token of the current line, which may be the ending EOL                          */
   call T_InsertToken p, 4

   /*  p is now the start of the new line                                                             */
   t_class.p = 'COMMENT'
   t_type.p = 'COMMENT_BEG'
   t_val.p = '/'||'*'
   t_prefix.p = ''

   p = p +1
   t_class.p = 'COMMENT'
   t_type.p = 'COMMENT_VAL'
   t_val.p = Tval
   t_prefix.p = ''

   p = p +1
   t_class.p = 'COMMENT'
   t_type.p = 'COMMENT_END'
   t_val.p = '*'||'/'
   t_prefix.p = ''

   p = p +1
   t_class.p = 'FILE_EOL'
   t_type.p = 'REXX_EOC'
   t_val.p = ''
   t_prefix.p = ''

   return 0
   /* === End of C_InsertCommentLine ================================================================ */

   C_InsertEOL: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL C_InsertEOL <t>                                                                      */
   /*                                                                                                 */
   /*  in a multiline comment, inserts a new End-Of-Line at (ie, before) token t                      */
   /*                                                                                                 */
   /*  unused, maybe should be tested to see if it still works                                        */
   /* =============================================================================================== */
   parse arg t
   ret = T_InsertToken(t)
   if ret \= 0 then return ret
   t_class.t = 'FILE_EOL'
   t_type.t = 'COMMENT_EOL'
   t_val.t = ''
   t_prefix.t = ''
   return  0
   /* === End of C_InsertEOL ======================================================================== */

   C_IsDivider: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       ret = C_IsDivider(t)                                                                      */
   /*  where t is a COMMENT_VAL token                                                                 */
   /*                                                                                                 */
   /*  returns True if token t is any kind of divider                                                 */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   if t_class.t \= 'COMMENT' then return 0
   if C_CommentType(t) \= 'FULL' then return 0
   if S_DividerType(t_val.t) == '' then return 0
   return 1
   /* === End of C_IsDivider ======================================================================== */

   C_IsFooter: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        if C_IsFooter(t) then                                                                    */
   /*  where t is a COMMENT_VAL token                                                                 */
   /*                                                                                                 */
   /*  returns True if a comment value is a footer line, like " == End of <label> "                   */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if C_CommentType(t) \= 'FULL' then return 0
   if abbrev(translate(t_val.t), translate(G.0footerID))  then return 1
   return 0
   /* === End of C_IsFooter ========================================================================= */

   C_JustifyRight: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL C_JustifyRight(t)                                                                    */
   /*  where t is a COMMENT_VAL token                                                                 */
   /*                                                                                                 */
   /*  adjusts the right side of a comment                                                            */
   /*  called by C_Fmt_Full, C_Fmt_Right                                                              */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , width
   if width == '' then width = Opt.0Width
   if debug.0arg then call T_Valid t, Esigl
   if t_type.t \= 'COMMENT_VAL' then return C_Errorhandler(t)
   ctype = C_CommentType(t)
   if wordpos(ctype, 'FULL RIGHT') == 0 then return C_Errorhandler(t)

   /*  lastP == start of comment_val; C_width == width of comment_val so the "@/" is at far right     */
   lastP = T_TokenColumn(t)
   C_width = width - (lastP -1) -2
   if length(t_val.t) = C_width then return 0                                     /* no change needed */

   /*  strip trailing spaces and replace them with the spacing we want:                               */
   /*   there should be at least one space before "@/" even if it breaks the right-justification      */
   c_val = StripStr(t_val.t, 't')||' '

   /*  if comment text is too short, good                                                             */
   if length(c_val) < C_width then c_val = left(c_val, C_width, ' ')
   /*  if comment text is still too long, we cannot fix that here.                                    */
   /*  Shifting or reformatting the text happens in other routines                                    */
   t_val.t = c_val
   return 0
   /* === End of C_JustifyRight ===================================================================== */

   C_MoveLeft: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL C_MoveLeft(t)                                                                        */
   /*  where t is a COMMENT_VAL token                                                                 */
   /*                                                                                                 */
   /*  moves a right comment to the left, do not justify on right                                     */
   /*  so this:                                                                                       */
   /*        IF a THEN                                      /@  right comment                    @/   */
   /*  becomes this:                                                                                  */
   /*        IF a THEN    /@  right comment  @/                                                       */
   /*                                                                                                 */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   if t_type.t \= 'COMMENT_VAL' then return C_Errorhandler(t)
   if C_CommentType(t) \= 'RIGHT' then return C_Errorhandler(t)
   /*  strip extra spaces from comment_val                                                            */
   call C_StripVal t

   /*  shorten/lengthen the comment prefix to 4 spaces                                                */
   call T_Prefix t -1, '    '
   return 0
   /* === End of C_MoveLeft ========================================================================= */

   C_MoveRight: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL C_MoveRight(t)                                                                       */
   /*  where t is a COMMENT_VAL token                                                                 */
   /*                                                                                                 */
   /*  moves a comment to the right-hand side                                                         */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   if t_type.t \= 'COMMENT_VAL' then return C_Errorhandler(t)
   if C_CommentType(t) \= 'RIGHT' then return C_Errorhandler(t)
   /*  lastP == end of Rexx statement +1                                                              */
   lastP = T_TokenColumn(t -1)
   /*  strip extra spaces from comment_val                                                            */
   call C_StripVal t

   /*  calculate prefix to align end of comment with right margin, but with a prefix of at least      */
   /*  four spaces. If the comment still runs past the right margin, we cannot fix that.              */
   commLen = length(t_val.t) +4
   linelen = lastP -1
   newprefix = Opt.0Width - linelen - commLen
   newprefix = max(3, newprefix)

   prefix = copies(' ', newprefix)
   call T_Prefix t -1, prefix
   return 0
   /* === End of C_MoveRight ======================================================================== */

   C_OtherEnd: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       ret = C_OtherEnd(t)                                                                       */
   /*  where t is a /@ or @/ token                                                                    */
   /*                                                                                                 */
   /*  returns the other end of a comment                                                             */
   /*  if token t is COMMENT_BEG, return token of COMMENT_END                                         */
   /*  if token t is COMMENT_END, return token of COMMENT_BEG                                         */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   select
      when abbrev(t_type.t, 'COMMENT_BEG') then do
            t_start = t +1 ; t_end = t_type.0 ; step = 1
         end
      when abbrev(t_type.t, 'COMMENT_END') then do
            t_start = t -1 ; t_end = 1 ; step = -1
         end
      otherwise
         call C_Errorhandler t, Esigl
         return ''
   end

   level = 1
   do e = t_start to t_end by step
      select
         when abbrev(t_type.e, 'COMMENT_BEG') then
            level = level + step
         when abbrev(t_type.e, 'COMMENT_END') then
            level = level - step
         otherwise nop
      end
      if level = 0 then return e
   end e
   call ShowError 'BUG: at line' t_line.t', no match for token "'t_val.t'"'
   call ShowError '  ==>' L_AsString(t), ''
   call ShowError 'called from line' Esigl, ''
   call ShowError 'This is either a bug in the Tokeniser or in a task that changes comments', ''
   call Fatality
   return t_end
   /* === End of C_OtherEnd ========================================================================= */

   C_ShiftLeft: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL C_ShiftLeft t [, <prefix>]                                                           */
   /*  where t is a COMMENT_VAL token                                                                 */
   /*                                                                                                 */
   /*  shifts a Right comment to the left, closer to Rexx code, by changing prefix of ctBeg           */
   /*  the default prefix is '    ' (4 spaces)                                                        */
   /* =============================================================================================== */
   Esigl = sigl
   t = arg(1)
   if arg(2, 'e') then prefix = arg(2) ; else prefix = '    '
   if t_type.t \= 'COMMENT_VAL' then return C_Errorhandler(t)
   if C_CommentType(t) \= 'RIGHT' then return C_Errorhandler(t)
   ctBeg = t -1
   t_prefix.ctBeg = prefix
   return 0
   /* === End of C_ShiftLeft ======================================================================== */

   C_StripVal: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       ret = C_StripVal( t , [where] , [<add>])                                                  */
   /*  where <t> is a COMMENT_VAL token                                                               */
   /*  <where> is a strip() argument 'L' | 'T' | 'B'                                                  */
   /*  <add> is a string to add to both sides of the text. Default add is "  "                        */
   /*                                                                                                 */
   /*  trims spaces from the text inside a comment (of any CommentType)                               */
   /* =============================================================================================== */
   Esigl = sigl
   t = arg(1)
   if arg(2, 'e') then where = translate(arg(2)) ; else where = 'B'
   if arg(3, 'e') then add = arg(3) ; else add = ' '
   if debug.0arg then call T_Valid t, Esigl
   if t_type.t \= 'COMMENT_VAL' then return C_Errorhandler(t)
   if verify(where, 'BLT') > 0 then return C_Errorhandler(t)
   Cval = StripStr(t_val.t, where)
   if where = 'B' ,
    | where = 'L' then Cval = add||Cval
   if where = 'B' ,
    | where = 'T' then Cval = Cval||add
   t_val.t = Cval
   return 0
   /* === End of C_StripVal ========================================================================= */

   /*  CL_  clause-related procedures                                                          */ /*fold00*/
   /***************************************************************************************************/
   CL_AsString: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        r = CL_AsString(t[,<count>])                                                             */
   /*                                                                                                 */
   /*  returns the specified clause as a string                                                       */
   /*  Examples:                                                                                      */
   /*        r = CL_AsString(t)       returns the clause that token t is part of                      */
   /*        r = CL_AsString(t, -1)   returns the clause before the clause that token t is part of    */
   /*        r = CL_AsString(t, 1)    returns the clause after the clause that token t is part of     */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0

   p2 = CL_FirstWord(t, count)
   str = ''
   do w = p2 to t_type.0
      if t_type.w == 'REXX_EOC' then leave w
      if abbrev(t_type.w, 'REXX_') then
         str = str||t_prefix.w||t_val.w
   end w
   return strip(str)
   /* === End of CL_AsString ======================================================================== */

   CL_ClauseEOC: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = CL_ClauseEOC(t [, <count>])                                                           */
   /*       r = CL_ClauseEOC(t)                                                                       */
   /*                                                                                                 */
   /*  returns the token number that marks the end of a clause (EOC)                                  */
   /*  if no <count> is given, the clause that token t is in                                          */
   /*  if <count> is negative, the clause is <count> clauses before where token t is                  */
   /*  if <count> is positive, the clause is <count> clauses after where token t is                   */
   /*                                                                                                 */
   /*  A clause can be referred to by the number of its End-Of-Clause token.                          */
   /*                                                                                                 */
   /*  Comments and empty clauses are ignored here because to Rexx an empty clause is                 */
   /*  not a clause, it is just nothing.                                                              */
   /*  However, we need a way to identify an empty clause, so the badly-named CL_HasRexx()            */
   /*  function considers the "clause" as the space between two EOCs. (A function named               */
   /*  "The_Tokens_Between_the_Previous_EOC_and_The_Next_EOC()" would be impractical.)                */
   /*                                                                                                 */
   /*  This will be the basis for all CL_xxx functions that take a <count> argument,                  */
   /*  when I write them.                                                                             */
   /*                                                                                                 */
   /* ----------------------------------------------------------------------------------------------- */
   /*  Note that a label is not a clause. Sometimes it would be convenient to treat                   */
   /*  it as one. But I cannot tag a label as type REXX_W_LABEL, because Rexx does                    */
   /*  not consider it a word (also it would break the indenting). As in this example,                */
   /*  where the ELSE applies to the next statement, ignoring the label:                              */
   /*              else                                                                               */
   /*              Label:                                                                             */
   /*                 do a = 1 to 3                                                                   */
   /*                                                                                                 */
   /*  Tentative solution: in the CL_xxx functions only, I will consider a label to be a clause.      */
   /*  We'll see how it works...                                                                      */
   /*                                                                                                 */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl
   if \abbrev(t_type.t, 'REXX_') then do
         call ShowError 'BUG: invalid call to CL_ClauseEOC from line' Esigl 'in' LastLabel(Esigl)
         call Mark 't t_val.t t_type.t count', ''
         call Fatality
      end

   p = 0
   c = 0
   select
      when count < 0 then do
            t = R_PrevEOC(t)
            do p = t to 1 by -1
               n = p +1
               if t_type.p == 'REXX_EOC' then
                  if t_type.n \== 'REXX_EOC' then
                     if CL_HasRexx(p) then
                        c = c +1
               if c == abs(count) then leave
            end
         end
      when count >= 0 then
         do p = t to t_type.0 -1 by 1
            n = p -1
            if t_type.p == 'REXX_EOC' then
               if t_type.n \== 'REXX_EOC' then
                  if CL_HasRexx(p) then
                     c = c +1
            if c == count +1 then leave
         end
      otherwise nop
   end
   return p
   /* === End of CL_ClauseEOC ======================================================================= */

   CL_Clear: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call CL_Clear t [, <count>]                                                              */
   /*                                                                                                 */
   /*  Removes the specified clause by deleting tokens.                                               */
   /*                                                                                                 */
   /*  It does not remove the terminating End-Of-Clause token.                                        */
   /*  When deleting a clause, it would seem logical to also delete the EOC token.                    */
   /*  But this is currently only used by Task_AddCounter, where it is more convenient                */
   /*  to not delete the EOC.                                                                         */
   /*                                                                                                 */
   /*  If there is a comment embedded within the clause, which is very unlikely, the                  */
   /*  comment will also be deleted.                                                                  */
   /*                                                                                                 */
   /*  returns the number of tokens that were deleted                                                 */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*     to remove the clause that token t is in                                                     */
   /*        call CL_Clear t                                                                          */
   /*     to remove the clause before the clause that t is in                                         */
   /*        call CL_Clear t, -1                                                                      */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   start = CL_FirstWord(t, count)
   stop = CL_LastWord(t, count)
   num = stop - start +1
   call T_DeleteToken start, num
   return num
   /* === End of CL_Clear =========================================================================== */

   CL_FirstWord: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = CL_FirstWord(t [,<count>])                                                            */
   /*  Example:                                                                                       */
   /*       if T_Val(CL_FirstWord(t)) == 'CALL' ...                                                   */
   /*                                                                                                 */
   /*  given token t, returns the number of the first Rexx word in clause <count>.                    */
   /*  that is, Comment tokens and non-word Rexx tokens are ignored.                                  */
   /*  <count> is relative to the clause that t is in, so                                             */
   /*        CL_FirstWord(t, -1)                                                                      */
   /*  refers to the clause before the clause that token t is in                                      */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = CL_ClauseEOC(t, count)
   p = R_PrevEOC(p)
   p = R_NextWord(p)

   if p \== '' then
      return p
   else do
         call ShowError 'BUG: clause has no Rexx words, called from line 'Esigl
         call ShowError 'here is the context:', ''
         call ShowContext t
         call Fatality
      end
   return ''
   /* === End of CL_FirstWord ======================================================================= */

   CL_HasRexx: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     r = CL_HasRexx(t)                                                                           */
   /*     where t is an EOC token                                                                     */
   /*                                                                                                 */
   /*  returns True if the clause that ends at token t has any Rexx tokens.                           */
   /*                                                                                                 */
   /*  This is only used internally by the CL_ClauseEOC() function and it does not take               */
   /*  a <count> argument.                                                                            */
   /*  To Rexx, a clause without Rexx tokens is not a clause.                                         */
   /*  See comments under CL_ClauseEOC.                                                               */
   /* =============================================================================================== */
   parse arg t
   do tt = t to t_type.0
      if t_type.tt == 'REXX_EOC' then do
            t = tt
            leave tt
         end
   end tt
   do p = t -1 to 1 by -1
      if t_type.p == 'REXX_EOC' then
         leave p
      if abbrev(t_type.p, 'REXX_') then
         return 1
   end
   return 0
   /* === End of CL_HasRexx ========================================================================= */

   CL_LastWord: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = CL_LastWord(t [,<count>])                                                             */
   /*                                                                                                 */
   /*  given token t, returns the number of the last REXX Word token in the clause.                   */
   /*  that is, Comment tokens and non-word Rexx tokens are ignored                                   */
   /*  <count> is relative to the clause that t is in                                                 */
   /*  if clause has no Rexx words it returns '', which may cause an error later                      */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = CL_ClauseEOC(t, count)
   do c = p -1 to R_PrevEOC(p) by -1
      if abbrev(t_type.c, 'REXX_') then return c
   end c
   call ShowError 'BUG: clause has no Rexx words, called from line 'Esigl
   call ShowError 'here is the context:', ''
   call ShowContext t
   call Fatality
   return ''
   /* === End of CL_LastWord ======================================================================== */

   /*  L_   line-related procedures                                                            */ /*fold00*/
   /***************************************************************************************************/
   L_AsString: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_AsString(t [, <count>])                                                             */
   /*       r = L_AsString(t)                                                                         */
   /*                                                                                                 */
   /*  returns an entire line as a string by combining the prefixes and values of all tokens          */
   /*  if no <count> is given, returns the line that token t is in                                    */
   /*  if <count> is negative, returns the line that is <count> lines above token t                   */
   /*  if <count> is positive, returns the line that is <count> lines below token t                   */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   Last = L_LineEOL(t, count)
   if Last > t_type.0 then return ''
   First = L_PrevEOL(Last) +1
   if First > Last then return ''
   /*  for the first token, if indenting has already been calculated use indent.                      */
   /*  Otherwise use its old prefix                                                                   */
   if symbol('t_indent.0') == 'LIT' then
      outstring = t_prefix.First
   else
      outstring = copies(' ', format(t_indent.First * Opt.0Tab, , 0))
   outstring = outstring||t_val.First
   do i = First +1 to Last
      outstring = outstring||t_prefix.i||t_val.i
   end
   return strip(outstring, 't')
   /* === End of L_AsString ========================================================================= */

   L_ClauseCount: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        ret = L_ClauseCount(t [, <count>])                                                       */
   /*        IF L_ClauseCount(t, -1) > 1 THEN ...                                                     */
   /*  where t is any token in a line                                                                 */
   /*                                                                                                 */
   /*  returns the number of Rexx clauses in a line. This line has one clause:                        */
   /*           DO a = 1 to 3                                                                         */
   /*                                                                                                 */
   /*  this line has two clauses, because THEN is a clause                                            */
   /*           THEN DO a = 1 to 3  /@ comment @/                                                     */
   /*                                                                                                 */
   /*  it is the same as writing                                                                      */
   /*           THEN ; DO a = 1 to 3  /@ comment @/                                                   */
   /*                                                                                                 */
   /*  this line has 0 clauses:                                                                       */
   /*           /@ a comment @/                                                                       */
   /*                                                                                                 */
   /*  this line really has one and a half clauses because the End-Of-Clause token is on              */
   /*  another line, but it will be counted as two clauses:                                           */
   /*           THEN DO  /@ begin a multi-line comment                                                */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  this turns out to be trickier than I thought, because a line could end with the start of       */
   /*  a multi-line comment, or even contain a comment in the middle of a statement.                  */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if \L_HasRexx(t, count) then return 0
   eoc = 0
   do c = L_FirstRexxWord(t, count) +1 to L_LastRexxWord(t, count)
      if t_type.c == 'REXX_EOC' then do
            eoc = eoc +1
         end
   end
   return eoc +1
   /* === End of L_ClauseCount ====================================================================== */

   L_DeleteLine: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        CALL L_DeleteLine t [, <count>                                                           */
   /*                                                                                                 */
   /*  deletes the specified line                                                                     */
   /*  if no <count> is given, deletes the line that token t is in                                    */
   /*  returns the number of tokens that were deleted                                                 */
   /*                                                                                                 */
   /*  ==> remember that all tokens beyond the deletion point will have new numbers                   */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   Last = L_LineEOL(t, count)

   HowMany = Last - L_PrevEOL(Last)
   where = L_FirstToken(Last)

   call T_DeleteToken where, HowMany
   return HowMany
   /* === End of L_DeleteLine ======================================================================= */

   L_FirstRexxToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_FirstRexxToken(t [,<count>])                                                        */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*       IF T_Val(L_FirstRexxToken(t)) == ',' THEN                                                 */
   /*                                                                                                 */
   /*  given token t, returns the number of the first REXX token in the line                          */
   /*  that is, Comment tokens are ignored                                                            */
   /*  <count> is relative to the line that t is in                                                   */
   /*                                                                                                 */
   /*  IMPORTANT:                                                                                     */
   /*    If the line has no Rexx tokens, this returns '', which may cause an error later.             */
   /*    You might want to check with L_HasRexx() first, or use a combination of                      */
   /*    R_NextToken() and L_LastRexxToken() to skip over comments and empty lines                    */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = L_LineEOL(t, count)
   do c = L_PrevEOL(p) to p by 1
      if t_class.c == 'REXX' then return c
   end c
   call ShowError 'BUG: line has no Rexx tokens, called from line 'Esigl
   call ShowError 'here is the context:', ''
   call ShowContext t
   call Fatality
   return ''
   /* === End of L_FirstRexxToken =================================================================== */

   L_FirstRexxWord: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_FirstRexxWord(t [,<count>])                                                         */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*       if T_Val(L_FirstRexxWord(t)) == 'THEN' then                                               */
   /*                                                                                                 */
   /*  given token t, returns the number of the first Rexx word in the specified line                 */
   /*  that is, Comment tokens and non-word Rexx tokens are ignored.                                  */
   /*  <count> is relative to the line that t is in                                                   */
   /*                                                                                                 */
   /*  IMPORTANT:                                                                                     */
   /*    If the line has no Rexx words, this returns '', which may cause an error later.              */
   /*    You might want to check with L_HasRexxWord() first, or use a combination of                  */
   /*    R_NextWord() and L_LastRexxWord() to skip over comments and empty lines                      */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = L_LineEOL(t, count)
   do c = L_PrevEOL(p) to p by 1
      if abbrev(t_type.c, 'REXX_W_') then return c
   end c
   call ShowError 'BUG: line has no Rexx words, called from line 'Esigl
   call ShowError 'here is the context:', ''
   call ShowContext t
   call Fatality
   return ''
   /* === End of L_FirstRexxWord ==================================================================== */

   L_FirstToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_FirstToken(t [,<count>])                                                            */
   /*                                                                                                 */
   /*  Given token t, returns the number of the first token of the line.                              */
   /*  <count> is relative to the line that t is in                                                   */
   /*                                                                                                 */
   /*  Use this only if you already know that the line is not empty                                   */
   /*  You can check for an empty line with the L_IsEmpty() function                                  */
   /*  (if the line is empty, this returns the ending EOL of the line)                                */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = L_LineEOL(t, count)
   do c = p -1 to 1 by -1
      if abbrev(t_class.c, 'FILE_') then return c +1
   end c
   return p
   /* === End of L_FirstToken ======================================================================= */

   L_HasComment: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_HasComment(t [,<count>])                                                            */
   /*                                                                                                 */
   /*  returns True if the specified line contains a comment                                          */
   /*  <count> is relative to the line that t is in                                                   */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = L_LineEOL(t, count)
   do c = L_PrevEOL(p) to p
      if t_class.c == 'COMMENT' then return 1
   end
   return 0
   /* === End of L_HasComment ======================================================================= */

   L_HasLabel: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_HasLabel(t [,<count>])                                                              */
   /*                                                                                                 */
   /*  returns True if the line contains a Label                                                      */
   /*  <count> is relative to the line that t is in                                                   */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = L_LineEOL(t, count)
   do c = L_PrevEOL(p) to p
      if t_type.c == 'REXX_LABEL' then return 1
   end
   return 0
   /* === End of L_HasLabel ========================================================================= */

   L_HasRexx: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_HasRexx(t [,<count>])                                                               */
   /*                                                                                                 */
   /*  returns True if the line contains any Rexx tokens (class REXX)                                 */
   /*  <count> is relative to the line that t is in                                                   */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = L_LineEOL(t, count)
   do c = L_PrevEOL(p) to p
      if t_class.c == 'REXX' then return 1
   end
   return 0
   /* === End of L_HasRexx ========================================================================== */

   L_HasRexxWord: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_HasRexxWord(t [,<count>])                                                           */
   /*                                                                                                 */
   /*  returns True if the line contains a Rexx word (type REXX_W_xxx)                                */
   /*  <count> is relative to the line that t is in                                                   */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = L_LineEOL(t, count)
   do c = L_PrevEOL(p) to p
      if abbrev(t_type.c, 'REXX_W') then return 1
   end
   return 0
   /* === End of L_HasRexxWord ====================================================================== */

   L_IsContinued: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_IsContinued(t [,<count>])                                                           */
   /*  Example:                                                                                       */
   /*       IF L_IsContinued(t, -1) THEN     /@ if previous line is continued @/                      */
   /*                                                                                                 */
   /*  returns True if the line is a continued line ending with a comma                               */
   /*  <count> is relative to the line that t is in                                                   */
   /* =============================================================================================== */
   parse arg t , count
   if count = '' then count = 0
   if \L_HasRexx(t, count) then return 0
   p = L_LineEOL(t, count)
   l = L_LastRexxToken(p)
   if t_type.l == 'CONTINUE' then return 1
   return 0
   /* === End of L_IsContinued ====================================================================== */

   L_IsEmpty: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_IsEmpty(t [,<count>])                                                               */
   /*                                                                                                 */
   /*  returns True if the specified line is empty                                                    */
   /*  <count> is relative to the line that t is in                                                   */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   t = L_LineEOL(t, count)
   /*  bugfix-- if at first token, pretend the previous line is not empty                             */
   if t <= 1 then return 0
   if abbrev(T_Class(t), 'FILE_') ,
    & abbrev(T_Class(t -1), 'FILE_') then
      return 1
   else
      return 0
   /* === End of L_IsEmpty ========================================================================== */

   L_IsFooter: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        ret = L_IsFooter(t[,<count>])                                                            */
   /*  where t is any token in the current line                                                       */
   /*                                                                                                 */
   /*  Returns True if the specified line is a footer comment.                                        */
   /*  Otherwise returns False.                                                                       */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count == '' then count = 0
   c = L_LineEOL(t, count) -2
   if c <= 1 then return 0
   if t_type.c \== 'COMMENT_VAL' then return 0
   return C_IsFooter(c)
   /* === End of L_IsFooter ========================================================================= */

   L_LastClause: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        r = L_LastClause(t [,<count>)                                                            */
   /*        if t == L_LastClause(t, +1) ...                                                          */
   /*                                                                                                 */
   /*  Finds the last clause of a line. That is, it returns the token number                          */
   /*  of the first word of the last clause of the specified line.                                    */
   /*                                                                                                 */
   /*  Returns '' if there is no Rexx clause, which could cause an error later,                       */
   /*  so check the return value or use L_HasRexxWord() first.                                        */
   /*  A label is not considered a clause.                                                            */
   /*                                                                                                 */
   /*  fix: this is inconsistent with the CL_XXX functions, which consider a label to be a clause     */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   if \L_HasRexxWord(t, count) then return ''
   do s = L_LastRexxWord(t, count) to L_LineEOL(t, count -1) by -1
      if t_type.s == 'REXX_EOC' then
         return s +1
   end
   return ''
   /* === End of L_LastClause ======================================================================= */

   L_LastRexxToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_LastRexxToken(t [,<count>])                                                         */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*       if T_Val(L_LastRexxToken(t)) == ',' then                                                  */
   /*                                                                                                 */
   /*  given token t, returns the number of the last Rexx token in the line.                          */
   /*  That is, Comment tokens are ignored                                                            */
   /*  <count> is relative to the line that t is in                                                   */
   /*  if line has no Rexx tokens it returns '', which may cause an error later                       */
   /*  you can check first with L_HasRexx()                                                           */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = L_LineEOL(t, count)
   do c = p to L_PrevEOL(p) by -1
      if t_class.c == 'REXX' then return c
   end c
   call ShowError 'BUG: line has no Rexx tokens, called from line 'Esigl
   call ShowError 'here is the context:', ''
   call ShowContext t
   call Fatality
   return ''
   /* === End of L_LastRexxToken ==================================================================== */

   L_LastRexxWord: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_LastRexxWord(t [,<count>])                                                          */
   /*                                                                                                 */
   /*  Examples:                                                                                      */
   /*       r = L_LastRexxWord(t, -1)                                                                 */
   /*       if T_Val(L_LastRexxWord(t)) == 'THEN' then                                                */
   /*                                                                                                 */
   /*  given token t, returns the number of the last REXX token in the line.                          */
   /*  that is, Comment tokens and non-word Rexx tokens are ignored                                   */
   /*  <count> is relative to the line that t is in                                                   */
   /*  if the line has no Rexx words it returns '', which may cause an error later                    */
   /*  You should check first with L_HasRexxWord()                                                    */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = L_LineEOL(t, count)
   do c = p to L_PrevEOL(p) by -1
      if abbrev(t_type.c, 'REXX_W_') then return c
   end c
   call ShowError 'BUG: line has no Rexx words, called from line 'Esigl
   call ShowError 'here is the context:', ''
   call ShowContext t
   call Fatality
   return ''
   /* === End of L_LastRexxWord ===================================================================== */

   L_LastToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_LastToken(t [,<count>])                                                             */
   /*                                                                                                 */
   /*  given token t, returns the number of the last token of the line, NOT counting                  */
   /*  the End-Of-Line token                                                                          */
   /*                                                                                                 */
   /*  <count> is relative to the line that t is in                                                   */
   /*                                                                                                 */
   /*  Use this only if you already know that the line is not empty; test with L_IsEmpty() first      */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = L_LineEOL(t, count)
   do c = p to t_type.0
      if abbrev(t_class.c, 'FILE_') then return c -1
   end c
   return p
   /* === End of L_LastToken ======================================================================== */

   L_LineEOL: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = L_LineEOL(t [, <count>])                                                              */
   /*                                                                                                 */
   /*  returns the token number that marks the end of a line (EOL)                                    */
   /*  if no <count> is given, the line that token t is in                                            */
   /*  if <count> is negative, the line is <count> lines above where token t is                       */
   /*  if <count> is positive, the line is <count> lines below where token t is                       */
   /* ----------------------------------------------------------------------------------------------- */
   /*  In all of the L_xxx() functions a line is defined as 0 or more tokens + EOL.                   */
   /*  A line may be identified by the token number of its ending EOL.                                */
   /*                                                                                                 */
   /*  L_LineEOL() is the basis for all of the L_xxx functions that take a <count>                    */
   /*  argument. They all call it directly or indirectly to get the true starting                     */
   /*  point.                                                                                         */
   /*                                                                                                 */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl

   p = 0
   c = 0
   select
      when count < 0 then do
            t = L_PrevEOL(t)
            do p = t to 1 by -1
               if abbrev(t_class.p, 'FILE_') then c = c +1
               if c == abs(count) then leave
               if p == 1 then leave
            end
         end
      when count >= 0 then
         do p = t to t_type.0 by 1
            if abbrev(t_class.p, 'FILE_') then c = c +1
            if c == count +1 then leave
         end
      otherwise nop
   end
   return p
   /* === End of L_LineEOL ========================================================================== */

   L_LineOf: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        IF L_LineOf(<this>, <that>)                                                              */
   /*        IF L_LineOf(t, u)                                                                        */
   /*        IF L_LineOf(t, t +1)                                                                     */
   /*                                                                                                 */
   /*  returns the line that <that> is in, relative to <this>                                         */
   /*  -2 means <that> is 2 lines before <this>                                                       */
   /*   0 means they are on the same line                                                             */
   /*  +1 means <that> is 1 line after <this>                                                         */
   /*                                                                                                 */
   /*  unlike other L_xxx functions, this does not take a <count> argument                            */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , u
   if debug.0arg then do
         call T_Valid t, Esigl
         call T_Valid u, Esigl
      end
   count = 0
   select
      when t > u then
         do c = t to u by -1
            if t_class.c == 'FILE_EOL' then count = count -1
         end c

      when t < u then do
            if t_class.u == 'FILE_EOL' then u = u -1
            do c = t to u by 1
               if t_class.c == 'FILE_EOL' then count = count +1
            end c
         end
      otherwise
         nop
   end   /*  select  */
   return count
   /* === End of L_LineOf =========================================================================== */

   L_MoveComments: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       call L_MoveComments(t, <count>)                                                           */
   /*  where t is any token in the line                                                               */
   /*  <count> is relative to the line of t                                                           */
   /*                                                                                                 */
   /*  Moves all comments in the line containing token t to the specified line                        */
   /*                                                                                                 */
   /*  Examples:                                                                                      */
   /*     call L_MoveComments(t, -1)                                                                  */
   /*        moves all comments to the previous line                                                  */
   /*     call L_MoveComments(t, 1)                                                                   */
   /*        moves all comments to the next line                                                      */
   /*                                                                                                 */
   /*  The moved comments will be to the right of any existing comments                               */
   /*                                                                                                 */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , tgt
   if debug.0arg then call T_Valid t, Esigl

   if \L_HasComment(t) then return 0
   /*  if the previous line has a comment, shorten it                                                 */
   if L_HasComment(t, tgt) then
      do s = L_FirstToken(t, tgt) to L_LastToken(t, tgt)
         if t_type.s == 'COMMENT_VAL' then call C_StripVal s, 'T'
      end

   s = L_FirstToken(t)
   do while s <= L_LastToken(t)
      if t_class.s == 'COMMENT' then do
            if t_type.s == 'COMMENT_BEG' then t_prefix.s = '   '
            if t_type.s == 'COMMENT_VAL' then call C_StripVal s, 'T'
            call T_MoveToken s, L_LastToken(t, tgt) +1
            t = t +1
         end
      else
         s = s +1
   end
   return 0
   /* === End of L_MoveComments ===================================================================== */

   L_OnSameLine: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       IF L_OnSameLine(t, u) THEN                                                                */
   /*                                                                                                 */
   /*  returns True if tokens t and u are on the same line                                            */
   /* =============================================================================================== */
   parse arg t1 , t2
   if t1 > t2 then inc = -1 ; else inc = 1
   do t = t1 to t2 by inc
      if abbrev(t_class.t, 'FILE') then return 0
   end
   return 1
   /* === End of L_OnSameLine ======================================================================= */

   L_PrevEOL: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        r = L_PrevEOL(t)                                                                         */
   /*                                                                                                 */
   /*  given token t, returns the EOL at the start of the line that t is in.                          */
   /*  Or more accurately, the EOL of the previous line.                                              */
   /*  If t is an EOL, returns the previous EOL                                                       */
   /*                                                                                                 */
   /*  this is equivalent to L_LineEOL(t, -1)                                                         */
   /*  Unlike other L_xxx functions, it does not take a <count> argument                              */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , arg2
   if arg2 \== '' then do
         call ShowError 'BUG: too many arguments to L_PrevEOL'
         call Mark 't arg2 Esigl', ''
         call Fatality
      end
   if debug.0arg then call T_Valid t, Esigl

   if abbrev(t_class.t, 'FILE_') then t = t -1
   do c = t to 1 by -1
      if abbrev(t_class.c, 'FILE_') then return c
   end c
   return 1
   /* === End of L_PrevEOL ========================================================================== */

   /*  O_   Other procedures                                                                   */ /*fold00*/
   O_InsertLine: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL O_InsertLine <t>                                                                     */
   /*                                                                                                 */
   /*  inserts a new line at (ie, before) token t                                                     */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*        CALL O_InsertLine t, 'DO a = 1 to 3'                                                     */
   /*    inserts                                                                                      */
   /*        DO a = 1 to 3                                                                            */
   /*    as one token at position t                                                                   */
   /*  the type will be REXX_W_LITERAL and the class will be OTHER                                    */
   /*                                                                                                 */
   /*  token t will normally be the first token of a line, so this will insert a                      */
   /*  new line of text before it.                                                                    */
   /*                                                                                                 */
   /*  This is for the convenience of tasks that need to edit a file by adding a line,                */
   /*  but do not need it broken up into separate tokens. This should not be used if it               */
   /*  will be followed by token-oriented tasks. Unless you are sure it will not do any harm.         */
   /* =============================================================================================== */
   parse arg t , val

   call T_InsertToken t, 2
   p = t -1
   t_class.t = 'OTHER'
   t_type.t = 'REXX_W_LITERAL'
   t_val.t = val
   if t_class.p \== 'FILE_EOL' then
      t_prefix.t = ' '
   else
      t_prefix.t = ''
   t = t +1
   t_class.t = 'FILE_EOL'
   t_type.t = 'REXX_EOC'
   t_val.t = ''
   t_prefix.t = ''
   return 0
   /* === End of O_InsertLine ======================================================================= */

   O_InsertToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL O_InsertToken <t>                                                                    */
   /*                                                                                                 */
   /*  inserts a new token at before token t                                                          */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*     a line says                                                                                 */
   /*        Label: procedure expose Frodo                                                            */
   /*  and t points to the End-Of-Line token                                                          */
   /*        CALL O_InsertToken t, 'Bilbo'                                                            */
   /*    changes it to                                                                                */
   /*        Label: procedure expose Frodo Bilbo                                                      */
   /*                                                                                                 */
   /*  the type will be REXX_W_LITERAL and the class will be OTHER                                    */
   /*  the new token will have a prefix of one space character                                        */
   /*                                                                                                 */
   /*  This is for the convenience of tasks that need to edit a file, but only the text               */
   /*  is important. This should generally not be used if it will be followed by                      */
   /*  token-oriented tasks. Unless you are sure it will not do any harm.                             */
   /* =============================================================================================== */
   parse arg t , val

   call T_InsertToken t, 1
   next = t +1
   t_class.t = 'OTHER'
   t_type.t = 'REXX_W_LITERAL'
   t_val.t = val
   t_prefix.t = ' '
   if t_class.next \== 'FILE_EOL' ,
    & t_prefix.next == '' then
      t_prefix.next = ' '
   return 0
   /* === End of O_InsertToken ====================================================================== */

   /* ----------------------------------------------------------------------------------------------- */
   /*  R_   Rexx-related procedures                                                            */ /*fold00*/
   /***************************************************************************************************/

   R_ArgNum: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*      if R_ArgNum(t) = ...                                                                       */
   /*                                                                                                 */
   /*  Returns the argument number of a function that token t is part of.                             */
   /*  For instance, in                                                                               */
   /*       x = value(y, 'fee'||'fie')                                                                */
   /*  or                                                                                             */
   /*       call value y, 'fee'||'fie'                                                                */
   /*  if token t is the "y", then                                                                    */
   /*       R_ArgNum(t) returns 1                                                                     */
   /*                                                                                                 */
   /*  if token t is the "'fee'", then                                                                */
   /*       R_ArgNum(t) returns 2                                                                     */
   /*                                                                                                 */
   /*  if token t is not an argument, returns 0                                                       */
   /*                                                                                                 */
   /*  This does not try to validate the input, so you might want to make sure that                   */
   /*  token t is (part of) one of the arguments to the function. (?)                                 */
   /* =============================================================================================== */
   parse arg t
   k = CL_FirstWord(t)
   c = 1
   s = t
   do while s >= k
      if t_type.s == 'REXX_W_COMMA' then
         c = c +1
      if t_type.s == 'REXX_W_(' then
         return c
      if t_type.s == 'REXX_W_)' then
         s = R_MatchPar(s)
      if s = k then
         if translate(t_val.s) == 'CALL' then
            return c
      s = s -1
   end
   return 0
   /* === End of R_ArgNum =========================================================================== */

   R_AtAnyAssignment: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        IF R_AtAnyAssignment(t) THEN                                                             */
   /*                                                                                                 */
   /*  returns True if token t is a variable being assigned a value, like A in                        */
   /*           A = B                                                                                 */
   /*           A = function()                                                                        */
   /*           A += B                                                                                */
   /*           DO A = 1 to 3                                                                         */
   /*           PARSE VAR Foo A B                                                                     */
   /*           PULL A                                                                                */
   /*           A.1 = B                                                                               */
   /*           ARG A                                                                                 */
   /*  in the case of a stem variable, token t may be any of the "A", ".", or "1" tokens              */
   /*                                                                                                 */
   /*  It does not return True for lines like                                                         */
   /*           IF A = B THEN                                                                         */
   /*           CALL VALUE 'varN', NewValue                                                           */
   /*                                                                                                 */
   /*  Do not confuse this with R_AtAssignment(t), which only tests for a simple                      */
   /*  assignment, not DO, ARG, or PARSE statements                                                   */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if t_type.t \== 'REXX_W_SYMBOL' then
      return 0

   if T_Type(R_NextWord(R_Tail(t))) == 'REXX_W_ASSIGN' then
      return 1

   if R_KeyVal(R_PrevWord(R_Stem(t))) == 'DO' ,
    & T_Val(R_NextWord(R_Tail(t))) == '=' then                                       /* do A = 1 to 3 */
      return 1

   w1 = CL_FirstWord(t)
   if R_KeyVal(w1) == 'ARG' then
      return 1                                                                               /* arg A */
   if wordpos(R_KeyVal(w1), 'PARSE PULL') > 0 then do
         do p = w1 to R_Stem(t)
            if wordpos(translate(t_val.p), 'ARG PULL SOURCE WITH VAR VERSION') > 0 then
               leave p
         end
         if translate(t_val.p) == 'VAR' then
            p = R_Tail(R_NextWord(p))
         p = R_NextWord(p)
         /*  p is the beginning of the template                                                       */
         if t >= p then do
               /*  if token t is in parentheses, it is being used, not assigned                       */
               if T_Type(t -1) \== 'REXX_W_(' then
                  return 1
               else
                  return 0
            end
         else return 0
      end



   /*  If t is not the start of the clause, then it is not a simple assignment.                       */
   if R_Stem(t) \== w1 then
      return 0                                                                           /* not A = B */
   /*  if not an assignment, it is probably an unquoted system command                                */
   return 0
   /* === End of R_AtAnyAssignment ================================================================== */

   R_AtAssignment: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        IF R_AtAssignment(t) THEN                                                                */
   /*                                                                                                 */
   /*  returns True if token t is a variable being assigned, like A in                                */
   /*           A = B                                                                                 */
   /*           A = function()                                                                        */
   /*           A += B                                                                                */
   /*           A.1 = B                                                                               */
   /*  in the case of a stem variable, token t may be any of the "A", ".", or "1" tokens              */
   /*                                                                                                 */
   /*  It does =not= return True for lines like                                                       */
   /*           DO A = 1 to 3                                                                         */
   /*           PARSE VAR Foo A B                                                                     */
   /*                                                                                                 */
   /*  It does =not= return True for lines like                                                       */
   /*           IF A = B THEN                                                                         */
   /*                                                                                                 */
   /*  Do not confuse this with R_AtAnyAssignment(t), which returns True for DO, ARG, and             */
   /*  PARSE instructions also.                                                                       */
   /* =============================================================================================== */
   parse arg t
   if t_type.t \== 'REXX_W_SYMBOL' then return 0
   op = R_NextWord(R_Tail(t))
   if t_type.op == 'REXX_W_ASSIGN' then return 1
   if t_type.op \= 'REXX_W_COMPARE' then
      if right(t_val.op, 1) == '=' then do
            call ShowError 'BUG: token' op 'should be type REXX_W_ASSIGN'
            call ShowError '     it is type' t_type.op, ''
            call ShowError '     it is "'t_val.op'" in line' t_line.op, ''
            call ShowError '     'Qline(t_line.op), ''
            call Fatality
            return 1
         end
   return 0
   /* === End of R_AtAssignment ===================================================================== */

   R_EndEOC: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        r = R_EndEOC(t)                                                                          */
   /*                                                                                                 */
   /*  returns the ending EOC of the clause that token t is in                                        */
   /*  if t is an EOC, returns t.                                                                     */
   /*                                                                                                 */
   /*  Do not think of this as "the next EOC after this EOC";                                         */
   /*  it means "the ending EOC of this clause".                                                      */
   /*                                                                                                 */
   /*  unlike CL_ClauseEOC(t), token t can be of any type                                             */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   do c = t to t_type.0 by 1
      if t_type.c == 'REXX_EOC' then leave
   end c
   return c
   /* === End of R_EndEOC =========================================================================== */

   R_EOCisOptional: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        r = R_EOCisOptional(t)                                                                   */
   /*                                                                                                 */
   /*  returns 1 if token t is an EOC and a written ";" is optional; that is,                         */
   /*  it can be an implied (hidden) EOC or an End-Of-Line token                                      */
   /*  returns 0 if token t must be a ";"                                                             */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if t_type.t \= 'REXX_EOC' then do
         call ShowError 'BUG: incorrect call to R_EOCisOptional from line' Esigl
         call ShowError '     token' t 'is type' t_type.t, ''
         call Fatality
      end
   if abbrev(t_class.t, 'FILE_') then return 1
   if T_Type(R_NextToken(t)) == 'REXX_EOC' then return 1
   if wordpos(translate(T_Val(t -1)), 'THEN ELSE OTHERWISE :') > 0 then return 1
   if translate(T_Val(t +1)) == 'THEN' then return 1
   return 0
   /* === End of R_EOCisOptional ==================================================================== */

   R_Infunction: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        FunctionName = R_Infunction(t)                                                           */
   /*  if token t is inside a function, returns the name of the function                              */
   /*  otherwise returns ''                                                                           */
   /*  the ( and ) are considered inside the function                                                 */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*        foo = translate(word('a b c', 1))                                                        */
   /*  if t_val.t == 'a b c', then returns "word"                                                     */
   /*  if t_val.t == 'word', then returns "translate"                                                 */
   /* =============================================================================================== */
   parse arg t

   level = -1
   if t_type.t == 'REXX_W_)' then t = t -1
   do i = t to 1 by -1
      select
         when level == 0 then
            return t_val.i
         when t_type.i == 'REXX_W_(' then
            level = level +1
         when t_type.i == 'REXX_W_)' then
            level = level -1
         when t_type.i == 'REXX_EOC' then
            leave i
         otherwise
            nop
      end   /*  select  */
   end i
   return ''
   /* === End of R_Infunction ======================================================================= */

   R_InsertEOL: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL R_InsertEOL <t>                                                                      */
   /*                                                                                                 */
   /*  inserts a new End-Of-Line at (ie, before) token t                                              */
   /*  to insert an EOL in a multiline comment, use C_InsertEOL                                       */
   /* =============================================================================================== */
   parse arg t
   ret = T_InsertToken(t)
   if ret \= 0 then return ret
   t_class.t = 'FILE_EOL'
   t_type.t = 'REXX_EOC'
   t_val.t = ''
   t_prefix.t = ''
   return  0
   /* === End of R_InsertEOL ======================================================================== */

   R_IsHexBin: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        IF R_IsHexBin(t) then ...                                                                */
   /*                                                                                                 */
   /*  returns True if token t is a constant in the form of a hex or binary string like               */
   /*        'FF'x                                                                                    */
   /*        '01010101'b                                                                              */
   /*  this is for formatting purposes; it does not check the string for valid characters             */
   /* =============================================================================================== */
   parse arg t
   if \abbrev(t_type.t, 'REXX_W_CONSTANT') then return 0
   if left(t_val.t, 1) \= '"' ,
    & left(t_val.t, 1) \= "'" then return 0
   if right(translate(t_val.t), 2) == "'B" then return 1
   if right(translate(t_val.t), 2) == "'X" then return 1
   if right(translate(t_val.t), 2) == '"B' then return 1
   if right(translate(t_val.t), 2) == '"X' then return 1
   return 0
   /* === End of R_IsHexBin ========================================================================= */

   R_KeyVal: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        IF R_KeyVal(t) == ...                                                                    */
   /*        where t is a token number                                                                */
   /*                                                                                                 */
   /*  Returns the value of a keyword token, but makes sure the token is really a                     */
   /*  keyword or sub-keyword, not a variable with the same name.                                     */
   /*  If the token is not a keyword, returns ''                                                      */
   /*  It =does= uppercase the returned value                                                         */
   /*                                                                                                 */
   /*  because                                                                                        */
   /*        IF R_KeyVal(t +1) == 'DO'                                                                */
   /*  is easier to type than                                                                         */
   /*        IF T_Type(t +1) == 'REXX_W_KEY' & translate(T_Val(t +1)) == 'DO'                         */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if abbrev(t_type.t, 'REXX_W_KEY') then
      return translate(t_val.t)
   return ''
   /* === End of R_KeyVal =========================================================================== */

   R_MatchELSE: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        i = R_MatchELSE(e)                                                                       */
   /*                                                                                                 */
   /*  returns the matching IF of an ELSE keyword                                                     */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then do
         call T_Valid t, Esigl
         if R_KeyVal(t) \== 'ELSE' then do
               call ShowError 'BUG: invalid call to R_MatchELSE from line' Esigl
               call Mark 't t_type.t t_val.t'
               call Fatality
            end
      end

   InIf = 0
   s = t
   do while s > 2
      s = s -1
      Tval = R_KeyVal(s)
      select
         when Tval == 'END' then
            s = R_OtherEnd(s)
         when Tval == 'ELSE' then
            InIf = InIf +1
         when Tval == 'IF' then do
               if InIf == 0 then
                  leave
               else
                  InIf = InIf -1
            end
         otherwise nop
      end
   end
   return s
   /* === End of R_MatchELSE ======================================================================== */

   R_MatchPar: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = R_MatchPar(t)                                                                         */
   /*                                                                                                 */
   /*  where t is a left or right parenthesis                                                         */
   /*  finds the matching parenthesis of token t                                                      */
   /*  if no match was found, returns t                                                               */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl

   select
      when t_type.t == 'REXX_W_(' then do
            t_start = t +1 ; t_end = t_type.0 ; step = 1
         end
      when t_type.t == 'REXX_W_)' then do
            t_start = t -1 ; t_end = 1 ; step = -1
         end
      otherwise
         call ShowError 'BUG: invalid call to R_MatchPar from line' Esigl
         call Mark 't t_type.t t_val.t'
         call Fatality
   end   /*  select  */

   level = 1
   do e = t_start to t_end by step
      select
         when t_type.e == 'REXX_W_(' then level = level + step
         when t_type.e == 'REXX_W_)' then level = level - step
         when t_type.e == 'REXX_EOC' then do
               call ShowError 'Warning: in R_MatchPar no matching parenthesis for token' t '"'t_val.t'"'
               return t
            end
         otherwise nop
      end
      if level = 0 then return e
   end e
   return t
   /* === End of R_MatchPar ========================================================================= */

   R_MoveClauseDown: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call R_MoveClauseDown t [,'newline']                                                     */
   /*  where token t is the first token of a clause                                                   */
   /*                                                                                                 */
   /*  Moves clause(s) from the end of a line to the beginning of the next line.                      */
   /*  if the second argument is 'newline', clause(s) will be moved to a new line.                    */
   /*  otherwise added before whatever is already on the next line.                                   */
   /*                                                                                                 */
   /*  Comments are not moved                                                                         */
   /*                                                                                                 */
   /*  Token t no longer has to be the last clause on the line.                                       */
   /*                                                                                                 */
   /*  Unlike MoveClauseUp(), this will never merge the right-side comments                           */
   /* =============================================================================================== */
   /*  the explanatory comments assume that token t is the DO in this:                                */
   /*           IF a <EOC> THEN <EOC> DO b = 1 to 5   /@ comment @/<EOL>                              */
   /*                 NOP                             /@ comment @/<EOL>                              */
   /*              END b                                                                              */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , Opt
   if translate(Opt) == 'NEWLINE' then
      newline = 1
   else
      newline = 0

   if \(t_type.t == 'REXX_W_KEY' ,
    | R_AtAssignment(t)) then do
         call ShowError 'BUG: Incorrect call to R_MoveClauseDown from line' Esigl
         call Fatality
      end

   /*  if newline, it's easy. Just change the EOC to EOL, making the last clause a new                */
   /*  line. A right-side comment will be carried to the new line, which is probably                  */
   /*  not what user wants, so move the comment back up.                                              */
   /*  fix: could probably also do this if there are no comments                                      */
   if newline then do
         eoc = R_PrevEOC(t)
         call T_Class eoc, 'FILE_EOL'
         call T_Val eoc, ''
         call T_Prefix eoc, ''
         call T_Type eoc, 'REXX_EOC'
         call L_MoveComments t, -1
         return 0
      end
   /*  otherwise we need to move the clause one word at a time to the beginning of the                */
   /*  next line, ignoring intervening comments.                                                      */
   /*  locate the End-of-Line and End-of-Clause tokens                                                */
   /*  eoc is the hidden EOC between THEN and DO                                                      */
   /*  eol is the EOL before the next Rexx word, skipping over any comments and empty lines           */
   eol = R_NextWord(L_LineEOL(t)) -1
   eoc = R_PrevEOC(t)

   /*  indenting prefix of NOP no longer applies; we want "; NOP"                                     */
   call T_Prefix R_NextWord(eol), ' '

   /*  the EOC before DO will not be needed; reuse it by moving it before NOP                         */
   /*  moving is faster than deleting and inserting tokens                                            */
   s = eoc
   call T_MoveToken eoc, eol
   /*  note that everything between the old and new positions has now moved                           */
   /*  one position to the left. Update locations of EOC and EOL.                                     */
   eoc = eol
   eol = eol -1
   /*  the second line is now                                                                         */
   /*        <EOC> NOP                                                                                */

   /*  move the statement after the comment                                                           */
   /*  to watch the lines changing, insert a "call msg L_AsString(s)" line below                      */
   do while t_class.s = 'REXX'
      call T_MoveToken s, eol
      /*   call msg L_AsString(s)                                                                     */
      /*   call msg L_AsString(s, +1)                                                                 */
   end
   /*  the second line is now                                                                         */
   /*        DO b = 1 to 5 <EOC> NOP                                                                  */

   /*  decide if the EOC should be a visible ';'                                                      */
   if R_EOCisOptional(eoc) then
      t_val.eoc = ''
   else
      t_val.eoc = ';'
   /*  the second line is now                                                                         */
   /*        DO b = 1 to 5; NOP                                                                       */
   return 0
   /* === End of R_MoveClauseDown =================================================================== */

   R_MoveClauseUp: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call R_MoveClauseUp t                                                                    */
   /*  where t is the first token of the clause, and first token of the line                          */
   /*                                                                                                 */
   /*     the clause at the beginning of a line is in the form                                        */
   /*              <EOL> Clause <EOC>                                                                 */
   /*     at the end of a line is                                                                     */
   /*              <EOC> Clause <EOL>                                                                 */
   /*                                                                                                 */
   /* =============================================================================================== */
   Esigl = sigl
   /*  t is the keyword we will move                                                                  */
   parse arg t

   if t_type.t \= 'REXX_W_KEY' ,
    | t \= L_FirstRexxWord(t) then do
         call ShowError 'BUG: Incorrect call to R_MoveClauseUp from line 'Esigl
         t_firstword = L_FirstRexxWord(t)
         call ShowError 't == 't 't_type.t ==' t_type.t 'first word ==' t_val.t_firstword, ''
         call Mark 't t_type.t t_val.t t_firstword t_type.t_firstword t_val.t_firstword'
         call ShowContext t, 2
         call Fatality
      end

   if L_ClauseCount(t) == 1 ,
    & L_HasComment(t) then do
         /*  If the clause to move is the only one on the line, move the comment up also.             */
         /*  this is what the user probably wants, but it also avoids leaving the comment on          */
         /*  a line by itself, which would require inserting a new FILE_EOC token.                    */
         /*  If there were no comment after previous line, it would be a simple matter of             */
         /*  swapping the attributes of the EOC and EOL tokens. But we must assume there              */
         /*  is a comment.                                                                            */

         /*  we need a landmark token that will not change as we move tokens.                         */
         t_ref = R_PrevWord(t)
         call L_MoveComments t, L_LineOf(t, R_PrevWord(t))
         /*  locate our keyword again because the numbering may have changed:                         */
         t = R_NextWord(t_ref)
      end

   /*  change  IF a <EOL> /@ comment1 @/ THEN /@ comment2 @/ <EOC> DO <EOL>                           */
   /*  to      IF a <EOC> THEN /@ comment1 @/ /@ comment2 @/ <EOL> DO <EOL>                           */
   /*  move the End-Of-Line token of the upper line to after the THEN and remember the                */
   /*  new position. Do not change it to an EOC yet, because that will change the line count.         */
   s_eoc = R_EndEOC(t)
   tgt = L_LastRexxWord(R_PrevWord(t)) +1

   /*  the first and last words of the statement to move                                              */
   t_first = t
   t_last =  CL_LastWord(t)

   /*     move EOC after THEN to previous line before THEN                                            */
   call T_MoveToken s_eoc, tgt
   call T_Class tgt, 'REXX'
   call T_Type tgt, 'REXX_EOC'
   call T_Val tgt, ''
   call T_Prefix tgt, ''
   /*  update pointers                                                                                */
   t = t +1
   tgt = tgt +1
   t_first = t_first +1
   t_last = t_last +1

   /*  in the new line we want only one space before THEN                                             */
   t_prefix.t = ' '

   /*  move each token of the clause to its new position                                              */
   do src = t_first to t_last
      call T_MoveToken src, tgt
      tgt = tgt +1
   end
   return 0
   /* === End of R_MoveClauseUp ===================================================================== */

   R_NextLabel: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = R_NextLabel(t)                                                                        */
   /*  where t is any token                                                                           */
   /*                                                                                                 */
   /*  returns the token number of the next label, If none, returns the token number of EOF           */
   /* =============================================================================================== */
   parse arg t
   do s = t +1 to t_type.0 -1
      if t_type.s == 'REXX_LABEL' then
         return s
   end s
   return t_type.0
   /* === End of R_NextLabel ======================================================================== */

   R_NextToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = R_NextToken(t)                                                                        */
   /*                                                                                                 */
   /*  finds the next Rexx =token=                                                                    */
   /*  it ignores the "," line continuation character and comments                                    */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   do c = t +1 to t_type.0
      if abbrev(t_type.c, 'REXX_') then return c
   end c
   return t_type.0
   /* === End of R_NextToken ======================================================================== */

   R_NextWord: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = R_NextWord(t)                                                                         */
   /*                                                                                                 */
   /*  finds the next Rexx =word=                                                                     */
   /*  it ignores Labels, End-Of-Clause, End-Of-Line, Continuation, and comments                      */
   /*  if there is no next word, returns t_type.0 (the ending FILE_EOF token)                         */
   /*                                                                                                 */
   /*  this is the same as R_Word(t, +1)                                                              */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   do c = t +1 to t_type.0
      if abbrev(t_type.c, 'REXX_W_') then return c
   end c
   return t_type.0
   /* === End of R_NextWord ========================================================================= */

   R_OtherEnd: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = R_OtherEnd(t)                                                                         */
   /*                                                                                                 */
   /*  if token t is DO or SELECT, this finds the matching END                                        */
   /*  if token t is END, this finds the matching DO or SELECT                                        */
   /*  if token t is OTHERWISE, this finds the END that closes both SELECT and OTHERWISE.             */
   /* =============================================================================================== */
   /*  this expects valid code. To check for unbalanced DO/END, use MatchDoEnd()                      */
   /* ----------------------------------------------------------------------------------------------- */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   Tval = translate(t_val.t)
   select
      when t_type.t == 'REXX_W_KEY' ,
         & Tval = 'END' then do
            t_start = t -1 ; t_end = 1 ; step = -1
         end
      when t_type.t == 'REXX_W_KEY' ,
         & wordpos(Tval, 'DO SELECT OTHERWISE') > 0 then do
            t_start = t +1 ; t_end = t_type.0 ; step = 1
         end
      otherwise
         call ShowError 'BUG: invalid call to R_OtherEnd from line' Esigl
         call Mark 't t_type.t t_val.t'
         call Fatality
   end
   level = 1
   do e = t_start to t_end by step
      Tval = translate(t_val.e)
      select
         when t_type.e \= 'REXX_W_KEY' then nop
         when Tval = 'DO' then level = level + step
         when Tval = 'SELECT' then level = level + step
         when Tval = 'END' then level = level - step
         otherwise nop
      end
      if level = 0 then return e
   end e
   call ShowError 'Warning: in R_OtherEnd, failed to find other end'
   call ShowError 'of "'t_val.t'" at line' t_line.t, ''
   return t
   /* === End of R_OtherEnd ========================================================================= */

   R_PrevEOC: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        r = R_PrevEOC(t)                                                                         */
   /*                                                                                                 */
   /*  given token t, returns the EOC at the start of the clause that t is in.                        */
   /*  Or more accurately, the EOC of the previous clause.                                            */
   /*  If t is an EOC, returns the previous EOC                                                       */
   /*                                                                                                 */
   /*  Note: this is used only by the CL_ functions. Probably what you want is                        */
   /*  CL_ClauseEOC(t, -1), which ignores empty clauses and empty lines                               */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl

   do c = t -1 to 2 by -1
      if t_type.c == 'REXX_EOC' then leave
   end c
   return c
   /* === End of R_PrevEOC ========================================================================== */

   R_PrevLabel: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = R_PrevLabel(t)                                                                        */
   /*  where t is any token                                                                           */
   /*                                                                                                 */
   /*  returns the token number of the previous label, If none, returns the BOF token                 */
   /* =============================================================================================== */
   parse arg t
   do s = t -1 to 2 by -1
      if t_type.s == 'REXX_LABEL' then
         return s
   end s
   return 1
   /* === End of R_PrevLabel ======================================================================== */

   R_PrevToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = R_PrevToken(t)                                                                        */
   /*                                                                                                 */
   /*  finds the previous Rexx =token=                                                                */
   /*  it ignores the "," line continuation character and comments                                    */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   do c = t -1 to 1 by -1
      if abbrev(t_type.c, 'REXX_') then return c
   end c
   return 1
   /* === End of R_PrevToken ======================================================================== */

   R_PrevWord: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = R_PrevWord(t)                                                                         */
   /*                                                                                                 */
   /*  finds the previous Rexx =word=                                                                 */
   /*  it ignores Labels, End-Of-Clause, End-Of-Line, Continuation, and comments                      */
   /*  if there is no previous word, returns 1 (the starting FILE_BOF token)                          */
   /*                                                                                                 */
   /*  this is the same as R_Word(t, -1)                                                              */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   do c = t -1 to 1 by -1
      if abbrev(t_type.c, 'REXX_W_') then return c
   end c
   return 1
   /* === End of R_PrevWord ========================================================================= */

   R_StartOfTemplate: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*          x = StartOfTemplate(t)                                                                 */
   /*     where t is the PARSE keyword                                                                */
   /*                                                                                                 */
   /*  returns the first word of the template in a PARSE instruction                                  */
   /*                                                                                                 */
   /*  this exists for the sake of Toker_AtFunction(), but it might be useful in the main program     */
   /* =============================================================================================== */
   parse arg t
   if translate(t_val.t) \= 'PARSE' then do
         call ShowError 'BUG: invalid argument' t '('t_val.t') to StartOfTemplate'
         call Fatality
         return t
      end

   i = R_NextToken(t)
   if wordpos(translate(t_val.i), 'UPPER LOWER CASELESS') > 0 then do
         i = R_NextToken(i)
      end
   select
      when wordpos(translate(t_val.i), 'ARG PULL SOURCE VAR VERSION') > 0 then do
            i = R_NextToken(R_NextToken(i))
            return i
         end
      when translate(t_val.i) = 'LINEIN' then do
            i = R_NextToken(i)                                                    /* left parenthesis */
            if t_type.i == 'REXX_W_(' then do
                  i = R_MatchPar(i)                                              /* right parenthesis */
                  i = R_NextToken(i)
               end
            return i
         end
      when translate(t_val.i) = 'VALUE' then do
            do forever
               i = R_NextToken(i)
               if translate(t_val.i) = 'WITH' then do
                     i = R_NextToken(i)
                     return i
                  end
               /*  avoid endless loop if "WITH" is missing                                            */
               if t_type.i == 'REXX_EOC' then do
                     if Opt.0verbose then do
                           call Msg '[StartOfTemplate] PARSE VALUE without WITH at line 't_line.t
                           call Msg '  ' Qline(t_line.t)
                        end
                     return i
                  end
            end
         end
      otherwise
         if Opt.0verbose then do
               call Msg '[StartOfTemplate] parsing error of PARSE, at line 't_line.t
               call Msg '   unknown sub-keyword' '"'t_val.i'"'
               call Msg '  ' Qline(t_line.i)
            end
   end   /*  select  */
   return i
   /* === End of R_StartOfTemplate ================================================================== */

   R_Stem: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     r = R_Stem(t)                                                                               */
   /*                                                                                                 */
   /*  returns the leftmost part of a stem variable that token t is part of.                          */
   /*  If token t is a simple variable, returns t                                                     */
   /*  see also R_Tail, S_CompoundVar                                                                 */
   /*                                                                                                 */
   /*  R_Stem and R_Tail can be used to skip over elements of a stem variable, as in                  */
   /*        do while t < t_type.0                                                                    */
   /*           t = t +1                                                                              */
   /*           t = R_Tail(t)                                                                         */
   /*        end                                                                                      */
   /* =============================================================================================== */
   parse arg t
   if t_type.t \== 'REXX_W_SYMBOL' then return t
   do c = t to 2 by -1
      if t_type.c \== 'REXX_W_SYMBOL' then return c +1
      if t_prefix.c \== '' then return c
   end c
   return t
   /* === End of R_Stem ============================================================================= */

   R_Tail: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     r = R_Tail(t)                                                                               */
   /*                                                                                                 */
   /*  returns the rightmost part of a stem variable that token t is part of.                         */
   /*  If token t is a simple variable, returns t                                                     */
   /*  see also R_Stem, S_CompoundVar                                                                 */
   /* =============================================================================================== */
   parse arg t
   if t_type.t \== 'REXX_W_SYMBOL' then return t
   do c = t to t_type.0 -1 by 1
      if t_type.c \== 'REXX_W_SYMBOL' then return c -1
      n = c +1
      if t_prefix.n \== '' then return c
   end c
   return t
   /* === End of R_Tail ============================================================================= */

   R_Word: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = R_Word(t, <count>)                                                                    */
   /*  where token t is a Rexx word                                                                   */
   /*                                                                                                 */
   /*  finds the Rexx =word= that is <count> words from token t                                       */
   /*  Examples:                                                                                      */
   /*           r = R_Word(t, 1)                                                                      */
   /*     returns the next Rexx word after token t                                                    */
   /*           r = R_Word(t, -2)                                                                     */
   /*     returns the Rexx word two words before token t                                              */
   /*                                                                                                 */
   /*  it ignores labels, End-Of-Clause, End-Of-Line, Continuation, and comments                      */
   /*  if the specified Rexx word is not found, returns 1 (the FILE_BOF token) or                     */
   /*  t_type.0 (the FILE_EOF token)                                                                  */
   /* =============================================================================================== */
   /*  not currently used, not sure what I wanted it for                                              */
   Esigl = sigl
   parse arg t , count
   if count = '' then count = 0
   if debug.0arg then call T_Valid t, Esigl
   if \abbrev(t_type.t, 'REXX_W') then do
         call ShowError 'BUG: in R_Word, token t is not a Rexx word'
         call Mark 'Esigl t_type.t t_val.t'
         call Fatality
      end
   w = 0
   select
      when count > 0 then do
            do c = t +1 to t_type.0
               if abbrev(t_type.c, 'REXX_W_') then
                  w = w +1
               if w == count then
                  return c
            end c
            return t_type.0
         end   /*  when  */
      when count < 0 then do
            do c = t -1 to 2 by -1
               if abbrev(t_type.c, 'REXX_W_') then
                  w = w -1
               if w == count then
                  return c
            end c
            return 1
         end   /*  when  */
      otherwise
         return t
   end   /*  select  */
   /* === End of R_Word ============================================================================= */

   /*  S_   string-related procedures                                                          */ /*fold00*/
   /***************************************************************************************************/
   S_CompoundVar: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     string = S_CompoundVar(t)                                                                   */
   /*                                                                                                 */
   /*  returns a stem variable as a string. That is,                                                  */
   /*        Foo.A                                                                                    */
   /*  will be tokenized as three tokens "Foo" "." "A"                                                */
   /*  if token t is any of those tokens, S_CompoundVar(t) will return a single string "Foo.A"        */
   /* =============================================================================================== */
   parse arg t
   str = ''
   if t_type.t \== 'REXX_W_SYMBOL' then return ''
   do c = R_Stem(t) to R_Tail(t)
      str = str||t_val.c
   end c
   return str
   /* === End of S_CompoundVar ====================================================================== */

   S_DividerType: procedure expose (Shared_vars) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        dtype = S_DividerType(<string>)                                                          */
   /*        dtype = S_DividerType(t_val.t)                                                           */
   /*                                                                                                 */
   /*  returns the type of divider that a string is:                                                  */
   /*        '#' for '#############################################'                                  */
   /*        '*' for '*********************************************'                                  */
   /*        '=' for '============================================='                                  */
   /*        '-' for '---------------------------------------------'                                  */
   /*        '_' for '_____________________________________________'                                  */
   /*        'F' for a "End of <routine>" footer                                                      */
   /*        ''  if not a divider                                                                     */
   /*                                                                                                 */
   /*  the decision is now based on the last 30 characters, so the beginning of the                   */
   /*  divider can contain text as well as divider characters                                         */
   /*                                                                                                 */
   /*  note that this operates on any string; the argument is not a token number                      */
   /*  This will be the basis of other divider functions                                              */
   /* =============================================================================================== */
   parse arg str

   if abbrev(translate(str), translate(G.0footerID)) then return 'F'
   Div_chars = '*=-_#'                                           /* only these characters are allowed */
   len = length(str)
   str = StripStr(str)
   if length(str) +2 < len then return ''                      /* more than one blank at each end: No */
   if length(str) < 20 then return ''                                                /* too short, No */
   if verify(right(str, 30), Div_chars) > 0 then return ''      /* has other chars than Div_chars, No */
   c1 = right(str, 1)                                         /* compare last char to rest of string; */
   if verify(right(str, 30), c1) > 0 then return ''              /* if not all the same character, No */
   return c1
   /* === End of S_DividerType ====================================================================== */

   S_IsQuoted: procedure expose (Shared_vars) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       IF S_IsQuoted('foo') then ...                                                             */
   /*                                                                                                 */
   /*  returns 1 if a string is enclosed in single or double quotes                                   */
   /* =============================================================================================== */
   parse arg str
   if (left(str, 1) == "'" ,
    | left(str, 1) == '"') ,
    & left(str, 1) == right(str, 1) then
      return 1
   return 0
   /* === End of S_IsQuoted ========================================================================= */

   S_UnComment: procedure expose (Shared_vars) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       string = S_UnComment <string>                                                             */
   /*                                                                                                 */
   /*  strips the beginning /@ and ending @/ from a comment.                                          */
   /*  it does not strip spaces from the remaining text                                               */
   /* =============================================================================================== */
   parse arg str
   /*  only need one call to StripStr() because comment Begin and End use the same characters         */
   if left(strip(str), 2) == '/'||'*' then
      str = StripStr(strip(str), 'B', '/'||'*')
   return str
   /* === End of S_UnComment ======================================================================== */

   /*  T_   token-related procedures                                                           */ /*fold00*/
   /***************************************************************************************************/
   T_AsString: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        string = T_AsString(T1, T2)                                                              */
   /*                                                                                                 */
   /*  returns a string composed of all prefixes and values of tokens <T1> through <T2>               */
   /*  the calling routine might want to trim the leading blanks                                      */
   /* =============================================================================================== */
   parse arg t_from , t_to
   if t_to == '' then t_to = t_from
   outstring = ''
   do i = t_from to t_to
      outstring = outstring||t_prefix.i||t_val.i
   end
   return outstring
   /* === End of T_AsString ========================================================================= */

   T_Class: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       string = T_Class(<t>, [<new class>])                                                      */
   /*       string = T_Class(t)                                                                       */
   /*       CALL T_Class t, 'COMMENT'                                                                 */
   /*                                                                                                 */
   /*  returns the class of token t                                                                   */
   /*  if there is a second argument, the class will be changed to it first                           */
   /* =============================================================================================== */
   Esigl = sigl
   t = arg(1)
   if debug.0arg then call T_Valid t, Esigl
   if arg(2, 'e') then t_class.t = arg(2)
   return t_class.t
   /* === End of T_Class ============================================================================ */

   T_Col: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       string = T_Col(<t>, [<new column>])                                                       */
   /*       string = T_Col(t)                                                                         */
   /*       CALL T_Col t, '10'                                                                        */
   /*                                                                                                 */
   /*  returns the column of token t                                                                  */
   /*  if there is a second argument, the t_col.t will be changed to it first                         */
   /*  But changing the column of a token makes no sense                                              */
   /* =============================================================================================== */
   Esigl = sigl
   t = arg(1)
   if debug.0arg then call T_Valid t, Esigl
   if arg(2, 'e') then t_col.t = arg(2)
   return t_col.t
   /* === End of T_Col ============================================================================== */

   T_DeleteToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL T_DeleteToken <where> [, <how many>]                                                 */
   /*                                                                                                 */
   /*  Deletes <how many> new elements at position <where> in all of the t_xxx arrays.                */
   /*  Tasks should call this instead of changing the arrays themselves.                              */
   /* =============================================================================================== */
   /*  this deletes element(s) in each of the arrays listed in G.0token_vars. If I add a new          */
   /*  token array or change the name of one, nothing here needs to be rewritten                      */
   /* ----------------------------------------------------------------------------------------------- */
   Esigl = sigl
   parse arg t , HowMany
   if HowMany == '' then HowMany = 1
   if debug.0arg then do
         call T_Valid t, Esigl
         if \datatype(HowMany, 'w') then do
               call ShowError 'In T_DeleteToken, invalid argument: "'HowMany'"'
               call Fatality
            end
      end

   Tlist = G.0token_vars
   do while Tlist \= ''
      parse var Tlist AttrName Tlist
      !vName = value('AttrName')
      call StemDelete !vName, t, HowMany
   end   /*  of tlist  */
   return 0
   /* === End of T_DeleteToken ====================================================================== */

   T_FloatToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        CALL T_FloatToken t, column                                                              */
   /*        where t is the token number and column is the column number to move token t to           */
   /*                                                                                                 */
   /*  Moves a token to the given column, or as close as possible, by changing its prefix.            */
   /*  Token t will usually be the beginning of a comment, but this could be used for other tokens.   */
   /*  Note that the column count does not include the left margin.                                   */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , where
   if debug.0arg then do
         call T_Valid t, Esigl
         if \datatype(where, 'w') then do
               call ShowError 'In T_FloatToken, invalid argument: "'where'"'
               call Fatality
            end
      end

   beginT = T_TokenColumn(t)
   if where \> beginT then
      where = beginT +2
   t_prefix.t = copies(' ', where - beginT)
   return 0
   /* === End of T_FloatToken ======================================================================= */

   T_Indent: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = T_Indent(<t>, [new indent])                                                           */
   /*       r = T_Indent(t)                                                                           */
   /*       CALL T_Indent t, 1                                                                        */
   /*                                                                                                 */
   /*  returns the Indent value of token t                                                            */
   /*  if there is a second argument, the Indent will be changed to it first                          */
   /* =============================================================================================== */
   Esigl = sigl
   t = arg(1)
   if debug.0arg then call T_Valid t, Esigl
   if arg(2, 'e') then t_indent.t = arg(2)
   return t_indent.t
   /* === End of T_Indent =========================================================================== */

   T_InsertToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL T_InsertToken <where> [, <how many>]                                                 */
   /*                                                                                                 */
   /*  inserts <how many> new elements at position <where> in all of the t_xxx arrays.                */
   /*  Tasks should call this instead of changing the arrays themselves.                              */
   /*  Example:                                                                                       */
   /*          /@ insert a new token, then fill in the class, type, and value @/                      */
   /*          call T_InsertToken t , 1                                                               */
   /*          t_class.t = 'REXX'                                                                     */
   /*          t_type.t = 'REXX_EOC'                                                                  */
   /*          t_val.t = ';'                                                                          */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg t , HowMany
   if HowMany == '' then HowMany = 1
   if debug.0arg then do
         call T_Valid t, Esigl
         if \datatype(HowMany, 'w') then do
               call ShowError 'In T_InsertToken, invalid argument(s): "'t'"  "'HowMany'"'
               call Fatality
            end
      end

   Tlist = G.0token_vars
   do while Tlist \= ''
      parse var Tlist AttrName Tlist
      !vName = value('AttrName')
      call StemInsert !vName, t, HowMany
   end
   return 0
   /* === End of T_InsertToken ====================================================================== */

   T_InsertContinuation: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     call T_InsertContinuation t                                                                 */
   /*                                                                                                 */
   /*  inserts a "," and EOL at token t to create a continued line                                    */
   /* =============================================================================================== */
   parse arg t
   call T_InsertToken t, 2
   t_class.t = 'REXX'
   t_type.t = 'CONTINUE'
   t_val.t = ','
   t_prefix.t = ' '
   call T_Class t +1, 'FILE_EOL'
   call T_Type t +1, 'NULL'
   call T_Val t +1, ''
   call T_Prefix t +1, ''
   return 0
   /* === End of T_InsertContinuation =============================================================== */

   T_Line: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       string = T_Line(<t>, [<new line>])                                                        */
   /*       string = T_Line(t)                                                                        */
   /*       CALL T_Line t, '10'                                                                       */
   /*                                                                                                 */
   /*  returns the line of token t                                                                    */
   /*  if there is a second argument, the t_line.t will be changed to it first                        */
   /*  But changing the line of a token makes no sense                                                */
   /* =============================================================================================== */
   Esigl = sigl
   t = arg(1)
   if debug.0arg then call T_Valid t, Esigl
   if arg(2, 'e') then t_line.t = arg(2)
   return t_line.t
   /* === End of T_Line ============================================================================= */

   T_MoveToken: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL T_MoveToken <old> , <new>                                                            */
   /*                                                                                                 */
   /*  moves a token from position <old> to position <new> (in all of the token arrays)               */
   /*                                                                                                 */
   /*  The logic of this is perhaps not intuitive-- if the tokens are 1 2 3 4 and you want to         */
   /*  move 1 to be before 4, you would use                                                           */
   /*        call T_MoveToken 1, 3                                                                    */
   /* =============================================================================================== */
   parse arg t_from , t_to
   if t_from > t_to then
      incr = -1
   else
      incr = 1

   /*  save to a temp var                                                                             */
   Tlist = G.0token_vars
   do while Tlist \= ''
      parse var Tlist AttrName Tlist
      !vName = value('AttrName')
      /*  some arrays, like t_indent, may not be filled yet                                           */
      if t_to > value(!vName||0) +1 then iterate
      call value 'tmp_'||!vName||t_from, value(!vName||t_from)
   end

   /*  shift everything between From and To                                                           */
   do t = t_from to t_to - incr by incr
      n = t + incr
      Tlist = G.0token_vars
      do while Tlist \= ''
         parse var Tlist AttrName Tlist
         !vName = value('AttrName')
         if t > value(!vName||0) +1 then do
               iterate
            end
         call value !vName||t, value(!vName||n)
      end
   end

   /*  restore to new position from temp var                                                          */
   Tlist = G.0token_vars
   do while Tlist \= ''
      parse var Tlist AttrName Tlist
      !vName = value('AttrName')
      if t_to > value(!vName||0) +1 then iterate
      call value !vName||t_to, value('tmp_'||!vName||t_from)
   end
   return 0
   /* === End of T_MoveToken ======================================================================== */

   T_Prefix: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       string = T_Prefix(<t>, [<new prefix>])                                                    */
   /*       string = T_Prefix(t)                                                                      */
   /*        CALL T_Prefix t, ' '                                                                     */
   /*                                                                                                 */
   /*  returns the prefix of token t                                                                  */
   /*  if there is a second argument, the prefix will be changed to it first                          */
   /* =============================================================================================== */
   Esigl = sigl
   t = arg(1)
   if debug.0arg then call T_Valid t, Esigl
   if arg(2, 'e') then t_prefix.t = arg(2)
   return t_prefix.t
   /* === End of T_Prefix =========================================================================== */

   T_ReadOnly: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = T_ReadOnly(<token>, [new value])                                                      */
   /*       r = T_ReadOnly(t)                                                                         */
   /*       CALL T_ReadOnly t, 1                                                                      */
   /*                                                                                                 */
   /*  returns the ReadOnly status of token <n>                                                       */
   /*  if there is a second argument, the ReadOnly flag will be changed to it first                   */
   /* =============================================================================================== */
   Esigl = sigl
   t = arg(1)
   if debug.0arg then call T_Valid t, Esigl
   if arg(2, 'e') then t_readonly.t = arg(2)
   return t_readonly.t
   /* === End of T_ReadOnly ========================================================================= */

   T_TokenColumn: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = T_TokenColumn(t)                                                                      */
   /*                                                                                                 */
   /*  given token t, returns the column in the line where token t would start.                       */
   /*  Or more accurately, where the prefix of token t would start.                                   */
   /*  Or more usefully, it is the next column after the end of the previous token.                   */
   /* =============================================================================================== */
   /*  The lines do not exist yet, so it does not even know what the line number is, but              */
   /*  ==>the indenting must have already been calculated                                             */
   /*  this was written for the comment formatter so it can resize Right-side comments.               */
   /* ----------------------------------------------------------------------------------------------- */
   Esigl = sigl
   parse arg t
   if debug.0arg then call T_Valid t, Esigl
   /*  p is relative to the first text column. that is, p = Opt.0margin +1                            */
   p = 1
   word1 = L_FirstToken(t)
   /*  for the first token, use the indent value because the new indent may not have                  */
   /*  been applied yet                                                                               */
   p = p + format(t_indent.word1 * Opt.0Tab, , 0) + length(t_val.word1)
   do c = word1 +1 to t -1
      p = p + length(t_prefix.c) + length(t_val.c)
   end
   return p
   /* === End of T_TokenColumn ====================================================================== */

   T_Type: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       string = T_Type(<t>, [<new type>])                                                        */
   /*       r = T_Type(t)                                                                             */
   /*       CALL T_Type t, 'COMMENT_VAL'                                                              */
   /*                                                                                                 */
   /*  returns the type of token <n>                                                                  */
   /*  if there is a second argument, the type will be changed to it first                            */
   /* =============================================================================================== */
   Esigl = sigl
   t = arg(1)
   if debug.0arg then call T_Valid t, Esigl
   if arg(2, 'e') then t_type.t = arg(2)
   return t_type.t
   /* === End of T_Type ============================================================================= */

   T_Val: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       string = T_Val(<t>, [<new value>])                                                        */
   /*       string = T_Val(t)                                                                         */
   /*       CALL T_Val t, 'this is a comment'                                                         */
   /*                                                                                                 */
   /*  returns the value of token t                                                                   */
   /*  if there is a second argument, the value will be changed to it first                           */
   /* =============================================================================================== */
   Esigl = sigl
   t = arg(1)
   if debug.0arg then call T_Valid t, Esigl
   if arg(2, 'e') then t_val.t = arg(2)
   return t_val.t
   /* === End of T_Val ============================================================================== */

   T_Valid: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call T_Valid t, Esigl                                                                    */
   /*        if T_Valid(t) then ...                                                                   */
   /*                                                                                                 */
   /*  verifies that t is a whole number between 1 and t_type.0                                       */
   /*  If not, it shows a diagnostic message and (possibly) exits 'BUG'                               */
   /*                                                                                                 */
   /*  Useful when writing a new task, to validate out-of-range arguments.                            */
   /*                                                                                                 */
   /*  Any routine that uses this should start with "Esigl = sigl" because Esigl is used              */
   /*  in the message here.                                                                           */
   /* =============================================================================================== */
   TVsigl = sigl
   parse arg t , Esigl
   if datatype(t, 'W') ,
    & t >= 1 ,
    & t <= t_type.0 then return 1
   call ShowError '', ''
   call ShowError '[T_Valid] BUG: invalid token number "'t'" in "'LastLabel(TVsigl)'" line' TVsigl
   call ShowError '      'sourceline(TVsigl)
   if Esigl \= '' then do
         call ShowError 'Esigl == line' Esigl '(in 'LastLabel(Esigl)')', ''
         call ShowError '      'sourceline(Esigl)
      end
   call Fatality
   return 0
   /* === End of T_Valid ============================================================================ */

   /*  U_   procedures for untokenized lines                                                   */ /*fold00*/
   /***************************************************************************************************/
   U_DividerType: procedure expose (Shared_vars) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       dtype = U_DividerType(<string>)                                                           */
   /*  where <string> is an untokenized line                                                          */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*       dtype = U_DividerType(SrcIn.ln)                                                           */
   /*       dtype = U_DividerType(L_AsString(t))                                                      */
   /*                                                                                                 */
   /*  returns the divider type of the given line--                                                   */
   /*  '#', '*', '-', 'F', '', etc.                                                                   */
   /*                                                                                                 */
   /*  See also S_DividerType                                                                         */
   /* =============================================================================================== */
   parse arg str
   str = StripStr(str)
   if left(str, 2) \= '/'||'*' then return ''
   if right(str, 2) \= '*'||'/' then return ''
   str = substr(str, 3, length(str) -4)                                            /* strip /@ and @/ */
   dtype = S_DividerType(str)
   return dtype
   /* === End of U_DividerType ====================================================================== */

   U_IsComment: procedure expose (Shared_vars) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       if U_IsComment(<string>) then ...                                                         */
   /*                                                                                                 */
   /*  returns True if <string> is a comment                                                          */
   /* =============================================================================================== */
   parse arg str
   str = strip(str)
   if left(str, 2) \= '/'||'*' then return 0
   if right(str, 2) \= '*'||'/' then return 0
   return  1
   /* === End of U_IsComment ======================================================================== */

   U_LabelList: procedure expose (Shared_vars) /*fold01*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     LabelList = U_LabelList()                                                                   */
   /*                                                                                                 */
   /*  returns a list of all labels in the SrcIn. array                                               */
   /*                                                                                                 */
   /*  This is only used when no tokenizing is done, so the LabelList() function will not work        */
   /* =============================================================================================== */
   list = '[Unlabeled]'
   incomment = 0
   do l = 1 to SrcIn.0
      if pos('/'||'*', SrcIn.l) > 0 ,
       | pos('*'||'/', SrcIn.l) > 0 then
         incomment = Incomment(SrcIn.l, incomment)
      if incomment <= 0 then do
            if HasLabel(SrcIn.l) then do
                  parse var SrcIn.l label ':' .
                  list = list strip(label)
               end
         end
   end l
   return list
   /* === End of U_LabelList ======================================================================== */

   Help_Misc:  /*  this is a dummy routine for "help <subject>" to find  */ /*fold00*/
   /* =============================================================================================== */
   /*  HELP: progress idle dialect                                                                    */
   /*                                                                                                 */
   /*     Options that are not specific to one command or task:                                       */
   /*                                                                                                 */
   /*        -dialect=<dialect>      Force RexxTool.cmd to treat the file as <dialect>,               */
   /*                                regardless of what it seems to be.                               */
   /*                                <dialect> can be either Regina or RexxSAA.                       */
   /*                                'Rexx' can be used for 'RexxSAA'                                 */
   /*                                                                                                 */
   /*                                If -dialect is not used:                                         */
   /*                                   If the first line of the file starts with                     */
   /*                                         EXTPROC regina.exe                                      */
   /*                                   or                                                            */
   /*                                         #! /usr/bin/regina                                      */
   /*                                   RexxTool.cmd will assume -Dialect=REGINA                      */
   /*                                   otherwise it will assume the dialect is the same as           */
   /*                                   the interpreter running RexxTool.cmd.                         */
   /*                                                                                                 */
   /*                                If you use -dialect, RexxTool.cmd will happily try               */
   /*                                to format a cmd.exe batch file or even a bash script.            */
   /*                                                                                                 */
   /*                                                                                                 */
   /*        -idle                   run at lower priority, to avoid slowing down other processes     */
   /*                                On my old, slow computer I needed this. On my new                */
   /*                                SMP computer I don't see as much difference.                     */
   /*                                Does not work for the Regina interpreter.                        */
   /*                                                                                                 */
   /*                                                                                                 */
   /*        -progress               show some progress messages to tell you what is happening        */
   /*                                -progress can also help prevent some tasks from                  */
   /*                                hogging the CPU. I do not understand why that is.                */
   /*                                On by default. Best to leave it that way.                        */
   /*                                The progress display is not as good under Regina/Linux           */
   /*                                because of deficiencies in the Regutil library.                  */
   /*                                                                                                 */
   /* =============================================================================================== */
   /*  this is a dummy task to hold documentation for switches that are not specific                  */
   /*  to any script or task                                                                          */
   /* === End of Help_Misc ========================================================================== */

   Help_Todo:  /*  this is a dummy routine for "help <subject>" to find  */ /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Todo                                                                                     */

   /*
          akm: these are notes to myself about things I might want to do

          * test the Toker_SubKeywords routine to see if it identifies all the keywords and
            sub-keywords correctly. What it really needs is a sample file with all the
            possible combinations of keywords and sub-keywords.

          * Depends does not include support for RXU functions like RxLoadMacroSpace
            or RxCallByName.  It probably should.

          * Add more BIFs from commonly-used libraries.
            The Format command uses the BIF lists for recapitalizing, and the Depends
            command uses them to distinguish BIFs from external commands. It is easy to add
            support for a new library-- just add a list of the BIFs it provides to the
            Init_DefineBIFs routine and add the library name to the BIFs.0OS2Libraries
            variable.

          * make it faster--
               * ShowError() is very slow, because LastLabel() uses Sourceline().
                  Normally, ShowError is never called anyway, but when there are many
                  errors it could become annoying.
                  --added -verbose switch; without it Sourceline is not used.
               * Use -AddCount to find the most-used routines and optimize them
                  --made some improvements, need to do more
               * Eliminating unneeded exposed variables does not make a noticeable difference.
               * The Help and DevHelp commands are annoyingly slow. Mostly because they must
                 track the comment level in order to exclude commented-out routines. Is it
                 better to make them fast or foolproof?

         *  the help/documentation could use improving. It's time to either simplify it
            or make it better but more complicated:
               documentation for a command X should be in Script_X, Documentation for switches
               that command X uses should also be in Script_X, but probably also in the header
               of the task where the switch is used. Except that would be repetitious. And it
               is possible that a switch could have different meanings in different scripts.

         *  it would be good to have the documentation in an INF file. It might be possible
            to adapt the Usage command to produce a simple IPF file, but I have no idea
            how to do that.

         *  Someone might want a way to preserve aligned comments like
                     IF a THEN         /@ xxx xxxx xx          @/
                        DO             /@ xxxxxx xxxxxxx xxxx  @/
                           nop         /@ xxx                  @/
                        END            /@ xx xxxxxxxxxxx xxxx  @/
               --it would mean checking the t_col.t value of the comment

         * wrap/unwrap lines to fit the line length
           --some parts are done; needs more work by someone.

         * Most functions will handle tabs correctly, but maybe not all of them
           --need to do some testing with files that have tabs

         *  Add support for Regina. Maybe someone who knows more about Linux can help.
               --see Regina.txt

         *  Object Rexx support?  --Not by me. I suspect it would be better to use this
            as the basis of a separate Object Rexx formatter, but don't know enough about
            Object Rexx to even be sure of that.

         * it would be easy to make a syntax highlighter from this-- insert HTML tags
           or ANSI sequences for each type of token. I'm not sure who would need
           such a thing.

         * some tasks break the tokenizing by substituting a token with a string that should
           rightfully be several tokens. Normally that's good because it is faster. But it
           means the task cannot be combined with other tasks that need accurate tokenizing.
           Maybe have a way to recombine tokens into a new SrcIn array, then re-tokenize
           and continue. I'm thinking of a hypothetical pre-processor task that could be
           done before formatting.

         * clarify terminology in comments-- reserve "procedure" for a routine that is
           really a procedure; a "block" is a block of code that includes a labeled routine
           and its associated comments. Also for variable names.

         * also see any comments that start with "fix:"

   */
   /* =============================================================================================== */
   /* === End of Help_Todo ========================================================================== */

   Help_Types: /*  this is a dummy routine for "help <subject>" to find  */ /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Types Token Tokens Type Class                                                            */

   /*
      Rexx code consists of a series of tokens, which are the words, symbols, strings,
      and numbers that you see. Tokens are grouped into clauses. See the example below.

      In this program, the tokens are represented by a set of arrays that hold
      various attributes of the tokens. The Tokeniser creates six arrays of attributes.
      For token number N,
         t_val.N        is the word, symbol, etc. that you see.
         t_prefix.N     is the string of blanks before the value.
         t_line.N       is the line number where the token was found. If you create
                        a new token, you must set its value, prefix, etc., but you can
                        leave the line number blank.
         t_col.N        is the column where the token was found.
                        You can probably ignore this array. It is provided by the Tokeniser
                        and currently is used only by the Prep_UnMargin stage.
         t_class.N
         t_type.N       class and type hold strings that describe what kind of token X is. See below.

      The formatter adds two more attributes:
         t_readonly.N   if True, formatting tasks are not allowed to move the token or change
                        its prefix.
         t_indent.N     shows how much token N will be indented when the formatting is done,
                        if token N is at the start of a line.

         Here is a list of the possible values for the t_class and t_type arrays:

            CLASSES (the t_class. array)
               COMMENT     --Comment tokens
               REXX        --for Rexx tokens; any token that has meaning for Rexx
               FILE_EOL    --End Of Line, also known as Line-Ending, Line Feed, or CRLF
               FILE_BOF    --Beginning Of File, the first token; a dummy token that is
                             convenient for some purposes.
               FILE_EOF    --End Of File; the last token; likewise a convenient dummy.
                             (not to be confused with the EOF character '1A'x or Ctrl-Z,
                             which is not relevant here)
               OTHER       --this is a special class that is not part of the Tokeniser.
                             It is only used by certain tasks that add new lines to
                             the file. See procedures O_InsertLine and O_InsertToken.

            TYPES (the t_type. array)
            Types For COMMENT tokens:
               The Rexx interpreter simply ignores comments, but a formatter cannot do that.
               COMMENT_BEG             --Begin Comment, a '/' and '*'
               COMMENT_END             --End Comment, a '*' and '/'
               COMMENT_VAL             --the comment text between the Begin and End,
                                         including leading and trailing spaces
               COMMENT_EOL             --a line-ending within a multi-line comment
            The C_CommentType() function can give a more specific type for COMMENT_VAL tokens.

            Types for REXX tokens:
               the REXX_W_ types are Rexx "words"; tokens that are part of a statement:
               REXX_W_COMMA            --a comma
               REXX_W_CONSTANT         --a constant: numbers, and hex/binary strings, and a few
                                         things that do not fit other categories
               REXX_W_KEY              --a Rexx keyword, like PARSE, SAY, IF, THEN
               REXX_W_KEY2             --a Rexx sub-keyword, like WITH, TO, ON, PULL
               REXX_W_LITERAL          --a literal string, in single or double quotes
               REXX_W_PROCEDURE        --name of a routine or function, internal, external, or built-in
               REXX_W_SYMBOL           --a Rexx symbol, usually a variable name,
                                         but could be an unrecognized token, like a system command
               REXX_W_NOT              --the NOT sign \ (also , and in Regina ~ and ^)
               REXX_W_LOGIC            --a logical (Boolean) operator, |, &, or &&
               REXX_W_COMPARE          --a comparison (conditional) operator, like =, ==, <=, etc.
               REXX_W_CONCAT           --the || concatenation operator
               REXX_W_ASSIGN           --the assignment operator '=', also Regina shortcut operators
               REXX_W_MATH             --math operators like /, *, **, +, -
               REXX_W_(                --the left parenthesis character
               REXX_W_)                --the right parenthesis character

               these are not part of a statement but have meaning to Rexx:
               REXX_LABEL              --the label that identifies a routine
               REXX_COLON              --the colon after the label
               REXX_EOC                --End Of Clause; it may be a ";" (semicolon) or a line-ending.
                                         It may also be an extra token that the Tokeniser adds
                                         where a ";" is optional. Before and after THEN, for example.
                                         These "hidden" EOCs make it easier for RexxTool.cmd to
                                         analyze statements.
               REXX_UNKNOWN            --you should never see this. It indicates an error in either
                                         the source file or in RexxTool.cmd

               these are token types that have no meaning to Rexx:
               CONTINUE                --a comma that indicates a continued line
               NULL                    --a FILE_EOL line-ending that is disabled by a CONTINUE comma
               COLON2                  --the "::" token used by Object Rexx
                                         (even though RexxTool.cmd does not do Object Rexx)

            The FILE_XXX tokens all have Type REXX_EOC.

         --------------------------------------------------------------------------------------------------
         If you use the -debug=token switch, you will get two extra files--
         the <filename>_Tokens_Begin file shows how each line was split into tokens and the
         tokens were identified by the Tokeniser, The <filename>_Tokens_End file shows the
         tokens after the Formatter has changed them. The best way to understand tokenizing is
         to look at those files.

         Here is a sample of Rexx code.  The comments are changed to "/@" and "@/" so as
         not to confuse Rexx.

                    /@===@/
                    if a == 'FF'x then
                       say 'Yes'


         which is tokenized into this:
              Class       Type                  {Prefix}{Value}
              -----       ----                  ---------------
              FILE_BOF    REXX_EOC              {}{}

              COMMENT     COMMENT_BEG           {}{/@}
              COMMENT     COMMENT_VAL           {}{===}
              COMMENT     COMMENT_END           {}{@/}
              FILE_EOL    REXX_EOC              {}{}

              REXX        REXX_W_KEY            {}{if}
              REXX        REXX_W_SYMBOL         { }{a}
              REXX        REXX_W_COMPARE        { }{==}
              REXX        REXX_W_CONSTANT       { }{'FF'x}
              REXX        REXX_EOC              {}{}
              REXX        REXX_W_KEY            { }{then}
              FILE_EOL    REXX_EOC              {}{}

              REXX        REXX_W_KEY            {}{say}
              REXX        REXX_W_LITERAL        { }{'Yes'}
              FILE_EOL    REXX_EOC              {}{}

              FILE_EOF    REXX_EOC              {}{}
         */
   /* =============================================================================================== */
   /* === End of Help_Types ========================================================================= */

   Help_Hack:  /*  this is a dummy routine for "help <subject>" to find  */ /*fold00*/
   /* =============================================================================================== */
   /*  HELP: Hack Hacking                                                                             */

   /*
            I (Anton Monroe) have tried to make it (relatively) easy for other people
            to add new features to RexxTool.cmd. If you want to improve it:

            The comments in RexxTool.cmd are probably worth reading.

            This file has over 15,000 lines and 200 procedures. It is easier to manage
            with a folding text editor. If you use FTE or eFTE, you should see my folds.

            My ideas about things to fix or improve can be found by searching the
            file for the string "fix:"  Also try "RexxTool.cmd help todo".

            The program works by breaking up Rexx code into tokens, identifying them,
            manipulating the tokens, and then putting them back together. See
                     RexxTool.cmd help types
            The best way to understand the tokens is to run "RexxTool.cmd format" with
            the -debug=token switch, which will create two files with information about
            the tokens.  The first shows how each line was split into tokens, the
            second describes the tokens after the formatter has worked on them.

            The routines that start with "Toker_" make up the tokenising stage.
            In the comments I sometimes refer to the tokenizing stage as the Toker.
            It's just an abbreviation that my fingers made up because they got tired
            of typing "Tokenizer". Or "Tokeniser".

            Most of the work is done by looping through the list of tokens. An important
            principle is that the program should not try to do too much in one pass.
            Instead, the work is divided into tasks, each of which makes a separate pass
            through the token list. It may be less efficient this way, but it is much
            easier to maintain, and it is much easier to add new tasks.

            The routines mostly fall into three groups-- tools, tasks, and scripts.

            TOOLS are the lowest level. Generally a tool works on one token, or in
            some cases, a group of tokens. To make tools easier to find, most of
            them have a prefix that is a clue to what that tool works on--
            C_ for Comment, CL_ for Clause, L_ for Line, O_ for Other, B_ for Block,
            R_ for Rexx, S_ for strings, T_ for Token, and U_ for Untokenized.
            For instance, the L_LastToken(t) function returns the last token in the
            line that token t is in. For descriptions of all the tool procedures, type

                     RexxTool.cmd devhelp *_*

            A TASK procedure performs one action on the whole set of tokens. Or
            sometimes it does several related actions. For example, Task_Recap
            searches the token list and corrects the capitalization of any variables,
            labels, or built-in-functions that it finds. Task_Indent calculates
            how many spaces each line should be indented.

            Some tasks use intermediate-level procedures that are best considered
            part of the task, although a couple of them are used by more than one task.

            The Prep_xxx routines are tasks that make some changes to the tokens to
            get them ready for formatting.

            A SCRIPT is simply a series of tasks to be executed. Usually it corresponds
            to a "command" on the command line. So
                  RexxTool.cmd FORMAT OldFile.cmd NewFile.cmd
            will execute the routine called Script_Format.

            The Toker_xxx routines are the Tokeniser that creates the arrays of
            tokens. Unless you discover a bug in the Tokeniser you don't really
            need to know how it works.

            There is a diagram at the top of RexxTool.cmd that shows the work flow.

            What the tasks do is controlled by variables in the Opt. stem variable.
            The tail names correspond to the command-line switches.

            If you want to add or change something, I suggest that you write a new
            task with a switch to control it. Or some things might be better as a
            section added to an existing task which can be executed or not depending
            on the value of a configuration variable. For instance, if you want to
            change something about the indenting, add a new section to the
            Task_Indent2 routine. If you want words spaced more compactly,
            you can change Task_Respace. Or better, write a replacement called
            something like Task_Respace_Compact.

            So to add a new task called Mangle, you will need to
            (1) write it in a procedure named Task_Mangle, Include a header that gives
                a description of what it does. You can see the format of the header in
                Task_template or one of the existing tasks.
            (2) in the Init_Config procedure, add one line for each command-line switch
                See the examples in Init_Config.
            (3) Near the end of the Init_Config procedure, add the name of the switch
                to the ValidSwitch variable, so the command-line parser knows it is a
                legal switch. You do not need to write code to parse the new switch,
                the GetOpts procedure will handle that.
            (4) if Mangle is a formatting task, add a call to Task_Mangle in the
                Script_Format procedure. Or you might want to write a Script_Mangle
                routine that calls Task_Mangle.

            if you add a script, you will need to add a line in section Main.
            Something like
                  if Command == 'MANGLE' then call Script_Mangle

            See Style_akm and Script_Quick for some examples of what you can do with
            a custom script.

            A STYLE is simply a built-in set of configuration options. See Style_akm
            and Style_alternate. If you add a Style_xxx routine, add its name to the
            G.0Styles variable in the Init_Config routine.

            For details about how command-line switches are parsed, see
                  RexxTool.cmd devhelp GetOpts
            If you have any doubts about how a command line is interpreted, type
                  RexxTool.cmd <switches> -ShowGetOpts

         Here are some writing conventions I try to follow:

            As much as possible, a task should be self-contained. It should do one
            thing, or several related things. Ideally it will not depend on whether
            other tasks have been done, but that is not always possible.
            Even the documentation goes in the procedure itself, in the procedure
            header. The "help <subject>" command works by extracting the help text
            from the appropriate header(s). You can use the Task_template procedure
            as a starting point for a new task.

            If a task is turned on or off by an Opt.0xxx variable, the task should check
            Opt.0xxx itself; it should not depend on a script to decide whether to
            call the task or not. The self-contained principle again.

            Tokens are described by a set of arrays whose names start with "t_".
            Their names are listed in the "all_tokens" variable, which is:
               t_class. t_type. t_prefix. t_val. t_line. t_col. t_indent. t_readonly.
            See "RexxTool.cmd help types"

            Some tokens are marked as Read Only by setting the t_readonly attribute.
            "read only" is inaccurate; it really means "do not change the position
            or spacing of this token". Before changing a token T a task should check
            the value of t_readonly.T. Tasks are allowed to change the value of
            a ReadOnly token. Re-capitalizing or changing the name of a variable,
            for instance.

            A task should not usually count tokens itself.  For example, a Rexx
            statement might be
                    IF a == b THEN DO something
               or
                    IF a == b THEN /@ comment @/
                       DO something
               or
                    IF a == b THEN ; DO something
            if THEN is token t, you cannot assume that DO is token number t +1. The
            only safe way to find the next word is with the function R_NextWord(t).

            I try to do things in the same way I think about them-- if what you want
            is "the last token on the previous line", then
                     P = L_LastToken(t, -1)
            will find it.  But is that what you really want? The previous line may be
            a comment, without any Rexx tokens at all.  If what you really mean is
            "the Rexx word before the first word in this line", then use
                     P = R_PrevWord(L_FirstRexxWord(t))

            For most global variables, the tail of a stem variable starts
            with "0" to prevent confusion with normal variable names. When you write
            a new routine, I cannot expect you to know what names have already been
            used in global variables.

            The "Opt." stem holds all the configuration variables. If you add one,
            read the comments in the Init_Config routine.

            The "G." stem is a collection of global variables that can be used wherever needed.

            Shared_vars is a list of variables that are needed in many places.
            Any procedure should use "EXPOSE (Shared_vars)". Even if your routine does
            not use any of the Shared variables itself, it might call routines that
            do need them. An error-handler, if nothing else.

            All exits from the program are done with
                     CALL Egress <exit status>
            See comments under the Egress label

            Output is done by writing to the SrcOut. array. Usually with
                     CALL Out <string>
            which is just an easier way of writing
                     ln = SrcOut.0 +1
                     SrcOut.ln = <string>
                     SrcOut.0 = ln
            When a script is done, it then sends the SrcOut. array to a file
            or stdout by calling ArrayToFile.

            Warnings to the user, about a defective file for example, go to stderr with
                     CALL Msg <string>
            Warnings about a possible bug in RexxTool.cmd itself go to stderr with
                     CALL ShowError <string>


         Debugging tools and error-handling:

            The -debug switch can be used to set some flags that control whether
            extra debugging will be done. For example. "-debug=eoc,loop" sets
            debug.0eoc and debug.0loop to 1. debug.0fatal will cause the program
            to abort immediately if a serious bug is detected. Otherwise it will
            show a warning and continue. debug.0arg validates arguments to many
            of the tool functions. Turning it off can save a few seconds of running
            time. The other debug categories have little effect on performance.
            debug.0loop and the Loopcounter routine provides a way to detect and
            break out of an endless loop. In some places it is always on to prevent
            an endless loop caused by defective code in the source file.

            "CALL ShowError <message>" can be used at any point to show a message on
            stderr. It will automatically include the line number it was called from.
            If -verbose is used it will also show the name of the procedure it was
            called from.

            "CALL Mark '<variable list>', <message>" can be used at any point to show
            the current values of the listed variables. As in
                  CALL Mark 't t_val.t t_type.t', 'top of do forever loop'
            There is also a DumpVars() routine, but Mark is probably more useful.
            I use Mark a lot. I have never gotten used to Trace.

            Some C_xxx functions use the C_Errorhandler function to report an error.
            It needs improving and adapting to other routines

            The RunoutHandler routine is to detect a routine that fails to return.
            When I am writing a new routine, I can put
                       SIGNAL RunoutHandler
            before the next label to make sure execution does not fall through. This
            does a better job than Rexx does of showing where the mistake really is.

            At any point in the program you can dump the current state of the token
            list by inserting a line like
                      CALL SaveTokensF <file name>
            Also see the SaveTokensT routine, which can be called at the end of the
            Toker stage.

            The Qline(t_line.t) function can be used to show the line of the source file
            that token t is in.

            "CALL ShowContext t" will show several lines before and after token t
            They can be useful in error messages.

            the SubID() function returns the name of the current routine (of RexxTool.cmd)

            Other routines used in error and progress messages include
            LastLabel(), LineNo()

         About indirect variables:
           There are many places in RexxTool.cmd where I use what I suppose you could
           call an "indirect variable". There may be a better term. For example
               !ProcN = 'Foo'
               Exposed.!ProcN = 'aaa'
               if Exposed.!ProcN == ...
           which is the same as writing
               Exposed.Foo = 'aaa'
               if Exposed.Foo == ...

           But that is not the recommended way to assign its value. It should be
               call value 'Exposed.'!ProcN, 'aaa'
               if value('Exposed.'!ProcN) == ...

           The Rexx documentation I have seen does not really explain why. The reason
           seems to be that my way creates a variable with a case-sensitive name.
           The above example creates a variable called EXPOSED.Foo, and any reference
           to it must have the value of !ProcN capitalized the same way.  Using value()
           uppercases the value of the !ProcN variable and creates EXPOSED.FOO, which
           can be referred to using any case.

           I only figured this out recently. I will not rewrite this script to do it
           the "correct" way because my style is easier to write and read, and since I
           always uppercase the value of !ProcN it makes no difference. But I don't want
           to give the impression my way is a good example to follow.

           If you want to see how Rexx can have case-sensitive variable names, run this
           example:

              if RxFuncQuery('SysLoadFuncs') then do
                    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
                    call SysLoadFuncs
                 end

              !Name = 'Crocus'
              call value 'Flower.'!Name, '"Crocus" using Value()'
              Flower.!Name = '"Crocus" without using Value()'
              !Name = 'CrOcUs'
              Flower.!Name = '"CrOcUs" without using Value()'
              call SysDumpVariables

         */
   /* =============================================================================================== */
   /* === End of Help_Hack ========================================================================== */

   Egress: /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     call Egress <exit type>                                                                     */
   /*                                                                                                 */
   /*  All exits come through here.                                                                   */
   /*                                                                                                 */
   /*  OKAY    == normal exit                                                                         */
   /*  USAGE   == user error                                                                          */
   /*  NOTREXX == File is not Rexx                                                                    */
   /*  FILE    == defective file                                                                      */
   /*  CHANGED == File was changed                                                                    */
   /*  ERROR   == error detected by Task_CheckError/Script_CheckOnly                                  */
   /*  BUG     == probable bug in this program                                                        */
   /*                                                                                                 */
   /* =============================================================================================== */
   Esigl = sigl
   signal off NoValue
   signal off Syntax

   if Opt.0idle == 1 then do
         /*  return priority to what we presume it started at:                                        */
         if G.0Interpreter == 'REGINA' then
            nop
         else
            call SysSetPriority 2, 0
      end
   parse arg ExitCode
   ExitCode = translate(ExitCode)
   if Opt.0progress == 1 then call Progress ''

   /*    handle my typing mistakes                                                                    */
   if ExitCode = 'OKAY' then exit 0
   if ExitCode = '' then exit 0
   if ExitCode = 0 then exit 0
   if abbrev(ExitCode, 'USAGE') then exit 1
   if abbrev(ExitCode, 'NOTREXX') then exit 2
   if abbrev(ExitCode, 'FILE') then exit 3
   if abbrev(ExitCode, 'CHANGED') then exit 4
   if abbrev(ExitCode, 'ERROR') then exit 5
   if abbrev(ExitCode, 'BUG') then exit 9
   /*    in case I used some other exit string and forgot to add it here:                             */
   call lineout 'stderr', 'invalid argument "'ExitCode'" to Egress from line' Esigl
   exit 99
   /* === End of Egress ============================================================================= */

   LoopCounter: /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     call LoopCounter                                                                            */
   /*                                                                                                 */
   /*  When writing a routine that loops, it is easy for the control variable to not                  */
   /*  be incremented and cause an endless loop.                                                      */
   /*                                                                                                 */
   /*  This is a debugging tool to prevent that. To use it, just add "CALL LoopCounter"               */
   /*  at the start of the loop. If there have been more than loop_max loops, this                    */
   /*  will break out of the loop and show the values of any variables listed in                      */
   /*  G.0track_vars. You can delete the calls to it when a routine is perfected.                     */
   /*                                                                                                 */
   /*  The calling routine does not need to set any variables, but it can set                         */
   /*  loop_ctr or loop_max to whatever is convenient. Make sure loop_max is                          */
   /*  high enough to allow all the legitimate passes through the loop.                               */
   /*  Later: that should be done automatically now-- the default loop_max is based on                */
   /*  the file size. It may be much higher than is needed, but it doesn't matter. The                */
   /*  important thing is that there is a limit and that it not be too low.                           */
   /*                                                                                                 */
   /*  Note that loop_max and loop_ctr are not global, they are local to                              */
   /*  the calling routine.                                                                           */
   /* =============================================================================================== */
   if symbol('loop_max') == 'LIT' then
      loop_max = max(50000, stream(G.0InfileN, 'c', 'query size'))
   if symbol('loop_ctr') == 'LIT' then
      loop_ctr = 1
   else
      loop_ctr = loop_ctr +1

   if loop_ctr >= loop_max then do
         Esigl = sigl
         call ShowError 'Over maximum loop limit ('loop_max') at line 'Esigl
         call ShowError 'probably because of an endless loop in the' LastLabel(Esigl) 'routine', ''
         if G.0track_vars \== '' then
            call Mark G.0track_vars
         call Egress 'BUG'
      end
   return 0
   /* === End of LoopCounter ======================================================================== */

   RunoutHandler: /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     signal RunoutHandler                                                                        */
   /*                                                                                                 */
   /*  A debugging tool for a routine that does not return and execution falls through                */
   /*  into the next routine. "signal RunoutHandler" after the end of a routine can                   */
   /*  catch the error before Rexx does, and give a more accurate idea of where the                   */
   /*  problem is. It does not usually catch errors caused by an unbalanced DO/END.                   */
   /* =============================================================================================== */
   Esigl = sigl
   Err_func = LastLabel(Esigl)
   IDstring = G.0meID||'[RunoutHandler]'
   call ShowError IDstring 'Failed to return from routine"'Err_func'" at line 'Esigl
   call Egress 'BUG'
   /* === End of RunoutHandler ====================================================================== */

   Syntax: /*fold00*/
   /* =============================================================================================== */
   /*  depends on variables: G.0me sigl                                                               */
   /*  depends on functions: LastLabel(), ShowError(), Egress()                                       */
   /* ----------------------------------------------------------------------------------------------- */
   /*  traps syntax errors and shows a better error message than Rexx does. In particular,            */
   /*  an incomplete DO/SELECT/IF is fairly common and the error message from Rexx is no              */
   /*  help at all.                                                                                   */
   /*                                                                                                 */
   /*  Some errors cannot be handled by an error handler, like REX06 "unmatched comment or quote"     */
   /*  (Which makes sense. If a defective script cannot be tokenized, then it cannot be run, so       */
   /*  the handler never exists. But the help message for REX06 implies that error 6 may not be       */
   /*  detected until later, so there is something I don't understand.)                               */
   /*                                                                                                 */
   /*  I have found one or two files that have calls so deeply nested that it causes                  */
   /*  a "Control Stack Full" error from OS/2 Rexx, in the Tree command and                           */
   /*  in Task_CheckExpose. There is probably not much I can do about that.                           */
   /*                                                                                                 */
   /*  This will always be a work in progress                                                         */
   /* =============================================================================================== */
   SyntaxSigl = sigl
   call lineout 'stderr', ''
   /*  sometimes a "Control Stack Full" will be trapped here and sometimes not. I don't know why.     */
   /*  If the control stack is full, any DO or SELECT after this will probably trigger                */
   /*  another (untrapped) error 11. Get the important part of the message onto the                   */
   /*  screen before that happens, without using DO.                                                  */
   /*  I have not figured out why a second error 11 sometimes happens and sometimes doesn't.          */

   if rc = 11 then
      call lineout 'stderr', ,
           '[SYNTAX Handler] Rexx error 11: Control Stack Full at line' SyntaxSigl||G.0CRLF ,
           '                      ==>' strip(sourceline(SyntaxSigl))||G.0CRLF ,
           '                There may be another error 11 after this'||G.0CRLF
   else do
         select
            when rc == 14 then do
                  /*  incomplete DO/SELECT/IF                                                         */
                  /* Rexx reports an error at the end of file, which is useless                       */
                  call ShowError '[SYNTAX Handler]'
                  call ShowError '    Syntax error 'rc' which is "'errortext(rc)'"', ''
                  if SyntaxSigl >= sourceline() then do
                        call ShowError '    Rexx says it is at line 'SyntaxSigl', which is the end of file.', ''
                        call ShowError '    To find where the error really is, try ', ''
                        call ShowError '        "'G.0me 'checkonly' G.0me'"', ''
                     end
                  else do
                        call ShowError '    at line 'SyntaxSigl, ''
                        call ShowError '       ==> 'strip(sourceline(SyntaxSigl)), ''
                     end
               end
            otherwise
               Err_func = LastLabel(SyntaxSigl)
               call ShowError '[SYNTAX Handler] Syntax error at line 'SyntaxSigl 'in label' Err_func
               call ShowError '==>' sourceline(SyntaxSigl), ''
               call ShowError 'Error # 'rc' is "'errortext(rc)'"', ''
               if condition('d') \= '' then call ShowError 'condition(D) is "'condition('d')'"', ''
         end   /*  select  */
      end
   if G.0Interpreter == 'REXXSAA' then
      call ShowError 'for more help, type "helpmsg REX'rc'"', ''
   if G.0track_vars \= '' then
      call Mark G.0track_vars
   call Egress 'BUG'
   /* === End of Syntax ============================================================================= */

   Error: /*fold00*/
   /* =============================================================================================== */
   /*  I suppose I should have some sort of error handler, but I have never seen an error             */
   /*  condition that triggers it.                                                                    */
   /* =============================================================================================== */
   Esigl = sigl
   call Msg G.0meID '[Error Handler]'
   Cname = condition('C')
   call Msg G.0meID_ind 'A 'Cname' error occurred at line number 'Esigl
   call Msg G.0meID_ind 'Sourceline = 'sourceline(sigl)
   call Msg G.0meID_ind 'Error 'rc': 'errortext(rc)
   exit 9
   /* === End of Error ============================================================================== */

   NoValue: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   signal off NoValue
   _ = sigl
   Err_func = LastLabel(_)
   call Msg ''
   call Msg G.0meID'[NOVALUE Handler] at line '_ 'under label' Err_func
   call Msg G.0meID_ind 'for variable "'condition('d')'"'
   call Msg G.0meID_ind '   ==> 'sourceline(_)
   if symbol('Esigl') == 'VAR' then do
         call Msg G.0meID_ind 'called from line' Esigl 'under label' LastLabel(Esigl)
      end
   call Egress 'BUG'
   /* === End of NoValue ============================================================================ */

   /***************************************************************************************************/
   /*  Templates:                                                                                     */
   /***************************************************************************************************/

   Task_Template: procedure expose (Shared_vars) (all_tokens) SrcOut. /*fold00*/
   /* =============================================================================================== */
   /*  HELP: template switch1 switch2                                                                 */
   /*                                                                                                 */
   /*  -template              do this task                                                            */
   /*  -template=0            do not do this task                                                     */
   /*                                                                                                 */
   /*  -switch1               <description of what this switch does>                                  */
   /*  -switch2               <description of what this switch does>                                  */
   /*                                                                                                 */
   /*  This is a template for new tasks. Replace "template" with the task name and                    */
   /*  describe what the task does in this header.                                                    */
   /*                                                                                                 */
   /*  The first line of the header is a list of switches or other subjects that are documented       */
   /*  in this header. "RexxTool.cmd help switch1" will display this header                           */
   /*                                                                                                 */
   /* =============================================================================================== */
   if \Opt.0template then return 0
   /*  SubID() returns the name of the current routine                                                */
   SubID = SubID()

   /*  You may want to check the switch for valid values:                                             */
   ValidOpts = 'ON OFF'
   if Opt.0Switch1 \= '' ,
    & wordpos(translate(Opt.0Switch1), ValidOpts) == 0 then do
         call Msg 'unrecognized option: "-Switch1='Opt.0Switch1'"'
      end

   if Opt.0progress then call Progress SubID

   /*  loop through the token list. Most tasks will not use the first and last tokens                 */
   do t = 2 to t_type.0 -1
      nop
      nop
      nop
   end t

   /*  OR do this if t or t_type.0 will change inside the loop                                        */
   t = 2
   do while t < t_type.0
      nop
      nop
      nop
      nop
      t = t +1
   end   /*  while t < t_type.0  */

   if Opt.0progress then call Progress
   return 0
   /* === End of Task_Template ====================================================================== */

   T_Template: procedure expose (Shared_vars) (all_tokens) /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       r = T_template(t)                                                                         */
   /*       CALL T_template t                                                                         */
   /*  where t is a token number                                                                      */
   /* ----------------------------------------------------------------------------------------------- */
   /*  a template for a new tool procedure.  Give it a suitable name. For example:                    */
   /*    C_xxx for Comment-related routine                                                            */
   /*    CL_xxx for a Clause-related routine                                                          */
   /*    R_xxx for Rexx code                                                                          */
   /*    T_xxx for tokens                                                                             */
   /*                                                                                                 */
   /*  Describe the procedure in this header. The DevHelp command will extract each header            */
   /*  that starts with "Usage:"                                                                      */
   /* =============================================================================================== */
   /*  save the SIGL that this routine was called from. It can be useful for error messages           */
   Esigl = sigl
   parse arg t
   /*  if debugging, verify that t is a valid token number                                            */
   if debug.0arg then call T_Valid t, Esigl
   nop
   nop
   nop
   nop
   return 0
   /* === End of T_Template ========================================================================= */

   /*** the Tokeniser *********************************************************************************/ /*fold00*/
   /*  The Toker_xxx routines make up the Tokeniser.                                                  */
   /***************************************************************************************************/

   Toker: procedure expose (Shared_vars) (all_tokens) /*fold01*/
   /* =============================================================================================== */
   /*  this is the entry point of the Tokeniser                                                       */
   /*  The Tokeniser breaks up Rexx code into a list of tokens, identified by type, and               */
   /*  adds them to the list represented by stem variables t_type., t_val., etc.                      */
   /*                                                                                                 */
   /*  note that the Tokeniser removes trailing blanks at the end of a line, on the grounds           */
   /*  that they are of no conceivable use.                                                           */
   /*                                                                                                 */
   /*  The various Toker_XXX routines are all part of the Tokeniser.                                  */
   /*                                                                                                 */
   /*  The Toker_Rexx routine is mostly based on the mini-tokeniser in Toby Thurston's                */
   /*  checkfunc.cmd. I have changed it a lot, undeterred by the fact that I did not                  */
   /*  understand all of it.                                                                          */
   /*                                                                                                 */
   /*  These are notes to myself in case I need to understand this again sometime.                    */
   /*                                                                                                 */
   /*  The variables that will be used by Toker_store start with "st_"                                */
   /*                                                                                                 */
   /*        SrcIn.      == an array of lines from the source file                                    */
   /*        SrcIn.0     == total number of lines                                                     */
   /*                                                                                                 */
   /*   the working variables in the Tokeniser loop:                                                  */
   /*        st_line      == current line number                                                      */
   /*        c            == a 1-character string; the current character being examined               */
   /*        c2           == a 1-character string; the next after the current character               */
   /*        cc           == a 2-character string; 'c' plus 'c2'                                      */
   /*        p            == active position in the line                                              */
   /*                        note that 'p' is one position to the right of 'c'                        */
   /*        st_col       == the column where the current token starts.                               */
   /*                        When the end of token is found, the token is stored and                  */
   /*                        st_col is updated to the start of the next token.                        */
   /*        incomment    == comment nesting level, it can be > 1 if in a nested comment              */
   /*                                                                                                 */
   /*                                                                                                 */
   /*  The end result is a set of arrays, holding the tokens' class, type, line, column,              */
   /*  prefix, and value.                                                                             */
   /*                                                                                                 */
   /*  the Tokeniser loop works like this:                                                            */
   /*        Initialize the search for the next token                                                 */
   /*            set 'st_col' to the current position 'p', presumably the start of the token          */
   /*            set 'c' to the 1 character at position 'p'                                           */
   /*            set 'cc' to the 2 characters at position 'p'                                         */
   /*            increment 'p'                                                                        */
   /*        If there are leading spaces before a token, then                                         */
   /*            move p to where the token really starts                                              */
   /*            save the string of spaces. they are the prefix attribute of the token                */
   /*            Initialize again                                                                     */
   /*        Examine 'c' and 'cc' to identify a token.                                                */
   /*            Sometimes that involves subordinate loops, incrementing 'p', and using               */
   /*            extra variables like 'c2' to see what characters come after 'c'.                     */
   /*        At any place in the loop where a token is identified--                                   */
   /*            set 'st_val' and 'st_prefix' to the contents of the token and its prefix             */
   /*            set st_class and st_type to describe the token                                       */
   /*            set st_line and st_col to the line and column where the token was found              */
   /*            call Toker_Store to add the token to the list                                        */
   /*            ensure that 'p' is set correctly for the next search                                 */
   /*            ==> and do nothing else                                                              */
   /*            iterate                                                                              */
   /*                                                                                                 */
   /* =============================================================================================== */

   /* =============================================================================================== */
   /*  initialize many variables used by Toker_* routines                                             */
   /* =============================================================================================== */
   /*  the arrays that will hold the tokens and attributes:                                           */
   t_class.      = ''   ; t_class.0      = 0                                   /* class of each token */
   t_type.       = ''   ; t_type.0       = 0                                    /* type of each token */
   t_line.       = 0    ; t_line.0       = 0                                    /* line of each token */
   t_col.        = 0    ; t_col.0        = 0                                  /* column of each token */
   t_prefix.     = ''   ; t_prefix.0     = 0                                  /* prefix of each token */
   t_val.        = ''   ; t_val.0        = 0                                   /* value of each token */
   t_readonly.   = 0    ; t_readonly.0   = 0                                        /* readonly flags */
   /*  note that the t_readonly and t_indent stems are not included; they are added later             */
   Tkr. = ''
   Tkr.0token_vars = 't_class. t_type. t_prefix. t_val. t_line. t_col.'
   Tkr.0dialect = G.0dialect
   Tkr.0Keywords = 'ADDRESS ARG CALL DO DROP' ,
                   'EXIT IF INTERPRET ITERATE' ,
                   'LEAVE NOP NUMERIC OPTIONS PARSE' ,
                   'PROCEDURE PULL PUSH QUEUE RETURN SAY SELECT' ,
                   'SIGNAL THEN TRACE USE' ,
                   'ELSE OTHERWISE END WHEN' ,
                   'USE'
   Tkr.0ExitCode = ''
   Tkr.0NOT_ops = '\'
   Tkr.0charset = '09'x||' ' ,
                  ||'_abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ?!' ,
                  ||'1234567890+/-*%&|=<>\' ,
                  ||'.():"'';,'
   Tkr.0typeset = 'bb' ,
                  ||'sssssssssssssssssssssssssssssssssssssssssssssssssssssss' ,
                  ||'nnnnnnnnnnoooooooooooo' ,
                  ||'.():"'';,'
   /* ----------------------------------------------------------------------------------------------- */
   /*  Adjustments for Regina:                                                                        */
   /*  Regina keyword list and charset are always used. Not sure why I did it that way                */
   /*  but it does no harm                                                                            */
   Tkr.0Keywords = Tkr.0Keywords 'UPPER'
   Tkr.0charset = '$#@'||Tkr.0charset
   Tkr.0typeset = 'sss'||Tkr.0typeset
   if Tkr.0dialect == 'REGINA' then do
         Tkr.0NOT_ops = Tkr.0NOT_ops||'^~'
         Tkr.0charset = '~^'||Tkr.0charset
         Tkr.0typeset = 'oo'||Tkr.0typeset
      end
   /* ----------------------------------------------------------------------------------------------- */

   /* =============================================================================================== */
   /*  Now do the tokenising:                                                                         */
   call Toker_Rexx
   /*  and identify sub-keywords                                                                      */
   call Toker_SubKeywords
   call Toker_IDfunctions
   return Tkr.0ExitCode
   /* === End of Toker ============================================================================== */

   Toker_Rexx: /*fold01*/
   /* =============================================================================================== */
   c = ''                                                                  /* character at position p */
   cc = ''                                                              /* 2 characters at position p */
   c2 = ''                                                                  /* next character after c */
   p = 1                                                                      /* position in the line */
   /* st_xxx variables describe the current token and are used by Toker_store                         */
   st_class = 'REXX'
   st_type = ''
   st_prefix = ''
   st_val = ''
   st_line = 1
   st_col = p
   L_len = length(SrcIn.st_line)                                            /* length of current line */

   loop_ctr = 0
   Tkr.0ORexxWarned = 0
   if Opt.0progress then call Progress G.0meID'[Toker] looking for tokens...'

   /*  Add a Beginning of File token just to make it easier to do some things.                        */
   /*  Without it, a loop that uses token t-1 on each pass will error if t = 1.                       */
   st_line = 1
   st_class = 'FILE_BOF'
   st_type = 'REXX_EOC'
   st_val = ''
   st_prefix = ''
   st_col = 1
   call Toker_Store

   /* ----------------------------------------------------------------------------------------------- */
   /*  I sometimes see code samples that have the starting comment somewhere other                    */
   /*  than the beginning of the first line. Try to recognize it for what it is                       */
   /*  intended to be.                                                                                */
   /*  I could just remove the empty lines, but who knows, someone might have a reason                */
   /*  to want them.                                                                                  */
   /*  The Toker always strips trailing blanks, because I cannot think of                             */
   /*  any possible use for them.                                                                     */
   do st_line = 1 to SrcIn.0
      if StripStr(SrcIn.st_line) == '' then do
            st_class = 'FILE_EOL'
            st_type = 'REXX_EOC'
            st_val = ''
            st_col = 1
            call Toker_Store
         end
      else
         leave st_line
   end st_line
   L_len = length(SrcIn.st_line)
   /* ----------------------------------------------------------------------------------------------- */

   /*  Regina on Linux, or even under 4OS2, may be identified by a #! line.                           */
   /*  Store it as a comment with empty COMMENT_END token                                             */
   if left(StripStr(SrcIn.st_line), 2) == '#!' then do
         /*  a tokeniser should preserve any leading spaces; removing them is the formatter's job     */
         st_prefix = ''
         do p = 1 to L_len
            c = substr(SrcIn.st_line, p, 1)
            if pos(c, ' '||'09'x) == 0 then leave
            st_prefix = st_prefix||c
         end p
         st_val = substr(SrcIn.st_line, p, 2)
         st_class = 'COMMENT'
         st_type = 'COMMENT_BEG'
         st_col = p
         call Toker_Store
         st_val = substr(SrcIn.st_line, p +2)
         st_class = 'COMMENT'
         st_type = 'COMMENT_VAL'
         st_col = p +2
         call Toker_Store
         st_val = ''
         st_class = 'COMMENT'
         st_type = 'COMMENT_END'
         p = L_len +1
         st_col = p
         call Toker_Store
      end

   /*  likewise for an OS/2 EXTPROC line                                                              */
   if translate(word(StripStr(SrcIn.st_line), 1)) == 'EXTPROC' then do
         st_prefix = ''
         do p = 1 to L_len
            c = substr(SrcIn.st_line, p, 1)
            if pos(c, ' '||'09'x) == 0 then leave
            st_prefix = st_prefix||c
         end p
         st_val = substr(SrcIn.st_line, p, 7)
         st_class = 'COMMENT'
         st_type = 'COMMENT_BEG'
         st_col = p
         call Toker_Store
         st_val = substr(SrcIn.st_line, p +7)
         st_class = 'COMMENT'
         st_type = 'COMMENT_VAL'
         st_col = p +7
         call Toker_Store
         st_val = ''
         st_class = 'COMMENT'
         st_type = 'COMMENT_END'
         p = L_len +1
         st_col = p
         call Toker_Store
      end

   /* =============================================================================================== */
   /*  This is the main loop that reads the SrcIn array                                               */
   /*  IMPORTANT: to prevent stupid 'endless loop' mistakes, do the tests in this order:              */
   /*      check for EOF (is line > last line)                                                        */
   /*      check for EOL (is p > last column of line)                                                 */
   /*      initialize for a new token                                                                 */
   /*      then do the string comparisons                                                             */
   /* =============================================================================================== */
   do forever
      /* -------------------------------------------------------------------------------------------- */
      /*  EOF  (End Of File)                                                                          */
      /*  bug check                                                                                   */
      if st_line > SrcIn.0  then do
            tokHigh = t_type.0
            if t_class.tokHigh \= 'FILE_EOL' then do
                  call ShowError  'Error: reached the last line but there is no EOL token'
                  call ShowError  'This is probably a bug in the Tokeniser', ''
                  st_class = 'FILE_EOL'
                  st_type = 'REXX_EOC'
                  st_val = ''
                  st_col = L_len +1
                  call Toker_Store
                  if wordpos('BUG', Tkr.0ExitCode) == 0 then Tkr.0ExitCode = Tkr.0ExitCode 'BUG'
               end
            st_class = 'FILE_EOF'
            st_type = 'REXX_EOC'
            st_val = ''
            st_col = L_len +1
            call Toker_Store
            leave
         end
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      /*  EOL (End Of Line)                                                                           */
      /*  if the line is continued by a comma, the type of the comma token and the EOL token          */
      /*  will be corrected in Toker_store                                                            */
      if p > L_len then do
            st_class = 'FILE_EOL'
            st_type = 'REXX_EOC'
            st_val = ''
            st_col = L_len +1
            call Toker_Store
            st_line = st_line +1
            L_len = length(SrcIn.st_line)
            p = 1
            iterate
         end
      /* -------------------------------------------------------------------------------------------- */

      /*  Now set up for the next token:                                                              */
      st_class = 'REXX'
      st_type = ''
      st_prefix = ''
      st_val = ''
      st_col = p
      c = substr(SrcIn.st_line, p, 1)
      cc = substr(SrcIn.st_line, p, 2)
      p = p +1
      type = translate(c, Tkr.0typeset, Tkr.0charset)
      if debug.0loop then call LoopCounter

      /* -------------------------------------------------------------------------------------------- */
      /*  BLANK is a <space> or <tab> character.                                                      */
      /*  The leading spaces before a token are the prefix. Do not call Toker_store or                */
      /*  iterate-- just save the blanks to st_prefix, move p to the start of the token,              */
      /*  and keep going.                                                                             */
      if type == 'b' then do
            do while p <= L_len
               c = substr(SrcIn.st_line, p, 1)
               if pos(c, ' '||'09'x) == 0 then
                  leave
               p = p +1
            end
            st_prefix = substr(SrcIn.st_line, st_col, p - st_col)
            st_col = p
            c = substr(SrcIn.st_line, p, 1)
            cc = substr(SrcIn.st_line, p, 2)
            p = p +1
         end
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      /*  EMPTY LINE (next pass will store the EOL token)                                             */
      if c == '' then iterate
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      /*  COMMENT: handling comments moved to a separate routine for readability                      */
      if cc == '/'||'*' then do
            /*  move p back so Toker_Comment can start over                                           */
            p = p -1
            call Toker_Comment
            iterate
         end

      /*  a REGINA or OBJECT REXX COMMENT                                                             */
      /*  This is so simple we might as well do it here instead of in Toker_Comment                   */
      /*  In this:                                                                                    */
      /*              say --1                                                                         */
      /*  Rexx interprets it as "say 1".                                                              */
      /*  Regina interprets it as "say ''" with a trailing comment.                                   */
      /*  so we need to check the dialect                                                             */
      /*  If the dialect is not Regina, this will detect a Regina-style comment only if               */
      /*  it is at the start of a line (or clause). That is the best I can do for now.                */
      if cc == '--' then do
            if Tkr.0dialect == 'REGINA' ,
             | T_Type(t_type.0) == 'REXX_EOC' then do
                  /*  Note that we store an empty COMMENT_END token, for the sake of formatter        */
                  /*  routines that expect normal three-part comments.                                */
                  st_val = substr(SrcIn.st_line, st_col, 2)
                  st_class = 'COMMENT'
                  st_type = 'COMMENT_BEG'
                  call Toker_Store
                  p = p +1
                  st_col = p
                  st_val = substr(SrcIn.st_line, p)
                  st_type = 'COMMENT_VAL'
                  call Toker_Store
                  st_val = ''
                  st_type = 'COMMENT_END'
                  call Toker_Store
                  st_class = 'REXX'
                  p = L_len +1
                  iterate
                  if Tkr.0dialect \= 'REGINA' then do
                        if Opt.0warn then do
                              call Msg 'Tokeniser Warning: Regina comment at line 'st_line
                              call Msg Qline(st_line)
                           end
                        if wordpos('NOTREXX', Tkr.0ExitCode) == 0 then
                           Tkr.0ExitCode = Tkr.0ExitCode 'NOTREXX'
                     end
               end
         end   /*  if '--'  */
      /* -------------------------------------------------------------------------------------------- */

      /*  at this point we know it is the start of some kind of Rexx token.                           */
      type = translate(c, Tkr.0typeset, Tkr.0charset)

      /* -------------------------------------------------------------------------------------------- */
      /*  LITERAL (in single or double quotes)                                                        */
      if type == '"' ,
       | type == "'" then do
            do forever
               c = substr(SrcIn.st_line, p, 1)
               p = p +1
               if c == type then do
                     if substr(SrcIn.st_line, p, 1) == c then
                        p = p +1                                           /* step over escaped quote */
                     else
                        leave   /*  the ending quote, leave loop  */
                  end
               if p > L_len then do
                     /*    reached end of line without finding a matching quote                       */
                     if Opt.0warn then do
                           if Opt.0progress then call Progress
                           call Msg 'Tokeniser Warning: No matching quote at line 'st_line
                           call Msg Qline(st_line)
                        end
                     if wordpos('FILE', Tkr.0ExitCode) == 0 then Tkr.0ExitCode = Tkr.0ExitCode 'FILE'
                     leave
                  end
            end   /*  forever  */
            st_val = substr(SrcIn.st_line, st_col, p - st_col)
            st_type = 'REXX_W_LITERAL'
            call Toker_Store
            iterate                                                            /* to top of main loop */
         end   /*  end if quoted string  */
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      /*  SYMBOL                                                                                      */
      /*  stems and tails will be split later in Toker_Fixup                                          */
      if type == 's' then do
            /*  advance position to a character that is not a legal symbol character                  */
            do forever
               if p > L_len then leave
               c = substr(SrcIn.st_line, p, 1)
               type = translate(c, Tkr.0typeset, Tkr.0charset)
               if verify(type, 'sn.') > 0 then leave
               p = p +1
            end
            st_val = substr(SrcIn.st_line, st_col, p - st_col)
            st_type = 'REXX_W_SYMBOL'
            call Toker_Store
            iterate
         end   /*  if type == 's'  */
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      /*  SYMBOL starting with "."    (global variables in Regina)                                    */
      /*  the "." must be followed by a symbol character. A number starting with "." is               */
      /*  handled below under CONSTANT                                                                */
      if type == '.' then do
            c2 = substr(SrcIn.st_line, p, 1)
            type2 = translate(c2, Tkr.0typeset, Tkr.0charset)
            if type2 == 's' then do
                  do forever
                     if p > L_len then leave
                     c = substr(SrcIn.st_line, p, 1)
                     type = translate(c, Tkr.0typeset, Tkr.0charset)
                     if verify(type, 'sn.') > 0 then leave
                     p = p +1
                  end
                  st_val = substr(SrcIn.st_line, st_col, p - st_col)
                  st_type = 'REXX_W_SYMBOL'
                  call Toker_Store
                  iterate
               end
         end   /*  if type == '.'  */
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      if type == 'o' then do
            c2 = substr(SrcIn.st_line, p, 1)
            /* -------------------------------------------------------------------------------------- */
            /* LOGICAL (BOOLEAN) operators; also CONCATENATION                                        */
            if pos(c, '&|') > 0 then do
                  if c \= c2 then do
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        st_type = 'REXX_W_LOGIC'
                        call Toker_Store
                        iterate
                     end
                  else do
                        p = p +1
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        if st_val == '||' then
                           st_type = 'REXX_W_CONCAT'
                        else
                           st_type = 'REXX_W_LOGIC'
                        call Toker_Store
                        iterate
                     end
               end
            /* -------------------------------------------------------------------------------------- */
            /*  unmatched COMMENT_END                                                                 */
            if c == '*' ,
             & c2 == '/' then do
                  /*  sometimes a Rexx file has an unmatched COMMENT_END string, "@/"                 */
                  /*  Rexx does not seem to care, but it can confuse the tokeniser or formatter.      */
                  /*  So recognize it here. Some places in this program assume that every             */
                  /*  comment ending has a matching beginning and a comment value, so put             */
                  /*  in empty tokens for them.                                                       */
                  p = p +1
                  if Opt.0warn then do
                        if Opt.0progress then call Progress
                        call Msg 'Tokeniser Warning: Unmatched End-Of-Comment at line 'st_line
                        call Msg Qline(st_line)
                     end
                  if wordpos('FILE', Tkr.0ExitCode) == 0 then Tkr.0ExitCode = Tkr.0ExitCode 'FILE'
                  st_class = 'COMMENT'
                  st_type = 'COMMENT_BEG'
                  st_val = ''
                  call Toker_Store
                  st_type = 'COMMENT_VAL'
                  st_val = ''
                  call Toker_Store
                  st_val = substr(SrcIn.st_line, st_col, p - st_col)
                  st_type = 'COMMENT_END'
                  call Toker_Store
                  st_class = 'REXX'
                  iterate
               end
            /* -------------------------------------------------------------------------------------- */
            /*  MATH OPERATOR, single                                                                 */
            if pos(c, '+-%') > 0 then do
                  st_val = substr(SrcIn.st_line, st_col, p - st_col)
                  st_type = 'REXX_W_MATH'
                  call Toker_Store
                  iterate
               end
            /* -------------------------------------------------------------------------------------- */
            /* MATH OPERATOR, single or double                                                        */
            if pos(c, '/'||'*') > 0 then do
                  if c \= c2 then do
                        /*  single operator                                                           */
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        st_type = 'REXX_W_MATH'
                        call Toker_Store
                        iterate
                     end
                  else do
                        /*  double operator                                                           */
                        p = p +1
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        st_type = 'REXX_W_MATH'
                        call Toker_Store
                        iterate
                     end
               end
            /* -------------------------------------------------------------------------------------- */
            /* ASSIGNMENT, single or double                                                           */
            /*  Toker_Fixup will sort out whether "=" and "==" are really COMPARE operators           */
            /*  if a Regina shortcut, Toker_Store will fix it                                         */
            if c == '=' then do
                  if c \= c2 then do
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        st_type = 'REXX_W_ASSIGN'
                        call Toker_Store
                        iterate
                     end
                  else do
                        p = p +1
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        st_type = 'REXX_W_ASSIGN'
                        call Toker_Store
                        iterate
                     end
               end
            /* -------------------------------------------------------------------------------------- */
            /* COMPARE operators (also known as CONDITIONAL operators)                                */
            if pos(c, '<>') > 0 then do
                  if pos(c2, '<>=') = 0 then do
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        st_type = 'REXX_W_COMPARE'
                        call Toker_Store
                        iterate
                     end
                  p = p +1
                  if c \= c2 then do
                        /* <= or <> or >= or ><                                                       */
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        st_type = 'REXX_W_COMPARE'
                        call Toker_Store
                        iterate
                     end
                  c2 = substr(SrcIn.st_line, p, 1)
                  if c2 \= '=' then do
                        /* << or >>                                                                   */
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        st_type = 'REXX_W_COMPARE'
                        call Toker_Store
                        iterate
                     end
                  /* <<= or >>=                                                                       */
                  p = p +1
                  st_val = substr(SrcIn.st_line, st_col, p - st_col)
                  st_type = 'REXX_W_COMPARE'
                  call Toker_Store
                  iterate
               end   /*  end if c is < or >  */
            /* -------------------------------------------------------------------------------------- */
            /*  NOT operator                                                                          */
            if pos(c, Tkr.0NOT_ops) > 0 then do
                  st_val = substr(SrcIn.st_line, st_col, p - st_col)
                  st_type = 'REXX_W_NOT'
                  call Toker_Store
                  iterate
               end
            /* -------------------------------------------------------------------------------------- */
            /*  an unexpected operator; this should never happen                                      */
            call ShowError 'Tokeniser warning: unexpected operator character: "'c'" at line' st_line
            call ShowError Qline(st_line), ''
            if wordpos('BUG', Tkr.0ExitCode) == 0 then Tkr.0ExitCode = Tkr.0ExitCode 'BUG'
            st_val = substr(SrcIn.st_line, st_col, p - st_col)
            st_type = 'REXX_UNKNOWN'
            call Toker_Store
            iterate
         end   /*  end if type == 'o'  */
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      /*  CONSTANT                                                                                    */
      /*  constant starts with number or '.' (type '.'), but can include certain type 's' characters  */
      /*  a variable starting with "." was done above at SYMBOL                                       */
      if type == 'n' ,
       | type == '.' then do
            do forever
               if p > L_len then leave
               c = substr(SrcIn.st_line, p, 1)
               type = translate(c, Tkr.0typeset, Tkr.0charset)
               if verify(type, 'sn.') > 0 then leave
               p = p +1
            end
            st_val = substr(SrcIn.st_line, st_col, p - st_col)
            st_type = 'REXX_W_CONSTANT'
            call Toker_Store
            iterate
         end   /*  if type == 'n'  */
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      /*  COLON                                                                                       */
      if type == ':' then do
            c2 = substr(SrcIn.st_line, p, 1)
            if c == ':'then do
                  if c \= c2 then do
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        st_type = 'REXX_COLON'
                        call Toker_Store
                        iterate
                     end
                  else do
                        /*  the Toker does not do Object Rexx, but we recognize a '::'                */
                        /*  to avoid a cascade of error messages                                      */
                        p = p +1
                        st_val = substr(SrcIn.st_line, st_col, p - st_col)
                        st_type = 'COLON2'
                        if Opt.0warn then
                           if \Tkr.0ORexxWarned then do
                                 call Msg 'Tokeniser Warning: Object Rexx Directive at line 'st_line
                                 call Msg Qline(st_line)
                                 Tkr.0ORexxWarned = 1
                              end
                        if wordpos('NOTREXX', Tkr.0ExitCode) == 0 then
                           Tkr.0ExitCode = Tkr.0ExitCode 'NOTREXX'
                        call Toker_Store
                        iterate
                     end
               end   /*  if c == ':'  */
         end   /*  if type == ':'  */
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      /*  OTHER                                                                                       */
      /*  should be ";", ",", "(", ")"                                                                */
      st_val = substr(SrcIn.st_line, st_col, p - st_col)
      if length(st_val) \== 1 then do
            call ShowError 'BUG: unexpected value for st_val "'st_val'" in line' st_line
            call ShowError Qline(st_line), ''
            call ShowError 'This indicates a Tokenisr bug involving the "st_col" and "p" variables', ''
            if wordpos('BUG', Tkr.0ExitCode) == 0 then
               Tkr.0ExitCode = Tkr.0ExitCode 'BUG'
         end

      select
         when st_val = ';' then st_type = 'REXX_EOC'
         when st_val = ',' then st_type = 'REXX_W_COMMA'
         when st_val == '1A'x then do
               /*  bugfix-- in OS/2 Rexx, the linein() function treats an EOF character               */
               /*  ('1A'x, Ctrl-Z) like an EOL, but Regina's linein() treats it as a                  */
               /*  printable character. If one is found, convert it to EOL.                           */
               st_class = 'FILE_EOL'
               st_type = 'REXX_EOC'
               st_val = ''
               SrcIn.st_line = strip(SrcIn.st_line, , '1A'x)
            end
         otherwise
            /*  the only other possibilities should be "(" and ")". Anything else is                  */
            /*  probably an error in the file                                                         */
            st_type = 'REXX_W_'||st_val
            if verify(type, '();,') > 0 then do
                  st_type = 'REXX_UNKNOWN'
                  if wordpos('FILE', Tkr.0ExitCode) == 0 then
                     Tkr.0ExitCode = Tkr.0ExitCode 'FILE'
                  if Opt.0warn then do
                        call Msg 'Tokeniser warning:' ,
                             'Unexpected token "'st_val'" at line' st_line
                        call Msg Qline(st_line)
                     end
               end
      end   /*  select  */
      call Toker_Store
      iterate
      /* -------------------------------------------------------------------------------------------- */
   end   /*  of outer DO forever loop  */

   if Opt.0progress then call Progress
   return Tkr.0ExitCode
   /* === End of Toker_Rexx ========================================================================= */

   Toker_Comment: /*fold01*/
   /* =============================================================================================== */
   /*  This is a special purpose comment tokeniser. Functionally it is just a block of code           */
   /*  within Toker_Rexx and shares all of its variables.                                             */
   /*                                                                                                 */
   /*  Toker_Rexx does not store comment tokens or care about comments at all.                        */
   /*     Later: but Toker_Rexx now does the Regina-style comments                                    */
   /*  When it sees the start of a comment it will just pass control to here. This sub will           */
   /*  store the COMMENT_BEG token, then read and store tokens until the end of the comment.          */
   /*  It stores the COMMENT_END token and returns control to Toker_Rexx.                             */
   /*  Variables like 'st_line', 'p', 'c', etc. are updated so Toker_Rexx can continue.               */
   /* =============================================================================================== */
   incomment = 0                                                             /* comment nesting level */
   do forever
      st_class = 'COMMENT'
      cc = substr(SrcIn.st_line, p, 2)
      p = p +1
      p2 = p +1
      st_val = ''
      if debug.0loop then call LoopCounter

      if st_line > SrcIn.0  then do
            call ShowError 'BUG: in Toker_Comment, st_line > SrcIn.0'
            call ShowError 'This should never happen', ''
            if wordpos('BUG', Tkr.0ExitCode) == 0 then Tkr.0ExitCode = Tkr.0ExitCode 'BUG'
            incomment = 0
            return 0
         end

      if p > L_len then do
            /*   EOL within a multiline comment. store the Comment, which may be ''; then store EOL   */
            /*  note that the type is COMMENT_EOL, because it will not be seen by Rexx. Unlike        */
            /*  the End-Of-Line between two comments, which counts as a Rexx End-Of-Clause.           */
            st_val = substr(SrcIn.st_line, st_col)
            st_type = 'COMMENT_VAL'
            call Toker_Store

            if st_line == SrcIn.0 then do
                  /*  Unexpected End of File while still in a comment. Add a dummy End-Of-Comment     */
                  /*  because some things expect comments to have three parts                         */
                  /*  Then leave; Toker_Rexx will handle the EOL                                      */
                  st_val = ''
                  st_col = L_len +1
                  st_type = 'COMMENT_END'
                  call Toker_Store
                  p = p +1
                  st_col = p
                  incomment = 0
                  if Opt.0warn then do
                        call Msg 'Tokeniser Warning: unexpected End Of File before end of comment'
                        call Msg '                   This is usually because of a missing "'||'*'||'/'||'"'
                     end
                  if wordpos('FILE', Tkr.0ExitCode) == 0 then Tkr.0ExitCode = Tkr.0ExitCode 'FILE'
                  return 0
               end
            else do
                  st_val = ''
                  st_col = L_len +1
                  st_type = 'COMMENT_EOL'
                  st_class = 'FILE_EOL'
                  call Toker_Store
                  st_val = ''
                  st_line = st_line +1
                  L_len = length(SrcIn.st_line)
                  p = 1
                  st_col = p
                  iterate
               end
         end   /*  if p > L_len  */

      if cc == '/'||'*' then do
            /*  a nested comment is not a token, but we track the nesting level                       */
            if incomment = 0 then do
                  /*  the "/@" that begins a comment can have a prefix, which Toker_Rexx has          */
                  /*  already saved in st_prefix.                                                     */
                  st_val = substr(SrcIn.st_line, st_col, p2 - st_col)
                  st_type = 'COMMENT_BEG'
                  call Toker_Store
                  p = p +1
                  st_col = p
               end
            else do
                  /*  bump p to end of cc as if we were at the end of a token, but keep st_col        */
                  /*  because this is not really the end of the comment token.                        */
                  /*  (fixes bug caused by nested comments with adjacent @/@/ )                       */
                  p = p +1
               end
            incomment = incomment +1
            iterate
         end

      if cc == '*'||'/' then do
            incomment = incomment -1
            select
               when incomment < 0 then do
                     st_val = substr(SrcIn.st_line, st_col, p2 - st_col)
                     st_type = 'COMMENT_END'
                     call Toker_Store
                     p = p +1
                     st_col = p
                     incomment = 0
                     return 0
                  end
               when incomment = 0 then do
                     /* end of the comment; store Comment; store COMMENT_END; return to main Toker    */
                     p = p -1
                     st_val = substr(SrcIn.st_line, st_col, p - st_col)
                     st_type = 'COMMENT_VAL'
                     call Toker_Store
                     st_col = p
                     p = p2
                     st_val = substr(SrcIn.st_line, st_col, p2 - st_col)
                     st_type = 'COMMENT_END'
                     call Toker_Store
                     st_col = p
                     return 0
                  end
               otherwise
                  /*  bump p to end of cc as if we were at the end of a token, but keep st_col        */
                  /*  because this is not really the end of the comment token.                        */
                  /*  (fixes bug caused by nested comments with adjacent @/@/ )                       */
                  p = p +1
            end   /*  select  */
         end   /*  end if cc == end of comment  */
   end   /*  end of DO forever  */
   signal RunoutHandler
   /* === End of Toker_Comment ====================================================================== */

   Toker_Store: /*fold01*/
   /* =============================================================================================== */
   /*  This is part of the Tokeniser. It is called by Toker_Rexx and Toker_Comment to                 */
   /*  store a token in the arrays.                                                                   */
   /*                                                                                                 */
   /*  Some tokens must be "fixed up" because Toker_Rexx cannot correctly identify them               */
   /*  by looking at just one token. Usually because it requires looking ahead to                     */
   /*  tokens that have not been parsed yet.                                                          */
   /*  Most fixups are done by calling Toker_Fixup at the end of each clause.                         */
   /* =============================================================================================== */
   /*  store this token in all arrays:                                                                */
   h = t_type.0 +1
   t_class.h    = st_class        ; t_class.0 = h
   t_type.h     = st_type         ; t_type.0 = h
   t_line.h     = st_line         ; t_line.0 = h
   t_col.h      = st_col          ; t_col.0 = h
   t_prefix.h   = st_prefix       ; t_prefix.0 = h
   t_val.h      = st_val          ; t_val.0 = h
   /*  once the token is stored, we know its prefix is invalid. Toker manages the                     */
   /*  other st_xxx variables. Not sure why I did it this way.                                        */
   st_prefix = ''
   if Opt.0progress then
      if t_line.h // 100 = 0 then
         call Progress G.0meID' stored token '||h||' from line '||st_line' of 'SrcIn.0
   /* ----------------------------------------------------------------------------------------------- */
   /*  avoid error if there is no previous token                                                      */
   /*  also, the LastEOC might as well be initialized here as anywhere else                           */
   if h == 1 then do
         Tkr.0LastEOC = 1
         return
      end
   /* ----------------------------------------------------------------------------------------------- */
   /*  fixup for CONTINUED LINE:                                                                      */
   /*  an EOL is nullified by a preceding comma                                                       */
   /*  R_PrevToken ignores intervening comment tokens                                                 */
   if t_class.h == 'FILE_EOL' then do
         if T_Type(R_PrevToken(h)) == 'REXX_W_COMMA' then do
               t_type.h = 'NULL'
               call T_Type R_PrevToken(h), 'CONTINUE'
            end
      end
   /* ----------------------------------------------------------------------------------------------- */
   /*  fixup for Regina SHORTCUT assignment operators                                                 */
   if t_type.h == 'REXX_W_ASSIGN' then
      if t_val.h == '=' then do
            prev = h -1
            if wordpos(t_val.prev, '+ - * / % // | & && ||') > 0 then do
                  t_val.prev = t_val.prev||t_val.h
                  t_type.prev = 'REXX_W_ASSIGN'
                  call Toker_DeleteToken h
               end
         end
   /* ----------------------------------------------------------------------------------------------- */
   /*  see comment at start of Toker_Fixup                                                            */
   if t_type.h == 'REXX_EOC' then do
         call Toker_Fixup
         Tkr.0LastEOC = t_type.0
      end
   return
   /* === End of Toker_Store ======================================================================== */

   Toker_Fixup: procedure expose (Shared_vars) (all_tokens) Tkr. /*fold01*/
   /* =============================================================================================== */
   /*  called by Toker_Store at the end of each clause to do some fixups that involve                 */
   /*  inserting or deleting tokens. StemInsert/StemDelete are very slow if they must                 */
   /*  shift many elements in a large array, so we do this while there are only a few                 */
   /*  elements to shift.                                                                             */
   /*  These things must wait until the end of a clause because they must look ahead                  */
   /*  to examine subsequent tokens.                                                                  */
   /*  Keywords are also identified here because inserting hidden EOCs depends on them                */
   /* =============================================================================================== */

   /* =============================================================================================== */
   /*  FIRST LOOP                                                                                     */
   /*  identify keywords and fix hex/binary constants, then insert hidden EOCs where needed           */
   t = Tkr.0LastEOC
   do while t < t_type.0
      t = t +1
      if t_class.t \== 'REXX' then iterate
      Tval = translate(t_val.t)
      next = R_NextToken(t)
      select
         /* ----------------------------------------------------------------------------------------- */
         /*  KEYWORD:                                                                                 */
         /*  if token t looks like a keyword, and is at the beginning of a clause, but is             */
         /*  not at an assignment, and is not label, then it is a real keyword, unless I              */
         /*  overlooked something.                                                                    */
         /*  Sub-keywords can be identified later, after all the tokens are stored                    */
         when wordpos(Tval, Tkr.0Keywords) > 0 ,
            & T_Type(R_PrevToken(t)) == 'REXX_EOC' ,
            & T_Type(R_NextWord(t)) \= 'REXX_W_ASSIGN' ,
            & t_val.next \= ':' then do
               t_type.t = 'REXX_W_KEY'
            end
         /* ----------------------------------------------------------------------------------------- */
         /*  THEN:                                                                                    */
         /*  IF a == b THEN c = d                                                                     */
         /*  May not have been caught above because THEN sometimes has no preceding EOC, and          */
         /*  the hidden REXX_EOC tokens have not been added yet. Rule out a variable named            */
         /*  THEN-- a keyword THEN must be after an IF or WHEN clause.                                */
         /*  Note that if THEN has no preceding EOC, CL_FirstWord() will see THEN as part             */
         /*  of the IF clause                                                                         */
         when Tval == 'THEN' then do
               testW = translate(T_Val(CL_FirstWord(R_PrevWord(t))))
               if wordpos(testW, 'IF WHEN') > 0 ,
                & t_val.next \= ':' then
                  t_type.t = 'REXX_W_KEY'
            end
         /* ----------------------------------------------------------------------------------------- */
         /*  ELSE:                                                                                    */
         /*  if a == b THEN c = d ELSE c = e                                                          */
         /*  May not have been caught above because ELSE sometimes has no preceding EOC, and          */
         /*  the hidden REXX_EOC tokens have not been added yet. Rule out a variable named            */
         /*  ELSE-- a keyword ELSE must be after a THEN clause.                                       */
         /*  Note that it uses CL_FirstWord(x, -1) == THEN because THEN has now                       */
         /*  had hidden EOCs added and therefore is the preceding clause.                             */
         when Tval == 'ELSE' then do
               if translate(T_Val(CL_FirstWord(R_PrevWord(t), -1))) == 'THEN' ,
                & t_val.next \= ':' then do
                     t_type.t = 'REXX_W_KEY'
                  end
            end   /*  when ELSE  */
         /* ----------------------------------------------------------------------------------------- */
         /*  HEX or BINARY CONSTANTS                                                                  */
         /*  the Toker will identify hex or binary numbers like                                       */
         /*          'FF'x    or   '10101010'b                                                        */
         /*  as a LITERAL token and a SYMBOL token.                                                   */
         /*  It is more convenient to think of it as one token with type of REXX_W_CONSTANT.          */
         /*                                                                                           */
         /*  use verify() == 0 instead of datatype() because datatype() allows spaces in hex          */
         /*  strings (at byte boundaries) but not in a binary string even when a space is legal--     */
         /*              SAY '0101 0010'B                      yields "R", but                        */
         /*              SAY datatype('0101 0010', 'B')        yields "0"                             */
         /*  We do not need to validate the string, just decide what the user intended.               */
         when t_type.t == 'REXX_W_LITERAL' ,
            & t_type.next == 'REXX_W_SYMBOL' ,
            & t_prefix.next == '' then do
               select
                  when verify(translate(Unquoted(t_val.t)), ' 0123456789ABCDEF') == 0 ,
                     & translate(t_val.next) == 'X' then do
                        t_type.t = 'REXX_W_CONSTANT'
                        t_val.t = t_val.t||t_val.next
                        call Toker_DeleteToken next
                     end   /*  when  */
                  when verify(Unquoted(t_val.t), ' 01') == 0 ,
                     & translate(t_val.next) == 'B' then do
                        t_type.t = 'REXX_W_CONSTANT'
                        t_val.t = t_val.t||t_val.next
                        call Toker_DeleteToken next
                     end   /*  when  */
                  otherwise nop
               end   /*  select  */
            end   /*  when t_type.t == LITERAL  */
         /* ----------------------------------------------------------------------------------------- */
         /*  EXPONENTIAL CONSTANTS                                                                    */
         /*  Toker_Rexx will see "12345E04" as one CONSTANT token-- "12345E04"                        */
         /*  but a + or - will make it see "12345E+04" as three tokens-- "12345E" "+" "04"            */
         /*  combine the three into one CONSTANT token                                                */
         when t_type.t == 'REXX_W_CONSTANT' ,
            & right(Tval, 1) == 'E' then do
               if t < t_type.0 -1 then do
                     if t_type.next == 'REXX_W_MATH' ,
                      & t_prefix.next == '' ,
                      & T_Type(t +2) == 'REXX_W_CONSTANT' ,
                      & T_Prefix(t +2) == '' then do
                           t_val.t = t_val.t||t_val.next||T_Val(t +2)
                           /*  note the deletion order is from highest to lowest. Otherwise           */
                           /*  'next' would be beyond the valid range of token numbers                */
                           call Toker_DeleteToken t +2
                           call Toker_DeleteToken next
                        end
                  end
            end   /*  if CONSTANT  */
         /* ----------------------------------------------------------------------------------------- */
         otherwise nop
      end   /*  select token t  */
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      /*  HIDDEN EOC:                                                                                 */
      /*  some words need hidden End-Of-Clause tokens inserted. What I call a hidden EOC              */
      /*  is the optional ";" or End-Of-Line that is implied in certain places.                       */
      /*  For instance,                                                                               */
      /*        IF a THEN DO                                                                          */
      /*  is a shorter way of writing                                                                 */
      /*        IF a ; THEN ; DO                                                                      */
      /*  or                                                                                          */
      /*        IF a                                                                                  */
      /*        THEN                                                                                  */
      /*        DO                                                                                    */
      /*  adding the implied EOC tokens will make it easier for later tasks to identify clauses       */
      /* -------------------------------------------------------------------------------------------- */
      if wordpos(translate(t_val.t), 'THEN ELSE OTHERWISE :') > 0 then do
            EOC_AddBefore = 0 ; EOC_AddAfter = 0
            select
               when t_type.t == 'REXX_W_KEY' then do
                     select
                        when Tval == 'THEN' then do
                              if abbrev(T_Type(R_PrevToken(t)), 'REXX_W') then
                                 EOC_AddBefore = 1
                              if abbrev(T_Type(R_NextToken(t)), 'REXX_W') then
                                 EOC_AddAfter = 1
                           end
                        when Tval == 'ELSE' then
                           if abbrev(T_Type(R_NextToken(t)), 'REXX_W') then
                              EOC_AddAfter = 1
                        when Tval == 'OTHERWISE' then
                           if abbrev(T_Type(R_NextToken(t)), 'REXX_W') then
                              EOC_AddAfter = 1
                        otherwise nop
                     end   /*  select  */
                  end
               when t_type.t == 'REXX_COLON' then
                  if abbrev(T_Type(R_NextToken(t)), 'REXX_W') then
                     EOC_AddAfter = 1
               otherwise nop
            end   /*  select  */

            if EOC_AddBefore then do
                  call Toker_InsertToken t
                  /*  keep next pointing to the last token                                            */
                  next = next +1
                  /*  t now points to the new token, shift t to the keyword again                     */
                  /*  and fill in the blanks                                                          */
                  t = t +1
                  prev = t -1
                  t_class.prev = 'REXX'
                  t_type.prev = 'REXX_EOC'
                  t_line.prev = t_line.t
                  t_col.prev = t_col.t
                  t_prefix.prev = ''
                  t_val.prev = ''
               end

            if EOC_AddAfter then do
                  call Toker_InsertToken t +1
                  next = t +1
                  t_class.next = 'REXX'
                  t_type.next = 'REXX_EOC'
                  t_line.next = t_line.t
                  t_col.next = t_col.t + length(t_val.t)
                  t_prefix.next = ''
                  t_val.next = ''
               end
         end   /*  if THEN ELSE OTHERWISE  */
   end   /*  do while t < t_type.0  */
   /* =============================================================================================== */

   /* =============================================================================================== */
   /*  SECOND LOOP                                                                                    */
   /* ----------------------------------------------------------------------------------------------- */
   /*  SPLIT STEM VARIABLES:                                                                          */
   /*  This splits a stem variable into its component tokens.                                         */
   /*                                                                                                 */
   /*  This is the kind of unlikely possibility that needs to be dealt with:                          */
   /*                                                                                                 */
   /*        X = Function.Name.with.Dots/@ a comment @/()                                             */
   /*                                                                                                 */
   /*  A function name can have dots in it, and the comment between the name and                      */
   /*  the left parenthesis is legal. So the stem-splitter must look ahead to find the "(",           */
   /*  to recognize that the string "Function.Name.with.Dots" is a function name and                  */
   /*  should not be split.                                                                           */
   /*                                                                                                 */
   /*  This should be after everything in the first loop, because they were written                   */
   /*  earlier and might not understand split stem variables                                          */
   t = Tkr.0LastEOC
   do while t < t_type.0
      t = t +1
      prev = R_PrevWord(t)
      if t_type.t == 'REXX_W_SYMBOL' then do
            if pos('.', t_val.t) > 1 then do
                  w1 = CL_FirstWord(t)
                  /*  exclude anything that is not a stem variable:                                   */
                  select
                     when T_Type(t +1) == 'REXX_COLON' then nop
                     /*  Toker_AtFunction may return a false positive if token t                      */
                     /*  is really a sub-keyword that has not been identified yet.                    */
                     /*  That's okay here; a sub-keyword should be excluded also.                     */
                     when Toker_AtFunction(t) then nop
                     when wordpos(translate(t_val.prev), 'CALL SIGNAL') > 0 then nop
                     /*  bugfix for "signal on error name Label.With.Dots"                            */
                     when wordpos(R_KeyVal(w1), 'CALL SIGNAL') > 0 ,
                        & translate(t_val.prev) == 'NAME' then nop
                     otherwise
                        Tval = t_val.t
                        do while pos('.', Tval) > 0
                           stem = substr(Tval, 1, pos('.', Tval) -1)
                           tail = strip(substr(Tval, pos('.', Tval) +1))
                           /*  cut this token down to the stem                                        */
                           t_val.t = stem
                           /*  the dot is a separate token                                            */
                           d = t +1
                           call Toker_InsertToken d
                           t_class.d = t_class.t
                           t_type.d = 'REXX_W_SYMBOL'
                           t_line.d = t_line.t
                           t_col.d = t_col.t + length(t_val.t)
                           t_prefix.d = ''
                           t_val.d = '.'
                           /*  if no tail, then Tval was just "stem." ; nothing more to do            */
                           if tail == '' then do
                                 t = d
                                 leave
                              end

                           /*  make a new token from the tail, copying attributes from the original,  */
                           /*  but adjusting prefix and column                                        */
                           n = d +1
                           call Toker_InsertToken n
                           t_class.n = t_class.t
                           t_type.n = t_type.t
                           t_line.n = t_line.t
                           t_col.n = t_col.t + length(t_val.t) + length(t_val.d)
                           t_prefix.n = ''
                           t_val.n = tail
                           /*  get ready to split the tail again, if needed                           */
                           t = n
                           Tval = tail
                        end   /*  do while pos  */
                  end   /*  select  */
               end   /*  if pos()  */
         end   /*  if SYMBOL  */
   end   /*  while t < t_type.0  */
   /* =============================================================================================== */

   /* =============================================================================================== */
   /*  THIRD LOOP                                                                                     */
   do t = Tkr.0LastEOC to t_type.0 -1
      /* -------------------------------------------------------------------------------------------- */
      /*  REXX_W_ASSIGN to REXX_W_COMPARE                                                             */
      /*  if token t is "=" and not really used for assignment, change its type.                      */
      /*  note the order under SELECT; a "==" or "=" is a comparison only if it already               */
      /*  failed the Toker_IsAssignment test                                                          */
      if t_type.t == 'REXX_W_ASSIGN' then do
            select
               when Toker_IsAssignment(t) then
                  nop
               when t_val.t == '==' ,
                  | t_val.t == '=' then
                  t_type.t = 'REXX_W_COMPARE'
               otherwise nop
            end   /*  select  */
         end   /*  if REXX_W_ASSIGN  */
      /* -------------------------------------------------------------------------------------------- */
      /*  CONSTANT to SYMBOL                                                                          */
      /*  akm: leading numbers and dots are not legal in variable names, but Rexx accepts them        */
      /*  in labels and function names. These are legal uses of label names:                          */
      /*               123:                                                                           */
      /*               CALL .123                                                                      */
      /*               SAY .123()                                                                     */
      /*               SAY .123/@ a comment @/()                                                      */
      /*  the Toker misidentifies those names as CONSTANT.  The only way to know is to look           */
      /*  at the previous and next Rexx tokens                                                        */
      /*                                                                                              */
      /*  these will be changed from SYMBOL to PROCEDURE in the normal processing below               */
      if t_type.t == 'REXX_W_CONSTANT' then do
            prev = R_PrevWord(t)
            next = R_NextWord(t)
            select
               when t_type.prev == 'REXX_W_KEY' ,
                  & wordpos(translate(t_val.prev), 'CALL SIGNAL') > 0 then
                  /*  "call 123"                                                                      */
                  t_type.t = 'REXX_W_SYMBOL'
               when t_type.next == 'REXX_W_(' then
                  /*  "123()"                                                                         */
                  t_type.t = 'REXX_W_SYMBOL'
               when t_type.next == 'REXX_COLON' then
                  /*  "123:"                                                                          */
                  t_type.t = 'REXX_W_SYMBOL'
               when T_Type(t -1) == 'REXX_EOC' then
                  /*  "123"                                                                           */
                  t_type.t = 'REXX_W_SYMBOL'
               otherwise
                  nop
            end   /*  select  */
         end   /*  if a constant  */
      /* -------------------------------------------------------------------------------------------- */

      /* -------------------------------------------------------------------------------------------- */
      /*  LABEL                                                                                       */
      /*  if a symbol is followed by a colon, it must be a label                                      */
      if t_type.t == 'REXX_COLON' then do
            prev = R_PrevToken(t)
            select
               when (t_type.prev = 'REXX_W_SYMBOL' ,
                  | t_type.prev = 'REXX_W_CONSTANT') then
                  t_type.prev = 'REXX_LABEL'
               otherwise
                  /*  this should never happen in valid Rexx code, but it can if the file has a       */
                  /*  mistaken cmd.exe/4OS2 style label, where t_val.prev == '', or if the label      */
                  /*  has an illegal character.                                                       */
                  if Opt.0warn then do
                        call Msg 'Tokeniser Warning:'
                        'Error identifying "'t_val.prev'" as label at line 't_line.t
                        call Msg Qline(t_line.t)
                     end
                  if wordpos('FILE', Tkr.0ExitCode) == 0 then Tkr.0ExitCode = Tkr.0ExitCode 'FILE'
            end   /*  select  */
         end   /*  if COLON  */
      /* -------------------------------------------------------------------------------------------- */
   end t
   return 0
   /* === End of Toker_Fixup ======================================================================== */

   Toker_SubKeywords: procedure expose (Shared_vars) (all_tokens) Tkr. /*fold01*/
   /* =============================================================================================== */
   /*  This is the part of the Toker that identifies sub-keywords. The keywords are                   */
   /*  identified in Toker_Fixup                                                                      */
   /* ----------------------------------------------------------------------------------------------- */
   /*  Words like 'To', 'By', and 'Forever' can sometimes be used as variable names,                  */
   /*  so we need to parse each clause to accurately identify sub-keywords.                           */
   /*                                                                                                 */
   /*  The calls to loopcounter() are always on here, because there might be some kind                */
   /*  of defective code that this routine does not anticipate.                                       */
   /*                                                                                                 */
   /*  fix: there may be bugs in this section; it needs a sample file to                              */
   /*        systematically test all these keywords                                                   */
   /* =============================================================================================== */
   if Opt.0progress then call Progress SubID()

   t = 0
   do while t < t_type.0
      t = t +1
      call LoopCounter
      if t_type.t \= 'REXX_W_KEY' then iterate

      i = t
      keyW = translate(t_val.i)
      select   /*  value of t_val.i  */
         /* ----------------------------------------------------------------------------------------- */
         when keyW = 'ADDRESS' then do
               /*  fix: I do not use ADDRESS much, so this may be incomplete                          */
               i = R_NextToken(i)
               select
                  when t_type.i == 'REXX_EOC' then nop
                  when t_type.i == 'REXX_W_SYMBOL' ,
                     & translate(t_val.i) == 'VALUE' then do
                        t_type.i = 'REXX_W_KEY2'
                     end
                  otherwise
                     if t_type.i == 'REXX_W_SYMBOL' then
                        t_type.i = 'REXX_W_CONSTANT'
               end   /*  select  */
               if Tkr.0dialect == 'REGINA' then
                  do while t_type.i \== 'REXX_EOC'
                     if wordpos(translate(t_val.i), ,
                        'WITH INPUT NORMAL NOEOL OUTPUT APPEND REPLACE ERROR STREAM STEM LIFO FIFO') > 0 then
                        t_type.i = 'REXX_W_KEY2'
                     i = i +1
                  end
            end   /*  when ADDRESS  */
         /* ----------------------------------------------------------------------------------------- */
         when keyW = 'CALL' then do
               /*  see comments under SIGNAL                                                          */
               i = R_NextToken(i)
               select
                  when translate(t_val.i) = 'OFF' then do
                        /*  "call off Error"                                                          */
                        t_type.i = 'REXX_W_KEY2'
                        i = i +1
                        t_type.i = 'REXX_W_KEY2'
                     end
                  when translate(t_val.i) = 'ON' then do
                        t_type.i = 'REXX_W_KEY2'
                        nn = i +2
                        if translate(t_val.nn) = 'NAME' then do
                              /*  as in "call on error name Errorhandler"                             */
                              i = i +1
                              t_type.i = 'REXX_W_KEY2'
                              i = i +1
                              t_type.i = 'REXX_W_KEY2'
                              i = i +1
                              t_type.i = 'REXX_W_PROCEDURE'
                           end
                        else do
                              /*  as in "call on error"                                               */
                              i = R_NextToken(i)
                              t_type.i = 'REXX_W_KEY2'
                           end
                     end   /*  when ON  */
                  when translate(t_val.i) = 'VALUE' then do
                        t_type.i = 'REXX_W_KEY2'
                     end
                  otherwise
                     /*  "call LabelA"                                                                */
                     t_type.i = 'REXX_W_PROCEDURE'
               end   /*  select  */
            end   /*  when CALL  */
         /* ----------------------------------------------------------------------------------------- */
         when keyW = 'DO' then do
               i = R_NextToken(i)
               op = R_NextWord(R_Tail(i))
               if T_Val(R_NextWord(R_Tail(i))) == '=' then
                  /*  i is a variable being assigned, skip it                                         */
                  i = op
               do forever
                  select
                     when t_type.i == 'REXX_EOC' then leave
                     when t_type.i == 'REXX_W_SYMBOL' ,
                        & wordpos(translate(t_val.i), 'TO BY FOR FOREVER UNTIL WHILE') > 0 then
                        t_type.i = 'REXX_W_KEY2'
                     otherwise nop
                  end   /*  select  */
                  i = R_NextToken(i)
               end   /*  do forever  */

            end   /*  when DO  */
         /* ----------------------------------------------------------------------------------------- */
         when keyW = 'NUMERIC' then
            do forever
               call LoopCounter
               i = R_NextToken(i)
               if t_type.i == 'REXX_EOC' then
                  leave
               else
                  if t_type.i = 'REXX_W_SYMBOL' ,
                   & wordpos(translate(t_val.i), 'DIGITS FORM SCIENTIFIC ENGINEERING VALUE FUZZ') > 0 then
                     t_type.i = 'REXX_W_KEY2'
            end
         /* ----------------------------------------------------------------------------------------- */
         /*  fix: test the parse section better                                                       */
         when keyW = 'PARSE' then do
               i = R_NextToken(i)
               if wordpos(translate(t_val.i), 'UPPER LOWER CASELESS') > 0 then do
                     t_type.i = 'REXX_W_KEY2'
                     i = R_NextToken(i)
                  end
               select
                  when wordpos(translate(t_val.i), 'ARG PULL SOURCE VAR VERSION LINEIN') > 0 then
                     t_type.i = 'REXX_W_KEY2'
                  when translate(t_val.i) = 'VALUE' then do
                        t_type.i = 'REXX_W_KEY2'
                        do forever
                           /*  avoid endless loop if "WITH" is missing                                */
                           if t_type.i == 'REXX_EOC' then do
                                 if wordpos('FILE', Tkr.0ExitCode) == 0 then
                                    Tkr.0ExitCode = Tkr.0ExitCode 'FILE'
                                 if Opt.0warn then do
                                       call Msg 'Tokeniser warning: PARSE VALUE without WITH, line 't_line.t
                                       call Msg Qline(t_line.t)
                                    end
                                 leave
                              end
                           i = R_NextToken(i)
                           if t_type.i = 'REXX_W_SYMBOL' ,
                            & translate(t_val.i) = 'WITH' then do
                                 t_type.i = 'REXX_W_KEY2'
                                 leave
                              end
                        end
                     end   /*  if t_val.i == VALUE  */
                  otherwise
                     if wordpos('FILE', Tkr.0ExitCode) == 0 then
                        Tkr.0ExitCode = Tkr.0ExitCode 'FILE'
                     if Opt.0warn then do
                           call Msg 'Tokeniser parsing error of PARSE, after line 't_line.t
                           call Msg '   unknown sub-keyword' '"'t_val.i'"'
                           call Msg '  ' Qline(t_line.i)
                        end
               end   /*  select  */
            end   /*  when PARSE  */
         /* ----------------------------------------------------------------------------------------- */
         when keyW = 'PROCEDURE' then do
               G.0HasProcedures = 1
               i = R_NextToken(i)
               if translate(t_val.i) = 'EXPOSE' then do
                     t_type.i = 'REXX_W_KEY2'
                  end
            end
         /* ----------------------------------------------------------------------------------------- */
         when keyW = 'SIGNAL' then do
               i = R_NextToken(i)
               select
                  when translate(t_val.i) = 'OFF' then do
                        /*  "signal off Error"                                                        */
                        t_type.i = 'REXX_W_KEY2'
                        i = i +1
                        t_type.i = 'REXX_W_KEY2'
                     end

                  when translate(t_val.i) = 'ON' then do
                        t_type.i = 'REXX_W_KEY2'
                        nn = i +2
                        if translate(t_val.nn) = 'NAME' then do
                              /*  as in "signal on error name Errorhandler"                           */
                              i = i +1
                              t_type.i = 'REXX_W_KEY2'
                              i = i +1
                              t_type.i = 'REXX_W_KEY2'
                              i = i +1
                              t_type.i = 'REXX_W_PROCEDURE'
                           end
                        else do
                              /*  as in "signal on error"                                             */
                              i = R_NextToken(i)
                              t_type.i = 'REXX_W_KEY2'
                           end
                     end   /*  when ON  */

                  when translate(t_val.i) = 'VALUE' then do
                        t_type.i = 'REXX_W_KEY2'
                     end
                  otherwise
                     /*  "signal LabelA"                                                              */
                     t_type.i = 'REXX_W_PROCEDURE'
               end   /*  select  */
            end   /*  when SIGNAL  */
         /* ----------------------------------------------------------------------------------------- */
         when keyW == 'TRACE' then do
               i = R_NextToken(i)
               if translate(t_val.i) = 'VALUE' then do
                     t_type.i = 'REXX_W_KEY2'
                  end
               else
                  if t_type.i == 'REXX_W_SYMBOL' then
                     t_type.i = 'REXX_W_CONSTANT'
            end   /*  when TRACE  */
         /* ----------------------------------------------------------------------------------------- */
         when keyW == 'USE' then do
               /*  "use arg" is an Object Rexx command                                                */
               if wordpos('NOTREXX', Tkr.0ExitCode) == 0 then
                  Tkr.0ExitCode = Tkr.0ExitCode 'FILE NOTREXX'
               if Opt.0warn then do
                     if \Tkr.0ORexxWarned then do
                           call Msg 'Tokeniser Warning: Object Rexx keyword at line 't_line.i
                           call Msg Qline(t_line.i)
                           Tkr.0ORexxWarned = 1
                        end
                  end
            end
         /* ----------------------------------------------------------------------------------------- */
         otherwise
            /*  keywords that do not use sub-keywords                                                 */
            nop
            /* -------------------------------------------------------------------------------------- */
      end   /*  select t_val.i  */
      t = i
   end   /*  while t < t_type.0  */
   if Opt.0progress then call Progress
   return 0
   /* === End of Toker_SubKeywords ================================================================== */

   Toker_IDfunctions: procedure expose (Shared_vars) (all_tokens) Tkr. /*fold01*/
   /* =============================================================================================== */
   /*  bugfix-- identifying functions was moved to here, after identification of sub-keywords,        */
   /*  because it is less prone to errors. See comments in Toker_AtFunction                           */
   /* ----------------------------------------------------------------------------------------------- */
   /*  FUNCTION                                                                                       */
   /*  identify function names and give them a special type:                                          */
   /*  look for "<token>("                                                                            */
   /*  there can even be a comment between the symbol and '(', but not a space                        */
   /*                                                                                                 */
   /*  ==> note that if token t is a LITERAL (in quotes) it will be changed to                        */
   /*  PROCEDURE, which it is, but this could be misleading. It depends on what a task                */
   /*  wants to do with it. A recapitalizing task will not want to recap a procedure                  */
   /*  name in quotes.                                                                                */
   /* =============================================================================================== */
   if Opt.0progress then call Progress SubID()
   do t = 2 to t_type.0 -1
      /*  looking at the next token partly duplicates what Toker_AtFunction does, but                 */
      /*  this prevents a lot of unneeded calls to it, which saves about a second of time             */
      /*  (when doing RexxTool.cmd)                                                                   */
      n = R_NextToken(t)
      if t_type.n == 'REXX_W_(' then
         if t_prefix.n == '' then
            if Toker_AtFunction(t) then
               t_type.t = 'REXX_W_PROCEDURE'
   end t
   if Opt.0progress then call Progress
   return 0
   /* === End of Toker_IDfunctions ================================================================== */

   Toker_IsAssignment: procedure expose (Shared_vars) (all_tokens) Tkr. /*fold01*/
   /* =============================================================================================== */
   /*  returns True if token t is the operator in a simple assignment statement, like                 */
   /*           a = b                                                                                 */
   /*           a = function()                                                                        */
   /*           Stem.Tail = b                                                                         */
   /*           a += b      (Regina shortcut assignment)                                              */
   /*           a == b      (which is illegal)                                                        */
   /*           do a = 1 to 3                                                                         */
   /*                                                                                                 */
   /*  returns False for anything else                                                                */
   /*                                                                                                 */
   /*  this is only used by the Toker to check the accuracy of a REXX_W_ASSIGN type                   */
   /* =============================================================================================== */
   parse arg t
   if t_type.t \= 'REXX_W_ASSIGN' then return 0
   p = R_PrevWord(t)
   if t_type.p \= 'REXX_W_SYMBOL' then return 0
   p = R_Stem(p)

   if R_KeyVal(R_PrevWord(p)) == 'DO' ,
    & t_val.t == '=' then
      return 1

   if p \== CL_FirstWord(t) then return 0
   if right(t_val.t, 1) == '=' then return 1
   return 0
   /* === End of Toker_IsAssignment ================================================================= */

   Toker_AtFunction: procedure expose (Shared_vars) (all_tokens) Tkr. /*fold01*/
   /* =============================================================================================== */
   /*  returns True if Token t is a function name, as in "Foo()"                                      */
   /*                                                                                                 */
   /*  this assumes that sub-keywords have already been identified                                    */
   /* =============================================================================================== */
   /*  Dagnabbit, how do we determine that ':' is not a procedure in:                                 */
   /*              PARSE VALUE SUBSTR(string, start) WITH disk':'(find_file) .                        */
   /*  or 'while' in                                                                                  */
   /*              do while(t>0)                                                                      */
   /*  this handles those cases, but there should be a more general way to identify                   */
   /*  a function, without resorting to a separate parser for each keyword                            */
   /*  fix: test this in every way I can think of                                                     */
   /* ----------------------------------------------------------------------------------------------- */
   parse arg t
   n = R_NextToken(t)
   /*  Add more exclusions here as I discover problems                                                */
   /*  next Rexx token must be "(" without a space before it                                          */
   if t_type.n \= 'REXX_W_(' then return 0
   if t_prefix.n \== '' then return 0
   /*  Foo, 'Foo', or 12.3 could be function names, otherwise not a function                          */
   if wordpos(t_type.t, 'REXX_W_SYMBOL REXX_W_LITERAL REXX_W_CONSTANT') == 0 then return 0
   /*  if not a legal label name it cannot be a function. Note this includes Regina characters        */
   if verify(translate(Unquoted(t_val.t)), '_ABCDEFGHIJKLMNOPQRSTUVWXYZ?!1234567890.$#@') ,
      \= 0 then return 0
   /*  if it part of a PARSE template, it cannot be a function (as far as I can tell)                 */
   w1 = CL_FirstWord(t)
   if translate(t_val.w1) == 'PARSE' then
      if t >= R_StartOfTemplate(w1) then return 0
   /*  passed all the exclusion tests; it must be a function                                          */
   return 1
   /* === End of Toker_AtFunction =================================================================== */

   Toker_InsertToken: procedure expose (Shared_vars) (all_tokens) Tkr. /*fold01*/
   /* =============================================================================================== */
   /*  Insert <HowMany> new tokens at position <t>                                                    */
   /*  This is similar to T_InsertToken but for the Toker; it uses a different list of                */
   /*  token attributes                                                                               */
   /*  This should only be used by the Toker                                                          */
   /* ----------------------------------------------------------------------------------------------- */
   Esigl = sigl
   parse arg t , HowMany
   if HowMany == '' then HowMany = 1
   if debug.0arg then do
         call T_Valid t, Esigl
         if \datatype(HowMany, 'w') then do
               call ShowError 'In Toker_InsertToken, invalid argument(s): "'t'"  "'HowMany'"'
               call Fatality
            end
      end
   Tlist = Tkr.0token_vars
   do while Tlist \= ''
      parse var Tlist AttrName Tlist
      !vName = value('AttrName')
      call StemInsert !vName, t, HowMany
   end   /*  of Tlist  */
   return 0
   /* === End of Toker_InsertToken ================================================================== */

   Toker_DeleteToken: procedure expose (Shared_vars) (all_tokens) Tkr. /*fold01*/
   /* =============================================================================================== */
   /*  Deletes <HowMany> elements at position <t> in all of the t_xxx arrays                          */
   /*  listed in Tkr.0token_vars.                                                                     */
   /*  This should only be used by the Toker                                                          */
   /* ----------------------------------------------------------------------------------------------- */
   Esigl = sigl
   parse arg t , HowMany
   if HowMany == '' then HowMany = 1
   if debug.0arg then do
         call T_Valid t, Esigl
         if \datatype(HowMany, 'w') then do
               call ShowError 'In Toker_DeleteToken, invalid argument(s): "'t'"  "'HowMany'"'
               call Fatality
            end
      end

   Tlist = Tkr.0token_vars
   do while Tlist \= ''
      parse var Tlist AttrName Tlist
      !vName = value('AttrName')
      call StemDelete !vName, t, HowMany
   end   /*  of Tlist  */
   return 0
   /* === End of Toker_DeleteToken ================================================================== */

   /*** 0 functions maintained by RexxMerge.cmd *******************************************************/ /*fold00*/
   ShowGetOpts: procedure expose (Shared_vars) ArgC ArgV. /*fold01*/
   0ShowGetOpts:
   /* =============================================================================================== */
   /*    show how the command line was interpreted by GetOpts()                                       */
   /* =============================================================================================== */
   say copies('=', 80)
   say ' GetOpt returned these variables:'
   VarList = 'Opt.0optlist 'Opt.0optlist' ArgC'
   do i = 0 to ArgC
      VarList = VarList' ArgV.'i
   end
   do while VarList \= ''
      parse var VarList thisvar VarList
      say '    'overlay('=  'value(thisvar), thisvar, 25)
   end
   say copies('=', 80)
   return 0
   /* === End of 0ShowGetOpts ======================================================================= */

   ShowError: procedure expose (Shared_vars) /*fold01*/
   0ShowError:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        call ShowError <Arg1> [, <Arg2>]                                                         */
   /* ----------------------------------------------------------------------------------------------- */
   /*   sends a message to stderr, adding a header with the program name, function name,              */
   /*   and line number to show where the message comes from.                                         */
   /*   Use '' as the second arg to improve the display of a multi-line message:                      */
   /*                                                                                                 */
   /*   Example: in function Add, to show a multi-line message:                                       */
   /*       call ShowError 'Warning: unexpected value for variable varA: "'varA'"'                    */
   /*       call ShowError 'It should be a number', ''                                                */
   /*   displays:                                                                                     */
   /*       [Foo.cmd:Add:33] Warning: unexpected value for variable varA: "foo"                       */
   /*                        It should be a number                                                    */
   /*                                                                                                 */
   /* =============================================================================================== */
   seSigl = sigl

   if Opt.0verbose \= 0 then
      seID = '['G.0me':'LastLabel(seSigl)':'seSigl']'
   else
      seID = '['G.0me':'seSigl']'

   if arg(2, 'e') ,
    & arg(2) = '' then
      seID = copies(' ', length(seID))
   else
      call lineout 'stderr', ''

   call lineout 'stderr', seID arg(1)
   return 0
   /* === End of 0ShowError ========================================================================= */

   Mark: /*fold01*/
   0Mark:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL Mark [<list of variable names>] [, <string>]                                         */
   /* ----------------------------------------------------------------------------------------------- */
   /*  Shows the filename and line number it was called from, the current values of the               */
   /*  variables named, and an informational string                                                   */
   /*  Arg(1) is a list of variable names                                                             */
   /*  Arg(2) can be any descriptive text                                                             */
   /*  Examples:                                                                                      */
   /*      CALL mark 'varA varB', 'At top of the "DO x = 1 to 5" loop'                                */
   /*  If no variable names are given, it will just show a message:                                   */
   /*      CALL mark '', 'At top of the "DO x = 1 to 5" loop'                                         */
   /*                                                                                                 */
   /*  This must share all the variables of whatever routine it was called from.                      */
   /*  I cannot think of a way to do it without setting at least one variable here,                   */
   /*  so reserve the variable '_' for the use of Mark and similar routines.                          */
   /* =============================================================================================== */
   _ = sigl
   call lineout 'stderr', ''
   if arg(2, 'e') ,
    & arg(2) = '' then nop
   else call lineout 'stderr', '[Mark]['value('G.0me')':'_']' arg(2)
   do _ = 1 to 99
      if word(arg(1), _) = '' then leave _
      if symbol(word(arg(1), _)) = 'VAR' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||'= "'||value(word(arg(1), _))||'"'
      if symbol(word(arg(1), _)) = 'LIT' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||' is not defined'
      if symbol(word(arg(1), _)) = 'BAD' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||' is not a variable'
   end _
   drop _
   return 0
   /* === End of 0Mark ============================================================================== */

   LastLabel: procedure expose (Shared_vars) /*fold01*/
   0LastLabel:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       ret = LastLabel()                                                                         */
   /*       ret = LastLabel(<line number>)                                                            */
   /* --------