/** $(c2)
 *
 * Copyright (C) 2022 David Azarewicz <david@88watts.net>
 * All rights reserved.
 *
 * LICENSE
 *
 * You may copy this source code, use it in derivative works, or use it in any
 * other way, provided the copyright notice above is not removed.
 *
 * This source code is provided to you with absolutely no warranty whatsoever.
 * There is no guarantee that you will be able to use this code, or build the
 * program, or use it in any way. It is provided solely as a reference so that
 * you may understand how the program works.
 *
 * The author is not responsible in any way for any problem caused by this
 * software, whether it is used correctly or not.
 */
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "usbcalls.h"
#include "usbtype.h"

static UCHAR ucBuffer[4096];

char *ClassToString(int iClass)
{
 char *pchRetVal;

 pchRetVal = "";

 if (iClass==1) pchRetVal="(Audio)";
 if (iClass==2) pchRetVal="(Communication)";
 if (iClass==3) pchRetVal="(HID)";
 if (iClass==6) pchRetVal="(Image)";
 if (iClass==7) pchRetVal="(Printer)";
 if (iClass==8) pchRetVal="(MSD)";
 if (iClass==14) pchRetVal="(Video)";
 if (iClass==224) pchRetVal="(Wireless Controller)";
 if (iClass==255) pchRetVal="(Vendor Specific)";

 return pchRetVal;
}

char *ProtocolToString(int iClass, int iSubClass, int iProtocol)
{
 char *pchRetVal;

 pchRetVal = "";

  if (iClass==3)
  {
    if (iProtocol==1) pchRetVal="(Keyboard)";
    if (iProtocol==2) pchRetVal="(Mouse)";
  }
  if (iClass==224)
  {
    if (iSubClass==1)
    {
      if (iProtocol==1) pchRetVal="(Bluetooth)";
      if (iProtocol==2) pchRetVal="(UWB Radio)";
      if (iProtocol==3) pchRetVal="(Remote NDIS)";
      if (iProtocol==4) pchRetVal="(Bluetooth AMP)";
    }
  }

  return pchRetVal;
}

int main(int argc, char **argv)
{
  APIRET rc;
  ULONG ulNumDevices;
  ULONG ulDevice;
  ULONG ulLen;
  DeviceDescriptor *pDevDesc;
  int iBufIx;
  CONFIGDESC *pTmpDesc;
  CONFIGDESC *pCfg;
  INTERFACEDESC *pIf;
  ENDPOINTDESC *pEp;
  GETDEVINFODATA GetInfoData;
  USBVERSION QueryVersion;
  #ifdef DOSTATUS
  USBHANDLE DevHandle;
  USHORT usStatus;
  #endif

  ulLen = sizeof(USBVERSION);
  rc = UsbQueryVersion( &QueryVersion, &ulLen );
  if (rc)
  {
    printf("Old USBCALLS doesn't report version information.\n");
  } else {
    printf("USBCALLS version %d.%02d USBRESMGR version %d.%02d\n",
           QueryVersion.DllMajor, QueryVersion.DllMinor,
           QueryVersion.UsbMajor, QueryVersion.UsbMinor);
  }

  rc = UsbQueryNumberDevices(&ulNumDevices);
  if (rc) printf("UsbQueryNumberDevices failed rc=%d\n");
  printf("%d devices found\n", ulNumDevices);

  for (ulDevice=0; ulDevice<ulNumDevices; ulDevice++)
  {
    printf("\nDevice %d:\n", ulDevice);

    memset(&GetInfoData, 0, sizeof(GetInfoData));
    ulLen = sizeof(GetInfoData);
    rc = UsbQueryDeviceInfo(ulDevice+1, &ulLen, (UCHAR*)&GetInfoData);
    if (!rc)
    {
      printf("  ctrlID=%d DevAdr=%d ConfigVal=%d IfNo=%d",
             GetInfoData.ctrlID, GetInfoData.deviceAddress,
             GetInfoData.bConfigurationValue, GetInfoData.bInterfaceNumber);
      printf(" Speed=%d Port=%d HubIx=%d\n",
             GetInfoData.SpeedDevice, GetInfoData.portNum,
             GetInfoData.parentHubIndex);
    }

    memset(ucBuffer, 0, sizeof(ucBuffer));
    ulLen = sizeof(ucBuffer);
    rc = UsbQueryDeviceReport( ulDevice+1, &ulLen, ucBuffer);
    if (rc)
    {
      printf("UsbQueryDeviceReport failed rc=%d\n", rc);
      continue;
    }
    pDevDesc = (DeviceDescriptor*)ucBuffer;
    if ((pDevDesc->bLength != sizeof(DeviceDescriptor)) && (pDevDesc->bDescriptorType != 1))
    {
      printf("Bad Descriptor. Size=%d Type=%d\n", pDevDesc->bLength, pDevDesc->bDescriptorType);
      continue;
    }

    printf("  USB Release number=0x%x DevReleaseNumber=0x%x\n",
      pDevDesc->bcdUSB, pDevDesc->bcdDevice);
    printf("  Class=%d SubClass=%d Protocol=%d MaxPacketSize=%d\n",
      pDevDesc->bDeviceClass, pDevDesc->bDeviceSubClass,
      pDevDesc->bDeviceProtocol, pDevDesc->bMaxPacketSize0);
    printf("  Vendor:Product=%04x:%04x\n",
      pDevDesc->idVendor, pDevDesc->idProduct);
    printf("  iManufacturer=%d iProduct=%d iSerialNumber=%d\n",
      pDevDesc->iManufacturer, pDevDesc->iProduct, pDevDesc->iSerialNumber);

    #ifdef DOSTATUS
    if (UsbOpen(&DevHandle, pDevDesc->idVendor, pDevDesc->idProduct, 0xffff, 0))
    {
      printf("Cannot open device\n");
      DevHandle = 0;
    }

    if (DevHandle)
    {
      rc = UsbDeviceGetStatus(DevHandle, (UCHAR*)&usStatus);
      printf("  DeviceStatus=%04x rc=%x\n", usStatus, rc);
    }
    #endif

    for (iBufIx = sizeof(DeviceDescriptor); iBufIx<ulLen; iBufIx+=pTmpDesc->ucLen)
    {
      pTmpDesc=(CONFIGDESC*)&ucBuffer[iBufIx];
      if (pTmpDesc->ucLen == 0)
      {
        printf("  Error: Invalid zero length descriptor\n");
        break;
      }
      switch (pTmpDesc->ucType)
      {
      case 2: /* Configuration */
        if (pTmpDesc->ucLen == sizeof(CONFIGDESC))
        {
          pCfg = (CONFIGDESC*)pTmpDesc;
          printf("  Configuration: NumInterfaces=%d ConfigValue=%d StringIx=%d Attr=0x%x MaxPower=%d\n",
               pCfg->ucNumInterfaces, pCfg->ucConfigValue, pCfg->ucStringIx, pCfg->ucAttributes, pCfg->ucMaxPower);
        }
        break;
      case 4: /* Interface */
        if (pTmpDesc->ucLen == sizeof(INTERFACEDESC))
        {
          pIf = (INTERFACEDESC*)pTmpDesc;
          printf("  Interface %d: Alt=%d nEP=%d Class=%d%s SubClass=%d Protocol=%d%s\n",
               pIf->ucInterfaceNumber, pIf->ucAlternate, pIf->ucNumEndpoints,
               pIf->ucClass, ClassToString(pIf->ucClass), pIf->ucSubClass,
               pIf->ucProtocol, ProtocolToString(pIf->ucClass, pIf->ucSubClass, pIf->ucProtocol));

          #ifdef DOSTATUS
          if (DevHandle)
          {
            rc = UsbInterfaceGetStatus(DevHandle, pIf->ucInterfaceNumber, (UCHAR*)&usStatus);
            printf("  InterfaceStatus=%04x rc=%x\n", usStatus, rc);
          }
          #endif
        }
        break;
      case 5: /* Endpoint */
        if (pTmpDesc->ucLen == sizeof(ENDPOINTDESC))
        {
          pEp = (ENDPOINTDESC*)pTmpDesc;
          printf("  Endpoint: Adr=0x%x Attr=0x%x MaxPktSize=%d Interval=%d\n",
               pEp->ucAddress, pEp->ucAttributes, pEp->usMaxPacketSize, pEp->ucInterval);

          #ifdef DOSTATUS
          if (DevHandle)
          {
            rc = UsbEndpointGetStatus(DevHandle, pEp->ucAddress, (UCHAR*)&usStatus);
            printf("  EndpointStatus=%04x rc=%x\n", usStatus, rc);
          }
          #endif
        }
        break;
      default:
        break;
      }
    } /* descriptor for loop */

    #ifdef DOSTATUS
    UsbClose(DevHandle);
    #endif

  } /* device for loop */

  return 0;
}
