/** $(c2)
 *
 * Copyright (C) 2022 David Azarewicz <david@88watts.net>
 * All rights reserved.
 *
 * LICENSE
 *
 * You may copy this source code, use it in derivative works, or use it in any
 * other way, provided the copyright notice above is not removed.
 *
 * This source code is provided to you with absolutely no warranty whatsoever.
 * There is no guarantee that you will be able to use this code, or build the
 * program, or use it in any way. It is provided solely as a reference so that
 * you may understand how the program works.
 *
 * The author is not responsible in any way for any problem caused by this
 * software, whether it is used correctly or not.
 */
#define INCL_DOSDEVICES
#define INCL_DOSDEVIOCTL
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "usbtype.h"
#ifdef AZARLIB
#include <\src\common\azarproto.h>
#endif

#define  IOCAT_USBRES            0xA0  // USB Resource Manager
#define  IOCTLF_GET_VERSION      0x30  // GetVersion
#define  IOCTLF_GET_NUMDEVICES   0x31  // GetNumberDevices
#define  IOCTLF_GET_DEVICEREPORT 0x32  // GetDeviceReport
#define  IOCTLF_GET_DEVICEINFO   0x39  // GetDeviceInfo
#define MAX_LEN 40

typedef struct
{
  ULONG  ulDevHandle;
  ULONG  ulInfoLength;
} DEVGETINFO;

static HFILE g_hUSBDrv;
static UCHAR ucBuffer[4096];
static char *cDevSpeed[4] = {"FS", "LS", "HS", "SS"};

ULONG IoctlOpenDriver(void)
{
  ULONG ulAction;
  APIRET rc;

  rc = DosOpen("USBRESM$", &g_hUSBDrv, &ulAction, 0, FILE_NORMAL, OPEN_ACTION_OPEN_IF_EXISTS,
               OPEN_ACCESS_READWRITE | OPEN_FLAGS_NOINHERIT | OPEN_SHARE_DENYNONE, 0);
  if (rc) g_hUSBDrv = 0;
  return rc;
}

ULONG IoctlQueryNumberDevices(void)
{
  APIRET rc;
  ULONG ulNumDev;

  if (!g_hUSBDrv) return 0;

  rc = DosDevIOCtl(g_hUSBDrv, IOCAT_USBRES, IOCTLF_GET_NUMDEVICES,
                   NULL, 0, NULL, &ulNumDev, sizeof(ulNumDev), NULL);
  if (rc) ulNumDev = 0;

  return ulNumDev;
}

ULONG IoctlQueryDeviceInfo(ULONG ulDevNumber, ULONG *pulBufLen, UCHAR *pucData)
{
  APIRET rc;

  if (!g_hUSBDrv) return 1;

  rc = DosDevIOCtl(g_hUSBDrv,IOCAT_USBRES,IOCTLF_GET_DEVICEINFO,
                   &ulDevNumber, sizeof(ulDevNumber), NULL,
                   pucData, *pulBufLen, pulBufLen);

  return rc;
}

ULONG IoctlQueryDeviceReport(ULONG ulDevNumber, ULONG *pulBufLen, UCHAR *pucData)
{
  APIRET rc;
  DEVGETINFO Parm;

  if (!g_hUSBDrv) return 1;

  if (pucData==NULL) *pulBufLen = 0;

  Parm.ulDevHandle = ulDevNumber;
  Parm.ulInfoLength = *pulBufLen;

  rc = DosDevIOCtl(g_hUSBDrv, IOCAT_USBRES, IOCTLF_GET_DEVICEREPORT,
                    &Parm, sizeof(DEVGETINFO), NULL,
                    pucData, *pulBufLen, pulBufLen);

  *pulBufLen = Parm.ulInfoLength;

  return rc;
}

#ifndef AZARLIB
char * _System UsbClassToString(int iClass)
{
  char *pchRetVal;

  pchRetVal = "";

  switch (iClass)
  {
  case 1:
    pchRetVal="(Audio)";
    break;
  case 2:
    pchRetVal="(Communication)";
    break;
  case 3:
    pchRetVal="(HID)";
    break;
  case 5:
    pchRetVal="(Physical)";
    break;
  case 6:
    pchRetVal="(Image)";
    break;
  case 7:
    pchRetVal="(Printer)";
    break;
  case 8:
    pchRetVal="(MSD)";
    break;
  case 9:
    pchRetVal="(Hub)";
    break;
  case 0x0a:
    pchRetVal="(CDC-Data)";
    break;
  case 0x0b:
    pchRetVal="(Smart Card)";
    break;
  case 0x0d:
    pchRetVal="(Security)";
    break;
  case 0x0e:
    pchRetVal="(Video)";
    break;
  case 0x0f:
    pchRetVal="(Healthcare)";
    break;
  case 0x10:
    pchRetVal="(Audio/Video)";
    break;
  case 0xdc:
    pchRetVal="(Diagnostic)";
    break;
  case 0xe0:
    pchRetVal="(Wireless Controller)";
    break;
  case 0xef:
    pchRetVal="(Misc)";
    break;
  case 0xfe:
    pchRetVal="(Application Specific)";
    break;
  case 0xff:
    pchRetVal="(Vendor Specific)";
    break;
  default:
    break;
  }

  return pchRetVal;
}

char * _System UsbProtocolToString(int iClass, int iSubClass, int iProtocol)
{
  char *pchRetVal;
  static char achBuf[24];

  pchRetVal = "";

  switch (iClass)
  {
  case 1: /* Audio */
    if (iProtocol==0) pchRetVal="(Class 1)";
    if (iProtocol==0x20) pchRetVal="(Class 2)";
    if (iProtocol==0x30) pchRetVal="(Class 3)";
    break;
  case 3: /* HID */
    if (iProtocol==1) pchRetVal="(Keyboard)";
    if (iProtocol==2) pchRetVal="(Mouse)";
    break;
  case 0xe0: /* Wireless */
    if (iSubClass!=1) break;
    if (iProtocol==1) pchRetVal="(Bluetooth)";
    if (iProtocol==2) pchRetVal="(UWB Radio)";
    if (iProtocol==3) pchRetVal="(Remote NDIS)";
    if (iProtocol==4) pchRetVal="(Bluetooth AMP)";
    break;
  default:
    break;
  }

  return pchRetVal;
}
#endif

char *GetDeviceType(UCHAR *pucBuf, ULONG ulLen)
{
  static char achRetVal[64];
  DeviceDescriptor *pDevDesc;
  int iBufIx;
  INTERFACEDESC *pDevIf;
  char *pVal1;
  char *pVal2;

  achRetVal[0] = 0;

  pDevDesc = (DeviceDescriptor *)pucBuf;

  if (pDevDesc->bDeviceClass == 9)
  {
    return UsbProtocolToString(pDevDesc->bDeviceClass, pDevDesc->bDeviceSubClass, pDevDesc->bDeviceProtocol);
  }

  for (iBufIx = sizeof(DeviceDescriptor); iBufIx<ulLen; iBufIx+=pDevIf->ucLen)
  {
    pDevIf = (INTERFACEDESC*)&pucBuf[iBufIx];
    if (pDevIf->ucLen == 0) break;

    pVal1 = NULL;
    pVal2 = NULL;

    if (pDevIf->ucType == 4) /* Interface */
    {
      pVal1 = UsbClassToString(pDevIf->ucClass);
      pVal2 = UsbProtocolToString(pDevIf->ucClass, pDevIf->ucSubClass, pDevIf->ucProtocol);
    }

    if ( pVal1 && *pVal1 && ((strlen(achRetVal)+strlen(pVal1))<MAX_LEN) )
    {
      strcat(achRetVal, pVal1);
      if ( pVal2 && *pVal2 && ((strlen(achRetVal)+strlen(pVal2))<MAX_LEN) )
      {
        strcat(achRetVal, "-");
        strcat(achRetVal, pVal2);
      }
    }
  }
  return achRetVal;
}

int main(int argc, char **argv)
{
  APIRET rc;
  ULONG ulNumDevices;
  ULONG ulDevice;
  ULONG ulLen;
  DeviceDescriptor *pDevDesc;
  GETDEVINFODATA GetInfoData;
  char achVendor[40];
  char achDevice[40];

  if (IoctlOpenDriver())
  {
    printf("Cannot open usb driver\n");
    return 1;
  }

  ulNumDevices = IoctlQueryNumberDevices();
  printf("%d devices found\n", ulNumDevices);

  for (ulDevice=0; ulDevice<ulNumDevices; ulDevice++)
  {
    printf("\nDevice %d:\n", ulDevice);

    memset(&GetInfoData, 0, sizeof(GetInfoData));
    ulLen = sizeof(GetInfoData);
    rc = IoctlQueryDeviceInfo(ulDevice+1, &ulLen, (UCHAR*)&GetInfoData);
    if (!rc)
    {
      printf("  Controller %d Hub %d Port %d Speed %s\n", GetInfoData.ctrlID,
             GetInfoData.parentHubIndex, GetInfoData.portNum, cDevSpeed[GetInfoData.SpeedDevice&3]);
    }

    memset(ucBuffer, 0, sizeof(ucBuffer));
    ulLen = sizeof(ucBuffer);
    rc = IoctlQueryDeviceReport(ulDevice+1, &ulLen, ucBuffer);
    if (rc)
    {
      printf("UsbQueryDeviceReport failed rc=%d\n", rc);
      continue;
    }
    pDevDesc = (DeviceDescriptor*)ucBuffer;
    if ((pDevDesc->bLength != sizeof(DeviceDescriptor)) && (pDevDesc->bDescriptorType != 1))
    {
      printf("Bad Descriptor. Size=%d Type=%d\n", pDevDesc->bLength, pDevDesc->bDescriptorType);
      continue;
    }

    printf("  USB Release=%03x / %04x %s\n", pDevDesc->bcdUSB, pDevDesc->bcdDevice, GetDeviceType(ucBuffer, ulLen));
    *achVendor = 0;
    *achDevice = 0;
    #ifdef AZARLIB
    UsbIdToString(pDevDesc->idVendor, pDevDesc->idProduct, achVendor, sizeof(achVendor), achDevice, sizeof(achDevice));
    #endif
    printf("  %04x:%04x %s %s\n", pDevDesc->idVendor, pDevDesc->idProduct, achVendor, achDevice);
  } /* device for loop */

  if (g_hUSBDrv) DosClose(g_hUSBDrv);
  return 0;
}
