/**/

   /***************************************************************************************************/
   /*    pstat.cmd by Anton Monroe <amonroe5301@gmail.com>                                            */
   /*                                                                                                 */
   /*  This is a front end for the pstat.exe command that reformats the output so it                  */
   /*  is easier to read. For example, it adds extra header lines every few lines                     */
   /*  lines so I do not have to scroll to the top of the listing to see what each                    */
   /*  column is. It also removes the redundant carriage-returns produced by pstat.exe                */
   /*                                                                                                 */
   /*  Usage is the same as pstat.exe                                                                 */
   /*  type 'pstat.cmd -help' for help about pstat.cmd                                                */
   /*  type 'pstat.cmd /?' for help about pstat.exe                                                   */
   /*                                                                                                 */
   /*  To redefine how things should be displayed, edit the list of variables in the                  */
   /*  DefineSection routine below. The 'ToShow' variable for each section determines                 */
   /*  which columns are shown and in what order.                                                     */
   /*                                                                                                 */
   /*  This is intended for a wide screen of over 110 characters per line, but if you                 */
   /*  pipe it into a pager it won't really matter what your screen width is.                         */
   /*                                                                                                 */
   /*  I wrote this years ago. The basic idea might be something I invented, or it                    */
   /*  might be based on code from somebody else. I wish I remembered so I could give                 */
   /*  credit where credit is due.                                                                    */
   /*                                                                                                 */
   /*  recent changes:                                                                                */
   /*    Tue Jun 19 2018                                                                              */
   /*        added check to make sure args start with '/' and fixed endless loop if pstat.exe         */
   /*        returned nothing.                                                                        */
   /*    Sat Aug  2 2025                                                                              */
   /*        automatically determine ScreenWidth and LinesPerPage                                     */
   /*        can still be overridden by editing defineSection                                         */
   /*    Sun Aug  3 2025                                                                              */
   /*        handle "/P:<pid>" switch, which has been a missing feature for a long time               */
   /*    Sun Aug  3 2025                                                                              */
   /*        fixed parsing to handle the unlikely case of a file name with spaces                     */
   /*    Thu Aug  7 2025                                                                              */
   /*        pstat.cmd now returns whatever exit code that pstat.exe returned                         */
   /*        cleaned up a lot of loose ends, unused return codes, obsolete error-checks               */
   /*        and debugging stuff.                                                                     */
   /***************************************************************************************************/
   /*    about the carriage-return problem:                                                           */
   /*    PSTAT puts an extra carriage return after each line in the header except the last line.      */
   /*    Or it might be easier to think of it as "each header line is preceded by CR CR LF"           */
   /*    so it gives something like this:                                                             */
   /*                                                                                                 */
   /*          <0D><0D><0A>                                                                           */
   /*          Process and Thread Information<0D><0D><0A>                                             */
   /*          <0D><0D><0A>                                                                           */
   /*          Parent<0D><0D><0A>                                                                     */
   /*          Process   Process   Session   Process   Thread<0D><0D><0A>                             */
   /*          ID        ID        ID       Name       ID    Priority   Block ID   State              */
   /*                                                                                                 */
   /*    Many apps I have seen treat a line ending with <0D><0D><0A> as one line                      */
   /*    LINEIN('stdin') treats it as two blank lines                                                 */
   /*    LINEIN('QUEUE:') treats it as one line containing <0D>, then a blank line                    */
   /*                                                                                                 */
   /*    That means extra complications in recognizing a section header.                              */
   /*    I do not want to depend on the extra lines; it would not work for pstat output that is       */
   /*    filtered through something else, or from some (hypothetical) other version of pstat.exe.     */
   /*                                                                                                 */
   /*    if all you need to do is strip the extra carriage returns, you could just use sed--          */
   /*          pstat.exe | sed -r -e s/(\x0D)+$//g                                                    */
   /***************************************************************************************************/
   /*  How it works:                                                                                  */
   /*  Any argument that starts with "-" is a switch for pstat.cmd. So far there is only one.         */
   /*  Arguments that start with "/" are simply passed on to pstat.exe.                               */
   /*                                                                                                 */
   /*  The output from pstat.exe is saved in the queue.                                               */
   /*                                                                                                 */
   /*  The top level routine reads each line until it recognizes the beginning of a                   */
   /*  section.  Say the start of the Process and Thread section. It calls the doThread               */
   /*  subroutine, which knows how to parse the lines in that section.                                */
   /*                                                                                                 */
   /*  doThread reads its section of the output, filters and formats the information                  */
   /*  according to certain rules, and displays it. When doThread reaches the end of its              */
   /*  section, it returns.                                                                           */
   /*                                                                                                 */
   /*  The top level then searches for the next section header.                                       */
   /*                                                                                                 */
   /*  The rules that determine how each section displays its information are done in                 */
   /*  the DefineSection routine. It has an area for each section where you can change                */
   /*  the variables that determine what columns to display, which order the columns are in,          */
   /*  and how wide each column is. You can also change the page width and length there.              */
   /*                                                                                                 */
   /*  If pstat.cmd does not understand a line, it will just print it preceded by <line no>::         */
   /*  If you see a line like that, it means there is a bug in pstat.cmd.                             */
   /*  If there is an error parsing a line, pstat.cmd will beep and show a message.                   */
   /***************************************************************************************************/

   Main: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   /* Load RexxUtil Library                                                                           */
   if RxFuncQuery('SysLoadFuncs') then do
         call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
         call SysLoadFuncs
      end

   call on Error
   signal on NoValue

   /*  In most of my scripts I use the G. stem for storing global variables and Opt. for              */
   /*  options based on command line switches. In this case neither one amounts to much.              */
   G. = ''
   parse source . . G.0me
   G.0me = filespec('name', G.0me)
   parse upper arg args

   /*    the user would eventually figure out that the only way to get                                */
   /*    pstat.exe to show help is with "/?".  Try to save him some trouble                           */
   if wordpos(args, '/? /HELP /H') > 0 then do
         'pstat.exe /?'
         return 1
      end

   Opt. = 0
   pstatargs = ''
   do while args \= ''
      parse var args thisarg args
      select
         when left(thisarg, 1) == '/' then
            pstatargs = pstatargs translate(thisarg)
         when left(thisarg, 1) == '-' then
            if translate(thisarg) == '-WIDE' then
               Opt.0wide = 1
            else do
                  call Help
                  exit 1
               end
         otherwise
            call Help
            exit 1
      end   /*  select  */
   end
   pstatargs = strip(pstatargs)
   if pstatargs == '' then
      pstatargs = '/A'

   parse value SysTextScreenSize() with G.0LinesPerPage G.0ScreenWidth
   G.0LinesPerPage = G.0LinesPerPage -5
   G.0ScreenWidth = G.0ScreenWidth -1
   /* flush queue                                                                                     */
   do while queued() > 0
      parse pull dummy
   end
   /* ----------------------------------------------------------------------------------------------- */

   /* save PSTAT output to queue, with blank line at end                                              */
   'pstat.exe 'pstatargs' | rxqueue.exe'
   pstatrc = rc
   /*  NextLine complains if queue does not end with empty line. Main complains if there is           */
   /*  only one empty line. I won't try to figure them out                                            */
   if queued() > 0 then
      queue ''
   Qctr = 0
   wline = ''
   inline = ''
   outline = ''

   call PrintLine ''
   do while queued() > 0
      call NextLine
      do while wline == '' ,
             & queued() > 0
         call NextLine
      end
      select
         when wline == 'Process and Thread Information' then
            if left(pstatargs, 2) == '/P' then
               call doProc1
            else
               call doThread
         when wline == 'System Semaphore Information' then
            call doSemaphore
         when wline == 'Named Shared Memory Information' then
            call doSharedMem
         when wline == 'Run-Time Link Libraries' then
            call doLibraries
         when wline == 'Process Version Information' then
            call doPVersion
         when wline == 'Run-Time Link Library Information' then
            call doLibInfo
         otherwise
            CurrentSection = 'NoSection'
            /* if everything goes right, this will never happen                                       */
            outline = inline
            outline = Qctr'::'outline
            call PrintLine outline
      end
   end
   return pstatrc
   /* === End of Main =============================================================================== */

   Help: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   call lineout 'stderr', 'Usage:'
   call lineout 'stderr', '   pstat.cmd [-wide] [<argument(s) for pstat.exe>]'
   call lineout 'stderr', '      -wide       do not truncate right-hand column even if wider than screen'
   call lineout 'stderr', '      /?          show help for pstat.exe'
   call lineout 'stderr', 'Example:'
   call lineout 'stderr', '   pstat.cmd -wide /A'
   call lineout 'stderr', ''
   call lineout 'stderr', ''
   call lineout 'stderr', 'the help for pstat.exe is:'
   'pstat.exe /?'
   return
   /* === End of Help =============================================================================== */

   DefineSection: /*fold00*/
   /* =============================================================================================== */
   /*     for each section, sets a lot of variables that define how that                              */
   /*     section should be shown                                                                     */
   /* PID, PPID, etc are names for the elements of the array. Don't change them                       */
   /* fsize.PID      is the width of the PID column.                                                  */
   /* hdr.PID        is the label at the top of the PID column                                        */
   /*                hdr.PID must be shorter than fsize.PID characters                                */
   /* ToShow         is a list of which columns to show and in what order                             */
   /* pstat.         array that contains lines of pstat.exe's section header, SPACE()ed.              */
   /*                it is used to verify that we are reading the correct header.                     */
   /* hdr.0          is just a convenient place to store the number of possible fields.               */
   /* ToShow         list of fields to show for the section and in what order                         */
   /*                                                                                                 */
   /* G.0ScreenWidth    number of characters in a horizontal line                                     */
   /* G.0LinesPerPage   how often to print header                                                     */
   /* =============================================================================================== */
   parse arg args

   CurrentSection = args
   args = translate(args)
   Lctr = 0
   /*  akm: page size is size of current window, in my case 124 x 52. To use something                */
   /*  different, uncomment and edit the lines below. But if you change them, you may                 */
   /*  also need to change some of the fsize.XXX variables. Like fsize.SemName and                    */
   /*  fsize.LibLong in the Semaphores, Libraries, and LibInfo sections below.                        */
   /*                                                                                                 */
   /*  If the content of a field is longer than the field length, it will be truncated                */
   /*  on the left. If the field length is 0, it will not be truncated, but that only                 */
   /*  works for the right-most field. I try to put the longest fields on the right.                  */
   /*                                                                                                 */
   /*    G.0ScreenWidth = 110                                                                         */
   /*    G.0LinesPerPage = 47                                                                         */

   /* the defaults:  do this every time because I don't want to accidentally inherit                  */
   /*                some changes set by a previous section                                           */
   PID         = 1     ;     fsize.PID         = 6     ;     hdr.PID       = 'PID'
   PPID        = 2     ;     fsize.PPID        = 6     ;     hdr.PPID      = 'PPID'
   SID         = 3     ;     fsize.SID         = 6     ;     hdr.SID       = 'SID'
   PName       = 4     ;     fsize.PName       = 50    ;     hdr.PName     = 'Process Name'
   TID         = 5     ;     fsize.TID         = 6     ;     hdr.TID       = 'TID'
   Priority    = 6     ;     fsize.Priority    = 10    ;     hdr.Priority  = 'Priority'
   BID         = 7     ;     fsize.BID         = 10    ;     hdr.BID       = 'Block ID'
   State       = 8     ;     fsize.State       = 9     ;     hdr.State     = 'State'
   Index       = 9     ;     fsize.Index       = 7     ;     hdr.Index     = 'Index'
   Ref         = 10    ;     fsize.Ref         = 6     ;     hdr.Ref       = 'Refs'
   Req         = 11    ;     fsize.Req         = 6     ;     hdr.Req       = 'Reqs'
   Flag        = 12    ;     fsize.Flag        = 7     ;     hdr.Flag      = 'Flag'
   SemName     = 13    ;     fsize.SemName     = 24    ;     hdr.SemName   = 'Owned Semaphore'
   Handle      = 14    ;     fsize.Handle      = 8     ;     hdr.Handle    = 'Handle'
   Selector    = 15    ;     fsize.Selector    = 8     ;     hdr.Selector  = 'Select'
   SharedName  = 16    ;     fsize.SharedName  = 40    ;     hdr.SharedName  = 'Shared Memory Name'
   LibLong     = 17    ;     fsize.LibLong     = 64    ;     hdr.LibLong   = 'Library'
   LibShort    = 18    ;     fsize.LibShort    = 13    ;     hdr.LibShort  = 'Short Name'
   Version     = 19    ;     fsize.Version     = 9     ;     hdr.Version   = 'Version'
   BldLvl      = 20    ;     fsize.BldLvl      = 7     ;     hdr.BldLvl    = 'Build'
   SubBldLvl   = 21    ;     fsize.SubBldLvl   = 10    ;     hdr.SubBldLvl = 'SubBuild'
   SMPVer      = 22    ;     fsize.SMPVer      = 9     ;     hdr.SMPVer    = 'SMP Ver'
   hdr.0 = 22

   /*  the complete ToShow list is never used, so comment it out                                      */
   /*     ToShow = PID PPID SID PName TID Priority BID State Index Ref Req Flag                       */
   /*     ToShow = ToShow SemName Handle Selector SharedName LibLong LibShort                         */
   /*     ToShow = ToShow Version BldLvl SubBldLvl SMPVer                                             */
   ToShow = ''
   /*  default section title is taken from pstat.exe section header. You can change it below          */
   SectionTitle = strip(wline)

   /* Make any section-specific changes here                                                          */
   /*  SectionTitle is optional; if you don't give one it will use the original header line           */
   /*  from pstat.exe.                                                                                */
   select
      when args == '' then
         nop
      when args == 'THREAD' then do                                                       /* as in /C */
            /*    SectionTitle = 'Process and Thread Information'                                     */
            pstat.1 = 'Parent'
            pstat.2 = 'Process Process Session Process Thread'
            pstat.3 = 'ID ID ID Name ID Priority Block ID State'
            pstat.0 = 3
            ToShow = PID PPID SID PName TID Priority BID State
            /*  Say you do not like the default, you want the process name in the last column,        */
            /*  with a 70-character width.  Change it to:                                             */
            ToShow = PID PPID SID TID BID Priority State PName
            /*  or if you are not interested in the Block ID and State                                */
            /*  ToShow = PID PPID SID TID Priority PName                                              */
            fsize.PName  = 70
            if Opt.0wide then
               fsize.PName  = 0
         end
      when args == 'PROC1' then do                                                  /* as in /P:<pid> */
            /*    SectionTitle = 'Process and Thread Information'                                     */
            pstat.1 = 'Parent Run-Time'
            pstat.2 = 'Process Process Session Process Link Shared'
            pstat.3 = 'ID ID ID Name Libraries Memory Names'
            pstat.0 = 3
            ToShow = PID PPID SID PName LibLong SharedName
            fsize.PName = 45
            fsize.LibLong = 40
            if Opt.0wide then
               fsize.SharedName = 0
         end
      when args == 'PROC2' then do                                                  /* as in /P:<pid> */
            pstat.1 = 'Thread'
            pstat.2 = 'ID Priority State Block ID Owned Semaphores'
            pstat.0 = 2
            ToShow = TID Priority State BID SemName
            if Opt.0wide then
               fsize.SemName = 0
         end
      when args == 'SEMAPHORE' then do                                                    /* as in /S */
            /*    SectionTitle = 'System Semaphore Information'                                       */
            pstat.1 = 'Owning Process No. of No. of Semaphore'
            pstat.2 = '(Name ID Session) Index References Requests Flag Name'
            pstat.0 = 2
            ToShow = PName PID SID Index Ref Req Flag SemName
            if Opt.0wide then
               fsize.SemName = 0
         end
      when args == 'SHAREDMEM' then do                                                    /* as in /M */
            /*    SectionTitle = 'Named Shared Memory Information'                                    */
            pstat.1 = 'No. of'
            pstat.2 = 'Handle Selector References Shared Memory Name'
            pstat.0 = 2
            ToShow  = Handle Selector Ref SharedName
            /*  akm: increased this from 40 to show more of the name;                                 */
            fsize.SharedName = 70
            if Opt.0wide then
               fsize.SharedName  = 0

         end
      when args == 'LIBRARIES' then do                                                    /* as in /L */
            /*    SectionTitle = 'Run-time Link Libraries'                                            */
            pstat.1 = 'Process Process Session'
            pstat.2 = 'Name ID ID Library List'
            pstat.0 = 2
            ToShow  = PName PID SID LibLong
            /*  akm: some libraries have long paths                                                   */
            /*  fsize.LibLong = 75                                                                    */
            if Opt.0wide then
               fsize.LibLong = 0
         end
      when args == 'PVERSION' then do                                                     /* as in /F */
            /*    SectionTitle = 'Process Version Information'                                        */
            pstat.1 = 'Sub-'
            pstat.2 = 'Process Short Version Build Build SMP Process'
            pstat.3 = 'ID Name Number Level Level Version Name'
            pstat.0 = 3
            ToShow  = PID LibShort Version BldLvl SubBldLvl SMPVer PName
            if Opt.0wide then
               fsize.PName = 0
         end

      when args == 'LIBINFO' then do                                                      /* as in /F */
            /*    SectionTitle = 'Run-time Link Library Information'                                  */
            pstat.1 = 'Sub-'
            pstat.2 = 'Short Version Build Build SMP Library'
            pstat.3 = 'Name Number Level Level Version Name'
            pstat.0 = 3
            ToShow  = LibShort Version BldLvl SubBldLvl SMPVer LibLong
            /*  akm: some libraries have long paths                                                   */
            /*  fsize.LibLong = 75                                                                    */
            if Opt.0wide then
               fsize.LibLong = 0
         end
      otherwise
         call lineout 'stderr', 'unrecognized Setup arg "'args'"'
   end

   /*  used for drawing the headers                                                                   */
   HLine1 = copies('#', G.0ScreenWidth)
   HLine2 = copies('-', G.0ScreenWidth)
   /*  if we forgot or did not need to set a SectionTitle, use the original line from pstat.exe       */
   if SectionTitle = '' then
      SectionTitle = wline
   return
   /* === End of DefineSection ====================================================================== */

   doLibInfo: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   call DefineSection 'LibInfo'
   call PrintSectionTitle
   call CheckPstatHdr
   call PrintHeader

   do forever
      call NextLine
      /* empty line is the end of section                                                             */
      if wline == '' then
         return
      /*  this has to be parsed by position because often the version and BldLvl fields are blank     */
      parse var wline fld.LibShort 10 fld.Version 21 fld.BldLvl 31 fld.SubBldLvl 41 fld.SMPVer 49 fld.LibLong
      do i = 1 to hdr.0
         fld.i = strip(fld.i)
      end
      call PrintFmt
   end
   return
   /* === End of doLibInfo ========================================================================== */

   doLibraries: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   call DefineSection 'Libraries'
   call PrintSectionTitle
   call CheckPstatHdr
   call PrintHeader

   do forever
      call NextLine
      /* empty line is the end of section                                                             */
      if wline == '' then
         return
      /*  some lines have 3 words, like "(kernel) <pid> <sid>" or "VDM <pid> <sid>", some             */
      /*  have "<Process> <pid> <sid> <Library>", some have just "<Library>". And the                 */
      /*  fields are not consistently aligned so we cannot use position to parse by.                  */
      /*  If the library name has spaces this will not parse correctly                                */
      /*        --fixed, if we can safely assume the library name is always preceded by 6 spaces      */
      if words(wline) == 3 then
         parse var wline fld.PName fld.PID fld.SID
      else do
            rwline = reverse(wline)
            /*              parse var rwline fld.LibLong fld.SID fld.PID fld.PName                    */
            parse var rwline fld.LibLong '      'fld.SID fld.PID fld.PName
            do i = 1 to hdr.0
               fld.i = reverse(strip(fld.i))
            end
         end
      call PrintFmt
   end
   return
   /* === End of doLibraries ======================================================================== */

   doPVersion: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   call DefineSection 'PVersion'
   call PrintSectionTitle
   call CheckPstatHdr
   call PrintHeader

   do forever
      call NextLine
      /* empty line is the end of section                                                             */
      if wline == '' then
         return
      /* fix: untested because this section seems to be always empty                                  */
      parse var wline fld.PID fld.LibShort fld.Version fld.BldLvl fld.SubBldLvl fld.SMPVer fld.PName
      call PrintFmt
   end
   return
   /* === End of doPVersion ========================================================================= */

   doProc1: /*fold00*/
   /* =============================================================================================== */
   /*  "/P:<pid>" needs special handling because it has two headers under one section title           */
   /* =============================================================================================== */
   call DefineSection 'Proc1'
   call PrintSectionTitle
   call CheckPstatHdr
   call PrintHeader

   call NextLine
   /*     parse var wline fld.PID fld.PPID fld.SID fld.PName fld.LibLong fld.SharedName dummy         */
   /*  parse from both ends, because LibLong may be ''.  PName seems to be followed by 8 spaces       */
   /*  so maybe we can use that to identify the end of it                                             */
   parse var wline fld.PID fld.PPID fld.SID fld.PName '        ' rest
   fld.PName = strip(fld.PName)
   rest = reverse(rest)
   parse var rest fld.SharedName fld.LibLong
   fld.SharedName = reverse(strip(fld.SharedName))
   fld.LibLong = reverse(strip(fld.LibLong))
   call PrintFmt
   do forever
      call NextLine
      if wline == '' then do
            call PrintLine ''
            call doProc2
            return
         end
      parse var wline fld.LibLong fld.SemName dummy
      if dummy \= '' then
         call ParseError 'dummy == "'dummy'"'
      call PrintFmt
   end
   return
   /* === End of doProc1 ============================================================================ */

   doProc2: /*fold00*/
   /* =============================================================================================== */
   /*  "/P:<pid>" needs special handling because it has two headers under one section title           */
   /*  there is no "call PrintSectionTitle" in this one                                               */
   /* =============================================================================================== */
   call DefineSection 'Proc2'
   call CheckPstatHdr
   call PrintHeader

   do forever
      call NextLine
      if wline == '' then
         return
      parse var wline fld.TID fld.Priority fld.State fld.BID fld.SemName dummy
      if dummy \= '' then
         call ParseError 'dummy =="'dummy'"'
      call PrintFmt
   end
   return
   /* === End of doProc2 ============================================================================ */

   doSemaphore: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   call DefineSection 'Semaphore'
   call PrintSectionTitle
   call CheckPstatHdr
   call PrintHeader

   do forever
      call NextLine
      if wline == '' then do
            return
         end
      /* if it starts with a drive+path, it's a full line                                             */
      /*  a process name could have spaces, so parse the line in reverse                              */
      if substr(wline, 2, 1) == ':' then do
            rwline = reverse(wline)
            parse var rwline fld.SemName fld.Flag fld.Req fld.Ref fld.Index fld.SID fld.PID fld.PName
            do i = 1 to hdr.0
               fld.i = reverse(fld.i)
            end i
         end
      else do
            parse var wline fld.Index fld.Ref fld.Req fld.Flag fld.SemName
         end
      call PrintFmt
   end
   return
   /* === End of doSemaphore ======================================================================== */

   doSharedMem: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   call DefineSection 'SharedMem'
   call PrintSectionTitle
   call CheckPstatHdr
   call PrintHeader

   do forever
      call NextLine
      /* empty line is the end of section                                                             */
      if wline == '' then
         return
      parse var wline fld.Handle fld.Selector fld.Ref fld.SharedName .
      call PrintFmt
   end
   return
   /* === End of doSharedMem ======================================================================== */

   doThread: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   call DefineSection 'Thread'
   call PrintSectionTitle
   call CheckPstatHdr
   call PrintHeader

   do forever
      call NextLine
      if wline == '' then
         return
      /*  parse line from both ends in case fld.PName has spaces                                      */
      wline = reverse(wline)
      parse var wline fld.State fld.BID fld.Priority fld.TID wline
      do f = TID to State
         fld.f = reverse(fld.f)
      end
      wline = reverse(wline)
      parse var wline fld.PID fld.PPID fld.SID fld.PName
      call PrintFmt
   end
   return
   /* === End of doThread =========================================================================== */

   CheckPstatHdr: /*fold00*/
   /* =============================================================================================== */
   /* look at pstat header, checking lines to make sure we are in the right place                     */
   /* =============================================================================================== */
   i = 0
   do while i < pstat.0
      i = i + 1
      wline = '' ; do while wline == '' ; call NextLine ; end
      if space(wline) \= pstat.i then do
            call Mark 'CurrentSection wline i pstat.i', 'in CheckPstatHdr'
            call lineout 'stderr', '     got: "'space(wline)'"'
            call lineout 'stderr', 'expected: "'pstat.i'"'
         end
   end
   call NextLine
   if wline \= '' then
      call ParseError
   return
   /* === End of CheckPstatHdr ====================================================================== */

   Fmt: procedure expose G. fld. fsize. ToShow /*fold00*/
   /* =============================================================================================== */
   /*      build new line from the fld. array                                                         */
   /*      clear the fld. array                                                                       */
   /*      return new line                                                                            */
   /* =============================================================================================== */
   outline = ''
   do i = 1 to words(ToShow)
      f = word(ToShow, i)
      fld.f = strip(fld.f)
      if fsize.f > 0 then do
            if length(fld.f) >= fsize.f then do
                  /*  truncate if longer than field width                                             */
                  fld.f = '~'right(fld.f, (fsize.f - 3))
               end
            outline = outline||left(fld.f, fsize.f)
         end
      else
         /*  if fsize.f == 0, do not truncate or pad fld.f                                            */
         outline = outline||fld.f
   end
   return outline
   /* === End of Fmt ================================================================================ */

   NextLine: /*fold00*/
   /* =============================================================================================== */
   /*    gets next line and sets:                                                                     */
   /*      inline == original line                                                                    */
   /*      wline  == line without leading/trailing spaces                                             */
   /* =============================================================================================== */
   inline = ''
   wline = ''
   fld. = ''
   if queued() < 1 then do
         call lineout 'stderr', '[NextLine] unexpected end of file, called from line 'sigl
         signal Error
      end
   else do
         inline = linein('queue:')
         /* remove spurious <CR> that pstat.exe adds                                                  */
         inline = strip(inline, , '0D'x)
         wline = strip(inline)
      end
   Qctr = Qctr + 1
   return
   /* === End of NextLine =========================================================================== */

   PrintSectionTitle: /*fold00*/
   /* =============================================================================================== */
   /* print the beginning of a section                                                                */
   /* =============================================================================================== */
   call PrintLine HLine1
   call PrintLine center(SectionTitle, G.0ScreenWidth)
   /*     call PrintLine HLine1                                                                       */
   return
   /* === End of PrintSectionTitle ================================================================== */

   PrintFmt: /*fold00*/
   /* =============================================================================================== */
   /*  formats and prints the fields of one line. If line is blank, does nothing.                     */
   /*  also prints header every G.0LinesPerPage lines                                                 */
   /* =============================================================================================== */
   outline = Fmt()
   if strip(outline) \= '' then do
         call PrintLine outline
         Lctr = Lctr + 1
         if Lctr >= G.0LinesPerPage then do
               call PrintHeader
               Lctr = 0
            end
      end
   return
   /* === End of PrintFmt =========================================================================== */

   PrintHeader: /*fold00*/
   /* =============================================================================================== */
   /* print our column headers                                                                        */
   /* =============================================================================================== */
   do i = 1 to hdr.0
      fld.i = hdr.i
   end
   call PrintLine HLine2
   call PrintLine Fmt()
   call PrintLine HLine2
   return
   /* === End of PrintHeader ======================================================================== */

   PrintLine: procedure expose G. /*fold00*/
   /* =============================================================================================== */
   /*    prints the finished line                                                                     */
   /* =============================================================================================== */
   parse arg args
   call lineout 'stdout', strip(args, 'T')
   return
   /* === End of PrintLine ========================================================================== */

   ParseError: /*fold00*/
   /* =============================================================================================== */
   /* handles an error by printing the original line                                                  */
   /* =============================================================================================== */
   parse arg args
   call beep 262, 75
   call lineout 'stderr', '# Parsing Error at 'G.0me':'CurrentSection':'sigl
   call lineout 'stderr', '#    ==> line 'Qctr': "'inline'"'
   if args \= '' then
      call lineout 'stderr', '# 'args
   return
   /* === End of ParseError ========================================================================= */

   /*  the 0-functions below are standard routines I use that are maintained automatically            */
   Error: /*fold00*/
   /* =============================================================================================== */
   /*  a better error-handler might be nice, but I have never seen this triggered anyway              */
   /* =============================================================================================== */
   call lineout 'stderr', '[Error] at line 'sigl', quitting'
   exit 99
   /* === End of Error ============================================================================== */

   Mark: /*fold00*/
   0Mark:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL Mark [<list of variable names>] [, <string>]                                         */
   /* ----------------------------------------------------------------------------------------------- */
   /*  Shows the filename and line number it was called from, the current values of the               */
   /*  variables named, and an informational string                                                   */
   /*  Arg(1) is a list of variable names                                                             */
   /*  Arg(2) can be any descriptive text                                                             */
   /*  Examples:                                                                                      */
   /*      CALL mark 'varA varB', 'At top of the "DO x = 1 to 5" loop'                                */
   /*  If no variable names are given, it will just show a message:                                   */
   /*      CALL mark '', 'At top of the "DO x = 1 to 5" loop'                                         */
   /*                                                                                                 */
   /*  This must share all the variables of whatever routine it was called from.                      */
   /*  I cannot think of a way to do it without setting at least one variable here,                   */
   /*  so reserve the variable '_' for the use of Mark and similar routines.                          */
   /* =============================================================================================== */
   _ = sigl
   call lineout 'stderr', ''
   if arg(2, 'e') ,
    & arg(2) = '' then nop
   else call lineout 'stderr', '[Mark]['value('G.0me')':'_']' arg(2)
   do _ = 1 to 99
      if word(arg(1), _) = '' then leave _
      if symbol(word(arg(1), _)) = 'VAR' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||'= "'||value(word(arg(1), _))||'"'
      if symbol(word(arg(1), _)) = 'LIT' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||' is not defined'
      if symbol(word(arg(1), _)) = 'BAD' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||' is not a variable'
   end _
   drop _
   return 0
   /* === End of 0Mark ============================================================================== */

   NoValue: /*fold00*/
   0NoValue:
   /* =============================================================================================== */
   /*  A simple NoValue handler.                                                                      */
   /*  The only variables it uses are _ and __, which it assumes are not used elsewhere               */
   /* ----------------------------------------------------------------------------------------------- */
   signal off NoValue
   _ = sigl
   parse source . . __
   __ = filespec('name', __)

   call lineout 'stderr', '['__'][NOVALUE Handler] at line '_' for variable "'condition('d')'"'
   call lineout 'stderr', '     ==> 'sourceline(_)
   call lineout 'stderr', 'NOVALUE can also be triggered by a missing CALL'
   exit 9
   /* === End of 0NoValue =========================================================================== */


