/**/

   /***************************************************************************************************/
   /*  Shows a list or tree diagram of running processes                                              */
   /*  Inspired by the Linux 'pstree' command, but it does a lot less                                 */
   /*  Anton Monroe <amonroe5301@gmail.com> March 2025                                                */
   /***************************************************************************************************/

   Main: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   /* Load RexxUtil Library                                                                           */
   if RxFuncQuery('SysLoadFuncs') then do
         call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
         call SysLoadFuncs
      end
   if RxFuncQuery('rxqprocstatus') then do
         call RxFuncAdd 'rxuinit', 'rxu', 'rxuinit'
         call RxuInit
      end
   signal on NoValue

   G. = ''
   parse source . . G.0me
   G.0me = filespec('name', G.0me)
   G.0meID = '['G.0me']'

   Proc.       = ''
   Shared_vars = 'G. Opt. Proc. Action f. IndentLevel sigl Out.'
   sortopts    = 'SID PPID PID PNAME TYPE'
   IndentLevel = 0
   Out.        = ''
   Out.0       = 0

   parse arg args
   Opt. = '0'
   interpret GetOpts(args)
   if Opt.0help then do
         call Help
         exit 1
      end

   if Opt.0hex then
      Opt.0x = Opt.0hex
   Opt.0sort = 'PID'
   if Opt.0sid then
      Opt.0sort = 'SID'
   if Opt.0pid then
      Opt.0sort = 'PID'
   if Opt.0ppid then
      Opt.0sort = 'PPID'
   if Opt.0type then
      Opt.0sort = 'TYPE'
   if Opt.0name then
      Opt.0sort = 'PNAME'
   if Opt.0pname then
      Opt.0sort = 'PNAME'
   Action = translate(ArgV.1)
   if Action == '' then Action = 'TREE'

   /*    lengths of padded sid/pid/etc strings for formatting output                                  */
   f. = ''
   f.sidlen = 13
   f.pidlen = 8
   f.typelen = 6
   f.pnamelen = 24

   if wordpos(SysQueryProcessCodePage(), '850 437') > 0 ,
    & Opt.0ascii \= 1 then do
         f.horchar  = ''
         f.vertchar = ''
         f.Lchar    = ''
         f.Tchar    = ''
      end
   else do
         f.horchar  = '_'
         f.vertchar = '|'
         f.Lchar    = '|'
         f.Tchar    = '|'
      end

   /* =============================================================================================== */
   /*  End of setup; the action starts here                                                           */
   /* =============================================================================================== */
   call MakeProcList
   select
      when Action == 'TREE' then
         ret = ShowTree(0)
      when Action == 'LIST' then do
            if wordpos(translate(Opt.0sort), sortopts) > 0 then do
                  call ShowProcList Opt.0sort
               end
            else do
                  call lineout 'stderr', 'invalid sort option "'Opt.0sort'", use one of'
                  call lineout 'stderr', sortopts
                  exit 1
               end
         end
      when Action == 'HELP' then do
            call Help
            exit 1
         end
      otherwise do
            call lineout 'stderr', 'I do not understand "'ArgV.1'"'
            exit 1
         end
   end   /*  select  */
   do i = 1 to Out.0
      say Out.i
   end i
   say ''
   exit 0
   /* === End of Main =============================================================================== */

   ShowTree: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*    shows information about all processes whose PPID == parentPID                                */
   /*    for each process i, it prints information about i, then calls ShowTree(Proc.i.0ppid) to      */
   /*    show the processes whose parent is process i.                                                */
   /*                                                                                                 */
   /*    Returns: 0 if success,                                                                       */
   /*             1 if error                                                                          */
   /* =============================================================================================== */
   /*    SysInit == PID 1, it is the parent of PMShell                                                */
   /*    PMShell is parent of things started from desktop                                             */
   /*    Detached sessions have PPID = 0                                                              */
   /*    I have no idea why there is a second SysInit                                                 */
   /* =============================================================================================== */
   Esigl = sigl
   parse arg parentPID
   ret = 0
   /*    at the top of the tree, show a header also                                                   */
   if parentPID == '0' then do
         IndentLevel = 1
         call Out FormatProc()
         call Out FormatProc('-')
         call Out FormatProc(IndexOf(0))
      end

   /*    Make a temporary list of PIDs in the idx. array, and sort the list so we can show the        */
   /*    processes in numerical order.                                                                */
   j = 0
   idx. = 0
   IndentLevel = IndentLevel +1
   do i = 1 to Proc.0
      if strip(Proc.i.0ppid) == parentPID then do
            if Proc.i.0pid == 0 then do
                  call lineout 'stderr', 'Error: got Proc.i.0pid == 0  ; i = 'i
                  call Mark 'i idx.0 Proc.0 Proc.i.0ppid Proc.i.0pid Proc.i.0Pname Proc.i.0status Proc.i.0statusDesc Proc.i.0type'
                  exit 9
               end
            j = j +1
            /*  how high can PIDs be? Eight digits should be more than enough, surely                 */
            idx.j = right(Proc.i.0pid, 8)
            idx.0 = j
         end
   end i

   if idx.0 > 0 then do
         call SysStemSort 'idx.', 'A', 'I'
         do j = 1 to idx.0
            i = IndexOf(strip(idx.j))
            call Out FormatProc(i)
            ret = ShowTree(Proc.i.0pid)
         end j
      end
   IndentLevel = IndentLevel -1

   if ret \= 0 then do
         call LogIt 'Error: recursive call to ShowTree() returned 'ret
         call LogIt '       IndentLevel == 'IndentLevel '  i == 'i '    Process == 'Proc.i.0pname
      end
   if IndentLevel < 1 then do
         ret = 1
         call LogIt 'Error: invalid Indentlevel == 'IndentLevel
      end

   if parentPID == '0' then do
         /*  end of recursion, end of output, now draw vertical connecting lines                      */
         do ln = 1 to Out.0
            lpos = pos(f.Lchar, Out.ln)
            if lpos > 0 then
               do l = ln -1 to 1 by -1
                  testchar = substr(Out.l, lpos, 1)
                  if testchar == f.Lchar then do
                        Out.l = overlay(f.Tchar, Out.l, lpos)
                     end
                  if testchar \== ' ' then leave l
                  Out.l = overlay(f.vertchar, Out.l, lpos)
               end l
         end ln
      end   /*  if parentPID == 0  */
   return ret
   /* === End of ShowTree =========================================================================== */

   FormatProc: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*    returns a formatted string ready for printing; called by ShowTree, maybe by others           */
   /*    moved into a function for readability and so the messy details and user options that affect  */
   /*    how things are displayed can be kept in one place                                            */
   /*                                                                                                 */
   /*    Input == the index in the Proc. array ; that is, the process is Proc.i.                      */
   /*    if == '', this returns header; if == '-' returns a line under the header                     */
   /*    otherwise build strings for pid, sid, process name                                           */
   /* =============================================================================================== */
   parse arg i

   StatusStr = ''
   StatusDescstr = ''
   select
      when i == '' then do
            PIDstr = '[PID]'
            PIDstr    = right(PIDstr, f.pidlen, ' ')
            SIDstr = '[SID]'
            Typestr = '[Type]'
            PNamestr = 'Process Name'
         end
      when i == '-' then do
            PIDstr = copies(f.horchar, 5)
            PIDstr    = right(PIDstr, f.pidlen, ' ')
            SIDstr = copies(f.horchar, 5)
            Typestr = copies(f.horchar, 7)
            PNamestr = copies(f.horchar, 12)
         end
      otherwise do
            PIDstr = Proc.i.0pid
            SIDstr = Proc.i.0sid
            StatusStr = Proc.i.0status
            StatusDescstr = Proc.i.0StatusDesc
            if Opt.0x then do
                  PIDstr = d2x(PIDstr)
                  SIDstr = d2x(SIDstr)
                  StatusStr = d2x(StatusStr)
               end
            PNamestr = Proc.i.0pname

            select
               when Proc.i.0type == '0' then
                  Typestr = 'FS'
               when Proc.i.0type == '1' then
                  Typestr = 'VDM'
               when Proc.i.0type == '2' then
                  Typestr = 'VIO'
               when Proc.i.0type == '3' then
                  Typestr = 'PM'
               when Proc.i.0type == '4' then
                  Typestr = 'DT'
               otherwise
                  Typestr = Proc.i.0type
            end   /*  select  */

            PIDstr = '['PIDstr']'
            SIDstr = '['SIDstr']'

            /*    If pid is not 0, add horizontal lines to point to parent                            */
            /*  Vertical lines will be added after the whole array is done                            */
            if Proc.i.0pid == 0 then do
                  PIDstr    = right(PIDstr, f.pidlen, ' ')
               end
            else do
                  PIDstr   = right(PIDstr, f.pidlen, f.horchar)
                  PIDstr   = f.Lchar||f.horchar||PIDstr
               end
            if Opt.0status ,
             & Proc.i.0status > 0 then
               PNamestr = PNamestr'   status: 'StatusStr' 'StatusDescstr
         end
   end   /*  end select  */

   /*    pad strings to fixed length:                                                                 */
   SIDstr   = right(SIDstr, f.pidlen, ' ')
   Typestr   = right(Typestr, f.typelen, ' ')
   blank = copies(' ', f.pidlen -1)
   Outstr = copies(blank, IndentLevel)
   ProcessStr = PIDstr'  'PNamestr

   Outstr = SIDstr||Outstr||ProcessStr

   /*    add hints at right                                                                           */
   if Proc.i.0Pdesc \= '' then do
         Pdescpos = max(length(Outstr) +3, 80)
         Outstr = overlay('('Proc.i.0Pdesc')', Outstr, Pdescpos)
      end

   /*    also show session types:                                                                     */
   Outstr = Typestr||' '||Outstr

   return Outstr
   /* === End of FormatProc ========================================================================= */

   ShowProcList: /*fold00*/
   /* =============================================================================================== */
   /*    lists the processes; argument is the name of the field to sort by.  Default is PID           */
   /*    the following define the positions within the string of the fields                           */
   /*    but I decided to ignore key2, it looks better to sort by everything to the right of key1     */
   /* =============================================================================================== */
   sid1     = 1
   sid2     = sid1 + f.sidlen
   ppid1    = sid2 +1
   ppid2    = ppid1 + f.pidlen
   pid1     = ppid2 +1
   pid2     = pid1 + f.pidlen
   type1    = pid2 +1
   type2    = type1 + f.pidlen
   pname1   = type2 +1
   pname2   = pname1 + f.pnamelen
   parse arg key
   if key == '' then key = 'pid'
   if translate(key) == 'NAME' then key = 'pname'
   firstchar = value(key||'1')
   lastchar  = value(key||'2')
   /*    change that, sort by rest of line                                                            */
   lastchar = pname2

   Disp. = ''
   do i = 1 to Proc.0
      if Proc.i.0pid \= 0 then do
            Typestr = Proc.i.0type
            /*    I assume the process types match what is documented for SysProcessType()            */
            /*    0 = FullScreen, 1 == Real Mode (DOS), 2 == VIO (windowable) 3 == PM, 4 == Detached  */
            select
               when Proc.i.0type == '0' then
                  Typestr = 'FS'
               when Proc.i.0type == '1' then
                  Typestr = 'VDM'
               when Proc.i.0type == '2' then
                  Typestr = 'VIO'
               when Proc.i.0type == '3' then
                  Typestr = 'PM'
               when Proc.i.0type == '4' then
                  Typestr = 'DT'
               otherwise
                  Typestr = Proc.i.0type
            end   /*  select  */

            PNamestr = Proc.i.0pname
            if Proc.i.0Pdesc \= '' then
               PNamestr = PNamestr'  ('Proc.i.0Pdesc')'
            if Opt.0x then
               Disp.i = right(d2x(Proc.i.0sid), f.sidlen) right(d2x(Proc.i.0ppid), f.pidlen) ,
                        right(d2x(Proc.i.0pid), f.pidlen) right(Typestr, f.pidlen)'     'PNamestr
            else
               Disp.i = right(Proc.i.0sid, f.sidlen) right(Proc.i.0ppid, f.pidlen) ,
                        right(Proc.i.0pid, f.pidlen) right(Typestr, f.pidlen)'     'PNamestr

            /*    ugly hack to see if status has anything useful:                                     */
            if Opt.0status then do
                  if Proc.i.0status \= 0 then do
                        StatusStr = strip(Proc.i.0status, 'L', '0')
                        StatusDescstr = Proc.i.0StatusDesc
                        StatusLine = copies(' ', pname1)'      status: 'StatusStr'  'StatusDescstr
                        Disp.i = Disp.i||'0D'x'0A'x||StatusLine
                     end
               end   /*  end if Opt.0status  */
         end   /*  end if Proc.i.0pid \= 0  */
   end i
   Disp.0 = Proc.0
   ret = SysStemSort('Disp.', 'A', 'I', , , firstchar, lastchar)
   if ret == 0 then do
         call Out copies('-', f.sidlen) copies('-', f.pidlen) copies('-', f.pidlen) ,
              copies('-', f.pidlen) '    'copies('-', f.pnamelen)
         call Out right('Session ID', f.sidlen) right('PPID', f.pidlen) right('PID', f.pidlen) ,
              right('Type', f.pidlen)'     Process Name'
         call Out copies('-', f.sidlen) copies('-', f.pidlen) copies('-', f.pidlen) ,
              copies('-', f.pidlen) '    'copies('-', f.pnamelen)
         do i = 1 to Disp.0
            call Out Disp.i
         end
         return 0
      end
   else do
         call LogIt 'Error: SysStemSort() returned 'ret
         return
      end
   /* === End of ShowProcList ======================================================================= */

   IndexOf: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*    In: decimal PID                                                                              */
   /*    Out: index, that is, InPID == Proc.<index>.pid                                               */
   /* =============================================================================================== */
   parse arg InPID
   if InPID == '' then return ''
   ret = ''
   do i = 1 to Proc.0
      if Proc.i.0pid == InPID then do
            ret = i
            leave i
         end
   end
   return ret
   /* === End of IndexOf ============================================================================ */

   MakeProcList: /*fold00*/
   /* =============================================================================================== */
   /*    stem q. has complete output of rxqProcStatus, copy only what we need into Proc.              */
   /*    and use more understandable variable names.  Numbers are hex, so decimalize them.            */
   /*    I assume the process types match what is documented for SysProcessType()                     */
   /*    0 = FullScreen, 1 == Real Mode (DOS), 2 == VIO (windowable) 3 == PM, 4 == Detached           */
   /*    status is a number and sometimes explanatory text                                            */
   /* =============================================================================================== */
   call RxQProcStatus 'q.'
   Proc. = ''
   do i = 1 to q.0P.0
      Proc.i.0pid =    x2d(q.0P.i.1)
      Proc.i.0ppid =   x2d(q.0P.i.2)
      Proc.i.0type =   x2d(q.0P.i.3)
      parse var q.0P.i.4 Proc.i.0status '/' Proc.i.0StatusDesc
      Proc.i.0status = x2d(Proc.i.0status)
      Proc.i.0sid =    x2d(q.0P.i.5)
      Proc.i.0pname =   q.0P.i.6
   end i
   Proc.0 = q.0P.0
   /*    make an entry for a fake process with PID 0                                                  */
   i = Proc.0 +1
   Proc.i.0pid = 0
   Proc.i.0ppid = ''
   Proc.i.0type = ''
   Proc.i.0status = '0'
   Proc.i.0StatusDesc = ''
   Proc.i.0sid = 0
   Proc.i.0pname = 'God, I suppose'
   Proc.0 = i

   if Opt.0debug then
      do i = 1 to Proc.0
         call Mark 'i Proc.i.0pid Proc.i.0ppid Proc.i.0type Proc.i.0status Proc.i.0StatusDesc Proc.i.0sid Proc.i.0Pname'
      end i

   call ReadPIDfiles
   return 0
   /* === End of MakeProcList ======================================================================= */

   ReadPIDfiles: /*fold00*/
   /* =============================================================================================== */
   /*    get hints about a few processes by checking in /var/run for pid files                        */
   /*    akm: this probably is only useful for me                                                     */
   /* =============================================================================================== */
   Unixroot = OS2Var('Unixroot')
   call SysFileTree Unixroot'\var\run\*.pid', 'pidfile', 'FOS'
   do i = 1 to pidfile.0
      line1 = linein(pidfile.i, 1, 1)
      call stream pidfile.i, 'c', 'CLOSE'
      if datatype(line1, 'W') then do
            ProcNum = IndexOf(line1)
            if ProcNum \= '' then
               Proc.ProcNum.0Pdesc = FileBaseName(pidfile.i)
         end
   end i
   return
   /* === End of ReadPIDfiles ======================================================================= */

   Out: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*  Adds a line to the output array                                                                */
   /* =============================================================================================== */
   parse arg args

   ln = Out.0 +1
   Out.ln = args
   Out.0 = ln
   return 0
   /* === End of Out ================================================================================ */

   LogIt: procedure expose (Shared_vars) /*fold00*/
   /* =============================================================================================== */
   /*    shows an error message                                                                       */
   /*    automatically adds timestamp and line number                                                 */
   /* =============================================================================================== */
   parse arg logstr
   logstr = '['date() time()'] ['sigl']'  logstr
   call lineout 'stderr' logstr
   return
   /* === End of LogIt ============================================================================== */

   Help: /*fold00*/
   /* =============================================================================================== */
   /* =============================================================================================== */
   call lineout 'stderr', 'Usage:'
   call lineout 'stderr', '   'G.0me '[action] [-option] ...'
   call lineout 'stderr', 'where'
   call lineout 'stderr', '   <action>  <option>'
   call lineout 'stderr', '    tree                     (default) show a tree of processes'
   call lineout 'stderr', '             -x or -hex      show PID, PPID, SID as hex numbers'
   call lineout 'stderr', '             -ascii          force using ASCII characters for lines'
   call lineout 'stderr', '                             default is to use line-drawing characters'
   call lineout 'stderr', '                             if possible'
   call lineout 'stderr', ''
   call lineout 'stderr', '    list                     show a list of processes'
   call lineout 'stderr', '             -x or -hex      show PID, PPID, SID as hex numbers'
   call lineout 'stderr', '             -sort=<column>  sort by SID, PPID, PID, TYPE, or PNAME'
   call lineout 'stderr', '                             default is PID'
   call lineout 'stderr', '              or'
   call lineout 'stderr', '             -<column>       "-pid" means "-sort=pid", etc.'
   call lineout 'stderr', '             -status         show process status, if non-zero'
   call lineout 'stderr', '                             this is not very useful'
   call lineout 'stderr', ''
   call lineout 'stderr', 'also, for debugging:'
   call lineout 'stderr', '            -debug           show whatever debugging information that'
   call lineout 'stderr', '                             might be included'
   call lineout 'stderr', ''
   call lineout 'stderr', 'Examples:'
   call lineout 'stderr', '    'G.0me'               show process tree with decimal numbers'
   call lineout 'stderr', '    'G.0me' -x            show process tree with hex numbers'
   call lineout 'stderr', '    'G.0me' list -x       list processes with hex numbers'
   call lineout 'stderr', '    'G.0me' list -sid     list processes, sorted by session ID'
   call lineout 'stderr', ''
   return 1
   /* === End of Help =============================================================================== */

   NoValue: /*fold00*/
   /* =============================================================================================== */
   /*  signal on NoValue                                                                              */
   /* ----------------------------------------------------------------------------------------------- */
   /*  depends on variables: G.0meID sigl                                                             */
   /* ----------------------------------------------------------------------------------------------- */
   signal off NoValue
   _ = sigl
   call lineout 'stderr', G.0meID'[NOVALUE Handler] at line '_' for variable "'condition('d')'"'
   call lineout 'stderr', '   ==> {'sourceline(_)'}'
   exit 9
   /* === End of NoValue ============================================================================ */

   Mark: /*fold00*/
   0Mark:
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*       CALL Mark [<list of variable names>] [, <string>]                                         */
   /* ----------------------------------------------------------------------------------------------- */
   /*  Shows the filename and line number it was called from, the current values of the               */
   /*  variables named, and an informational string                                                   */
   /*  Arg(1) is a list of variable names                                                             */
   /*  Arg(2) can be any descriptive text                                                             */
   /*  Examples:                                                                                      */
   /*      CALL mark 'varA varB', 'At top of the "DO x = 1 to 5" loop'                                */
   /*  If no variable names are given, it will just show a message:                                   */
   /*      CALL mark '', 'At top of the "DO x = 1 to 5" loop'                                         */
   /*                                                                                                 */
   /*  This must share all the variables of whatever routine it was called from.                      */
   /*  I cannot think of a way to do it without setting at least one variable here,                   */
   /*  so reserve the variable '_' for the use of Mark and similar routines.                          */
   /* =============================================================================================== */
   _ = sigl
   call lineout 'stderr', ''
   if arg(2, 'e') ,
    & arg(2) = '' then nop
   else call lineout 'stderr', '[Mark]['value('G.0me')':'_']' arg(2)
   do _ = 1 to 99
      if word(arg(1), _) = '' then leave _
      if symbol(word(arg(1), _)) = 'VAR' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||'= "'||value(word(arg(1), _))||'"'
      if symbol(word(arg(1), _)) = 'LIT' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||' is not defined'
      if symbol(word(arg(1), _)) = 'BAD' then
         call lineout 'stderr', '      '||left(word(arg(1), _), 30)||' is not a variable'
   end _
   drop _
   return 0
   /* === End of 0Mark ============================================================================== */

   FileBaseName: procedure /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        name = FileBaseName(<filename>)                                                          */
   /*  returns basename of a filename, like 4os2 "%@name[]" or almost like                            */
   /*  Unix "basename -s"                                                                             */
   /* =============================================================================================== */
   parse arg fileN
   fileN = filespec('N', fileN)
   if fileN == '' then return ''
   if pos('.', fileN) == 0 then return fileN
   return substr(fileN, 1, lastpos('.', fileN) -1)
   /* === End of FileBaseName ======================================================================= */

   GetOpts: procedure /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     INTERPRET GetOpts(args [,valid])                                                            */
   /*                                                                                                 */
   /*  where 'args' is the command line to be parsed, and                                             */
   /*  'valid' is an optional space-separated list of valid switches                                  */
   /*  'help' and '?' are always valid                                                                */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*     parse arg args                                                                              */
   /*     interpret GetOpts(args, 'switchA switchB')                                                  */
   /*                                                                                                 */
   /* ----------------------------------------------------------------------------------------------- */
   /*  Switches must start with '-'. For example:                                                     */
   /*                                                                                                 */
   /*         -Foo                (Opt.0Foo == 1)                                                     */
   /*         -Foo=1              (Opt.0Foo == 1)                                                     */
   /*         -!Foo or -Foo=0     (Opt.0Foo == 0)                                                     */
   /*         -Foo=string         (Opt.0Foo == 'string')                                              */
   /*         -Foo="string"       (Opt.0Foo == 'string')                                              */
   /*         -Foo=a,b,c          (Opt.0Foo == 'a,b,c')                                               */
   /*         -Foo="a, b, c"      (Opt.0Foo == 'a, b, c')                                             */
   /*         -Foo="a b c"        (Opt.0Foo == 'a b c')                                               */
   /*         -Foo=123            (Opt.0Foo == '123')                                                 */
   /*         -Foo= or -Foo=''    (Opt.0Foo == '')                                                    */
   /*                                                                                                 */
   /*  switches must not have spaces before or after the '='                                          */
   /*  values with spaces must be in quotes. the quotes will be stripped.                             */
   /*                                                                                                 */
   /*  a switch without a value is set to '1'                                                         */
   /*        -switchA                                                                                 */
   /*  is the same as                                                                                 */
   /*        -switchA=1                                                                               */
   /*                                                                                                 */
   /*  a switch can be negated with "!", so                                                           */
   /*        -!switchA                                                                                */
   /*  is the same as                                                                                 */
   /*        -switchA=0                                                                               */
   /*                                                                                                 */
   /*     -? is the equivalent of -help                                                               */
   /*     they will both result in:                                                                   */
   /*        Opt.0help    == 1                                                                        */
   /*                                                                                                 */
   /*  Interpreting what GetOpts returns will create several variables that the                       */
   /*  caller can use--                                                                               */
   /*     Opt.              is a stem variable that holds whatever was specified by switches          */
   /*     Opt.0optlist      a space-separated list of options that were read                          */
   /*     Opt.0errors       number of errors; that is, when GetOpts() did not understand something    */
   /*                                                                                                 */
   /*                       elements of the Opt. compound variable all start with '0', to avoid       */
   /*                       name conflicts with simple variables                                      */
   /*                                                                                                 */
   /*     ArgV.             is a stem that holds the non-switch arguments                             */
   /*                       (ArgV.0 is the name of the .cmd script, NOT the number of items           */
   /*                       in the stem)                                                              */
   /*     ArgC              is the number of non-switch arguments                                     */
   /*                                                                                                 */
   /*  Example:                                                                                       */
   /*     Foo.cmd -dir=up -msg="that way" -v "From here"                                              */
   /*     will result in these variables:                                                             */
   /*        Opt.0optlist == opt.0dir opt.0msg                                                        */
   /*        Opt.0dir     == up                                                                       */
   /*        Opt.0msg     == that way                                                                 */
   /*        Opt.0v       == 1                                                                        */
   /*        ArgV.0       == Foo.cmd                                                                  */
   /*        ArgV.1       == From here                                                                */
   /*        ArgC         == 1                                                                        */
   /*                                                                                                 */
   /*  in the case of errors, GetOpts() will set Opt.0errors to the number of errors                  */
   /*  found, and cause the caller to show a message about it on stderr. The caller                   */
   /*  should probably abort if Opt.0errors is greater than 0.                                        */
   /* =============================================================================================== */
   /* Some important variables (inside GetOpts) are:                                                  */
   /*    Str.SetOpts       == commands the the caller will execute to set options                     */
   /*                          in the Opt. stem variable                                              */
   /*    Str.SetArgs       == commands the the caller will execute to put the arguments               */
   /*                          in the ArgV. and ArgC variables                                        */
   /*    Str.Errors        == commands the caller will execute to report errors                       */
   /*                          reading the command line                                               */
   /*    ReturnStr         == all of the above, combined and returned to caller                       */
   /*    flag.             == variables for GetOpt's internal use                                     */
   /*                          flag.error == number of errors trying to read command line             */
   /*    Str.SwitchList    == list of options (switches) used on command line                         */
   /*    ArgC              == number of non-switch arguments                                          */
   /*    ArgV.             == array of non-switch arguments                                           */
   /*    valid             == list of valid switches                                                  */
   /*                                                                                                 */
   /* =============================================================================================== */
   flag. = ''
   flag.errors = 0
   str. = ''
   ReturnStr = ''
   ArgC = 0
   ArgV. = ''
   parse arg args , valid
   do i = 1 to WordsQ(args)
      thisArg = WordQ(args, i)
      if left(thisArg, 1) == '-' then
         call GetOpts.ReadSwitch thisArg
      else do
            /*    not a switch, add to Arg list                                                       */
            ArgC = ArgC +1
            ArgV.ArgC = thisArg
         end
   end
   /* ----------------------------------------------------------------------------------------------- */
   /*    now put it all together into a series of commands for the caller to interpret                */
   if flag.errors > 0 then do
         str.SwitchList = strip(str.SwitchList' Opt.0Errors')
      end
   str.SetOpts = str.SetOpts' Opt.0Errors = 'flag.errors ';'
   str.SetOpts = str.SetOpts 'Opt.0optlist = "'str.SwitchList'" ;'

   str.SetArgs = 'ArgV. = "" ; parse source . . ArgV.0 ;  ArgC = "'ArgC'" ;'
   do i = 1 to ArgC
      varVal = Unquoted(ArgV.i)
      str.SetArgs = str.SetArgs' ArgV.'i' = 'Quoted(varVal)' ;'
   end

   ReturnStr = str.errors str.SetOpts str.SetArgs
   return ReturnStr

   GetOpts.ReadSwitch: procedure expose flag. str. valid
   /* =============================================================================================== */
   /*  this is part of GetOpts()                                                                      */
   /* =============================================================================================== */
   parse arg thisArg
   Negate = 0
   switch = strip(thisArg, 'L', '-')
   if left(switch, 1) == '!' then do
         switch = strip(switch, 'L', '!')
         Negate = 1
      end
   parse var switch optName '=' optVal
   select
      /* -? and -help have fixed meanings and are allowed even if not included in 'valid'             */
      when translate(optName) == '?' then
         optName  = 'Opt.0Help'
      when translate(optName) == 'HELP' then
         optName  = 'Opt.0Help'
      when optName == '' then do
            str.errors = ,
              str.errors||'call lineout "stderr", "[GetOpts] `-` without a switch name" ;'
            flag.errors = flag.errors +1
            return
         end
      when valid \= '' ,
         & wordpos(translate(optName), translate(valid)) == 0 then do
            str.errors = str.errors|| ,
                         'call lineout "stderr", "[GetOpts] I do not understand switch `'thisArg'`" ;'
            flag.errors = flag.errors +1
            return
         end
      otherwise
         optName = 'Opt.0'optName
   end   /*  select  */

   select
      when Negate then
         if optVal == '' then
            optVal = 0
         else do
               str.errors = str.errors|| ,
                            'call lineout "stderr", "[GetOpts] I do not understand switch `'thisArg'`" ;'
               flag.errors = flag.errors +1
               return
            end
      when optVal == '' ,
         & right(switch, 1) \= '=' then
         /*  "-switch=" means Opt.0switch == '', not 1                                                */
         optVal = '1'
      otherwise
         /*    try to be tolerant of unusual arguments; only strip quotes if they are balanced        */
         /*    and only one level of nested quotes                                                    */
         optVal = Unquoted(optVal)
   end   /*  select  */

   if wordpos(optName, str.SetOpts) > 0 then do
         str.errors = ,
           str.errors||'call lineout "stderr", "[GetOpts] duplicate switch `'thisArg'`" ;'
         flag.errors = flag.errors +1
      end
   else do
         str.SetOpts = str.SetOpts' 'optName' = 'Quoted(optVal)' ;'
         str.SwitchList = strip(str.SwitchList' 'optName)
      end
   return
   /* === End of GetOpts ============================================================================ */

   OS2Var: procedure /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        VarValue = OS2var(<variable name>)                                                       */
   /*        OS2Path = OS2var('PATH')                                                                 */
   /*                                                                                                 */
   /*     returns value of a variable in the OS/2 environment,                                        */
   /*     or '' if not defined                                                                        */
   /* =============================================================================================== */
   parse arg Pname
   return value(Pname, , 'OS2ENVIRONMENT')
   /* === End of OS2Var ============================================================================= */

   Quoted: procedure /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        string = Quoted(string)                                                                  */
   /*                                                                                                 */
   /*    if possible, quotes a string with the kind of quotes it does not contain                     */
   /*        {He said "no"}   --> {'He said "no"'}                                                    */
   /*        {He said 'no'}   --> {"He said 'no'"}                                                    */
   /*    otherwise uses double quotes and lets the caller deal with the errors                        */
   /* =============================================================================================== */
   parse arg string
   select
      when pos('"', string) == 0 then
         newstring = '"'string'"'
      when pos("'", string) == 0 then
         newstring = "'"string"'"
      otherwise
         newstring = '"'string'"'
   end
   return newstring
   /* === End of Quoted ============================================================================= */

   Unquoted: procedure /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*        string = Unquoted(string)                                                                */
   /*                                                                                                 */
   /*    strips quotes from a string if and only if it is enclosed in a balanced pair                 */
   /*    of single or double quotes                                                                   */
   /* =============================================================================================== */
   parse arg string
   if length(string) < 2 then return string
   select
      when left(string, 1) == '"' ,
         & right(string, 1) == '"' then
         string = substr(string, 2, length(string) -2)
      when left(string, 1) == "'" ,
         & right(string, 1) == "'" then
         string = substr(string, 2, length(string) -2)
      otherwise
         nop
   end
   return string
   /* === End of Unquoted =========================================================================== */

   WordQ: procedure /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     foo = WordQ(<string>, <word number>, <separator characters>, <quote characters>)            */
   /*                                                                                                 */
   /*  The last two parameters are optional, so it will work                                          */
   /*  as a replacement for the Rexx Word() function.                                                 */
   /* ----------------------------------------------------------------------------------------------- */
   /*  A better Word() function--                                                                     */
   /*  It recognizes a quoted string as one word                                                      */
   /*  Optional parameters let you decide which characters                                            */
   /*  count as word separators and as quotes.                                                        */
   /*                                                                                                 */
   /*  Default word separators are <space> and <tab>                                                  */
   /*  Default quote characters are single, double, and                                               */
   /*  back quotes-- ' " `                                                                            */
   /* =============================================================================================== */

   parse arg str , thisnum , chrsW , chrsQ
   if chrsW == '' then chrsW = ' '||d2c(9)
   if chrsQ == '' then chrsQ = '"'||"'"||'`'
   i         = 1
   quote     = ''
   numwords  = 0
   Thisword  = ''
   w1        = 0
   w2        = 0

   /* find start of first word by trimming blanks and word characters                                 */
   do while i <= length(str) ,
          & pos(substr(str, i, 1), chrsW) > 0
      i = i +1
   end
   if i <= length(str) then numwords = numwords +1
   if numwords == thisnum then w1 = i

   do while (i <= length(str))
      if pos(substr(str, i, 1), chrsQ) \= 0 then do
            /* start of quoted string;
             nothing counts until we get to the matching quote (or end of string) */
            quote = substr(str, i, 1)
            tsti = i +1
            /*  bugfix for "-opt=''"                                                                  */
            do while tsti < length(str) ,
                   & substr(str, tsti, 1) \= quote
               tsti = tsti +1
            end
            if substr(str, tsti, 1) == quote then
               /* found the matching quote                                                            */
               i = tsti
            else
               /* ran out of string with no closing quote, so ignore this quote                       */
               nop
            i = i +1
         end   /*  quote  */

      if pos(substr(str, i, 1), chrsW) \= 0 ,
       & i <= length(str) then do
            /* a word separator, skip all blanks, find start of next word                             */
            /* unless the end of the word we want, then quit                                          */
            if w1 > 0 then do
                  w2 = i -1
                  leave
               end
            do while (pos(substr(str, i, 1), chrsW) \= 0) ,
                   & (i <= length(str))
               i = i +1
            end
            if i <= length(str) then numwords = numwords +1
            /* if this is the start of the word we want, note position in string                      */
            if numwords == thisnum then w1 = i
         end   /*  wordsep  */
      else
         /* a normal character                                                                        */
         i = i +1
   end   /*  do i  */
   if i > length(str) then i = length(str)
   if w1 > 0 then do
         /* we know w1, if we ran out of string, w2 is the end of string                              */
         if w2 == 0 then w2 = i
         /* found start and end of word, copy it                                                      */
         wlen = w2 - w1 +1
         Thisword = substr(str, w1, wlen)
      end
   return Thisword
   /* === End of WordQ ============================================================================== */

   WordsQ: procedure /*fold00*/
   /* =============================================================================================== */
   /*  Usage:                                                                                         */
   /*     foo = WordsQ(string, <separator characters>, <quote characters>)                            */
   /*                                                                                                 */
   /*  The last two parameters are optional, so it will work                                          */
   /*  as a drop-in replacement for the Rexx Word() function.                                         */
   /* ----------------------------------------------------------------------------------------------- */
   /*  A better Words() function--                                                                    */
   /*  It recognizes a quoted string as one word                                                      */
   /*  Optional parameters let you decide which characters                                            */
   /*  count as word separators and as quotes.                                                        */
   /*                                                                                                 */
   /*  Default word separators are <space> and <tab>                                                  */
   /*  Default quote characters are single, double, and                                               */
   /*  back quotes-- ' " `                                                                            */
   /* =============================================================================================== */

   parse arg str , chrsW , chrsQ
   if chrsW == '' then chrsW = ' '||d2c(9)
   if chrsQ == '' then chrsQ = '"'||"'"||'`'
   i         = 1
   quote     = ''
   numwords  = 0

   /* find start of first word by trimming blanks and word characters                                 */
   do while i <= length(str) ,
          & pos(substr(str, i, 1), chrsW) > 0
      i = i +1
   end
   if i <= length(str) then numwords = numwords +1

   do while (i <= length(str))
      if pos(substr(str, i, 1), chrsQ) \= 0 then do
            /* start of quoted string;
             nothing counts until we get to the matching quote (or end of string) */
            quote = substr(str, i, 1)
            tsti = i +1
            /*  bugfix for "-opt=''"                                                                  */
            do while tsti < length(str) ,
                   & substr(str, tsti, 1) \= quote
               tsti = tsti +1
            end
            if substr(str, tsti, 1) == quote then
               /* found the matching quote                                                            */
               i = tsti
            else
               /* ran out of string with no closing quote, so ignore this quote                       */
               nop
            i = i +1
         end   /*  quote  */

      if pos(substr(str, i, 1), chrsW) \= 0 ,
       & i <= length(str) then do
            /* a word separator, skip all blanks, find start of next word                             */
            do while (pos(substr(str, i, 1), chrsW) \= 0) ,
                   & (i <= length(str))
               i = i +1
            end
            if i <= length(str) then numwords = numwords +1
         end
      else
         /* a normal character                                                                        */
         i = i +1
   end   /*  do i  */
   return numwords
   /* === End of WordsQ ============================================================================= */

