/**
 * Event class that indicates something happened within a cribbage board
 * component.  These events could be a move is complete or other events.
 * An instance of this class is passed back through the CribBoardListener
 * interface.  A cribbage board instance can and will throw all events with
 * the exception of custom events. <br><br>
 * <b>Changes:</b>
 * <ul>
 * <li> 2001-09-03 - Initial Release.
 * </ul>
 * @author Michael G. Slack
 * @author slack@attglobal.net
 * @created 2001-09-03
 * @version 2001-09-03 Version 1.00
*/
public class CribBoardEvent
    extends java.util.EventObject
{
    // public statics
    /** No event constant. */
    public static final int CBE_NO_EVENT = 0;

    /** Custom event generated constant. */
    public static final int CBE_CUSTOM_EVENT = 1;

    /** A crib board move has been started event (the peg is now in motion). */
    public static final int CBE_MOVE_STARTED = 2;

    /** A crib board move has been completed.  If a player has won, a CBE_PLAYER_WON
     * event is thrown in the place of this event.
    */
    public static final int CBE_MOVE_COMPLETED = 3;

    /** The cribbage board has been reset. */
    public static final int CBE_CRIB_BOARD_RESET = 4;

    /** One of the players has won (reached max score).  This event type also
     * indicates that the peg move is complete.
    */
    public static final int CBE_PLAYER_WON = 5;

    // private references
    /** Holder for event type. */
    private int event = CBE_NO_EVENT;

    /** Holder for custom event (String). */
    private String custom = "";

    /** Holder for the winning players value. */
    private int winner = CribBoard.NO_PLAYER;

    // ---------------------- Constructors --------------------------------

    /** Constructor to create an cribbage board event.
     * @param source Object that initiated the event.
     * @param event An int representing the event that occured.
    */
    public CribBoardEvent(Object source, int event)
    {
        super(source);

        // check validity of event...
        if ((event >= CBE_CUSTOM_EVENT) && (event <= CBE_PLAYER_WON))
            this.event = event; // otherwise, leave as no event
    }

    /** Constructor to create an event object with a custom event string.
     * @param source Object that initiated the event.
     * @param customEvent A String containing the custom event string.
    */
    public CribBoardEvent(Object source, String customEvent)
    {
        this(source, CBE_CUSTOM_EVENT);

        if (customEvent != null) custom = customEvent;
    }

    /** Constructor to create an event object with a winning player.
     * @param source Object that initiated the event.
     * @param won A boolean indicating a player has won.  If false, NO_PLAYER
     * is set in the event.  This should be true to set the winner passed in
     * in the event.
     * @param winner A int representing the winning player.  This parameter is
     * ignored if the won parameter is false.
    */
    public CribBoardEvent(Object source, boolean won, int winner)
    {
        this(source, CBE_PLAYER_WON);

        if (won) this.winner = winner;
    }

    // ------------------------- Public Methods ----------------------------

    /** Method used to return the event generated.
     * @return A int representing the event generated.
    */
    public int getEvent()
    {
        return event;
    }

    /** Method used to return the custom event associated with this instance.
     * The string returned will never be null.
     * @return String representing a custom event generated by the cribbage
     * board (if one was).  May be an empty string.
    */
    public String getCustomEvent()
    {
        return custom;
    }

    /** Method used to return the winning player to the caller.
     * @return A int representing the winning player.
    */
    public int getWinningPlayer()
    {
        return winner;
    }

    /** Method to return the value of the event as a string to the caller.
     * @return A string representation of the event.
    */
    public String toString()
    {
        String sRet = "CribBoardEvent [";

        switch (event) {
            case CBE_NO_EVENT:
                sRet += "No event";
                break;
            case CBE_CUSTOM_EVENT:
                sRet += "Custom event - '" + custom + "'";
                break;
            case CBE_MOVE_STARTED:
                sRet += "Peg move started";
                break;
            case CBE_MOVE_COMPLETED: 
                sRet += "Peg move completed";
                break;
            case CBE_CRIB_BOARD_RESET:
                sRet += "Cribbage board reset";
                break;
            case CBE_PLAYER_WON:
                sRet += "Player (" + winner + ") has won";
                break;
            default:
                sRet += "Unknown (" + event + ")";
                break;
        }

        sRet += "]";

        return sRet;
    }
}
