import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import com.slackandassociates.cards.*;
import com.slackandassociates.cards.playingcards.*;
import com.slackandassociates.ImageCanvas;
import com.slackandassociates.Utilities;
import com.slackandassociates.SwingWorker;
import com.slackandassociates.dialogs.AboutDlg;

/** 
 * Class defines a game of cribbage (or at least the interface into one). <br>
 * NOTE: bogus/incomplete code is marked with '// --->'. <br><br>
 * <b>Changes:</b>
 * <ul>
 * <li> 2001-09-26 - Initial Release (finished except for bug testing).
 * <li> 2001-09-28 - Modified to use the CribbageProps class for initial
 * configuration information (including paths).
 * <li> 2001-09-29 - Turned off 'debug', cleaned up code.
 * <li> 2001-09-30 - Fixed a 'bug' in the muggins processing (moved the
 * check to the bottom of the board move event).
 * <li> 2001-10-02 - Moved 'waitForMoves' reset into finishGame method so that
 * the 'gameOver' flag is set prior to the move wait ending.
 * <li> 2001-10-07 - Put in a fix to catch a delayed 'muggins' message (go
 * through 'movePeg' even if the points are zero so that the boad move start
 * and end events are ran).
 * <li> 2001-10-08 - Added link into CribbageHelp class.
 * <li> 2001-10-13 - Added a few more properties to make the about box processing
 * easier for future releases.
 * <li> 2001-10-17 - Fixed a minor issue with last card point and computer playing
 * out his hand.  Was also making player press 'go' twice.
 * <li> 2002-01-05 - Modified to use the about dialog from the utility library
 * (slack.jar).
 * <li> 2002-03-31 - Add import of ImageCanvas (moved to jar).
 * <li> 2002-06-22 - Modified to use 'centerWindowOnScreen' method.
 * <li> 2002-06-26 - Modified with 'new' AboutDlg call.
 * <li> 2002-07-11 - Modified to use player/computer name properties.
 * <li> 2002-07-13 - Added a check for game over in comp move thread.
 * <li> 2003-08-10 - Fixed font to work under GCD Java 1.4.1.
 * <li> 2004-01-21 - Updated the CribbageEngine to move class variables into the
 * method using them.
 * <li> 2005-01-21 - Modified to use cards v2 library, try to fix thread issues.
 * <li> 2005-04-28 - Modified to remove 'dead' variable in setupCenter method.
 * <li> 2005-08-31 - Changed cribbage board reference to use interface instead of
 * StdCribBoard.
 * <li> 2005-12-20 - Added an 'invokeLater' call to the main routine to help get
 * swing initialized properly.
 * </ul>
 * @author Michael G. Slack
 * @author slack@attglobal.net
 * @created 2001-09-26
 * @version 2005-12-20 Version 2.04
*/
public class Cribbage extends JApplet
{
    private static final boolean DEBUG_ON = false;

    // window size constants
    private static final int W_WIDTH = 650;
    private static final int W_HEIGHT = 500;

    // constants representing the computer/human players
    private static final int C_PLAYER = 0; // computer
    private static final int H_PLAYER = 1; // human

    // constant strings (maybe put in resource bundle?)
    private static final String S_APPTITLE = "Cribbage";
    private static final String S_DISCARD = "Discard";
    private static final String S_KEEP = "Keep";
    private static final String S_PLAY = "Play";
    private static final String S_PLAY_GAME = "Play Game";
    private static final String S_NEW_GAME = "New Game";
    private static final String S_GO = "Go";
    private static final String S_CONTINUE = "Continue";
    private static final String S_BOARD_SOUNDS = "Board Sound On";
    private static final String S_LASTCARD = " gets one point for last card.";

    // card deck and images
    private CardDeck cards = new CardDeck(PlayingCardDeck.PC_DECK, PlayingCard.class);
    private PlayingCardImageCache cImgs = new PlayingCardImageCache();
    private PlayingCardEnum placeHolder = (PlayingCardEnum) PlayingCardEnum.JC_CARDSYMBOL_VAL3;

    // window elements/components
    private JFrame frame = null;
    private Container frame_cp = null;
    private JMenuBar mb = new JMenuBar();
    private JMenu m1 = new JMenu("File");
    private JMenuItem mi1 = new JMenuItem("Print");
    private JMenuItem mi2 = new JMenuItem("Exit");
    private JMenu m2 = new JMenu("Help");
    private JMenuItem mi3 = new JMenuItem("Contents");
    private JMenuItem mi4 = new JMenuItem("About");

    // message box images
    private ImageIcon mbOKImg = null;
    private ImageIcon mbStopImg = null;

    // board component
    private CribBoard cbBoard = new StdCribBoard(false);

    // other components used
    private ImageCanvas icUpCard = null;
    private JLabel lbl1 = new JLabel("Cards Left: ", SwingConstants.LEFT);
    private JLabel lblCL = new JLabel("0", SwingConstants.CENTER);
    private JLabel lbl2 = new JLabel("Game Tally: ", SwingConstants.LEFT);
    private JLabel lblGT = new JLabel("0", SwingConstants.CENTER);
    private JLabel lbl3 = new JLabel("Score:", SwingConstants.LEFT);
    private JLabel lblCompScore = new JLabel("0", SwingConstants.CENTER);
    private JLabel lblPlayerScore = new JLabel("0", SwingConstants.CENTER);
    private JLabel lbl4 = new JLabel("Games:", SwingConstants.LEFT);
    private JLabel lblCompGames = new JLabel("0", SwingConstants.CENTER);
    private JLabel lblPlayerGames = new JLabel("0", SwingConstants.CENTER);
    private JButton newBtn = new JButton(S_PLAY_GAME);
    private JButton goBtn = new JButton(S_GO);
    private JButton contBtn = new JButton(S_CONTINUE);
    private JCheckBox cbSndOn = new JCheckBox(S_BOARD_SOUNDS);
    private JLabel lblCrib = new JLabel("[CRIB]");
    private PlayArea playArea = new PlayArea();
    private ImageCanvas c1 = new ImageCanvas(cImgs.getCardImage(placeHolder));
    private ImageCanvas c2 = new ImageCanvas(cImgs.getCardImage(placeHolder));
    private ImageCanvas c3 = new ImageCanvas(cImgs.getCardImage(placeHolder));
    private ImageCanvas c4 = new ImageCanvas(cImgs.getCardImage(placeHolder));
    private ImageCanvas c5 = new ImageCanvas(cImgs.getCardImage(placeHolder));
    private ImageCanvas c6 = new ImageCanvas(cImgs.getCardImage(placeHolder));
    private JButton b1 = new JButton(S_DISCARD);
    private JButton b2 = new JButton(S_DISCARD);
    private JButton b3 = new JButton(S_DISCARD);
    private JButton b4 = new JButton(S_DISCARD);
    private JButton b5 = new JButton(S_DISCARD);
    private JButton b6 = new JButton(S_DISCARD);
//  "TimesRoman"
    private Font btnFnt = new Font("Helvetica", Font.PLAIN, 12);

    // handlers
    private ButtonHandler btnhnd = null;
    private CribBoardHandler crbhnd = null;
    private CribPlayHandler crbply = null;

    // card hand instances needed by the game (assume 2 player board)
    private CardHand playedCards = new CardHand(8, false); // play area hand
    private CardHand[] hands = new CardHand[2];     // card hands (6 slots) - original deal
    private CardHand[] playHands = new CardHand[2]; // card hands to play from (4 slots)
    private CardHand crib = new CardHand(4);

    // other instances
    private CribbageEngine cribEng = new CribbageEngine();
    private PlayingCard upCard = null;
    private boolean cribFlag = true; // true = players crib, false = computers crib
    private int iDiscardCount = 0, mugginsPoints = 0;
    private ImageCanvas[] cardImgs = new ImageCanvas[6];
    private JButton[] buttons = new JButton[6];
    private int handTally = 0, playerGamesWon = 0, compGamesWon = 0;
    private boolean goFlagP = false, goFlagC = false, scoreLastPoint = true;
    private boolean gameOver = false, pPlayedLast = false, X31 = false;
    private boolean waitForMoves = false, pAlwaysStart = false;

    // property instance
    private CribbageProps props = new CribbageProps(this);

    // ------------------------ Constructor -----------------------------------

    /** Cribbage game constructor.  Used to initialize some of the cribbage game
     * globals.
    */
    public Cribbage()
    {
        // set up card hand arrays now
        for (int i = 0; i < 2; i++) {
            hands[i] = new CardHand(6);
            hands[i].setCompMode(Card.JC_COMP_NOSUIT);
            playHands[i] = new CardHand(4);
            playHands[i].setCompMode(Card.JC_COMP_NOSUIT);
        }
        crib.setCompMode(Card.JC_COMP_NOSUIT);

        // assign image/button references to array (easier to handle)
        cardImgs[0] = c1; cardImgs[1] = c2; cardImgs[2] = c3;
        cardImgs[3] = c4; cardImgs[4] = c5; cardImgs[5] = c6;
        buttons[0] = b1; buttons[1] = b2; buttons[2] = b3;
        buttons[3] = b4; buttons[4] = b5; buttons[5] = b6;
    }

    // ----------------------- Private Methods --------------------------------

    /** Method used to create the application window frame and about dialog. */
    private void createFrame()
    {
        // create 'application' (non-applet) specific items here
        frame = new JFrame(S_APPTITLE);
        frame.setIconImage(Utilities.loadImage(this, getToolkit(), props.getImagePath(),
                                               "cribbage.gif"));

        // add frame close listener (java 1.3+ only)
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        // --- old style (anonymous inner class) ---
        //  frame.addWindowListener(new WindowAdapter() {
        //                                  public void windowClosing(WindowEvent e) {
        //                                      System.exit(0);
        //                                  }
        //                              });
    }

    /** Method used to setup the application menu bar. */
    private void setupMenu()
    {
        mi1.addActionListener(new ActionListener() { // print
                                      public void actionPerformed(ActionEvent evt) {
                                          try {
                                              Toolkit t = frame.getToolkit();
                                              PrintJob job = t.getPrintJob(frame,
                                                                           "Cribbage Print",
                                                                           null);
                                              Graphics g = job.getGraphics();
                                              frame.printComponents(g);
                                              g.dispose();
                                              job.end();
                                          }
                                          catch (Exception e) {
                                              System.err.println("Print failed: " + e);
                                          }
                                      }
                                  });
        m1.add(mi1);
        m1.addSeparator();
        mi2.addActionListener(new ActionListener() { // exit
                                      public void actionPerformed(ActionEvent evt) {
                                          System.exit(0);
                                      }
                                  });
        m1.add(mi2);
        mb.add(m1);
        mi3.addActionListener(new ActionListener() { // help contents (index)
                                      public void actionPerformed(ActionEvent evt) {
                                          new CribbageHelp(frame, 
                                                           props.getHelpPath()).show();
                                      }
                                  });
        m2.add(mi3);
        m2.addSeparator();
        mi4.addActionListener(new ActionListener() { // about
                                      public void actionPerformed(ActionEvent evt) {
                                          new AboutDlg(frame,
                                                       S_APPTITLE,
                                                       props.getVersionInfo(),
                                                       "Michael G. Slack",
                                                       props.getAuthorEmail()).show();
                                      }
                                  });
        m2.add(mi4);
        mb.add(m2);
        frame.setJMenuBar(mb);
    }

    /** Method used to setup the initial label colors used by the game. */
    private void setupLabels()
    {
        lblCompScore.setOpaque(true);
        lblCompScore.setBackground(Color.blue);
        lblCompScore.setForeground(Color.white);
        lblCompGames.setOpaque(true);
        lblCompGames.setBackground(Color.blue);
        lblCompGames.setForeground(Color.white);
        lblPlayerScore.setOpaque(true);
        lblPlayerScore.setBackground(Color.red);
        lblPlayerScore.setForeground(Color.white);
        lblPlayerGames.setOpaque(true);
        lblPlayerGames.setBackground(Color.red);
        lblPlayerGames.setForeground(Color.white);
        lblCrib.setOpaque(true);
        lblCrib.setBackground(Color.lightGray);
        lblCrib.setForeground(Color.white);
    }

    /** Method used to set buttons size so all are the same. */
    private void setButtonSize(JButton b)
    {
        Dimension d = new Dimension(100, 27);

        b.setPreferredSize(d);
        b.setMaximumSize(d);
        b.setMinimumSize(d);
        // set font to button
        b.setFont(btnFnt);
    }

    /** Method used to setup the center panel of the cribbage game board.
    */
    private void setupCenter()
    {
        Box centerPanel = Box.createHorizontalBox();
        Box v1 = Box.createVerticalBox();
        Box v2 = Box.createVerticalBox();
        Box v3 = Box.createVerticalBox();
        Box v4 = Box.createVerticalBox();
        Dimension dd = new Dimension(5, 16);
        // Dimension bb = new Dimension(88, 27);

        setupLabels();

        v1.add(Box.createVerticalStrut(5));
        v1.add(lbl1);
        v1.add(lbl2);
        v1.add(lbl3);
        v1.add(lbl4);
        v1.add(Box.createVerticalGlue());
        v2.add(Box.createVerticalStrut(5));
        v2.add(lblCL);
        v2.add(lblGT);
        v2.add(lblCompScore);
        v2.add(lblCompGames);
        v2.add(Box.createVerticalGlue());
        v3.add(Box.createVerticalStrut(5));
        v3.add(Box.createRigidArea(dd));
        v3.add(Box.createRigidArea(dd));
        v3.add(lblPlayerScore);
        v3.add(lblPlayerGames);
        v3.add(Box.createVerticalGlue());
        newBtn.addActionListener(btnhnd);
        setButtonSize(newBtn);
        v4.add(newBtn);
        goBtn.addActionListener(btnhnd);
        setButtonSize(goBtn);
        goBtn.setEnabled(false);
        v4.add(goBtn);
        contBtn.addActionListener(btnhnd);
        setButtonSize(contBtn);
        contBtn.setEnabled(false);
        v4.add(contBtn);
        cbSndOn.addActionListener(new ActionListener() { // checkbox (sound on)
                                          public void actionPerformed(ActionEvent evt) {
                                              if (cbBoard.hasSoundEffects()) {
                                                  // change the sound state in the board
                                                  cbBoard.setSoundOn(!cbBoard.isSoundOn());
                                              }
                                          }
                                      });
        v4.add(cbSndOn);
        v4.add(lblCrib);

        centerPanel.add(Box.createHorizontalGlue());
        centerPanel.add(v1);
        centerPanel.add(v2);
        centerPanel.add(v3);
        centerPanel.add(v4);
        centerPanel.add(Box.createHorizontalGlue());

        frame_cp.add(centerPanel, BorderLayout.CENTER);
    }

    /** Method used to create the card + button panel to use for displaying
     * players hand with.
    */
    private JPanel createCardPanel(ImageCanvas c, JButton b)
    {
        JPanel jpRet = new JPanel();
        int w = PlayingCardImageCache.IMAGE_WIDTH;
        int h = PlayingCardImageCache.IMAGE_HEIGHT;
        Dimension d = new Dimension(w+10, h+32); // allow space for button

        jpRet.setPreferredSize(d); // need this in order to set items in absolute locations
        jpRet.setMinimumSize(d);   // need this and the next to keep items together
        jpRet.setMaximumSize(d);   // - see above (note: need both in order to work)
        jpRet.setLayout(null);
        c.setBounds(new Rectangle(5, 1, w, h));
        b.setBounds(new Rectangle(1, h+4, w+8, 27));
        b.setFont(btnFnt);
        b.setVisible(false);
        jpRet.add(c, null);
        jpRet.add(b, null);

        return jpRet;
    }

    /** Method used to setup the south panel of the game layout. */
    private void setupSouth()
    {
        Box bb = Box.createHorizontalBox();

        bb.add(createCardPanel(c1, b1));
        bb.add(createCardPanel(c2, b2));
        bb.add(createCardPanel(c3, b3));
        bb.add(createCardPanel(c4, b4));
        bb.add(createCardPanel(c5, b5));
        bb.add(createCardPanel(c6, b6));
        bb.add(Box.createGlue());

        b1.addActionListener(btnhnd);
        b2.addActionListener(btnhnd);
        b3.addActionListener(btnhnd);
        b4.addActionListener(btnhnd);
        b5.addActionListener(btnhnd);
        b6.addActionListener(btnhnd);

        frame_cp.add(bb, BorderLayout.SOUTH);
    }

    /** Method used to setup the main panels used. */
    private void setupPanels()
    {
        JPanel westPanel = new JPanel();
        JPanel eastPanel = new JPanel();

        frame_cp.add((Component) cbBoard, BorderLayout.NORTH);
        icUpCard = new ImageCanvas(cImgs.getCardImage(props.getCardBackImage()));
        westPanel.add(icUpCard);
        frame_cp.add(westPanel, BorderLayout.WEST);
        setupCenter();
        eastPanel.add(playArea);
        frame_cp.add(eastPanel, BorderLayout.EAST);
        setupSouth();
    }

    /** Method used to move the peg for a player an amount.
     * @param who Which players peg to move.
     * @param amt How much (score) to move peg.
    */
    private void movePeg(int who, int amt)
    {
        try {
            // if (amt > 0) cbBoard.movePeg(who, amt);
            cbBoard.movePeg(who, amt);
        }
        catch (Exception e) {
            System.err.println("movePeg() error: " + e);
        }
    }

    // ------------------------ Protected Methods ---------------------------

    /** Method used to get the points from the player they have in their
     * hand or the crib hand (if theirs).  Moves the peg the points entered.
     * @param cribFlg A boolean representing that the points to be gotten are
     * from the crib hand.
    */
    protected void get_Points(boolean cribFlg)
    {
        int aPts = 0, ePts = 0;

        // figure out how many points player actually has :)
        if (cribFlg)
            aPts = cribEng.figure_Points(crib, true, upCard);
        else
            aPts = cribEng.figure_Points(hands[H_PLAYER], false, upCard);

        // get how many points player thinks they have...
        CribbageScoreDlg csd = new CribbageScoreDlg(frame, cribFlg, aPts, mbStopImg);
        csd.show();
        ePts = csd.getPoints();

        // score the points for the player to what they think they had
        movePeg(H_PLAYER, ePts);

        // did they miss any :) - check for later...
        mugginsPoints = aPts - ePts;
    }

    /** Method used to fire off a play event. */
    protected void firePlayEvent(int event)
    {
        CribPlayEvent cpe = new CribPlayEvent(this, event);

        // pass it off to the listener (there's only one...)
        crbply.update(cpe);
    }

    /** Method used to reset the upcard to card back image (no upcard). */
    protected void resetUpcard()
    {
        if (upCard != null) {// reset upcard
            upCard = null;
            icUpCard.setImage(cImgs.getCardImage(props.getCardBackImage()));
        }
    }

    /** Method used to display the current scores on the screen. */
    protected void showScore()
    {
        try {
            lblCompScore.setText("" + cbBoard.getScore(C_PLAYER));
            lblPlayerScore.setText("" + cbBoard.getScore(H_PLAYER));
        }
        catch (Exception e) {
            System.err.println("showScore() error: " + e);
        }
    }

    /** Method used to initialize the game (either to play first time or new game). */
    protected void init_Game()
    {
        try {
            cbBoard.resetBoard(); // reset board event will reset other things of the game
        }
        catch (Exception e) {
            System.err.println("init_Game() - board reset: " + e);
        }
    }

    /** Method used to clear off (and reset some variables) the play area. */
    protected void clearPlayedCards()
    {
        handTally = 0; goFlagP = false; goFlagC = false;
        lblGT.setText("" + handTally);
        playedCards.removeAll();
        playArea.repaint();
    }

    /** Method used to start a cribbage hand. */
    protected void startHand()
    {
        clearPlayedCards();
        resetUpcard();
        iDiscardCount = 0;
        cards.shuffle();
        if (cribFlag)
            lblCrib.setBackground(Color.red);  // player's crib
        else
            lblCrib.setBackground(Color.blue); // computer's crib
        scoreLastPoint = true;
        // empty card hands
        for (int i = 0; i < hands.length; i++) {
            hands[i].removeAll();
            playHands[i].removeAll();
        }
        crib.removeAll();
        // deal cards
        for (int i = 0; i < 6; i++) {
            hands[C_PLAYER].add(cards.getNextCard()); // computer's hand
            hands[H_PLAYER].add(cards.getNextCard()); // player's hand
            buttons[i].setVisible(true);
            buttons[i].setEnabled(true);
            buttons[i].setText(S_DISCARD);
        }
        // show player's cards
        for (int i = 0; i < 6; i++) {
            cardImgs[i].setImage(cImgs.getCardImage((PlayingCard) hands[H_PLAYER].cardAt(i)));
        }
    }

    /** Method used to close out the game and prepare (potentially) for a new
     * game.
    */
    protected void finishGame(int player)
    {
        gameOver = true; waitForMoves = false;
        if (player == H_PLAYER) {
            showMessageBox("You've won, congratulations.");
            playerGamesWon++;
            lblPlayerGames.setText("" + playerGamesWon);
        }
        else {
            showMessageBox("I've won.");
            compGamesWon++;
            lblCompGames.setText("" + compGamesWon);
        }
        // reset some of the game now
        contBtn.setEnabled(false);
        goBtn.setEnabled(false);
        for (int i = 0; i < 6; i++) {
            buttons[i].setVisible(false);
            cardImgs[i].setImage(cImgs.getCardImage(placeHolder));
        }
    }

    /** Method used to deal the up card. */
    protected void getUpCard()
    {
        upCard = (PlayingCard) cards.getNextCard();
        icUpCard.setImage(cImgs.getCardImage(upCard));
        if (upCard.getCardPointValue() == PlayingCard.JPC_JACK) {
            // ---> message (if verbose mode)???
            if (cribFlag)
                movePeg(H_PLAYER, 2);
            else
                movePeg(C_PLAYER, 2);
        }
    }

    /** Method used to enable or disable the players card buttons.
     * @param enab If true, enables the controls, else disables them.
    */
    protected void enablePlayerBtns(boolean enab)
    {
        for (int i = 0; i < 6; i++) {
            if (buttons[i].isVisible()) {
                buttons[i].setEnabled(enab);
            }
        }
    }

    /** Method used to enable or disable the buttons/other controls in the
     * cribbage game.
     * @param enab If true, enables the controls, else disables them.
    */
    protected void enableControls(boolean enab)
    {
        newBtn.setEnabled(enab);
        cbSndOn.setEnabled(enab);
        enablePlayerBtns(enab);
    }

    /** Method used to display a message box (with OK button) with the 
     * specified message.
     * @param msg String containing the message to display.
    */
    protected void showMessageBox(String msg)
    {
        // if (cbBoard.isSoundOn()) getToolkit().beep();
        JOptionPane.showMessageDialog(frame, msg, "Information",
                                      JOptionPane.INFORMATION_MESSAGE, mbOKImg);
    }

    /** Method used to play a card from the playHands into the playedHand
     * for one of the players.
     * @param who Which player to play the card for.
     * @param cardIndex Which card to play from the hand.
    */
    protected void play_Card(int who, int cardIndex)
    {
        pPlayedLast = (who == H_PLAYER); // true if player played, else false for computer
        // play card
        int t = playHands[who].cardAt(cardIndex).getCardPointValueFace10();
        playedCards.add(playHands[who].remove(cardIndex));
        // refresh the play area now
        handTally += t;
        lblGT.setText("" + handTally);
        playArea.repaint();
    }

    /** Method used to do the play for a player or computer.
     * @param who Which player to play the card for.
     * @param cardIndex Which card to play from the hand.
    */
    protected void doThePlay(int who, int cardIndex)
    {
        ArrayList a1 = new ArrayList();
        int k = 0;
        String s = (who == H_PLAYER) ? props.getPlayName() : props.getCompName();

        play_Card(who, cardIndex);
        k = cribEng.pointsInGame(playedCards);
        a1 = cribEng.getScoreMessages();
        if (!a1.isEmpty()) { // show score messages
            for (Iterator itr = a1.iterator(); itr.hasNext(); )
                showMessageBox(s + (String) itr.next());
        }
        if (k > 0) movePeg(who, k); // need this now
        if (handTally == 31) {
            clearPlayedCards();
            X31 = true;
        }
        else {
            X31 = false;
        }
    }

    /** Method used to display card hands for scoring. */
    protected void show_Cards(CardHand hnd)
    {
        for (int i = 0; i < 4; i++) {
            buttons[i].setVisible(false);
            cardImgs[i].setImage(cImgs.getCardImage((PlayingCard) hnd.cardAt(i)));
        }
    }

    // ------------------------- Public Methods -----------------------------

    /** Method used to initialize the application/applet. */
    public void init()
    {
        Image temp = null;
        String sPath = props.getImagePath();
        boolean bSOn = props.getGameSoundsOn();
        pAlwaysStart = props.getAlwaysStart();

        // load message box images
        temp = Utilities.loadImage(this, getToolkit(), sPath, "info.gif");
        mbOKImg = new ImageIcon(temp);
        temp = Utilities.loadImage(this, getToolkit(), sPath, "hand.gif");
        mbStopImg = new ImageIcon(temp);

        // setup the handler classes
        btnhnd = new ButtonHandler();
        crbhnd = new CribBoardHandler();
        crbply = new CribPlayHandler();

        // get content panel to drop components on
        if (frame != null) { // only set up menu if application
            frame_cp = frame.getContentPane();
            setupMenu();
        }
        else { // setup this if applet
            frame_cp = this.getContentPane();
            frame_cp.setLayout(new BorderLayout());
        }

        // do other setup
        setupPanels();

        // setup some other things
        cbBoard.addCribBoardListener(crbhnd);
        try {
            cbBoard.initializeBoard();
        }
        catch (Exception e) {
            System.err.println("Cribbage board initialization error: " + e);
        }

        // setup sound now if board allows and set in props file
        if (bSOn) {
            cbBoard.setSoundOn(bSOn);
            cbSndOn.setSelected(bSOn);
        }

        // setup more frame stuff
        if (frame != null) {
            frame.setSize(W_WIDTH, W_HEIGHT);
            Utilities.centerWindowOnScreen(frame, W_WIDTH, W_HEIGHT);
            frame.setResizable(false);
            frame.setVisible(true);
        }

        // load card images
        new CardLoader().start();
    }

    /** Method used to start the process from a 'main' method. */
    public void go()
    {
        init();
        start(); // just in case we need...
    }

    /** Method used as entry point if running as an application. */
    public static void main(String[] args)
    {
        SwingUtilities.invokeLater(new Runnable() {
                                     public void run() {
                                         Cribbage win = new Cribbage();

                                         win.createFrame();
                                         win.go();
                                     }
                                   });
    }

    // ----------------------- Inner Classes --------------------------

    /**
     * Class used to initialize the card images during startup.  Runs as
     * a 'SwingWorker' thread to 'waitfor' the card images until all are
     * loaded.
    */
    class CardLoader extends SwingWorker
    {
        /** Run 'waitfor' routine until all cards images fully loaded.
         * Return a string containing 'success'.
        */
        public Object construct()
        {
            cImgs.waitForImages(frame_cp);
            return "Success";
        }

        /** Update U/I to indicate load is finished. */
        public void finished()
        {
            // do nothing...
        }
    }

    /** Class used to handle the cribbage board events for Cribbage. */
    class CribBoardHandler implements CribBoardListener
    {
        /** CribBoardHandler constructor. */
        public CribBoardHandler() { }

        /** Method used to show score and enable controls. */
        private void doIt(boolean endOfGame)
        {
            showScore();
            if (mugginsPoints > 0) { // hehe
                showMessageBox("MUGGINS! You missed " + mugginsPoints +
                               " points (which I get).");
                movePeg(C_PLAYER, mugginsPoints);
                mugginsPoints = 0;
            }
            else {
                enableControls(true);
                if (!endOfGame) waitForMoves = false; // reset in finishGame if eOG
            }
        }

        /** Method called when a cribbage board event occured.
         * @param event The cribbage board event that occured.
        */
        public void update(CribBoardEvent event)
        {
            int evt = event.getEvent();

            // what happened?
            if (DEBUG_ON) System.out.println(event);

            // process events
            if (evt == CribBoardEvent.CBE_MOVE_STARTED) {
                waitForMoves = true;
                enableControls(false);
            }
            else if (evt == CribBoardEvent.CBE_MOVE_COMPLETED) {
                if (DEBUG_ON) {
                    try {
                        System.out.println("Current Score: (0):" + cbBoard.getScore(0) +
                                           " (1):" + cbBoard.getScore(1));
                    }
                    catch (Exception e) { }
                }
                doIt(false);
            }
            else if (evt == CribBoardEvent.CBE_PLAYER_WON) {
                int iPlayer = event.getWinningPlayer();

                mugginsPoints = 0;
                doIt(true);
                finishGame(iPlayer);
            }
            else if (evt == CribBoardEvent.CBE_CRIB_BOARD_RESET) {
                gameOver = false;
                showScore(); // should be zero at this point...
                lblCrib.setBackground(Color.lightGray);
                resetUpcard();
            }
        }
    }

    /** Class used to handle some 'button' events for Cribbage. */
    class ButtonHandler implements ActionListener
    {
        /** ButtonHandler constructor. */
        public ButtonHandler() { }

        /** Method used to discard cards into crib hand from selected hand. */
        private void doDiscard(int who, int[] what)
        {
            crib.add(hands[who].remove(what[0]));
            crib.add(hands[who].remove(what[1]));
            hands[who].compressHand();
        }

        /** Method used to handle the continue button event. */
        private void handleContinue()
        {
            // computers discards
            int[] dis = cribEng.analyzeHandForDiscards(hands[C_PLAYER]);
            if (DEBUG_ON) {
                System.out.println("Computers discards: " + dis[0] + ", " + dis[1]);
            }
            doDiscard(C_PLAYER, dis);
            lblCL.setText("" + hands[C_PLAYER].getCardCount());

            // players discards
            int j = 0;
            for (int i = 0; i < 6; i++) {
                if (buttons[i].getText().equals(S_KEEP)) dis[j++] = i;
                if (j == 2) break;
            }
            if (DEBUG_ON) {
                System.out.println("Players discards: " + dis[0] + ", " + dis[1]);
            }
            doDiscard(H_PLAYER, dis);

            // copy hands to hands to play with (need original to score at end of hand)
            for (int i = 0; i < 4; i++) {
                playHands[C_PLAYER].add(hands[C_PLAYER].cardAt(i));
                playHands[H_PLAYER].add(hands[H_PLAYER].cardAt(i));
            }

            // reset player card display
            for (int i = 0; i < 6; i++) {
                buttons[i].setText(S_PLAY);
                if (i > 3) { // only need to show first 4
                    buttons[i].setVisible(false);
                    cardImgs[i].setImage(cImgs.getCardImage(placeHolder));
                }
                else {
                    cardImgs[i].setImage(cImgs.getCardImage((PlayingCard) playHands[H_PLAYER].cardAt(i)));
                }
            }

            // finish setting up
            getUpCard();
            contBtn.setEnabled(false);
            if (cribFlag)
                firePlayEvent(CribPlayEvent.CPE_PLAYERS_PLAY);
            else
                firePlayEvent(CribPlayEvent.CPE_COMPUTERS_PLAY);
        }

        /** Method used to handle the 'go' button event processing. */
        private void handleGo()
        {
            // player really has no play???
            for (int i = 0; i < 4; i++) {
                if ((buttons[i].isVisible()) && 
                    ((handTally +
                      playHands[H_PLAYER].cardAt(i).getCardPointValueFace10()) <= 31)) {
                    showMessageBox("Shame on you, you have a play.");
                    return;
                }
            }
            // process 'go' now
            if (goFlagC) { // comp has already gone
                showMessageBox(props.getPlayName() + S_LASTCARD);
                movePeg(H_PLAYER, 1);
                clearPlayedCards();
            }
            else { // maybe move on???
                if (playHands[C_PLAYER].getCardCount() > 0) {
                    goFlagP = true;
                }
                else {
                    scoreLastPoint = false;
                    showMessageBox(props.getCompName() + S_LASTCARD);
                    movePeg(C_PLAYER, 1);
                    clearPlayedCards();
                }
            }
            // let computer go now
            firePlayEvent(CribPlayEvent.CPE_COMPUTERS_PLAY);
        }

        /** Method to handle the button actions. */
        public void actionPerformed(ActionEvent evt)
        {
            String btn = evt.getActionCommand();

            if (btn.equals(S_CONTINUE)) {
                handleContinue();
            }
            else if ((btn.equals(S_DISCARD)) || (btn.equals(S_KEEP))) {
                if (btn.equals(S_DISCARD)) {
                    iDiscardCount++;
                    ((JButton) evt.getSource()).setText(S_KEEP);
                }
                else {
                    iDiscardCount--;
                    ((JButton) evt.getSource()).setText(S_DISCARD);
                }
                if (iDiscardCount == 2) { // setup for rest of game play
                    contBtn.setEnabled(true);
                }
                else {
                    contBtn.setEnabled(false);
                }
            }
            else if (btn.equals(S_GO)) {
                handleGo();
            }
            else if (btn.equals(S_PLAY)) {
                int j = 0, k = 0;
                for (int i = 0; i < 4; i++)
                    if (buttons[i] == evt.getSource()) j = i; // which card is it?
                k = handTally + 
                    playHands[H_PLAYER].cardAt(j).getCardPointValueFace10();
                if (k > 31) { // can't play this...
                    showMessageBox("That totals more than 31, try again.");
                    return;
                }
                buttons[j].setVisible(false);
                cardImgs[j].setImage(cImgs.getCardImage(placeHolder));
                doThePlay(H_PLAYER, j);
                if ((!gameOver) && (!goFlagC))
                    firePlayEvent(CribPlayEvent.CPE_COMPUTERS_PLAY);
            }
            else if ((btn.equals(S_PLAY_GAME)) || (btn.equals(S_NEW_GAME))) {
                if (btn.equals(S_PLAY_GAME)) { // assume game initialized already
                    ((JButton) evt.getSource()).setText(S_NEW_GAME); // reset label in btn
                }
                else {
                    // ---> prompt user - start new game?
                    init_Game(); // initialize game
                }
                if (pAlwaysStart)
                    cribFlag = true;
                else
                    cribFlag = (((int) Math.floor(Math.random() * 2)) == 0);
                if (cribFlag)
                    showMessageBox("Your crib to start.");
                else
                    showMessageBox("My crib to start.");
                startHand();
            }
        }
    }

    /** Class used as a base class for play event threads. This class contains
     * a bogus run method and all functions used by all event threads.
    */
    class BaseT extends Thread
    {
        /** Constructor. */
        public BaseT() { }

        /** Method used to check if the hand is over and done with. */
        public boolean isHandOver()
        {
            boolean bRet = false;

            if ((playHands[C_PLAYER].getCardCount() == 0) &&
                (playHands[H_PLAYER].getCardCount() == 0)) {
                bRet = true;
                firePlayEvent(CribPlayEvent.CPE_SCORE_HAND);
            }

            return bRet;
        }

        /** Method to wait until board movement is complete before moving on. */
        public void waitOnBoard()
        {
            try {
                do {
                    sleep(250);
                } while (waitForMoves);
            }
            catch (InterruptedException e) { }
        }

        /** Bogus run method.  Is over-rode by child classes. */
        public void run()
        {
            return;
        }
    }

    /** Class used to handle the players play event as a thread. */
    class PlayPlayT extends BaseT
    {
        /** Constructor. */
        public PlayPlayT() { }

        /** Thread process method. */
        public void run()
        {
            if (!gameOver) {
                waitOnBoard();
                if (isHandOver()) return;
                if (playHands[H_PLAYER].getCardCount() > 0) {
                    // it is our turn now
                    goBtn.setEnabled(true); // ---> should be using events for this!!!!!
                    enablePlayerBtns(true);
                    // ---> verbose mode - show message about turn??
                }
                else {
                    firePlayEvent(CribPlayEvent.CPE_COMPUTERS_PLAY);
                }
            }
        }
    }

    /** Class used to handle the computers play event as a thread. */
    class CompPlayT extends BaseT
    {
        /** Constructor. */
        public CompPlayT() { }

        /** Method used to score last point for computer. */
        private void doOneMove()
        {
            showMessageBox(props.getCompName() + S_LASTCARD);
            movePeg(C_PLAYER, 1);
            clearPlayedCards();
        }

        /** Method used to process the computer play event. */
        public void run()
        {
            boolean done = false;
            int cc = -1;

            if (!gameOver) {
                // disable players options
                goBtn.setEnabled(false); // ---> should be using events for this!!!!!
                enablePlayerBtns(false);
                waitOnBoard();
                // is hand/game over? (gameOver added with v1.13)
                if ((isHandOver()) || (gameOver)) return;
                // play card(s) from computer
                if (playHands[C_PLAYER].getCardCount() > 0) {
                    // ---> verbose mode - show message about computers play???
                    do { // may be doing multiple moves...
                        waitOnBoard();
                        cc = cribEng.getCardToPlay(playedCards, playHands[C_PLAYER]);
                        if (cc != -1) { // have a card to play
                            doThePlay(C_PLAYER, cc);
                            lblCL.setText("" + playHands[C_PLAYER].getCardCount());
                        }
                        else { // process 'go'
                            if (goFlagP) { // player has already gone
                                doOneMove();
                            }
                            else { // computer pushes go first
                                if (playHands[H_PLAYER].getCardCount() != 0) {
                                    showMessageBox("I don't have a play, Go.");
                                    goFlagC = true;
                                }
                                else {
                                    scoreLastPoint = false;
                                    showMessageBox("I have no play, you get last point.");
                                    movePeg(H_PLAYER, 1);
                                    clearPlayedCards();
                                }
                            }
                        }
                        done = ((!goFlagP) ||
                                (playHands[C_PLAYER].getCardCount() == 0) ||
                                (gameOver));
                    } while (!done);
                    waitOnBoard();
                    // fix (1.08) so that 'last card' point is correct (I think)
                    if ((playHands[C_PLAYER].getCardCount() == 0) &&
                        (playHands[H_PLAYER].getCardCount() != 0) &&
                        (goFlagP)) { // player has gone, comp played out cards...
                        doOneMove();
                        waitOnBoard();
                        goFlagP = true; // reset this just in case
                    }
                }
                // give turn to player now
                if (!gameOver) firePlayEvent(CribPlayEvent.CPE_PLAYERS_PLAY);
            }
        }
    }

    /** Class to handle the score event as a thread. */
    class ScoreT extends BaseT
    {
        /** Constructor. */
        public ScoreT() { }

        public void run()
        {
            int pts = 0;

            if (!gameOver) {
                waitOnBoard();
                // score point for last card?
                if ((!X31) && (scoreLastPoint)) {
                    if (pPlayedLast) {
                        showMessageBox(props.getPlayName() + S_LASTCARD);
                        movePeg(H_PLAYER, 1);
                    }
                    else {
                        showMessageBox(props.getCompName() + S_LASTCARD);
                        movePeg(C_PLAYER, 1);
                    }
                    waitOnBoard();
                }
                if (!gameOver) { // could've won with last card point
                    clearPlayedCards();
                    // ---> verbose mode, show message who scores first???
                    pts = cribEng.figure_Points(hands[C_PLAYER], false, upCard);
                    if (cribFlag) { // computer scores first
                        show_Cards(hands[C_PLAYER]);
                        showMessageBox(props.getCompName() + " has " + pts + " points.");
                        movePeg(C_PLAYER, pts);
                        waitOnBoard();
                        if (!gameOver) { // score player
                            show_Cards(hands[H_PLAYER]);
                            get_Points(false);
                            waitOnBoard();
                            if (!gameOver) { // get crib points from player
                                show_Cards(crib);
                                get_Points(true);
                                waitOnBoard();
                            }
                        }
                    }
                    else { // player scores first
                        show_Cards(hands[H_PLAYER]);
                        get_Points(false);
                        waitOnBoard();
                        if (!gameOver) { // score computer now
                            show_Cards(hands[C_PLAYER]);
                            showMessageBox(props.getCompName() + " has " + pts + " points.");
                            movePeg(C_PLAYER, pts);
                            waitOnBoard();
                            if (!gameOver) { // score crib for computer
                                show_Cards(crib);
                                pts = cribEng.figure_Points(crib, true, upCard);
                                showMessageBox("There are " + pts + " points in the crib.");
                                movePeg(C_PLAYER, pts);
                                waitOnBoard();
                            }
                        }
                    }
                }
                if (!gameOver) { // setup for next hand
                    cribFlag = !cribFlag;
                    startHand();
                }
            }
        }
    }

    /** Class used to handle the cribbage play events for the cribbage game. */
    class CribPlayHandler implements CribPlayListener
    {
        /** Constructor to create the handler. */
        public CribPlayHandler() { }

        /** Method used to listen for play events on.
         * @param event CribPlayEvent the occured.
        */
        public void update(CribPlayEvent event)
        {
            int evt = event.getEvent();

            // what happened?
            if (DEBUG_ON) System.out.println(event);

            // process the event
            if (evt == CribPlayEvent.CPE_PLAYERS_PLAY) {
                new PlayPlayT().start();
            }
            else if (evt == CribPlayEvent.CPE_COMPUTERS_PLAY) {
                new CompPlayT().start();
            }
            else if (evt == CribPlayEvent.CPE_SCORE_HAND) {
                new ScoreT().start();
            }
        }
    }

    /** Class (component) used to deal with the cribbage playing area.  Basically
     * component sets up a set size playing area and handles the painting of the
     * playing area for the game.  Created as an inner class so that it can take
     * advantage of the 'played' cards hand to paint with.
    */
    class PlayArea extends javax.swing.JComponent
    {
        /** Playing area size reference. */
        private Dimension dim = new Dimension(184, 110);

        /** Constructor to create the cribbage playing area (where the cards are played). */
        public PlayArea()
        {
            super();

            // set size
            setPreferredSize(dim);
        }

        /** Method to return the accessibilty context to the caller.  This class
         * returns null.
         * @return A AccessibleContext object for this class.  There is currently no
         * support for this so only a null is returned for this method.
        */
        public javax.accessibility.AccessibleContext getAccessibleContext()
        {
            return null;
        }

        /** Overrode method used to display the playing area on the screen.
         * @param g Graphic instance used to draw with.
        */
        public void paint(Graphics g)
        {
            int x = 4, y = 6;

            // set up background border color
            g.setColor(Color.gray);
            g.fillRect(0, 0, dim.width, dim.height);

            // paint the played cards now (if any)
            int count = playedCards.getCardCount();
            for (int i = 0; i < count; i++) {
                Card card = playedCards.cardAt(i);
                if (card != CardHand.EMPTY_CARD) { // paint it
                    g.drawImage(cImgs.getCardImage((PlayingCard) card), x, y, this);
                    x += PlayingCardImageCache.IMAGE_OFFSET;
                }
            }
        }
    }
}
