import java.util.*;
import java.io.*;
import com.slackandassociates.cards.CardEnum;
import com.slackandassociates.cards.playingcards.PlayingCardEnum;

/**
 * Class defines the 'settable' properites and settings used by the Cribbage
 * game along with the defaults.  Properties are stored in a properties file
 * called 'Cribbage.properties' and should be accessible via the Properties
 * class. <br>
 * There are a few properties that should not be changed.  These are marked
 * in the property file. <br>
 * The following are the properties available for setting:
 * <ul>
 * <li> Cribbage.ImagePath = <path to cribbage game images (including board images)>.
 * Defaults to 'images/'.
 * <li> Cribbage.SoundPath = <path to cribbage game sounds (including board sounds)>.
 * Defaults to 'sounds/'.
 * <li> Cribbage.SoundsOn = <anything for true, default is false>.  Sets the game
 * sounds on option to 'on' from the start if set.
 * <li> Cribbage.CardBackImage = <0 through 5>.  Indexes into a private array of the
 * card back image enumerations.  Default is '2'.
 * <li> Cribbage.AlwaysStart = <anything for true>.  Cheat mode parameter (must be
 * set in the properties file) that sets the cribflag to the player at game start
 * always (no 'draw' for first crib).
 * <li> Cribbage.HelpPath = <path to cribbage game help html files>.  Defaults to
 * 'help/'.
 * <li> Cribbage.ComputerName = <string containing a name>.  Defaults to "Computer".
 * <li> Cribbage.PlayerName = <string containing a name>.  Defaults to "Player".
 * </ul>
 * <b>Changes:</b>
 * <ul>
 * <li> 2001-09-27 - Initial Release.
 * <li> 2001-10-06 - Added property for the help path.
 * <li> 2001-10-13 - Added version/email properties (not user changable).
 * <li> 2002-07-11 - Added computer/player name properties.
 * <li> 2005-01-20 - Modified for cards v2 library.
 * </ul>
 * @author Michael G. Slack
 * @author slack@attglobal.net
 * @created 2001-09-27
 * @version 2005-01-20 Version 1.04
*/
public class CribbageProps
{
    // private statics

    /** Properties prepender (key lead). */
    private static final String S_LEAD = "Cribbage.";

    /** Properties file name. */
    private static final String S_PROPSNAME = S_LEAD + "properties";

    /** Default value for the path to the images. */
    private static final String S_IMAGEPATH = "images/";

    /** Default value for the path to the sound files used. */
    private static final String S_SOUNDPATH = "sounds/";

    /** Default value for card back image (as index into array). */
    private static final int I_CARDBACK = 2;

    /** Default value for the path to the help html files. */
    private static final String S_HELPPATH = "help/";

    /** Default value for computer name. */
    private static final String S_COMPNAME = "Computer";

    /** Default value for player name. */
    private static final String S_PLAYNAME = "Player";

    // private references

    /** Card backs (represented as an array). */
    private CardEnum[] cardBacks = {PlayingCardEnum.JC_CARDBACK_VAL1,
                                    PlayingCardEnum.JC_CARDBACK_VAL2,
                                    PlayingCardEnum.JC_CARDBACK_VAL3,
                                    PlayingCardEnum.JC_CARDBACK_VAL4,
                                    PlayingCardEnum.JC_CARDBACK_VAL5,
                                    PlayingCardEnum.JC_CARDBACK_VAL6};

    /** Properties object used. */
    private Properties props = null;

    // ------------------------- Constructor -------------------------------

    /** Constructor - readies for property read.
     * @param parent Parent object creating the properties object.
    */
    public CribbageProps(Object parent)
    {
        props = new Properties(); // no defaults

        try { // to load properties now
            InputStream in = parent.getClass().getResourceAsStream(S_PROPSNAME);
            props.load(in);
            in.close();
        }
        catch (Exception e) {
            System.err.println("Properties failed to load: " + e);
        }
    }

    // ------------------------- Public Methods -----------------------------

    /** Method to return the version information string to the caller. */
    public String getVersionInfo()
    {
        if (props != null)
            return props.getProperty(S_LEAD + "Version");
        else
            return "";
    }

    /** Method used to return the author email string to the caller. */
    public String getAuthorEmail()
    {
        if (props != null)
            return props.getProperty(S_LEAD + "eMail");
        else
            return "";
    }

    /** Method returns the image path from the properties to the caller. */
    public String getImagePath()
    {
        String sRet = S_IMAGEPATH;

        if (props != null) {
            String sTemp = props.getProperty(S_LEAD + "ImagePath");
            if ((sTemp != null) && (!"".equals(sTemp))) sRet = sTemp;
        }

        return sRet;
    }

    /** Method returns the sound path from the properties to the caller. */
    public String getSoundPath()
    {
        String sRet = S_SOUNDPATH;

        if (props != null) {
            String sTemp = props.getProperty(S_LEAD + "SoundPath");
            if ((sTemp != null) && (!"".equals(sTemp))) sRet = sTemp;
        }

        return sRet;
    }

    /** Method to return if game sounds will be turned on from the beginning of
     * the game (as opposed to the user turning them on later).  The user can
     * still turn them off.
    */
    public boolean getGameSoundsOn()
    {
        boolean bRet = false;

        if (props != null) {
            String ss = props.getProperty(S_LEAD + "SoundsOn");
            if ((ss != null) && (!"".equals(ss))) bRet = true;
        }

        return bRet;
    }

    /** Method to return the card back image to use (as a card image enumeration)
     * to the caller.
    */
    public PlayingCardEnum getCardBackImage()
    {
        PlayingCardEnum ceRet = (PlayingCardEnum) cardBacks[I_CARDBACK];

        if (props != null) {
            String ss = props.getProperty(S_LEAD + "CardBackImage");
            if ((ss != null) && (!"".equals(ss))) {
                int ii = 0;
                try {
                    ii = Integer.parseInt(ss);
                }
                catch (Exception e) {
                    ii = I_CARDBACK;
                }
                if ((ii < 0) || (ii >= cardBacks.length)) ii = I_CARDBACK;
                ceRet = (PlayingCardEnum) cardBacks[ii];
            }
        }

        return ceRet;
    }

    /** Method to return if the player always gets first crib during game
     * start.  If false, first crib is 'drawn' (randomized).
    */
    public boolean getAlwaysStart()
    {
        boolean bRet = false;

        if (props != null) {
            String ss = props.getProperty(S_LEAD + "AlwaysStart");
            if ((ss != null) && (!"".equals(ss))) bRet = true;
        }

        return bRet;
    }

    /** Method returns the help path from the properties to the caller. */
    public String getHelpPath()
    {
        String sRet = S_HELPPATH;

        if (props != null) {
            String sTemp = props.getProperty(S_LEAD + "HelpPath");
            if ((sTemp != null) && (!"".equals(sTemp))) sRet = sTemp;
        }

        return sRet;
    }

    /** Method returns the computers name from the properties to the caller. */
    public String getCompName()
    {
        String sRet = S_COMPNAME;

        if (props != null) {
            String sTemp = props.getProperty(S_LEAD + "ComputerName");
            if ((sTemp != null) && (!"".equals(sTemp))) sRet = sTemp;
        }

        return sRet;
    }

    /** Method returns the computers name from the properties to the caller. */
    public String getPlayName()
    {
        String sRet = S_PLAYNAME;

        if (props != null) {
            String sTemp = props.getProperty(S_LEAD + "PlayerName");
            if ((sTemp != null) && (!"".equals(sTemp))) sRet = sTemp;
        }

        return sRet;
    }
}
