/* sort of like a xcopy command, but will copy only missing files
   or optionally where the 'datestamp' (date, time, size) has changed.
   Also, subsequent lower level subdirectories can be copied too with '/s'

   In general, any time a file is copied, the "A" attribute is set on the destination
                        if not copied, existing file on dest "A" attribute reset

   Options:  /u  update only newer changed existing files (missing files not are copied)
             /a  copy src files that have the "A" attribute set even if otherwsie the same
             /m  copy src files with "A" attribute set (as above) and reset the src "A" attribute
             /d  to compare date, time, and size to determine "newest" and copy if different
                    otherwise only missing files are copied.
             /b  bi-directional of destination back to source based on "newest" date/time
             /s  for include subdirectories
                 variation: /s<N> up to N levels deep, default no limit
             /h  to include hidden and system files too
             /fno FileName output actions to "FileName" rather than executing them.
             /v  display version # of the exec
             /t  Tracing options  (/t2 is even more verbose tracing)
             /td    stop a predefined TRACE("?R") points
             /tr<N> stop when recursing to sub directory level N  (N defaults to 1)
             /l  to enable .LONGNAME extended attributes
                 (Note, this can improve performance of this Rexx script
                        if the attribute checks are not needed.)
                 (Note, other EAs are handled per "copy" & "xcopy" rules.)
             /EA?  copy failed EAs as separate file suffixed as "-$EAs"
             /lst fn  means 'fn' is a file containing the list of source files vs a file mask
                       (fn typically generated from 'DriveDir.cmd')

   Examples:
       xeroxit x: /s    will copy missing or date/time/size changed from the current directory on x: and subdirs
                        to the current directory.
*/

szVersion = '2.1.1'

/* remember where we presently are */
here = directory()

/* use our name as a base for an 'errors' file in the current directory */
parse source OS . pgm .
exname = filespec('name',pgm)
ldot = lastpos('.',exname)
if ldot < 2 then ldot = length(exname) + 1
exbase = left(exname,ldot-1)
parse arg args '(' recursion_level dest_is8.3 errors .

if translate(left(OpSys,7)) = 'WINDOWS' | translate(left(OpSys,4)) = 'OS/2' then do
   /* Insure the SysXXXXX functions are ready to use */
   Call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
   Call SysLoadFuncs
end
  
/* Read screen size to implement paging the output if requested by '/p' */
parse value SysTextScreenSize() with rows cols

if args = '' | args = '?' then do
   say 'using' pgm '   version' szVersion
   say 'Function:  Makes a copy of missing (or changed files)'
   say '  Any file copied during the run will have the destination "A" attribute set.'
   say '  Destination files not updated will have destination "A" attribute reset.'
   say ''
   say 'Syntax' exbase 'source_mask <dest> </s> </d> </v> </v2> </h> </l-> </erase>'
   say '  where <dest> defaults to the current directory'
   say ' Options:'
   say '   /u  update only existing files. (Missing files are NOT copied.)'
   say '   /a  copy matching src files that have the "A" attribute set'
   say '          regardless of date/time/size compare test.'
   say '   /m  same as /a plus reset the "A" src attribute if copy was successful'
   say '   /d to compare date, time, and size'
   say '        allows copying of changed files that already exist'
   say '        by selecting the newest as "best" version.'
   say '        Without the /d option, only missing files are copied.'
   say '   /b  bi-directional of destination back to source of "newest".'
   say '         This implies the /d and /u options automatically.'
   say '   /s for include all appropriate subdirectories.'
   say '        /s<N> subdirectories up to N levels of nesting.'
   say '   /h to include hidden and system files & directories too.'
   '@pause'
   say '   /fno FileName output actions to "FileName" rather than executing them.'
   say '   /t for tracing options  (/t2 is even more verbose tracing)'
   say '       /td    stop a predefined TRACE("?R") points'
   say '       /tr<N> debug recursion level N where N defaults to 1st recursion.'
   say '   /l to enable preserving the "longfile" name on a FAT file system'
   say '             using the .LONGNAME EA on FAT file systems.'
   say '        By default, .LONGNAME Extended Attribute is not set.'
   say '        Other extended attributes are handled by the normal'
   say '             "copy" and "xcopy" Operating System functions.'
   say '   /EA?  copy failed EAs as separate file suffixed as "<filename>-$EAs".'
   say '   /erase will erase existing destination files and subdirectories '
   say '      that do not have a correspondence to the source.'
   say '      Use /erase- to erase extraneous files in subdirectories'
   say '         BUT preserve extraneous files in the top subdirectory.'
   say '   /v  display version revision level'
   exit 100
end

fTrace = 0           /* we aren't tracing the logical decisions */
fTraceDebug = 0      /* set to 1 to stop at predefined TRACE("?R") points */
iTraceRecursion = 0  /* stop for tracing going into a recursion level */
mask = ''            /* mask of what to copy ...i.e. everything */
fnc = 0              /* we only have a source file mask, might be used later */
dest = '.'           /* assume current directory  */
bCompare = 0         /* compare flag set from the /d option to compare dates & size */
tattrib = '**-*-'    /* assume no hidden nor system files are considered */
lfn_support = 0      /* assume we don't want to logically handle long file names */
erase_orphans = 0    /* do not erase extraneous files on the destination */
inclSubdirs = 0      /* assume no recursion into subdirectories */
lstFN = ''           /* assume no indirect lstFile containing the names to copy */
bBiDirectional = 0   /* '1' with copy dest files to source if missing */
bUpdateOnly = 0      /* 0 = missing & changed files, 1 = update only existing files */
szOutputFile = ''    /* default execution the the appropriate commands */

     /* the '/a' and '/m' options are mutually exclusive and don't mimic xcopy rules */
bConsiderA = 0       /* ignoring 'A', only looking at 'HRS' */
                     /*   otherwise copy if source 'A' is set & dest 'A' not set */
bMarkCopied = 0      /* set the 'A' dest attribute if copied, otherwise reset it */

fHandleEAs = 0       /* 1 will try copying the file without EA's if needed */
EA_Alternate_Suffix = '-$EAs'  /* and put the EAs into corresponding filename with this suffix */

/* parse the command line arguments */
origArgs = ''
do while args <> ''
   parse var args next args
   if left(next,1) = '/' | left(next,1) = '-' then do
      origArgs = origArgs next
      next = translate(substr(next,2))
      if left(next,1) = 'S' then do
         inclSubdirs = 1
         si = substr(next,2)
         if datatype(si) = 'NUM'
            then max_recursion_level = si
            else max_recursion_level = 32767         /* effectively, unlimitted recursion */
      end
      if left(next,1) = 'D' then bCompare = 1
      if left(next,1) = 'B' then do 
         bBiDirectional = 1
         bCompare = 1 
         bUpdateOnly = 1
      end
      if left(next,3) = 'FNO' then do                /* file name output */
         /* assume name immediately follows */ 
         szOutputFile = substr(word(origArgs,words(OrigArgs)),5)   /* preserve the original case */
         if length(szOutputFile) = 0 then do
            parse var args szOutputFile args         /* it must be the next arg */
            if left(szOutputFile,1) = '/' then do    /* does it look like we did not get a filename */
               say 'ERROR: /fno option requires a file name'
               exit 8
            end
         end
         else /* we want to standardize the filename to be the next word, not butted up with the /f0 option */ 
            origArgs = delword(origArgs,words(origArgs)) '/fno'

         /* anchor the output file name to a full path for recursion */
         if filespec('drive',szOutputFile) = '' & filespec('path',szOutputFile) = '' then do
            p = directory()
            if right(p,1) <> '\'
               then p = p||'\'
            szOutputFile = p||szOutputFile           
         end
         origArgs = origArgs szOutputFile            /* update for recursion */
      end
      if left(next,1) = 'U' then do
         bUpdateOnly = 1
         bCompare = 1
      end
      if left(next,1) = 'A' then bConsiderA = 1       /* regardless, copy if the the 'A' attribute set in the source */
      if left(next,1) = 'M' then do                   /* like A but also reset src "A" attribute */
         bMarkCopied = 1
         bConsiderA = 1
      end
      if left(next,1) = 'T' then do                   /* trace */
         if translate(substr(next,2,1)) = 'D'         /* trace Debug vs Trace level */
            then fTraceDebug = 1
         else if translate(substr(next,2,1)) = 'R' then do /* trace a recursion level */
            iTraceRecursion = substr(next,3)          /* a specific subdirectory recursion level requested? */
            if iTraceRecursion < 1 | datatype(iTraceRecursion) <> 'NUM'
               then iTraceRecursion = 1               /* default to 1st level recursion */
         end                                          
         else do
            fTrace = substr(next,2)
            if fTrace = '' then fTrace = 1
         end
      end
      if left(next,1) = 'H' then tattrib = '*****'     /* consider Hidden & System files too */
      if left(next,3) = 'LST'
         then parse var args lstFN args
      else if left(next,1) = 'L' then lfn_support = 1

      if left(next,3) = 'EA?' then do 
         fHandleEAs = 1 
         say ' "/EA?" means: Will attempt to put EAs into <filename>'||EA_Alternate_Suffix 'if the copy fails.'
      end

      if left(next,5) = 'ERASE' then erase_orphans = 1   /* erase orphans even in 1st recursion */
      if left(next,6) = 'ERASE-' then erase_orphans = 2  /* erase only from subdirectories */
      if left(next,1) = 'V' then do
         say 'using' pgm '   version' szVersion
         exit 100
      end
   end
   else do
      /* handle quoted file names that may have embedded blanks */
      if left(next,1) == '"' & right(next,1) \= '"' then do
         endQuote = pos('"',args)
         next = next substr(args,1,endQuote)
         args = substr(args,endQuote + 1)
      end
      next = strip(next,'B','"')

      fnc = fnc + 1
      select
         when fnc = 1 then mask = next
         when fnc = 2 then dest = next
         otherwise do
            say 'invalid argument' next
            exit 8
         end
      end
   end
end

if recursion_level = '' then do
   rc = time('R')               /* reset & start elapsed time counter */
   say pgm ' Begin execution at:' date() time()
   recursion_level = 0
   if right(here,1) \= '\'
      then errors = here||'\'||exbase||'.err'
      else errors = here||exbase||'.err'
   if stream(errors,'c','query exists') \= '' then '@erase' errors
   if szOutputFile <> '' 
      then if stream(szOutputFile,'c','query exists') \= '' then '@erase' szOutputFile
   say 'Errors (if any) will be accumulated into' errors
end

/* remember the current working directory on the destination drive */
dest_CWD = directory(filespec('drive',dest))
back = directory(here)                         /* undo damage from the directory call */
recursion_level = recursion_level + 1

if fTrace > 1
   then trace("?R")
/* get information so we can compute relative directory location */
source_drive = filespec('drive',mask)
source_path  = filespec('path',mask)
relativeRoot = source_drive||source_path
mask  = filespec('name',mask)
if mask = '' then mask = '*'

source_cwd   = directory(source_drive)
if right(source_cwd,1)  \= '\'
   then source_cwd = source_cwd||'\'

/* does the mask contain a relative up directory indicator? */
do p=0 by 1 while left(source_path,3) = '..\' 
   source_path = substr(source_path,4)
end
if p > 0 then do                /* if traversing relative up in the directory tree */
   relativeRoot = source_cwd
   do while p > 0               /* remove the appropriate number of subdirectories from the current directory */
      relativeRoot = filespec('drive',relativeRoot)||filespec('path',strip(relativeRoot,'T','\'))
      p = p -1
   end
   relativeRoot = relativeRoot || source_path
end
else if left(source_path,1) \= '\'
   then relativeRoot = source_cwd||source_path

/* get information on all the SOURCE files fitting that mask */
if fTrace > 1 then say '   processing' mask 'at' relativeRoot
if lstFN = ''
   then rc = SysFileTree(relativeRoot||mask,srcFiles,'FL',tattrib)
else do
   si = 0
   status = stream(lstFN,'c','open read')
   Do While left(status,5) = 'READY'
      data = linein(lstFN)
      status = stream(lstFN,'s')
      if left(status,5) = 'READY' then do
         si = si + 1
         srcFiles.si = data
      end
   End
   status = stream(lstFN,'c','close')
   srcFiles.0 = si
end
if srcFiles.0 = 0 & pos('*',mask) = 0 then do
   /* nothing was found, so was the original specification a subdirectory? */
   rc = SysFileTree(mask,srcDirs,'DL',tattrib)
   if srcDirs.0 = 1 then do
      source_path  = filespec('path',mask)
      if source_path = ''
         then mask = directory(mask)||'\*'
         else mask = mask||'\*'
      rc = SysFileTree(mask,srcFiles,'FL',tattrib)
   end
end

/* get information about this subdirectory so we can copy its state onto the destination */
rc = SysFileTree(left(relativeRoot,length(relativeRoot)-1),thisDirectory,'L')
parse var thisDirectory.1 dt tm . curDirectoryAttr curDestDirectory
curDestDirectory = strip(curDestDirectory)
curDirectoryDateTime = dt tm
if left(curDirectoryAttr,1) = 'A'
   then copyDirectoryAttr = '+*'
   else copyDirectoryAttr = '-*'
if substr(curDirectoryAttr3,1) = 'H'
   then copyDirectoryAttr = copyDirectoryAttr||'+'
   else copyDirectoryAttr = copyDirectoryAttr||'-'
if substr(curDirectoryAttr4,1) = 'R'
   then copyDirectoryAttr = copyDirectoryAttr||'+'
   else copyDirectoryAttr = copyDirectoryAttr||'-'
if substr(curDirectoryAttr5,1) = 'S'
   then copyDirectoryAttr = copyDirectoryAttr||'+'
   else copyDirectoryAttr = copyDirectoryAttr||'-'

/* remember where we are */
back = directory(here)

if right(dest,1) = ':'
   then dest = dest||'.\'

if dest \= '' & right(dest,1) \= '\'
   then dest = dest||'\'

/* if we don't know if the destination supports native long file names */
if dest_is8.3 = '' then do
   /* try to generate a file name that is not 8.3 */
   fqDestName = SysTempFileName(filespec('drive',dest)||'\$2345678?.123?')
   if fqDestName \= ''
      then dest_is8.3 = 0       /* native long file name support */
      else dest_is8.3 = 1       /* limited to DOS FAT 8.3 file names */
end

if recursion_level > 1 then do      /* IF it isn't the top level starting point */
   /* we need to insure the destination subdirectory fits the 8.3 restriction */
   rest = dest
   fqDestName = ''
   do while pos('\',rest) > 0
      /* as necessary, create short 8.3 names of each directory in the path */
      parse var rest longname '\' rest
      shortName = shortName(longname)
      fqDestName = fqDestName || shortName

      /* if the directory doesn't exist, create it */
      back = directory(fqDestName)
      if back = '' then do
         /* make the directory & set its datetime stamp and attributes */
         rc = SysMkDir(strip(fqDestName))
         rc = SysSetFileDateTime(fqDestName,curDirectoryDateTime)
         rc = SysFileTree(fqDestName,destFiles,'L',,copyDirectoryAttr)

         /* as necessary, set the .LONGNAME attribute */
         if lfn_support & shortName <> translate(longname) then
            rc = SysPutEA(fqDestName,'.LONGNAME','FDFF'x||D2C(LENGTH(longname))||'00'x||longname)
      end
      /* get us back to original conditions on the destination */
      back = directory(dest_CWD)

      fqDestName = fqDestName || '\'
   end
   dest = fqDestName

   /* get us back to the original Current Working DIrectories */
   back = directory(dest_CWD)
   back = directory(here)
end

/* list all the existing files in the DESTINATION directory */
rc = SysFileTree(dest||mask,destFiles,'FL',tattrib)

/* and get the longname attribute of the existing DESTINATION files */
Do di = 1 to destFiles.0
   /* reformat the variable to quote the file name(s) */
   cur = destFiles.di
   parse var cur dtHere tmHere sizeHere attrHere fnHere
   fnHere = strip(fnHere)
   destFileName = filespec('name',fnHere)
   if lfn_support then do
      rc = SysGetEA(fnHere,'.LONGNAME','longname')
      if rc = 0 & length(longname) > 4
         then destFileName = delstr(longname,1,4)
   end
   destFiles.di = dtHere tmHere sizeHere attrHere '"'||fnHere||'" "'||destFileName||'"'

   /* just the essential case insensitive file name for sorting purposes */
   key.di = translate(destFileName)
End

/* insure the list of existing destination names is sorted into ascending order */
say 'Sorting' destFiles.0 'destination (existing files)...'
Do di = destFiles.0 to 1 by -1 until ordered = 1
   ordered = 1                /* assume list is ordered (Class bubble sort) */
   Do ni = 2 to di
      p = ni - 1
      if key.p > key.ni then do
         /* swap places in the list */
         this = key.p
         key.p  = key.ni          /* swap the keys */
         key.ni = this
         cur = destFiles.p        /* swap the data */
         destFiles.p  = destFiles.ni
         destFiles.ni = cur

         ordered = 0              /* list has changes to ripple */
      end
   End

   /* if no changes detected then we are sorted */
   if bReordered = 0 then leave
End

/* get the longname for each source file */
if lfn_support
   then say 'Checking for the .LONGNAME extended attribute...'
Do si = 1 to srcFiles.0
   src = srcFiles.si
   parse var src d t s a sfn
   sfn = strip(sfn)
   sfn = strip(sfn,'B','"')
   if lfn_support then do
      rc = SysGetEA(sfn,'.LONGNAME','srcLongname')
      if rc \= 0 | length(srcLongname) <= 4 then srcLongname = ''
   end
   else srcLongname = ''
   srcFiles.si = d t s a '"'||sfn||'" "'||filespec('name',delstr(srcLongname,1,4))||'"'

   /* just the essential data for sorting purposes */
   key.si = translate(filespec('name',sfn))
end

/* insure the list of source names is sorted into ascending order */
say 'Sorting source files...'
Do si = srcFiles.0 to 1 by -1 until ordered = 1
   ordered = 1                /* assume list is ordered (Class bubble sort) */
   Do ni = 2 to si
      p = ni - 1
      if key.p > key.ni then do
         /* swap places in the list */
         this = key.p
         key.p  = key.ni          /* swap the keys */
         key.ni = this
         cur = srcFiles.p         /* swap the data */
         srcFiles.p  = srcFiles.ni
         srcFiles.ni = cur

         ordered = 0              /* list has changes to ripple */
      end
   End

   /* if no changes detected then we are sorted */
   if bReordered = 0 then leave
End

/* if we are to erase orphan files */
if erase_orphans <> 0 & recursion_level >= erase_orphans then do

   if fTraceDebug > 0
      then trace("?R")

   say 'checking' destFiles.0 'existing destination files at' dest
   /* for each existing file in the destination directory */
   lastMatch = 0
   Do di = 1 to destFiles.0
      /* see if this existing file still exists in the source files list */
      cur = destFiles.di
      parse var cur . . . a '"' fn '"' . '"' longname '"'
      destFileName = translate(filespec('name',fn))

      bFound = 0
      peekAt = lastMatch + 1
      Do si = peekAt to srcFiles.0
         src = srcFiles.si
         parse var src . . . . '"' sfn '"' . '"' srcLongname '"'
         srcFileName = translate(filespec('name',sfn))

         /* first, see if we can find it by the long name EA */
         if lfn_support then do
            if srcLongname \= '' & translate(srcLongname) = destFileName then bFound = 1
            else if srcLongname \='' & translate(srcLongname) = translate(longname) then bFound = 1
            else if longname \= ''& translate(longname) = srcFileName then bFound = 1
         end

         /* now check for it via the native file system name */
         if srcFileName = destFileName then bFound = 1

         if bFound = 1 then do
            lastMatch = si  /* we might be able to take short next time */
            leave
         end
         else if lfn_support = 0 then do  /* if only care about file system names */
            /* as the list is ordered,
               if source file is past existing destination file names,
                  then we'll never find it in the source list, so it is orphaned. */
            if srcFileName > destFileName
               then leave
         end
         else if si = peekAt & lastMatch \= 0 then do
            /* we peeked ahead at the next item, but our guess was wrong.
               So as lfn processing is needed, we resume searching from item # 1 */
            lastMatch = 0
            si = 0
         end
      End

      /* if it wasn't found, then it is an orphaned file */
      if bFound = 0 then do
         fn = '"'||fn||'"'
         say '@erase orphan file:' fn
         relatedEA_file = ''
         if fHandleEAs > 0
            then relatedEA_file = stream(strip(fn,'B','"')||EA_Alternate_Suffix,'c','query exists')
         if szOutputFile <> '' then do
            rc = lineout(szOutputFile,'@REM file:' fn ' does not exist in' curDestDirectory)
         end

         /* conditional processing of hidden & system files */
         if (substr(tattrib,3,1) = '*' | substr(tattrib,5,1) = '*') ,
            & (pos('H',translate(a)) > 0 | pos('S',translate(a)) > 0)
            then if szOutputFile = ''
               then '@attrib -h -s' fn
               else rc = lineout(szOutputFile,'@attrib -h -s' fn)

         /* always allow removing read only files */
         if pos('R',translate(a)) > 0
            then if szOutputFile = ''
               then '@attrib -r' fn
               else rc = lineout(szOutputFile,'@attrib -r' fn)

         if szOutputFile = ''
            then '@erase' fn
            else rc = lineout(szOutputFile,'@erase' fn)

         /* if we created a related EA file, then we should remove that one too. */
         if relatedEA_file <> '' then do
            if szOutputFile = ''
               then '@erase' relatedEA_file
               else rc = lineout(szOutputFile,'@erase' relatedEA_file)
         end
      end
   End

   if fTraceDebug > 0
      then trace("?R")

   /* are there any directories that no longer exist */
   rc = SysFileTree(dest||mask,destDirs,'DL',tattrib)
   rc = SysFileTree(relativeRoot,srcDirs,'DL',tattrib)
   Do di = 1 to destDirs.0
      aDir = destDirs.di
      parse var aDir . . . . destDir
      destDir = strip(destDir)
      destDirName = filespec('name',destDir)
      longname = ''
      if lfn_support then do
         rc = SysGetEA(destDir,'.LONGNAME','longname')
         if rc = 0 & length(longname) > 4
            then longname = delstr(longname,1,4)
      end

      bFound = 0
      Do si = 1 to srcDirs.0
         src = srcDirs.si
         parse var src . . . . srcDir
         srcDir = strip(srcDir)
         srcDirName = filespec('name',srcDir)
         srcLongname = ''

         if lfn_support then do
            rc = SysGetEA(srcDir,'.LONGNAME','srcLongname')
            if rc = 0 & length(srcLongname) > 4
               then srcLongname = delstr(srcLongname,1,4)
         end

         /* first, see if we can find it by the long name EA */
         if lfn_support then do
            if srcLongname \= '' & translate(srcLongname) = translate(destDirName) then bFound = 1
            else if srcLongname \='' & translate(srcLongname) = translate(longname) then bFound = 1
            else if longname \= '' & translate(longname) = translate(srcDirName) then bFound = 1
         end

         /* now check for it via the native file system name */
         if srcDirName = destDirName then bFound = 1

         if bFound = 1 then leave
      End

      if bFound = 0 then Do
         say 'directory to be deleted' destDir
         if szOutputFile = ''
            then '@call rd! "'||destDir||'" (OK'
            else do
               rc = lineout(szOutputFile,'@REM Directory does not exist on the source')
               rc = lineout(szOutputFile,'@rd! "'||destDir||'" (OK')
            end
      end
   End
End

/* for each one of the candidate files */
say 'checking' srcFiles.0 'source candidate files from' relativeRoot
if fTraceDebug = 1 then do
   say ''
   say 'ready for 1st file: "'||srcFiles.1||'"'
   trace("?R")
end

consecutiveErrorCount = 0
lastMatch = 0
Do si = 1 to srcFiles.0
   cur = srcFiles.si
   copy = 'Y'         /* assume we need to copy it */
   bFound = 0

   /* parse the current file information from the source */
   parse var cur d t s a '"' sfn '"' . '"' srcLongname '"'
   if fTrace >= 1 then say '  checking' d t sfn
   sfn = strip(sfn)                                /* sfn = fully qualified source */
   srcFileName = translate(filespec('name',sfn))
   fqDestName = dest||filespec('name',sfn)         /* fqDestName = fully qualified destination */

   /* check for the existance of the requested file */
   peekAt = lastMatch + 1
   Do di = peekAt to destFiles.0
      parse value destFiles.di with dtHere tmHere sizeHere attrHere '"' fn '"' . '"' longname '"'
      destFileName = translate(filespec('name',fn))

      /* first, see if we can find it by the long name EA */
      if lfn_support then do
         if translate(longname) = srcFileName | srcLongname = longname then do
            copy = 'N'
            bFound = 1
            leave
         end
      end

      /* or find it by the the native file system names */
      if srcFileName = destFileName then do
         copy = 'N'
         bFound = 1
         lastMatch = di  /* we might be able to take short next time */
         leave
      end
      else if lfn_support = 0 then do  /* if only care about file system names */
         /* as the list is ordered,
            if destination list past source list,
               then we'll never find it so we should leave and copy it */
         if srcFileName < destFileName
            then leave
      end
      else if di = peekAt & lastMatch \= 0 then do
         /* we peeked ahead at the next item, but our guess was wrong.
            So as lfn processing is needed, we resume searching from item # 1 */
         lastMatch = 0
         di = 0
      end
   End

   /* if it wasn't found and only updating existing files, don't copy it */
   if bFound = 0 & bUpdateOnly
      then copy = 'N'

   /* if it wasn't missing, see if we need to compare other attributes */
   if bFound = 1 & copy = 'N' then do
      If bCompare > 0 Then Do
         If s <> sizeHere ,        /* size changed so copy 'yes' */
            | (bBiDirectional = 1 & d <> dtHere | tmHere <> t) , /* will pick which way later */
            | (bBiDirectional = 0 & (d > dtHere | (d = dtHere & t > tmHere))) /* source is newer */
         then do
            copy = 'Y'   /* timestamp or size changed, so copy it */
            if fTrace > 2
               then say 'for' sfn 'datestamps:' dtHere 'vs' d '  ' tmHere 'vs' t '  size:' sizeHere 'vs' s
         end
      End
      if copy = 'N' then do            /* otherwise did the attributes change */

         /* if 'A' is to considered & it is set in source OR HRS not the same */
         if (bConsiderA & left(a,1) = 'A') ,
                | substr(attrHere,3) <> substr(a,3) then do
            copy = 'Y'   /* attributes changed, so copy it */
            if fTrace > 2
               then say 'for' sfn 'the attributes' attrHere 'vs' a 'changed'
         end
      end
   End

   if fTraceDebug = 1 then do
      trace("?R")
      say 'Analyzed; copy status =' copy 'for' sfn
   end

   /* do we need to update something? */
   If copy = 'Y' Then Do
      if fTrace < 1 then say '  processing' sfn

      if szOutputFile <> '' 
         then rc = lineout(szOutputFile,'REM for' sfn)

      /* validate the destination directory hierarchy
         and fully qualify the destination file path */
      fqDestName = ''
      rest = dest || substr(sfn,length(relativeRoot)+1)
      do while pos('\',rest) > 0
         parse var rest adir '\' rest
         if dest_is8.3
            then adir = shortName(adir)
         if fqDestName \= '' then fqDestName = fqDestName || '\'
         fqDestName = fqDestName || adir
         there = directory(fqDestName)
         if there = '' then do
            if fTrace > 1 then say '   making subdirectory' fqDestName

            if szOutputFile <> '' 
               then rc = lineout(szOutputFile,'MD' strip(fqDestName))
               else rc = SysMkDir(strip(fqDestName))
         end
         back = directory(dest_CWD)
         back = directory(here)
      end
      if fqDestName \= '' then fqDestName = fqDestName || '\'

      /* if copying from a VFAT drive with a long name */
      setLFN_EA = 0
      if (lfn_support & length(srcLongname) > 0 & dest_is8.3 = 0)
         then destFileName = srcLongname    /* use the source longname attribute */
      else do  /* we don't want long names on the destination */
         if dest_is8.3 then do
            destFileName = shortName(rest)
            if translate(destFilename) \= translate(rest) & lfn_support
               then setLFN_EA = 1
         end
         else destFileName = rest
      end
      fqDestName = fqDestName||destFileName

      /* if bi-directional copy to sync the latest version */
      if bBiDirectional then do
         /* when the 'source' is older than the match */
         if dtHere > d | ((dtHere = d) & (tmHere > t)) then do

            if sizeHere = 0 & s <> 0 then do
               if fTraceDebug = 1 then trace("?R") 
      
               /* Warn that the "newer" destination file is 0 bytes.
                  This would appear to be a mistake so we will overlay the 0 byte file with the source file. */
               emsg = "Error: 0 byte destination file had newer date, but we'll ignore that and update it from" sfn 'of size' s
               say emsg
               rc = lineout(errors,emsg)
            end
            else do
               /* swap source and destinations */
               cur = sfn
               sfn = fqDestName
               
               /* reset the critical variables */
               rc = SysFileTree(sfn,srcEntity,'FL') /* dest is newer so make it the current source */
               parse value srcEntity.1 with d t s a .
               
               rc = SysFileTree(cur,destEntity,'FL') /* the file we will replace */
               parse value destEntity.1 with dtHere tmHere sizeHere attrHere fqDestName
               fqDestName = strip(fqDestName)
            end
         end
      end

      /* Copied files always marked as "A", replicate other source attributes to the destination */
      newAttr = '+*'    /* directory attribute always stays the same */
      if substr(a,3,1) = 'H'
         then newAttr = newAttr||'+'
         else newAttr = newAttr||'-'
      if substr(a,4,1) = 'R'
         then newAttr = newAttr||'+'
         else newAttr = newAttr||'-'
      if substr(a,5,1) = 'S'
         then newAttr = newAttr||'+'
         else newAttr = newAttr||'-'

      /* reset any readonly attribute on the destination file so it can be replaced,
         and might as well reset H & S too */
      if szOutputFile = ''
         then rc = SysFileTree(fqDestName,destEntity,'FL',,'**---') /* the file we will replace */

      if fTraceDebug = 1 then do
         trace("?R")
         say 'Ready for copy action...'
      end

      if szOutputFile <> '' then do
         theCommand = 'xcopy /h /o /t /v "'||sfn||'" "'||filespec('drive',fqDestName)||filespec('path',fqDestName)||'*"' 
         rc = lineout(szOutputFile,theCommand)
         if substr(newAttr,1,1) = '-'
            then theCommand = "-a"
            else theCommand = "+a"
         if substr(newAttr,3,1) = '-'
            then theCommand = theCommand "-h"
            else theCommand = theCommand "+h"
         if substr(newAttr,4,1) = '-'
            then theCommand = theCommand "-r"
            else theCommand = theCommand "+r"
         if substr(newAttr,5,1) = '-'
            then theCommand = theCommand "-s"
            else theCommand = theCommand "+s"
         rc = lineout(szOutputFile,'attrib' theCommand '"'||fqDestName||'"')
         rc = 0
      end
      else do
         /* if it is a 'regular' file, just copy it */
         if pos('H',a) = 0 & pos('S',a) = 0 & s > 0
           then '@copy "'||sfn||'" /b "'||fqDestName||'" /v'
         else do  /* we let xcopy do the heavy lifting,
                       but to avoid a prompt by xcopy, we overlay a dummy file */
           rc = stream(fqDestName,'c','open write')
           rc = stream(fqDestName,'c','close')
           rc = 0        /* assume file open write & close worked */
           if s > 0
              then '@xcopy /h /o /t /v "'||sfn||'" "'||fqDestName||'"'
         end
         
         /* set the attributes of the copied file */
         rcAttrs = SysFileTree(fqDestName,copiedResult,'FL',,newAttr)
         
         /* checking for a bug with NetGear NAS and extended attributes failures */
         if copiedResult.0 = 1 & translate(left(OS,4)) = 'OS/2' & fHandleEAs > 0 then do
         
            /* did we get a 0 length file that shouldn't be? */
            parse var copiedResult.1 . . newSz oldAttr newName
            newName = strip(newName)
            if s > 0 & newSz = 0 then do
               /* we need a couple of temporary files in the original source location */
               noEAname = SysTempFileName(filespec('drive',sfn)||filespec('path',sfn)||'$TMP_NOEA.???')
               theEAname = SysTempFileName(filespec('drive',sfn)||filespec('path',sfn)||'$TMP_EAS.???')
         
               /* assume our work around does not work */
               copyRC = -1    /* assume can't copy it without EAs */
         
               /* copy original file as a temporary version in same location */
               '@copy "'||sfn||'" /b "'||noEAname||'" /v'
         
               /* try splitting the EAs off into a temporary file */
               '@eautil /s' noEAname theEAname
               rcAttrs = SysFileTree(theEAname,copiedResult,'FL')
               parse var copiedResult.1 . . eaSize .
               rcAttrs = SysFileTree(noEAname,copiedResult,'FL')
         
               /* if we created the base file without EAs */
               parse var copiedResult.1 . . newSz .
               if copiedResult.0 = 1 & s = newSz then do
                  /* Warn of the error and copy that file to the destination */
                  emsg = "Error:" sfn "dropping" eaSize "bytes of EAs."
                  say emsg
                  rc = lineout(errors,emsg)
         
                  /* delete the garbage and try copying to the destination again */
                  rc = SysFileDelete(fqDestName)
                  '@copy "'||strip(noEAname)||'" /b "'||fqDestName||'" /v'
                  copyRC = rc
                  alternateDestEA_Name = fqDestName||EA_Alternate_Suffix
                  '@copy "'||strip(theEAname)||'" /b "'||alternateDestEA_Name ||'" /v'
                  copyEARC = rc
                  if copyEARC = 0 then do
                     emsg = "Warning:" sfn "EAs were copied as '"||alternateDestEA_Name||'".'
                     say emsg
                     rc = lineout(errors,emsg)
                  end
                  else rc = SysFileDelete(alternateDestEA_Name)
         
                  /* what result did we get? */
                  rcAttrs = SysFileTree(fqDestName,copiedResult,'FL',,newAttr)
               end
         
               /* clean up our temporary files and prepare to continue onward */
               rc = SysFileDelete(theEAname)
               rc = SysFileDelete(noEAname)
               rc = copyRC
            end
         end

         /* does it look like it copied as best as we could? */
         parse var copiedResult.1 . . newSz .
         if rc \= 0 | copiedResult.0 <> 1 | newSz <> s then do
            emsg = "Error" rc 'copying "'||sfn||'" to "'||fqDestName||'"'
            rc = lineout(errors,emsg)
            consecutiveErrorCount = consecutiveErrorCount + 1
            if consecutiveErrorCount > 3 | newSz = 0 then do
               if newSz = 0 then do
                  emsg = 'A zero byte file was created instead.'
                  say emsg
                  rc = lineout(errors,emsg)
                  emsg = "This could be because EAs aren't fully supported on the destination."
                  say emsg
                  rc = lineout(errors,emsg)
                  emsg = "Try restarting with the /I option to drop the EAs."
                  say emsg
                  rc = lineout(errors,emsg)
               end
               emsg = 'Too many consecutive errors.  Aborting!'
               say emsg
               rc = lineout(errors,emsg)
               leave
            end
         end
         else if lfn_support & setLFN_EA then do
            if srcLongname = '' then srcLongname = filespec('name',sfn)
            rc = SysPutEA(fqDestName,'.LONGNAME','FDFF'x||D2C(LENGTH(srcLongname))||'00'x||srcLongname)
            if rc = 0 then consecutiveErrorCount = 0
         end
         else if dest_is8.3 = 0 & lfn_support then do
            /* remove any unnecessary longname attribute copied from the source */
            rc = SysPutEA(fqDestName,'.LONGNAME','')
            if rc = 0 then consecutiveErrorCount = 0
         end
         else consecutiveErrorCount = 0

         /* if the copy was successful and option to reset (/m) the source "A" attribute */
         if (0 = consecutiveErrorCount) & bMarkCopied then do
            if szOutputFile = ''
               then rc = SysFileTree(sfn,copiedResult,'FL',,"-****")
               else rc = lineout(szOutputFile,'attrib -a "'||sfn||'"')
         end
      end
   End
   else if bFound & left(attrHere,1) = 'A' then do    /* no need to copy so reset the 'A' attribute if it was set */
      /* reset the 'A' destination attrib, only currently copied files during this run have 'A' set */
      if szOutputFile = ''
         then rcAttrs = SysFileTree(fqDestName,copiedResult,'FL',,'-****')
         else rcAttrs = lineout(szOutputFile,'attrib -a "'||fqDestName||'"')
   end
End

if inclSubdirs & recursion_level <= max_recursion_level then do
   /* close the file to prepare for recursion */
   if szOutputFile <> ''
      then rc = stream(szOutputFile,'c','close')

   /* get the names of the immediate subdirectories */
   rc = SysFileTree(relativeRoot,subDirs,'DO',tattrib)
   do d = 1 to subDirs.0
      /* determine the best name for the next subdirectory */
      if lfn_support
         then rc = SysGetEA(subDirs.d,'.LONGNAME','srcLongname')
         else srcLongname = ''
      if rc = 0 & length(srcLongname) > 4
         then newRelativeDest = delstr(srcLongname,1,4)
         else newRelativeDest = filespec('name',subdirs.d)

      if iTraceRecursion >= recursion_level then do
         /* enable tracing at this recursion level */
         w = wordpos('/TD',translate(origArgs))
         if w > 0
            then origArgs = delword(origArgs,w,1)
         origArgs = '/TD' origArgs
      end

      if fTrace > 2 then say '   recursing' subDirs.d '  .LONGNAME="'||srcLongname||'"  newRelativeDest='||newRelativeDest

      if dest \= '' then newRelativeDest = dest || newRelativeDest
      cmdLine = '"' || subdirs.d || '\' || mask || '"'
      cmdLine = cmdLine '"' || newRelativeDest || '"'
      cmdLine = '@call' pgm cmdLine origArgs '(' recursion_level dest_is8.3 errors
      cmdLine

      /* count errors at the sub level but continue onward... */
      if rc > 0 then do
         consecutiveErrorCount = consecutiveErrorCount + rc
      end   
   end
end

if recursion_level <= 1 then do
   elapsedTime = time('E')
   elapsedMin = trunc(elapsedTime / 60)
   elapsedSec = elapsedTime - (60 * elapsedMin)
   elapsedHr = trunc(elapsedMin / 60)
   elapsedMin = elapsedMin - (60 * elapsedHr)
   elapsedTime = elapsedHr||':'||elapsedMin||':'|| elapsedSec
   say 'End execution at:' date() time() ' Processing time:' elapsedTime 'Errors:' consecutiveErrorCount
end

/* restore the current directories on the source and destination drives */
back = directory(dest_CWD)
back = directory(here)

exit consecutiveErrorCount

/* shorten the input name to 8.3 syntax if necessary, or return it as it is */
shortName: procedure expose fqDestName dest_is8.3
   parse arg rest
   if dest_is8.3 = 1 then do
      parse upper var rest eight '.' three
      if length(eight) > 8 | length(three) > 3 then do
         longname = rest
         b6 = translate(strip(substr(eight||"      ",1,6)))
         e3 = translate(strip(substr(three||"   ",1,3)))
         if length(e3) > 0
            then e3 = '.'||e3
         do u = 1 to 9
            shortName = fqDestName||b6||'~'||u||e3
            if stream(shortName,'c','query exists') = '' then leave
            if lfn_support then do
               rc = SysGetEA(shortName,'.LONGNAME','existing_longname')
               if rc = 0 & delstr(existing_longname,1,4) = longname then leave
            end
         end
         if u > 9 then do
            /* we exhausted the most common names, make a unique one */
            b2 = substr(b6||" ",1,2)
            shortName = fqDestName||b2||"????~1"||e3
            shortName = SysTempFileName(shortName)
         end
         rest = filespec('name',shortName)
      end
   end
return rest
