/**
 * Interface used to standardized access and manipulation methods for a
 * cribbage board.  The interface allows several different types of boards
 * to be designed/used as long as they implement this interface. <br>
 * The cribbage board is responsible for maintaining the players score.  The
 * board should also allow for the display of the score, though this is not
 * manditory.  All cribbage boards should extend "javax.swing.JComponent".<br>
 * The cribbage board is also responsible for maintaing the number of
 * players allowed (normally 2 or 4 in some configurations).  More than
 * two is optional though and may not be supported by games. <br>
 * If sound is wanted (for peg moves, etc.), it should only be used if the
 * setSoundOn method is called with a parameter of 'true'.  Otherwise sound
 * should be left off (default state). <br>
 * <b>NOTE:</b> The implementing board object needs to generate events of
 * the 'CribBoardEvent' type to notify the listeners when a particular event
 * happens. <br><br>
 * <b>Changes:</b>
 * <ul>
 * <li> 2001-08-01 - Initial Release.
 * </ul>
 * @see CribBoardEvent
 * @author Michael G. Slack
 * @author slack@attglobal.net
 * @created 2001-08-01
 * @version 2001-08-01 Version 1.00
*/
public interface CribBoard
{
    // public constants
    /** Static representing 'no player'. */
    public static final int NO_PLAYER = -1;

    /** Method used to add listener to the cribbage board.  When a cribbage
     * board event occurs, the listeners 'update()' method is called with the
     * specific CribBoardEvent.
     * @param listener CribBoardListener instance to add to the cribbage board.
    */
    public void addCribBoardListener(CribBoardListener listener);

    /** Method used to remove a cribbage board listener from the board.
     * @param listener CribBoardListener instance to remove from the board.
    */
    public void removeCribBoardListener(CribBoardListener listener);

    /** Method used to initialize anything/resources needed by the cribbage
     * board.  This method should be used to load any graphics, components
     * etc. used by the board.  In addition, when called, this method should
     * setup the board into the initial start state.
     * @exception A CribbageInitException should be thrown to the caller if
     * the board cannot be initialized (can't find graphics, etc.).
    */
    public void initializeBoard() throws CribbageInitException;

    /** Method used to reset the board back to it's initial (starting) state.
     * @throws CribbageNotInitException if the board has not been initialized.
    */
    public void resetBoard() throws CribbageNotInitException;

    /** Method used to move a players peg one space.  The method should also
     * increment the players score one point.
     * @param iPlayer An int representing the player to move the peg of.
     * For consistencies sake, players should start at 0 (player 1), 1 (player2),
     * etc.
     * @return Set to true if the player has won, otherwise, set to false.
     * @throws CribbageRangeException if the player is out of range.
     * @throws CribbageNotInitException if the board has not been initialized.
    */
    public boolean movePeg(int iPlayer)
        throws CribbageRangeException, CribbageNotInitException;

    /** Method used to move a players peg multiple spaces.  The method should
     * also increment the players score by the number of spaces moved.
     * @param iPlayer An int representing the player to move the peg of.
     * For consistencies sake, players should start at 0 (player 1), 1 (player2),
     * etc.
     * @param iMoves A int indicating the number of spaces to move the players peg.
     * @return Set to true if the player has won, otherwise, set to false.
     * @throws CribbageRangeException if the player is out of range.
     * @throws CribbageNotInitException if the board has not been initialized.
    */
    public boolean movePeg(int iPlayer, int iMoves)
        throws CribbageRangeException, CribbageNotInitException;

    /** Method used to return a players score to the caller.
     * @param iPlayer An int representing the player to get the score of.
     * For consistencies sake, players should start at 0 (player 1), 1 (player2),
     * etc.
     * @return The players score.
     * @throws CribbageRangeException if the player is out of range.
    */
    public int getScore(int iPlayer) throws CribbageRangeException;

    /** Method used to return the maximum amount of points a player could
     * get using this cribbage board.
     * @return The maximum score value.
    */
    public int getMaxScore();

    /** Method used to return if a specified player has won the game or not.
     * @param iPlayer An int representing the player to check to see they've won 
     * or not.  For consistencies sake, players should start at 0 (player 1), 
     * 1 (player2), etc.
     * @return Return true if the player has won, otherwise return false.
     * @throws CribbageRangeException if the player is out of range.
    */
    public boolean hasWon(int iPlayer) throws CribbageRangeException;

    /** Method used to return the int representing the player that has won.  If
     * no player has won, then NO_PLAYER should be returned.
     * @return A int representing the player that has won, or NO_PLAYER if no player
     * has won.
    */
    public int getWinner();

    /** Method used to get the maximum number of players supported by the cribbage
     * board.
     * @return Should be an int representing the maximum number of players.
    */
    public int getMaxNumberOfPlayers();

    /** Method used to get the current number of players playing the cribbage
     * board.  Typically, this would be the same number as is returned by
     * getMaxNumberOfPlayers, but could be any number less then that and greater
     * than two (2).
     * @return Sould be the current number of players of the board.  For common
     * two player boards, this number will equal the return of getMaxNumberOfPlayers.
    */
    public int getNumberOfPlayers();

    /** Method used to set the current number of players (up to the max supported
     * by the board).  This method may not do anything for two player boards other
     * than throwing a range error if 1 &lt; iNumPlayers &gt; max players.  The
     * exception being thrown is up to the board implementation.
     * @param iNumPlayers The int value of the number of players (2 to max) that 
     * will be playing the board.
     * @throws CribbageRangeException may be thrown if iNumPlayers is out of range.
     * This method may also be left empty (no code) if so desired and setting the
     * number of players doesn't make sense.
     */
    public void setNumberOfPlayers(int iNumPlayers) throws CribbageRangeException;

    /** Method used to return if the board supports sound effects.
     * If the board does not support sound effects, this method should return false.
     * @return A boolean indicating board has sound effects or not.
    */
    public boolean hasSoundEffects();

    /** Method used to set if sound effects are enabled or not.  This method may not
     * do anything if sound effects are not implemented in the cribbage board.  Sound
     * effects should be left off by default unless this method is called.
     * @param bSoundOn Turn on sound effects if true, otherwise turn them off (default
     * setting).
    */
    public void setSoundOn(boolean bSoundOn);

    /** Method used to see if sound effects (if supported) are on or off.
     * @return A boolean indicating if the sound is on or off.  Should return false
     * if sound is not supported.
    */
    public boolean isSoundOn();
}
