import java.io.*;
import java.util.*;
import com.slackandassociates.cards.*;
import com.slackandassociates.cards.playingcards.PlayingCard;
import com.slackandassociates.cards.playingcards.PlayingCardDeck;

/** 
 * Class defines a cribbage game engine.  The class relies on the cards
 * package Card and CardHand definitions.  The game engine provides a
 * method to determine the score of a cribbage hand (will use the upcard
 * if passed in), pick the best cards to discard into the crib from the
 * initial 6 (used mainly to determine cards to discard by a computer 
 * opponent, but could be used by a hint routine), determine the points 
 * scored during game play (requires an 8 card hand to store cards in
 * that have been played), and a method to determine the best card to play
 * from a hand of 4 (usually used for determining the card to play for the
 * computer opponent, but could be used in a hint routine). <br>
 * The class should be used by instanciating an instance during game startup
 * and keeping alive a reference to the engine as game play continues. <br>
 * The cribbage engine is based off of code originally written by David
 * Addison (1986?) in Basic that was converted to Pascal and then eventually
 * to object Pascal (Sibyl). <br><br>
 * <b>Changes:</b>
 * <ul>
 * <li> 2001/08/05 - Initial release.
 * <li> 2001/08/25 - Added a 'getScoreMessages' method.
 * <li> 2004-01-21 - Moved some private class variables to be local within methods
 *                   so it is more thread safe.  Still have one more variable reference
 *                   not sure of what to do with yet.
 * <li> 2005-01-19 - Modified to use cards v2 library.
 * <li> 2005-03-28 - Removed commented out 'PlayingCard' casts and removed one last
 * cast made incorrectly for cards v2 support.  Engine correctly uses just the Card
 * interface now.
 * </ul>
 * @see com.slackandassociates.cards.Card
 * @see com.slackandassociates.cards.CardHand
 * @see com.slackandassociates.cards.playingcards.PlayingCard
 * @author Michael G. Slack
 * @author slack@attglobal.net
 * @created 2001-08-05
 * @version 2005-03-28 Version 1.03
*/
public class CribbageEngine
    implements Serializable
{
    // private statics
    /** Static used to determine score of runs. */
    private static final int[][] CONQ = {{1,1,1,2,3,9}, {1,1,2,2,3,12}, {1,1,2,3,3,12},
                                         {1,1,2,3,4,8}, {1,2,2,2,3, 9}, {1,2,2,3,3,12},
                                         {1,2,2,3,4,8}, {1,2,3,3,3, 9}, {1,2,3,3,4, 8},
                                         {1,2,3,4,4,8}, {1,2,3,4,5, 5}};

    /** Static used to determine score of runs. */
    private static final int[][] CONR = {{1,1,2,3,6}, {1,2,2,3,6}, {1,2,3,3,6}, {1,2,3,4,4}};

    /** Static used to determine score of a run. */
    private static final int[]   CONS = {1,2,3,3};

    /** Static used to determine discards to use (different hand combinations). */
    private static final int[][] CONV = {{0,1,2,3,4,5,0}, {0,1,2,4,3,5,0}, {0,1,2,5,3,4,0},
                                         {0,1,3,4,2,5,0}, {0,1,3,5,2,4,0}, {0,1,4,5,2,3,0},
                                         {0,2,3,4,1,5,0}, {0,2,3,5,1,4,0}, {0,2,4,5,1,3,0},
                                         {0,3,4,5,1,2,0}, {1,2,3,4,0,5,0}, {1,2,3,5,0,4,0},
                                         {1,2,4,5,0,3,0}, {1,3,4,5,0,2,0}, {2,3,4,5,0,1,0}};

    // private references
    // -- some other mechanism needs to be defined for this so that the engine can be --
    // -- truely thread safe                                                          --
    private ArrayList scoreMsgs = new ArrayList(4); // won't typically have more than two msgs

    // --------------------------- Constructor -----------------------------

    /** Default constructor to create the cribbage game engine. */
    public CribbageEngine()
    {
        // empty constructor
    }

    // -------------------------- Private Methods ---------------------------

    /** Method used to check to see if the hand contains a jack that matches
     * the suit of the upcard.
     * @param hand CardHand to check for a jack in.
     * @return One point if a jack is in the hand with the same suit of the
     * upcard, otherwise zero.
    */
    private int checkForJack(CardHand hand, boolean bUpCard, int iUpSuit)
    {
        int iRet = 0;

        if (bUpCard) {
            for (int i = 0; i < 4; i++) {
                Card pc = hand.cardAt(i);
                if ((pc != null) && (pc.getCardSuit() == iUpSuit) &&
                    (pc.getCardPointValue() == PlayingCard.JPC_JACK)) {
                    iRet = 1;
                    break;
                }
            }
        }

        return iRet;
    }

    /** Method used to check for flushes in the hand.  A flush is valid
     * if all cards in a players hand have the same suit (plus an extra
     * point if match upcard suit) or if all of the cards and the upcard
     * suit of the cribhand match.
     * @param hand CardHand to check for flush in.
     * @param bCribScore A boolean indicating the hand checking is a 
     * cribhand instead of a players hand.
     * @return Zero if no flush in the hand or 4 to 5 points.
    */
    private int checkFlush(CardHand hand, boolean bCribScore, int iUpSuit)
    {
        int iRet = 0;
        int p = 4;

        for (int i = 0; i < 3; i++) {
            Card pc1 = hand.cardAt(i);
            Card pc2 = hand.cardAt(i+1);

            if ((pc1 == null) || (pc2 == null) ||
                (pc1.getCardSuit() != pc2.getCardSuit())) {
                p = 0;
                break;
            }
        }

        // score points now
        Card pc = hand.cardAt(0);

        if ((!bCribScore) && (p == 4)) {
            iRet = 4;
            if ((pc != null) && (pc.getCardSuit() == iUpSuit)) iRet++;
        }
        else if ((p == 4) && (pc != null) && (pc.getCardSuit() == iUpSuit)) {
            iRet = 5;
        }

        return iRet;
    }

    /** Method used to check for scores of 15's in the hand.  Will
     * loop through all possible combinations of 15.
     * @param hand CardHand to check to see if contains sequences of 15.
     * @return Zero through the number of 15's x 2.
    */
    private int check15(CardHand hand, int iUpValue)
    {
        int iRet = 0;

        // check for 15 between two cards
        for (int i = 0; i < 4; i++) {
            for (int j = i+1; j < 5; j++) {
                Card pc1 = hand.cardAt(i);
                Card pc2 = hand.cardAt(j);
                int ii = 0;

                if ((j == 4) && (pc1 != null))
                    ii = pc1.getCardPointValueFace10() + iUpValue;
                else if ((pc1 != null) && (pc2 != null))
                    ii = pc1.getCardPointValueFace10() + pc2.getCardPointValueFace10();

                if (ii == 15) iRet += 2;
            }
        }

        // check for 15 between three cards
        for (int i = 0; i < 3; i++) {
            for (int j = i+1; j < 4; j++) {
                for (int k = j+1; k < 5; k++) {
                    Card pc1 = hand.cardAt(i);
                    Card pc2 = hand.cardAt(j);
                    Card pc3 = hand.cardAt(k);
                    int ii = 0;

                    if ((pc1 != null) && (pc2 != null))
                        ii = pc1.getCardPointValueFace10() + pc2.getCardPointValueFace10();

                    if (k == 4)
                        ii += iUpValue;
                    else if (pc3 != null)
                        ii += pc3.getCardPointValueFace10();

                    if (ii == 15) iRet += 2;
                }
            }
        }

        // check for 15 between four cards
        for (int i = 0; i < 2; i++) {
            for (int j = i+1; j < 3; j++) {
                for (int k = j+1; k < 4; k++) {
                    for (int l = k+1; l < 5; l++) {
                        Card pc1 = hand.cardAt(i);
                        Card pc2 = hand.cardAt(j);
                        Card pc3 = hand.cardAt(k);
                        Card pc4 = hand.cardAt(l);
                        int ii = 0;

                        if ((pc1 != null) && (pc2 != null) && (pc3 != null))
                            ii = pc1.getCardPointValueFace10() +
                                 pc2.getCardPointValueFace10() +
                                 pc3.getCardPointValueFace10();

                        if (l == 4)
                            ii += iUpValue;
                        else if (pc4 != null)
                            ii += pc4.getCardPointValueFace10();

                        if (ii == 15) iRet += 2;
                    }
                }
            }
        }

        // check for 15 with all 4 cards + upcard
        int iii = iUpValue;
        for (int i = 0; i < 4; i++) {
            Card pc = hand.cardAt(i);
            if (pc != null) iii += pc.getCardPointValueFace10();
        }
        if (iii == 15) iRet += 2;

        return iRet;
    }

    /** Method used to check for pairs, three of kinds, etc.
     * @param hand CardHand to check for duplicate cards in.
     * @return Zero, 2 (for each pair), 6 (for 3 of a kind) and
     * 12 (for 4 of a kind).
    */
    private int checkDups(CardHand hand, boolean bUpCard, int iUpV)
    {
        int iRet = 0;
        int jj[] = new int[13];

        for (int i = 0; i < jj.length; i++)
            jj[i] = 0;

        // run check for dups
        for (int i = 0; i < 4; i++) {
            Card pc = hand.cardAt(i);
            if (pc != null) jj[pc.getCardPointValue()-1]++;
        }
        if (bUpCard) jj[iUpV-1]++;

        // score dups now...
        for (int i = 0; i < jj.length; i++) {
            switch (jj[i]) {
                case 2: iRet += 2; // pair
                        break;
                case 3: iRet += 6; // three of a kind
                        break;
                case 4: iRet += 12; // four of a kind
                        break;
                default: break; // nothing...
            }
        }

        return iRet;
    }

    /** Method used to check for a run (straight) in the card hand.
     * @param hand CardHand to check for runs in.
     * @return Zero or the value of the runs (may be more than one).
     * Maximum that will return is 12.
    */
    private int checkRuns(CardHand hand, int iUpV)
    {
        int iRet = 0;
        int[] vv = new int[5];
        int t = 0, l = 0, m = 0;
        int[][] q = new int[11][6];
        int[][] r = new int[4][5];
        int[] s = new int[4];

        for (int i = 0; i < 4; i++) {
            Card pc = hand.cardAt(i);
            if (pc != null) 
                vv[i] = pc.getCardPointValue();
            else
                vv[i] = 0;
        }
        vv[4] = iUpV;

        // sort
        for (int i = 0; i < vv.length; i++) {
            for (int j = i; j < vv.length; j++) {
                if (vv[i] > vv[j]) { // swap
                    t = vv[j]; vv[j] = vv[i]; vv[i] = t;
                }
            }
        }

        // check runs now (first pass - 5 card run/mult runs using all 5)
        for (int i = 0; i < 11; i++)
            for (int j = 0; j < 6; j++)
                q[i][j] = CONQ[i][j];
        l = vv[0] - q[0][0];
        for (int i = 0; i < 11; i++)
            for (int j = 0; j < 5; j++)
                q[i][j] += l;
        for (int i = 0; i < 11; i++) {
            for (int j = 0; j < 5; j++) {
                if (vv[j] != q[i][j]) {
                    break;
                }
                else if (j == 4) {
                    return q[i][5]; // return run points
                }
            }
        }
        // second pass - 4 card run/mult runs in 4 cards
        for (int i = 0; i < 4; i++)
            for (int j = 0; j < 5; j++)
                r[i][j] = CONR[i][j];
        for (int i = 0; i < 2; i++) {
            m = vv[i] - r[0][0];
            for (int j = 0; j < 4; j++)
                for (int k = 0; k < 4; k++)
                    r[j][k] += m;
            for (int j = 0; j < 4; j++) {
                for (int k = 0; k < 4; k++) {
                    if (vv[k+i] != r[j][k]) {
                        break;
                    }
                    else if (k == 3) {
                        return r[j][4];
                    }
                }
            }
        }
        // third pass - 3 card run somewhere in hand
        for (int i = 0; i < 4; i++)
            s[i] = CONS[i];
        for (int i = 0; i < 3; i++) {
            m = vv[i] - s[0];
            for (int j = 0; j < 3; j++)
                s[j] += m;
            for (int j = 0; j < 3; j++) {
                if (vv[i+j] != s[j]) {
                    break;
                }
                else if (j == 2) {
                    return s[3];
                }
            }
        }

        return iRet;
    }

    /** Method used to check to see if a run has been played during cribbage
     * game play.
     * @param playedCards CardHand containing the cards played during the
     * game.  Hand should have enough space for eight cards and not be sorted.
     * @param end The int value to check runs up to.
     * @return The value of any run count (3 to potentially 8).  Will be zero
     * if no runs are in the played cards.
    */
    private int checkForRun(CardHand playedCards, int end)
    {
        int iRet = 0;
        int[] jj = new int[20];
        int x = 0;

        for (int i = 0; i < jj.length; i++)
            if (i < 10)
                jj[i] = 0;
            else
                jj[i] = 14;
        x = playedCards.getCardCount();
        for (int i = 0; i < x; i++) { // store here last card to first card played...
            Card pc = playedCards.cardAt(x-i-1);
            jj[10+i] = pc.getCardPointValue();
        }
        // sort now...
        for (int k = 0; k < (end-1); k++) {
            for (int j = k+1; j < end; j++) {
                if (jj[k+10] > jj[j+10]) { // swap...
                    x = jj[k+10]; jj[k+10] = jj[j+10]; jj[j+10] = x;
                }
            }
        }
        // check run...
        for (int i = 0; i < (end-1); i++)
            if (jj[i+10] != (jj[i+11]-1)) return iRet;
        // have at least 'end' card run
        return end;
    }

    // --------------------------- Public Methods ----------------------------

    /** Method used to return a list of score messages (Strings) generated by
     * the engine during game play ('pointsInGame' method).  For example " gets
     * 2 points for fifteen.".  Messages are not generated at any other time.
     * The list returned may contain more than one message or it may contain
     * no messages.
     * @return An ArrayList with the engine score messages in it.  This list
     * is a copy that can be manipulated however the caller sees fit.
    */
    public ArrayList getScoreMessages()
    {
        return new ArrayList(scoreMsgs); // keep our collection private and uneditable
    }

    /** Method used to determine the amount of points a cribbage hand has
     * (with or without the upcard).  The points available in the hand are
     * returned (which could be zero).
     * @param hand CardHand to get points of.  Hand checked needs to have at
     * least 4 cards and they must be the first four cards in the hand (use
     * 'compressHand()' on a 6 card hand after discarding).
     * @param bCribScore Set to true if scoring the crib hand, otherwise set
     * to false.  Crib hands can only count flushes if all cards and the upcard
     * are the same suit.
     * @param upCard Card that is the cribbage upcard for the hand.  This could
     * be set to null if the points a hand has (without the upcard) is wanted.
     * @return The number of points in the cribbage hand.
    */
    public int figure_Points(CardHand hand, boolean bCribScore, Card upCard)
    {
        int iRet = 0;

        // moved to being local references (instead of defined to the class)
        int iUpSuit     = 25;
        int iUpValue    = 25;
        int iUpV        = 25;
        boolean bUpCard = false;

        // setup upcard stuff (if needed)
        if ((upCard != null) && (upCard instanceof PlayingCard)) {
            bUpCard = true;
            iUpSuit = upCard.getCardSuit();
            iUpValue = upCard.getCardPointValueFace10();
            iUpV = upCard.getCardPointValue();
        }

        // clear out message list
        scoreMsgs.clear();

        // figure points now
        iRet += checkForJack(hand, bUpCard, iUpSuit);
        iRet += checkFlush(hand, bCribScore, iUpSuit);
        iRet += check15(hand, iUpValue);
        iRet += checkDups(hand, bUpCard, iUpV);
        iRet += checkRuns(hand, iUpV);

        return iRet;
    }

    /** Method used to analyze a 6 card cribbage hand to determine what cards
     * would be best discarded into the crib.  Method is mainly used to determine
     * computer moves, but could be used to implement a hint facility also.
     * @param hand CardHand to analyze for discards.  Must contain 6 cards, else
     * -1 is returned for both discard indexes.
     * @return An int array of two elements with the indexes of the cards to discard
     * contained inside.  May contain -1 (in both spots) if an error occurred in the
     * analysis (not enough cards in hand, etc.).
    */
    public int[] analyzeHandForDiscards(CardHand hand)
    {
        int[] iiRet = {-1, -1};

        if (hand.getCardCount() == 6) { // have a cribbage hand to analyze
            int[][] v = new int[15][7];
            int b9 = 0, p9 = 0, j = 0;
            int[] ii = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
            int[] jj = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

            for (int i = 0; i < 15; i++)
                for (int k = 0; k < 7; k++)
                    v[i][k] = CONV[i][k];
            // figure points for each hand combination (15 of them)
            for (int i = 0; i < 15; i++) {
                CardHand tmpHand = new CardHand(6);
                tmpHand.setCompMode(Card.JC_COMP_NOSUIT);
                // create temporary hand of cards from hand
                for (int k = 0; k < 4; k++)
                    tmpHand.add(hand.cardAt(v[i][k]));
                // figure points for created hand
                v[i][6] = figure_Points(tmpHand, false, null);
                if (v[i][6] > p9) p9 = v[i][6]; // p9 has largest amount of points so far
            }
            // more than one hand with largest amount of points?
            for (int i = 0; i < 15; i++)
                if (v[i][6] == p9) {
                    ii[j++] = i; // save off hand combo with high points
                }
            // set index of hand combo with high points (this may be the only one)
            b9 = ii[0];
            if (j != 1) { // had multiple hands with high score
                int c9 = 5, zz = 1; // check for hands with '5's in them
                do {
                    p9 = 0;
                    for (int i = 0; i < jj.length; i++)
                        jj[i] = 0;
                    // does hand combo contain card(s) valued c9?
                    for (int i = 0; i < j; i++) {
                        // check each card in high score combination
                        for (int k = 0; k < 4; k++) {
                            int l = v[ii[i]][k];
                            Card pc = hand.cardAt(l);
                            if ((pc != null) && (pc.getCardPointValue() == c9)) { // mark it
                                jj[i]++;
                                if (jj[i] > p9) p9 = jj[i]; // maybe more than one...
                            }
                        }
                    }
                    // more than one hand combo with our card in it?
                    int y = 0;
                    for (int i = 0; i < j; i++) {
                        if ((jj[i] != 0) && (jj[i] == p9)) { // maybe
                            y++; b9 = ii[i];
                        }
                    }
                    // do we have more than one hand combo?
                    if (y != 1) { // maybe none...
                        if (y != 0) { // rebuild list of hand combos to look at
                            for (int i = (j-2); i >= 0; i--) {
                                if (jj[i] == 0) {
                                    for (int k = i; k < (j-2); k++) {
                                        ii[k] = ii[k+1];
                                    }
                                }
                            }
                            j = y; // reset count to number of hands left...
                        }
                        zz++;
                        switch(zz) {
                            case 2: c9 = 8;  // eights?
                                    break;
                            case 3: c9 = 7;  // sevens?
                                    break;
                            case 4: c9 = 11; // jacks?
                                    break;
                            case 5: c9 = 1;  // aces?
                                    break;
                            default: b9 = ii[(int) Math.floor(Math.random() * j)]; // pick one..
                                     zz = 6;
                                     break;
                        }
                    }
                    else { // only one - have it picked, now exit...
                        zz = 6;
                    }
                } while (zz < 6);
            }
            // set indexes to pass back
            iiRet[0] = v[b9][4];
            iiRet[1] = v[b9][5];
        }

        return iiRet;
    }

    /** Method used to tally up the point value of all of the played cards.
     * @param playedCards CardHand containing the cards played during the
     * game.  Hand should have enough space for eight cards and not be sorted.
     * @return Total point value of all played cards.
    */
    public int getGameTally(CardHand playedCards)
    {
        int iRet = 0;
        int x = playedCards.getCardCount();

        for (int i = 0; i < x; i++) {
            Card c = playedCards.cardAt(i);
            if (c != CardHand.EMPTY_CARD)
                iRet += c.getCardPointValueFace10();
        }

        return iRet;
    }

    /** Method used to return the number of points a hand scores in the game.  This
     * method is generally called after a card is played to get the points (if any)
     * scored.<br>
     * It is up to the caller to manage the played cards hand.  If, for instance, 31
     * points is reached, the caller must clear the played cards and reset everything.
     * The caller should use the 'getGameTally()' method to check where the played
     * point total is.<br>
     * It is also up to the caller to make sure the hand total does not get above 31.
     * @param playedCards CardHand containing the cards played during the
     * game.  Hand should have enough space for eight cards and not be sorted.
     * @return Number of points (maybe zero) scored in game play.
    */
    public int pointsInGame(CardHand playedCards)
    {
        int iRet = 0;
        int tally = getGameTally(playedCards);
        int cardsPlayed = playedCards.getCardCount();
        int n = 2, flg = 0;

        // clear out message list
        scoreMsgs.clear();

        if (cardsPlayed > 1) {
            if (tally == 15) { // score two points (have 15 with play)
                iRet += 2;
                scoreMsgs.add(" gets 2 points for fifteen.");
            }
            if (tally == 31) { // score two points (have 31 with play)
                iRet += 2;     // playedCards need to be reset for game to continue...
                scoreMsgs.add(" gets 2 points for thirtyone.");
            }
            // score dups...
            if ((cardsPlayed - 2) > 2) n = cardsPlayed - 2;
            for (int i = cardsPlayed; i >= n; i--) {
                Card c1 = playedCards.cardAt(i-1);
                Card c2 = playedCards.cardAt(i-2);

                if ((c1 != CardHand.EMPTY_CARD) && (c2 != CardHand.EMPTY_CARD)) {
                    if (c1.getCardPointValue() != c2.getCardPointValue())
                        break;
                }
                else
                    break;
                // score dups now
                switch (cardsPlayed-i+1) {
                    case 1: iRet += 2;
                            flg = 1;
                            break;
                    case 2: iRet += 4; // cumulitive points
                            flg = 2;
                            break;
                    case 3: iRet += 6; //  " " "
                            flg = 3;
                            break;
                }
            }
            if (flg != 0) { // messages for duplicates
                switch(flg) {
                    case 1: scoreMsgs.add(" gets 2 points for a pair.");
                            break;
                    case 2: scoreMsgs.add(" gets 6 points for three-of-a-kind.");
                            break;
                    case 3: scoreMsgs.add(" gets 12 points for four-of-a-kind.");
                            break;
                }
            }
            // check for runs?
            if (cardsPlayed > 2) { // have more than two cards, check 'em
                n = 0;
                int k = 3, nn = 0;

                while (k <= cardsPlayed) {
                    n = checkForRun(playedCards, k);
                    if (n != 0) nn = n;
                    k++;
                }
                // score message (if needed)
                if (nn != 0)
                    scoreMsgs.add(" gets " + nn + " points for a " + nn + " card run.");

                iRet += nn;
            }
        }

        return iRet;
    }

    /** Method used to return the best card to play for the given hand and
     * the given set of cards already played.  Will return a -1 if no card
     * can be played.  This method is generally used to determine a computer
     * oponents move, but could be used to implement a play hint for other
     * players.
     * @param playedCards CardHand containing the cards played during the
     * game.  Hand should have enough space for eight cards and not be sorted.
     * @param hand CardHand to get the best card to play from.  The card hand
     * must contain space for a least 4 cards.
     * @return The index of the card in the hand that is the best card to
     * play.  Will be -1 if no card is available to play in the hand.
    */
    public int getCardToPlay(CardHand playedCards, CardHand hand)
    {
        int iRet = -1;
        int lastCard = playedCards.getCardCount(); // save this for now
        int p = 0, p9 = 0, j = 0, k = 0;
        int[] ii = new int[4];
        int[] jj = new int[4];

        for (int i = 0; i < 4; i++) {
            Card c = hand.cardAt(i);
            if (c != CardHand.EMPTY_CARD) {
                if ((getGameTally(playedCards) + c.getCardPointValueFace10()) <= 31) {
                    playedCards.add(c); // add temporarily..
                    p = pointsInGame(playedCards);
                    playedCards.remove(lastCard); // remove now (played cards not sorted!!!)
                    if (p > p9) { // save off reference...
                        p9 = p; iRet = i;
                    }
                    // check to make sure never throwing five as first card (unless have to)
                    if ((c.getCardPointValue() == 5) && (playedCards.getCardCount() == 0)) {
                        jj[k++] = i;
                    }
                    else { // this may be a card we could play..
                        ii[j++] = i;
                    }
                }
            }
        }
        // select a card to play (if no points can be scored)
        if ((iRet == -1) && (j != 0)) {
            if (j == 1)
                iRet = ii[0];
            else // we have multiple cards we could play, pick one randomly...
                iRet = ii[(int) Math.floor(Math.random() * j)];
        }
        // still don't have a card to play, check secondary play list
        //  - may only be a '5' we can play...
        if ((iRet == -1) && (k != 0)) {
            if (k == 1)
                iRet = jj[0];
            else // have more than one five in hand...
                iRet = jj[(int) Math.floor(Math.random() * k)];
        }

        // clear out message list (analyze methods don't have messages...)
        scoreMsgs.clear();

        return iRet; // may still not have a card to play (=GO=)
    }

    // ---------------------- Inner Class (test runner) ------------------------

    /** Inner class used to test the CribbageEngine class.
     * @see CribbageEngine
    */
    public static final class Test
    {
        CribbageEngine ce = new CribbageEngine();

        CardDeck deck = new CardDeck(PlayingCardDeck.PC_DECK, PlayingCard.class);
        CardHand playedCards = new CardHand(8, false);
        CardHand hand1 = new CardHand(6);
        CardHand hand2 = new CardHand(6);
        CardHand h1 = new CardHand(4);
        CardHand h2 = new CardHand(4);
        CardHand crib = new CardHand(4);
        Card upCard = null;

        /** Method used to setup the engine test.
        */
        private void setup()
        {
            // set comp mode to nosuit (engine needs this mode)
            hand1.setCompMode(Card.JC_COMP_NOSUIT);
            hand2.setCompMode(Card.JC_COMP_NOSUIT);
            h1.setCompMode(Card.JC_COMP_NOSUIT);
            h2.setCompMode(Card.JC_COMP_NOSUIT);
            // ready deck
            deck.shuffle();
            // deal cards
            for (int i = 0; i < 6; i++) {
                hand1.add(deck.getNextCard());
                hand2.add(deck.getNextCard());
            }
        }

        /** Method used to display the hands used in the test.
         * @param disToo Show crib hand if true.
         * @param plydToo Show played cards if true.
         * @param pCards Show hands playing from if true.
        */
        private void showHands(boolean disToo, boolean plydToo, boolean pCards)
        {
            System.out.println("----- Hands -----");
            if (pCards) {
                System.out.println("Hand 1: " + h1);
                System.out.println("Hand 2: " + h2);
            }
            else {
                System.out.println("Hand 1: " + hand1);
                System.out.println("Hand 2: " + hand2);
            }
            if (disToo)
                System.out.println("Crib: " + crib);
            if (plydToo)
                System.out.println("Played: " + playedCards);
        }

        /** Method used to discard cards into the crib from players hands.
         * @param dis1 Discard indexes for hand 1.
         * @param dis2 Discard indexes for hand 2.
        */
        private void doDiscard(int[] dis1, int[] dis2)
        {
            crib.add(hand1.remove(dis1[0]));
            crib.add(hand1.remove(dis1[1]));
            crib.add(hand2.remove(dis2[0]));
            crib.add(hand2.remove(dis2[1]));
            hand1.compressHand();
            hand2.compressHand();
            upCard = deck.getNextCard();
        }

        /** Method used to play a few rounds of cribbage (first two cards
         * of each hand only).
        */
        private void doPlay() // first two cards only...
        {
            int p1 = -1, p2 = -1, h1p = 0, h2p = 0;
            ArrayList a1 = new ArrayList();
            ArrayList a2 = new ArrayList();

            for (int i = 0; i < 4; i++) {
                h1.add(hand1.cardAt(i));
                h2.add(hand2.cardAt(i));
            }

            for (int i = 0; i < 2; i++) {
                p1 = ce.getCardToPlay(playedCards, h1);
                if (p1 != -1) playedCards.add(h1.remove(p1));
                h1p = ce.pointsInGame(playedCards);
                a1 = ce.getScoreMessages();
                p2 = ce.getCardToPlay(playedCards, h2);
                if (p2 != -1) {
                    playedCards.add(h2.remove(p2));
                    h2p = ce.pointsInGame(playedCards);
                    a2 = ce.getScoreMessages();
                }
                else {
                    h2p = 0;
                    a2.clear();
                }
                System.out.println("--- Playing ---");
                System.out.println("Hand 1 plays - "+p1+", scores: "+h1p);
                if (!a1.isEmpty()) { // show score messages
                    for (Iterator itr = a1.iterator(); itr.hasNext(); )
                        System.out.println(" Hand 1" + (String) itr.next());
                }
                System.out.println("Hand 2 plays - "+p2+", scores: "+h2p);
                if (!a2.isEmpty()) { // show score messages
                    for (Iterator itr = a2.iterator(); itr.hasNext(); )
                        System.out.println(" Hand 2" + (String) itr.next());
                }
                showHands(false, true, true);
            }
        }

        /** Method used to display the points for each hand and for the crib
         * hand.
        */
        private void doPoints()
        {
            int p1 = ce.figure_Points(hand1, false, upCard);
            int p2 = ce.figure_Points(hand2, false, upCard);
            int p3 = ce.figure_Points(crib, true, upCard);
            System.out.println("----- Scores (with upcard) -----");
            System.out.println("Hand 1: "+p1+", Hand 2: "+p2+", Crib: "+p3);
        }

        /** Method used to process the cribbage engine test.
        */
        public void go()
        {
            // set up
            setup();
            // show hands
            showHands(false, false, false);
            // determine discards
            int[] dis1 = ce.analyzeHandForDiscards(hand1);
            int[] dis2 = ce.analyzeHandForDiscards(hand2);
            System.out.println("----- Have crib discards now -----");
            System.out.println("Hand 1 discards (index's): " + dis1[0] + ", " + dis1[1]);
            System.out.println("Hand 2 discards (index's): " + dis2[0] + ", " + dis2[1]);
            // discard - get up card...
            doDiscard(dis1, dis2);
            showHands(true, false, false);
            System.out.println("----- Upcard dealt -----");
            System.out.println("Upcard - " + upCard);
            doPlay(); // only first two cards...
            doPoints();
        }

        /** Method used to start the engine test with (entry method).
         * @param args Ignored.
        */
        public static void main(String[] args)
        {
            Test ct = new Test();
            ct.go();
        }
    }
}
