import java.io.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.html.*;
import java.net.*;
import com.slackandassociates.Utilities;

/**
 * Class defines a simple help facility for the Java Cribbage game.  Some of
 * the code was borrowed (heavily) from the SwingSet2 demo application (if it
 * looks familar it's because it is!!!). <br><br>
 * <b>Changes:</b>
 * <ul>
 * <li> 2001/10/06 - Initial release.
 * <li> 2001/10/12 - Cleaned up code.
 * <li> 2002/01/05 - Modified to use utility class from slack.jar.
 * <li> 2005-01-19 - Modified to use 'createTestFrame' method in 'main'.
 * </ul>
 * @author Michael G. Slack
 * @author slack@attglobal.net
 * @created 2001-10-06
 * @version 2005-01-19 Version 1.03
*/
public class CribbageHelp extends JDialog
    implements HyperlinkListener
{
    // window size constants
    private static final int D_HLP_WIDTH = 500;
    private static final int D_HLP_HEIGHT = 320;

    // string constants
    private static final String S_LOADMSG = 
        "<html><body><br><br>Loading page, please wait...</body></html>";
    private static final String S_ERR1 = "<html><body>Page load failed for page (<b>";
    private static final String S_ERR2 = "</b>).</body></html>";

    // components used by dialog
    private JEditorPane help = new JEditorPane();
    private JScrollPane scroller = new JScrollPane(help);
    private JPanel dPnl1 = new JPanel();
    private JButton dB1 = new JButton("Exit");

    // private references
    private String helpPath = "";
    private String gPgNm = "fake";
    private URL gURLPg = null;

    // -------------------------- Constructor -----------------------------

    public CribbageHelp(JFrame parent, String helpPath)
    {
        super(parent, "Cribbage Help", true);

        this.helpPath = helpPath;

        setupHelpDlg(parent);
    }

    // ------------------------- Private Methods -----------------------------

    /** Method used to setup the about dialog box. */
    private void setupHelpDlg(JFrame parent)
    {
        Container cp = getContentPane();

        setSize(D_HLP_WIDTH, D_HLP_HEIGHT);
        Utilities.centerDialog(parent, this);
        cp.setLayout(new BorderLayout());
        cp.add(scroller, BorderLayout.CENTER);
        dB1.addActionListener(new ActionListener() {
                                      public void actionPerformed(ActionEvent evt) {
                                          dispose(); // close the dialog
                                      }
                                  });
        dPnl1.add(dB1);
        cp.add(dPnl1, BorderLayout.SOUTH);
        initEditorPane();
    }

    /** Method used to initialize (to start page) the help pane. */
    private void initEditorPane()
    {
        help.setEditorKit(new HTMLEditorKit());
        help.setEditable(false);
        help.addHyperlinkListener(this);
        // load initial page now
        loadPage("index.html", null);
    }

    // ------------------------ Protected Methods ----------------------------

    /** Method to load page sent in to selected html page (uses a thread to
     * load it).
     * @param pgNm String containing the page name to load into the JEditorPane.
    */
    protected void loadPage(String pgNm, URL pg)
    {
        help.setText(S_LOADMSG);
        gPgNm = pgNm; // set to global so it can be used in anonymous inner class
        gURLPg = pg;  // ditto

        Runnable r = new Runnable() {
                             public void run() {
                                 String sErr = S_ERR1 + gPgNm + S_ERR2;
                                 URL in = null;

                                 try {
                                     String ss = helpPath + gPgNm;
                                     if (gURLPg == null)
                                         in = this.getClass().getResource(ss);
                                     else
                                         in = gURLPg;
                                     if (in != null)
                                         help.setPage(in);
                                     else {
                                         System.err.println("Help file returned null - "+ss);
                                         help.setText(sErr);
                                     }
                                 }
                                 catch (Exception e) {
                                     System.err.println("Failed help file read " +
                                                        "(" + gPgNm + "): " + e);
                                     help.setText(sErr);
                                 }
                             }
                         };
        SwingUtilities.invokeLater(r);
    }

    // ------------------------- Public Methods ------------------------------

    /** Method used to respond to the hyperlink events from the html help files. */
    public void hyperlinkUpdate(HyperlinkEvent e)
    {
        if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
            if (e instanceof HTMLFrameHyperlinkEvent) {
                HTMLDocument doc = (HTMLDocument) help.getDocument();
                doc.processHTMLFrameHyperlinkEvent((HTMLFrameHyperlinkEvent) e);
            }
            else {
                loadPage("", e.getURL());
            }
        }
    }

    /** Method used for testing dialog. */
    public static void main(String[] args)
    {
        JFrame frame = Utilities.createTestFrame("Help Test", D_HLP_WIDTH, D_HLP_HEIGHT);

        new CribbageHelp(frame, "help/").show();
    }
}
