import java.util.*;
import java.io.*;
import java.awt.*;
import javax.swing.SwingUtilities;
import javax.swing.event.EventListenerList;
import javax.sound.sampled.*;
import com.slackandassociates.Utilities;

/**
 * Class used to implement a standard 121 point cribbage board (around
 * twice).  This board implements a sound 'tick' when moving the pegs
 * for each player from hole to hole (if sounds are wanted). <br>
 * Note: Board does not display score.  It is up to the caller to display
 * score.  Board uses blue and red pegs to represent players. <br><br>
 * <b>Changes:</b>
 * <ul>
 * <li> 2001-08-03 - Initial Release.
 * <li> 2001-09-28 - Modified to use the CribbageProps class for paths.
 * <li> 2001-10-07 - Added a check in the move peg thread to see if iMoves > 0
 *                   along with score less than MAX_SCORE.
 * <li> 2001-10-30 - Added a call to 'invokeLater()' in the 'fireCribEvent'
 *                   method.
 * <li> 2002-01-05 - Modified to use the Utilities class from slack.jar.
 * <li> 2002-06-15 - Modified wait on PegSound thread (use join() instead of flag).
 * </ul>
 * @author Michael G. Slack
 * @author slack@attglobal.net
 * @created 2001-08-03
 * @version 2002-06-15 Version 1.05
*/
public class StdCribBoard extends javax.swing.JComponent
    implements CribBoard
{
    // private constants
    /** Static representing board width (pixels). */
    private static final int BRD_WIDTH = 641;

    /** Static representing board height (pixels). */
    private static final int BRD_HEIGHT = 156;

    /** Static representing the maximum number of players allowed. */
    private static final int MAX_PLAYERS = 2;

    /** Static representing the maximum score that can be won with this board. */
    private static final int MAX_SCORE = 121;

    /** Static message string for not initialized messages. */
    private static final String NOT_INIT_MSG = "Cribbage board not initialized.";

    /** Static representing the sound file (wav) used by this board. */
    private static final String SND_FILE_NAME = "tac.wav";

    /** Static representing a empty peg hole. */
    private static final int CB_EMPTY = 1;

    /** Static representing a blue peg in the hole. */
    private static final int CB_BLUE = 2;

    /** Static representing a red peg in the hole. */
    private static final int CB_RED = 3;

    /** Static representing the board at startup (or reset).
     * <pre>
     *  - Blank position = 0
     *  - Empty peg (hole) = 1
     *  - Blue peg (in hole) = 2
     *  - Red peg (in hole) = 3
     *  - Arrow picture (-&gt;) in position = 4
     * </pre>
    */
    private static final int[][] DEF_BOARD =
        {{0,2,0,4,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0},
         {0,2,0,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0},
         {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},
         {0,3,0,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0},
         {0,3,0,4,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1,0}};

    /** Static representing the positions to move 'pegs' to in the board structure.
     * Used by the peg move method to correctly 'move' the pegs around the board.
     * Size is [2][121][2].
    */
    private static final int[][][] BOARD_POS = {{{4,0},  {5,0},  {6,0},  {7,0},  {8,0},
                                                 {10,0}, {11,0}, {12,0}, {13,0}, {14,0},
                                                 {16,0}, {17,0}, {18,0}, {19,0}, {20,0},
                                                 {22,0}, {23,0}, {24,0}, {25,0}, {26,0},
                                                 {28,0}, {29,0}, {30,0}, {31,0}, {32,0},
                                                 {34,0}, {35,0}, {36,0}, {37,0}, {38,0},
                                                 {38,1}, {37,1}, {36,1}, {35,1}, {34,1},
                                                 {32,1}, {31,1}, {30,1}, {29,1}, {28,1},
                                                 {26,1}, {25,1}, {24,1}, {23,1}, {22,1},
                                                 {20,1}, {19,1}, {18,1}, {17,1}, {16,1},
                                                 {14,1}, {13,1}, {12,1}, {11,1}, {10,1},
                                                 {8,1},  {7,1},  {6,1},  {5,1},  {4,1},
                                                 {4,0},  {5,0},  {6,0},  {7,0},  {8,0},
                                                 {10,0}, {11,0}, {12,0}, {13,0}, {14,0},
                                                 {16,0}, {17,0}, {18,0}, {19,0}, {20,0},
                                                 {22,0}, {23,0}, {24,0}, {25,0}, {26,0},
                                                 {28,0}, {29,0}, {30,0}, {31,0}, {32,0},
                                                 {34,0}, {35,0}, {36,0}, {37,0}, {38,0},
                                                 {38,1}, {37,1}, {36,1}, {35,1}, {34,1},
                                                 {32,1}, {31,1}, {30,1}, {29,1}, {28,1},
                                                 {26,1}, {25,1}, {24,1}, {23,1}, {22,1},
                                                 {20,1}, {19,1}, {18,1}, {17,1}, {16,1},
                                                 {14,1}, {13,1}, {12,1}, {11,1}, {10,1},
                                                 {8,1},  {7,1},  {6,1},  {5,1},  {4,1},
                                                 {1,1}},
                                                {{4,4},  {5,4},  {6,4},  {7,4},  {8,4},
                                                 {10,4}, {11,4}, {12,4}, {13,4}, {14,4},
                                                 {16,4}, {17,4}, {18,4}, {19,4}, {20,4},
                                                 {22,4}, {23,4}, {24,4}, {25,4}, {26,4},
                                                 {28,4}, {29,4}, {30,4}, {31,4}, {32,4},
                                                 {34,4}, {35,4}, {36,4}, {37,4}, {38,4},
                                                 {38,3}, {37,3}, {36,3}, {35,3}, {34,3},
                                                 {32,3}, {31,3}, {30,3}, {29,3}, {28,3},
                                                 {26,3}, {25,3}, {24,3}, {23,3}, {22,3},
                                                 {20,3}, {19,3}, {18,3}, {17,3}, {16,3},
                                                 {14,3}, {13,3}, {12,3}, {11,3}, {10,3},
                                                 {8,3},  {7,3},  {6,3},  {5,3},  {4,3},
                                                 {4,4},  {5,4},  {6,4},  {7,4},  {8,4},
                                                 {10,4}, {11,4}, {12,4}, {13,4}, {14,4},
                                                 {16,4}, {17,4}, {18,4}, {19,4}, {20,4},
                                                 {22,4}, {23,4}, {24,4}, {25,4}, {26,4},
                                                 {28,4}, {29,4}, {30,4}, {31,4}, {32,4},
                                                 {34,4}, {35,4}, {36,4}, {37,4}, {38,4},
                                                 {38,3}, {37,3}, {36,3}, {35,3}, {34,3},
                                                 {32,3}, {31,3}, {30,3}, {29,3}, {28,3},
                                                 {26,3}, {25,3}, {24,3}, {23,3}, {22,3},
                                                 {20,3}, {19,3}, {18,3}, {17,3}, {16,3},
                                                 {14,3}, {13,3}, {12,3}, {11,3}, {10,3},
                                                 {8,3},  {7,3},  {6,3},  {5,3},  {4,3},
                                                 {1,3}}};

    // private stuff
    /** Flag indicating the board is initialized. */
    private boolean bInitialized = false;

    /** Board size reference. */
    private Dimension dim = new Dimension(BRD_WIDTH, BRD_HEIGHT);

    /** List of listeners for the cribbage board events. */
    private EventListenerList listenerList = new EventListenerList();

    /** Board image (arrow image). */
    private Image imgArrow = null;

    /** Board image (blank space image). */
    private Image imgBlank = null;

    /** Board image (blue peg in space image). */
    private Image imgBluePeg = null;

    /** Board image (no peg in space image). */
    private Image imgEmptyPeg = null;

    /** Board image (red peg in space image). */
    private Image imgRedPeg = null;

    /** Flag value indicating that the sound clip was loaded and is available. */
    private boolean bSoundLoaded = false;

    /** Flag value indicating if sound is on or off. */
    private boolean bSoundOn = false;

    /** Board tick sound (always loaded even if sound not wanted). */
    private byte[] snd;

    /** Sound length. */
    private int snd_len = 0;

    /** Board audio format (initialized during sound load). */
    private AudioFormat format = null;

    /** Board audio line info (initialized if sound is turned on). */
    private DataLine.Info info = null;

    /** Board audio line (initialized if sound is turned on/released on sound off). */
    private SourceDataLine sdl = null;

    /** Array of players current scores. */
    private int[] playerScore = {0, 0};

    /** Current board layout. */
    private int[][] board = new int[5][40];

    /** Saved peg position player 1. */
    private int p1PegP = 120;

    /** Saved peg position player 2. */
    private int p2PegP = 120;

    /** Property instance with some cribbage props in it. */
    private CribbageProps props = new CribbageProps(this);

    // ------------------------ Inner Classes --------------------------------

    /** Class (thread) used to play the peg move sound.
    */
    class PegSound extends Thread
    {
        // ---------------------------- Constructors --------------------------------

        /** Empty Constructor. */
        public PegSound() { }

        // ------------------------ Public Methods -----------------------------------

        /** Thread run method - does the peg 'tick' sound. */
        public void run()
        {
            // play it
            try {
                sdl.start();
                sdl.write(snd, 0, snd.length);
                sdl.drain();
                sdl.stop();
            }
            catch (Exception e) {
                System.err.println("Sound Playback Error: " + e);
                bSoundLoaded = false;
            }

            return;
        }
    }

    /** Class used to move the peg in the board for the board object.
    */
    class MovePeg extends Thread
    {
        /** Passed in instance of board component. */
        private javax.swing.JComponent owner = null;

        /** Player to move peg of. */
        private int iPlayer = 0;

        /** Number of moves to make for the peg. */
        private int iMoves = 0;

        // ---------------------------- Constructors --------------------------------

        /** Empty Constructor (should not use). */
        public MovePeg() { }

        /** Constructor for Pause. */
        public MovePeg(javax.swing.JComponent owner, int iPlayer, int iMoves)
        {
            this.owner = owner;
            this.iPlayer = iPlayer;
            this.iMoves = iMoves;
        }

        // ---------------------- Private Methods -----------------------------------

        /** Method used to move a players peg one space.  This method does the
         * actual drawing to the board of the move.
         * @param bc A int representing the column of the board where the peg moved from.
         * @param br A int representing the row of the board where the peg moved from.
         * @param nc A int representing the column of the board where the peg moves to.
         * @param nr A int representing the row of the board where the peg moves to.
         * @param c A int representing the peg color to be moved.
        */
        private void moveOneSpace(int bc, int br, int nc, int nr, int c)
        {
            // do peg move
            board[br][bc] = CB_EMPTY;
            board[nr][nc] = c;
            if (owner != null) owner.repaint();
        }

        /** Method used to play the peg move sound if sounds are on.
        */
        private void playPegSound()
        {
            PegSound ps = null;

            if ((bSoundLoaded) && (bSoundOn)) {
                ps = new PegSound();
                ps.start();
            }
            try {
                if (((bSoundLoaded) && (!bSoundOn)) || (!bSoundLoaded))
                    sleep(200);
                else
                    ps.join();
            }
            catch (InterruptedException ie) { }
        }

        // ------------------------ Public Methods -----------------------------------

        /** Thread run method - does the peg move/plays 'tick' sound. */
        public void run()
        {
            if ((playerScore[iPlayer] < MAX_SCORE) && (iMoves > 0)) { // only move if can move
                int fp = playerScore[iPlayer];
                int bp = (iPlayer == 0) ? p1PegP : p2PegP;
                int c  = (iPlayer == 0) ? CB_BLUE : CB_RED;
                int bc = 0, br = 0, nc = 0, nr = 0, count = 0;

                // setup for move
                if (fp != 0) { // get back position to clear
                    bc = BOARD_POS[iPlayer][bp][0];
                    br = BOARD_POS[iPlayer][bp][1];
                    if (iPlayer == 0)
                        p1PegP = fp - 1;
                    else
                        p2PegP = fp - 1;
                }
                else { // just clear front peg position
                    bc = 1;
                    br = (iPlayer == 0) ? 0 : 4;
                }

                // do all of move in one swoop
                for (int i = 0; i < iMoves; i++) {
                    // play sound
                    playPegSound();
                    // move the peg now
                    fp = playerScore[iPlayer];
                    playerScore[iPlayer]++;
                    nc = BOARD_POS[iPlayer][fp][0];
                    nr = BOARD_POS[iPlayer][fp][1];
                    moveOneSpace(bc, br, nc, nr, c); 
                    br = nr; bc = nc;
                    if (playerScore[iPlayer] >= MAX_SCORE) break;
                }
            }

            // fire the 'done' event
            if (playerScore[iPlayer] >= MAX_SCORE)
                fireCribEvent(CribBoardEvent.CBE_PLAYER_WON, "", iPlayer);
            else
                fireCribEvent(CribBoardEvent.CBE_MOVE_COMPLETED, "", NO_PLAYER);

            return;
        }
    }

    // ------------------------- Constructors --------------------------------

    /** Constructor that creates the cribbage board instance.  The game board
     * is automatically initialized during construction.
    */
    public StdCribBoard()
    {
        this(true);
    }

    /** Constructor that creates the cribbage board instance.  The game board
     * can be initialized if wanted or not.  If not initialized, the
     * 'initializeBoard()' method must be called before using.
     * @param bInitNow A boolean value indicating if the game board is initialized
     * during construction or not.
    */
    public StdCribBoard(boolean bInitNow)
    {
        super();

        // set size
        setPreferredSize(dim);

        if (bInitNow) {
            try {
                initializeBoard();
            }
            catch (Exception e) {
                System.err.println("StdCribBoard constructor (init): " + e.toString());
            }
        }
    }

    // ------------------------ Private Methods ------------------------------

    /** Method used to load a image and pass back the image to the caller.  Uses
     * the image path constant to locate the image on.
     * @param sImg String containing the name of the image to load.
     * @return The loaded image or null if not found.
    */
    private Image loadImage(String sImg)
    {
        return Utilities.loadImage(this, getToolkit(), props.getImagePath(), sImg);
    }

    /** Method used to load the board sound image from board resources.  Will
     * set bSoundLoaded to true if no errors detected in the load.
    */
    private void loadSound()
    {
        AudioInputStream ain = null;

        // assume all will be well
        bSoundLoaded = true;

        try {
            String sPth = props.getSoundPath();
            InputStream in = this.getClass().getResourceAsStream(sPth + SND_FILE_NAME);
            BufferedInputStream bs = new BufferedInputStream(in);
            ain = AudioSystem.getAudioInputStream(bs);
            int r = 0, bufSize = 0;

            // initialize some of the globals now
            format = ain.getFormat();
            bufSize = (int) ain.getFrameLength() * format.getFrameSize();
            snd = new byte[bufSize];

            // read sound bytes now
            snd_len = 0;
            do {
                r = ain.read();
                if (r != -1) {
                    snd[snd_len++] = (byte) r;
                }
            } while (r != -1);

            if (snd_len < bufSize) { // don't think it was all read
                throw new CribbageException("Only read " + snd_len + " bytes of " + bufSize);
            }
        }
        catch (Exception e) {
            System.err.println("Sound Load Error: " + e);
            bSoundLoaded = false;
        }
        finally {
            try { if (ain != null) ain.close(); } catch (Exception e) { }
        }
    }

    // ---------------------- Protected Methods ------------------------------

    // used by anonymous inner class in fireCribEvent...
    private CribBoardEvent cbe = null;

    /** Method used to fire off a cribbage board event to the registered listeners.
     * @param event Event to send to the listeners.
     * @param custom Custom event to send to the listeners.
     * @param winner Winning player if any.
    */
    protected void fireCribEvent(int event, String custom, int winner)
    {
        // create the event to pass on to event listeners
        if (winner != NO_PLAYER)
            cbe = new CribBoardEvent(this, true, winner);
        else if ((custom != null) && (!"".equals(custom)))
            cbe = new CribBoardEvent(this, custom);
        else
            cbe = new CribBoardEvent(this, event);

        // set the event processing into the event thread
        SwingUtilities.invokeLater(new Runnable() {
                                     public void run() {
                                       // store reference internally (no over-write)
                                       CribBoardEvent cbe2 = cbe;
                                       // Guaranteed to return a non-null array
                                       Object[] listeners = listenerList.getListenerList();

                                       // Process the listeners last to first, notifying
                                       // those that are interested in this event
                                       for (int i = listeners.length - 2; i >= 0; i -= 2) {
                                         if (listeners[i] == CribBoardListener.class) {
                                           // call it
                                           ((CribBoardListener) listeners[i+1]).update(cbe2);
                                         }
                                       }
                                     }
                                   });
    }

    // ---------------------- Accessibility Method ---------------------------

    /** Method to return the accessibilty context to the caller.  This class
     * returns null.
     * @return A AccessibleContext object for this class.  There is currently no
     * support for this so only a null is returned for this method.
    */
    public javax.accessibility.AccessibleContext getAccessibleContext()
    {
        return null;
    }

    // ---------------------- Public Methods (CribBoard) ---------------------

    /** Overrode method used to display the board on the screen.
     * @param g Graphic instance used to draw with.
    */
    public void paint(Graphics g)
    {
        int x = 0, y = 1;

        // set up background border color
        g.setColor(Color.black);
        g.fillRect(0, 0, dim.width, dim.height);

        // draw up board if initialized...
        if (bInitialized) {
            // set board pictures into this
            for (int i = 0; i < 5; i++) {
                x = 1;
                for (int j = 0; j < 40; j++) {
                    switch(board[i][j]) {
                        case 0: g.drawImage(imgBlank, x, y, this);
                                break;
                        case 1: g.drawImage(imgEmptyPeg, x, y, this);
                                break;
                        case 2: g.drawImage(imgBluePeg, x, y, this);
                                break;
                        case 3: g.drawImage(imgRedPeg, x, y, this);
                                break;
                        case 4: g.drawImage(imgArrow, x, y, this);
                                break;
                        default: break;
                    }
                    x += imgBlank.getWidth(null) + 1;
                }
                y += imgBlank.getHeight(null) + 1;
            }
        }
    }

    /** Method used to add listener to this cribbage board.  When a specified
     * event occurs, the registered listeners 'update()' event is called with
     * the details.
     * @param listener CribBoardListener instance to add to the cribbage board.
    */
    public void addCribBoardListener(CribBoardListener listener)
    {
        listenerList.add(CribBoardListener.class, listener);
    }

    /** Method used to remove a cribbage board listener from the board.
     * @param listener CribBoardListener instance to remove from the board.
    */
    public void removeCribBoardListener(CribBoardListener listener)
    {
        listenerList.remove(CribBoardListener.class, listener);
    }

    /** Method used to initialize graphics and other resources used by the cribbage
     * board.  The board is setup in it's initial starting point by this method.
     * @exception A CribbageInitException if the resources can't be loaded.
    */
    public void initializeBoard() throws CribbageInitException
    {
        if (!bInitialized) {
            bInitialized = true; // assume we are good...

            // load images for cribbage board
            imgArrow = loadImage("arrow.gif");
            imgBlank = loadImage("blank.gif");
            imgBluePeg = loadImage("bluepeg.gif");
            imgEmptyPeg = loadImage("emptypeg.gif");
            imgRedPeg = loadImage("redpeg.gif");

            // did they all load?
            bInitialized = ((imgArrow != null) && (imgBlank != null) &&
                            (imgBluePeg != null) && (imgEmptyPeg != null) &&
                            (imgRedPeg != null));

            // load sound sample
            loadSound();

            // reset and repaint board
            if (bInitialized) {
                try {
                    resetBoard();
                }
                catch (Exception e) {
                    bInitialized = false;
                }
            }
        }
    }

    /** Method used to reset the board back to it's initial (starting) state.
     * @throws CribbageNotInitException if the board has not been initialized.
    */
    public void resetBoard() throws CribbageNotInitException
    {
        if (!bInitialized)
            throw new CribbageNotInitException("resetBoard(): " + NOT_INIT_MSG);

        // reset players scores
        for (int i = 0; i < playerScore.length; i++)
            playerScore[i] = 0;

        // reset board to statup board positions now
        for (int i = 0; i < 5; i++)
            for (int j = 0; j < 40; j++)
                board[i][j] = DEF_BOARD[i][j];

        // set peg position indexes
        p1PegP = MAX_SCORE - 1;
        p2PegP = MAX_SCORE - 1;

        // draw it now
        repaint();

        // notify listeners
        fireCribEvent(CribBoardEvent.CBE_CRIB_BOARD_RESET, "", NO_PLAYER);
    }

    /** Method used to move a players peg one space.  In addition, the players
     * score is incremented by one point also.  Note: if sound is enabled, a
     * peg move sound is played for the move.
     * @param iPlayer An int representing the player to move the peg of.
     * @return A true value indicates the player has won, otherwise false is returned.
     * @throws A CribbageRangeException if the player is out of range.
     * @throws CribbageNotInitException if the board has not been initialized.
    */
    public boolean movePeg(int iPlayer)
        throws CribbageRangeException, CribbageNotInitException
    {
        return movePeg(iPlayer, 1);
    }

    /** Method used to move a players peg multiple spaces.  This method also increases
     * the players score by the amount sent in.<br>
     * Note: Even though this method returns instantly (with the correct player won
     * information), the score is not up-to-date until the move is complete (per the
     * move complete event).
     * @param iPlayer An int representing the player to move the peg of.
     * @param iMoves A int indicating the number of spaces to move the players peg.
     * @return A true value indicates the player has won, otherwise false is returned.
     * @throws A CribbageRangeException if the player is out of range.
     * @throws CribbageNotInitException if the board has not been initialized.
    */
    public boolean movePeg(int iPlayer, int iMoves)
        throws CribbageRangeException, CribbageNotInitException
    {
        if (!bInitialized)
            throw new CribbageNotInitException("movePeg(int): " + NOT_INIT_MSG);

        if ((iPlayer < 0) || (iPlayer >= MAX_PLAYERS))
            throw new CribbageRangeException("Player is out of range - ("+iPlayer+")");

        fireCribEvent(CribBoardEvent.CBE_MOVE_STARTED, "", NO_PLAYER);

        boolean bRet = ((playerScore[iPlayer] + iMoves) >= MAX_SCORE);

        new MovePeg(this, iPlayer, iMoves).start();

        return bRet;
    }

    /** Method used to return a players score to the caller.
     * @param iPlayer An int representing the player to get the score of.
     * @return The players score.
     * @throws A CribbageRangeException if the player is out of range.
    */
    public int getScore(int iPlayer) throws CribbageRangeException
    {
        if ((iPlayer < 0) || (iPlayer >= MAX_PLAYERS))
            throw new CribbageRangeException("Player is out of range - ("+iPlayer+")");

        return playerScore[iPlayer];
    }

    /** Method used to return the maximum amount of points a player could
     * get using this cribbage board.
     * @return The maximum score value.
    */
    public int getMaxScore()
    {
        return MAX_SCORE;
    }

    /** Method used to return if a specified player has won the game or not.
     * @param iPlayer An int representing the player to check to see they've won 
     * or not.
     * @return Return true if the player has won, otherwise return false.
     * @throws A CribbageRangeException if the player is out of range.
    */
    public boolean hasWon(int iPlayer) throws CribbageRangeException
    {
        if ((iPlayer < 0) || (iPlayer >= MAX_PLAYERS))
            throw new CribbageRangeException("Player is out of range - ("+iPlayer+")");

        return (playerScore[iPlayer] >= MAX_SCORE);
    }

    /** Method used to return the int representing the player that has won (if any).
     * @return A int representing the player that has won, or NO_PLAYER if no player
     * has won.
    */
    public int getWinner()
    {
        int iRet = NO_PLAYER;

        for (int i = 0; i < playerScore.length; i++) {
            try {
                if (hasWon(i)) {
                    iRet = i;
                    break;
                }
            }
            catch (Exception e) { }
        }

        return iRet;
    }

    /** Method used to get the maximum number of players supported by this cribbage
     * board.
     * @return The maximum number of players that the board can support.
    */
    public int getMaxNumberOfPlayers()
    {
        return MAX_PLAYERS;
    }

    /** Method used to get the current number of players playing the cribbage
     * board.  For this board, this number is the same as what is returned by
     * getMaxNumberOfPlayers() - (only two players, have to have two players).
     * @return The current number of players playing.
    */
    public int getNumberOfPlayers()
    {
        return MAX_PLAYERS;
    }

    /** Method used to set the current number of players (up to the max supported
     * by the board).
     * @param iNumPlayers The int value of the number of players (2 to max) that 
     * will be playing the board.
     * @throws A CribbageRangeException is thrown if the number of players exceeds
     * the max number supported by the board or if the number is less than 2.
     */
    public void setNumberOfPlayers(int iNumPlayers) throws CribbageRangeException
    {
        if ((iNumPlayers < 2) || (iNumPlayers > MAX_PLAYERS))
            throw new CribbageRangeException("Max players can only be set to " +
                                              MAX_PLAYERS + " - tried: " + iNumPlayers + ".");

        // do nothing (only two are allowed with this board)
    }

    /** Method used to return if the board has sound effects.  This method does
     * not indicate if the sound is on or off, just that sound is available if
     * desired.
     * @return A boolean indicating board has sound effects or not.
    */
    public boolean hasSoundEffects()
    {
        return bSoundLoaded;
    }

    /** Method used to set if sound effects (peg movement) is enabled or not.
     * @param bSndOn Turn on peg move sound if true, otherwise turn it off.
    */
    public void setSoundOn(boolean bSndOn)
    {
        if ((bSndOn) && (bSoundLoaded)) { // initialize the sound objects if not already
            try {
                // aquire the output line and open it
                if (info == null) {
                    info = new DataLine.Info(SourceDataLine.class, format);
                }
                if (sdl == null) {
                    sdl = (SourceDataLine) AudioSystem.getLine(info);
                }
                if (!bSoundOn) { // only if sound is not currently on
                    sdl.open(format, snd_len);
                }
            }
            catch (Exception e) {
                System.err.println("Sound On Error: " + e);
                bSoundLoaded = false;
            }
        }
        else if (bSoundLoaded) { // turning sound off - release the line
            try {
                if (sdl != null) { // sound has been turned on...
                    sdl.close();
                }
                sdl = null;
            }
            catch (Exception ee) {
                System.err.println("Sound Off Error: " + ee);
            }
        }

        // set sound on or off
        if (bSoundLoaded) bSoundOn = bSndOn;
    }

    /** Method used to see if peg move sounds are on or off.
     * @return A boolean indicating if the sound is on or off.
    */
    public boolean isSoundOn()
    {
        return bSoundOn;
    }
}
